// Licensed under the MIT license.
//
// Copyright © 2006-2009 Simon Thum
// Copyright © 2008-2012 Kristian Høgsberg
// Copyright © 2010-2012 Intel Corporation
// Copyright © 2010-2011 Benjamin Franzke
// Copyright © 2011-2012 Collabora, Ltd.
// Copyright © 2013-2014 Jonas Ådahl
// Copyright © 2013-2015 David Herrmann <dh.herrmann@gmail.com>
// Copyright © 2013-2025 Red Hat, Inc.
// Copyright © 2025 Dmitry Sapozhnikov
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice (including the next
// paragraph) shall be included in all copies or substantial portions of the
// Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include <sys/epoll.h>   // ::epoll_create1()
//todo use SIGEV_SIGNAL instead of timerfd_create
#include <sys/timerfd.h> // ::timerfd_create() ::timerfd_settime()
#include <sys/stat.h>    // ::fstat()
#include <linux/input.h> // EV_*
#include <fnmatch.h>     // ::fnmatch()
#include <dirent.h>      // ::dirent
#include <fcntl.h>       // O_RDWR | O_NONBLOCK | O_CLOEXEC
#include <sys/inotify.h> // ::inotify

#define CASE_RETURN_STRING(a) case a: return #a

using WacomDevice = void*;

namespace netxs::lixx // li++, libinput++.
{
    #if defined(DEBUG)
    template<class ...Args>
    void log(auto format, Args... args)
    {
        auto f = "lixx: "s + format;
        netxs::log(f, std::forward<Args>(args)...);
    }
    #else
        #define log(...) noop()
    #endif

    enum read_flags
    {
        LIBEVDEV_READ_FLAG_SYNC       = 1 << 0,
        LIBEVDEV_READ_FLAG_NORMAL     = 1 << 1,
        LIBEVDEV_READ_FLAG_FORCE_SYNC = 1 << 2,
        LIBEVDEV_READ_FLAG_BLOCKING   = 1 << 3,
    };
    enum event_filter_status
    {
        EVENT_FILTER_NONE,
        EVENT_FILTER_MODIFIED,
        EVENT_FILTER_DISCARD,
    };
    enum sync_states
    {
        SYNC_NONE,
        SYNC_NEEDED,
        SYNC_IN_PROGRESS,
    };
    enum touch_states
    {
        TOUCH_OFF,
        TOUCH_STARTED,
        TOUCH_STOPPED,
        TOUCH_ONGOING,
        TOUCH_CHANGED,
    };
    enum libinput_arbitration_state
    {
        ARBITRATION_NOT_ACTIVE,
        ARBITRATION_IGNORE_ALL,
        ARBITRATION_IGNORE_RECT,
    };
    enum libinput_device_caps
    {
        EVDEV_DEVICE_KEYBOARD        = 1 << 0,
        EVDEV_DEVICE_POINTER         = 1 << 1,
        EVDEV_DEVICE_GESTURE         = 1 << 2,
        EVDEV_DEVICE_TOUCH           = 1 << 3,
        EVDEV_DEVICE_SWITCH          = 1 << 4,
        EVDEV_DEVICE_TABLET_PAD      = 1 << 5,
        EVDEV_DEVICE_TABLET          = 1 << 6,
    };
    enum ud_type_enum
    {
        UDEV_MOUSE         = 1 << 1,
        UDEV_POINTINGSTICK = 1 << 2,
        UDEV_TOUCHPAD      = 1 << 3,
        UDEV_TABLET        = 1 << 4,
        UDEV_TABLET_PAD    = 1 << 5,
        UDEV_JOYSTICK      = 1 << 6,
        UDEV_KEYBOARD      = 1 << 7,
    };
    enum libinput_device_tags
    {
        EVDEV_TAG_NONE               = 0,
        EVDEV_TAG_EXTERNAL_MOUSE     = 1 << 0,
        EVDEV_TAG_INTERNAL_TOUCHPAD  = 1 << 1,
        EVDEV_TAG_EXTERNAL_TOUCHPAD  = 1 << 2,
        EVDEV_TAG_TRACKPOINT         = 1 << 3,
        EVDEV_TAG_KEYBOARD           = 1 << 4,
        EVDEV_TAG_LID_SWITCH         = 1 << 5,
        EVDEV_TAG_INTERNAL_KEYBOARD  = 1 << 6,
        EVDEV_TAG_EXTERNAL_KEYBOARD  = 1 << 7,
        EVDEV_TAG_TABLET_MODE_SWITCH = 1 << 8,
        EVDEV_TAG_TABLET_TOUCHPAD    = 1 << 9,
        EVDEV_TAG_VIRTUAL            = 1 << 10,
    };
    //todo unify, combine with ud_type_enum and rename to ID_INPUT_*
    enum evdev_ud_device_tags
    {
        EVDEV_UDEV_TAG_INPUT         = 1 << 0,
        EVDEV_UDEV_TAG_KEYBOARD      = 1 << 1,
        EVDEV_UDEV_TAG_MOUSE         = 1 << 2,
        EVDEV_UDEV_TAG_TOUCHPAD      = 1 << 3,
        EVDEV_UDEV_TAG_TOUCHSCREEN   = 1 << 4,
        EVDEV_UDEV_TAG_TABLET        = 1 << 5,
        EVDEV_UDEV_TAG_JOYSTICK      = 1 << 6,
        EVDEV_UDEV_TAG_ACCELEROMETER = 1 << 7,
        EVDEV_UDEV_TAG_TABLET_PAD    = 1 << 8,
        EVDEV_UDEV_TAG_POINTINGSTICK = 1 << 9,
        EVDEV_UDEV_TAG_TRACKBALL     = 1 << 10,
        EVDEV_UDEV_TAG_SWITCH        = 1 << 11,
        EVDEV_UDEV_TAG_PURETABLET    = 1 << 12,
    };
    enum evdev_button_scroll_state
    {
        BUTTONSCROLL_IDLE,
        BUTTONSCROLL_BUTTON_DOWN, // Button is down.
        BUTTONSCROLL_READY,       // Ready for scroll events.
        BUTTONSCROLL_SCROLLING,   // Have sent scroll events.
    };
    enum evdev_button_scroll_lock_state
    {
        BUTTONSCROLL_LOCK_DISABLED,
        BUTTONSCROLL_LOCK_IDLE,
        BUTTONSCROLL_LOCK_FIRSTDOWN,
        BUTTONSCROLL_LOCK_FIRSTUP,
        BUTTONSCROLL_LOCK_SECONDDOWN,
    };
    enum evdev_middlebutton_state
    {
        MIDDLEBUTTON_IDLE,
        MIDDLEBUTTON_LEFT_DOWN,
        MIDDLEBUTTON_RIGHT_DOWN,
        MIDDLEBUTTON_MIDDLE,
        MIDDLEBUTTON_LEFT_UP_PENDING,
        MIDDLEBUTTON_RIGHT_UP_PENDING,
        MIDDLEBUTTON_IGNORE_LR,
        MIDDLEBUTTON_IGNORE_L,
        MIDDLEBUTTON_IGNORE_R,
        MIDDLEBUTTON_PASSTHROUGH,
    };
    enum libinput_config_accel_profile
    {
        LIBINPUT_CONFIG_ACCEL_PROFILE_NONE     = 0,      // Placeholder for devices that don't have a configurable pointer acceleration profile.
        LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT     = 1 << 0, // A flat acceleration profile. Pointer motion is accelerated by a constant (device-specific) factor, depending on the current speed.
        LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE = 1 << 1, // An adaptive acceleration profile. Pointer acceleration depends on the input speed. This is the default profile for most devices.
        LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM   = 1 << 2, // A custom acceleration profile. Device movement acceleration depends on user defined custom acceleration functions for each movement type.
    };
    enum libinput_switch
    {
        LIBINPUT_SWITCH_LID = 1,    // The laptop lid was closed when the switch state is @ref LIBINPUT_SWITCH_STATE_ON, or was opened when it is @ref LIBINPUT_SWITCH_STATE_OFF.
        LIBINPUT_SWITCH_TABLET_MODE,// This switch indicates whether the device is in normal laptop mode or behaves like a tablet-like device where the primary interaction is usually a touch screen. When in tablet mode, the keyboard and touchpad are usually inaccessible.
    };
    enum libinput_event_type
    {
        LIBINPUT_EVENT_NONE = 0,      // This is not a real event type, and is only used to tell the user that no new event is available in the queue.
        LIBINPUT_EVENT_DEVICE_ADDED,  // Signals that a device has been added to the context. The device will not be read until the next time the user calls libinput_dispatch() and data is available.
        LIBINPUT_EVENT_DEVICE_REMOVED,// Signals that a device has been removed. No more events from the associated device will be in the queue or be queued after this event.
        LIBINPUT_EVENT_KEYBOARD_KEY = 300,
        LIBINPUT_EVENT_POINTER_MOTION = 400,
        LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE,
        LIBINPUT_EVENT_POINTER_BUTTON,
        LIBINPUT_EVENT_POINTER_AXIS,
        LIBINPUT_EVENT_POINTER_SCROLL_WHEEL,     // A scroll event from a wheel. This event is sent is sent **in addition** to the @ref LIBINPUT_EVENT_POINTER_AXIS event for all events with a libinput_event_pointer_get_axis_source() of @ref LIBINPUT_POINTER_AXIS_SOURCE_WHEEL. Ignore @ref LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
        LIBINPUT_EVENT_POINTER_SCROLL_FINGER,    // A scroll event caused by the movement of one or more fingers on a device. This event is sent is sent **in addition** to the @ref LIBINPUT_EVENT_POINTER_AXIS event for all events with a libinput_event_pointer_get_axis_source() of @ref LIBINPUT_POINTER_AXIS_SOURCE_FINGER. Ignore @ref LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
        LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS,// A scroll event from a continuous scroll source, e.g. button scrolling. This event is sent is sent **in addition** to the @ref LIBINPUT_EVENT_POINTER_AXIS event for all events with a libinput_event_pointer_get_axis_source() of @ref LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS. Ignore @ref LIBINPUT_EVENT_POINTER_AXIS if you are processing this event.
        LIBINPUT_EVENT_TOUCH_DOWN = 500,
        LIBINPUT_EVENT_TOUCH_UP,
        LIBINPUT_EVENT_TOUCH_MOTION,
        LIBINPUT_EVENT_TOUCH_CANCEL,
        LIBINPUT_EVENT_TOUCH_FRAME,            // Signals the end of a set of touchpoints at one device sample time. This event has no coordinate information attached.
        LIBINPUT_EVENT_TABLET_TOOL_AXIS = 600, // One or more axes have changed state on a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability. This event is only sent when the tool is in proximity, see @ref LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY for details.
        LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY,  // Signals that a tool has come in or out of proximity of a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
        LIBINPUT_EVENT_TABLET_TOOL_TIP,        // Signals that a tool has come in contact with the surface of a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
        LIBINPUT_EVENT_TABLET_TOOL_BUTTON,     // Signals that a tool has changed a logical button state on a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_TOOL capability.
        LIBINPUT_EVENT_TABLET_PAD_BUTTON = 700,// A button pressed on a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
        LIBINPUT_EVENT_TABLET_PAD_RING,        // A status change on a tablet ring with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
        LIBINPUT_EVENT_TABLET_PAD_STRIP,       // A status change on a strip on a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
        LIBINPUT_EVENT_TABLET_PAD_KEY,         // A key pressed on a device with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
        LIBINPUT_EVENT_TABLET_PAD_DIAL,        // A status change on a tablet dial with the @ref LIBINPUT_DEVICE_CAP_TABLET_PAD capability.
        LIBINPUT_EVENT_GESTURE_SWIPE_BEGIN = 800,
        LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE,
        LIBINPUT_EVENT_GESTURE_SWIPE_END,
        LIBINPUT_EVENT_GESTURE_PINCH_BEGIN,
        LIBINPUT_EVENT_GESTURE_PINCH_UPDATE,
        LIBINPUT_EVENT_GESTURE_PINCH_END,
        LIBINPUT_EVENT_GESTURE_HOLD_BEGIN,
        LIBINPUT_EVENT_GESTURE_HOLD_END,
        LIBINPUT_EVENT_SWITCH_TOGGLE = 900,
    };
    enum libinput_config_status
    {
        LIBINPUT_CONFIG_STATUS_SUCCESS = 0, // Config applied successfully.
        LIBINPUT_CONFIG_STATUS_UNSUPPORTED, // Configuration not available on this device.
        LIBINPUT_CONFIG_STATUS_INVALID,     // Invalid parameter range.
    };
    enum libinput_config_drag_lock_state
    {
        LIBINPUT_CONFIG_DRAG_LOCK_DISABLED,        // Drag lock is to be disabled, or is currently disabled.
        LIBINPUT_CONFIG_DRAG_LOCK_ENABLED_TIMEOUT, // Drag lock is to be enabled in timeout mode, or is currently enabled in timeout mode.
        LIBINPUT_CONFIG_DRAG_LOCK_ENABLED = LIBINPUT_CONFIG_DRAG_LOCK_ENABLED_TIMEOUT,// Legacy spelling for LIBINPUT_CONFIG_DRAG_LOCK_ENABLED_TIMEOUT.
        LIBINPUT_CONFIG_DRAG_LOCK_ENABLED_STICKY,  // Drag lock is to be enabled in sticky mode, or is currently enabled in sticky mode.
    };
    enum libinput_config_scroll_method
    {
        LIBINPUT_CONFIG_SCROLL_NO_SCROLL      = 0,      // Never send scroll events instead of pointer motion events. This has no effect on events generated by scroll wheels.
        LIBINPUT_CONFIG_SCROLL_2FG            = 1 << 0, // Send scroll events when two fingers are logically down on the device.
        LIBINPUT_CONFIG_SCROLL_EDGE           = 1 << 1, // Send scroll events when a finger moves along the bottom or right edge of a device.
        LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN = 1 << 2, // Send scroll events when a button is down and the device moves along a scroll-capable axis.
    };
    enum libinput_config_click_method
    {
        LIBINPUT_CONFIG_CLICK_METHOD_NONE         = 0,      // Do not send software-emulated button events. This has no effect on events generated by physical buttons.
        LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS = 1 << 0, // Use software-button areas to generate button events.
        LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER  = 1 << 1, // The number of fingers decides which button press to generate.
    };
    enum libinput_config_3fg_drag_state
    {
        LIBINPUT_CONFIG_3FG_DRAG_DISABLED,    // Drag is to be disabled, or is currently disabled.
        LIBINPUT_CONFIG_3FG_DRAG_ENABLED_3FG, // Drag is to be enabled for 3 fingers, or is currently enabled.
        LIBINPUT_CONFIG_3FG_DRAG_ENABLED_4FG, // Drag is to be enabled for 4 fingers, or is currently enabled.
    };
    enum libinput_config_send_events_mode
    {
        LIBINPUT_CONFIG_SEND_EVENTS_ENABLED                    = 0,      // Send events from this device normally. This is a placeholder mode only, any device detected by libinput can be enabled. Do not test for this value as bitmask.
        LIBINPUT_CONFIG_SEND_EVENTS_DISABLED                   = 1 << 0, // Do not send events through this device. Depending on the device, this may close all file descriptors on the device or it may leave the file descriptors open and route events through a different device.
        LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE = 1 << 1, // If an external pointer device is plugged in, do not send events, from this device. This option may be available on built-in touchpads.
    };
    enum wheel_event
    {
        WHEEL_EVENT_SCROLL_ACCUMULATED,
        WHEEL_EVENT_SCROLL,
        WHEEL_EVENT_SCROLL_TIMEOUT,
        WHEEL_EVENT_SCROLL_DIR_CHANGED,
    };
    enum key_type
    {
        KEY_TYPE_NONE,
        KEY_TYPE_KEY,
        KEY_TYPE_BUTTON,
    };
    enum mt_palm_state
    {
        MT_PALM_NONE,
        MT_PALM_NEW,
        MT_PALM_IS_PALM,
        MT_PALM_WAS_PALM, // This touch sequence was a palm but isn't now.
    };
    enum wheel_state
    {
        WHEEL_STATE_NONE,
        WHEEL_STATE_ACCUMULATING_SCROLL,
        WHEEL_STATE_SCROLLING,
    };
    enum wheel_direction
    {
        WHEEL_DIR_UNKNOW,
        WHEEL_DIR_VPOS,
        WHEEL_DIR_VNEG,
        WHEEL_DIR_HPOS,
        WHEEL_DIR_HNEG,
    };
    enum debounce_state
    {
        DEBOUNCE_STATE_IS_UP = 100,
        DEBOUNCE_STATE_IS_DOWN,
        DEBOUNCE_STATE_IS_DOWN_WAITING,
        DEBOUNCE_STATE_IS_UP_DELAYING,
        DEBOUNCE_STATE_IS_UP_DELAYING_SPURIOUS,
        DEBOUNCE_STATE_IS_UP_DETECTING_SPURIOUS,
        DEBOUNCE_STATE_IS_DOWN_DETECTING_SPURIOUS,
        DEBOUNCE_STATE_IS_UP_WAITING,
        DEBOUNCE_STATE_IS_DOWN_DELAYING,
        DEBOUNCE_STATE_DISABLED = 999,
    };
    enum switch_reliability
    {
        RELIABILITY_RELIABLE,
        RELIABILITY_UNRELIABLE,
        RELIABILITY_WRITE_OPEN,
    };
    enum evdev_event_type
    {
        EVDEV_NONE                = 0,
        EVDEV_ABSOLUTE_TOUCH_DOWN = 1 << 0,
        EVDEV_ABSOLUTE_MOTION     = 1 << 1,
        EVDEV_ABSOLUTE_TOUCH_UP   = 1 << 2,
        EVDEV_ABSOLUTE_MT         = 1 << 3,
        EVDEV_KEY                 = 1 << 4,
        EVDEV_RELATIVE_MOTION     = 1 << 5,
        EVDEV_BUTTON              = 1 << 6,
    };
    enum debounce_event
    {
        DEBOUNCE_EVENT_PRESS = 50,
        DEBOUNCE_EVENT_RELEASE,
        DEBOUNCE_EVENT_TIMEOUT,
        DEBOUNCE_EVENT_TIMEOUT_SHORT,
        DEBOUNCE_EVENT_OTHERBUTTON,
    };
    enum suspend_trigger
    {
        SUSPEND_NO_FLAG         = 0,
        SUSPEND_EXTERNAL_MOUSE  = 1 << 0,
        SUSPEND_SENDEVENTS      = 1 << 1,
        SUSPEND_LID             = 1 << 2,
        SUSPEND_TABLET_MODE     = 1 << 3,
    };
    enum libinput_pointer_axis
    {
        LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL,
        LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL,
    };
    enum tap_event
    {
        TAP_EVENT_TOUCH = 12,
        TAP_EVENT_MOTION,
        TAP_EVENT_RELEASE,
        TAP_EVENT_BUTTON,
        TAP_EVENT_TIMEOUT,
        TAP_EVENT_THUMB,
        TAP_EVENT_PALM,
        TAP_EVENT_PALM_UP,
    };
    enum evdev_middlebutton_event
    {
        MIDDLEBUTTON_EVENT_L_DOWN,
        MIDDLEBUTTON_EVENT_R_DOWN,
        MIDDLEBUTTON_EVENT_OTHER,
        MIDDLEBUTTON_EVENT_L_UP,
        MIDDLEBUTTON_EVENT_R_UP,
        MIDDLEBUTTON_EVENT_TIMEOUT,
        MIDDLEBUTTON_EVENT_ALL_UP,
    };
    enum libinput_pointer_axis_source
    {
        LIBINPUT_POINTER_AXIS_SOURCE_WHEEL = 1, // The event is caused by the rotation of a wheel.
        LIBINPUT_POINTER_AXIS_SOURCE_FINGER,    // The event is caused by the movement of one or more fingers on a device.
        LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS,// The event is caused by the motion of some device.
        LIBINPUT_POINTER_AXIS_SOURCE_WHEEL_TILT,// @deprecated This axis source is deprecated as of libinput 1.16. The event is caused by the tilting of a mouse wheel rather than its rotation. This method is commonly used on mice without separate horizontal scroll wheels.
    };
    enum gesture_event
    {
        GESTURE_EVENT_RESET,
        GESTURE_EVENT_END,
        GESTURE_EVENT_CANCEL,
        GESTURE_EVENT_FINGER_DETECTED,
        GESTURE_EVENT_FINGER_SWITCH_TIMEOUT,
        GESTURE_EVENT_TAP_TIMEOUT,
        GESTURE_EVENT_HOLD_TIMEOUT,
        GESTURE_EVENT_HOLD_AND_MOTION_START,
        GESTURE_EVENT_POINTER_MOTION_START,
        GESTURE_EVENT_SCROLL_START,
        GESTURE_EVENT_SWIPE_START,
        GESTURE_EVENT_PINCH_START,
        GESTURE_EVENT_3FG_DRAG_START,
        GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT,
    };
    enum scroll_event
    {
        SCROLL_EVENT_TOUCH,
        SCROLL_EVENT_MOTION,
        SCROLL_EVENT_RELEASE,
        SCROLL_EVENT_TIMEOUT,
        SCROLL_EVENT_POSTED,
    };
    enum tp_edge
    {
        EDGE_NONE   = 0,
        EDGE_RIGHT  = 1 << 0,
        EDGE_BOTTOM = 1 << 1,
    };
    enum touchpad_event
    {
        TOUCHPAD_EVENT_NONE           = 0,
        TOUCHPAD_EVENT_MOTION         = 1 << 0,
        TOUCHPAD_EVENT_BUTTON_PRESS   = 1 << 1,
        TOUCHPAD_EVENT_BUTTON_RELEASE = 1 << 2,
        TOUCHPAD_EVENT_OTHERAXIS      = 1 << 3,
        TOUCHPAD_EVENT_TIMESTAMP      = 1 << 4,
    };
    enum tp_gesture_state
    {
        GESTURE_STATE_NONE,
        GESTURE_STATE_UNKNOWN,
        GESTURE_STATE_HOLD,
        GESTURE_STATE_HOLD_AND_MOTION,
        GESTURE_STATE_POINTER_MOTION,
        GESTURE_STATE_SCROLL_START,
        GESTURE_STATE_SCROLL,
        GESTURE_STATE_PINCH_START,
        GESTURE_STATE_PINCH,
        GESTURE_STATE_SWIPE_START,
        GESTURE_STATE_SWIPE,
        GESTURE_STATE_3FG_DRAG_START,
        GESTURE_STATE_3FG_DRAG,
        GESTURE_STATE_3FG_DRAG_RELEASED,
        GESTURE_STATE_COUNT,
    };
    enum touch_state
    {
        TOUCH_NONE,
        TOUCH_HOVERING,
        TOUCH_BEGIN,
        TOUCH_UPDATE,
        TOUCH_MAYBE_END,
        TOUCH_END,
    };
    enum tp_tap_touch_state
    {
        TAP_TOUCH_STATE_IDLE = 16, // Not in touch.
        TAP_TOUCH_STATE_TOUCH,     // Touching, may tap.
        TAP_TOUCH_STATE_DEAD,      // Exceeded motion/timeout.
    };
    enum tp_edge_scroll_touch_state
    {
        EDGE_SCROLL_TOUCH_STATE_NONE,
        EDGE_SCROLL_TOUCH_STATE_EDGE_NEW,
        EDGE_SCROLL_TOUCH_STATE_EDGE,
        EDGE_SCROLL_TOUCH_STATE_AREA,
    };
    enum touch_palm_state
    {
        TOUCH_PALM_NONE,
        TOUCH_PALM_EDGE,
        TOUCH_PALM_TYPING,
        TOUCH_PALM_TRACKPOINT,
        TOUCH_PALM_TOOL_PALM,
        TOUCH_PALM_PRESSURE,
        TOUCH_PALM_TOUCH_SIZE,
        TOUCH_PALM_ARBITRATION,
    };
    enum tp_tap_state
    {
        TAP_STATE_IDLE = 4,
        TAP_STATE_TOUCH,
        TAP_STATE_HOLD,
        TAP_STATE_1FGTAP_TAPPED,
        TAP_STATE_2FGTAP_TAPPED,
        TAP_STATE_3FGTAP_TAPPED,
        TAP_STATE_TOUCH_2,
        TAP_STATE_TOUCH_2_HOLD,
        TAP_STATE_TOUCH_2_RELEASE,
        TAP_STATE_TOUCH_3,
        TAP_STATE_TOUCH_3_HOLD,
        TAP_STATE_TOUCH_3_RELEASE,
        TAP_STATE_TOUCH_3_RELEASE_2,
        TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP,
        TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP,
        TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP,
        TAP_STATE_1FGTAP_DRAGGING_OR_TAP,
        TAP_STATE_2FGTAP_DRAGGING_OR_TAP,
        TAP_STATE_3FGTAP_DRAGGING_OR_TAP,
        TAP_STATE_1FGTAP_DRAGGING,
        TAP_STATE_2FGTAP_DRAGGING,
        TAP_STATE_3FGTAP_DRAGGING,
        TAP_STATE_1FGTAP_DRAGGING_WAIT,
        TAP_STATE_2FGTAP_DRAGGING_WAIT,
        TAP_STATE_3FGTAP_DRAGGING_WAIT,
        TAP_STATE_1FGTAP_DRAGGING_2,
        TAP_STATE_2FGTAP_DRAGGING_2,
        TAP_STATE_3FGTAP_DRAGGING_2,
        TAP_STATE_DEAD, // Finger count exceeded.
    };
    enum tp_thumb_state
    {
        THUMB_STATE_FINGER,
        THUMB_STATE_JAILED,
        THUMB_STATE_PINCH,
        THUMB_STATE_SUPPRESSED,
        THUMB_STATE_REVIVED,
        THUMB_STATE_REVIVED_JAILED,
        THUMB_STATE_DEAD,
    };
    enum tp_jump_state
    {
        JUMP_STATE_IGNORE = 0,
        JUMP_STATE_EXPECT_FIRST,
        JUMP_STATE_EXPECT_DELAY,
    };
    enum libinput_device_capability
    {
        LIBINPUT_DEVICE_CAP_KEYBOARD    = 0,
        LIBINPUT_DEVICE_CAP_POINTER     = 1,
        LIBINPUT_DEVICE_CAP_TOUCH       = 2,
        LIBINPUT_DEVICE_CAP_TABLET_TOOL = 3,
        LIBINPUT_DEVICE_CAP_TABLET_PAD  = 4,
        LIBINPUT_DEVICE_CAP_GESTURE     = 5,
        LIBINPUT_DEVICE_CAP_SWITCH      = 6,
    };
    enum tablet_status
    {
        TABLET_NONE                    = 0,
        TABLET_AXES_UPDATED            = 1 << 0,
        TABLET_BUTTONS_PRESSED         = 1 << 1,
        TABLET_BUTTONS_DOWN            = 1 << 2,
        TABLET_BUTTONS_RELEASED        = 1 << 3,
        TABLET_TOOL_UPDATED            = 1 << 4,
        TABLET_TOOL_IN_CONTACT         = 1 << 5,
        TABLET_TOOL_LEAVING_PROXIMITY  = 1 << 6,
        TABLET_TOOL_OUT_OF_PROXIMITY   = 1 << 7,
        TABLET_TOOL_ENTERING_PROXIMITY = 1 << 8,
        TABLET_TOOL_ENTERING_CONTACT   = 1 << 9,
        TABLET_TOOL_LEAVING_CONTACT    = 1 << 10,
        TABLET_TOOL_OUT_OF_RANGE       = 1 << 11,
        TABLET_TOOL_OUTSIDE_AREA       = 1 << 12,
    };
    enum slot_state_enum
    {
        SLOT_STATE_NONE,
        SLOT_STATE_BEGIN,
        SLOT_STATE_UPDATE,
        SLOT_STATE_END,
    };
    enum pad_toggle_button_target_mode
    {
        MODE_NEXT = -1,
        MODE_0,
        MODE_1,
        MODE_2,
        MODE_3,
    };
    enum libinput_tablet_pad_ring_axis_source
    {
        LIBINPUT_TABLET_PAD_RING_SOURCE_UNKNOWN = 1,
        LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER,// The event is caused by the movement of one or more fingers on the ring.
    };
    enum libinput_tablet_pad_strip_axis_source
    {
        LIBINPUT_TABLET_PAD_STRIP_SOURCE_UNKNOWN = 1,
        LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER,// The event is caused by the movement of one or more fingers on the strip.
    };
    enum pad_axes
    {
        PAD_AXIS_NONE   = 0,
        PAD_AXIS_RING1  = 1 << 0,
        PAD_AXIS_RING2  = 1 << 1,
        PAD_AXIS_STRIP1 = 1 << 2,
        PAD_AXIS_STRIP2 = 1 << 3,
        PAD_AXIS_DIAL1  = 1 << 4,
        PAD_AXIS_DIAL2  = 1 << 5,
    };
    enum pad_status
    {
        PAD_NONE             = 0,
        PAD_AXES_UPDATED     = 1 << 0,
        PAD_BUTTONS_PRESSED  = 1 << 1,
        PAD_BUTTONS_RELEASED = 1 << 2,
    };
    enum button_event
    {
        BUTTON_EVENT_NONE = 0,
        BUTTON_EVENT_IN_BOTTOM_R = 30,
        BUTTON_EVENT_IN_BOTTOM_M,
        BUTTON_EVENT_IN_BOTTOM_L,
        BUTTON_EVENT_IN_TOP_R,
        BUTTON_EVENT_IN_TOP_M,
        BUTTON_EVENT_IN_TOP_L,
        BUTTON_EVENT_IN_AREA,
        BUTTON_EVENT_UP,
        BUTTON_EVENT_PRESS,
        BUTTON_EVENT_RELEASE,
        BUTTON_EVENT_TIMEOUT,
    };
    enum libinput_config_accel_type
    {
        LIBINPUT_ACCEL_TYPE_FALLBACK,// The default acceleration type used as a fallback when other acceleration types are not provided.
        LIBINPUT_ACCEL_TYPE_MOTION,  // Acceleration type for regular pointer movement. This type is always supported.
        LIBINPUT_ACCEL_TYPE_SCROLL,  // Acceleration type for scroll movement. This type is supported by mouse and touchpad.
    };
    enum libinput_tablet_tool_proximity_state
    {
        LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_OUT,
        LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN,
    };
    enum directions
    {
        N  = 1 << 0,
        NE = 1 << 1,
        E  = 1 << 2,
        SE = 1 << 3,
        S  = 1 << 4,
        SW = 1 << 5,
        W  = 1 << 6,
        NW = 1 << 7,
        UNDEFINED_DIRECTION = 0xff
    };
    enum libinput_tablet_tool_tip_state
    {
        LIBINPUT_TABLET_TOOL_TIP_UP,
        LIBINPUT_TABLET_TOOL_TIP_DOWN,
    };
    enum libinput_tablet_tool_type
    {
        LIBINPUT_TABLET_TOOL_TYPE_NONE = 0,
        LIBINPUT_TABLET_TOOL_TYPE_PEN  = 1, // A generic pen.
        LIBINPUT_TABLET_TOOL_TYPE_ERASER,   // Eraser.
        LIBINPUT_TABLET_TOOL_TYPE_BRUSH,    // A paintbrush-like tool.
        LIBINPUT_TABLET_TOOL_TYPE_PENCIL,   // Physical drawing tool, e.g. Wacom Inking Pen.
        LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH, // An airbrush-like tool.
        LIBINPUT_TABLET_TOOL_TYPE_MOUSE,    // A mouse bound to the tablet.
        LIBINPUT_TABLET_TOOL_TYPE_LENS,     // A mouse tool with a lens.
        LIBINPUT_TABLET_TOOL_TYPE_TOTEM,    // A rotary device with positional and rotation data.
    };
    enum libinput_tablet_tool_axis
    {
        LIBINPUT_TABLET_TOOL_AXIS_NONE       = 0,
        LIBINPUT_TABLET_TOOL_AXIS_X          = 1,
        LIBINPUT_TABLET_TOOL_AXIS_Y          = 2,
        LIBINPUT_TABLET_TOOL_AXIS_DISTANCE   = 3,
        LIBINPUT_TABLET_TOOL_AXIS_PRESSURE   = 4,
        LIBINPUT_TABLET_TOOL_AXIS_TILT_X     = 5,
        LIBINPUT_TABLET_TOOL_AXIS_TILT_Y     = 6,
        LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z = 7,
        LIBINPUT_TABLET_TOOL_AXIS_SLIDER     = 8,
        LIBINPUT_TABLET_TOOL_AXIS_REL_WHEEL  = 9,
        LIBINPUT_TABLET_TOOL_AXIS_SIZE_MAJOR = 10,
        LIBINPUT_TABLET_TOOL_AXIS_SIZE_MINOR = 11,
    };
    enum pressure_heuristic_enum
    {
        PRESSURE_HEURISTIC_STATE_PROXIN1, // First proximity in event.
        PRESSURE_HEURISTIC_STATE_PROXIN2, // Second proximity in event.
        PRESSURE_HEURISTIC_STATE_DECIDE,  // Decide on offset now.
        PRESSURE_HEURISTIC_STATE_DONE,    // Decision's been made, live with it.
    };
    enum bustype
    {
        BT_UNKNOWN,
        BT_USB,
        BT_BLUETOOTH,
        BT_PS2,
        BT_RMI,
        BT_I2C,
        BT_SPI,
    };
    enum match_flags
    {
        M_NAME      = 1 << 0,
        M_BUS       = 1 << 1,
        M_VID       = 1 << 2,
        M_PID       = 1 << 3,
        M_DMI       = 1 << 4,
        M_UDEV_TYPE = 1 << 5,
        M_DT        = 1 << 6,
        M_VERSION   = 1 << 7,
        M_UNIQ      = 1 << 8,
        M_LAST      = M_UNIQ,
    };
    enum quirk
    {
        QUIRK_NONE = 0,
        QUIRK_MODEL_ALPS_SERIAL_TOUCHPAD = 100,
        QUIRK_MODEL_APPLE_TOUCHPAD,
        QUIRK_MODEL_APPLE_TOUCHPAD_ONEBUTTON,
        QUIRK_MODEL_BOUNCING_KEYS,
        QUIRK_MODEL_CHROMEBOOK,
        QUIRK_MODEL_CLEVO_W740SU,
        QUIRK_MODEL_DELL_CANVAS_TOTEM,
        QUIRK_MODEL_HP_PAVILION_DM4_TOUCHPAD,
        QUIRK_MODEL_HP_ZBOOK_STUDIO_G3,
        QUIRK_MODEL_INVERT_HORIZONTAL_SCROLLING,
        QUIRK_MODEL_LENOVO_SCROLLPOINT,
        QUIRK_MODEL_LENOVO_T450_TOUCHPAD,
        QUIRK_MODEL_LENOVO_X1GEN6_TOUCHPAD,
        QUIRK_MODEL_LENOVO_X230,
        QUIRK_MODEL_SYNAPTICS_SERIAL_TOUCHPAD,
        QUIRK_MODEL_SYSTEM76_BONOBO,
        QUIRK_MODEL_SYSTEM76_GALAGO,
        QUIRK_MODEL_SYSTEM76_KUDU,
        QUIRK_MODEL_TABLET_MODE_NO_SUSPEND,
        QUIRK_MODEL_TABLET_MODE_SWITCH_UNRELIABLE,
        QUIRK_MODEL_TOUCHPAD_VISIBLE_MARKER,
        QUIRK_MODEL_TRACKBALL,
        QUIRK_MODEL_WACOM_TOUCHPAD,
        QUIRK_MODEL_PRESSURE_PAD,
        QUIRK_MODEL_TOUCHPAD_PHANTOM_CLICKS,

        _QUIRK_LAST_MODEL_QUIRK_, // Guard: do not modify.

        QUIRK_ATTR_SIZE_HINT = 300,
        QUIRK_ATTR_TOUCH_SIZE_RANGE,
        QUIRK_ATTR_PALM_SIZE_THRESHOLD,
        QUIRK_ATTR_LID_SWITCH_RELIABILITY,
        QUIRK_ATTR_KEYBOARD_INTEGRATION,
        QUIRK_ATTR_TRACKPOINT_INTEGRATION,
        QUIRK_ATTR_TPKBCOMBO_LAYOUT,
        QUIRK_ATTR_PRESSURE_RANGE,
        QUIRK_ATTR_PALM_PRESSURE_THRESHOLD,
        QUIRK_ATTR_RESOLUTION_HINT,
        QUIRK_ATTR_TRACKPOINT_MULTIPLIER,
        QUIRK_ATTR_THUMB_PRESSURE_THRESHOLD,
        QUIRK_ATTR_USE_VELOCITY_AVERAGING,
        QUIRK_ATTR_TABLET_SMOOTHING,
        QUIRK_ATTR_THUMB_SIZE_THRESHOLD,
        QUIRK_ATTR_MSC_TIMESTAMP,
        QUIRK_ATTR_EVENT_CODE,
        QUIRK_ATTR_INPUT_PROP,
        QUIRK_ATTR_IS_VIRTUAL,

        _QUIRK_LAST_ATTR_QUIRK_, // Guard: do not modify.
    };
    enum libinput_device_model
    {
        EVDEV_MODEL_DEFAULT                   = 0,
        EVDEV_MODEL_WACOM_TOUCHPAD            = 1 << 1,
        EVDEV_MODEL_SYNAPTICS_SERIAL_TOUCHPAD = 1 << 2,
        EVDEV_MODEL_ALPS_SERIAL_TOUCHPAD      = 1 << 3,
        EVDEV_MODEL_LENOVO_T450_TOUCHPAD      = 1 << 4,
        EVDEV_MODEL_APPLE_TOUCHPAD_ONEBUTTON  = 1 << 5,
        EVDEV_MODEL_LENOVO_SCROLLPOINT        = 1 << 6,
        // udev tags, not true quirks.
        EVDEV_MODEL_TEST_DEVICE               = 1 << 20,
        EVDEV_MODEL_TRACKBALL                 = 1 << 21,
        EVDEV_MODEL_LENOVO_X220_TOUCHPAD_FW81 = 1 << 22,
    };
    enum button_state_enum
    {
        BUTTON_STATE_NONE,
        BUTTON_STATE_AREA,
        BUTTON_STATE_BOTTOM,
        BUTTON_STATE_TOP,
        BUTTON_STATE_TOP_NEW,
        BUTTON_STATE_TOP_TO_IGNORE,
        BUTTON_STATE_IGNORE,
    };

    constexpr ui32 evdev_usage_from_code(ui32 type, ui32 code) { return (type << 16) | code;  }
    constexpr ui16 evdev_usage_type(ui32 usage)                { return (ui16)(usage >> 16);  }
    constexpr ui16 evdev_usage_code(ui32 usage)                { return (ui16)usage & 0xFFFF; }
    struct evdev // The enum doesn't need to contain all event codes, only the ones we use in libinput - add to here as required.      * The order doesn't matter either since each enum value is just the type | code value anyway, keep it in somewhat logical groups where possible.
    {
        static constexpr auto success             = 0;
        static constexpr auto sync                = 1;
        static constexpr auto pressed             = 1;
        static constexpr auto released            = 0;
        static constexpr auto syn_report          = evdev_usage_from_code(EV_SYN, SYN_REPORT);
        static constexpr auto key_reserved        = evdev_usage_from_code(EV_KEY, KEY_RESERVED);
        static constexpr auto key_esc             = evdev_usage_from_code(EV_KEY, KEY_ESC);
        static constexpr auto key_micmute         = evdev_usage_from_code(EV_KEY, KEY_MICMUTE);
        static constexpr auto key_ok              = evdev_usage_from_code(EV_KEY, KEY_OK);
        static constexpr auto key_lights_toggle   = evdev_usage_from_code(EV_KEY, KEY_LIGHTS_TOGGLE);
        static constexpr auto key_als_toggle      = evdev_usage_from_code(EV_KEY, KEY_ALS_TOGGLE);
        static constexpr auto key_max             = evdev_usage_from_code(EV_KEY, KEY_MAX);
        static constexpr auto btn_left            = evdev_usage_from_code(EV_KEY, BTN_LEFT);
        static constexpr auto btn_right           = evdev_usage_from_code(EV_KEY, BTN_RIGHT);
        static constexpr auto btn_middle          = evdev_usage_from_code(EV_KEY, BTN_MIDDLE);
        static constexpr auto btn_side            = evdev_usage_from_code(EV_KEY, BTN_SIDE);
        static constexpr auto btn_extra           = evdev_usage_from_code(EV_KEY, BTN_EXTRA);
        static constexpr auto btn_forward         = evdev_usage_from_code(EV_KEY, BTN_FORWARD);
        static constexpr auto btn_back            = evdev_usage_from_code(EV_KEY, BTN_BACK);
        static constexpr auto btn_task            = evdev_usage_from_code(EV_KEY, BTN_TASK);
        static constexpr auto btn_joystick        = evdev_usage_from_code(EV_KEY, BTN_JOYSTICK);
        static constexpr auto btn_0               = evdev_usage_from_code(EV_KEY, BTN_0);
        static constexpr auto btn_1               = evdev_usage_from_code(EV_KEY, BTN_1);
        static constexpr auto btn_2               = evdev_usage_from_code(EV_KEY, BTN_2);
        static constexpr auto btn_stylus          = evdev_usage_from_code(EV_KEY, BTN_STYLUS);
        static constexpr auto btn_stylus2         = evdev_usage_from_code(EV_KEY, BTN_STYLUS2);
        static constexpr auto btn_stylus3         = evdev_usage_from_code(EV_KEY, BTN_STYLUS3);
        static constexpr auto btn_touch           = evdev_usage_from_code(EV_KEY, BTN_TOUCH);
        static constexpr auto btn_tool_pen        = evdev_usage_from_code(EV_KEY, BTN_TOOL_PEN);
        static constexpr auto btn_tool_rubber     = evdev_usage_from_code(EV_KEY, BTN_TOOL_RUBBER);
        static constexpr auto btn_tool_brush      = evdev_usage_from_code(EV_KEY, BTN_TOOL_BRUSH);
        static constexpr auto btn_tool_pencil     = evdev_usage_from_code(EV_KEY, BTN_TOOL_PENCIL);
        static constexpr auto btn_tool_airbrush   = evdev_usage_from_code(EV_KEY, BTN_TOOL_AIRBRUSH);
        static constexpr auto btn_tool_mouse      = evdev_usage_from_code(EV_KEY, BTN_TOOL_MOUSE);
        static constexpr auto btn_tool_lens       = evdev_usage_from_code(EV_KEY, BTN_TOOL_LENS);
        static constexpr auto btn_tool_quinttap   = evdev_usage_from_code(EV_KEY, BTN_TOOL_QUINTTAP);
        static constexpr auto btn_tool_doubletap  = evdev_usage_from_code(EV_KEY, BTN_TOOL_DOUBLETAP);
        static constexpr auto btn_tool_tripletap  = evdev_usage_from_code(EV_KEY, BTN_TOOL_TRIPLETAP);
        static constexpr auto btn_tool_quadtap    = evdev_usage_from_code(EV_KEY, BTN_TOOL_QUADTAP);
        static constexpr auto btn_tool_finger     = evdev_usage_from_code(EV_KEY, BTN_TOOL_FINGER);
        static constexpr auto btn_misc            = evdev_usage_from_code(EV_KEY, BTN_MISC);
        static constexpr auto btn_gear_up         = evdev_usage_from_code(EV_KEY, BTN_GEAR_UP);
        static constexpr auto btn_dpad_up         = evdev_usage_from_code(EV_KEY, BTN_DPAD_UP);
        static constexpr auto btn_dpad_right      = evdev_usage_from_code(EV_KEY, BTN_DPAD_RIGHT);
        static constexpr auto btn_trigger_happy   = evdev_usage_from_code(EV_KEY, BTN_TRIGGER_HAPPY);
        static constexpr auto btn_trigger_happy40 = evdev_usage_from_code(EV_KEY, BTN_TRIGGER_HAPPY40);
        static constexpr auto rel_x               = evdev_usage_from_code(EV_REL, REL_X);
        static constexpr auto rel_y               = evdev_usage_from_code(EV_REL, REL_Y);
        static constexpr auto rel_wheel           = evdev_usage_from_code(EV_REL, REL_WHEEL);
        static constexpr auto rel_wheel_hi_res    = evdev_usage_from_code(EV_REL, REL_WHEEL_HI_RES);
        static constexpr auto rel_hwheel          = evdev_usage_from_code(EV_REL, REL_HWHEEL);
        static constexpr auto rel_hwheel_hi_res   = evdev_usage_from_code(EV_REL, REL_HWHEEL_HI_RES);
        static constexpr auto rel_dial            = evdev_usage_from_code(EV_REL, REL_DIAL);
        static constexpr auto abs_x               = evdev_usage_from_code(EV_ABS, ABS_X);
        static constexpr auto abs_y               = evdev_usage_from_code(EV_ABS, ABS_Y);
        static constexpr auto abs_z               = evdev_usage_from_code(EV_ABS, ABS_Z);
        static constexpr auto abs_rx              = evdev_usage_from_code(EV_ABS, ABS_RX);
        static constexpr auto abs_ry              = evdev_usage_from_code(EV_ABS, ABS_RY);
        static constexpr auto abs_rz              = evdev_usage_from_code(EV_ABS, ABS_RZ);
        static constexpr auto abs_pressure        = evdev_usage_from_code(EV_ABS, ABS_PRESSURE);
        static constexpr auto abs_distance        = evdev_usage_from_code(EV_ABS, ABS_DISTANCE);
        static constexpr auto abs_throttle        = evdev_usage_from_code(EV_ABS, ABS_THROTTLE);
        static constexpr auto abs_wheel           = evdev_usage_from_code(EV_ABS, ABS_WHEEL);
        static constexpr auto abs_misc            = evdev_usage_from_code(EV_ABS, ABS_MISC);
        static constexpr auto abs_tilt_x          = evdev_usage_from_code(EV_ABS, ABS_TILT_X);
        static constexpr auto abs_tilt_y          = evdev_usage_from_code(EV_ABS, ABS_TILT_Y);
        static constexpr auto abs_mt_slot         = evdev_usage_from_code(EV_ABS, ABS_MT_SLOT);
        static constexpr auto abs_mt_position_x   = evdev_usage_from_code(EV_ABS, ABS_MT_POSITION_X);
        static constexpr auto abs_mt_position_y   = evdev_usage_from_code(EV_ABS, ABS_MT_POSITION_Y);
        static constexpr auto abs_mt_tool_type    = evdev_usage_from_code(EV_ABS, ABS_MT_TOOL_TYPE);
        static constexpr auto abs_mt_tracking_id  = evdev_usage_from_code(EV_ABS, ABS_MT_TRACKING_ID);
        static constexpr auto abs_mt_touch_major  = evdev_usage_from_code(EV_ABS, ABS_MT_TOUCH_MAJOR);
        static constexpr auto abs_mt_touch_minor  = evdev_usage_from_code(EV_ABS, ABS_MT_TOUCH_MINOR);
        static constexpr auto abs_mt_orientation  = evdev_usage_from_code(EV_ABS, ABS_MT_ORIENTATION);
        static constexpr auto abs_mt_pressure     = evdev_usage_from_code(EV_ABS, ABS_MT_PRESSURE);
        static constexpr auto sw_lid              = evdev_usage_from_code(EV_SW, SW_LID);
        static constexpr auto sw_tablet_mode      = evdev_usage_from_code(EV_SW, SW_TABLET_MODE);
        static constexpr auto msc_scan            = evdev_usage_from_code(EV_MSC, MSC_SCAN);
        static constexpr auto msc_serial          = evdev_usage_from_code(EV_MSC, MSC_SERIAL);
        static constexpr auto msc_timestamp       = evdev_usage_from_code(EV_MSC, MSC_TIMESTAMP);
    };

    static constexpr auto libinput_tablet_tool_type_min = LIBINPUT_TABLET_TOOL_TYPE_PEN;
    static constexpr auto libinput_tablet_tool_type_max = LIBINPUT_TABLET_TOOL_TYPE_LENS;
    static constexpr auto libinput_tablet_tool_axis_cnt = LIBINPUT_TABLET_TOOL_AXIS_SIZE_MINOR + 1;

    // For the Lenovo X230 custom accel. do not touch.
    static constexpr auto x230_threshold                        = datetime::round<fp64, std::chrono::microseconds>(400us); // In units/us.
    static constexpr auto x230_acceleration                     = 2.0; // Unitless factor.
    static constexpr auto x230_incline                          = 1.1; // Unitless factor.
    static constexpr auto x230_magic_slowdown                   = 0.4; // Unitless.
    static constexpr auto x230_tp_magic_low_res_factor          = 4.0; // Unitless.

    static constexpr auto libinput_accel_npoints_min            = 2;
    static constexpr auto libinput_accel_npoints_max            = 64; // Custom acceleration function max number of points an arbitrary limit of sample points it should be more than enough for everyone.
    static constexpr auto libinput_accel_step_max               = 10000;
    static constexpr auto libinput_accel_point_min_value        = 0;
    static constexpr auto libinput_accel_point_max_value        = 10000;

    static constexpr auto max_velocity_diff                     = datetime::round<fp64, std::chrono::microseconds>(1ms); // Units/us.
    static constexpr auto middlebutton_timeout                  = 50ms;
    static constexpr auto debounce_timeout_bounce               = 25ms;
    static constexpr auto debounce_timeout_spurious             = 12ms;
    static constexpr auto wheel_scroll_timeout                  = 500ms;
    static constexpr auto default_keyboard_activity_timeout_1   = 200ms;
    static constexpr auto default_keyboard_activity_timeout_2   = 500ms;
    static constexpr auto default_trackpoint_event_timeout      = 40ms;
    static constexpr auto default_trackpoint_activity_timeout   = 300ms;
    static constexpr auto default_draglock_timeout_period       = 300ms;
    static constexpr auto default_drag_timeout_period_base      = 160ms;
    static constexpr auto default_drag_timeout_period_perfinger = 20ms;
    static constexpr auto default_tap_timeout_period            = 300ms; // Old laptops can't handle double taps with an interval of <200ms.
    static constexpr auto timer_warning_limit                   = 20ms; // We only warn if we're more than 20ms behind.
    static constexpr auto motion_timeout                        = 1000ms;
    static constexpr auto first_motion_time_interval            = 7ms;  // Random but good enough interval for very first event.
    static auto           forced_proxout_timeout                = 50ms; // The tablet sends events every ~2ms , 50ms should be plenty enough to detect out-of-range. This value is higher during test suite runs.
    static constexpr auto palm_timeout                          = 200ms;
    static constexpr auto thumb_timeout                         = 100ms;
    static constexpr auto default_button_enter_timeout          = 100ms;
    static constexpr auto default_button_leave_timeout          = 300ms;
    static constexpr auto default_button_scroll_timeout         = 200ms;
    static constexpr auto default_scroll_lock_timeout           = 300ms;
    static constexpr auto default_gesture_hold_timeout          = 180ms;
    static constexpr auto default_gesture_switch_timeout        = 100ms;
    static constexpr auto default_gesture_swipe_timeout         = 150ms;
    static constexpr auto default_gesture_pinch_timeout         = 300ms;
    static constexpr auto quick_gesture_hold_timeout            = 40ms;
    static constexpr auto active_threshold                      = 100ms;
    static constexpr auto inactive_threshold                    = 50ms;
    static constexpr auto default_scroll_event_timeout          = 100ms;
    static constexpr auto minimum_acceleration_threshold        = datetime::round<fp64, std::chrono::microseconds>(200us); // In units/us.
    static constexpr auto default_acceleration_threshold        = datetime::round<fp64, std::chrono::microseconds>(400us); // In units/us.
    static constexpr auto default_acceleration                  = 2.0; // Unitless factor.
    static constexpr auto default_incline                       = 1.1; // Unitless factor.
    static constexpr auto fake_finger_overflow                  = 1ul << 7;
    static constexpr auto touchpad_history_length               = 4;
    static constexpr auto default_mouse_dpi                     = 1000;
    static constexpr auto vendor_id_apple                       = 0x5ac;
    static constexpr auto vendor_id_chicony                     = 0x4f2;
    static constexpr auto vendor_id_logitech                    = 0x46d;
    static constexpr auto vendor_id_wacom                       = 0x56a;
    static constexpr auto vendor_id_synaptics_serial            = 0x002;
    static constexpr auto product_id_apple_kbd_touchpad         = 0x273;
    static constexpr auto product_id_apple_appletouch           = 0x21a;
    static constexpr auto product_id_synaptics_serial           = 0x007;
    static constexpr auto product_id_wacom_ekr                  = 0x0331;
    consteval auto mm_to_dpi_normalized(auto mm) { return lixx::default_mouse_dpi / 25.4 * mm; }
    static constexpr auto default_scroll_threshold              = lixx::mm_to_dpi_normalized(3);
    static constexpr auto default_wheel_click_angle             = 15;
    static constexpr auto event_code_undefined                  = 0xffff;
    static constexpr auto thumb_ignore_speed_threshold          = 20; // mm/s.
    static constexpr auto default_tap_move_threshold            = 1.3; // mm.
    static constexpr auto pinch_disambiguation_move_threshold   = 1.5; // mm.
    static constexpr auto tp_magic_slowdown_flat                = 0.2968;
    static constexpr auto tp_magic_slowdown                     = 0.2968; // Unitless factor.
    static constexpr auto hold_and_motion_threshold             = 0.5; // mm.
    static constexpr auto tablet_history_length                 = 4;

    static constexpr auto clock_type     = CLOCK_MONOTONIC;
    static constexpr auto max_slots      = 0x100;
    static constexpr auto min_queue_size = 0x100;
    static constexpr auto abs_mt_min     = ABS_MT_SLOT;
    static constexpr auto abs_mt_max     = ABS_MT_TOOL_Y;
    static constexpr auto abs_mt_cnt     = lixx::abs_mt_max - lixx::abs_mt_min + 1;
    static constexpr auto valid_flags    = LIBEVDEV_READ_FLAG_NORMAL | LIBEVDEV_READ_FLAG_SYNC | LIBEVDEV_READ_FLAG_FORCE_SYNC | LIBEVDEV_READ_FLAG_BLOCKING;
    static constexpr auto tap_button_map = std::to_array<std::array<ui32, 3>>(
    {
        { evdev::btn_left, evdev::btn_right, evdev::btn_middle }, // 1/2/3 finger click maps to left/right/middle.
        { evdev::btn_left, evdev::btn_middle, evdev::btn_right }, // 1/2/3 finger click maps to left/middle/right.
    });
    static constexpr fp64 v_us2ms(fp64 units_per_us) { return units_per_us * 1000.0; }
    static constexpr fp64 v_us2s(fp64 units_per_us)  { return units_per_us * 1000000.0; }

    using fp64_range = netxs::limits<fp64>;
    using si32_range = netxs::limits<si32>;
    using fp64_rect  = netxs::xysz<fp64>;
    using fp64_coor  = netxs::xy2d<fp64>;
    using si32_rect  = netxs::xysz<si32>;
    using si32_coor  = netxs::xy2d<si32>;
    static constexpr auto zero_coor = fp64_coor{};

    using fd_t = os::fd_t;
    template<class T>
    struct libinput_timer_t;
    using libinput_event_listener = std::function<void(time, struct libinput_event&)>;
    using libinput_event_listener_sptr        = sptr<libinput_event_listener>;
    using libinput_sptr                       = sptr<struct libinput_t>;
    using libinput_timer_sptr                 = sptr<struct libinput_timer_t<struct libinput_timer_host>>;
    using libinput_device_sptr                = sptr<struct libinput_device_t>;
    using libinput_tablet_tool_sptr           = sptr<struct libinput_tablet_tool>;
    using libinput_paired_keyboard_sptr       = sptr<struct libinput_paired_keyboard>;
    using libinput_tablet_pad_mode_group_sptr = sptr<struct libinput_tablet_pad_mode_group>;
    using match_sptr                          = sptr<struct match_t>;
    using quirks_sptr                         = sptr<struct quirks_t>;
    using section_sptr                        = sptr<struct section_t>;
    using property_sptr                       = sptr<struct property_t>;
    using ud_device_sptr                      = sptr<struct ud_device_t>;
    using tp_dispatch_sptr                    = sptr<struct tp_device>;
    using event_source_sptr                   = sptr<struct event_source_t>;
    using pad_dispatch_sptr                   = sptr<struct pad_device>;
    using motion_filter_sptr                  = sptr<struct motion_filter>;
    using pad_led_group_sptr                  = sptr<struct pad_led_group>;
    using tablet_dispatch_sptr                = sptr<struct tablet_device>;
    using custom_accel_function_sptr          = sptr<struct custom_accel_function>;
    using pointer_delta_smoothener_sptr       = sptr<struct pointer_delta_smoothener>;

    using button_state_t = std::bitset<KEY_CNT>;
    using tablet_axes_bitset = std::bitset<lixx::libinput_tablet_tool_axis_cnt>;
    using input_prop = std::pair<ui32, bool>;

    struct input_event_t : ::input_event
    {
        netxs::time input_event_time() const
        {
            return netxs::time{} + std::chrono::seconds{ input_event_sec } + std::chrono::microseconds{ input_event_usec };
        }
    };
    struct abs_info_t : ::input_absinfo
    {
        constexpr operator bool ()                            const { return resolution != 0; }
        fp64 absinfo_range()                                  const { return (fp64)(maximum - minimum + 1); }
        fp64 absinfo_scale_axis(fp64 v, fp64 to_range)        const { return (v - minimum) * to_range / absinfo_range(); }
        si32 axis_range_percentage(fp64 percent)              const { return (maximum - minimum) * percent / 100.0 + minimum; }
        si32 invert_axis()                                    const { return maximum - (value - minimum); }
        fp64 absinfo_normalize_value(si32 v)                  const { return std::min(1.0, std::max(0.0, (fp64)(v - minimum) / (maximum - minimum))); }
        fp64 absinfo_normalize()                              const { return absinfo_normalize_value(value); }
        fp64 absinfo_convert_to_mm(fp64 v)                    const { return (v - minimum) / resolution; }
        si32 pressure_offset_to_absinfo(fp64 pressure_offset) const { return (maximum - minimum) * pressure_offset + minimum; }
        fp64 pressure_offset_from_absinfo(fp64 v)             const { return (v - minimum) / (maximum - minimum); }
    };
    struct matrix
    {
        fp32 val[3][3] = {}; // row/col

        void matrix_mult_vec(si32_coor& p) const
        {
            p.x = (si32)(p.x * val[0][0] + p.y * val[0][1] + val[0][2]);
            p.y = (si32)(p.x * val[1][0] + p.y * val[1][1] + val[1][2]);
        }
        void matrix_mult_vec_double(fp64_coor& p)
        {
            p.x = p.x * val[0][0] + p.y * val[0][1] + val[0][2];
            p.y = p.x * val[1][0] + p.y * val[1][1] + val[1][2];
        }
        bool matrix_is_identity() const
        {
            return (val[0][0] == 1
                 && val[0][1] == 0
                 && val[0][2] == 0
                 && val[1][0] == 0
                 && val[1][1] == 1
                 && val[1][2] == 0
                 && val[2][0] == 0
                 && val[2][1] == 0
                 && val[2][2] == 1);
        }
        void matrix_init_identity()
        {
            ::memset(val, 0, sizeof(val));
            val[0][0] = 1;
            val[1][1] = 1;
            val[2][2] = 1;
        }
        void matrix_from_farray6(std::array<fp32, 6> const& values)
        {
            matrix_init_identity();
            val[0][0] = values[0];
            val[0][1] = values[1];
            val[0][2] = values[2];
            val[1][0] = values[3];
            val[1][1] = values[4];
            val[1][2] = values[5];
        }
        void matrix_to_farray6(auto& values) const
        {
            values[0] = val[0][0];
            values[1] = val[0][1];
            values[2] = val[0][2];
            values[3] = val[1][0];
            values[4] = val[1][1];
            values[5] = val[1][2];
        }
        void matrix_init_translate(fp32 x, fp32 y)
        {
            matrix_init_identity();
            val[0][2] = x;
            val[1][2] = y;
        }
        void matrix_init_scale(fp32 sx, fp32 sy)
        {
            matrix_init_identity();
            val[0][0] = sx;
            val[1][1] = sy;
        }
        void matrix_mult(matrix const& m1, matrix const& m2)
        {
            auto m = matrix{}; // Allow for dest == m1 or dest == m2.
            for (auto row = 0; row < 3; row++)
            {
                for (auto col = 0; col < 3; col++)
                {
                    auto v = 0.0;
                    for (auto i = 0; i < 3; i++)
                    {
                        v += m1.val[row][i] * m2.val[i][col];
                    }
                    m.val[row][col] = v;
                }
            }
            ::memcpy(val, m.val, sizeof(m));
        }
        void matrix_init_rotate(si32 degrees)
        {
            auto s = std::sin(netxs::pi * degrees / 180.0);
            auto c = std::cos(netxs::pi * degrees / 180.0);
            matrix_init_identity();
            val[0][0] = c;
            val[0][1] = -s;
            val[1][0] = s;
            val[1][1] = c;
        }
    };
    struct libinput_config_accel
    {
        struct libinput_config_accel_custom_func
        {
            fp64              step{};
            std::vector<fp64> points;
        };
        struct custom_t
        {
            libinput_config_accel_custom_func fallback;
            libinput_config_accel_custom_func motion;
            libinput_config_accel_custom_func scroll;
        };

        libinput_config_accel_profile profile = {};
        custom_t                      custom{};
    };
    struct libinput_device_config_scroll_method
    {
        ui32                          (*get_methods)            (libinput_device_sptr li_device) = {};
        libinput_config_status        (*set_method)             (libinput_device_sptr li_device, libinput_config_scroll_method method) = {};
        libinput_config_scroll_method (*get_method)             (libinput_device_sptr li_device) = {};
        libinput_config_scroll_method (*get_default_method)     (libinput_device_sptr li_device) = {};
        libinput_config_status        (*set_button)             (libinput_device_sptr li_device, ui32 button) = {};
        ui32                          (*get_button)             (libinput_device_sptr li_device) = {};
        ui32                          (*get_default_button)     (libinput_device_sptr li_device) = {};
        libinput_config_status        (*set_button_lock)        (libinput_device_sptr li_device, bool scroll_button_lock_enabled) = {};
        bool                          (*get_button_lock)        (libinput_device_sptr li_device) = {};
        bool                          (*get_default_button_lock)(libinput_device_sptr li_device) = {};
    };
    struct libinput_device_config_natural_scroll
    {
        si32                  (*has)                (libinput_device_sptr li_device) = {};
        libinput_config_status(*set_enabled)        (libinput_device_sptr li_device, si32 enabled) = {};
        si32                  (*get_enabled)        (libinput_device_sptr li_device) = {};
        si32                  (*get_default_enabled)(libinput_device_sptr li_device) = {};
    };
    struct libinput_device_config_accel
    {
        si32                         (*available)          (libinput_device_sptr li_device) = {};
        libinput_config_status       (*set_speed)          (libinput_device_sptr li_device, fp64 speed) = {};
        fp64                         (*get_speed)          (libinput_device_sptr li_device) = {};
        fp64                         (*get_default_speed)  (libinput_device_sptr li_device) = {};
        ui32                         (*get_profiles)       (libinput_device_sptr li_device) = {};
        libinput_config_status       (*set_profile)        (libinput_device_sptr li_device, libinput_config_accel_profile) = {};
        libinput_config_accel_profile(*get_profile)        (libinput_device_sptr li_device) = {};
        libinput_config_accel_profile(*get_default_profile)(libinput_device_sptr li_device) = {};
        libinput_config_status       (*set_accel_config)   (libinput_device_sptr li_device, libinput_config_accel& accel_config) = {};
    };
    struct libinput_device_config_left_handed
    {
        si32                  (*has)        (libinput_device_sptr li_device) = {};
        libinput_config_status(*set)        (libinput_device_sptr li_device, si32 left_handed) = {};
        si32                  (*get)        (libinput_device_sptr li_device) = {};
        si32                  (*get_default)(libinput_device_sptr li_device) = {};
    };
    struct libinput_device_config_middle_emulation
    {
        si32                   (*available)  (libinput_device_sptr li_device) = {};
        libinput_config_status (*set)        (libinput_device_sptr li_device, bool middle_emulation_enabled) = {};
        bool                   (*get)        (libinput_device_sptr li_device) = {};
        bool                   (*get_default)(libinput_device_sptr li_device) = {};
    };
        struct libinput_device_config_calibration
        {
            si32                  (*has_matrix)        (libinput_device_sptr li_device) = {};
            libinput_config_status(*set_matrix)        (libinput_device_sptr li_device, std::array<fp32, 6> const& matrix) = {};
            si32                  (*get_matrix)        (libinput_device_sptr li_device, std::array<fp32, 6>& matrix) = {};
            si32                  (*get_default_matrix)(libinput_device_sptr li_device, std::array<fp32, 6>& matrix) = {};
        };
        struct libinput_device_config_tap
        {
            si32                           (*count)                       (libinput_device_sptr li_device) = {};
            libinput_config_status         (*set_enabled)                 (libinput_device_sptr li_device, bool tap_state_enabled) = {};
            bool                           (*get_enabled)                 (libinput_device_sptr li_device) = {};
            bool                           (*get_default)                 (libinput_device_sptr li_device) = {};
            libinput_config_status         (*set_map)                     (libinput_device_sptr li_device, bool use_lmr_map) = {};
            bool                           (*get_map)                     (libinput_device_sptr li_device) = {};
            bool                           (*get_default_map)             (libinput_device_sptr li_device) = {};
            libinput_config_status         (*set_drag_enabled)            (libinput_device_sptr li_device, bool drag_enbled) = {};
            bool                           (*get_drag_enabled)            (libinput_device_sptr li_device) = {};
            bool                           (*get_default_drag_enabled)    (libinput_device_sptr li_device) = {};
            libinput_config_status         (*set_draglock_enabled)        (libinput_device_sptr li_device, libinput_config_drag_lock_state) = {};
            libinput_config_drag_lock_state(*get_draglock_enabled)        (libinput_device_sptr li_device) = {};
            libinput_config_drag_lock_state(*get_default_draglock_enabled)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_area
        {
            si32                  (*has_rectangle)        (libinput_device_sptr li_device) = {};
            libinput_config_status(*set_rectangle)        (libinput_device_sptr li_device, fp64_rect rectangle) = {};
            fp64_rect             (*get_rectangle)        (libinput_device_sptr li_device) = {};
            fp64_rect             (*get_default_rectangle)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_click_method
        {
            ui32                         (*get_methods)                (libinput_device_sptr li_device) = {};
            libinput_config_status       (*set_method)                 (libinput_device_sptr li_device, libinput_config_click_method method) = {};
            libinput_config_click_method (*get_method)                 (libinput_device_sptr li_device) = {};
            libinput_config_click_method (*get_default_method)         (libinput_device_sptr li_device) = {};
            libinput_config_status       (*set_clickfinger_map)        (libinput_device_sptr li_device, bool use_lmr_map) = {};
            bool                         (*get_clickfinger_map)        (libinput_device_sptr li_device) = {};
            bool                         (*get_default_clickfinger_map)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_dwt
        {
            si32                   (*is_available)       (libinput_device_sptr li_device) = {};
            libinput_config_status (*set_enabled)        (libinput_device_sptr li_device, bool dwt_enabled) = {};
            bool                   (*get_enabled)        (libinput_device_sptr li_device) = {};
            bool                   (*get_default_enabled)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_dwtp
        {
            si32                   (*is_available)       (libinput_device_sptr li_device) = {};
            libinput_config_status (*set_enabled)        (libinput_device_sptr li_device, bool dwtp_enabled) = {};
            bool                   (*get_enabled)        (libinput_device_sptr li_device) = {};
            bool                   (*get_default_enabled)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_rotation
        {
            si32                  (*is_available)     (libinput_device_sptr li_device) = {};
            libinput_config_status(*set_angle)        (libinput_device_sptr li_device, ui32 degrees_cw) = {};
            ui32                  (*get_angle)        (libinput_device_sptr li_device) = {};
            ui32                  (*get_default_angle)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_gesture
        {
            libinput_config_status (*set_hold_enabled)(libinput_device_sptr li_device, bool hold_enabled) = {};
            bool                   (*get_hold_enabled)(libinput_device_sptr li_device) = {};
            bool                   (*get_hold_default)(libinput_device_sptr li_device) = {};
        };
        struct libinput_device_config_3fg_drag
        {
            si32                          (*count)      (libinput_device_sptr li_device) = {};
            libinput_config_status        (*set_enabled)(libinput_device_sptr li_device, libinput_config_3fg_drag_state enable) = {};
            libinput_config_3fg_drag_state(*get_enabled)(libinput_device_sptr li_device) = {};
            libinput_config_3fg_drag_state(*get_default)(libinput_device_sptr li_device) = {};

        };
        struct libinput_device_config
        {
            libinput_device_config_tap*              tap{};
            libinput_device_config_calibration*      calibration{};
            libinput_device_config_area*             area{};
            libinput_device_config_accel*            accel{};
            libinput_device_config_natural_scroll*   natural_scroll{};
            libinput_device_config_left_handed*      left_handed{};
            libinput_device_config_scroll_method*    scroll_method{};
            libinput_device_config_click_method*     click_method{};
            libinput_device_config_middle_emulation* middle_emulation{};
            libinput_device_config_dwt*              dwt{};
            libinput_device_config_dwtp*             dwtp{};
            libinput_device_config_rotation*         rotation{};
            libinput_device_config_gesture*          gesture{};
            libinput_device_config_3fg_drag*         drag_3fg{};
        };

    struct libinput_tablet_tool
    {
        struct pressure_t
        {
            fp64_range              range; // We're assuming that the *configured* pressure range is per tool, not per tablet. The *adjusted* thresholds are then per-tablet.
            fp64_range              wanted_range;
            bool                    has_configured_range{};
            ui32                    tablet_id{};
            abs_info_t              abs_info; // The configured axis we actually work with.
            si32_range              threshold; // In device coordinates.
            fp64                    threshold_offset{};
            bool                    threshold_has_offset{};
            pressure_heuristic_enum heuristic_state{}; // This gives us per-tablet heuristic state which is arguably wrong but >99% of users have one tablet and it's easier to implement it this way.

            fp64 normalize_pressure(si32 abs_value)
            {
                // Note: the upper threshold takes the offset into account so that
                //            |- 4% -|
                // min |------X------X-------------------------| max
                //            |      |
                //            |      + upper threshold / tip trigger
                //            +- offset and lower threshold
                //
                // The axis is scaled into the range [lower, max] so that the lower threshold is 0 pressure.
                auto abs = abs_info;
                abs.minimum = threshold.min;
                return abs.absinfo_normalize_value(abs_value);
            }
            void set_pressure_offset(fp64 offset_in_percent)
            {
                threshold_offset = offset_in_percent;
                threshold_has_offset = true;
                // Adjust the tresholds accordingly - we use the same gap (4% in device coordinates) between upper and lower as before which isn't technically correct (our range shrunk) but it's easy to calculate.
                auto units = abs_info.pressure_offset_to_absinfo(offset_in_percent);
                auto gap = threshold.max - threshold.min;
                threshold.min = units;
                threshold.max = units + gap;
            }
        };

        ui32                      serial{};
        ui32                      tool_id{};
        libinput_tablet_tool_type type{};
        tablet_axes_bitset        axis_caps_bits;
        std::bitset<KEY_MAX>      buttons_bits;
        pressure_t                pressure;

        si32 pressure_range_is_available()
        {
            return axis_caps_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE];
        }
        libinput_config_status pressure_range_set(fp64_range range)
        {
            if (range.min < 0.0 || range.min >= 1.0 || range.max <= 0.0 || range.max > 1.0 || range.max <= range.min)
            {
                return LIBINPUT_CONFIG_STATUS_INVALID;
            }
            else
            {
                pressure.wanted_range = range;
                pressure.has_configured_range = true;
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
        }
        void pressure_range_get(fp64_range& range)
        {
            range = pressure.wanted_range;
        }
        void pressure_range_get_default(fp64_range& range)
        {
            range.min = 0.0;
            range.max = 1.0;
        }
    };

            struct pointer_delta_smoothener
            {
                span threshold{};
                span value{};

                static pointer_delta_smoothener_sptr create(span event_delta_smooth_threshold, span event_delta_smooth_value)
                {
                    auto s_ptr = ptr::shared<pointer_delta_smoothener>();
                    s_ptr->threshold = event_delta_smooth_threshold;
                    s_ptr->value = event_delta_smooth_value;
                    return s_ptr;
                }
            };
            struct pointer_tracker
            {
                fp64_coor delta; // Delta to most recent event.
                time      now{}; // us.
                ui32      dir{};

                fp64 calculate_trackers_velocity(time new_now, pointer_delta_smoothener_sptr smoothener)
                {
                    auto tdelta = new_now - now + 1us;
                    if (smoothener && tdelta < smoothener->threshold)
                    {
                        tdelta = smoothener->value;
                    }
                    return delta.hypot() / datetime::round<fp64, std::chrono::microseconds>(tdelta); // Units/us.
                }
                fp64 trackers_velocity_after_timeout(pointer_delta_smoothener_sptr smoothener)
                {
                    // First movement after timeout needs special handling.
                    // When we trigger the timeout, the last event is too far in the past to use it for velocity calculation across multiple tracker values. Use the motion timeout itself to calculate the speed rather than the last tracker time. This errs on the side of being too fast for really slow movements but provides much more useful initial movement in normal use-cases (pause, move, pause, move).
                    return calculate_trackers_velocity(now + lixx::motion_timeout, smoothener);
                }
                static ui32 get_direction(fp64_coor p)
                {
                    auto dir = (ui32)UNDEFINED_DIRECTION;
                    if (std::abs(p.x) < 2.0 && std::abs(p.y) < 2.0)
                    {
                             if (p.x > 0.0 && p.y > 0.0) dir = S | SE | E;
                        else if (p.x > 0.0 && p.y < 0.0) dir = N | NE | E;
                        else if (p.x < 0.0 && p.y > 0.0) dir = S | SW | W;
                        else if (p.x < 0.0 && p.y < 0.0) dir = N | NW | W;
                        else if (p.x > 0.0)              dir = NE | E | SE;
                        else if (p.x < 0.0)              dir = NW | W | SW;
                        else if (p.y > 0.0)              dir = SE | S | SW;
                        else if (p.y < 0.0)              dir = NE | N | NW;
                    }
                    else
                    {
                        // Calculate r within the interval [0 to 8)
                        //   r = [0 .. 2π] where 0 is North
                        //   d_f = r / 2π  ([0 .. 1))
                        //   d_8 = 8 * d_f
                        auto r = std::atan2(p.y, p.x);
                        r = std::fmod(r + 2.5 * netxs::pi, 2 * netxs::pi);
                        r *= 4 * M_1_PI;
                        // Mark one or two close enough octants.
                        auto d1 = (si32)(r + 0.9) % 8;
                        auto d2 = (si32)(r + 0.1) % 8;
                        dir = (1ul << d1) | (1ul << d2);
                    }
                    return dir;
                }
            };
        struct pointer_trackers
        {
            std::vector<pointer_tracker>  trackers;
            ui32                          cur_tracker{};
            pointer_delta_smoothener_sptr smoothener;

            pointer_tracker& trackers_by_offset(ui32 offset = 0)
            {
                auto index = (ui32)((cur_tracker + trackers.size() - offset) % trackers.size());
                return trackers[index];
            }
            void trackers_reset(time now)
            {
                for (auto offset = 1u; offset < trackers.size(); offset++)
                {
                    auto& tracker = trackers_by_offset(offset);
                    tracker.now = {};
                    tracker.dir = 0;
                    tracker.delta.x = 0;
                    tracker.delta.y = 0;
                }
                auto& tracker = trackers_by_offset();
                tracker.now = now;
                tracker.dir = UNDEFINED_DIRECTION;
            }
            void trackers_init(si32 ntrackers)
            {
                trackers.resize(ntrackers);
                cur_tracker = 0;
                smoothener  = nullptr;
            }
            void trackers_feed(fp64_coor delta, time now)
            {
                assert(!trackers.empty());
                for (auto& ts : trackers)
                {
                    ts.delta += delta;
                }
                cur_tracker = (cur_tracker + 1) % trackers.size();
                trackers[cur_tracker].delta = {};
                trackers[cur_tracker].now   = now;
                trackers[cur_tracker].dir   = pointer_tracker::get_direction(delta);
            }
            fp64 trackers_velocity(time now)
            {
                auto result = 0.0;
                auto initial_velocity = 0.0;
                auto dir = trackers_by_offset(0).dir;
                for (auto offset = 1u; offset < trackers.size(); offset++) // Find least recent vector within a timelimit, maximum velocity diff and direction threshold.
                {
                    auto& tracker = trackers_by_offset(offset);
                    if (tracker.now > now) break; // Bug: time running backwards.
                    if (now - tracker.now > lixx::motion_timeout) // Stop if too far away in time.
                    {
                        if (offset == 1u)
                        {
                            result = tracker.trackers_velocity_after_timeout(smoothener);
                        }
                        break;
                    }
                    auto velocity = tracker.calculate_trackers_velocity(now, smoothener);
                    dir &= tracker.dir; // Stop if direction changed.
                    if (dir == 0)
                    {
                        if (offset == 1) // First movement after dirchange - velocity is that of the last movement.
                        {
                            result = velocity;
                        }
                        break;
                    }
                    if (initial_velocity == 0.0 || offset <= 2) // Always average the first two events. On some touchpads where the first event is jumpy, this somewhat reduces pointer jumps on slow motions.
                    {
                        result = initial_velocity = velocity;
                    }
                    else // Stop if velocity differs too much from initial.
                    {
                        auto velocity_diff = std::abs(initial_velocity - velocity);
                        if (velocity_diff > lixx::max_velocity_diff) break;
                        result = velocity;
                    }
                }
                return result; // Units/us.
            }
        };
    struct motion_filter : ptr::enable_shared_from_this<motion_filter>
    {
        si32                          dpi;
        fp64                          velocity;      // Units/us.
        fp64                          last_velocity; // Units/us.
        pointer_trackers              trackers;
        libinput_config_accel_profile type{};
        fp64                          speed_adjustment{}; // Normalized [-1, 1].

        motion_filter(si32 dpi = 1)
            :         dpi{ dpi },
                 velocity{ 0.0 },
            last_velocity{ 0.0 }
        { }
        virtual ~motion_filter()
        { }

        virtual fp64      apply_acceleration(fp64 velocity)                                      { return velocity; }
        virtual fp64_coor filter(         [[maybe_unused]] fp64_coor unaccelerated, void*, time) { return {}; }
        virtual fp64_coor filter_constant([[maybe_unused]] fp64_coor unaccelerated, time)        { return {}; }
        virtual fp64_coor filter_scroll(  [[maybe_unused]] fp64_coor unaccelerated, time)        { return {}; }
        virtual bool      set_speed([[maybe_unused]] fp64 speed_adjustment)                      { return {}; }
        virtual bool      set_accel_config([[maybe_unused]] libinput_config_accel& accel_config) { return {}; }
        virtual void      restart(time now)                                                      { trackers.trackers_reset(now); }

        bool filter_set_speed(fp64 speed_adjustment)
        {
            return set_speed(speed_adjustment);
        }
        fp64 filter_get_speed()
        {
            return speed_adjustment;
        }
        libinput_config_accel_profile filter_get_type()
        {
            return type;
        }
        bool filter_set_accel_config(libinput_config_accel& accel_config)
        {
            assert(type == accel_config.profile);
            return set_accel_config(accel_config);
        }
        fp64_coor filter_dispatch_scroll(fp64_coor unaccelerated, time stamp)
        {
            return filter_scroll(unaccelerated, stamp);
        }
        fp64_coor filter_dispatch(fp64_coor unaccelerated, void* data, time now)
        {
            return filter(unaccelerated, data, now);
        }
        fp64 calculate_acceleration_simpsons(fp64_coor unaccelerated, time now)
        {
            trackers.trackers_feed(unaccelerated, now);
            velocity = trackers.trackers_velocity(now);
            // Use Simpson's rule to calculate the average acceleration between the previous motion and the most recent.
            auto factor = apply_acceleration(velocity)
                        + apply_acceleration(last_velocity)
                        + apply_acceleration((last_velocity + velocity) / 2.0) * 4.0;
            last_velocity = velocity;
            factor /= 6.0;
            return factor; // Unitless factor.
        }
        fp64_coor normalize_for_dpi(fp64_coor coor)
        {
            return coor * lixx::default_mouse_dpi / dpi;
        }
    };
    struct trackpoint_accelerator : motion_filter
    {
        fp64 multiplier;
        fp64 speed_factor;

        struct trackpoint_accelerator_impl_t
        {
            trackpoint_accelerator& accel;
                fp64 trackpoint_accel_profile(fp64 velocity)
                {
                    velocity = v_us2ms(velocity); // Make it units/ms.
                    // Just a nice-enough curve that provides fluid factor conversion from the minimum speed up to the real maximum.
                    // Generated by https://www.mycurvefit.com/ with input data
                    //    0    0.3
                    //    0.1  1
                    //    0.4  3
                    //    0.6  4
                    auto factor = 10.06254 + (0.3 - 10.06254) / (1 + pow(velocity / 0.9205459, 1.15363));
                    factor *= accel.speed_factor;
                    return factor;
                }
            fp64_coor trackpoint_accelerator_filter(fp64_coor unaccelerated, time now)
            {
                auto multiplied = unaccelerated * accel.multiplier;
                accel.trackers.trackers_feed(multiplied, now);
                auto velocity = accel.trackers.trackers_velocity(now);
                auto f = trackpoint_accel_profile(velocity);
                auto coords = multiplied * f;
                return coords;
            }
            fp64_coor trackpoint_accelerator_filter_noop(fp64_coor unaccelerated)
            {
                return unaccelerated * accel.multiplier;
            }
                fp64 speed_factor1(fp64 s)
                {
                    // Maps the [-1, 1] speed setting into a constant acceleration
                    // range. This isn't a linear scale, we keep 0 as the 'optimized'
                    // mid-point and scale down to 0 for setting -1 and up to 5 for
                    // setting 1. On the premise that if you want a faster cursor, it
                    // doesn't matter as much whether you have 0.56789 or 0.56790,
                    // but for lower settings it does because you may lose movements.
                    // *shrug*.
                    //
                    // Magic numbers calculated by MyCurveFit.com, data points were
                    //  0.0 0.0
                    //  0.1 0.1 (because we need 4 points)
                    //  1   1
                    //  2   5
                    //
                    //  This curve fits nicely into the range necessary.
                    s += 1; // Map to [0, 2].
                    return 435837.2 + (0.04762636 - 435837.2) / (1.0 + pow(s / 240.4549, 2.377168));
                }
            bool trackpoint_accelerator_set_speed(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                accel.speed_adjustment = speed_adjustment;
                accel.speed_factor = speed_factor1(speed_adjustment);
                return true;
            }
        };

        trackpoint_accelerator(fp64 multiplier, bool use_velocity_averaging)
            :   multiplier{ multiplier },
              speed_factor{ 0.0 }
        {
            // Trackpoints are special. They don't have a movement speed like a
            // mouse or a finger, instead they send a stream of events based on
            // the pressure applied.
            //
            // Physical ranges on a trackpoint are the max values for relative
            // deltas, but these are highly device-specific and unreliable to
            // measure.
            //
            // Instead, we just have a constant multiplier we have in the quirks system.
            assert(multiplier > 0.0);
            type         = LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE;
            trackers.trackers_init(use_velocity_averaging ? 16 : 2);
            trackers.smoothener = pointer_delta_smoothener::create(10ms, 10ms);
        }

        trackpoint_accelerator_impl_t impl{ *this };
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time now) { return impl.trackpoint_accelerator_filter(unaccelerated, now); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)            { return impl.trackpoint_accelerator_filter_noop(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)            { return impl.trackpoint_accelerator_filter_noop(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                          { return impl.trackpoint_accelerator_set_speed(speed_adjustment); }
    };
    struct pointer_accelerator_flat : motion_filter
    {
        fp64 factor;

        struct pointer_accelerator_flat_impl_t
        {
            pointer_accelerator_flat& accel;
            fp64_coor accelerator_filter_flat(fp64_coor unaccelerated)
            {
                return unaccelerated * accel.factor;
            }
            fp64_coor accelerator_filter_noop_flat(fp64_coor unaccelerated)
            {
                // We map the unaccelerated flat filter to have the same behavior as the "accelerated" flat filter.
                // The filter by definition is flat, i.e. it does not actually
                // apply any acceleration (merely a constant factor) and we can assume
                // that a user wants all mouse movement to have the same speed, mapped 1:1 to the input speed.
                // Thus we apply the same factor to our non-accelerated motion - this way
                // things like button scrolling end up having the same movement as pointer motion.
                return accelerator_filter_flat(unaccelerated);
            }
            bool accelerator_set_speed_flat(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                // Speed range is 0-200% of the nominal speed, with 0 mapping to the nominal speed. Anything above 200 is pointless, we're already skipping over ever second pixel at 200% speed.
                accel.factor = std::max(0.005, 1 + speed_adjustment);
                accel.speed_adjustment = speed_adjustment;
                return true;
            }
        };

        pointer_accelerator_flat(si32 dpi = 0)
            : motion_filter{ dpi },
                     factor{ 0.0 }
        {
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT;
        }

        pointer_accelerator_flat_impl_t impl{ *this };
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time) { return impl.accelerator_filter_flat(unaccelerated); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)        { return impl.accelerator_filter_noop_flat(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)        { return impl.accelerator_filter_noop_flat(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                      { return impl.accelerator_set_speed_flat(speed_adjustment); }
    };
    struct trackpoint_flat_accelerator : pointer_accelerator_flat
    {
        fp64 multiplier;

        trackpoint_flat_accelerator(fp64 multiplier)
            : multiplier{ multiplier }
        {
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT;
        }
    };
    struct tablet_accelerator_flat : motion_filter
    {
        fp64      factor;
        si32_coor resolution;
        fp64_coor resolution_scale; // 1000dpi : tablet res.

        struct tablet_accelerator_flat_impl_t
        {
            tablet_accelerator_flat& accel;
                fp64_coor tablet_accelerator_filter_flat_mouse(fp64_coor units)
                {
                    // Tablets are high res (Intuos 4 is 5080 dpi) and unmodified deltas are way too high.
                    // Slow it down to the equivalent of a 1000dpi mouse. The ratio of that is:
                    // ratio = 1000 / (resolution_per_mm * 25.4) i.e. on the Intuos4 it's a ratio of ~1/5.
                    return units * accel.resolution_scale * accel.factor;
                }
                fp64_coor tablet_accelerator_filter_flat_pen(fp64_coor units)
                {
                    // Tablet input is in device units, output is supposed to be in
                    // logical pixels roughly equivalent to a mouse/touchpad.
                    // This is a magical constant found by trial and error. On a 96dpi
                    // screen 0.4mm of movement correspond to 1px logical pixel which
                    // is almost identical to the tablet mapped to screen in absolute
                    // mode. Tested on a Intuos5, other tablets may vary.
                    static constexpr auto dpi_conversion = 96.0 / 25.4 * 2.5; // Unitless factor.
                    auto mm = units / accel.resolution;
                    auto accelerated = mm * accel.factor * dpi_conversion;
                    return accelerated;
                }
            fp64_coor tablet_accelerator_filter_flat(fp64_coor units, void* data)
            {
                auto tool = (libinput_tablet_tool*)data;
                auto type = tool->type;
                auto is_mouse = type == LIBINPUT_TABLET_TOOL_TYPE_MOUSE || type == LIBINPUT_TABLET_TOOL_TYPE_LENS;
                auto accelerated = is_mouse ? tablet_accelerator_filter_flat_mouse(units)
                                            : tablet_accelerator_filter_flat_pen(units);
                return accelerated;
            }
            bool tablet_accelerator_set_speed(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                accel.factor = speed_adjustment + 1.0;
                return true;
            }
        };

        tablet_accelerator_flat(si32_coor resolution)
            :         factor{ 1.0 },
                  resolution{ resolution },
            resolution_scale{ lixx::default_mouse_dpi / (25.4 * resolution.x), lixx::default_mouse_dpi / (25.4 * resolution.y) }
        {
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT;
        }

        tablet_accelerator_flat_impl_t impl{ *this };
        virtual fp64_coor filter(fp64_coor unaccelerated, void* data, time) { return impl.tablet_accelerator_filter_flat(unaccelerated, data); }
        virtual bool      set_speed(fp64 speed_adjustment)                  { return impl.tablet_accelerator_set_speed(speed_adjustment); }
    };
    struct custom_accel_function
    {
        time              last_time{};
        fp64              step{};
        std::vector<fp64> points;

        fp64 custom_accel_function_calculate_speed(fp64_coor unaccelerated, time now)
        {
            // Although most devices have a constant polling rate, and for fast
            // movements these distances do represent the actual speed,
            // for slow movements it is not the case.
            //
            // Since all devices have a finite resolution, real world events
            // for a slow smooth movement could look like:
            //   Event 1 - (0,  1) - time 0
            //   Event 2 - (0,  0) - time 7  - filtered (zero event)
            //   Event 3 - (1,  0) - time 14
            //   Event 4 - (0,  0) - time 21 - filtered (zero event)
            //   Event 5 - (0,  0) - time 28 - filtered (zero event)
            //   Event 6 - (0,  1) - time 35
            //
            // Not taking the time into account would mean interpreting those events as:
            //   Move 1 unit over 7 ms
            //   Pause for 7 ms
            //   Move 1 unit over 7 ms
            //   Pause for 14 ms
            //   Move 1 unit over 7ms
            //
            // Where in reality this was one smooth movement without pauses,
            // so after normalizing for time we get:
            //   Move 1 unit over 7 ms
            //   Move 1 unit over 14 ms
            //   Move 1 unit over 21ms
            //
            // which should give us better speed estimation.
            auto distance = unaccelerated.hypot(); // Calculate speed based on time passed since last event.
            if (now - last_time > lixx::motion_timeout) // Handle first event in a motion.
            {
                last_time = now - lixx::first_motion_time_interval;
            }
            auto dt = datetime::round<fp64, std::chrono::milliseconds>(now - last_time);
            auto speed = distance / dt; // Speed is in device-units per ms.
            last_time = now;
            return speed;
        }
        fp64 custom_accel_function_profile(fp64 speed_in)
        {
            auto i = (ui64)(speed_in / step); // Calculate the index of the first point used for interpolation.
            i = std::min<ui64>(i, points.size() - 2); // If speed is greater than custom curve's max speed, use last 2 points for linear extrapolation (same calculation as linear interpolation).
            auto x0 = step * i; // The 2 points used for linear interpolation.
            auto x1 = step * (i + 1);
            auto y0 = points[i];
            auto y1 = points[i + 1];
            auto speed_out = (y0 * (x1 - speed_in) + y1 * (speed_in - x0)) / step; // Linear interpolation.
            // We moved (dx, dy) device units within the last N ms. This gives us a
            // given speed S in units/ms, that's our accel input. Our curve says map
            // that speed S to some other speed S'.
            //
            // Our device delta is represented by the vector, that vector needs to
            // be modified to represent our intended speed.
            //
            // Example: we moved a delta of 7 over the last 7ms. Our speed is
            // thus 1 u/ms, our out speed is 2 u/ms because we want to double our
            // speed (points: [0.0, 2.0]). Our delta must thus be 14 - factor of 2,
            // or out-speed/in-speed.
            //
            // Example: we moved a delta of 1 over the last 7ms. Our input speed is
            // 1/7 u/ms, our out speed is 1/7 ms because we set up a flat accel
            // curve (points: [0.0, 1.0]). Our delta must thus be 1 - factor of 1,
            // or out-speed/in-speed.
            //
            // Example: we moved a delta of 1 over the last 21ms. Our input speed is
            // 1/21 u/ms, our out speed is 1 u/ms because we set up a fixed-speed
            // curve (points: [1.0, 1.0]). Our delta must thus be 21 - factor of 21,
            // or out-speed/in-speed.
            //
            // Example: we moved a delta of 21 over the last 7ms. Our input speed is
            // 3 u/ms, our out speed is 1 u/ms because we set up a fixed-speed
            // curve (points: [1.0, 1.0]). Our delta must thus be 7 - factor of 1/3,
            // or out-speed/in-speed.
            auto accel_factor = speed_out / speed_in; // Calculate the acceleration factor based on the user desired speed out.
            return accel_factor;
        }
        fp64_coor custom_accel_function_filter(fp64_coor unaccelerated, time now)
        {
            auto speed = custom_accel_function_calculate_speed(unaccelerated, now);
            auto accel_factor = custom_accel_function_profile(speed);
            auto accelerated = unaccelerated * accel_factor;
            return accelerated;
        }
        static custom_accel_function_sptr create_custom_accel_function(fp64 step, auto const& points)
        {
            auto npoints = points.size();
            if (npoints < lixx::libinput_accel_npoints_min || npoints > lixx::libinput_accel_npoints_max
             || step <= 0 || step > lixx::libinput_accel_step_max)
            {
                return {};
            }
            for (auto p : points)
            {
                if (p < lixx::libinput_accel_point_min_value || p > lixx::libinput_accel_point_max_value)
                {
                    return {};
                }
            }
            auto cf = ptr::shared<custom_accel_function>();
            cf->points.assign(points.begin(), points.end());
            cf->step = step;
            return cf;
        }
    };
    struct custom_accelerator : motion_filter
    {
        struct funcs_t
        {
            custom_accel_function_sptr fallback;
            custom_accel_function_sptr motion;
            custom_accel_function_sptr scroll;
        };
        funcs_t funcs;

        struct custom_accelerator_impl_t
        {
            custom_accelerator& accel;
                fp64_coor custom_accelerator_filter(libinput_config_accel_type accel_type, fp64_coor unaccelerated, time now)
                {
                    auto cf = accel.funcs.motion && accel_type == LIBINPUT_ACCEL_TYPE_MOTION ? accel.funcs.motion
                            : accel.funcs.scroll && accel_type == LIBINPUT_ACCEL_TYPE_SCROLL ? accel.funcs.scroll
                                                                                             : accel.funcs.fallback;
                    return cf->custom_accel_function_filter(unaccelerated, now);
                }
            fp64_coor custom_accelerator_filter_motion(fp64_coor unaccelerated, time now)
            {
                return custom_accelerator_filter(LIBINPUT_ACCEL_TYPE_MOTION, unaccelerated, now);
            }
            fp64_coor custom_accelerator_filter_fallback(fp64_coor unaccelerated, time now)
            {
                return custom_accelerator_filter(LIBINPUT_ACCEL_TYPE_FALLBACK, unaccelerated, now);
            }
            fp64_coor custom_accelerator_filter_scroll(fp64_coor unaccelerated, time now)
            {
                return custom_accelerator_filter(LIBINPUT_ACCEL_TYPE_SCROLL, unaccelerated, now);
            }
            bool custom_accelerator_set_speed([[maybe_unused]] fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                return true; // Noop, this function has no effect in the custom interface.
            }
            bool custom_accelerator_set_accel_config(libinput_config_accel& config)
            {
                auto fallback = custom_accel_function_sptr{};
                auto motion   = custom_accel_function_sptr{};
                auto scroll   = custom_accel_function_sptr{};
                auto ok = true;
                if (ok && config.custom.fallback.step)
                {
                    fallback = custom_accel_function::create_custom_accel_function(config.custom.fallback.step, config.custom.fallback.points);
                    ok = !!fallback;
                }
                if (ok && config.custom.motion.step)
                {
                    motion = custom_accel_function::create_custom_accel_function(config.custom.motion.step, config.custom.motion.points);
                    ok = !!motion;
                }
                if (ok && config.custom.scroll.step)
                {
                    scroll = custom_accel_function::create_custom_accel_function(config.custom.scroll.step, config.custom.scroll.points);
                    ok = !!scroll;
                }
                if (ok)
                {
                    accel.funcs.fallback = fallback;
                    accel.funcs.motion   = motion;
                    accel.funcs.scroll   = scroll;
                }
                return ok;
            }
        };

        custom_accelerator()
        {
            // The unit function by default, speed in = speed out, i.e. no acceleration.
            static constexpr auto default_step = 1.0;
            static constexpr auto default_points = std::to_array({ 0.0, 1.0 });
            // Initialize default acceleration, used as fallback.
            type           = LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM;
            funcs.fallback = custom_accel_function::create_custom_accel_function(default_step, default_points);
            // Don't initialize other acceleration functions. Those will be initialized if the user sets their points, otherwise the fallback acceleration function is used.
        }

        custom_accelerator_impl_t impl{ *this };
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time now) { return impl.custom_accelerator_filter_motion(unaccelerated, now); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time now)        { return impl.custom_accelerator_filter_fallback(unaccelerated, now); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time now)        { return impl.custom_accelerator_filter_scroll(unaccelerated, now); }
        virtual bool      set_speed(fp64 speed_adjustment)                          { return impl.custom_accelerator_set_speed(speed_adjustment); }
        virtual bool      set_accel_config(libinput_config_accel& accel_config)     { return impl.custom_accelerator_set_accel_config(accel_config); }
        virtual void      restart(time)                                             { } // Noop, this function has no effect in the custom interface.
    };
    struct pointer_accelerator : motion_filter
    {
        fp64 threshold; // 1000dpi units/us.
        fp64 accel;     // Unitless factor.
        fp64 incline;   // Incline of the function.

        struct pointer_accelerator_impl_t
        {
            pointer_accelerator& accel;
            fp64_coor accelerator_filter_linear(fp64_coor unaccelerated, time stamp)
            {
                return accel.normalize_for_dpi(unaccelerated) * accel.calculate_acceleration_simpsons(unaccelerated, stamp);
            }
            fp64_coor accelerator_filter_noop(fp64_coor unaccelerated)
            {
                return accel.normalize_for_dpi(unaccelerated);
            }
            bool accelerator_set_speed(fp64 new_speed_adjustment)
            {
                assert(new_speed_adjustment >= -1.0 && new_speed_adjustment <= 1.0);
                // Note: the numbers below are nothing but trial-and-error magic, don't read more into them other than "they mostly worked ok" delay when accel kicks in.
                accel.threshold = lixx::default_acceleration_threshold - datetime::round<fp64, std::chrono::microseconds>(250us) * new_speed_adjustment;
                if (accel.threshold < lixx::minimum_acceleration_threshold)
                {
                    accel.threshold = lixx::minimum_acceleration_threshold;
                }
                accel.accel = lixx::default_acceleration + new_speed_adjustment * 1.5; // Adjust max accel factor.
                accel.incline = lixx::default_incline + new_speed_adjustment * 0.75; // Higher speed -> faster to reach max.
                accel.speed_adjustment = new_speed_adjustment;
                return true;
            }
            fp64 pointer_accel_profile_linear(fp64 speed_in/* in normalized units */)
            {
                auto max_accel = accel.accel; // Unitless factor.
                auto threshold = accel.threshold; // 1000dpi units/us.
                auto incline = accel.incline;
                auto factor = fp64{}; // Unitless.
                // Our acceleration function calculates a factor to accelerate input deltas with.
                // The function is a double incline with a plateau, with a rough shape like this:
                //     accel
                //     factor
                //     ^
                //     |        /
                //     |  _____/
                //     | /
                //     |/
                //     +-------------> speed in
                // The two inclines are linear functions in the form
                //     y = ax + b
                //     where y is speed_out
                //           x is speed_in
                //           a is the incline of acceleration
                //           b is minimum acceleration factor
                // for speeds up to 0.07 u/ms, we decelerate, down to 30% of input speed.
                //     hence 1 = a * 0.07 + 0.3
                //         0.7 = a * 0.07 => a := 10
                //     deceleration function is thus:
                //         y = 10x + 0.3
                //  Note:
                //  -  0.07 u/ms as threshold is a result of trial-and-error and has no other intrinsic meaning.
                //  -  0.3 is chosen simply because it is above the Nyquist frequency for subpixel motion within a pixel.
                if (v_us2ms(speed_in) < 0.07)
                {
                    factor = 10 * v_us2ms(speed_in) + 0.3;
                    // Up to the threshold, we keep factor 1, i.e. 1:1 movement.
                }
                else if (speed_in < threshold)
                {
                    factor = 1;
                }
                else
                {
                    // Acceleration function above the threshold:
                    //   y = ax' + b
                    //   where T is threshold
                    //       x is speed_in
                    //       x' is speed
                    //       and
                    //       y(T) == 1
                    //   hence 1 = ax' + 1
                    //       => x' := (x - T)
                    factor = incline * v_us2ms(speed_in - threshold) + 1;
                }
                factor = std::min(max_accel, factor); // Cap at the maximum acceleration factor.
                return factor;
            }
        };

        pointer_accelerator(si32 dpi, bool use_velocity_averaging)
            : motion_filter{ dpi },
                  threshold{ lixx::default_acceleration_threshold },
                      accel{ lixx::default_acceleration },
                    incline{ lixx::default_incline }
        {
            trackers.trackers_init(use_velocity_averaging ? 16 : 2);
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE;
        }

        pointer_accelerator_impl_t impl{ *this };
        virtual fp64      apply_acceleration(fp64 velocity)                         { return impl.pointer_accel_profile_linear(velocity); }
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time now) { return impl.accelerator_filter_linear(unaccelerated, now); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)            { return impl.accelerator_filter_noop(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)            { return impl.accelerator_filter_noop(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                          { return impl.accelerator_set_speed(speed_adjustment); }
    };
    struct pointer_accelerator_low_dpi : pointer_accelerator
    {
        struct pointer_accelerator_low_dpi_impl_t
        {
            pointer_accelerator_low_dpi& accel;
            fp64 pointer_accel_profile_linear_low_dpi(fp64 speed_in/* in device units (units/us) */)
            {
                // Custom acceleration function for mice < 1000dpi.
                // At slow motion, a single device unit causes a one-pixel movement.
                // The threshold/max accel depends on the DPI, the smaller the DPI the
                // earlier we accelerate and the higher the maximum acceleration is. Result:
                // at low speeds we get pixel-precision, at high speeds we get approx. the
                // same movement as a high-dpi mouse.
                // Note: data fed to this function is in device units, not normalized.
                auto max_accel = accel.accel; // Unitless factor.
                auto threshold = accel.threshold; // Units/us.
                auto incline = accel.incline;
                auto dpi_factor = accel.dpi / (fp64)lixx::default_mouse_dpi;
                auto factor = fp64{}; // Unitless.
                // The dpi_factor is always < 1.0, increase max_accel, reduce the threshold so it kicks in earlier.
                max_accel /= dpi_factor;
                threshold *= dpi_factor;
                // See pointer_accel_profile_linear for a long description.
                     if (v_us2ms(speed_in) < 0.07) factor = 10 * v_us2ms(speed_in) + 0.3;
                else if (speed_in < threshold)     factor = 1;
                else                               factor = incline * v_us2ms(speed_in - threshold) + 1;
                factor = std::min(max_accel, factor);
                return factor;
            }
            fp64_coor accelerator_filter_low_dpi(fp64_coor unaccelerated, time now)
            {
                return unaccelerated * accel.calculate_acceleration_simpsons(unaccelerated, now);
            }
        };

        pointer_accelerator_low_dpi(si32 dpi, bool use_velocity_averaging)
            : pointer_accelerator{ dpi, use_velocity_averaging }
        { }

        pointer_accelerator_low_dpi_impl_t impl_low{ *this };
        virtual fp64      apply_acceleration(fp64 velocity)                { return impl_low.pointer_accel_profile_linear_low_dpi(velocity); }
        virtual fp64_coor filter(fp64_coor unaccelerated, void*, time now) { return impl_low.accelerator_filter_low_dpi(unaccelerated, now); }
    };
    struct touchpad_accelerator_flat : motion_filter
    {
        fp64 factor;

        struct touchpad_accelerator_flat_impl_t
        {
            touchpad_accelerator_flat& accel;
                fp64_coor accelerator_filter_touchpad_flat(fp64_coor unaccelerated)
                {
                    return accel.normalize_for_dpi(unaccelerated) * lixx::tp_magic_slowdown_flat * accel.factor; // You want flat acceleration, you get flat acceleration for the device.
                }
            fp64_coor accelerator_filter_noop_touchpad_flat(fp64_coor unaccelerated)
            {
                // We map the unaccelerated flat filter to have the same behavior as
                // the "accelerated" flat filter.
                // The filter by definition is flat, i.e. it does not actually
                // apply any acceleration (merely a constant factor) and we can assume
                // that a user wants all mouse movement to have the same speed, mapped
                // 1:1 to the input speed.
                //
                // Thus we apply the same factor to our non-accelerated motion - this way
                // things like gestures end up having the same movement as
                // pointer motion.
                return accelerator_filter_touchpad_flat(unaccelerated);
            }
            bool accelerator_set_speed_touchpad_flat(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                accel.factor = std::max(0.005, 1 + speed_adjustment);
                accel.speed_adjustment = speed_adjustment;
                return true;
            }
        };

        touchpad_accelerator_flat(si32 dpi)
            : motion_filter{ dpi },
                     factor{ 0.0 }
        {
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT;
        }

        touchpad_accelerator_flat_impl_t impl{ *this };
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time) { return impl.accelerator_filter_touchpad_flat(unaccelerated); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)        { return impl.accelerator_filter_noop_touchpad_flat(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)        { return impl.accelerator_filter_noop_touchpad_flat(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                      { return impl.accelerator_set_speed_touchpad_flat(speed_adjustment); }
    };
    struct pointer_accelerator_x230 : motion_filter
    {
        fp64 threshold; // Units/us.
        fp64 accel;     // Unitless factor.
        fp64 incline;   // Incline of the function.

        struct pointer_accelerator_x230_impl_t
        {
            pointer_accelerator_x230& accel;
            fp64_coor accelerator_filter_x230(fp64_coor raw, time stamp)
            {
                // This filter is a "do not touch me" filter. So the hack here is
                // just to replicate the old behavior before filters switched to
                // device-native dpi:
                // 1) Convert from device-native to 1000dpi normalized.
                // 2) Run all calculation on 1000dpi-normalized data.
                // 3) Apply accel factor no normalized data.
                auto unaccelerated = accel.normalize_for_dpi(raw);
                auto accelerated = unaccelerated * accel.calculate_acceleration_simpsons(unaccelerated, stamp);
                return accelerated;
            }
            fp64_coor accelerator_filter_constant_x230(fp64_coor unaccelerated)
            {
                const auto factor = lixx::x230_magic_slowdown / lixx::x230_tp_magic_low_res_factor;
                return accel.normalize_for_dpi(unaccelerated) * factor;
            }
            void accelerator_x230_restart(time stamp)
            {
                accel.trackers.trackers_reset(stamp);
            }
            bool accelerator_set_speed_x230(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                // Note: the numbers below are nothing but trial-and-error magic, don't read more into them other than "they mostly worked ok" delay when accel kicks in.
                accel.threshold = lixx::default_acceleration_threshold - datetime::round<fp64, std::chrono::microseconds>(250us) * speed_adjustment;
                if (accel.threshold < lixx::minimum_acceleration_threshold) accel.threshold = lixx::minimum_acceleration_threshold;
                // Adjust max accel factor.
                accel.accel = lixx::default_acceleration + speed_adjustment * 1.5;
                // Higher speed -> faster to reach max.
                accel.incline = lixx::default_incline + speed_adjustment * 0.75;
                accel.speed_adjustment = speed_adjustment;
                return true;
            }
            fp64 touchpad_lenovo_x230_accel_profile(fp64 speed_in/* 1000dpi-units/µs */)
            {
                // Those touchpads presents an actual lower resolution that what is advertised. We see some jumps from the cursor due to the big steps in X and Y when we are receiving data.
                // Apply a factor to minimize those jumps at low speed, and try keeping the same feeling as regular touchpads at high speed. It still feels slower but it is usable at least.
                auto max_accel = accel.accel * lixx::x230_tp_magic_low_res_factor; // Unitless factor.
                auto threshold = accel.threshold / lixx::x230_tp_magic_low_res_factor; // Units/us.
                auto incline   = accel.incline * lixx::x230_tp_magic_low_res_factor;
                // Note: the magic values in this function are obtained by trial-and-error. No other meaning should be interpreted.
                // The calculation is a compressed form of pointer_accel_profile_linear(), look at the git history of that function for an explanation of what the min/max/etc. does.
                speed_in *= lixx::x230_magic_slowdown / lixx::x230_tp_magic_low_res_factor;
                auto f1 = std::min(1.0, v_us2ms(speed_in) * 5); // Unitless.
                auto f2 = 1 + (v_us2ms(speed_in) - v_us2ms(threshold)) * incline;
                auto factor = std::min(max_accel, f2 > 1 ? f2 : f1); // Unitless.
                return factor * lixx::x230_magic_slowdown / lixx::x230_tp_magic_low_res_factor;
            }
        };

        pointer_accelerator_x230(si32 dpi, bool use_velocity_averaging)
            : motion_filter{ dpi },
                  threshold{ lixx::x230_threshold },
                      accel{ lixx::x230_acceleration }, // Unitless factor.
                    incline{ lixx::x230_incline } // Incline of the acceleration function.
        {
            // The Lenovo x230 has a bad touchpad. This accel method has been trial-and-error'd, any changes to it will require re-testing everything.
            // Don't touch this.
            trackers.trackers_init(use_velocity_averaging ? 16 : 2);
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE;
        }

        pointer_accelerator_x230_impl_t impl{ *this };
        virtual fp64      apply_acceleration(fp64 velocity)                         { return impl.touchpad_lenovo_x230_accel_profile(velocity); }
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time now) { return impl.accelerator_filter_x230(unaccelerated, now); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)            { return impl.accelerator_filter_constant_x230(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)            { return impl.accelerator_filter_constant_x230(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                          { return impl.accelerator_set_speed_x230(speed_adjustment); }
        virtual void      restart(time now)                                         {        impl.accelerator_x230_restart(now); }
    };
    struct touchpad_accelerator : motion_filter
    {
        fp64 threshold;    // mm/s.
        fp64 accel;        // Unitless factor.
        fp64 speed_factor; // Factor based on speed setting.

        struct touchpad_accelerator_impl_t
        {
            touchpad_accelerator& accel;
            fp64_coor accelerator_filter_touchpad(fp64_coor unaccelerated, time now)
            {
                auto accelerated = unaccelerated * accel.calculate_acceleration_simpsons(unaccelerated, now);
                return accel.normalize_for_dpi(accelerated);
            }
            fp64_coor touchpad_constant_filter(fp64_coor unaccelerated)
            {
                // We need to use the same baseline here as the accelerated code,
                // otherwise our unaccelerated speed is different to the accelerated
                // speed on the plateau.
                //
                // This is a hack, the baseline should be incorporated into the
                // lixx::tp_magic_slowdown so we only have one number here but meanwhile
                // this will do.
                static constexpr auto baseline = 0.9;
                return accel.normalize_for_dpi(unaccelerated) * baseline * lixx::tp_magic_slowdown;
            }
                fp64 speed_factor2(fp64 s)
                {
                    // Maps the [-1, 1] speed setting into a constant acceleration
                    // range. This isn't a linear scale, we keep 0 as the 'optimized'
                    // mid-point and scale down to 0.05 for setting -1 and up to 5 for
                    // setting 1. On the premise that if you want a faster cursor, it
                    // doesn't matter as much whether you have 0.56789 or 0.56790,
                    // but for lower settings it does because you may lose movements. *shrug*.
                    return pow(s + 1, 2.38) * 0.95 + 0.05;
                }
            bool touchpad_accelerator_set_speed(fp64 speed_adjustment)
            {
                assert(speed_adjustment >= -1.0 && speed_adjustment <= 1.0);
                accel.speed_adjustment = speed_adjustment;
                accel.speed_factor = speed_factor2(speed_adjustment);
                return true;
            }
                fp64 touchpad_accel_profile_linear(fp64 speed_in/* in device units/µs */)
                {
                    const auto threshold = accel.threshold; // mm/s.
                    const auto baseline = 0.9;
                    auto factor = 0.0; // Unitless.
                    speed_in = v_us2s(speed_in) * 25.4 / accel.dpi; // Convert to mm/s because that's something one can understand.
                    // Our acceleration function calculates a factor to accelerate input
                    // deltas with. The function is a double incline with a plateau,
                    // with a rough shape like this:
                    //     accel
                    //     factor
                    //     ^         ______
                    //     |        )
                    //     |  _____)
                    //     | /
                    //     |/
                    //     +-------------> speed in
                    // Except the second incline is a curve, but well, asciiart.
                    // The first incline is a linear function in the form
                    //     y = ax + b
                    //     where y is speed_out
                    //           x is speed_in
                    //           a is the incline of acceleration
                    //           b is minimum acceleration factor
                    // for speeds up to the lower threshold, we decelerate, down to 30% of input speed.
                    //     hence 1 = a * 7 + 0.3
                    //         0.7 = a * 7  => a := 0.1
                    //     deceleration function is thus:
                    //         y = 0.1x + 0.3
                    // The first plateau is the baseline.
                    // The second incline is a curve up, based on magic numbers obtained by trial-and-error.
                    // Above the second incline we have another plateau because
                    // by then you're moving so fast that extra acceleration doesn't help.
                    // Note:
                    // * The minimum threshold is a result of trial-and-error and has no other special meaning.
                    // * 0.3 is chosen simply because it is above the Nyquist frequency for subpixel motion within a pixel.
                    if (speed_in < 7.0)
                    {
                        factor = std::min(baseline, 0.1 * speed_in + 0.3);
                        // Up to the threshold, we keep factor 1, i.e. 1:1 movement.
                    }
                    else if (speed_in < threshold)
                    {
                        factor = baseline;
                    }
                    else
                    {
                        // Acceleration function above the threshold is a curve up to four times the threshold, because why not.
                        // Don't assume anything about the specific numbers though, this was all just trial and error by tweaking numbers here and there, then the formula was optimized doing basic maths.
                        // You could replace this with some other random formula that gives the same numbers and it would be just as correct.
                        auto upper_threshold = threshold * 4.0;
                        speed_in = std::min(speed_in, upper_threshold);
                        factor = 0.0025 * (speed_in / threshold) * (speed_in - threshold) + baseline;
                    }
                    // Once normalized, touchpads see the same acceleration as mice. that is technically correct but subjectively wrong, we expect a touchpad to be a lot slower than a mouse. Apply a magic factor to slow down all movements.
                    factor *= accel.speed_factor;
                    return factor * lixx::tp_magic_slowdown;
                }
        };

        touchpad_accelerator(si32 dpi, span event_delta_smooth_threshold, span event_delta_smooth_value, bool use_velocity_averaging)
            : motion_filter{ dpi },
                  threshold{ 130 },
                      accel{ 0.0 },
               speed_factor{ 0.0 }
        {
            type = LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE;
            trackers.trackers_init(use_velocity_averaging ? 16 : 2);
            trackers.smoothener = pointer_delta_smoothener::create(event_delta_smooth_threshold, event_delta_smooth_value);
        }

        touchpad_accelerator_impl_t impl{ *this };
        virtual fp64      apply_acceleration(fp64 velocity)                         { return impl.touchpad_accel_profile_linear(velocity); }
        virtual fp64_coor filter(         fp64_coor unaccelerated, void*, time now) { return impl.accelerator_filter_touchpad(unaccelerated, now); }
        virtual fp64_coor filter_constant(fp64_coor unaccelerated, time)            { return impl.touchpad_constant_filter(unaccelerated); }
        virtual fp64_coor filter_scroll(  fp64_coor unaccelerated, time)            { return impl.touchpad_constant_filter(unaccelerated); }
        virtual bool      set_speed(fp64 speed_adjustment)                          { return impl.touchpad_accelerator_set_speed(speed_adjustment); }
    };

    si32 libevdev_event_type_from_name(qiew name)
    {
             if (name == "EV_ABS"      ) return EV_ABS;
        else if (name == "EV_FF"       ) return EV_FF;
        else if (name == "EV_FF_STATUS") return EV_FF_STATUS;
        else if (name == "EV_KEY"      ) return EV_KEY;
        else if (name == "EV_LED"      ) return EV_LED;
        else if (name == "EV_MAX"      ) return EV_MAX;
        else if (name == "EV_MSC"      ) return EV_MSC;
        else if (name == "EV_PWR"      ) return EV_PWR;
        else if (name == "EV_REL"      ) return EV_REL;
        else if (name == "EV_REP"      ) return EV_REP;
        else if (name == "EV_SND"      ) return EV_SND;
        else if (name == "EV_SW"       ) return EV_SW;
        else if (name == "EV_SYN"      ) return EV_SYN;
        else return -1;
    }
    qiew libevdev_event_type_get_name(ui32 type)
    {
             if (type == EV_SYN      ) return "EV_SYN";
        else if (type == EV_KEY      ) return "EV_KEY";
        else if (type == EV_REL      ) return "EV_REL";
        else if (type == EV_ABS      ) return "EV_ABS";
        else if (type == EV_MSC      ) return "EV_MSC";
        else if (type == EV_SW       ) return "EV_SW";
        else if (type == EV_LED      ) return "EV_LED";
        else if (type == EV_SND      ) return "EV_SND";
        else if (type == EV_REP      ) return "EV_REP";
        else if (type == EV_FF       ) return "EV_FF";
        else if (type == EV_PWR      ) return "EV_PWR";
        else if (type == EV_FF_STATUS) return "EV_FF_STATUS";
        else if (type == EV_MAX      ) return "EV_MAX";
        else                           return "";
    }
    si32 libevdev_event_type_get_max(ui32 type)
    {
             if (type == EV_SYN) return SYN_MAX;
        else if (type == EV_ABS) return ABS_MAX;
        else if (type == EV_REL) return REL_MAX;
        else if (type == EV_KEY) return KEY_MAX;
        else if (type == EV_REP) return REP_MAX;
        else if (type == EV_MSC) return MSC_MAX;
        else if (type == EV_SW ) return SW_MAX;
        else if (type == EV_LED) return LED_MAX;
        else if (type == EV_SND) return SND_MAX;
        else if (type == EV_FF ) return FF_MAX;
        else                     return -1;
    }
    qiew libevdev_property_get_name(ui32 prop)
    {
             if (prop == INPUT_PROP_POINTER       ) return "INPUT_PROP_POINTER";
        else if (prop == INPUT_PROP_DIRECT        ) return "INPUT_PROP_DIRECT";
        else if (prop == INPUT_PROP_BUTTONPAD     ) return "INPUT_PROP_BUTTONPAD";
        else if (prop == INPUT_PROP_SEMI_MT       ) return "INPUT_PROP_SEMI_MT";
        else if (prop == INPUT_PROP_TOPBUTTONPAD  ) return "INPUT_PROP_TOPBUTTONPAD";
        else if (prop == INPUT_PROP_POINTING_STICK) return "INPUT_PROP_POINTING_STICK";
        else if (prop == INPUT_PROP_ACCELEROMETER ) return "INPUT_PROP_ACCELEROMETER";
        else if (prop == INPUT_PROP_MAX           ) return "INPUT_PROP_MAX";
        else                                        return "";
    }
    si32 libevdev_property_from_name(qiew name)
    {
             if (name == "INPUT_PROP_ACCELEROMETER" ) return INPUT_PROP_ACCELEROMETER;
        else if (name == "INPUT_PROP_BUTTONPAD"     ) return INPUT_PROP_BUTTONPAD;
        else if (name == "INPUT_PROP_DIRECT"        ) return INPUT_PROP_DIRECT;
        else if (name == "INPUT_PROP_MAX"           ) return INPUT_PROP_MAX;
        else if (name == "INPUT_PROP_POINTER"       ) return INPUT_PROP_POINTER;
        else if (name == "INPUT_PROP_POINTING_STICK") return INPUT_PROP_POINTING_STICK;
        else if (name == "INPUT_PROP_SEMI_MT"       ) return INPUT_PROP_SEMI_MT;
        else if (name == "INPUT_PROP_TOPBUTTONPAD"  ) return INPUT_PROP_TOPBUTTONPAD;
        else                                          return -1;
    }

    struct match_t
    {
        ui32     bits{};
        text     name2;
        text     uniq2;
        bustype  bus{};
        ui32     vendor{};
        ui32     product[64] = {}; // Zero-terminated.
        ui32     version{};
        text     dmi2; // DMI modalias with preceding "dmi:".
        ui32     ud_type{}; // We can have more than one type set, so this is a bitfield.
        text     dt2; // Device tree compatible (first) string.
    };
    struct quirk_tuples
    {
        struct tuples_t
        {
            si32 first;
            si32 second;
            si32 third;
        };
        std::array<tuples_t, 32> tuples;
        ui64                     ntuples{};
    };
    struct property_t
    {
        using prop_variants = std::variant<bool,
                                           ui32,
                                           si32,
                                           fp64,
                                           text,
                                           si32_coor,
                                           si32_range,
                                           quirk_tuples>;
        quirk         id{};
        prop_variants value;
    };
    struct section_t
    {
        bool                     has_match{};    // To check for empty sections.
        bool                     has_property{}; // To check for empty sections.
        text                     name2;          // The [Section Name].
        match_sptr               match;
        std::list<property_sptr> properties;
    };
    struct quirks_t
    {
        std::list<property_sptr> properties;          // These are not ref'd, just a collection of pointers.
        std::list<property_sptr> floating_properties; // Special properties for AttrEventCode and AttrInputCode, these are owned by us, not the section.

        void _quirk_merge_event_codes(property_sptr prop)
        {
            auto& prop_tuples = std::get<quirk_tuples>(prop->value);
            for (auto& p : properties)
            {
                if (p->id == prop->id) // We have a duplicated property, merge in with ours.
                {
                    auto& p_tuples = std::get<quirk_tuples>(p->value);
                    auto offset = p_tuples.ntuples;
                    for (auto j = 0ul; j < prop_tuples.ntuples; j++)
                    {
                        if (offset + j >= p_tuples.tuples.size()) break;
                        p_tuples.tuples[offset + j] = prop_tuples.tuples[j];
                        p_tuples.ntuples++;
                    }
                    return;
                }
            }
            // First time we add AttrEventCode: create a new property.
            // Unlike the other properties, this one isn't part of a section, it belongs to the quirks.
            auto newprop = ptr::shared<property_t>();
            newprop->id    = prop->id;
            newprop->value = prop_tuples;
            properties.emplace_back(newprop);
            floating_properties.emplace_back(newprop);
        }
        void _quirk_apply_section(section_sptr s)
        {
            for (auto p : s->properties)
            {
                log("property added: %s% from %s%", quirks_t::quirk_get_name(p->id), s->name2);
                // All quirks but AttrEventCode and AttrInputProp
                // simply overwrite each other, so we can just append the
                // matching property and, later when checking the quirk, pick
                // the last one in the array.
                //
                // The event codes/input props are special because they're lists
                // that may *partially* override each other, e.g. a section may
                // enable BTN_LEFT and BTN_RIGHT but a later section may disable
                // only BTN_RIGHT. This should result in BTN_LEFT force-enabled
                // and BTN_RIGHT force-disabled.
                //
                // To hack around this, those are the only ones where only ever
                // have one struct property in the list (not owned by a section)
                // and we simply merge any extra sections onto that.
                if (p->id == QUIRK_ATTR_EVENT_CODE || p->id == QUIRK_ATTR_INPUT_PROP)
                {
                    _quirk_merge_event_codes(p);
                }
                else
                {
                    properties.push_back(p);
                }
            }
        }
        void quirk_match_section(section_sptr s, match_sptr m)
        {
            auto matched_flags = 0u;
            for (auto flag = 1u; flag <= M_LAST; flag <<= 1)
            {
                auto prev_matched_flags = matched_flags;
                if (!(s->match->bits & flag)) continue; // Section doesn't have this bit set, continue.
                if (!(m->bits & flag)) // Couldn't fill in this bit for the match, so we do not match on it.
                {
                    log("%s% wants %s% but we don't have that", s->name2, quirks_t::matchflagname((match_flags)flag));
                    continue;
                }
                switch (flag)
                {
                    case M_NAME:        if (::fnmatch(s->match->name2.data(), m->name2.data(), 0) == 0) matched_flags |= flag; break;
                    case M_UNIQ:        if (::fnmatch(s->match->uniq2.data(), m->uniq2.data(), 0) == 0) matched_flags |= flag; break;
                    case M_BUS:         if (m->bus == s->match->bus)                                    matched_flags |= flag; break;
                    case M_VID:         if (m->vendor == s->match->vendor)                              matched_flags |= flag; break;
                    case M_PID:         for (auto mi : m->product)
                                        {
                                            if (mi == 0 || matched_flags & flag) break;
                                            for (auto si : s->match->product)
                                            {
                                                if (si == 0) break;
                                                if (mi == si)
                                                {
                                                    matched_flags |= flag;
                                                    break;
                                                }
                                            }
                                        }
                                        break;
                    case M_VERSION:     if (m->version == s->match->version)                          { matched_flags |= flag; } break;
                    case M_DMI:         if (::fnmatch(s->match->dmi2.data(), m->dmi2.data(), 0) == 0) { matched_flags |= flag; } break;
                    case M_DT:          if (::fnmatch(s->match->dt2.data(), m->dt2.data(), 0) == 0)   { matched_flags |= flag; } break;
                    case M_UDEV_TYPE:   if (s->match->ud_type & m->ud_type)                           { matched_flags |= flag; } break;
                    default: ::abort();
                }
                if (prev_matched_flags != matched_flags)
                {
                    log("%s% matches for %s%", s->name2, quirks_t::matchflagname((match_flags)flag));
                }
            }
            if (s->match->bits == matched_flags)
            {
                log("%s% is full match", s->name2);
                _quirk_apply_section(s);
            }
        }
        property_sptr _quirk_find_prop(quirk which)
        {
            for (auto p : properties | std::views::reverse) // Run backwards to only handle the last one assigned.
            {
                if (p->id == which)
                {
                    return p;
                }
            }
            return property_sptr{};
        }
        template<class T>
        bool quirks_get(quirk which, T& val)
        {
            if (auto p = _quirk_find_prop(which))
            {
                assert(std::holds_alternative<T>(p->value));
                val = std::get<T>(p->value);
                return true;
            }
            return faux;
        }
        static char const* quirk_get_name(quirk q)
        {
            switch (q)
            {
                case QUIRK_MODEL_ALPS_SERIAL_TOUCHPAD:          return "ModelALPSSerialTouchpad";
                case QUIRK_MODEL_APPLE_TOUCHPAD:                return "ModelAppleTouchpad";
                case QUIRK_MODEL_APPLE_TOUCHPAD_ONEBUTTON:      return "ModelAppleTouchpadOneButton";
                case QUIRK_MODEL_BOUNCING_KEYS:                 return "ModelBouncingKeys";
                case QUIRK_MODEL_CHROMEBOOK:                    return "ModelChromebook";
                case QUIRK_MODEL_CLEVO_W740SU:                  return "ModelClevoW740SU";
                case QUIRK_MODEL_DELL_CANVAS_TOTEM:             return "ModelDellCanvasTotem";
                case QUIRK_MODEL_HP_PAVILION_DM4_TOUCHPAD:      return "ModelHPPavilionDM4Touchpad";
                case QUIRK_MODEL_HP_ZBOOK_STUDIO_G3:            return "ModelHPZBookStudioG3";
                case QUIRK_MODEL_INVERT_HORIZONTAL_SCROLLING:   return "ModelInvertHorizontalScrolling";
                case QUIRK_MODEL_LENOVO_SCROLLPOINT:            return "ModelLenovoScrollPoint";
                case QUIRK_MODEL_LENOVO_T450_TOUCHPAD:          return "ModelLenovoT450Touchpad";
                case QUIRK_MODEL_LENOVO_X1GEN6_TOUCHPAD:        return "ModelLenovoX1Gen6Touchpad";
                case QUIRK_MODEL_LENOVO_X230:                   return "ModelLenovoX230";
                case QUIRK_MODEL_SYNAPTICS_SERIAL_TOUCHPAD:     return "ModelSynapticsSerialTouchpad";
                case QUIRK_MODEL_SYSTEM76_BONOBO:               return "ModelSystem76Bonobo";
                case QUIRK_MODEL_SYSTEM76_GALAGO:               return "ModelSystem76Galago";
                case QUIRK_MODEL_SYSTEM76_KUDU:                 return "ModelSystem76Kudu";
                case QUIRK_MODEL_TABLET_MODE_NO_SUSPEND:        return "ModelTabletModeNoSuspend";
                case QUIRK_MODEL_TABLET_MODE_SWITCH_UNRELIABLE: return "ModelTabletModeSwitchUnreliable";
                case QUIRK_MODEL_TOUCHPAD_VISIBLE_MARKER:       return "ModelTouchpadVisibleMarker";
                case QUIRK_MODEL_TOUCHPAD_PHANTOM_CLICKS:       return "ModelTouchpadPhantomClicks";
                case QUIRK_MODEL_TRACKBALL:                     return "ModelTrackball";
                case QUIRK_MODEL_WACOM_TOUCHPAD:                return "ModelWacomTouchpad";
                case QUIRK_MODEL_PRESSURE_PAD:                  return "ModelPressurePad";
                case QUIRK_ATTR_SIZE_HINT:                      return "AttrSizeHint";
                case QUIRK_ATTR_TOUCH_SIZE_RANGE:               return "AttrTouchSizeRange";
                case QUIRK_ATTR_PALM_SIZE_THRESHOLD:            return "AttrPalmSizeThreshold";
                case QUIRK_ATTR_LID_SWITCH_RELIABILITY:         return "AttrLidSwitchReliability";
                case QUIRK_ATTR_KEYBOARD_INTEGRATION:           return "AttrKeyboardIntegration";
                case QUIRK_ATTR_TRACKPOINT_INTEGRATION:         return "AttrPointingStickIntegration";
                case QUIRK_ATTR_TPKBCOMBO_LAYOUT:               return "AttrTPKComboLayout";
                case QUIRK_ATTR_PRESSURE_RANGE:                 return "AttrPressureRange";
                case QUIRK_ATTR_PALM_PRESSURE_THRESHOLD:        return "AttrPalmPressureThreshold";
                case QUIRK_ATTR_RESOLUTION_HINT:                return "AttrResolutionHint";
                case QUIRK_ATTR_TRACKPOINT_MULTIPLIER:          return "AttrTrackpointMultiplier";
                case QUIRK_ATTR_THUMB_PRESSURE_THRESHOLD:       return "AttrThumbPressureThreshold";
                case QUIRK_ATTR_USE_VELOCITY_AVERAGING:         return "AttrUseVelocityAveraging";
                case QUIRK_ATTR_TABLET_SMOOTHING:               return "AttrTabletSmoothing";
                case QUIRK_ATTR_THUMB_SIZE_THRESHOLD:           return "AttrThumbSizeThreshold";
                case QUIRK_ATTR_MSC_TIMESTAMP:                  return "AttrMscTimestamp";
                case QUIRK_ATTR_EVENT_CODE:                     return "AttrEventCode";
                case QUIRK_ATTR_INPUT_PROP:                     return "AttrInputProp";
                case QUIRK_ATTR_IS_VIRTUAL:                     return "AttrIsVirtual";
                default:
                    ::abort();
            }
        }
        static char const* matchflagname(match_flags f)
        {
            switch (f)
            {
                case M_NAME:      return "MatchName";        break;
                case M_BUS:       return "MatchBus";         break;
                case M_VID:       return "MatchVendor";      break;
                case M_PID:       return "MatchProduct";     break;
                case M_VERSION:   return "MatchVersion";     break;
                case M_DMI:       return "MatchDMIModalias"; break;
                case M_UDEV_TYPE: return "MatchUdevType";    break;
                case M_DT:        return "MatchDeviceTree";  break;
                case M_UNIQ:      return "MatchUniq";        break;
                default: ::abort();
            }
        }
    };

    struct event_source_t
    {
        std::function<void()> func;
        fd_t                  fd{ os::invalid_fd };
    };
    template<class T>
    struct libinput_timer_t : ptr::enable_shared_from_this<libinput_timer_t<T>>
    {
        T&                        owner;
        text                      timer_name;
        time                      expire;
        std::function<void(time)> func;

        libinput_timer_t(T& owner, text&& name, auto func)
            :         owner{ owner           },
                 timer_name{ std::move(name) },
                     expire{                 },
                       func{ std::move(func) }
        { }
        void cancel()
        {
            owner.libinput_timer_cancel(*this);
        }
        void start(time expire, bool allow_negative = {})
        {
            owner.libinput_timer_set(*this, expire, allow_negative);
        }
    };
    struct libinput_timer_host
    {
        using libinput_timer_t = lixx::libinput_timer_t<libinput_timer_host>;

        std::vector<libinput_timer_sptr> active;
        std::vector<libinput_timer_sptr> cached;
        event_source_sptr                source;
        fd_t                             fd{ os::invalid_fd };
        fd_t                             epoll_fd{ os::invalid_fd };
        time                             next_expiry{};
        std::vector<event_source_sptr>   source_destroy_list;

        libinput_timer_host() = default;
        ~libinput_timer_host()
        {
            os::close(fd);
            os::close(epoll_fd);
        }

        void clear()
        {
            active.clear();
            cached.clear();
            if (source)
            {
                libinput_remove_event_source(source);
            }
            os::close(fd);
            os::close(epoll_fd);
        }
        auto create(text name, auto func)
        {
            return ptr::shared<libinput_timer_t>(*this, std::move(name), std::move(func));
        }
        void libinput_timer_cancel(libinput_timer_t& timer)
        {
            if (timer.expire != time{})
            {
                timer.expire = {};
                libinput_timer_handler();
            }
        }
        void libinput_timer_set(libinput_timer_t& timer, time expire, [[maybe_unused]] bool allow_negative = {})
        {
            #ifndef NDEBUG
            auto now = datetime::now();
            if (expire < now)
            {
                if (!allow_negative && now - expire > lixx::timer_warning_limit)
                {
                    log("timer %s%: scheduled expiry is in the past (-%dms%), your system is too slow", timer.timer_name, now - expire);
                }
            }
            else if ((expire - now) > 5000ms)
            {
                log("timer %s%: offset more than 5s, now %d% expire %d%", timer.timer_name, now, expire);
            }
            #endif
            assert(expire != time{});
            if (timer.expire == time{}) // Push only if timer is inactive (not in the active list).
            {
                active.push_back(timer.This());
            }
            timer.expire = expire;
            libinput_timer_handler();
        }
        void libinput_timer_handler(time now = {})
        {
            auto earliest_expire = netxs::maxtime;
            std::swap(active, cached);
            active.clear(); // Get ready to start new timers.
            for (auto timer : cached) // Filter canceled timers and find smallest timeout.
            {
                if (timer->expire != time{})
                {
                    if (timer->expire <= now)
                    {
                        timer->expire = {};
                        timer->func(now); // The func may re-arm timers or trigger another unrelated timer to be cancelled and removed.
                    }
                    else // Keep not expired timers active.
                    {
                        if (timer->expire < earliest_expire)
                        {
                            earliest_expire = timer->expire;
                        }
                        active.push_back(timer);
                    }
                }
            }
            auto its = ::itimerspec{};
            if (earliest_expire != netxs::maxtime)
            {
                auto sec = datetime::round<ui64, std::chrono::seconds>(earliest_expire);
                auto nsec = datetime::round<ui64, std::chrono::nanoseconds>(earliest_expire - std::chrono::seconds{ sec });
                its.it_value.tv_sec = sec;
                its.it_value.tv_nsec = nsec;
            }
            if (0 != ::timerfd_settime(fd, TFD_TIMER_ABSTIME, &its, nullptr))
            {
                log("timer: timerfd_settime error: %s%", ::strerror(errno));
            }
            next_expiry = earliest_expire;
        }
        void libinput_timer_flush(time now)
        {
            if (next_expiry != time{} && next_expiry <= now)
            {
                libinput_timer_handler(now);
            }
        }
        void libinput_timer_dispatch()
        {
            auto discard = ui64{};
            auto r = ::read(fd, &discard, sizeof(discard));
            if (r == -1 && errno != EAGAIN)
            {
                log("timer: error %d% reading from timerfd (%s%)", errno, ::strerror(errno));
            }
            auto now = datetime::now();
            libinput_timer_handler(now);
        }
        event_source_sptr libinput_add_event_source(si32 fd, auto func)
        {
            auto source = ptr::shared<event_source_t>();
            source->func = std::move(func);
            source->fd = fd;
            auto ep = ::epoll_event{};
            ep.events = EPOLLIN;
            ep.data.ptr = source.get();
            if (0 > ::epoll_ctl(epoll_fd, EPOLL_CTL_ADD, fd, &ep))
            {
                source.reset();
            }
            return source;
        }
        bool libinput_timer_subsys_init()
        {
            fd = ::timerfd_create(lixx::clock_type, TFD_CLOEXEC | TFD_NONBLOCK);
            if (fd < 0)
            {
                os::close(epoll_fd);
                return faux;
            }
            else
            {
                source = libinput_add_event_source(fd, [&]{ libinput_timer_dispatch(); });
                return true;
            }
        }
        void libinput_remove_event_source(event_source_sptr& source)
        {
            if (source->fd != os::invalid_fd)
            {
                ::epoll_ctl(epoll_fd, EPOLL_CTL_DEL, source->fd, nullptr);
                source->fd = os::invalid_fd;
                source_destroy_list.push_back(source);
            }
            source.reset();
        }
    };

    struct libinput_event
    {
        libinput_event_type  type{};
        libinput_device_sptr li_device;
        time                 stamp{};

        libinput_event() = default;
        virtual ~libinput_event()
        { }

        libinput_device_sptr libinput_event_get_device()
        {
            return li_device;
        }
        virtual ui32                  libinput_event_keyboard_get_key()                                              { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_keyboard_get_key_state()                                        { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual libinput_switch       libinput_event_switch_get_switch()                                             { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_switch_get_switch_state()                                       { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_gesture_get_finger_count()                                      { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_gesture_get_cancelled()                                         { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_gesture_get_ds()                                                { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_gesture_get_ds_unaccelerated()                                  { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64                  libinput_event_gesture_get_scale()                                             { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64                  libinput_event_gesture_get_angle_delta()                                       { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_pointer_get_absolute_xy_transformed(fp64_coor /*size*/)         { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_pointer_get_ds()                                                { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual ui32                  libinput_event_pointer_get_button()                                            { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_pointer_get_button_state()                                      { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual si32                  libinput_event_pointer_has_axis(libinput_pointer_axis /*axis*/)                { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_pointer_get_scroll_value()                                      { if constexpr (debugmode) bad_event_type(__func__); return {}; }
        virtual fp64_coor             libinput_event_pointer_get_scroll_value_v120()                                 { if constexpr (debugmode) bad_event_type(__func__); return {}; }

        view event_type_to_str()
        {
            if constexpr (debugmode)
            switch (type)
            {
                CASE_RETURN_STRING(LIBINPUT_EVENT_DEVICE_ADDED);
                CASE_RETURN_STRING(LIBINPUT_EVENT_DEVICE_REMOVED);
                CASE_RETURN_STRING(LIBINPUT_EVENT_KEYBOARD_KEY);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_MOTION);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_BUTTON);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_AXIS);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_SCROLL_WHEEL);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_SCROLL_FINGER);
                CASE_RETURN_STRING(LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TOUCH_DOWN);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TOUCH_UP);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TOUCH_MOTION);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TOUCH_CANCEL);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TOUCH_FRAME);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_TOOL_AXIS);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_TOOL_TIP);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_TOOL_BUTTON);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_PAD_BUTTON);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_PAD_RING);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_PAD_STRIP);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_PAD_KEY);
                CASE_RETURN_STRING(LIBINPUT_EVENT_TABLET_PAD_DIAL);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_SWIPE_BEGIN);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_SWIPE_END);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_PINCH_BEGIN);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_PINCH_UPDATE);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_PINCH_END);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_HOLD_BEGIN);
                CASE_RETURN_STRING(LIBINPUT_EVENT_GESTURE_HOLD_END);
                CASE_RETURN_STRING(LIBINPUT_EVENT_SWITCH_TOGGLE);
                default: ::abort();
            }
            return {};
        }
        void bad_event_type([[maybe_unused]] view function_name)
        {
            log("Invalid event type %s% (%d%) used to call %s%()", event_type_to_str(), type, function_name);
        }
    };

    struct libinput_event_keyboard : libinput_event
    {
        ui32 key{};
        ui32 seat_key_count{};
        si32 state{};

        virtual ui32 libinput_event_keyboard_get_key()       override { return key; }
        virtual si32 libinput_event_keyboard_get_key_state() override { return state; }
    };
    struct libinput_event_empty : libinput_event
    {
        libinput_event_empty()
        {
            libinput_event::type = LIBINPUT_EVENT_NONE;
        }
    };
    struct libinput_event_device_notify : libinput_event
    {
        //
    };
    struct libinput_event_pointer : libinput_event
    {
        fp64_coor                    delta;
        fp64_coor                    delta_raw;
        si32_coor                    absolute;
        si32_coor                    discrete;
        si32_coor                    v120;
        ui32                         button{};
        ui32                         seat_button_count{};
        si32                         state{};
        libinput_pointer_axis_source source{};
        ui32                         active_axes{};
        abs_info_t                   absinfo_x{};
        abs_info_t                   absinfo_y{};

        virtual fp64_coor libinput_event_pointer_get_absolute_xy_transformed(fp64_coor size) override
        {
            auto xy = fp64_coor{ absinfo_x.absinfo_scale_axis(absolute.x, size.x),
                                 absinfo_y.absinfo_scale_axis(absolute.y, size.y) };
            return xy;
        }
        virtual fp64_coor libinput_event_pointer_get_ds()                             override { return delta; }
        virtual ui32      libinput_event_pointer_get_button()                         override { return button; }
        virtual si32      libinput_event_pointer_get_button_state()                   override { return state; }
        virtual si32      libinput_event_pointer_has_axis(libinput_pointer_axis axis) override { return !!(active_axes & (1ul << axis)); }
        virtual fp64_coor libinput_event_pointer_get_scroll_value()                   override { return delta; }
        virtual fp64_coor libinput_event_pointer_get_scroll_value_v120()              override { return v120; }
    };
    struct libinput_event_gesture : libinput_event
    {
        si32      finger_count{};
        si32      cancelled{};
        fp64_coor delta;
        fp64_coor delta_unaccel;
        fp64      scale{};
        fp64      angle{};

        virtual si32      libinput_event_gesture_get_finger_count()     override { return finger_count; }
        virtual si32      libinput_event_gesture_get_cancelled()        override { return cancelled; }
        virtual fp64_coor libinput_event_gesture_get_ds()               override { return delta; }
        virtual fp64_coor libinput_event_gesture_get_ds_unaccelerated() override { return delta_unaccel; }
        virtual fp64      libinput_event_gesture_get_scale()            override { return scale; }
        virtual fp64      libinput_event_gesture_get_angle_delta()      override { return angle; }
    };
    struct libinput_event_touch : libinput_event
    {
        si32      slot{};
        si32      seat_slot{};
        si32_coor point;
    };
    struct libinput_event_switch : libinput_event
    {
        libinput_switch sw{};
        si32            state{};

        virtual libinput_switch libinput_event_switch_get_switch()       override { return sw; }
        virtual si32            libinput_event_switch_get_switch_state() override { return state; }
    };
        struct libinput_tablet_pad_mode_group
        {
            libinput_device_sptr li_device;
            void*                user_data{};
            ui32                 index{};
            ui32                 num_modes{};
            ui32                 current_mode{};
            ui32                 button_mask{};
            ui32                 ring_mask{};
            ui32                 strip_mask{};
            ui32                 dial_mask{};
            ui32                 toggle_button_mask{};

            libinput_tablet_pad_mode_group() = default;
            virtual ~libinput_tablet_pad_mode_group() { }
        };
    struct libinput_event_tablet_pad : libinput_event
    {
        struct button_t
        {
            ui32 number{};
            si32 state{};
        };
        struct key_t
        {
            ui32 code{};
            si32 state{};
        };
        struct dial_t
        {
            fp64 v120{};
            si32 number{};
        };
        struct ring_t
        {
            libinput_tablet_pad_ring_axis_source source{};
            fp64                                 position{};
            si32                                 number{};
        };
        struct strip_t
        {
            libinput_tablet_pad_strip_axis_source source{};
            fp64                                  position{};
            si32                                  number{};
        };

        ui32                                mode{};
        libinput_tablet_pad_mode_group_sptr mode_group;
        button_t                            button;
        key_t                               key;
        dial_t                              dial;
        ring_t                              ring;
        strip_t                             strip;
    };
        struct tablet_axes
        {
            si32_coor  point;
            fp64_coor  delta;
            fp64_range size_limits;
            fp64       distance{};
            fp64       pressure{};
            fp64_coor  tilt;
            fp64       rotation{};
            fp64       slider{};
            fp64       wheel{};
            si32       wheel_discrete{};
        };
    struct libinput_event_tablet_tool : libinput_event
    {
        ui32                                 button{};
        si32                                 state{};
        ui32                                 seat_button_count{};
        tablet_axes                          axes;
        tablet_axes_bitset                   changed_axes_bits;
        libinput_tablet_tool_sptr            tool;
        libinput_tablet_tool_proximity_state proximity_state{};
        libinput_tablet_tool_tip_state       tip_state{};
        abs_info_t                           abs_info_x2;
        abs_info_t                           abs_info_y2;

        libinput_event_tablet_tool() = default;
    };

    struct evdev_event
    {
        ui32 usage{}; // This may be a value outside the known usages above but it's just an int.
        si32 value{};
    };
    struct evdev_frame
    {
        time                     stamp;
        std::vector<evdev_event> ev_events;

        evdev_frame()
            : stamp{}
        {
            ev_events.reserve(256);
            ev_events.push_back(evdev_event{ .usage = evdev::syn_report }); // evdev::syn_report(0) should be always at the end.
        }
        void evdev_frame_reset()
        {
            ev_events.resize(1);
            ev_events.front() = {}; // evdev::syn_report(0) should be always at the end.
        }
        void evdev_frame_append_input_event(input_event_t& event)
        {
            auto ev = evdev_event{ .usage = evdev_usage_from_code(event.type, event.code), .value = event.value };
            if (ev.usage == evdev::syn_report)
            {
                stamp = event.input_event_time();
            }
            else
            {
                ev_events.pop_back(); // Pop evdev::syn_report.
                ev_events.push_back(ev);
                ev_events.push_back({}); // evdev::syn_report(0) should be always at the end.
            }
        }
    };

    struct ud_device_t
    {
        struct slot_change_state
        {
            touch_states         state{};
            std::bitset<ABS_CNT> axes;
        };
        struct mt_sync_state
        {
            ui32 code{};
            si32 val[lixx::max_slots] = {};
        };
        struct evdev_abs_t
        {
            abs_info_t& absinfo_x;
            abs_info_t& absinfo_y;
            bool        is_fake_resolution{};
            si32        apply_calibration{};
            matrix      calibration;
            matrix      default_calibration; // From LIBINPUT_CALIBRATION_MATRIX.
            matrix      usermatrix; // As supplied by the caller.
            si32_coor   dimensions;
            si32_range  warning_range_x;
            si32_range  warning_range_y;
        };

        utf::unordered_map<text, text> properties;
        text                           sysname; // "eventX"
        text                           devpath; // "/dev/input/eventX"
        text                           devname; // uevent_name: "USB Optical Mouse M7"
        text                           phys;    // uevent_phys: "usb-0000:00:14.0-4/input0" or "i2c-ASUE140C:00" or "isa0060/serio0/input0"
        text                           uniq;    // uevent_uniq: ""
        text                           device_class; // Something like "touchpad", "tablet". Used for logs.
        fd_t                           fd{ os::invalid_fd };
        sync_states                    sync_state{};
        ui32                           model_flags{};

        ::input_id prod_info;
        si32       driver_version{};

        std::bitset<INPUT_PROP_MAX>         prop_bits;
        std::bitset<EV_MAX>                   ev_bits;
        std::bitset<ABS_CNT>                 abs_bits;
        std::bitset<REL_CNT>                 rel_bits;
        std::bitset<KEY_CNT>                 key_bits;
        std::bitset<LED_CNT>                 led_bits;
        std::bitset<MSC_CNT>                 msc_bits;
        std::bitset<SW_CNT>                   sw_bits;
        std::bitset<FF_CNT>                   ff_bits;
        std::bitset<REP_CNT>                 rep_bits;
        std::bitset<SND_CNT>                 snd_bits;

        std::array<abs_info_t, ABS_CNT> abs_values;
        std::array<si32, REP_CNT>       rep_values;
        std::bitset<KEY_CNT>            key_values;
        std::bitset<LED_CNT>            led_values;
        std::bitset<SW_CNT>              sw_values;

        std::vector<si32>              mt_slot_vals; // size = num_slots * lixx::abs_mt_cnt
        si32                           num_slots{};  // = mt_slot_vals.size() / lixx::abs_mt_cnt
        si32                           current_slot{};
        std::vector<slot_change_state> slot_changes;

        std::vector<input_event_t> queue; //todo use std::deque
        ui64                       queue_next{};  // Next event index.
        ui64                       queue_nsync{}; // Number of sync events.
        ::timeval                  last_event_time{};

        bool                       initialized;
        bool                       is_mt;
        evdev_abs_t                abs;

        ud_device_t(text eventX)
            :   sysname{ eventX },
                devpath{ "/dev/input/" + eventX },
            initialized{ _initialize() },
                  is_mt{ !evdev_is_fake_mt_device() && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_X)
                                                    && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_Y) },
                    abs{ abs_values[is_mt ? ABS_MT_POSITION_X : ABS_X], abs_values[is_mt ? ABS_MT_POSITION_Y : ABS_Y] }
        {
            if (initialized)
            {
                if constexpr (debugmode)
                {
                    auto mx = 0u;
                    for (auto& [propname, value] : properties) if (mx < propname.size()) mx = propname.size();
                    mx += 10;
                    log("  Property\r\x1b[%mx%CValue", mx);
                    log("  ---------------------------------");
                    for ([[maybe_unused]] auto& [propname, value] : properties)
                    {
                        log("  %propname%\r\x1b[%mx%C%value%", propname, mx, value);
                    }
                }
            }
        }
        bool _initialize()
        {
            bool ok = faux;
            auto new_fd = ::open(devpath.data(), O_RDONLY | O_NONBLOCK | O_NOCTTY);
            if (new_fd != os::invalid_fd)
            {
                ud_device_t::evdev_drain_fd(new_fd);
                ok = _libevdev_set_fd(new_fd);
                if (ok)
                {
                    _set_properties();
                    _sync_with_hwdb();
                }
                else
                {
                    log("  Device %% is not initialized", devpath);
                    os::close(new_fd);
                }
            }
            return ok;
        }
        void _sync_with_hwdb()
        {
            //todo sync with lixx::hwdb lookup in /etc/udev/*mouse.hwdb or /usr/lib/udev/*mouse.hwdb
            //auto udev_path1 = os::fs::path{ "/etc/udev/hwdb.d/" };
            //auto udev_path2 = os::fs::path{ "/usr/lib/udev/hwdb.d/" };
            //auto mouse_regex = std::regex{ ".*mouse.*\\.hwdb"s };
            //auto touchpad_regex = std::regex{ ".*touchpad.*\\.hwdb"s };
            //auto code = std::error_code{};
            //auto files = fs::directory_iterator{ udev_path1, code };
            //if (!code)
            //for (auto& entry : files)
            //{
            //    if (entry.is_regular_file())
            //    {
            //        auto filename = entry.path().filename().string();
            //        if (std::regex_match(filename, touchpad_regex))
            //        {
            //            set ID_INPUT_TOUCHPAD_INTEGRATION=internal | external
            //        }
            //        if (std::regex_match(filename, mouse_regex))
            //        {
            //            auto file = open(filename);
            //            while (auto regex_pattern = readline(file))
            //            {
            //                regex_pattern = std::regex_replace(regex_pattern, std::regex("\\."), "\\.");
            //                regex_pattern = std::regex_replace(regex_pattern, std::regex("\\*"), ".*");
            //                auto rec_regex = std::regex{ regex_pattern };
            //                set "MOUSE_DPI"
            //                    "MOUSE_WHEEL_CLICK_COUNT"
            //                    "MOUSE_WHEEL_CLICK_ANGLE"
            //                    "MOUSE_WHEEL_CLICK_COUNT_HORIZONTAL"
            //                    "MOUSE_WHEEL_CLICK_ANGLE_HORIZONTAL"
            //            }
            //        }
            //    }
            //}
        }
        void _set_properties()
        {
            auto accum = [](auto& bits, auto from, auto upto) { auto summ = 0; for (auto i = from; i < upto; i++) summ += (si32)bits[i]; return summ; };
            auto accum2= [](auto& bits, auto& ids)            { auto summ = 0; for (auto i : ids)                 summ += (si32)bits[i]; return summ; };
            auto allof = [](auto& bits, auto from, auto upto) { for (auto i = from; i < upto; i++) if (!bits[i]) return faux; return true; };
            auto anyof = [](auto& bits, auto from, auto upto) { for (auto i = from; i < upto; i++) if (bits[i]) return true; return faux; };
            properties["NAME"] = devname;
            properties["PHYS"] = phys;
            properties["UNIQ"] = uniq;
            properties["PRODUCT"] = utf::fprint("%%/%%/%%/%%", utf::to_hex(prod_info.bustype),
                                                               utf::to_hex(prod_info.vendor ),
                                                               utf::to_hex(prod_info.product),
                                                               utf::to_hex(prod_info.version));
            if (abs_bits[ABS_X] && abs_bits[ABS_Y]) //todo Sync with tp_init_default_resolution
            {
                auto& xinfo = abs_values[ABS_X];
                auto& yinfo = abs_values[ABS_Y];
                if (xinfo.resolution == 0) xinfo.resolution = (xinfo.maximum - xinfo.minimum) / 70; // Fallback to 70x50mm touchpad size.
                if (yinfo.resolution == 0) yinfo.resolution = xinfo.resolution;                     //
                auto mm = [](auto& xy){ return (xy.maximum - xy.minimum) / xy.resolution; };
                properties["ID_INPUT_WIDTH_MM" ] = std::to_string(mm(xinfo));
                properties["ID_INPUT_HEIGHT_MM"] = std::to_string(mm(yinfo));
            }
            auto like_accelerometer = prop_bits[INPUT_PROP_ACCELEROMETER] || (!ev_bits[EV_KEY] && abs_bits[ABS_X] && abs_bits[ABS_Y] && abs_bits[ABS_Z]);
            auto device_class = text{};
            if (like_accelerometer)
            {
                properties["ID_INPUT_ACCELEROMETER"] = "1";
                properties["ID_INPUT_MOUSE"]         = "1";
                device_class += " Accelerometer";
                device_class += " Mouse";
            }
            else
            {
                auto have_wheel       = ev_bits[EV_REL] && (rel_bits[REL_WHEEL] || rel_bits[REL_HWHEEL]);
                auto have_pad_buttons = key_bits[BTN_0] && key_bits[BTN_1] && !key_bits[BTN_TOOL_PEN];
                auto have_finger      = key_bits[BTN_TOOL_FINGER] && !key_bits[BTN_TOOL_PEN];
                auto have_touch       = key_bits[BTN_TOUCH];
                auto have_stylus      = key_bits[BTN_STYLUS];
                auto have_abs_xy      = abs_bits[ABS_X] && abs_bits[ABS_Y];
                auto have_mt_coords   = abs_bits[ABS_MT_POSITION_X] && abs_bits[ABS_MT_POSITION_Y] && (!abs_bits[ABS_MT_SLOT] || !abs_bits[ABS_MT_SLOT - 1]);
                auto joystick_button_count = 0;
                if (!key_bits[BTN_TASK]) // Exclude a case of mouse with more than 16 buttons, e.g. "Mad Catz M.M.O. TE".
                {
                    joystick_button_count += accum(key_bits, BTN_JOYSTICK,       BTN_THUMBR + 1);
                    joystick_button_count += accum(key_bits, BTN_TRIGGER_HAPPY1, BTN_TRIGGER_HAPPY40 + 1);
                    joystick_button_count += accum(key_bits, BTN_DPAD_UP,        BTN_DPAD_RIGHT + 1);
                }
                auto like_tablet        = faux;
                auto like_touchpad      = faux;
                auto like_abs_mouse     = faux;
                auto like_touchscreen   = faux;
                auto have_mouse_buttons = anyof(key_bits, BTN_MOUSE, BTN_TASK + 1);
                auto have_direct        = prop_bits[INPUT_PROP_DIRECT];
                auto have_pen           = key_bits[BTN_TOOL_PEN];
                if (have_abs_xy)
                {
                   ((have_stylus || have_pen    ) ? like_tablet      :
                    (have_finger && !have_direct) ? like_touchpad    :
                    (have_mouse_buttons         ) ? like_abs_mouse   : // VMware's USB mouse has no touch/pressure buttons but it has absolute axes.
                    (have_touch || have_direct  ) ? like_touchscreen :
                                                    have_abs_xy) = true;
                }
                if (have_mt_coords)
                {
                   ((have_stylus || have_pen    ) ? like_tablet      :
                    (have_finger && !have_direct) ? like_touchpad    :
                    (have_touch || have_direct  ) ? like_touchscreen :
                                                    have_mt_coords) = true;
                }
                auto like_tablet_pad = like_tablet && have_pad_buttons;
                auto have_rel_coords = ev_bits[EV_REL] && rel_bits[REL_X] && rel_bits[REL_Y];
                if (have_pad_buttons && have_wheel && !have_rel_coords)
                {
                    like_tablet = true;
                    like_tablet_pad = true;
                }
                auto like_joystick      = faux;
                auto joystick_axe_count = accum(abs_bits, ABS_RX, ABS_HAT3Y + 1);
                if (joystick_button_count || joystick_axe_count) // Distinguish keyboards/tablet-pads with random joystick buttons.
                {
                    static constexpr auto test_key_subset = std::to_array({ KEY_LEFTCTRL, KEY_CAPSLOCK, KEY_NUMLOCK, KEY_INSERT, KEY_MUTE, KEY_CALC, KEY_FILE, KEY_MAIL, KEY_PLAYPAUSE, KEY_BRIGHTNESSDOWN, });
                    auto count = accum2(key_bits, test_key_subset);
                    like_joystick = count <= 3 && joystick_button_count + joystick_axe_count >= 2 // The device has joystick buttons and axes but also a keyboard key subset.
                                                            && !(have_wheel && have_pad_buttons); // The device with a wheel and pad buttons is not a joystick.
                }
                auto like_trackball      = utf::to_lower(text{ devname }).find("trackball") != text::npos;
                auto like_switch         = ev_bits[EV_SW];
                auto like_mouse          = have_mouse_buttons && !(like_touchpad || like_tablet || like_joystick);
                auto like_pointing_stick = prop_bits[INPUT_PROP_POINTING_STICK] || (like_mouse && prod_info.bustype == BUS_I2C);
                auto is_mouse            = like_tablet || like_mouse || like_abs_mouse || like_touchpad || like_touchscreen || like_joystick || like_pointing_stick;
                auto like_keyboard       = ev_bits[EV_KEY] && allof(key_bits, KEY_ESC, KEY_D);
                auto is_wheel            = have_wheel && !is_mouse;
                auto like_key            = is_wheel || (ev_bits[EV_KEY] && (anyof(key_bits, KEY_RESERVED,   BTN_MISC)
                                                                         || anyof(key_bits, KEY_OK,         BTN_DPAD_UP)
                                                                         || anyof(key_bits, KEY_ALS_TOGGLE, BTN_TRIGGER_HAPPY)));
                like_mouse |= like_abs_mouse;
                if (like_key           ) { device_class += " Key";           properties["ID_INPUT_KEY"          ] = "1"; }
                if (like_mouse         ) { device_class += " Mouse";         properties["ID_INPUT_MOUSE"        ] = "1"; }
                if (like_switch        ) { device_class += " Switch";        properties["ID_INPUT_SWITCH"       ] = "1"; }
                if (like_tablet        ) { device_class += " Tablet";        properties["ID_INPUT_TABLET"       ] = "1"; }
                if (like_joystick      ) { device_class += " Joystick";      properties["ID_INPUT_JOYSTICK"     ] = "1"; }
                if (like_keyboard      ) { device_class += " Keyboard";      properties["ID_INPUT_KEYBOARD"     ] = "1"; }
                if (like_touchpad      ) { device_class += " Touchpad";      properties["ID_INPUT_TOUCHPAD"     ] = "1"; }
                if (like_trackball     ) { device_class += " Trackball";     properties["ID_INPUT_TRACKBALL"    ] = "1"; }
                if (like_tablet_pad    ) { device_class += " TabletPad";     properties["ID_INPUT_TABLET_PAD"   ] = "1"; }
                if (like_touchscreen   ) { device_class += " Touchscreen";   properties["ID_INPUT_TOUCHSCREEN"  ] = "1"; }
                if (like_pointing_stick) { device_class += " Pointingstick"; properties["ID_INPUT_POINTINGSTICK"] = "1"; }
            }
            properties["ID_INPUT"] = "1";
            log("Device is tagged as:%s%", device_class);
        }
        static void evdev_drain_fd(si32 fd)
        {
            auto events = std::array<input_event_t, 24>{};
            while (::read(fd, events.data(), sizeof(events)) == sizeof(events)) // Discard all pending events.
            { }
        }
        bool libinput_ud_device_is_virtual()
        {
            //todo use ioctl
            return faux;//syspath.starts_with("/sys/devices/virtual/input/");
        }
        qiew libinput_udev_prop(view prop)
        {
            auto iter = properties.find(prop);
            auto value = qiew{};
            if (iter != properties.end())
            {
                value = iter->second;
            }
            return value;
        }
        template<ui32 Type>
        si32 libevdev_has_event_type()
        {
            return Type == EV_SYN || ev_bits[Type];
        }
        template<ui32 Type>
        auto& select_bits()
        {
                 if constexpr (Type == EV_ABS) return abs_bits;
            else if constexpr (Type == EV_REL) return rel_bits;
            else if constexpr (Type == EV_KEY) return key_bits;
            else if constexpr (Type == EV_REP) return rep_bits;
            else if constexpr (Type == EV_MSC) return msc_bits;
            else if constexpr (Type == EV_LED) return led_bits;
            else if constexpr (Type == EV_SND) return snd_bits;
            else if constexpr (Type == EV_SW ) return  sw_bits;
            else if constexpr (Type == EV_FF ) return  ff_bits;
        }
        void for_each_bits(auto proc)
        {
            if (libevdev_has_event_type<EV_ABS>()) proc(abs_bits);
            if (libevdev_has_event_type<EV_REL>()) proc(rel_bits);
            if (libevdev_has_event_type<EV_KEY>()) proc(key_bits);
            if (libevdev_has_event_type<EV_REP>()) proc(rep_bits);
            if (libevdev_has_event_type<EV_MSC>()) proc(msc_bits);
            if (libevdev_has_event_type<EV_LED>()) proc(led_bits);
            if (libevdev_has_event_type<EV_SND>()) proc(snd_bits);
            if (libevdev_has_event_type<EV_SW >()) proc( sw_bits);
            if (libevdev_has_event_type<EV_FF >()) proc( ff_bits);
        }
        template<ui32 Type>
        bool libevdev_has_event_code(ui32 code)
        {
            if constexpr (Type == EV_SYN) return true;
            if (libevdev_has_event_type<Type>())
            {
                auto& bits = select_bits<Type>();
                if (code < bits.size()) return bits[code];
            }
            return faux;
        }
        template<ui32 Type>
        si32 libevdev_get_event_value(ui32 code)
        {
            auto value = 0;
            if (libevdev_has_event_code<Type>(code))
            {
                     if constexpr (Type == EV_ABS) value = abs_values[code].value;
                else if constexpr (Type == EV_KEY) value = key_values[code];
                else if constexpr (Type == EV_LED) value = led_values[code];
                else if constexpr (Type == EV_SW ) value = sw_values[code];
                else if constexpr (Type == EV_REP)
                {
                         if (code == REP_DELAY ) value = rep_values[REP_DELAY];
                    else if (code == REP_PERIOD) value = rep_values[REP_PERIOD];
                }
            }
            return value;
        }
        template<ui32 Type>
        void libevdev_disable_event_code(ui32 code)
        {
            auto& bits = select_bits<Type>();
            if (code < bits.size())
            {
                bits[code] = faux;
                if constexpr (Type == EV_ABS)
                {
                    if (code == ABS_MT_SLOT)
                    {
                        init_slots();
                    }
                    else if (code == ABS_MT_TRACKING_ID)
                    {
                        reset_tracking_ids();
                    }
                }
            }
        }
        template<ui32 Type>
        bool set_bits_by_type(ui32 code)
        {
            auto& bits = select_bits<Type>();
            auto ok = code < bits.size();
            if (ok)
            {
                bits[code] = true;
            }
            return ok;
        }
        template<ui32 Type>
        void libevdev_disable_event_type()
        {
            ev_bits[Type] = faux;
        }
        template<ui32 Type>
        bool libevdev_enable_event_type()
        {
            if (!libevdev_has_event_type<Type>())
            {
                ev_bits[Type] = true;
                if constexpr (Type == EV_REP)
                {
                    libevdev_enable_event_code<EV_REP>(REP_DELAY);
                    libevdev_enable_event_code<EV_REP>(REP_PERIOD);
                }
                return true;
            }
            else return faux;
        }
        si32 libevdev_disable_property(ui32 prop)
        {
            if (prop < prop_bits.size())
            {
                prop_bits[prop] = faux;
                return 0;
            }
            return -1;
        }
        bool parse_udev_flag(view property)
        {
            return libinput_udev_prop(property) == "1";
        }
        ui32 evdev_device_get_udev_tags()
        {
            static constexpr auto evdev_udev_tag_matches = std::to_array<std::pair<view, ui32>>(
            {
                { "ID_INPUT",               EVDEV_UDEV_TAG_INPUT         },
                { "ID_INPUT_KEYBOARD",      EVDEV_UDEV_TAG_KEYBOARD      },
                { "ID_INPUT_KEY",           EVDEV_UDEV_TAG_KEYBOARD      },
                { "ID_INPUT_MOUSE",         EVDEV_UDEV_TAG_MOUSE         },
                { "ID_INPUT_TOUCHPAD",      EVDEV_UDEV_TAG_TOUCHPAD      },
                { "ID_INPUT_TOUCHSCREEN",   EVDEV_UDEV_TAG_TOUCHSCREEN   },
                { "ID_INPUT_TABLET",        EVDEV_UDEV_TAG_TABLET        },
                { "ID_INPUT_TABLET_PAD",    EVDEV_UDEV_TAG_TABLET_PAD    },
                { "ID_INPUT_JOYSTICK",      EVDEV_UDEV_TAG_JOYSTICK      },
                { "ID_INPUT_ACCELEROMETER", EVDEV_UDEV_TAG_ACCELEROMETER },
                { "ID_INPUT_POINTINGSTICK", EVDEV_UDEV_TAG_POINTINGSTICK },
                { "ID_INPUT_TRACKBALL",     EVDEV_UDEV_TAG_TRACKBALL     },
                { "ID_INPUT_SWITCH",        EVDEV_UDEV_TAG_SWITCH        },
            });
            auto tags = ui32{};
            for (auto [name, tag] : evdev_udev_tag_matches)
            {
                if (parse_udev_flag(name))
                {
                    tags |= tag;
                }
            }
            if ((tags & (EVDEV_UDEV_TAG_TABLET | EVDEV_UDEV_TAG_TOUCHPAD | EVDEV_UDEV_TAG_TOUCHSCREEN)) == EVDEV_UDEV_TAG_TABLET)
            {
                tags |= EVDEV_UDEV_TAG_PURETABLET;
            }
            return tags;
        }
        abs_info_t& libevdev_get_abs_info(ui32 code)
        {
            static auto empty_abs_info = abs_info_t{};
            auto ok = libevdev_has_event_code<EV_ABS>(code);
            return ok ? abs_values[code] : empty_abs_info;
        }
            si32& get_slot_ref(si32 slot, si32 axis)
            {
                return mt_slot_vals[slot * lixx::abs_mt_cnt + axis - lixx::abs_mt_min];
            }
        si32 libevdev_get_slot_value(ui32 slot, ui32 code)
        {
            auto ok = libevdev_has_event_code<EV_ABS>(code)
                        && num_slots >= 0 && slot < (ui32)num_slots
                        && code >= lixx::abs_mt_min && code <= lixx::abs_mt_max;
            return ok ? get_slot_ref(slot, code) : 0;
        }
        si32 libevdev_set_slot_value(ui32 slot, ui32 code, si32 value)
        {
            auto rc = -1;
            if (libevdev_has_event_code<EV_ABS>(code))
            {
                if (num_slots != -1 && slot < (ui32)num_slots)
                {
                    if (code >= lixx::abs_mt_min && code <= lixx::abs_mt_max)
                    {
                        if (code == ABS_MT_SLOT)
                        {
                            if (value < 0 || value >= num_slots) return rc;
                            current_slot = value;
                        }
                        get_slot_ref(slot, code) = value;
                        rc = 0;
                    }
                }
            }
            return rc;
        }
            void reset_tracking_ids()
            {
                if (num_slots != -1 && libevdev_has_event_code<EV_ABS>(ABS_MT_TRACKING_ID))
                {
                    for (auto slot = 0; slot < num_slots; slot++)
                    {
                        libevdev_set_slot_value(slot, ABS_MT_TRACKING_ID, -1);
                    }
                }
            }
            void init_slots()
            {
                mt_slot_vals.clear();
                num_slots = -1;
                if (!libevdev_has_event_code<EV_ABS>(ABS_RESERVED) && libevdev_has_event_code<EV_ABS>(ABS_MT_SLOT)) // If device has ABS_RESERVED it is not a real multitouch device.
                {
                    auto& absinfo = abs_values[ABS_MT_SLOT];
                    num_slots = absinfo.maximum + 1;
                    mt_slot_vals.resize(num_slots * lixx::abs_mt_cnt);
                    current_slot = absinfo.value;
                    reset_tracking_ids();
                }
            }
        template<ui32 Type, class T = si32>
        void libevdev_enable_event_code(ui32 code, T data = 0) // data type: abs_info_t or si32.
        {
            if constexpr (Type != EV_SYN)
            {
                if (libevdev_enable_event_type<Type>()
                 && set_bits_by_type<Type>(code))
                {
                    if constexpr (Type == EV_ABS)
                    {
                        abs_values[code] = data;
                             if (code == ABS_MT_SLOT       ) init_slots();
                        else if (code == ABS_MT_TRACKING_ID) reset_tracking_ids();
                    }
                    else if constexpr (Type == EV_REP)
                    {
                        rep_values[code] = data;
                    }
                }
            }
        }
        si32 libevdev_enable_property(ui32 prop)
        {
            if (prop < prop_bits.size())
            {
                prop_bits[prop] = true;
                return 0;
            }
            return -1;
        }
        si32 libevdev_get_abs_fuzz(ui32 code)
        {
            auto& absinfo = libevdev_get_abs_info(code);
            return absinfo.fuzz;
        }
        si32 libevdev_get_abs_maximum(ui32 code)
        {
            auto& absinfo = libevdev_get_abs_info(code);
            return absinfo.maximum;
        }
        si32 libevdev_get_abs_resolution(ui32 code)
        {
            auto& absinfo = libevdev_get_abs_info(code);
            return absinfo.resolution;
        }
        void libevdev_set_abs_fuzz(ui32 code, si32 val)
        {
            if (libevdev_has_event_code<EV_ABS>(code)) abs_values[code].fuzz = val;
        }
        void libevdev_set_abs_maximum(ui32 code, si32 val)
        {
            if (libevdev_has_event_code<EV_ABS>(code)) abs_values[code].maximum = val;
        }
        void libevdev_set_abs_resolution(ui32 code, si32 val)
        {
            if (libevdev_has_event_code<EV_ABS>(code)) abs_values[code].resolution = val;
        }
        void libevdev_set_abs_info(ui32 code, abs_info_t& absinfo)
        {
            if (libevdev_has_event_code<EV_ABS>(code))
            {
                abs_values[code] = absinfo;
            }
        }
        si32 libevdev_change_fd(fd_t new_fd)
        {
            if (initialized)
            {
                fd = new_fd;
                if (fd != os::invalid_fd)
                {
                    return ::ioctl(fd, EVIOCSCLOCKID, &lixx::clock_type) ? -errno : 0;
                }
                return 0;
            }
            return -1;
        }
        si32 libevdev_fetch_slot_value(ui32 slot, ui32 code, si32& value)
        {
            auto ok = libevdev_has_event_code<EV_ABS>(code) && num_slots >= 0 && slot < (ui32)num_slots;
            if (ok)
            {
                value = libevdev_get_slot_value(slot, code);
            }
            return ok;
        }
        void libevdev_reset()
        {
            initialized = faux;
            fd = os::invalid_fd;
            devname = {};
            phys = {};
            uniq = {};
            mt_slot_vals.clear();
            num_slots    = -1;
            current_slot = -1;
            sync_state = SYNC_NONE;

            prod_info = {};
            driver_version = {};

            prop_bits.reset();
            ev_bits.reset();
            abs_bits.reset();
            rel_bits.reset();
            key_bits.reset();
            led_bits.reset();
            msc_bits.reset();
             sw_bits.reset();
             ff_bits.reset();
            rep_bits.reset();
            snd_bits.reset();

            abs_values = {};
            rep_values = {};
            key_values.reset();
            led_values.reset();
            sw_values.reset();
        }
            bool sync_mt_state(std::vector<slot_change_state>& changes_out)
            {
                auto ok = true;
                auto limit = std::min<ui64>(lixx::max_slots, changes_out.size());
                auto changes = std::array<slot_change_state, lixx::max_slots>{};
                for (auto axis = lixx::abs_mt_min; axis <= lixx::abs_mt_max; axis++)
                {
                    if (axis != ABS_MT_SLOT && libevdev_has_event_code<EV_ABS>(axis))
                    {
                        auto mt_state = mt_sync_state{ .code = (ui32)axis };
                        ok = 0 <= ::ioctl(fd, EVIOCGMTSLOTS(sizeof(mt_state)), &mt_state);
                        if (!ok) break;
                        for (auto slot = 0u; slot < limit; slot++)
                        {
                            auto prev = get_slot_ref(slot, axis);
                            auto next = mt_state.val[slot];
                            if (axis == ABS_MT_TRACKING_ID)
                            {
                                auto& state = changes[slot].state;
                                state = prev == -1 && next != -1                 ? TOUCH_STARTED
                                      : prev != -1 && next == -1                 ? TOUCH_STOPPED
                                      : prev != -1 && next != -1 && prev == next ? TOUCH_ONGOING
                                      : prev != -1 && next != -1 && prev != next ? TOUCH_CHANGED
                                                                                 : TOUCH_OFF;
                            }
                            if (prev != next)
                            {
                                changes[slot].axes[axis] = true;
                                changes[slot].axes[ABS_MT_SLOT] = true;
                                get_slot_ref(slot, axis) = next;
                            }
                        }
                    }
                }
                if (ok)
                {
                    std::copy(changes.begin(), changes.begin() + limit, changes_out.begin());
                }
                return ok;
            }
            void init_event_queue()
            {
                auto max_event_count = 1; // SYN_REPORT is always enqueued.
                for_each_bits([&](auto const& bits)
                {
                    for (auto code = 0u; code < bits.size(); code++)
                    {
                        max_event_count += bits[code];
                    }
                });
                auto extra_slot_count = num_slots;
                if (extra_slot_count > 1)
                {
                    extra_slot_count--; // Exclude the first slot.
                    auto num_mt_axes = 0;
                    for (auto code = (ui32)ABS_MT_SLOT; code <= (ui32)ABS_MAX; code++)
                    {
                        if (libevdev_has_event_code<EV_ABS>(code))
                        {
                            num_mt_axes++;
                        }
                    }
                    max_event_count += num_mt_axes * extra_slot_count;
                }
                max_event_count = std::max<ui64>(lixx::min_queue_size, max_event_count * 2);
                queue.resize(max_event_count);
                queue_next = 0;
            }
        bool _libevdev_set_fd(fd_t new_fd)
        {
            auto ok = faux;
            auto trim = [](text& s){ s.erase(std::find(s.begin(), s.end(), '\0'), s.end()); return true; };
            libevdev_reset();
            devname.assign(256, '\0');
            phys.assign(256, '\0');
            uniq.assign(256, '\0');
            auto rc = (0 <= ::ioctl(new_fd, EVIOCSCLOCKID, &lixx::clock_type))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(0,      sizeof(ev_bits)),    &ev_bits       ))
                   && (0 <= ::ioctl(new_fd, EVIOCGNAME(devname.size() - 1),        devname.data() ))                    && trim(devname)
                   && (0 <= ::ioctl(new_fd, EVIOCGPHYS(phys.size() - 1),           phys.data()    ) || errno == ENOENT) && trim(phys)
                   && (0 <= ::ioctl(new_fd, EVIOCGUNIQ(uniq.size() - 1),           uniq.data()    ) || errno == ENOENT) && trim(uniq)
                   && (0 <= ::ioctl(new_fd, EVIOCGID,                              &prod_info     ))
                   && (0 <= ::ioctl(new_fd, EVIOCGVERSION,                         &driver_version))
                   && (0 <= ::ioctl(new_fd, EVIOCGPROP(       sizeof(prop_bits     )), &prop_bits         ) || errno == EINVAL) // The case of a kernel without a properties support.
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_REL, sizeof(rel_bits  )), &rel_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_ABS, sizeof(abs_bits  )), &abs_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_LED, sizeof(led_bits  )), &led_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_KEY, sizeof(key_bits  )), &key_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_SW,  sizeof(sw_bits   )), &sw_bits       ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_MSC, sizeof(msc_bits  )), &msc_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_FF,  sizeof(ff_bits   )), &ff_bits       ))
                   && (0 <= ::ioctl(new_fd, EVIOCGBIT(EV_SND, sizeof(snd_bits  )), &snd_bits      ))
                   && (0 <= ::ioctl(new_fd, EVIOCGKEY(        sizeof(key_values)), &key_values    ))
                   && (0 <= ::ioctl(new_fd, EVIOCGLED(        sizeof(led_values)), &led_values    ))
                   && (0 <= ::ioctl(new_fd, EVIOCGSW(         sizeof(sw_values )), &sw_values     ));
            if (rc && ev_bits[EV_REP]) // rep_bits are always set if EV_REP is set.
            {
                rep_bits.set();
                rc = 0 <= ::ioctl(new_fd, EVIOCGREP, rep_values.data());
            }
            if (rc) for (auto axis = ABS_X; axis <= ABS_MAX; axis++)
            {
                if (abs_bits[axis])
                {
                    auto absinfo = abs_info_t{};
                    rc = 0 <= ::ioctl(new_fd, EVIOCGABS(axis), &absinfo);
                    if (!rc) break;
                    if (axis == ABS_MT_TRACKING_ID && absinfo.maximum == absinfo.minimum) // Fix an invalid ABS_MT_TRACKING_ID range.
                    {
                        absinfo.minimum = -1;
                        absinfo.maximum = 0xFFFF;
                    }
                    abs_values[axis] = absinfo;
                }
            }
            if (rc)
            {
                ok = true;
                fd = new_fd;
                init_slots();
                if (num_slots != -1)
                {
                    auto tmp = std::vector<slot_change_state>(num_slots);
                    sync_mt_state(tmp);
                }
                init_event_queue();
            }
            else
            {
                libevdev_reset();
            }
            return ok;
        }
                        void init_event(input_event_t& ev, ui32 type, ui32 code, si32 value)
                        {
                            ev.input_event_sec  = last_event_time.tv_sec;
                            ev.input_event_usec = last_event_time.tv_usec;
                            ev.type             = (ui16)type;
                            ev.code             = (ui16)code;
                            ev.value            = value;
                        }
                    bool queue_push_event(ui32 type, ui32 code, si32 value)
                    {
                        if (queue_next < queue.size())
                        {
                            auto& ev = queue[queue_next++];
                            init_event(ev, type, code, value);
                            return true;
                        }
                        else return faux;
                    }
                void drain_events()
                {
                    static constexpr auto max_iterations = 8; // =EVDEV_BUF_PACKETS fromn kernel/drivers/input/evedev.c
                    auto nelem = ui64{};
                    auto iterations = 0;
                    queue_pop_multiple(queue_next);
                    do
                    {
                        auto rc = read_more_events();
                        if (rc == -EAGAIN || rc < 0) return;
                        nelem = queue_next;
                        queue_pop_multiple(nelem);
                    }
                    while (iterations++ < max_iterations && nelem >= queue.size());
                }
                void terminate_slots(std::vector<slot_change_state>& changes, si32& last_reported_slot)
                {
                    static constexpr auto map = std::to_array<ui32>(
                    {
                        BTN_TOOL_FINGER, BTN_TOOL_DOUBLETAP, BTN_TOOL_TRIPLETAP, BTN_TOOL_QUADTAP, BTN_TOOL_QUINTTAP
                    });
                    auto touches_stopped = faux;
                    auto prev_ntouches = 0;
                    auto next_ntouches = 0;
                    for (auto slot = 0; slot < num_slots;  slot++) // Get counters for the BTN_TOOL_* emulation.
                    {
                        auto& state = changes[slot].state;
                        if (state == TOUCH_CHANGED || state == TOUCH_STOPPED)
                        {
                            queue_push_event(EV_ABS, ABS_MT_SLOT, slot);
                            queue_push_event(EV_ABS, ABS_MT_TRACKING_ID, -1);
                            last_reported_slot = slot;
                            touches_stopped = true;
                            prev_ntouches++;
                        }
                        else if (state == TOUCH_ONGOING)
                        {
                            prev_ntouches++;
                            next_ntouches++;
                        }
                    }
                    if (touches_stopped) // Split the touches state: stopped + started if any.
                    {
                        if (prev_ntouches > 0 && prev_ntouches <= (si32)map.size())
                        {
                            auto ev = input_event_t{};
                            ev.type = EV_KEY;
                            ev.code = (ui16)map[prev_ntouches - 1];
                            ev.value = 0;
                            queue_push_event(ev.type, ev.code, ev.value);
                            update_key_state(ev);
                        }
                        if (next_ntouches > 0 && next_ntouches <= (si32)map.size())
                        {
                            auto ev = input_event_t{};
                            ev.type = EV_KEY;
                            ev.code = (ui16)map[next_ntouches - 1];
                            ev.value = 1;
                            queue_push_event(ev.type, ev.code, ev.value);
                            update_key_state(ev);
                        }
                        queue_push_event(EV_SYN, SYN_REPORT, 0);
                    }
                }
                si32 sync_key_state()
                {
                    auto keystate = std::bitset<KEY_CNT>{};
                    auto rc = ::ioctl(fd, EVIOCGKEY(sizeof(keystate)), &keystate);
                    if (rc < 0) return -errno;
                    for (auto i = 0; i < KEY_CNT; i++)
                    {
                        auto prev = key_values[i];
                        auto next = keystate[i];
                        if (prev != next)
                        {
                            queue_push_event(EV_KEY, i, next);
                        }
                    }
                    key_values = keystate;;
                    return 0;
                }
                si32 sync_led_state()
                {
                    auto ledstate = std::bitset<LED_CNT>{};
                    auto rc = ::ioctl(fd, EVIOCGLED(sizeof(ledstate)), ledstate);
                    if (rc < 0) return -errno;
                    for (auto i = 0; i < LED_CNT; i++)
                    {
                        auto prev = led_values[i];
                        auto next = ledstate[i];
                        if (prev != next)
                        {
                            queue_push_event(EV_LED, i, next);
                        }
                    }
                    led_values = ledstate;
                    return 0;
                }
                si32 sync_sw_state()
                {
                    auto swstate = std::bitset<SW_CNT>{};
                    auto rc = ::ioctl(fd, EVIOCGSW(sizeof(swstate)), swstate);
                    if (rc < 0) return -errno;
                    for (auto i = 0; i < SW_CNT; i++)
                    {
                        auto prev = sw_values[i];
                        auto next = swstate[i];
                        if (prev != next)
                        {
                            queue_push_event(EV_LED, i, next);
                        }
                    }
                    sw_values = swstate;
                    return 0;
                }
                si32 sync_abs_state()
                {
                    for (auto i = ABS_X; i < ABS_CNT; i++)
                    {
                        if ((i < lixx::abs_mt_min || i > lixx::abs_mt_max) && abs_bits[i])
                        {
                            auto absinfo = abs_info_t{};
                            auto rc = ::ioctl(fd, EVIOCGABS(i), &absinfo);
                            if (rc < 0) return -errno;
                            auto& cur_value = abs_values[i].value;
                            if (cur_value != absinfo.value)
                            {
                                queue_push_event(EV_ABS, i, absinfo.value);
                                cur_value = absinfo.value;
                            }
                        }
                    }
                    return 0;
                }
                si32 push_mt_sync_events(std::vector<slot_change_state>& changes, si32 last_reported_slot)
                {
                    for (auto slot = 0; slot < num_slots; slot++)
                    {
                        if (changes[slot].axes[ABS_MT_SLOT])
                        {
                            auto have_slot_event = faux;
                            for (auto axis = lixx::abs_mt_min; axis <= lixx::abs_mt_max; axis++)
                            {
                                if (axis != ABS_MT_SLOT && libevdev_has_event_code<EV_ABS>(axis) && changes[slot].axes[axis])
                                {
                                    if (axis != ABS_MT_TRACKING_ID || get_slot_ref(slot, axis) != -1) // ABS_MT_TRACKING_ID is already sent.
                                    if (!have_slot_event)
                                    {
                                        queue_push_event(EV_ABS, ABS_MT_SLOT, slot);
                                        last_reported_slot = slot;
                                        have_slot_event = true;
                                    }
                                    queue_push_event(EV_ABS, axis, get_slot_ref(slot, axis));
                                }
                            }
                        }
                    }
                    auto absinfo = abs_info_t{};
                    auto rc = ::ioctl(fd, EVIOCGABS(ABS_MT_SLOT), &absinfo);
                    if (rc < 0) return -errno;
                    current_slot = absinfo.value;
                    if (current_slot != last_reported_slot)
                    {
                        queue_push_event(EV_ABS, ABS_MT_SLOT, current_slot);
                    }
                    return 0;
                }
            si32 do_sync_state()
            {
                auto changes_count = num_slots > 0 ? num_slots : 1;
                slot_changes.assign(changes_count, {});
                drain_events();
                auto want_mt_sync = faux;
                auto last_reported_slot = 0;
                auto rc = 0;
                if (num_slots > -1 && libevdev_has_event_code<EV_ABS>(ABS_MT_SLOT))
                {
                    want_mt_sync = true;
                    rc = sync_mt_state(slot_changes);
                    if (rc == 0) terminate_slots(slot_changes, last_reported_slot);
                    else         want_mt_sync = faux;
                }
                if (           libevdev_has_event_type<EV_KEY>()) rc = sync_key_state();
                if (           libevdev_has_event_type<EV_LED>()) rc = sync_led_state();
                if (           libevdev_has_event_type<EV_SW >())  rc = sync_sw_state();
                if (rc == 0 && libevdev_has_event_type<EV_ABS>()) rc = sync_abs_state();
                if (rc == 0 && want_mt_sync) push_mt_sync_events(slot_changes, last_reported_slot);
                queue_nsync = queue_next;
                if (queue_nsync > 0)
                {
                    queue_push_event(EV_SYN, SYN_REPORT, 0);
                    queue_nsync++;
                }
                return rc;
            }
                si32 queue_pop_multiple(ui64 n)
                {
                    auto remaining = queue_next;
                    n = std::min(n, remaining);
                    if (n)
                    {
                        remaining -= n;
                        ::memmove(queue.data(), &queue[n], remaining * sizeof(input_event_t));
                        queue_next = remaining;
                    }
                    return n;
                }
                si32 queue_pop_front(input_event_t& ev)
                {
                    auto remaining = queue_next;
                    auto n = std::min(ui64{ 1 }, remaining);
                    if (remaining > 0)
                    {
                        remaining--;
                        ev = queue.front();
                        ::memmove(queue.data(), &queue[1], remaining * sizeof(ev));
                        queue_next = remaining;
                    }
                    return n;
                }
            si32 queue_shift(input_event_t& ev)
            {
                return queue_pop_front(ev) == 1 ? 0 : 1;
            }
                template<ui32 Type>
                si32 libevdev_event_is_code(input_event_t& ev, ui32 code)
                {
                    if (ev.type != Type)
                    {
                        return 0;
                    }
                    auto& bits = select_bits<Type>();
                    return code < bits.size() && ev.code == code;
                }
            event_filter_status sanitize_event(input_event_t& ev, sync_states sync_state)
            {
                if (num_slots > -1 && libevdev_event_is_code<EV_ABS>(ev, ABS_MT_SLOT) && (ev.value < 0 || ev.value >= num_slots))
                {
                    ev.value = num_slots - 1;
                    return EVENT_FILTER_MODIFIED; // An invalid slot index received.
                }
                if (sync_state == SYNC_NONE && num_slots > -1 && libevdev_event_is_code<EV_ABS>(ev, ABS_MT_TRACKING_ID)
                 && ((ev.value == -1 && get_slot_ref(current_slot, ABS_MT_TRACKING_ID) == -1)
                  || (ev.value != -1 && get_slot_ref(current_slot, ABS_MT_TRACKING_ID) != -1)))
                {
                    return EVENT_FILTER_DISCARD; // A double tracking ID received.
                }
                return EVENT_FILTER_NONE;
            }
                si32 update_key_state(input_event_t& e)
                {
                    if (!libevdev_has_event_type<EV_KEY>() || e.code > KEY_MAX) return 1;
                    else
                    {
                        key_values[e.code] = e.value != 0;
                        return 0;
                    }
                }
                    si32 update_mt_state(input_event_t& e)
                    {
                        if (current_slot == -1) return 1;
                        else
                        {
                            if (e.code == ABS_MT_SLOT && num_slots > -1)
                            {
                                current_slot = e.value;
                                for (auto i = ABS_MT_SLOT + 1; i <= lixx::abs_mt_max; i++)
                                {
                                    if (libevdev_has_event_code<EV_ABS>(i))
                                    {
                                        abs_values[i].value = get_slot_ref(current_slot, i);
                                    }
                                }
                            }
                            else
                            {
                                get_slot_ref(current_slot, e.code) = e.value;
                            }
                            return 0;
                        }
                    }
                si32 update_abs_state(input_event_t& e)
                {
                    if (!libevdev_has_event_type<EV_ABS>() || e.code > ABS_MAX) return 1;
                    else
                    {
                        if (e.code >= lixx::abs_mt_min && e.code <= lixx::abs_mt_max)
                        {
                            update_mt_state(e);
                        }
                        abs_values[e.code].value = e.value;
                        return 0;
                    }
                }
                si32 update_led_state(input_event_t& e)
                {
                    if (!libevdev_has_event_type<EV_LED>() || e.code > LED_MAX) return 1;
                    else
                    {
                        led_values[e.code] = e.value != 0;
                        return 0;
                    }
                }
                si32 update_sw_state(input_event_t& e)
                {
                    if (!libevdev_has_event_type<EV_SW>() || e.code > SW_MAX) return 1;
                    else
                    {
                        sw_values[e.code] = e.value != 0;
                        return 0;
                    }
                }
            si32 update_state(input_event_t& e)
            {
                auto rc = 0;
                     if (e.type == EV_KEY) rc = update_key_state(e);
                else if (e.type == EV_ABS) rc = update_abs_state(e);
                else if (e.type == EV_LED) rc = update_led_state(e);
                else if (e.type == EV_SW ) rc = update_sw_state(e);
                last_event_time.tv_sec  = e.input_event_sec;
                last_event_time.tv_usec = e.input_event_usec;
                return rc;
            }
            si32 read_more_events()
            {
                if (queue.size() && queue_next != queue.size())
                {
                    auto free_elem = queue.size() - queue_next;
                    auto len = ::read(fd, &queue[queue_next], free_elem * sizeof(input_event_t));
                    if (len < 0) return -errno;
                    if (len > 0)
                    {
                        if (len % sizeof(input_event_t) != 0) return -EINVAL;
                        auto nev = len / sizeof(input_event_t);
                        auto nelem = queue_next + nev;
                        if (nelem <= queue.size())
                        {
                            queue_next = nelem;
                        }
                    }
                }
                return 0;
            }
        si32 libevdev_next_event(ui32 flags, input_event_t& ev)
        {
            auto rc = evdev::success;
            if (!initialized || fd == os::invalid_fd)
            {
                return -EBADF;
            }
            if (!(flags & lixx::valid_flags))
            {
                return -EINVAL;
            }
            if (flags & LIBEVDEV_READ_FLAG_SYNC)
            {
                if (sync_state == SYNC_NEEDED)
                {
                    rc = do_sync_state();
                    if (rc != evdev::success) return rc;
                    sync_state = SYNC_IN_PROGRESS;
                }
                if (queue_nsync == 0)
                {
                    sync_state = SYNC_NONE;
                    return -EAGAIN;
                }
            }
            else if (sync_state != SYNC_NONE)
            {
                auto ev = input_event_t{};
                while (queue_shift(ev) == 0) // Update state for all events.
                {
                    queue_nsync--;
                    if (sanitize_event(ev, sync_state) != EVENT_FILTER_DISCARD)
                    {
                        update_state(ev);
                    }
                }
                sync_state = SYNC_NONE;
            }
            auto filter_status = event_filter_status{};
            do
            {
                if (queue_next == 0)
                {
                    rc = read_more_events();
                    if (rc < 0 && rc != -EAGAIN) return rc;
                }
                if (flags & LIBEVDEV_READ_FLAG_FORCE_SYNC)
                {
                    sync_state = SYNC_NEEDED;
                    return evdev::sync;
                }
                if (queue_shift(ev) != 0)
                {
                    return -EAGAIN;
                }
                filter_status = sanitize_event(ev, sync_state);
                if (filter_status != EVENT_FILTER_DISCARD)
                {
                    update_state(ev);
                }
            }
            while (filter_status == EVENT_FILTER_DISCARD);
            rc = evdev::success;
            if (ev.type == EV_SYN && ev.code == SYN_DROPPED)
            {
                sync_state = SYNC_NEEDED;
                rc = evdev::sync;
            }
            if (flags & LIBEVDEV_READ_FLAG_SYNC && queue_nsync > 0)
            {
                queue_nsync--;
                rc = evdev::sync;
                if (queue_nsync == 0)
                {
                    sync_state = SYNC_NONE;
                }
            }
            return rc;
        }
        void evdev_disable_accelerometer_axes()
        {
            libevdev_disable_event_code<EV_ABS>(ABS_X);
            libevdev_disable_event_code<EV_ABS>(ABS_Y);
            libevdev_disable_event_code<EV_ABS>(ABS_Z);
            libevdev_disable_event_code<EV_ABS>(REL_X);
            libevdev_disable_event_code<EV_ABS>(REL_Y);
            libevdev_disable_event_code<EV_ABS>(REL_Z);
        }
        bool evdev_check_min_max(ui32 code)
        {
            if (auto& absinfo = libevdev_get_abs_info(code))
            if (absinfo.minimum == absinfo.maximum)
            {
                // Some devices have a sort-of legitimate min/max of 0 for ABS_MISC and above (e.g. Roccat Kone XTD). Don't ignore them, simply disable the axes so we won't get events, we don't know what to do with them anyway.
                if (absinfo.minimum == 0 && code >= ABS_MISC && code < ABS_MT_SLOT)
                {
                    log("Disabling EV_ABS %#x% on device (min == max == 0)", code);
                    libevdev_disable_event_code<EV_ABS>(code);
                }
                else
                {
                    log("Device has min == max on EV_ABS 'code=%%'", code);
                    return faux;
                }
            }
            return true;
        }
        bool evdev_is_fake_mt_device()
        {
            return libevdev_has_event_code<EV_ABS>(ABS_MT_SLOT) && num_slots == -1;
        }
        bool evdev_reject_device()
        {
            if (libevdev_has_event_code<EV_ABS>(ABS_X) ^ libevdev_has_event_code<EV_ABS>(ABS_Y))
            {
                return true;
            }
            if (libevdev_has_event_code<EV_REL>(REL_X) ^ libevdev_has_event_code<EV_REL>(REL_Y))
            {
                return true;
            }
            if (!evdev_is_fake_mt_device() && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_X) ^ libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_Y))
            {
                return true;
            }
            if (libevdev_has_event_code<EV_ABS>(ABS_X) || is_mt)
            {
                if (!!abs.absinfo_x != !!abs.absinfo_y)
                {
                    log("Device has only x or y resolution");
                    return true;
                }
            }
            for (auto code = 0u; code < ABS_CNT; code++)
            {
                if (code != ABS_MISC && code != ABS_MT_SLOT && code != ABS_MT_TOOL_TYPE && !evdev_check_min_max(code))
                {
                    return true;
                }
            }
            return faux;
        }
        void evdev_fix_android_mt() // If !evdev_is_fake_mt_device().
        {
            if (!libevdev_has_event_code<EV_ABS>(ABS_X) && !libevdev_has_event_code<EV_ABS>(ABS_Y))
            if (libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_X) && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_Y))
            {
                libevdev_enable_event_code<EV_ABS>(ABS_X, libevdev_get_abs_info(ABS_MT_POSITION_X));
                libevdev_enable_event_code<EV_ABS>(ABS_Y, libevdev_get_abs_info(ABS_MT_POSITION_Y));
            }
        }
        template<class Li>
        bool evdev_read_attr_res_prop(Li& li, si32_coor& res)
        {
            if (auto q = li.quirks_fetch_for_device(*this))
            {
                auto dim = si32_coor{};
                if (q->quirks_get(QUIRK_ATTR_RESOLUTION_HINT, dim))
                {
                    res = dim;
                    return true;
                }
            }
            return faux;
        }
        template<class Li>
        bool evdev_read_attr_size_prop(Li& li, si32_coor& size)
        {
            if (auto q = li.quirks_fetch_for_device(*this))
            {
                auto dim = si32_coor{};
                if (q->quirks_get(QUIRK_ATTR_SIZE_HINT, dim))
                {
                    size = dim;
                    return true;
                }
            }
            return faux;
        }
        template<class Li>
        void evdev_fix_abs_resolution(Li& li)
        {
            static constexpr auto fake_resolution = dot_11;
            abs.is_fake_resolution = faux;
            if (!abs.absinfo_x || !abs.absinfo_y) // Note: we *do not* override resolutions if provided by the kernel. If a device needs this, add it to 60-evdev.hwdb. The libinput property is only for general size hints where we can make educated guesses but don't know better.
            {
                auto mm = si32_coor{};
                auto res = fake_resolution;
                if (!evdev_read_attr_res_prop(li, res) && evdev_read_attr_size_prop(li, mm))
                {
                    res.x = abs.absinfo_x.absinfo_range() / mm.x;
                    res.y = abs.absinfo_y.absinfo_range() / mm.y;
                }
                abs.absinfo_x.resolution = res.x;
                abs.absinfo_y.resolution = res.y;
                abs.is_fake_resolution = res.x == fake_resolution.x;
            }
        }
        si32 evdev_read_fuzz_prop(ui32 code)
        {
            auto fuzz = 0;
            char name[32];
            auto rc = ::snprintf(name, sizeof(name), "LIBINPUT_FUZZ_%02x", code);
            if (rc == -1)
            {
                return 0;
            }
            auto prop = libinput_udev_prop(name);
            if (prop)
            {
                if (auto v = utf::to_int(prop); v && v.value() >= 0)
                {
                    fuzz = v.value();
                }
                else
                {
                    log("invalid LIBINPUT_FUZZ property value: %s%", prop);
                    return 0;
                }
            }
            // The udev callout should have set the kernel fuzz to zero. If the kernel fuzz is nonzero, something has gone wrong there, so let's complain but still use a fuzz of zero for our view of the device. Otherwise, the kernel will use the nonzero fuzz, we then use the same fuzz on top of the pre-fuzzed data and that leads to unresponsive behavior.
            auto& abs_info = libevdev_get_abs_info(code);
            if (abs_info.fuzz == 0)
            {
                return fuzz;
            }
            if (prop) log("kernel fuzz of %d% even with LIBINPUT_FUZZ_%02x% present", abs_info.fuzz, code);
            else      log("kernel fuzz of %d% but LIBINPUT_FUZZ_%02x% is missing", abs_info.fuzz, code);
            return 0;
        }
        template<class Li>
        void evdev_extract_abs_axes(Li& li, ui32 udev_tags)
        {
            evdev_fix_abs_resolution(li);
            if (is_mt)
            {
                if (auto fuzz_x = evdev_read_fuzz_prop(ABS_MT_POSITION_X))
                {
                    libevdev_set_abs_fuzz(ABS_MT_POSITION_X, fuzz_x);
                }
                if (auto fuzz_y = evdev_read_fuzz_prop(ABS_MT_POSITION_Y))
                {
                    libevdev_set_abs_fuzz(ABS_MT_POSITION_Y, fuzz_y);
                }
            }
            else
            {
                if (udev_tags & (EVDEV_UDEV_TAG_TOUCHPAD | EVDEV_UDEV_TAG_TOUCHSCREEN))
                {
                    auto fuzz_x = evdev_read_fuzz_prop(ABS_X);
                    libevdev_set_abs_fuzz(ABS_X, fuzz_x);
                    auto fuzz_y = evdev_read_fuzz_prop(ABS_Y);
                    libevdev_set_abs_fuzz(ABS_Y, fuzz_y);
                }
            }
            abs.dimensions.x = std::abs((si32)abs.absinfo_x.absinfo_range());
            abs.dimensions.y = std::abs((si32)abs.absinfo_y.absinfo_range());
        }
        template<class Li>
        bool evdev_device_has_model_quirk(Li& li, quirk model_quirk)
        {
            auto result = faux;
            assert(quirks_t::quirk_get_name(model_quirk) != nullptr);
            if (auto q = li.quirks_fetch_for_device(*this))
            {
                q->quirks_get(model_quirk, result);
            }
            return result;
        }
        template<class Li>
        void evdev_pre_configure_model_quirks(Li& li)
        {
            auto prop = text{};
            //auto is_virtual = faux;
            // Touchpad claims to have 4 slots but only ever sends 2.   https://bugs.freedesktop.org/show_bug.cgi?id=98100
            if (evdev_device_has_model_quirk(li, QUIRK_MODEL_HP_ZBOOK_STUDIO_G3))
            {
                libevdev_set_abs_maximum(ABS_MT_SLOT, 1);
            }
            // Generally we don't care about MSC_TIMESTAMP and it can cause unnecessary wakeups but on some devices we need to watch it for pointer jumps.
            auto q = li.quirks_fetch_for_device(*this);
            if (!q || !q->quirks_get(QUIRK_ATTR_MSC_TIMESTAMP, prop) || "watch"sv != prop)
            {
                libevdev_disable_event_code<EV_MSC>(MSC_TIMESTAMP);
            }
            if (q)
            {
                auto t = quirk_tuples{};
                if (q->quirks_get(QUIRK_ATTR_EVENT_CODE, t))
                {
                    for (auto i = 0u; i < t.ntuples; i++)
                    {
                        auto absinfo = abs_info_t{};
                        absinfo.minimum = 0;
                        absinfo.maximum = 1;
                        auto type = t.tuples[i].first;
                        auto code = t.tuples[i].second;
                        auto stat = t.tuples[i].third;
                             if (type == EV_ABS) set_event_type_code<EV_ABS>(stat, code, absinfo);
                        else if (type == EV_REL) set_event_type_code<EV_REL>(stat, code);
                        else if (type == EV_KEY) set_event_type_code<EV_KEY>(stat, code);
                        else if (type == EV_REP) set_event_type_code<EV_REP>(stat, code);
                        else if (type == EV_MSC) set_event_type_code<EV_MSC>(stat, code);
                        else if (type == EV_LED) set_event_type_code<EV_LED>(stat, code);
                        else if (type == EV_SND) set_event_type_code<EV_SND>(stat, code);
                        else if (type == EV_SW ) set_event_type_code<EV_SW >(stat, code);
                        else if (type == EV_FF ) set_event_type_code<EV_FF >(stat, code);
                        log("quirks: %s% %s% ('type=%% code=%%')", stat ? "enabling" : "disabling", libevdev_event_type_get_name(type), type, code);
                    }
                }
                if (q->quirks_get(QUIRK_ATTR_INPUT_PROP, t))
                {
                    for (auto i = 0u; i < t.ntuples; i++)
                    {
                        auto p = (ui32)t.tuples[i].first;
                        auto enable = t.tuples[i].second;
                        if (enable)
                        {
                            libevdev_enable_property(p);
                        }
                        else
                        {
                            #if HAVE_LIBEVDEV_DISABLE_PROPERTY
                            libevdev_disable_property(p);
                            #else
                            log("quirks: a quirk for this device requires newer libevdev than installed");
                            #endif
                        }
                        log("quirks: %s% %s% (%#x%)", enable ? "enabling" : "disabling", libevdev_property_get_name(p), p);
                    }
                }
                //if (!q->quirks_get(QUIRK_ATTR_IS_VIRTUAL, is_virtual))
                //{
                //    is_virtual = !::getenv("LIBINPUT_RUNNING_TEST_SUITE") && libinput_ud_device_is_virtual();
                //}
                //if (is_virtual)
                //{
                //    device_tags |= EVDEV_TAG_VIRTUAL;
                //}
            }
        }
        template<class Li>
        void evdev_read_model_flags(Li& li)
        {
            #define X(name) { QUIRK_MODEL_##name, EVDEV_MODEL_##name }
            static constexpr auto model_map = std::to_array<std::pair<lixx::quirk, libinput_device_model>>(
            {
                X(WACOM_TOUCHPAD           ),
                X(SYNAPTICS_SERIAL_TOUCHPAD),
                X(ALPS_SERIAL_TOUCHPAD     ),
                X(LENOVO_T450_TOUCHPAD     ),
                X(TRACKBALL                ),
                X(APPLE_TOUCHPAD_ONEBUTTON ),
                X(LENOVO_SCROLLPOINT       ),
            });
            #undef X
            auto all_model_flags = 0u;
            if (auto q = li.quirks_fetch_for_device(*this))
            {
                for (auto [quirk, model] : model_map)
                {
                    auto is_set = faux;
                    assert(!(all_model_flags & model)); // Check for flag re-use.
                    all_model_flags |= model;
                    if (q->quirks_get(quirk, is_set))
                    {
                        if (is_set)
                        {
                            log("tagged as %s%", quirks_t::quirk_get_name(quirk));
                            model_flags |= model;
                        }
                        else
                        {
                            log("untagged as %s%", quirks_t::quirk_get_name(quirk));
                            model_flags &= ~model;
                        }
                    }
                }
            }
            if (parse_udev_flag("ID_INPUT_TRACKBALL"))
            {
                log("tagged as trackball");
                model_flags |= EVDEV_MODEL_TRACKBALL;
            }
            // Device is 6 years old at the time of writing this and this was one of the few udev properties that wasn't reserved for private usage, so we need to keep this for backwards compat.
            if (parse_udev_flag("LIBINPUT_MODEL_LENOVO_X220_TOUCHPAD_FW81"))
            {
                log("tagged as trackball");
                model_flags |= EVDEV_MODEL_LENOVO_X220_TOUCHPAD_FW81;
            }
            if (parse_udev_flag("LIBINPUT_TEST_DEVICE"))
            {
                log("is a test device");
                model_flags |= EVDEV_MODEL_TEST_DEVICE;
            }
        }
        template<ui32 Type, class T = si32>
        void set_event_type_code(bool enable, ui32 code, T data = 0)
        {
            if (code == lixx::event_code_undefined)
            {
                if (enable) libevdev_enable_event_type<Type>();
                else        libevdev_disable_event_type<Type>();
            }
            else
            {
                if (enable) libevdev_enable_event_code<Type>(code, data);
                else        libevdev_disable_event_code<Type>(code);
            }
        }
        si32 parse_mouse_wheel_click_angle_property(qiew prop)
        {
            auto angle = 0;
            auto v = utf::to_int<si32>(prop);
            if (v && std::abs(v.value()) <= 360)
            {
                return angle = v.value();
            }
            return angle;
        }
        bool evdev_read_wheel_click_count_prop(qiew prop_name, fp64& angle)
        {
            angle = lixx::default_wheel_click_angle;
            if (auto prop = libinput_udev_prop(prop_name))
            {
                if (auto val = parse_mouse_wheel_click_angle_property(prop))
                {
                    angle = 360.0 / val;
                    return true;
                }
                log("mouse wheel click count is present but invalid, using %d% degrees for angle instead instead", lixx::default_wheel_click_angle);
            }
            return faux;
        }
        bool evdev_read_wheel_click_angle_prop(view prop_name, fp64& angle)
        {
            angle = lixx::default_wheel_click_angle;
            if (auto prop = libinput_udev_prop(prop_name))
            {
                if (auto val = parse_mouse_wheel_click_angle_property(prop))
                {
                    angle = val;
                    return true;
                }
                log("mouse wheel click angle is present but invalid, using %d% degrees instead", lixx::default_wheel_click_angle);
            }
            return faux;
        }
        fp64_coor evdev_read_wheel_click_props()
        {
            auto angles = fp64_coor{};
            if (evdev_read_wheel_click_count_prop("MOUSE_WHEEL_CLICK_COUNT", angles.y) // *_CLICK_COUNT should override *_CLICK_ANGLE.
             || evdev_read_wheel_click_angle_prop("MOUSE_WHEEL_CLICK_ANGLE", angles.y))
            {
                log("wheel: vertical click angle: %.2f%", angles.y);
            }
            if (evdev_read_wheel_click_count_prop("MOUSE_WHEEL_CLICK_COUNT_HORIZONTAL", angles.x)
             || evdev_read_wheel_click_angle_prop("MOUSE_WHEEL_CLICK_ANGLE_HORIZONTAL", angles.x))
            {
                log("wheel: horizontal click angle: %.2f%", angles.x);
            }
            else
            {
                angles.x = angles.y;
            }
            return angles;
        }
        auto evdev_device_get_size()
        {
            auto w = 0.0;
            auto h = 0.0;
            auto has_size = abs.absinfo_x && abs.absinfo_y
                        && (abs.absinfo_x.minimum != 0 || abs.absinfo_x.maximum != 1)
                        && (abs.absinfo_y.minimum != 0 || abs.absinfo_y.maximum != 1)
                        && !abs.is_fake_resolution
                        && abs.absinfo_x.resolution && abs.absinfo_y.resolution;
            if (has_size)
            {
                w = abs.absinfo_x.absinfo_convert_to_mm(abs.absinfo_x.maximum);
                h = abs.absinfo_y.absinfo_convert_to_mm(abs.absinfo_y.maximum);
            }
            return std::pair{ w, h };
        }
        bool totem_reject_device()
        {
            auto [w, h] = evdev_device_get_size();
            auto has_size       = w && h;
            auto has_slot       = libevdev_has_event_code<EV_ABS>(ABS_MT_SLOT);
            auto has_xy         = libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_X) && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_Y);
            auto has_tool_dial  = libevdev_has_event_code<EV_ABS>(ABS_MT_TOOL_TYPE) && libevdev_get_abs_maximum(ABS_MT_TOOL_TYPE) >= MT_TOOL_DIAL;
            auto has_touch_size = libevdev_get_abs_resolution(ABS_MT_TOUCH_MAJOR) > 0 || libevdev_get_abs_resolution(ABS_MT_TOUCH_MINOR) > 0;
            auto ok = has_xy && has_slot && has_tool_dial && has_size && has_touch_size;
            if (!ok) log("missing totem capabilities:%s%%s%%s%%s%%s%. Ignoring this device.", has_xy ? "" : " xy", has_slot ? "" : " slot", has_tool_dial ? "" : " dial", has_size ? "" : " resolutions", has_touch_size ? "" : " touch-size");
            return !ok;
        }
    };

    struct quirks_context_t
    {
        text                    dmi2;
        text                    dt2;
        std::list<section_sptr> sections;
        std::list<quirks_sptr>  quirk_list; // List of quirks handed to libinput, just for bookkeeping.
        bool                    initialized{};

        quirks_context_t()
        {
            auto quirks_pir_ptr = ::getenv("LIBINPUT_QUIRKS_DIR");
            auto data_path = text{};
            auto override_file = text{};
            if (!quirks_pir_ptr)
            {
                data_path     = ""s;//LIBINPUT_QUIRKS_DIR;
                override_file = ""s;//LIBINPUT_QUIRKS_OVERRIDE_FILE;
            }
            else
            {
                data_path = text{ quirks_pir_ptr };
            }
            dmi2 = init_dmi();
            dt2  = init_dt();
            initialized = (!dmi2.empty() || !dt2.empty())
                        && parse_files(data_path)
                        && (override_file.empty() || parse_file(override_file));
        }
        operator bool () const { return initialized; }

        match_sptr match_new(ud_device_t& ud_device)
        {
            auto m = ptr::shared<match_t>();
            if (dmi2.size())
            {
                m->dmi2 = dmi2;
                m->bits |= M_DMI;
            }
            if (dt2.size())
            {
                m->dt2 = dt2;
                m->bits |= M_DT;
            }
            if (auto str = ud_device.libinput_udev_prop("NAME"))
            {
                if (str.size() > 1 && str.front() == '"' && str.back() == '"') // Strip quotes.
                {
                    str.pop_front();
                    str.pop_back();
                }
                m->name2 = str;
                m->bits |= M_NAME;
            }
            if (auto str = ud_device.libinput_udev_prop("UNIQ"))
            {
                if (str.size() > 1 && str.front() == '"' && str.back() == '"') // Strip quotes.
                {
                    str.pop_front();
                    str.pop_back();
                }
                m->uniq2 = str;
                m->bits |= M_UNIQ;
            }
            if (auto str = ud_device.libinput_udev_prop("PRODUCT"))
            {
                auto product = 0u;
                auto vendor = 0u;
                auto bus = 0u;
                auto version = 0u;
                if (::sscanf(str.data(), "%x/%x/%x/%x", &bus, &vendor, &product, &version) == 4) // ID_VENDOR_ID/ID_PRODUCT_ID/ID_BUS aren't filled in for virtual devices so we have to resort to PRODUCT.
                {
                    m->product[0] = product;
                    m->product[1] = 0;
                    m->vendor = vendor;
                    m->version = version;
                    m->bits |= M_PID | M_VID | M_VERSION;
                    switch (bus)
                    {
                        case BUS_USB:       m->bus = BT_USB;       m->bits |= M_BUS; break;
                        case BUS_BLUETOOTH: m->bus = BT_BLUETOOTH; m->bits |= M_BUS; break;
                        case BUS_I8042:     m->bus = BT_PS2;       m->bits |= M_BUS; break;
                        case BUS_RMI:       m->bus = BT_RMI;       m->bits |= M_BUS; break;
                        case BUS_I2C:       m->bus = BT_I2C;       m->bits |= M_BUS; break;
                        case BUS_SPI:       m->bus = BT_SPI;       m->bits |= M_BUS; break;
                        default: break;
                    }
                }
            }
            static constexpr auto mappings = std::to_array<std::pair<view, ui32>>(
            {
                { "ID_INPUT_MOUSE"        , UDEV_MOUSE         },
                { "ID_INPUT_POINTINGSTICK", UDEV_POINTINGSTICK },
                { "ID_INPUT_TOUCHPAD"     , UDEV_TOUCHPAD      },
                { "ID_INPUT_TABLET"       , UDEV_TABLET        },
                { "ID_INPUT_TABLET_PAD"   , UDEV_TABLET_PAD    },
                { "ID_INPUT_JOYSTICK"     , UDEV_JOYSTICK      },
                { "ID_INPUT_KEYBOARD"     , UDEV_KEYBOARD      },
                { "ID_INPUT_KEY"          , UDEV_KEYBOARD      },
            });
            for (auto [prop, flag] : mappings)
            {
                if (ud_device.libinput_udev_prop(prop))
                {
                    m->ud_type |= flag;
                }
            }
            m->bits |= M_UDEV_TYPE;
            return m;
        }
        static bool read_uevent(qiew filepath, auto proc)
        {
            auto buffer = std::array<char, 4096>{};
            auto f = std::ifstream{ filepath, std::ios::binary };
            if (f.is_open())
            {
                f.read(buffer.data(), buffer.size());
                auto data = qiew{ buffer.data(), (size_t)f.gcount() };
                auto lines = utf::split<true>(data, "\n");
                for (auto l : lines)
                {
                    auto p = l.find('=');
                    if (p != text::npos)
                    {
                        auto prop_name  = l.substr(0, p);
                        auto prop_value = l.substr(p + 1);
                        proc(utf::to_upper(prop_name), utf::remove_quotes(prop_value));
                    }
                }
                return true;
            }
            else
            {
                log("Failed to open %filepath%, errno=%%", filepath, errno);
            }
            return faux;
        }
        text init_dmi() // Desktop Management Interface.
        {
            if (::getenv("LIBINPUT_RUNNING_TEST_SUITE")) return "dmi:";
            #if defined(__linux__)
                auto modalias = "dmi:*"s;
                auto dmi_uevent_file = "/sys/devices/virtual/dmi/id/uevent";
                read_uevent(dmi_uevent_file, [&](qiew prop_name, qiew prop_value)
                {
                    if (prop_name == "MODALIAS")
                    {
                        modalias = prop_value;
                    }
                });
                return modalias;
            #elif defined(__FreeBSD__)
                auto buf = std::array<char, KENV_MVALLEN + 1>{};
                auto bios_vendor      = -1 != ::kenv(KENV_GET, "smbios.bios.vendor"    , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto bios_version     = -1 != ::kenv(KENV_GET, "smbios.bios.version"   , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto bios_reldate     = -1 != ::kenv(KENV_GET, "smbios.bios.reldate"   , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto system_maker     = -1 != ::kenv(KENV_GET, "smbios.system.maker"   , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto system_product   = -1 != ::kenv(KENV_GET, "smbios.system.product" , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto system_version   = -1 != ::kenv(KENV_GET, "smbios.system.version" , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto planar_maker     = -1 != ::kenv(KENV_GET, "smbios.planar.maker"   , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto planar_product   = -1 != ::kenv(KENV_GET, "smbios.planar.product" , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto planar_version   = -1 != ::kenv(KENV_GET, "smbios.planar.version" , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto chassis_vendor   = -1 != ::kenv(KENV_GET, "smbios.chassis.vendor" , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto chassis_version  = -1 != ::kenv(KENV_GET, "smbios.chassis.version", buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto chassis_type     = -1 != ::kenv(KENV_GET, "smbios.chassis.type"   , buf.data(), buf.size()) ? text{ buf.data() } : ""s;
                auto chassis_type_num =   strcmp(chassis_type, "Desktop"    ) == 0 ? 0x3
                                        : strcmp(chassis_type, "Portable"   ) == 0 ? 0x8
                                        : strcmp(chassis_type, "Laptop"     ) == 0 ? 0x9
                                        : strcmp(chassis_type, "Notebook"   ) == 0 ? 0xA
                                        : strcmp(chassis_type, "Tablet"     ) == 0 ? 0x1E
                                        : strcmp(chassis_type, "Convertible") == 0 ? 0x1F
                                        : strcmp(chassis_type, "Detachable" ) == 0 ? 0x20
                                                                                    : 0x2;
                auto modalias = utf::fprintf("dmi:bvn%s%:bvr%s%:bd%s%:svn%s%:pn%s%:pvr%s%:rvn%s%:rn%s%:rvr%s%:cvn%s%:ct%d%:cvr%s%:",
                    bios_vendor, bios_version, bios_reldate, system_maker, system_product,
                    system_version, planar_maker, planar_product, planar_version, chassis_vendor,
                    chassis_type_num, chassis_version);
                return modalias;
            #else
                return "dmi:*"s;
            #endif
        }
        text init_dt() // Device Tree.
        {
            auto copy = text{};
            if (!::getenv("LIBINPUT_RUNNING_TEST_SUITE"))
            {
                auto filepath = "/sys/firmware/devicetree/base/compatible";
                auto buffer = std::array<char, 4096>{};
                auto f = std::ifstream{ filepath, std::ios::binary };
                if (f.is_open())
                {
                    f.read(buffer.data(), buffer.size());
                    copy = buffer.data(); // devicetree/base/compatible has multiple null-terminated entries but we only care about the first one here.
                }
            }
            return copy;
        }
        static bool strneq(char const* str1, char const* str2, si32 n)
        {
            return str1 && str2 ? ::strncmp(str1, str2, n) == 0 : str1 == str2;
        }
        static bool strendswith(char const* str, char const* suffix)
        {
            if (str == nullptr) return faux;
            auto slen = strlen(str);
            auto suffixlen = strlen(suffix);
            if (slen == 0 || suffixlen == 0 || suffixlen > slen) return faux;
            auto offset = slen - suffixlen;
            return strneq(&str[offset], suffix, suffixlen);
        }
        static si32 is_data_file(::dirent const* dir)
        {
            return strendswith(dir->d_name, ".quirks");
        }
        static si32 libinput_strverscmp(char const* l0, char const* r0)
        {
            auto l = (byte const*)l0;
            auto r = (byte const*)r0;
            auto i = 0ul;
            auto dp = 0ul;
            auto j = 0ul;
            auto z = 1;
            for (dp = i = 0; l[i] == r[i]; i++) // Find maximal matching prefix and track its maximal digit suffix and whether those digits are all zeros.
            {
                auto c = l[i];
                if (!c) return 0;
                if (!isdigit(c))
                {
                    dp = i + 1;
                    z = 1;
                }
                else if (c != '0')
                {
                    z=0;
                }
            }
            if (l[dp] != '0' && r[dp] != '0') // If we're not looking at a digit sequence that began with a zero, longest digit string is greater.
            {
                for (j = i; isdigit(l[j]); j++)
                {
                    if (!isdigit(r[j])) return 1;
                }
                if (isdigit(r[j])) return -1;
            }
            else if (z && dp < i && (isdigit(l[i]) || isdigit(r[i]))) // Otherwise, if common prefix of digit sequence is all zeros, digits order less than non-digits.
            {
                return (byte)(l[i] - '0') - (byte)(r[i] - '0');
            }
            return l[i] - r[i];
        }
        static si32 versionsort(::dirent const**a, ::dirent const**b)
        {
            return libinput_strverscmp((*a)->d_name, (*b)->d_name);
        }
        bool parse_match(section_sptr s, view key, view value) // Parse a MatchFooBar=banana line.
        {
            auto rc = true;
            auto check_set_bits = [&](auto field)
            {
                if (s->match->bits & field) rc = faux;
                else                        s->match->bits |= field;
                return rc;
            };
            assert(value.size() >= 1);
            if (key == "MatchName")
            {
                if (check_set_bits(M_NAME))
                {
                    s->match->name2 = value;
                }
            }
            else if (key == "MatchUniq")
            {
                if (check_set_bits(M_UNIQ))
                {
                    s->match->uniq2 = value;
                }
            }
            else if (key == "MatchBus")
            {
                if (check_set_bits(M_BUS))
                {
                         if (value == "usb"      ) s->match->bus = BT_USB;
                    else if (value == "bluetooth") s->match->bus = BT_BLUETOOTH;
                    else if (value == "ps2"      ) s->match->bus = BT_PS2;
                    else if (value == "rmi"      ) s->match->bus = BT_RMI;
                    else if (value == "i2c"      ) s->match->bus = BT_I2C;
                    else if (value == "spi"      ) s->match->bus = BT_SPI;
                    else
                    {
                        rc = faux;
                    }
                }
            }
            else if (key == "MatchVendor")
            {
                if (check_set_bits(M_VID))
                {
                    if (auto v = utf::to_int<ui32, 16>(value))
                    {
                        s->match->vendor = v.value();
                    }
                    else
                    {
                        rc = faux;
                    }
                }
            }
            else if (key == "MatchProduct")
            {
                static constexpr auto product_size = std::size(decltype(s->match->product){});
                auto product = std::array<ui32, product_size>{};
                auto strs = utf::split<true>(value, ";");
                auto head = strs.begin();
                for (auto& p : product)
                {
                    if (head == strs.end()) break;
                    auto str = *head++;
                    if (auto v = utf::to_int<ui32, 16>(str))
                    {
                        p = v.value();
                    }
                    else
                    {
                        rc = faux;
                        break;
                    }
                }
                if (strs.size() && rc == true)
                {
                    if (check_set_bits(M_PID))
                    {
                        ::memcpy(s->match->product, product.data(), sizeof(product));
                    }
                }
            }
            else if (key == "MatchVersion")
            {
                if (check_set_bits(M_VERSION))
                {
                    if (auto v = utf::to_int<ui32, 16>(value))
                    {
                        s->match->version = v.value();
                    }
                    else
                    {
                        rc = faux;
                    }
                }
            }
            else if (key == "MatchDMIModalias")
            {
                if (check_set_bits(M_DMI))
                {
                    if (!value.starts_with("dmi:"))
                    {
                        log("%s%: MatchDMIModalias must start with 'dmi:'", s->name2);
                        rc = faux;
                    }
                    else
                    {
                        s->match->dmi2 = value;
                    }
                }
            }
            else if (key == "MatchUdevType")
            {
                if (check_set_bits(M_UDEV_TYPE))
                {
                         if (value == "touchpad"     ) s->match->ud_type = UDEV_TOUCHPAD;
                    else if (value == "mouse"        ) s->match->ud_type = UDEV_MOUSE;
                    else if (value == "pointingstick") s->match->ud_type = UDEV_POINTINGSTICK;
                    else if (value == "keyboard"     ) s->match->ud_type = UDEV_KEYBOARD;
                    else if (value == "joystick"     ) s->match->ud_type = UDEV_JOYSTICK;
                    else if (value == "tablet"       ) s->match->ud_type = UDEV_TABLET;
                    else if (value == "tablet-pad"   ) s->match->ud_type = UDEV_TABLET_PAD;
                    else
                    {
                        rc = faux;
                    }
                }
            }
            else if (key == "MatchDeviceTree")
            {
                if (check_set_bits(M_DT))
                {
                    s->match->dt2 = value;
                }
            }
            else
            {
                log("Unknown match key '%s%'", key);
                rc = faux;
            }
            if (rc)
            {
                s->has_match = true;
            }
            return rc;
        }
        bool parse_model(section_sptr s, view key, view value)
        {
            auto b = value == "1";
            auto q = QUIRK_MODEL_ALPS_SERIAL_TOUCHPAD;
            assert(key.starts_with("Model"));
            do
            {
                if (key == quirks_t::quirk_get_name(q))
                {
                    auto p = ptr::shared<property_t>();
                    p->id = q;
                    p->value = b;
                    s->properties.push_back(p);
                    s->has_property = true;
                    return true;
                }
                q = (quirk)(q + 1);
            }
            while (q < _QUIRK_LAST_MODEL_QUIRK_);
            log("Unknown key %s% in %s%", key, s->name2);
            return faux;
        }
        bool parse_dimension_property(view prop, si32_coor& dim)
        {
            auto d = si32_coor{};
            auto ok = prop.size() && ::sscanf(prop.data(), "%dx%d", &d.x, &d.y) == 2 && d.x > 0 && d.y > 0;
            if (ok)
            {
                dim = d;
            }
            return ok;
        }
        bool parse_range_property(view prop, si32_range& range)
        {
            auto r = si32_range{};
            auto ok = prop.size() && (prop == "none" || (::sscanf(prop.data(), "%d:%d", &r.max, &r.min) == 2 && r.min < r.max)); // "max:min":  Pressure/touch size.
            if (ok)
            {
                range = r;
            }
            return ok;
        }
        bool parse_evcode_string(view s, si32& type_out, si32& code_out)
        {
            auto found = faux;
            if (s.starts_with("EV_"))
            {
                auto type = libevdev_event_type_from_name(s.data());
                found = type != -1;
                if (found)
                {
                    type_out = type;
                    code_out = lixx::event_code_undefined;
                }
            }
            else
            {
                static constexpr auto map = std::to_array<std::pair<view, si32>>(
                {
                    { "KEY_", EV_KEY },
                    { "BTN_", EV_KEY },
                    { "ABS_", EV_ABS },
                    { "REL_", EV_REL },
                    { "SW_",  EV_SW  },
                });
                for (auto [str, type] : map)
                {
                    if (s.starts_with(str))
                    {
                        auto code = -1;//todo is it too much? libevdev_event_code_from_name(type, s.data());
                        found = code != -1;
                        if (found)
                        {
                            type_out = type;
                            code_out = code;
                        }
                        break;
                    }
                }
            }
            return found;
        }
        bool parse_evcode_property(view prop, input_event_t* events, ui64& nevents)
        {
            // Parses a string of the format "+EV_ABS;+KEY_A;-BTN_TOOL_DOUBLETAP;-ABS_X;" where each element must be + or - (enable/disable) followed by a named event type OR a named event code OR a tuple in the form of EV_KEY:0x123, i.e. a named event type followed by a hex event code.
            // - events must point to an existing array of size nevents.
            // - nevents specifies the size of the array in events and returns the number of items, elements exceeding nevents are simply ignored, just make sure events is large enough for your use-case.
            // The results are returned as input events with type and code set, all other fields undefined. Where only the event type is specified, the code is set to lixx::event_code_undefined.
            // On success, events contains nevents events with each event's value set to 1 or 0 depending on the + or - prefix.
            auto rc = true;
            auto evs = std::array<input_event_t, 32>{}; // A randomly chosen max so we avoid crazy quirks.
            auto strv = utf::split(prop, ";");
            if (strv.empty() || strv.size() > evs.size())
            {
                rc = faux;
            }
            else
            {
                auto ncodes = std::min(nevents, (ui64)strv.size());
                for (auto idx = 0; strv[idx]; idx++)
                {
                    auto s = strv[idx].data();
                    auto c = *s++;
                    auto enable = faux;
                            if (c == '+') enable = true;
                    else if (c == '-') enable = faux;
                    else
                    {
                        rc = faux;
                        break;
                    }
                    auto type = 0;
                    auto code = 0;
                    auto crop = qiew{ s };
                    if (crop.find(':') == text::npos)
                    {
                        if (!parse_evcode_string(s, type, code))
                        {
                            rc = faux;
                            break;
                        }
                    }
                    else
                    {
                        auto consumed = 0;
                        char stype[13] = {}; // EV_FF_STATUS + '\0'.
                        if (::sscanf(s, "%12[A-Z_]:%x%n", stype, &code, &consumed) != 2
                            || ::strlen(s) != (ui64)consumed
                            || (type = libevdev_event_type_from_name(stype)) == -1
                            || code < 0 || code > libevdev_event_type_get_max(type))
                        {
                            rc = faux;
                            break;
                        }
                    }
                    evs[idx].type = type;
                    evs[idx].code = code;
                    evs[idx].value = enable;
                }
                if (rc)
                {
                    ::memcpy(events, evs.data(), ncodes * sizeof(*events));
                    nevents = ncodes;
                }
            }
            return rc;
        }
        bool parse_input_prop_property(view prop_str, std::array<input_prop, INPUT_PROP_CNT>& props_out, ui64& nprops)
        {
            // Parses a string of the format "+INPUT_PROP_BUTTONPAD;-INPUT_PROP_POINTER;+0x123;" where each element must be a named input prop OR a hexcode in the form 0x1234. The prefix for each element must be either '+' (enable) or '-' (disable).
            // - props must point to an existing array of size nprops.
            // - nprops specifies the size of the array in props and returns the number of elements, elements exceeding nprops are simply ignored, just make sure props is large enough for your use-case.
            // On success, props contains nprops elements.
            auto props = std::array<input_prop, INPUT_PROP_CNT>{}; // Doubling up on quirks is a bug.
            auto strv = utf::split(prop_str, ";");
            auto rc = strv.size() && strv.size() < props.size();
            if (rc)
            {
                auto count = std::min(nprops, (ui64)strv.size());
                for (auto idx = 0; strv[idx]; idx++)
                {
                    auto s = strv[idx].data();
                    auto c = *s++;
                    auto prop = 0u;
                    auto enable = faux;
                            if (c == '+') enable = true;
                    else if (c == '-') enable = faux;
                    else
                    {
                        rc = faux;
                        break;
                    }
                    auto crop = qiew{ s };
                    if (auto v = utf::to_int<ui32, 16>(crop))
                    {
                        prop = v.value();
                        if (prop > INPUT_PROP_MAX)
                        {
                            rc = faux;
                            break;
                        }
                    }
                    else
                    {
                        auto val = libevdev_property_from_name(s);
                        if (val == -1)
                        {
                            rc = faux;
                            break;
                        }
                        prop = (ui32)val;
                    }
                    props[idx].first  = prop;
                    props[idx].second = enable;
                }
                if (rc)
                {
                    props_out = props;
                    nprops = count;
                }
            }
            return rc;
        }
        bool parse_attr(section_sptr s, view key, qiew value)
        {
            auto p = ptr::shared<property_t>();
            auto rc = faux;
            auto dim = si32_coor{};
            auto range = si32_range{};
            if (key == quirks_t::quirk_get_name(QUIRK_ATTR_SIZE_HINT))
            {
                p->id = QUIRK_ATTR_SIZE_HINT;
                rc = parse_dimension_property(value, dim);
                if (rc) p->value = dim;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_TOUCH_SIZE_RANGE))
            {
                p->id = QUIRK_ATTR_TOUCH_SIZE_RANGE;
                rc = parse_range_property(value, range);
                if (rc) p->value = range;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_PALM_SIZE_THRESHOLD))
            {
                p->id = QUIRK_ATTR_PALM_SIZE_THRESHOLD;
                auto v = utf::to_int<ui32>(value);
                rc = !!v;
                if (rc) p->value = v.value();
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_LID_SWITCH_RELIABILITY))
            {
                p->id = QUIRK_ATTR_LID_SWITCH_RELIABILITY;
                rc = value == "reliable" || value == "write_open" || value == "unreliable";
                if (rc) p->value = text{ value };
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_KEYBOARD_INTEGRATION))
            {
                p->id = QUIRK_ATTR_KEYBOARD_INTEGRATION;
                rc = value == "internal" && value != "external";
                if (rc) p->value = text{ value };
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_TRACKPOINT_INTEGRATION))
            {
                p->id = QUIRK_ATTR_TRACKPOINT_INTEGRATION;
                rc = value == "internal" && value != "external";
                if (rc) p->value = text{ value };
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_TPKBCOMBO_LAYOUT))
            {
                p->id = QUIRK_ATTR_TPKBCOMBO_LAYOUT;
                rc = value == "below";
                if (rc) p->value = text{ value };
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_PRESSURE_RANGE))
            {
                p->id = QUIRK_ATTR_PRESSURE_RANGE;
                rc = parse_range_property(value, range);
                if (rc) p->value = range;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_PALM_PRESSURE_THRESHOLD))
            {
                p->id = QUIRK_ATTR_PALM_PRESSURE_THRESHOLD;
                auto v = utf::to_int<ui32>(value);
                rc = !!v;
                if (rc) p->value = v.value();
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_RESOLUTION_HINT))
            {
                p->id = QUIRK_ATTR_RESOLUTION_HINT;
                rc = parse_dimension_property(value, dim);
                if (rc) p->value = dim;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_TRACKPOINT_MULTIPLIER))
            {
                p->id = QUIRK_ATTR_TRACKPOINT_MULTIPLIER;
                auto v = utf::to_int<fp64>(value);
                rc = !!v;
                if (rc) p->value = v.value();
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_USE_VELOCITY_AVERAGING))
            {
                p->id = QUIRK_ATTR_USE_VELOCITY_AVERAGING;
                p->value = value == "1";
                rc = true;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_TABLET_SMOOTHING))
            {
                p->id = QUIRK_ATTR_TABLET_SMOOTHING;
                p->value = value == "1";
                rc = true;
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_THUMB_PRESSURE_THRESHOLD))
            {
                p->id = QUIRK_ATTR_THUMB_PRESSURE_THRESHOLD;
                auto v = utf::to_int<ui32>(value);
                rc = !!v;
                if (rc) p->value = v.value();
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_THUMB_SIZE_THRESHOLD))
            {
                p->id = QUIRK_ATTR_THUMB_SIZE_THRESHOLD;
                auto v = utf::to_int<ui32>(value);
                rc = !!v;
                if (rc) p->value = v.value();
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_MSC_TIMESTAMP))
            {
                p->id = QUIRK_ATTR_MSC_TIMESTAMP;
                rc = value == "watch";
                if (rc) p->value = text{ value };
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_EVENT_CODE))
            {
                auto events = std::array<input_event_t, 32>{};
                auto nevents = (ui64)events.size();
                p->id = QUIRK_ATTR_EVENT_CODE;
                if (parse_evcode_property(value, events.data(), nevents) && nevents)
                {
                    auto value_tuples = quirk_tuples{};
                    for (auto i = 0u; i < nevents; i++)
                    {
                        value_tuples.tuples[i].first = events[i].type;
                        value_tuples.tuples[i].second = events[i].code;
                        value_tuples.tuples[i].third = events[i].value;
                    }
                    value_tuples.ntuples = nevents;
                    p->value = value_tuples;
                    rc = true;
                }
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_INPUT_PROP))
            {
                auto props = std::array<input_prop, INPUT_PROP_CNT>{};
                auto nprops = (ui64)props.size();
                p->id = QUIRK_ATTR_INPUT_PROP;
                if (parse_input_prop_property(value, props, nprops) && nprops != 0)
                {
                    auto value_tuples = quirk_tuples{};
                    for (auto i = 0u; i < nprops; i++)
                    {
                        value_tuples.tuples[i].first  = props[i].first;
                        value_tuples.tuples[i].second = props[i].second;
                    }
                    value_tuples.ntuples = nprops;
                    p->value = value_tuples;
                    rc = true;
                }
            }
            else if (key == quirks_t::quirk_get_name(QUIRK_ATTR_IS_VIRTUAL))
            {
                p->id = QUIRK_ATTR_IS_VIRTUAL;
                p->value = value == "1";
                rc = true;
            }
            else
            {
                log("Unknown key %s% in %s%", key, s->name2);
            }
            if (rc)
            {
                s->properties.push_back(p);
                s->has_property = true;
            }
            return rc;
        }
        bool parse_value_line(section_sptr s, view line)
        {
            auto rc = faux;
            auto strv = utf::split(line, "=");
            if (strv.size() == 2)
            {
                auto key = strv[0];
                auto value = strv[1];
                if (key.size() && value.size() && value[0] != '"' && value[0] != '\'') // The value is not supposed to be in quotes.
                {
                            if (key.starts_with("Match")) rc = parse_match(s, key, value);
                    else if (key.starts_with("Model")) rc = parse_model(s, key, value);
                    else if (key.starts_with("Attr") ) rc = parse_attr( s, key, value);
                    else log("Unknown value prefix %s%", line);
                }
            }
            return rc;
        }
        bool parse_file(view path)
        {
            enum state_enum
            {
                STATE_SECTION,
                STATE_MATCH,
                STATE_MATCH_OR_VALUE,
                STATE_VALUE_OR_SECTION,
                STATE_ANY,
            };
            auto strstartswith = [](char const* str, char const* prefix)
            {
                if (str == nullptr) return faux;
                auto prefixlen = ::strlen(prefix);
                return prefixlen > 0 ? strneq(str, prefix, ::strlen(prefix)) : faux;
            };
            char line[512];
            auto rc = true;
            auto state = STATE_SECTION;
            auto section = ptr::shared<section_t>();
            auto lineno = -1;
            log("%s%", path);
            auto fp = ::fopen(path.data(), "r");
            if (!fp) // If the file doesn't exist that's fine. Only way this can happen is for the custom override file, all others are provided by scandir so they do exist. Short of races we don't care about.
            {
                if (errno == ENOENT) return true;
                log("%s%: failed to open file", path);
                rc = faux;
            }
            else while (::fgets(line, sizeof(line), fp))
            {
                lineno++;
                auto comment = strstr(line, "#");
                if (comment) // Comment points to # but we need to remove the preceding whitespaces too.
                {
                    comment--;
                    while (comment >= line)
                    {
                        if (*comment == ' ' || *comment == '\t')
                        {
                            comment--;
                        }
                    }
                    *(comment + 1) = '\0';
                }
                else // Strip the trailing newline.
                {
                    comment = strstr(line, "\n");
                    if (comment) *comment = '\0';
                }
                if (strlen(line) == 0) continue;
                // We don't use quotes for strings, so we really don't want erroneous trailing whitespaces.
                auto c = line[strlen(line) - 1];
                if (c == ' ' || c == '\t') // Strip the trailing newline.
                {
                    log("%s%:%d%: Trailing whitespace '%s%'", path, lineno, line);
                    rc = faux;
                    break;
                }
                c = line[0];
                if (c == '\0' || c == '\n' || c == '#')
                {
                    //
                }
                else if (c == ' ' || c == '\t') // Whitespaces not allowed.
                {
                    log("%s%:%d%: Preceding whitespace '%s%'", path, lineno, line);
                    rc = faux;
                    break;
                }
                else if (c == '[') // Section title.
                {
                    if (line[strlen(line) - 1] != ']')
                    {
                        log("%s%:%d%: Closing ] missing '%s%'", path, lineno, line);
                        rc = faux;
                        break;
                    }
                    if (state != STATE_SECTION && state != STATE_VALUE_OR_SECTION)
                    {
                        log("%s%:%d%: expected section before %s%", path, lineno, line);
                        rc = faux;
                        break;
                    }
                    if (section && (!section->has_match || !section->has_property))
                    {
                        log("%s%:%d%: previous section %s% was empty", path, lineno, section->name2);
                        rc = faux;
                        break; // Previous section was empty.
                    }
                    state = STATE_MATCH;
                    auto path_copy = text{ path };
                    section->name2 = utf::fprint("%s% (%s%)", line, ::basename(path_copy.data()));
                    sections.push_back(section);
                    break;
                }
                else // Entries must start with A-Z.
                {
                    if (line[0] < 'A' || line[0] > 'Z')
                    {
                        log("%s%:%d%: Unexpected line %s%", path, lineno, line);
                        rc = faux;
                        break;
                    }
                    if (state == STATE_SECTION)
                    {
                        log("%s%:%d%: expected [Section], got %s%", path, lineno, line);
                        rc = faux;
                        break;
                    }
                    else if (state == STATE_MATCH)
                    {
                        if (!strstartswith(line, "Match"))
                        {
                            log("%s%:%d%: expected MatchFoo=bar, have %s%", path, lineno, line);
                            rc = faux;
                            break;
                        }
                        state = STATE_MATCH_OR_VALUE;
                    }
                    else if (state == STATE_MATCH_OR_VALUE)
                    {
                        if (!strstartswith(line, "Match"))
                        {
                            state = STATE_VALUE_OR_SECTION;
                        }
                    }
                    else if (state == STATE_VALUE_OR_SECTION)
                    {
                        if (strstartswith(line, "Match"))
                        {
                            log("%s%:%d%: expected value or [Section], have %s%", path, lineno, line);
                            rc = faux;
                            break;
                        }
                    }
                    else if (state == STATE_ANY)
                    {
                        //
                    }
                    if (!parse_value_line(section, line))
                    {
                        log("%s%:%d%: failed to parse %s%", path, lineno, line);
                        rc = faux;
                        break;
                    }
                }
            }
            if (rc)
            {
                if (!section)
                {
                    log("%s%: is an empty file", path);
                    rc = faux;
                }
                else if ((!section->has_match || !section->has_property))
                {
                    log("%s%:%d%: previous section %s% was empty", path, lineno, section->name2);
                    rc = faux;
                }
            }
            if (fp)
            {
                ::fclose(fp);
            }
            return rc;
        }
        bool parse_files(qiew data_path)
        {
            //todo use os::fs
            auto namelist = (::dirent**)nullptr;
            auto ndev = ::scandir(data_path.data(), &namelist, is_data_file, versionsort);
            if (ndev <= 0)
            {
                //log("%s%: failed to find data files", data_path ? data_path : "empty path");
                return faux;
            }
            auto idx = 0;
            for (; idx < ndev; idx++)
            {
                char path[PATH_MAX];
                ::snprintf(path, sizeof(path), "%s/%s", data_path.data(), namelist[idx]->d_name);
                if (!parse_file(path)) break;
            }
            for (auto i = 0; i < ndev; i++)
            {
                ::free(namelist[i]);
            }
            ::free(namelist);
            return idx == ndev;
        }
    };

    struct libinput_t : ptr::enable_shared_from_this<libinput_t>
    {
        struct ud_monitor_t
        {
            static constexpr auto& active_tty_file = "/sys/devices/virtual/tty/tty0/active"; //todo incompatible with FreeBSD
            static constexpr auto& dev_input_path = "/dev/input";
            static constexpr auto dev_monitor_mode = IN_CREATE | IN_ATTRIB/*try again after mode access update*/;
            static constexpr auto tty_monitor_mode = IN_MODIFY;

            libinput_t&                              li;
            fd_t                                     inotify_fd;
            fd_t                                     wd_tty;
            fd_t                                     wd_dev;
            text                                     buffer;
            text                                     uevent_buffer;
            utf::unordered_map<text, ud_device_sptr> ud_device_list;
            text                                     initial_tty;
            text                                     current_tty;
            event_source_sptr                        ud_monitor_handler;

            ud_monitor_t(libinput_t& li)
                :           li{ li },
                    inotify_fd{ os::invalid_fd },
                        wd_tty{ os::invalid_fd },
                        wd_dev{ os::invalid_fd },
                 uevent_buffer(4096, '\0')
            {
                auto tmp1 = ::dup(STDIN_FILENO);
                auto tmp2 = ::inotify_init1(IN_CLOEXEC | IN_NONBLOCK); //todo Revise: ::inotify_init1() returns STDIN_FILENO for unknown reason despite the STDIN_FILENO is not closed.
                if (tmp2 == os::invalid_fd)
                {
                    log("Failed to initialize inotify context");
                }
                else if (tmp2 == STDIN_FILENO)
                {
                    inotify_fd = ::dup(tmp2);
                    os::close(tmp2);
                    ::dup2(tmp1, STDIN_FILENO);
                }
                else
                {
                    inotify_fd = tmp2;
                }
                os::close(tmp1);
                initial_tty = get_active_tty();
                current_tty = initial_tty;
                wd_tty = ::inotify_add_watch(inotify_fd, active_tty_file, tty_monitor_mode);
            }
            ~ud_monitor_t()
            {
                udev_monitor_disable_monitoring();
            }
            operator bool () const { return inotify_fd != os::invalid_fd; }

            text get_active_tty()
            {
                auto buffer = std::array<char, 10>{};
                auto f = std::ifstream{ active_tty_file };
                if (f.is_open())
                {
                    f.read(buffer.data(), buffer.size());
                    f.close();
                    auto ttynum = qiew{ buffer.data() };
                    utf::trim_back(ttynum, whitespaces);
                    if (ttynum.size()) log("Active tty changed to '%tty%'", ttynum);
                    return ttynum;
                }
                else return {};
            }
            bool add_by_name(qiew filename)
            {
                auto iter = ud_device_list.find(filename);
                if (iter == ud_device_list.end())
                {
                    auto ud_device_ptr = ptr::shared<ud_device_t>(filename);
                    if (ud_device_ptr->initialized && li.libinput_device_create(ud_device_ptr))
                    {
                        ud_device_list[filename] = ud_device_ptr;
                        return true;
                    }
                }
                return faux;
            }
            void add_devices()
            {
                auto length = 0u;
                ::ioctl(inotify_fd, FIONREAD, &length); // Get available events block size.
                if (length)
                {
                    buffer.resize(length);
                    length = ::read(inotify_fd, buffer.data(), buffer.size()); // Take events block.
                    if (!length)
                    {
                        log("Failed to read events. errno=%%", errno);
                        return;
                    }
                    auto crop = qiew{ buffer };
                    while (crop)
                    {
                        auto& event = *(::inotify_event*)crop.data();
                        if (event.wd == wd_tty)
                        {
                            current_tty = get_active_tty();
                        }
                        else if (event.wd == wd_dev)
                        {
                            auto filename = qiew{ event.name };
                            if (!(event.mask & IN_ISDIR) && (event.mask & dev_monitor_mode) && filename.starts_with("event")) // Created or access modified. Do nothing on delete, just wait for ENODEV.
                            {
                                add_by_name(filename);
                            }
                        }
                        crop.remove_prefix(sizeof(::inotify_event) + event.len);
                    }
                }
            }
            void udev_monitor_add_all_active_input_devices()
            {
                auto code = std::error_code{};
                auto events = os::fs::directory_iterator{ "/dev/input/", code };
                if (!code)
                for (auto& entry : events)
                {
                    auto filename = entry.path().filename().string();
                    if (filename.starts_with("event"))
                    {
                        add_by_name(filename);
                    }
                }
            }
            void udev_monitor_enable_monitoring()
            {
                if (inotify_fd != os::invalid_fd)
                {
                    wd_dev = ::inotify_add_watch(inotify_fd, dev_input_path, dev_monitor_mode);
                    if (wd_dev != os::invalid_fd)
                    {
                        ud_monitor_handler = li.timers.libinput_add_event_source(inotify_fd, [&]{ add_devices(); });
                    }
                    else log("Couldn't add watch to %s%", dev_input_path);
                }
                else log("Couldn't initialize inotify");
            }
            void udev_monitor_disable_monitoring()
            {
                if (auto fd_value = std::exchange(inotify_fd, os::invalid_fd); fd_value != os::invalid_fd)
                {
                    os::close(fd_value);
                    wd_dev = os::invalid_fd;
                    wd_tty = os::invalid_fd;
                }
            }
        };

        using event_variants = std::variant<libinput_event_empty,
                                            libinput_event_device_notify,
                                            libinput_event_keyboard,
                                            libinput_event_pointer,
                                            libinput_event_gesture,
                                            libinput_event_touch,
                                            libinput_event_switch,
                                            libinput_event_tablet_pad,
                                            libinput_event_tablet_tool>;

        libinput_timer_host                   timers;
        std::deque<event_variants>            event_queue;
        std::list<libinput_tablet_tool_sptr>  tool_list;
        time                                  last_event_time = {};
        time                                  dispatch_time = {};
        ui32                                  seat_slot_map = {};
        ui32                                  seat_button_count[KEY_CNT] = {};
        quirks_context_t                      quirks;
        ud_monitor_t                          ud_monitor;
        std::list<libinput_device_sptr>       device_list;

        libinput_device_sptr libinput_device_create(ud_device_sptr ud_device_ptr);
        static auto& empty_event()
        {
            static auto empty_event = libinput_event_device_notify{};
            return empty_event;
        }

        libinput_t()
            : ud_monitor{ *this }
        {
            timers.epoll_fd = ::epoll_create1(EPOLL_CLOEXEC);
            if (timers.epoll_fd != os::invalid_fd)
            {
                libinput_t::event_queue.resize(4);
                libinput_t::event_queue.clear();
                libinput_t::event_queue.emplace_back(empty_event()); // At least one event must be in the event queue (as previous).
                if (timers.libinput_timer_subsys_init())
                {
                    ud_monitor.udev_monitor_add_all_active_input_devices();
                    ud_monitor.udev_monitor_enable_monitoring();
                }
            }
        }

        operator bool () const { return timers.epoll_fd != os::invalid_fd; }

        quirks_sptr quirks_fetch_for_device(ud_device_t& ud_device)
        {
            if (quirks)
            {
                log("%s%: fetching quirks", ud_device.devpath);
                auto m = quirks.match_new(ud_device);
                auto q = ptr::shared<quirks_t>();
                for (auto s : quirks.sections)
                {
                    q->quirk_match_section(s, m);
                }
                if (q->properties.size())
                {
                    quirks.quirk_list.push_back(q);
                    return q;
                }
            }
            return {};
        }
        ui32 update_press_count(ui32 key_code, si32 state)
        {
            assert(key_code <= KEY_MAX);
            auto& press_count = seat_button_count[key_code];
                 if (state == evdev::pressed) press_count++;
            else if (press_count)             press_count--; // We might not have received the first PRESSED event.
            return press_count;
        }
        libinput_device_sptr libinput_add_device(qiew sysname)
        {
            auto ok = ud_monitor.add_by_name(sysname);
            return ok ? device_list.back() : libinput_device_sptr{};
        }
        void enumerate_active_devices(auto proc)
        {
            auto iter = device_list.begin();
            while (iter != device_list.end())
            {
                auto d = *iter++;
                if (!proc(d)) break;
            }
        }
        bool current_tty_is_active()
        {
            return ud_monitor && ud_monitor.initial_tty.size() && ud_monitor.initial_tty == ud_monitor.current_tty;
        }
        template<class T>
        auto& libinput_emplace_event()
        {
            auto& event_packet = std::get<T>(event_queue.emplace_back(T{}));
            return event_packet;
        }
        auto& libinput_get_event()
        {
            auto selector = [](auto& e)->libinput_event& { return e; };
            assert(!event_queue.empty());
            if (event_queue.size() > 1)
            {
                event_queue.pop_front(); // Pop/invalidate previous event.
                auto& event = std::visit(selector, event_queue.front());
                return event;
            }
            else
            {
                return (libinput_event&)empty_event();
            }
        }
        si32 libinput_dispatch()
        {
            static auto take_time_snapshot = byte{};
            auto ep = std::array<::epoll_event, 32>{};
            // Every 10 calls to libinput_dispatch() we take the current time so we can check the delay between our current time and the event timestamps.
                 if ((++take_time_snapshot % 10) == 0) dispatch_time = datetime::now();
            else if (dispatch_time != time{})          dispatch_time = {};
            auto count = ::epoll_wait(timers.epoll_fd, ep.data(), ep.size(), 0);
            if (count < 0) return -errno;
            for (auto i = 0; i < count; ++i)
            {
                auto& source = *(event_source_t*)ep[i].data.ptr;
                if (source.fd != os::invalid_fd)
                {
                    source.func();
                }
            }
            timers.source_destroy_list.clear();
            return 0;
        }
        #if HAVE_LIBWACOM
        WacomDeviceDatabase* libinput_libwacom_ref()
        {
            auto db = (WacomDeviceDatabase*)nullptr;
            if (!libwacom.db)
            {
                db = ::libwacom_database_new();
                if (!db)
                {
                    log("Failed to initialize libwacom context");
                    return NULL;
                }
                libwacom.db = db;
                libwacom.refcount = 0;
            }
            libwacom.refcount++;
            db = libwacom.db;
            return db;
        }
        #endif
        void libinput_libwacom_unref()
        {
            #if HAVE_LIBWACOM
            if (!libwacom.db) return;
            assert(libwacom.refcount >= 1);
            if (--libwacom.refcount == 0)
            {
                ::libwacom_database_destroy(libwacom.db);
                libwacom.db = {};
            }
            #endif
        }
        fd_t libinput_get_fd()
        {
            return timers.epoll_fd;
        }
    };

    struct libinput_device_t : ptr::enable_shared_from_this<libinput_device_t>
    {
        struct evdev_scroll_t
        {
            libinput_timer_sptr                   timer;
            libinput_device_config_scroll_method  config;
            libinput_config_scroll_method         method; // Currently enabled method, button.
            ui32                                  button{}; // evdev_usage_t
            time                                  button_down_time{};
            libinput_config_scroll_method         want_method{}; // Set during device init, used at runtime to delay changes until all buttons are up.
            ui32                                  want_button{}; // evdev_usage_t
            void                                (*change_scroll_method)(libinput_device_sptr li_device) = {}; // Checks if buttons are down and commits the setting.
            evdev_button_scroll_state             button_scroll_state{};
            fp64                                  threshold = 5.0; // Default may be overridden.
            fp64                                  direction_lock_threshold = 5.0; // Default may be overridden.
            ui32                                  direction{};
            fp64_coor                             buildup;
            libinput_device_config_natural_scroll config_natural;
            bool                                  natural_scrolling_enabled{}; // Set during device init if we want natural scrolling, used at runtime to enable/disable the feature.
            bool                                  invert_horizontal_scrolling{}; // Set during device init to invert direction of horizontal scrolling.
            fp64_coor                             wheel_click_angle; // Angle per REL_WHEEL click in degrees.
            evdev_button_scroll_lock_state        lock_state{};
            bool                                  want_lock_enabled{};
            bool                                  lock_enabled{};
        };
        struct evdev_left_handed_t
        {
            libinput_device_config_left_handed config;
            bool                               enabled{}; // Left-handed currently enabled.
            bool                               want_enabled{}; // Set during device init if we want left_handed config, used at runtime to delay the effect until buttons are up.
            void                             (*change_to_enabled)(libinput_device_sptr li_device) = {}; // Checks if buttons are down and commits the setting.
        };
        struct evdev_middlebutton_t
        {
            libinput_device_config_middle_emulation config;
            bool                                    enabled{}; // Middle-button emulation enabled.
            bool                                    enabled_default{};
            bool                                    want_enabled{};
            evdev_middlebutton_state                state{};
            libinput_timer_sptr                     timer;
            ui32                                    button_mask{};
            time                                    first_event_time{};
        };

        libinput_t&                             li;
        ud_device_sptr                          ud_device_ptr; // Owning the physical device.
        ud_device_t&                            ud_device;
        text                                    device_group; //todo Property for tablet touch arbitration. Set LIBINPUT_DEVICE_GROUP somewhere in settings (or in quirks) for devices intended to be in a group (e.g. tablet+stylus).
        std::list<libinput_event_listener_sptr> event_listeners;
        libinput_device_config                  config;
        libinput_config_send_events_mode        sendevents_current_mode = LIBINPUT_CONFIG_SEND_EVENTS_ENABLED;
        event_source_sptr                       source;
        bool                                    was_removed{};
        ui32                                    device_caps{};
        ui32                                    device_tags{};
        bool                                    is_suspended{};
        si32                                    dpi{ lixx::default_mouse_dpi }; // HW resolution.
        fp64                                    trackpoint_multiplier{};  // Trackpoint constant multiplier.
        bool                                    use_velocity_averaging{}; // Whether averaging should be applied on velocity calculation.
        evdev_scroll_t                          scroll;
        libinput_device_config_accel            pointer_config;
        motion_filter_sptr                      pointer_filter;
        byte                                    key_count[KEY_CNT] = {}; // Key counter used for multiplexing button events internally in libinput.
        evdev_left_handed_t                     dev_left_handed;
        evdev_middlebutton_t                    middlebutton;
        evdev_frame                             frame;

        libinput_device_t(libinput_t& li, ud_device_sptr ud_device_ptr)
            :          li{ li },
            ud_device_ptr{ ud_device_ptr },
                ud_device{ *ud_device_ptr }
        { }
        virtual ~libinput_device_t()
        { }

        virtual void                       process([[maybe_unused]] evdev_event& event, time)                                                { } // Process an evdev input event.
        virtual void                       suspend()                                                                                         { } // Device is being suspended.
        virtual void                        remove()                                                                                         { } // Device is being removed (may be nullptr).
        virtual void                       destroy()                                                                                         { } // Destroy an event dispatch handler and free all its resources.
        virtual void                  device_added([[maybe_unused]] libinput_device_sptr added_li_device)                                    { } // A new device was added.
        virtual void                device_removed([[maybe_unused]] libinput_device_sptr removed_li_device)                                  { } // A device was removed.
        virtual void              device_suspended([[maybe_unused]] libinput_device_sptr suspended_li_device)                                { } // A device was suspended.
        virtual void                device_resumed([[maybe_unused]] libinput_device_sptr resumed_li_device)                                  { } // A device was resumed.
        virtual void                    post_added()                                                                                         { } // Called immediately after the LIBINPUT_EVENT_DEVICE_ADDED event was sent.
        virtual void      touch_arbitration_toggle([[maybe_unused]] libinput_arbitration_state which, [[maybe_unused]] fp64_rect area, time) { } // For touch arbitration, called on the device that should enable/disable touch capabilities.
        virtual void touch_arbitration_update_rect([[maybe_unused]] fp64_rect area, time)                                                    { } // Called when touch arbitration is on, updates the area where touch arbitration should apply.
        virtual si32              get_switch_state([[maybe_unused]] libinput_switch which)                                                   { return {}; } // Return the state of the given switch.
        virtual void            left_handed_toggle([[maybe_unused]] bool left_handed_enabled)                                                { }

        virtual libinput_config_status sendevents_set_mode(libinput_config_send_events_mode mode)
        {
                 if (mode == sendevents_current_mode)              return LIBINPUT_CONFIG_STATUS_SUCCESS;
            else if (mode == LIBINPUT_CONFIG_SEND_EVENTS_ENABLED)  evdev_device_resume();
            else if (mode == LIBINPUT_CONFIG_SEND_EVENTS_DISABLED) evdev_device_suspend();
            else                                                   return LIBINPUT_CONFIG_STATUS_UNSUPPORTED; // No support for combined modes yet.
            sendevents_current_mode = mode;
            return LIBINPUT_CONFIG_STATUS_SUCCESS;
        }
        virtual libinput_config_send_events_mode sendevents_get_default_mode()
        {
            return LIBINPUT_CONFIG_SEND_EVENTS_ENABLED;
        }
        virtual ui32 sendevents_get_modes()
        {
            return LIBINPUT_CONFIG_SEND_EVENTS_DISABLED;
        }

        static si32_coor apply_hysteresis(si32_coor in, si32_coor center, si32_coor margin)
        {
            // Apply a hysteresis filtering to the coordinate in, based on the current
            // hysteresis center and the margin. If 'in' is within 'margin' of center,
            // return the center (and thus filter the motion). If 'in' is outside,
            // return a point on the edge of the new margin (which is an ellipse, usually
            // a circle). So for a point x in the space outside c + margin we return r:
            // ,---.       ,---.
            // | c |  x →  | r x
            // `---'       `---'
            //
            // The effect of this is that initial small motions are filtered. Once we
            // move into one direction we lag the real coordinates by 'margin' but any
            // movement that continues into that direction will always be just outside
            // margin - we get responsive movement. Once we move back into the other
            // direction, the first movements are filtered again.
            //
            // Returning the edge rather than the point avoids cursor jumps, as the
            // first reachable coordinate is the point next to the center (center + 1).
            // Otherwise, the center has a dead zone of size margin around it and the
            // first reachable point is the margin edge.
            //
            // @param in The input coordinate
            // @param center Current center of the hysteresis
            // @param margin Hysteresis width (on each side)
            // @return The new center of the hysteresis
            auto d = in - center;
            auto d2 = d * d;
            auto a = margin.x;
            auto b = margin.y;
            auto lag_x = 0.0;
            auto lag_y = 0.0;
            if (!a || !b) return in;
            // Basic equation for an ellipse of radii a,b:
            //   x²/a² + y²/b² = 1
            // But we start by making a scaled ellipse passing through the
            // relative finger location (dx,dy). So the scale of this ellipse is
            // the ratio of finger_distance to margin_distance:
            //   dx²/a² + dy²/b² = normalized_finger_distance²
            auto normalized_finger_distance = std::sqrt((fp64)d2.x / (a * a) + (fp64)d2.y / (b * b));
            // Which means anything less than 1 is within the elliptical margin.
            if (normalized_finger_distance < 1.0) return center;
            auto finger_distance = std::sqrt(d2.x + d2.y);
            auto margin_distance = finger_distance / normalized_finger_distance;
            // Now calculate the x,y coordinates on the edge of the margin ellipse where it intersects the finger vector. Shortcut: We achieve this by finding the point with the same gradient as dy/dx.
            if (d.x)
            {
                auto gradient = (fp64)d.y / d.x;
                lag_x = margin_distance / std::sqrt(gradient * gradient + 1);
                lag_y = std::sqrt((margin_distance + lag_x) * (margin_distance - lag_x));
            }
            else // Infinite gradient.
            {
                lag_x = 0.0;
                lag_y = margin_distance;
            }
            // The 'result' is the centre of an ellipse (radii a,b) which has been dragged by the finger moving inside it to 'in'. The finger is now touching the margin ellipse at some point: (±lag_x,±lag_y).
            auto result = si32_coor{};
            result.x = d.x >= 0 ? in.x - lag_x : in.x + lag_x;
            result.y = d.y >= 0 ? in.y - lag_y : in.y + lag_y;
            return result;
        }
        view middlebutton_state_to_str(evdev_middlebutton_state state)
        {
            switch (state)
            {
                CASE_RETURN_STRING(MIDDLEBUTTON_IDLE);
                CASE_RETURN_STRING(MIDDLEBUTTON_LEFT_DOWN);
                CASE_RETURN_STRING(MIDDLEBUTTON_RIGHT_DOWN);
                CASE_RETURN_STRING(MIDDLEBUTTON_MIDDLE);
                CASE_RETURN_STRING(MIDDLEBUTTON_LEFT_UP_PENDING);
                CASE_RETURN_STRING(MIDDLEBUTTON_RIGHT_UP_PENDING);
                CASE_RETURN_STRING(MIDDLEBUTTON_PASSTHROUGH);
                CASE_RETURN_STRING(MIDDLEBUTTON_IGNORE_LR);
                CASE_RETURN_STRING(MIDDLEBUTTON_IGNORE_L);
                CASE_RETURN_STRING(MIDDLEBUTTON_IGNORE_R);
            }
            return view{};
        }
        view middlebutton_event_to_str(evdev_middlebutton_event event)
        {
            switch (event)
            {
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_L_DOWN);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_R_DOWN);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_OTHER);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_L_UP);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_R_UP);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_TIMEOUT);
                CASE_RETURN_STRING(MIDDLEBUTTON_EVENT_ALL_UP);
            }
            return view{};
        }
        void libinput_device_add_event_listener(libinput_event_listener_sptr listener)
        {
            event_listeners.push_back(listener);
        }
        void libinput_device_remove_event_listener(libinput_event_listener_sptr listener)
        {
            std::erase_if(event_listeners, [&](auto& l){ return l == listener; });
        }
        bool evdev_device_is_virtual()
        {
            return device_tags & EVDEV_TAG_VIRTUAL;
        }
        void post_device_event(time now, libinput_event_type type, libinput_event& event)
        {
            event.stamp = now;
            event.type = type;
            event.li_device = This();
            for (auto& l : event_listeners)
            {
                auto& listener = *l;
                listener(now, event);
            }
        }
        si32 libinput_device_has_capability(libinput_device_capability capability)
        {
            switch (capability)
            {
                case LIBINPUT_DEVICE_CAP_POINTER:     return !!(device_caps & EVDEV_DEVICE_POINTER);
                case LIBINPUT_DEVICE_CAP_KEYBOARD:    return !!(device_caps & EVDEV_DEVICE_KEYBOARD);
                case LIBINPUT_DEVICE_CAP_TOUCH:       return !!(device_caps & EVDEV_DEVICE_TOUCH);
                case LIBINPUT_DEVICE_CAP_GESTURE:     return !!(device_caps & EVDEV_DEVICE_GESTURE);
                case LIBINPUT_DEVICE_CAP_TABLET_TOOL: return !!(device_caps & EVDEV_DEVICE_TABLET);
                case LIBINPUT_DEVICE_CAP_TABLET_PAD:  return !!(device_caps & EVDEV_DEVICE_TABLET_PAD);
                case LIBINPUT_DEVICE_CAP_SWITCH:      return !!(device_caps & EVDEV_DEVICE_SWITCH);
                default: return faux;
            }
        }
        bool device_has_cap(libinput_device_capability cap)
        {
            auto capability = view{};
            if (libinput_device_has_capability(cap)) return true;
            if constexpr (debugmode)
            {
                switch (cap)
                {
                    case LIBINPUT_DEVICE_CAP_POINTER:     capability = "CAP_POINTER";     break;
                    case LIBINPUT_DEVICE_CAP_KEYBOARD:    capability = "CAP_KEYBOARD";    break;
                    case LIBINPUT_DEVICE_CAP_TOUCH:       capability = "CAP_TOUCH";       break;
                    case LIBINPUT_DEVICE_CAP_GESTURE:     capability = "CAP_GESTURE";     break;
                    case LIBINPUT_DEVICE_CAP_TABLET_TOOL: capability = "CAP_TABLET_TOOL"; break;
                    case LIBINPUT_DEVICE_CAP_TABLET_PAD:  capability = "CAP_TABLET_PAD";  break;
                    case LIBINPUT_DEVICE_CAP_SWITCH:      capability = "CAP_SWITCH";      break;
                    default:                              capability = "CAP_UNKNOWN";     break;
                }
                log("Event for missing capability %s% on device \"%s%\"", capability, ud_device.devname);
            }
            return faux;
        }
        void pointer_notify_motion(time stamp, fp64_coor delta, fp64_coor raw)
        {
            if (device_has_cap(LIBINPUT_DEVICE_CAP_POINTER))
            {
                auto& motion_event = li.libinput_emplace_event<libinput_event_pointer>();
                motion_event.delta     = delta;
                motion_event.delta_raw = raw;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_MOTION, motion_event);
            }
        }
        void pointer_notify_motion_absolute(time stamp, si32_coor point)
        {
            if (device_has_cap(LIBINPUT_DEVICE_CAP_POINTER))
            {
                auto& motion_absolute_event = li.libinput_emplace_event<libinput_event_pointer>();
                motion_absolute_event.absolute  = point;
                motion_absolute_event.absinfo_x = ud_device.abs.absinfo_x;
                motion_absolute_event.absinfo_y = ud_device.abs.absinfo_y;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_MOTION_ABSOLUTE, motion_absolute_event);
            }
        }
        si32 libevdev_get_id_bustype()
        {
            return ud_device.prod_info.bustype;
        }
        ui32 libevdev_get_id_product()
        {
            return ud_device.prod_info.product;
        }
        si32 libevdev_get_id_vendor()
        {
            return ud_device.prod_info.vendor;
        }
        si32 libinput_device_config_left_handed_is_available()
        {
            return config.left_handed ? config.left_handed->has(This()) : 0;
        }
        si32 libinput_device_config_left_handed_get()
        {
            auto lh = libinput_device_config_left_handed_is_available();
            return lh ? config.left_handed->get(This()) : 0;
        }
        fp64_coor evdev_device_units_to_mm(si32_coor units)
        {
            auto mm = fp64_coor{};
            if (!ud_device.abs.absinfo_x || !ud_device.abs.absinfo_y)
            {
                log("%s%: is not an abs device (1)", ud_device.devname);
            }
            else
            {
                auto& absx = ud_device.abs.absinfo_x;
                auto& absy = ud_device.abs.absinfo_y;
                mm.x = (units.x - absx.minimum) / absx.resolution;
                mm.y = (units.y - absy.minimum) / absy.resolution;
            }
            return mm;
        }
        void evdev_device_check_abs_axis_range(ui32 usage, si32 value)
        {
            auto min = 0;
            auto max = 0;
            switch (usage)
            {
                case evdev::abs_x:
                case evdev::abs_mt_position_x:
                    min = ud_device.abs.warning_range_x.min;
                    max = ud_device.abs.warning_range_x.max;
                    break;
                case evdev::abs_y:
                case evdev::abs_mt_position_y:
                    min = ud_device.abs.warning_range_y.min;
                    max = ud_device.abs.warning_range_y.max;
                    break;
                default:
                    return;
            }
            if (value < min || value > max)
            {
                log("Axis %#x% value %d% is outside expected range [%d%, %d%]", usage, value, min, max);
            }
        }
        si32 evdev_update_key_down_count(ui32 usage, si32 pressed)
        {
            assert(usage >= evdev::key_reserved && usage <= evdev::key_max);
            auto code = evdev_usage_code(usage);
            auto& count = key_count[code];
                 if (pressed)   count++;
            else if (count > 0) count--;
            else
            {
                log("Releasing key 'type=EV_KEY code=%%' with count %d%", code, count);
            }
            if (count > 32)
            {
                log("Key count for 'type=EV_KEY code=%%' reached abnormal values", code);
            }
            return count;
        }
        void pointer_notify_axis_finger(time stamp, ui32 active_axes, fp64_coor delta)
        {
            if (device_has_cap(LIBINPUT_DEVICE_CAP_POINTER))
            {
                auto& axis_event = li.libinput_emplace_event<libinput_event_pointer>();
                axis_event.delta       = delta;
                axis_event.discrete    = {};
                axis_event.v120        = {};
                axis_event.source      = LIBINPUT_POINTER_AXIS_SOURCE_FINGER;
                axis_event.active_axes = active_axes;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_SCROLL_FINGER, axis_event);
                auto& axis_event_legacy = li.libinput_emplace_event<libinput_event_pointer>();
                axis_event_legacy = axis_event;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_AXIS, axis_event_legacy);
            }
        }
        void pointer_notify_axis_continuous(time stamp, ui32 active_axes, fp64_coor delta)
        {
            if (device_has_cap(LIBINPUT_DEVICE_CAP_POINTER))
            {
                auto& axis_event = li.libinput_emplace_event<libinput_event_pointer>();
                axis_event.delta       = delta;
                axis_event.discrete    = {};
                axis_event.v120        = {};
                axis_event.source      = LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS;
                axis_event.active_axes = active_axes;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_SCROLL_CONTINUOUS, axis_event);
                auto& axis_event_legacy = li.libinput_emplace_event<libinput_event_pointer>();
                axis_event_legacy = axis_event;
                post_device_event(stamp, LIBINPUT_EVENT_POINTER_AXIS, axis_event_legacy);
            }
        }
            bool evdev_is_scrolling(libinput_pointer_axis axis)
            {
                assert(axis == LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL || axis == LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL);
                return (scroll.direction & (1ul << axis)) != 0;
            }
            void evdev_start_scrolling(libinput_pointer_axis axis)
            {
                assert(axis == LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL || axis == LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL);
                scroll.direction |= (1ul << axis);
            }
        void evdev_notify_axis_continous(time stamp, ui32 active_axes, fp64_coor delta)
        {
            if (scroll.natural_scrolling_enabled)
            {
                delta = -delta;
            }
            pointer_notify_axis_continuous(stamp, active_axes, delta);
        }
        void evdev_notify_axis_finger(time stamp, ui32 active_axes, fp64_coor delta)
        {
            if (scroll.natural_scrolling_enabled)
            {
                delta = -delta;
            }
            pointer_notify_axis_finger(stamp, active_axes, delta);
        }
        void evdev_post_scroll(time stamp, libinput_pointer_axis_source source, fp64_coor delta)
        {
            if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL))   scroll.buildup.y += delta.y;
            if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL)) scroll.buildup.x += delta.x;
            auto trigger = scroll.buildup;
            // If we're not scrolling yet, use a distance trigger: movingast a certain distance starts scrolling.
            if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL) && !evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL))
            {
                if (std::abs(trigger.y) >= scroll.threshold) evdev_start_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL);
                if (std::abs(trigger.x) >= scroll.threshold) evdev_start_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL);
            }
            // We're already scrolling in one direction. Require somerigger speed to start scrolling in the other direction.
            else if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL))
            {
                if (std::abs(delta.y) >= scroll.direction_lock_threshold) evdev_start_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL);
            }
            else if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL))
            {
                if (std::abs(delta.x) >= scroll.direction_lock_threshold) evdev_start_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL);
            }
            auto event = delta;
            // We use the trigger to enable, but the delta from this event for the actual scroll movement. Otherwise we get a jump once scrolling engages.
            if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL)) event.y = 0.0;
            if (!evdev_is_scrolling(LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL)) event.x = 0.0;
            if (event)
            {
                auto axes = scroll.direction;
                if (event.y == 0.0) axes &= ~(1ul << LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL);
                if (event.x == 0.0) axes &= ~(1ul << LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL);
                     if (source == LIBINPUT_POINTER_AXIS_SOURCE_FINGER)     evdev_notify_axis_finger(stamp, axes, event);
                else if (source == LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS) evdev_notify_axis_continous(stamp, axes, event);
                else log("Posting invalid scroll source %d%", source);
            }
        }
        void evdev_transform_absolute(si32_coor& point)
        {
            if (ud_device.abs.apply_calibration)
            {
                ud_device.abs.calibration.matrix_mult_vec(point);
            }
        }
        ui32 evdev_to_left_handed(ui32 button)
        {
            if (dev_left_handed.enabled)
            {
                     if (button == evdev::btn_left)  return evdev::btn_right;
                else if (button == evdev::btn_right) return evdev::btn_left;
            }
            return button;
        }
                        void middlebutton_state_error([[maybe_unused]] evdev_middlebutton_event event)
                        {
                            log("Invalid event %s% in middle button state %s%", middlebutton_event_to_str(event), middlebutton_state_to_str(middlebutton.state));
                        }
                        void middlebutton_set_state(evdev_middlebutton_state state, time now)
                        {
                            switch (state)
                            {
                                case MIDDLEBUTTON_LEFT_DOWN:
                                case MIDDLEBUTTON_RIGHT_DOWN:
                                    middlebutton.timer->start(now + lixx::middlebutton_timeout);
                                    middlebutton.first_event_time = now;
                                    break;
                                case MIDDLEBUTTON_IDLE:
                                case MIDDLEBUTTON_MIDDLE:
                                case MIDDLEBUTTON_LEFT_UP_PENDING:
                                case MIDDLEBUTTON_RIGHT_UP_PENDING:
                                case MIDDLEBUTTON_PASSTHROUGH:
                                case MIDDLEBUTTON_IGNORE_LR:
                                case MIDDLEBUTTON_IGNORE_L:
                                case MIDDLEBUTTON_IGNORE_R:
                                    middlebutton.timer->cancel();
                                    break;
                            }
                            middlebutton.state = state;
                        }
                    si32 evdev_middlebutton_idle_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN: middlebutton_set_state(MIDDLEBUTTON_LEFT_DOWN, stamp); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN: middlebutton_set_state(MIDDLEBUTTON_RIGHT_DOWN, stamp); break;
                            case MIDDLEBUTTON_EVENT_OTHER: return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:
                            case MIDDLEBUTTON_EVENT_L_UP:
                            case MIDDLEBUTTON_EVENT_TIMEOUT:
                                middlebutton_state_error(event);
                                break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:
                                break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_ldown_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN: evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_MIDDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_OTHER:  evdev_pointer_notify_button(stamp, evdev::btn_left, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:   middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_L_UP:   evdev_pointer_notify_button(middlebutton.first_event_time, evdev::btn_left, evdev::pressed); evdev_pointer_notify_button(stamp, evdev::btn_left, evdev::released); middlebutton_set_state(MIDDLEBUTTON_IDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT:evdev_pointer_notify_button(middlebutton.first_event_time, evdev::btn_left, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP: middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_rdown_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:  evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_MIDDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_OTHER:   evdev_pointer_notify_button(middlebutton.first_event_time, evdev::btn_right, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    evdev_pointer_notify_button(middlebutton.first_event_time, evdev::btn_right, evdev::pressed); evdev_pointer_notify_button(stamp, evdev::btn_right, evdev::released); middlebutton_set_state(MIDDLEBUTTON_IDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_L_UP:    middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: evdev_pointer_notify_button(middlebutton.first_event_time, evdev::btn_right, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_middle_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:
                            case MIDDLEBUTTON_EVENT_R_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_OTHER:   evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::released); middlebutton_set_state(MIDDLEBUTTON_IGNORE_LR, stamp);        return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::released); middlebutton_set_state(MIDDLEBUTTON_LEFT_UP_PENDING, stamp);  break;
                            case MIDDLEBUTTON_EVENT_L_UP:    evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::released); middlebutton_set_state(MIDDLEBUTTON_RIGHT_UP_PENDING, stamp); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_lup_pending_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_OTHER:   middlebutton_set_state(MIDDLEBUTTON_IGNORE_L, stamp); return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_L_UP:    middlebutton_set_state(MIDDLEBUTTON_IDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_L_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN:
                                evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::pressed);
                                middlebutton_set_state(MIDDLEBUTTON_MIDDLE, stamp);
                                break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_rup_pending_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:  evdev_pointer_notify_button(stamp, evdev::btn_middle, evdev::pressed); middlebutton_set_state(MIDDLEBUTTON_MIDDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_OTHER:   middlebutton_set_state(MIDDLEBUTTON_IGNORE_R, stamp); return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    middlebutton_set_state(MIDDLEBUTTON_IDLE, stamp); break;
                            case MIDDLEBUTTON_EVENT_L_UP:    middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_passthrough_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:
                            case MIDDLEBUTTON_EVENT_R_DOWN:
                            case MIDDLEBUTTON_EVENT_OTHER:
                            case MIDDLEBUTTON_EVENT_R_UP:
                            case MIDDLEBUTTON_EVENT_L_UP:    return 0;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_set_state(MIDDLEBUTTON_IDLE, stamp); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_ignore_lr_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:
                            case MIDDLEBUTTON_EVENT_R_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_OTHER:   return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    middlebutton_set_state(MIDDLEBUTTON_IGNORE_L, stamp); break;
                            case MIDDLEBUTTON_EVENT_L_UP:    middlebutton_set_state(MIDDLEBUTTON_IGNORE_R, stamp); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT: middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_ignore_l_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_R_DOWN:  return 0;
                            case MIDDLEBUTTON_EVENT_OTHER:
                            case MIDDLEBUTTON_EVENT_R_UP:    return 0;
                            case MIDDLEBUTTON_EVENT_L_UP:    middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); break;
                            case MIDDLEBUTTON_EVENT_TIMEOUT:
                            case MIDDLEBUTTON_EVENT_ALL_UP:  middlebutton_state_error(event); break;
                        }
                        return 1;
                    }
                    si32 evdev_middlebutton_ignore_r_handle_event(time stamp, evdev_middlebutton_event event)
                    {
                        switch (event)
                        {
                            case MIDDLEBUTTON_EVENT_L_DOWN:  return 0;
                            case MIDDLEBUTTON_EVENT_R_DOWN:  middlebutton_state_error(event); break;
                            case MIDDLEBUTTON_EVENT_OTHER:   return 0;
                            case MIDDLEBUTTON_EVENT_R_UP:    middlebutton_set_state(MIDDLEBUTTON_PASSTHROUGH, stamp); break;
                            case MIDDLEBUTTON_EVENT_L_UP:    return 0;
                            case MIDDLEBUTTON_EVENT_TIMEOUT:
                            case MIDDLEBUTTON_EVENT_ALL_UP:  break;
                        }
                        return 1;
                    }
                si32 evdev_middlebutton_handle_event(time stamp, evdev_middlebutton_event event)
                {
                    auto rc = 0;
                    auto current = middlebutton.state;
                    switch (current)
                    {
                        case MIDDLEBUTTON_IDLE:             rc = evdev_middlebutton_idle_handle_event(       stamp, event); break;
                        case MIDDLEBUTTON_LEFT_DOWN:        rc = evdev_middlebutton_ldown_handle_event(      stamp, event); break;
                        case MIDDLEBUTTON_RIGHT_DOWN:       rc = evdev_middlebutton_rdown_handle_event(      stamp, event); break;
                        case MIDDLEBUTTON_MIDDLE:           rc = evdev_middlebutton_middle_handle_event(     stamp, event); break;
                        case MIDDLEBUTTON_LEFT_UP_PENDING:  rc = evdev_middlebutton_lup_pending_handle_event(stamp, event); break;
                        case MIDDLEBUTTON_RIGHT_UP_PENDING: rc = evdev_middlebutton_rup_pending_handle_event(stamp, event); break;
                        case MIDDLEBUTTON_PASSTHROUGH:      rc = evdev_middlebutton_passthrough_handle_event(stamp, event); break;
                        case MIDDLEBUTTON_IGNORE_LR:        rc = evdev_middlebutton_ignore_lr_handle_event(  stamp, event); break;
                        case MIDDLEBUTTON_IGNORE_L:         rc = evdev_middlebutton_ignore_l_handle_event(   stamp, event); break;
                        case MIDDLEBUTTON_IGNORE_R:         rc = evdev_middlebutton_ignore_r_handle_event(   stamp, event); break;
                        default: log("Invalid middle button state %d%", current); break;
                    }
                    log("middlebutton state0: %s% → %s% → %s%, rc %d%", middlebutton_state_to_str(current), middlebutton_event_to_str(event), middlebutton_state_to_str(middlebutton.state), rc);
                    return rc;
                }
                void evdev_middlebutton_apply_config()
                {
                    if (middlebutton.want_enabled == middlebutton.enabled) return;
                    if (middlebutton.button_mask != 0) return;
                    middlebutton.enabled = middlebutton.want_enabled;
                }
            bool evdev_middlebutton_filter_button(time stamp, ui32 button, si32 state)
            {
                auto event = evdev_middlebutton_event{};
                auto is_press = state == evdev::pressed;
                auto rc = 0;
                auto btnbit = (button - evdev::btn_left);
                auto old_mask = 0u;
                if (!middlebutton.enabled) return faux;
                switch (button)
                {
                    case evdev::btn_left:
                        if (is_press) event = MIDDLEBUTTON_EVENT_L_DOWN;
                        else          event = MIDDLEBUTTON_EVENT_L_UP;
                        break;
                    case evdev::btn_right:
                        if (is_press) event = MIDDLEBUTTON_EVENT_R_DOWN;
                        else          event = MIDDLEBUTTON_EVENT_R_UP;
                        break;
                    // BTN_MIDDLE counts as "other" and resets middle button emulation.
                    case evdev::btn_middle:
                    default:
                        event = MIDDLEBUTTON_EVENT_OTHER;
                        break;
                }
                if (button < evdev::btn_left || btnbit >= sizeof(middlebutton.button_mask) * 8)
                {
                    log("Button mask too small for 'code=%%'", button);
                    return true;
                }
                rc = evdev_middlebutton_handle_event(stamp, event);
                old_mask = middlebutton.button_mask;
                if (is_press) middlebutton.button_mask |= (1ul << btnbit);
                else          middlebutton.button_mask &= ~(1ul << btnbit);
                if (old_mask != middlebutton.button_mask && middlebutton.button_mask == 0)
                {
                    evdev_middlebutton_handle_event(stamp, MIDDLEBUTTON_EVENT_ALL_UP);
                    evdev_middlebutton_apply_config();
                }
                return rc;
            }
        void evdev_pointer_notify_physical_button(time stamp, ui32 button, si32 state)
        {
            if (!evdev_middlebutton_filter_button(stamp, button, state))
            {
                evdev_pointer_notify_button(stamp, button, state);
            }
        }
        void evdev_stop_scroll(time stamp, libinput_pointer_axis_source source)
        {
            if (scroll.direction != 0) // Terminate scrolling with a zero scroll event.
            {
                     if (source == LIBINPUT_POINTER_AXIS_SOURCE_FINGER)     pointer_notify_axis_finger(stamp, scroll.direction, {});
                else if (source == LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS) pointer_notify_axis_continuous(stamp, scroll.direction, {});
                else log("Stopping invalid scroll source %d%", source);
            }
            scroll.buildup.x = 0;
            scroll.buildup.y = 0;
            scroll.direction = 0;
        }
            void evdev_button_scroll_button(time stamp, si32 is_press)
            {
                // Where the button lock is enabled, we wrap the buttons into their own little state machine and filter out the events.
                switch (scroll.lock_state)
                {
                    case BUTTONSCROLL_LOCK_DISABLED: break;
                    case BUTTONSCROLL_LOCK_IDLE:       assert(is_press);  scroll.lock_state = BUTTONSCROLL_LOCK_FIRSTDOWN;  log("scroll lock: first down");  break;  // Handle event.
                    case BUTTONSCROLL_LOCK_FIRSTDOWN:  assert(!is_press); scroll.lock_state = BUTTONSCROLL_LOCK_FIRSTUP;    log("scroll lock: first up");    return; // Filter release event.
                    case BUTTONSCROLL_LOCK_FIRSTUP:    assert(is_press);  scroll.lock_state = BUTTONSCROLL_LOCK_SECONDDOWN; log("scroll lock: second down"); return; // Filter press event.
                    case BUTTONSCROLL_LOCK_SECONDDOWN: assert(!is_press); scroll.lock_state = BUTTONSCROLL_LOCK_IDLE;       log("scroll lock: idle");        break;  // Handle event.
                }
                if (is_press)
                {
                    if (scroll.button < evdev::btn_left + 5)
                    {
                        // For mouse buttons 1-5 (0x110 to 0x114) we apply a timeout before scrolling since the button could also be used for regular clicking.
                        auto allow_negative = faux;
                        scroll.button_scroll_state = BUTTONSCROLL_BUTTON_DOWN;
                        // Special case: if middle button emulation is enabled and our scroll button is the left or right button, we only get here *after* the middle button timeout has expired for that button press. The time passed is the button-down time though (which is in the past), so we have to allow for a negative timer to be set.
                        if (middlebutton.enabled && (scroll.button == evdev::btn_left || scroll.button == evdev::btn_right))
                        {
                            allow_negative = true;
                        }
                        scroll.timer->start(stamp + lixx::default_button_scroll_timeout, allow_negative);
                    }
                    else // For extra mouse buttons numbered 6 or more (0x115+) we assume it is dedicated exclusively to scrolling, so we don't apply the timeout in order to provide immediate scrolling responsiveness.
                    {
                        scroll.button_scroll_state = BUTTONSCROLL_READY;
                    }
                    scroll.button_down_time = stamp;
                    log("btnscroll: down");
                }
                else
                {
                    scroll.timer->cancel();
                    switch (scroll.button_scroll_state)
                    {
                        case BUTTONSCROLL_IDLE: log("invalid state IDLE for button up"); break;
                        case BUTTONSCROLL_BUTTON_DOWN:
                        case BUTTONSCROLL_READY:
                            log("btnscroll: cancel");
                            // If the button is released quickly enough or without scroll events, emit the button press/release events.
                            evdev_pointer_post_button(scroll.button_down_time, scroll.button, evdev::pressed);
                            evdev_pointer_post_button(stamp, scroll.button, evdev::released);
                            break;
                        case BUTTONSCROLL_SCROLLING:
                            log("btnscroll: up");
                            evdev_stop_scroll(stamp, LIBINPUT_POINTER_AXIS_SOURCE_CONTINUOUS);
                            break;
                    }
                    scroll.button_scroll_state = BUTTONSCROLL_IDLE;
                }
            }
        void evdev_pointer_notify_button(time stamp, ui32 button, si32 state)
        {
            if (scroll.method == LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN && button == scroll.button)
            {
                evdev_button_scroll_button(stamp, state);
            }
            else
            {
                evdev_pointer_post_button(stamp, button, state);
            }
        }
            void pointer_notify_button(time stamp, ui32 button, si32 state)
            {
                if (device_has_cap(LIBINPUT_DEVICE_CAP_POINTER))
                {
                    auto seat_button_count = li.update_press_count(button, state);
                    auto& button_event = li.libinput_emplace_event<libinput_event_pointer>();
                    button_event.button            = button;
                    button_event.seat_button_count = seat_button_count;
                    button_event.state             = state;
                    post_device_event(stamp, LIBINPUT_EVENT_POINTER_BUTTON, button_event);
                }
            }
        void evdev_pointer_post_button(time stamp, ui32 button, si32 state)
        {
            auto down_count = evdev_update_key_down_count(button, state);
            if ((state == evdev::pressed && down_count == 1) || (state == evdev::released && down_count == 0))
            {
                pointer_notify_button(stamp, evdev_usage_code(button), state);
                if (state == evdev::released)
                {
                    if (dev_left_handed.change_to_enabled) dev_left_handed.change_to_enabled(This());
                    if (scroll.change_scroll_method) scroll.change_scroll_method(This());
                }
            }
        }
        void evdev_notify_suspended_device()
        {
            if (!is_suspended)
            {
                auto self = This();
                for (auto d : li.device_list) // Fire on all but specified.
                {
                    if (d != self)
                    {
                        d->device_suspended(self);
                    }
                }
                is_suspended = true;
            }
        }
        void evdev_notify_resumed_device()
        {
            if (is_suspended)
            {
                auto self = This();
                for (auto d : li.device_list)
                {
                    if (d != self)
                    {
                        d->device_resumed(self);
                    }
                }
                is_suspended = faux;
            }
        }
        si32 evdev_need_mtdev()
        {
            return (libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_X) && libevdev_has_event_code<EV_ABS>(ABS_MT_POSITION_Y)
                && !libevdev_has_event_code<EV_ABS>(ABS_MT_SLOT));
        }
        void evdev_process_event(evdev_event& ev, time now)
        {
            #if EVENT_DEBUGGING
            evdev_print_event(ev, now);
            #endif
            li.timers.libinput_timer_flush(now);
            process(ev, now);
        }
        void evdev_device_dispatch_one(evdev_event& ev, time now)
        {
            evdev_process_event(ev, now);
        }
        void evdev_device_dispatch_frame(evdev_frame& frame)
        {
            for (auto& ev : frame.ev_events)
            {
                evdev_device_dispatch_one(ev, frame.stamp);
            }
        }
        si32 evdev_sync_device()
        {
            auto ev = input_event_t{};
            auto rc = evdev::success;
            frame.evdev_frame_reset();
            do
            {
                rc = ud_device.libevdev_next_event(LIBEVDEV_READ_FLAG_SYNC, ev);
                if (rc < 0) break;
                frame.evdev_frame_append_input_event(ev);
            }
            while (rc == evdev::sync);
            evdev_device_dispatch_frame(frame);
            return rc == -EAGAIN ? evdev::success : rc;
        }
        void evdev_note_time_delay(input_event_t& ev)
        {
            auto event_time = ev.input_event_time();
            if (li.dispatch_time != time{} && event_time <= li.dispatch_time) // If we have a current libinput_dispatch() snapshot, compare our event time with the one from the snapshot. If we have more than 10ms delay, complain about it. This catches delays in processing where there is no steady event flow and thus SYN_DROPPED may not get hit by the kernel despite us being too slow.
            {
                auto tdelta = li.dispatch_time - event_time;
                if (tdelta > 20ms)
                {
                    log("event processing lagging behind by %dms%, your system is too slow", tdelta);
                }
            }
        }
        void notify_removed_device()
        {
            auto& event = li.libinput_emplace_event<libinput_event_device_notify>();
            event.stamp     = datetime::now();
            event.type      = LIBINPUT_EVENT_DEVICE_REMOVED;
            event.li_device = This();
        }
        void notify_added_device()
        {
            auto& event = li.libinput_emplace_event<libinput_event_device_notify>();
            event.stamp     = datetime::now();
            event.type      = LIBINPUT_EVENT_DEVICE_ADDED;
            event.li_device = This();
        }
        void evdev_device_remove()
        {
            if (auto& timer = scroll.timer)       timer->cancel();
            if (auto& timer = middlebutton.timer) timer->cancel();
            evdev_device_suspend();
            remove();
            was_removed = true; // A device may be removed while suspended, mark it to skip re-opening a different device with the same node.
            notify_removed_device();
        }
        void evdev_device_dispatch()
        {
            auto ev = input_event_t{};
            auto rc = evdev::success;
            auto once = faux;
            frame.evdev_frame_reset();
            do // If the compositor is repainting, this function is called only once per frame and we have to process all the events available on the fd, otherwise there will be input lag.
            {
                rc = ud_device.libevdev_next_event(LIBEVDEV_READ_FLAG_NORMAL, ev);
                if (rc == evdev::sync)
                {
                    log("SYN_DROPPED event - some input events have been lost.");
                    ev.code = SYN_REPORT; // Send one more sync event so we handle all currently pending events before we sync up to the current state.
                    frame.evdev_frame_append_input_event(ev);
                    evdev_device_dispatch_frame(frame);
                    frame.evdev_frame_reset();
                    rc = evdev_sync_device();
                }
                else if (rc == evdev::success)
                {
                    if (!once)
                    {
                        evdev_note_time_delay(ev);
                        once = true;
                    }
                    frame.evdev_frame_append_input_event(ev);
                    if (ev.type == EV_SYN && ev.code == SYN_REPORT)
                    {
                        evdev_device_dispatch_frame(frame); //todo filter/drop kernel auto-repeat events (ev.value=1) for tp devices
                        frame.evdev_frame_reset();
                    }
                }
                else if (rc == -ENODEV)
                {
                    remove_device();
                    return;
                }
            }
            while (rc == evdev::success);
            if (frame.ev_events.size() > 1) // This should never happen, the kernel flushes only on SYN_REPORT.
            {
                log("event frame missing SYN_REPORT, forcing frame");
                evdev_device_dispatch_frame(frame);
            }
            if (rc != -EAGAIN && rc != -EINTR)
            {
                li.timers.libinput_remove_event_source(source);
            }
        }
        void remove_device()
        {
            auto self = This();
            std::erase_if(li.device_list, [&](auto d)
            {
                if (d == self)
                {
                    self->evdev_device_remove();
                    return true;
                }
                else
                {
                    d->device_removed(self);
                    return faux;
                }
            });
            std::erase_if(li.ud_monitor.ud_device_list, [&](auto d){ return d.second.get() == &ud_device; });
            log("Device '%%' removed", ud_device.devname);
        }
        si32 evdev_device_resume()
        {
            if (ud_device.fd != os::invalid_fd) return 0;
            if (was_removed) return -ENODEV;
            auto devnode = qiew{ ud_device.devpath };
            if (!devnode) return -ENODEV;
            auto new_fd = ::open(devnode.data(), O_RDWR | O_NONBLOCK | O_CLOEXEC);
            if (new_fd == os::invalid_fd) return -errno;
            ud_device_t::evdev_drain_fd(new_fd);
            ud_device.libevdev_change_fd(new_fd);
            // Re-sync libevdev's view of the device, but discard the actual events. Our device is in a neutral state already.
            auto ev = input_event_t{};
            ud_device.libevdev_next_event(LIBEVDEV_READ_FLAG_FORCE_SYNC, ev);
            while (ud_device.libevdev_next_event(LIBEVDEV_READ_FLAG_SYNC, ev) == evdev::sync)
            { }
            source = li.timers.libinput_add_event_source(ud_device.fd, [&]{ evdev_device_dispatch(); });
            if (!source)
            {
                return -ENOMEM;
            }
            evdev_notify_resumed_device();
            return 0;
        }
        void evdev_device_suspend()
        {
            evdev_notify_suspended_device();
            suspend();
            if (source)
            {
                li.timers.libinput_remove_event_source(source);
            }
            if (ud_device.fd != os::invalid_fd)
            {
                os::close(ud_device.fd);
                ud_device.fd = os::invalid_fd;
            }
        }
        si32_coor evdev_device_mm_to_units(fp64_coor mm)
        {
            // Convert the pair of coordinates in mm to device units. This takes the axis min into account, i.e. 0 mm  is equivalent to the min.
            auto units = si32_coor{};
            if (!ud_device.abs.absinfo_x || !ud_device.abs.absinfo_y)
            {
                log("%s%: is not an abs device (2)", ud_device.devname);
            }
            else
            {
                auto& absx = ud_device.abs.absinfo_x;
                auto& absy = ud_device.abs.absinfo_y;
                units.x = mm.x * absx.resolution + absx.minimum;
                units.y = mm.y * absy.resolution + absy.minimum;
            }
            return units;
        }
        bool evdev_device_has_model_quirk(quirk model_quirk)
        {
            return ud_device.evdev_device_has_model_quirk(li, model_quirk);
        }
        si32 libevdev_fetch_slot_value(ui32 slot, ui32 code, si32& value)
        {
            return ud_device.libevdev_fetch_slot_value(slot, code, value);
        }
        template<ui32 Type>
        si32 libevdev_has_event_code(ui32 code)
        {
            return ud_device.libevdev_has_event_code<Type>(code);
        }
        template<ui32 Type>
        void libevdev_disable_event_code(ui32 code)
        {
            ud_device.libevdev_disable_event_code<Type>(code);
        }
        si32 libevdev_has_property(ui32 prop)
        {
            return (prop < ud_device.prop_bits.size()) && ud_device.prop_bits[prop];
        }
        si32 libevdev_get_slot_value(ui32 slot, ui32 code)
        {
            return ud_device.libevdev_get_slot_value(slot, code);
        }
        template<ui32 Type>
        si32 libevdev_get_event_value(ui32 code)
        {
            return ud_device.libevdev_get_event_value<Type>(code);
        }
        si32 libevdev_get_abs_resolution(ui32 code)
        {
            return ud_device.libevdev_get_abs_resolution(code);
        }
        abs_info_t& libevdev_get_abs_info(ui32 code)
        {
            return ud_device.libevdev_get_abs_info(code);
        }
        auto evdev_device_get_size()
        {
            return ud_device.evdev_device_get_size();
        }
            static bool evdev_middlebutton_get(libinput_device_sptr li_device)
            {
                return li_device->middlebutton.want_enabled;
            }
            static bool evdev_middlebutton_get_default(libinput_device_sptr li_device)
            {
                return li_device->middlebutton.enabled_default;
            }
            static libinput_config_status evdev_middlebutton_set(libinput_device_sptr li_device, bool middle_emulation_enabled)
            {
                li_device->middlebutton.want_enabled = middle_emulation_enabled;
                li_device->evdev_middlebutton_apply_config();
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static si32 evdev_middlebutton_is_available([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 1;
            }
        void evdev_init_middlebutton(bool enable, bool want_config)
        {
            auto timer_name = utf::fprint("%s% middlebutton", ud_device.sysname);
            middlebutton.timer = li.timers.create(timer_name, [&](time now){ evdev_middlebutton_handle_event(now, MIDDLEBUTTON_EVENT_TIMEOUT); });
            middlebutton.enabled_default = enable;
            middlebutton.want_enabled    = enable;
            middlebutton.enabled         = enable;
            if (want_config)
            {
                middlebutton.config.available   = evdev_middlebutton_is_available;
                middlebutton.config.set         = evdev_middlebutton_set;
                middlebutton.config.get         = evdev_middlebutton_get;
                middlebutton.config.get_default = evdev_middlebutton_get_default;
                config.middle_emulation = &middlebutton.config;
            }
        }
            static si32 evdev_scroll_config_natural_has([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 1;
            }
            static libinput_config_status evdev_scroll_config_natural_set(libinput_device_sptr li_device, si32 enabled)
            {
                li_device->scroll.natural_scrolling_enabled = enabled ? true : faux;
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static si32 evdev_scroll_config_natural_get(libinput_device_sptr li_device)
            {
                return li_device->scroll.natural_scrolling_enabled ? 1 : 0;
            }
            static si32 evdev_scroll_config_natural_get_default([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 0; // Overridden in evdev-mt-touchpad.c for Apple touchpads.
            }
        void evdev_init_natural_scroll()
        {
            scroll.config_natural.has                 = evdev_scroll_config_natural_has;
            scroll.config_natural.set_enabled         = evdev_scroll_config_natural_set;
            scroll.config_natural.get_enabled         = evdev_scroll_config_natural_get;
            scroll.config_natural.get_default_enabled = evdev_scroll_config_natural_get_default;
            scroll.natural_scrolling_enabled          = faux;
            config.natural_scroll = &scroll.config_natural;
        }
        void tp_disable_abs_mt()
        {
            for (auto code = ABS_MT_SLOT; code <= ABS_MAX; code++)
            {
                libevdev_disable_event_code<EV_ABS>(code);
            }
        }
            static si32 evdev_left_handed_has([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 1; // This is only hooked up when we have left-handed configuration, so we can hardcode 1 here.
            }
            static libinput_config_status evdev_left_handed_set(libinput_device_sptr li_device, si32 left_handed)
            {
                li_device->dev_left_handed.want_enabled = left_handed ? true : faux;
                li_device->dev_left_handed.change_to_enabled(li_device);
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static si32 evdev_left_handed_get(libinput_device_sptr li_device)
            {
                return li_device->dev_left_handed.want_enabled; // Return the wanted configuration, even if it hasn't taken effect yet!
            }
            static si32 evdev_left_handed_get_default([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 0;
            }
        void evdev_init_left_handed(void(*change_to_left_handed)(libinput_device_sptr))
        {
            dev_left_handed.config.has         = evdev_left_handed_has;
            dev_left_handed.config.set         = evdev_left_handed_set;
            dev_left_handed.config.get         = evdev_left_handed_get;
            dev_left_handed.config.get_default = evdev_left_handed_get_default;
            config.left_handed = &dev_left_handed.config;
            dev_left_handed.enabled            = faux;
            dev_left_handed.want_enabled       = faux;
            dev_left_handed.change_to_enabled  = change_to_left_handed;
        }
                void evdev_device_calibrate(std::array<fp32, 6> const& calibration)
                {
                    auto scale = matrix{};
                    auto translate = matrix{};
                    auto transform = matrix{};
                    transform.matrix_from_farray6(calibration);
                    ud_device.abs.apply_calibration = !transform.matrix_is_identity();
                    ud_device.abs.usermatrix.matrix_from_farray6(calibration); // Back up the user matrix so we can return it on request.
                    if (!ud_device.abs.apply_calibration)
                    {
                        ud_device.abs.calibration.matrix_init_identity();
                        return;
                    }
                    auto sx = ud_device.abs.absinfo_x.absinfo_range();
                    auto sy = ud_device.abs.absinfo_y.absinfo_range();
                    // The transformation matrix is in the form:
                    //  [ a b c ]
                    //  [ d e f ]
                    //  [ 0 0 1 ]
                    // Where a, e are the scale components, a, b, d, e are the rotation
                    // component (combined with scale) and c and f are the translation
                    // component. The translation component in the input matrix must be
                    // normalized to multiples of the device width and height,
                    // respectively. e.g. c == 1 shifts one device-width to the right.
                    //
                    // We pre-calculate a single matrix to apply to event coordinates:
                    //     M = Un-Normalize * Calibration * Normalize
                    //
                    // Normalize: scales the device coordinates to [0,1]
                    // Calibration: user-supplied matrix
                    // Un-Normalize: scales back up to device coordinates
                    // Matrix maths requires the normalize/un-normalize in reverse order.
                    //
                    // - Un-Normalize.
                    translate.matrix_init_translate(ud_device.abs.absinfo_x.minimum, ud_device.abs.absinfo_y.minimum);
                    scale.matrix_init_scale(sx, sy);
                    scale.matrix_mult(translate, scale);
                    // - Calibrate.
                    transform.matrix_mult(scale, transform);
                    // - Normalize.
                    translate.matrix_init_translate(-ud_device.abs.absinfo_x.minimum / sx, -ud_device.abs.absinfo_y.minimum / sy);
                    scale.matrix_init_scale(1.0 / sx, 1.0 / sy);
                    scale.matrix_mult(translate, scale);
                    // - Store final matrix in device.
                    ud_device.abs.calibration.matrix_mult(transform, scale);
                }
            static si32 evdev_calibration_has_matrix(libinput_device_sptr li_device)
            {
                return li_device->ud_device.abs.absinfo_x && li_device->ud_device.abs.absinfo_y;
            }
            static libinput_config_status evdev_calibration_set_matrix(libinput_device_sptr li_device, std::array<fp32, 6> const& matrix_data)
            {
                li_device->evdev_device_calibrate(matrix_data);
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static si32 evdev_calibration_get_matrix(libinput_device_sptr li_device, std::array<fp32, 6>& matrix_data)
            {
                li_device->ud_device.abs.usermatrix.matrix_to_farray6(matrix_data);
                return !li_device->ud_device.abs.usermatrix.matrix_is_identity();
            }
            static si32 evdev_calibration_get_default_matrix(libinput_device_sptr li_device, std::array<fp32, 6>& matrix_data)
            {
                li_device->ud_device.abs.default_calibration.matrix_to_farray6(matrix_data);
                return !li_device->ud_device.abs.default_calibration.matrix_is_identity();
            }
        void evdev_init_calibration(libinput_device_config_calibration& calibration)
        {
            config.calibration = &calibration;
            calibration.has_matrix         = evdev_calibration_has_matrix;
            calibration.set_matrix         = evdev_calibration_set_matrix;
            calibration.get_matrix         = evdev_calibration_get_matrix;
            calibration.get_default_matrix = evdev_calibration_get_default_matrix;
        }
            void evdev_button_scroll_timeout()
            {
                scroll.button_scroll_state = BUTTONSCROLL_READY;
            }
            static ui32 evdev_scroll_get_methods([[maybe_unused]] libinput_device_sptr li_device)
            {
                return LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN;
            }
            static libinput_config_status evdev_scroll_set_method(libinput_device_sptr li_device, libinput_config_scroll_method method)
            {
                li_device->scroll.want_method = method;
                li_device->scroll.change_scroll_method(li_device);
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static libinput_config_scroll_method evdev_scroll_get_method(libinput_device_sptr li_device)
            {
                return li_device->scroll.want_method; // Return the wanted configuration, even if it hasn't taken effect yet!
            }
            static libinput_config_scroll_method evdev_scroll_get_default_method(libinput_device_sptr li_device)
            {
                auto on_button_down = (li_device->device_tags & EVDEV_TAG_TRACKPOINT)
                                  || (!li_device->libevdev_has_event_code<EV_REL>(REL_WHEEL) // Mice without a scroll wheel but with middle button have on-button scrolling by default.
                                   && !li_device->libevdev_has_event_code<EV_REL>(REL_HWHEEL)
                                   &&  li_device->libevdev_has_event_code<EV_KEY>(BTN_MIDDLE));
                return on_button_down ? LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN : LIBINPUT_CONFIG_SCROLL_NO_SCROLL;
            }
            static libinput_config_status evdev_scroll_set_button(libinput_device_sptr li_device, ui32 button)
            {
                li_device->scroll.want_button = evdev_usage_from_code(EV_KEY, button);
                li_device->scroll.change_scroll_method(li_device);
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static ui32 evdev_scroll_get_button(libinput_device_sptr li_device)
            {
                return evdev_usage_code(li_device->scroll.want_button); // Return the wanted configuration, even if it hasn't taken effect yet!
            }
            static ui32 evdev_scroll_get_default_button(libinput_device_sptr li_device)
            {
                if (li_device->libevdev_has_event_code<EV_KEY>(BTN_MIDDLE))
                {
                    return BTN_MIDDLE;
                }
                for (auto code = BTN_SIDE; code <= BTN_TASK; code++)
                {
                    if (li_device->libevdev_has_event_code<EV_KEY>(code))
                    {
                        return code;
                    }
                }
                if (li_device->libevdev_has_event_code<EV_KEY>(BTN_RIGHT))
                {
                    return BTN_RIGHT;
                }
                return 0;
            }
            static libinput_config_status evdev_scroll_set_button_lock(libinput_device_sptr li_device, bool scroll_button_lock_enabled)
            {
                li_device->scroll.want_lock_enabled = scroll_button_lock_enabled;
                li_device->scroll.change_scroll_method(li_device);
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static bool evdev_scroll_get_button_lock(libinput_device_sptr li_device)
            {
                return li_device->scroll.lock_state;
            }
            static bool evdev_scroll_get_default_button_lock([[maybe_unused]] libinput_device_sptr li_device)
            {
                return faux;
            }
        void evdev_init_button_scroll(void(*change_scroll_method)(libinput_device_sptr))
        {
            auto timer_name = utf::fprint("%s% btnscroll", ud_device.sysname);
            scroll.timer = li.timers.create(timer_name, [&](time){ evdev_button_scroll_timeout(); });
            scroll.config.get_methods             = evdev_scroll_get_methods;
            scroll.config.set_method              = evdev_scroll_set_method;
            scroll.config.get_method              = evdev_scroll_get_method;
            scroll.config.get_default_method      = evdev_scroll_get_default_method;
            scroll.config.set_button              = evdev_scroll_set_button;
            scroll.config.get_button              = evdev_scroll_get_button;
            scroll.config.get_default_button      = evdev_scroll_get_default_button;
            scroll.config.set_button_lock         = evdev_scroll_set_button_lock;
            scroll.config.get_button_lock         = evdev_scroll_get_button_lock;
            scroll.config.get_default_button_lock = evdev_scroll_get_default_button_lock;
            config.scroll_method = &scroll.config;
            scroll.method               = evdev_scroll_get_default_method(This());
            scroll.want_method          = scroll.method;
            scroll.button               = evdev_usage_from_code(EV_KEY, evdev_scroll_get_default_button(This()));
            scroll.want_button          = scroll.button;
            scroll.change_scroll_method = change_scroll_method;
        }
        bool evdev_is_fake_mt_device()
        {
            return ud_device.evdev_is_fake_mt_device();
        }
        void tablet_notify_proximity(time now, libinput_tablet_tool_sptr tool, libinput_tablet_tool_proximity_state proximity_state, tablet_axes_bitset& changed_axes, tablet_axes const& axes, abs_info_t const& x, abs_info_t const& y)
        {
            auto& proximity_event = li.libinput_emplace_event<libinput_event_tablet_tool>();
            proximity_event.axes              = axes;
            proximity_event.tool              = tool;
            proximity_event.proximity_state   = proximity_state;
            proximity_event.tip_state         = LIBINPUT_TABLET_TOOL_TIP_UP;
            proximity_event.abs_info_x2       = x;
            proximity_event.abs_info_y2       = y;
            proximity_event.changed_axes_bits = changed_axes;
            post_device_event(now, LIBINPUT_EVENT_TABLET_TOOL_PROXIMITY, proximity_event);
        }
        void tablet_notify_tip(time now, libinput_tablet_tool_sptr tool, libinput_tablet_tool_tip_state tip_state, tablet_axes_bitset& changed_axes, tablet_axes const& axes, abs_info_t const& x, abs_info_t const& y)
        {
            auto& tip_event = li.libinput_emplace_event<libinput_event_tablet_tool>();
            tip_event.axes              = axes;
            tip_event.tool              = tool;
            tip_event.proximity_state   = LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN;
            tip_event.tip_state         = tip_state;
            tip_event.abs_info_x2       = x;
            tip_event.abs_info_y2       = y;
            tip_event.changed_axes_bits = changed_axes;
            post_device_event(now, LIBINPUT_EVENT_TABLET_TOOL_TIP, tip_event);
        }
        void tablet_notify_axis(time now, libinput_tablet_tool_sptr tool, libinput_tablet_tool_tip_state tip_state, tablet_axes_bitset& changed_axes, tablet_axes const& axes, abs_info_t const& x, abs_info_t const& y)
        {
            auto& axis_event = li.libinput_emplace_event<libinput_event_tablet_tool>();
            axis_event.axes              = axes;
            axis_event.tool              = tool;
            axis_event.proximity_state   = LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN;
            axis_event.tip_state         = tip_state;
            axis_event.abs_info_x2       = x;
            axis_event.abs_info_y2       = y;
            axis_event.changed_axes_bits = changed_axes;
            post_device_event(now, LIBINPUT_EVENT_TABLET_TOOL_AXIS, axis_event);
        }
        void tablet_notify_button(time now, libinput_tablet_tool_sptr tool, libinput_tablet_tool_tip_state tip_state, tablet_axes const& axes, ui32 button, si32 state, abs_info_t const& x, abs_info_t const& y)
        {
            auto& button_event = li.libinput_emplace_event<libinput_event_tablet_tool>();
            button_event.button            = button;
            button_event.state             = state;
            button_event.seat_button_count = li.update_press_count(button, state);
            button_event.axes              = axes;
            button_event.tool              = tool;
            button_event.proximity_state   = LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN;
            button_event.tip_state         = tip_state;
            button_event.abs_info_x2       = x;
            button_event.abs_info_y2       = y;
            post_device_event(now, LIBINPUT_EVENT_TABLET_TOOL_BUTTON, button_event);
        }
        si32 libinput_device_config_tap_get_finger_count()
        {
            return config.tap ? config.tap->count(This()) : 0;
        }
        libinput_config_status libinput_device_config_tap_set_enabled(bool tap_state_enabled)
        {
            auto rc = LIBINPUT_CONFIG_STATUS_INVALID;
            auto fn = libinput_device_config_tap_get_finger_count();
            rc = fn ? config.tap->set_enabled(This(), tap_state_enabled)
                    : tap_state_enabled ? LIBINPUT_CONFIG_STATUS_UNSUPPORTED
                                        : LIBINPUT_CONFIG_STATUS_SUCCESS;
            return rc;
        }
            ui32 libinput_device_config_scroll_get_methods()
            {
                return config.scroll_method ? config.scroll_method->get_methods(This()) : 0u;
            }
        libinput_config_status libinput_device_config_scroll_set_method(libinput_config_scroll_method method)
        {
            auto rc = LIBINPUT_CONFIG_STATUS_INVALID;
            if (method == LIBINPUT_CONFIG_SCROLL_NO_SCROLL // Check method is a single valid method.
             || method == LIBINPUT_CONFIG_SCROLL_2FG
             || method == LIBINPUT_CONFIG_SCROLL_EDGE
             || method == LIBINPUT_CONFIG_SCROLL_ON_BUTTON_DOWN)
            {
                if ((libinput_device_config_scroll_get_methods() & method) != method) rc = LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
                else if (config.scroll_method)                                        rc = config.scroll_method->set_method(This(), method);
                else /* method must be _NO_SCROLL to get here */                      rc = LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            return rc;
        }
        bool libinput_device_config_accel_is_available()
        {
            return config.accel && config.accel->available(This());
        }
        libinput_config_status libinput_device_config_accel_set_speed(fp64 speed)
        {
            if (speed >= -1.0 && speed <= 1.0)
            {
                if (!libinput_device_config_accel_is_available()) return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
                else                                              return config.accel->set_speed(This(), speed);
            }
            else return LIBINPUT_CONFIG_STATUS_INVALID;
        }
        fp64 libinput_device_config_accel_get_speed()
        {
            if (!libinput_device_config_accel_is_available()) return 0;
            else                                              return config.accel->get_speed(This());
        }
        ui32 libinput_device_config_accel_get_profiles()
        {
            if (!libinput_device_config_accel_is_available()) return 0;
            else                                              return config.accel->get_profiles(This());
        }
        libinput_config_status libinput_device_config_accel_set_profile(libinput_config_accel_profile profile)
        {
            auto ok = profile == LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT
                   || profile == LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE
                   || profile == LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM;
                 if (!ok)                                                   return LIBINPUT_CONFIG_STATUS_INVALID;
            else if (libinput_device_config_accel_get_profiles() & profile) return config.accel->set_profile(This(), profile);
            else                                                            return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
        }
        void evdev_read_calibration_prop()
        {
            auto prop = ud_device.libinput_udev_prop("LIBINPUT_CALIBRATION_MATRIX");
            if (prop && ud_device.abs.absinfo_x && ud_device.abs.absinfo_y) // Parses a set of 6 space-separated floats.
            {
                auto calibration = std::array<fp32, 6>{};
                auto strv = utf::split<true>(prop, " ");
                if (strv.size() >= calibration.size())
                {
                    auto head = strv.begin();
                    for (auto& c : calibration)
                    {
                        auto& s = *head++;
                        if (auto v = utf::to_int<fp64>(s))
                        {
                            c = v.value();
                        }
                        else
                        {
                            log("Calibration matrix is broken");
                            return;
                        }
                    }
                    ud_device.abs.default_calibration.matrix_from_farray6(calibration);
                    evdev_device_calibrate(calibration);
                    log("Apply calibration: %f% %f% %f% %f% %f% %f%",
                            calibration[0],
                            calibration[1],
                            calibration[2],
                            calibration[3],
                            calibration[4],
                            calibration[5]);
                }
            }
        }
        void evdev_device_init_abs_range_warnings()
        {
            auto& x = ud_device.abs.absinfo_x;
            auto& y = ud_device.abs.absinfo_y;
            auto& w = ud_device.abs.dimensions.x;
            auto& h = ud_device.abs.dimensions.y;
            ud_device.abs.warning_range_x.min = x.minimum - 0.05 * w;
            ud_device.abs.warning_range_y.min = y.minimum - 0.05 * h;
            ud_device.abs.warning_range_x.max = x.maximum + 0.05 * w;
            ud_device.abs.warning_range_y.max = y.maximum + 0.05 * h;
        }
                void evdev_init_accel(libinput_config_accel_profile which)
                {
                    auto filter = motion_filter_sptr{};
                    if (which == LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM)
                    {
                        filter = ptr::shared<custom_accelerator>();
                    }
                    else if (device_tags & EVDEV_TAG_TRACKPOINT)
                    {
                        if (which == LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT)
                        {
                            filter = ptr::shared<trackpoint_flat_accelerator>(trackpoint_multiplier);
                        }
                        else
                        {
                            filter = ptr::shared<trackpoint_accelerator>(trackpoint_multiplier, use_velocity_averaging);
                        }
                    }
                    else
                    {
                        if (which == LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT)
                        {
                            filter = ptr::shared<pointer_accelerator_flat>(dpi);
                        }
                        else if (dpi < lixx::default_mouse_dpi)
                        {
                            filter = ptr::shared<pointer_accelerator_low_dpi>(dpi, use_velocity_averaging);
                        }
                        else
                        {
                            filter = ptr::shared<pointer_accelerator>(dpi, use_velocity_averaging);
                        }
                    }
                    evdev_device_init_pointer_acceleration(filter);
                }
            static libinput_config_status evdev_accel_config_set_speed(libinput_device_sptr li_device, fp64 speed)
            {
                auto ok = li_device->pointer_filter->filter_set_speed(speed);
                return ok ? LIBINPUT_CONFIG_STATUS_SUCCESS : LIBINPUT_CONFIG_STATUS_INVALID;
            }
            static libinput_config_status evdev_accel_config_set_profile(libinput_device_sptr li_device, libinput_config_accel_profile profile)
            {
                auto& filter = li_device->pointer_filter;
                if (filter->filter_get_type() != profile)
                {
                    auto speed = filter->filter_get_speed();
                    li_device->evdev_init_accel(profile);
                    evdev_accel_config_set_speed(li_device, speed);
                }
                return LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            static libinput_config_accel_profile evdev_accel_config_get_profile(libinput_device_sptr li_device)
            {
                return li_device->pointer_filter->filter_get_type();
            }
            static libinput_config_accel_profile evdev_accel_config_get_default_profile(libinput_device_sptr li_device)
            {
                if (!li_device->pointer_filter)
                {
                    return LIBINPUT_CONFIG_ACCEL_PROFILE_NONE;
                }
                else // No device has a flat profile as default.
                {
                    return LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE;
                }
            }
            static libinput_config_status evdev_set_accel_config(libinput_device_sptr li_device, libinput_config_accel& accel_config)
            {
                assert(evdev_accel_config_get_profile(li_device) == accel_config.profile);
                auto ok = li_device->pointer_filter->filter_set_accel_config(accel_config);
                return ok ? LIBINPUT_CONFIG_STATUS_SUCCESS : LIBINPUT_CONFIG_STATUS_INVALID;
            }
            static si32 evdev_accel_config_available([[maybe_unused]] libinput_device_sptr li_device) // This function is only called if we set up ptraccel, so we can reply with a resounding "Yes".
            {
                return 1;
            }
            static fp64 evdev_accel_config_get_speed(libinput_device_sptr li_device)
            {
                return li_device->pointer_filter->filter_get_speed();
            }
            static fp64 evdev_accel_config_get_default_speed([[maybe_unused]] libinput_device_sptr li_device)
            {
                return 0.0;
            }
            static ui32 evdev_accel_config_get_profiles(libinput_device_sptr li_device)
            {
                auto filter_ptr = li_device->pointer_filter;
                return filter_ptr ? LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE | LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT | LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM
                                  : LIBINPUT_CONFIG_ACCEL_PROFILE_NONE;
            }
        void evdev_device_init_pointer_acceleration(motion_filter_sptr filter)
        {
            pointer_filter = filter;
            if (config.accel == nullptr)
            {
                pointer_config.available           = evdev_accel_config_available;
                pointer_config.set_speed           = evdev_accel_config_set_speed;
                pointer_config.get_speed           = evdev_accel_config_get_speed;
                pointer_config.get_default_speed   = evdev_accel_config_get_default_speed;
                pointer_config.get_profiles        = evdev_accel_config_get_profiles;
                pointer_config.set_profile         = evdev_accel_config_set_profile;
                pointer_config.get_profile         = evdev_accel_config_get_profile;
                pointer_config.get_default_profile = evdev_accel_config_get_default_profile;
                pointer_config.set_accel_config    = evdev_set_accel_config;
                config.accel                       = &pointer_config;
                auto default_speed = evdev_accel_config_get_default_speed(This());
                evdev_accel_config_set_speed(This(), default_speed);
            }
        }
        void evdev_notify_added_device()
        {
            auto self = This();
            for (auto d : li.device_list)
            {
                if (d != self)
                {
                    d->device_added(self); // Notify existing device d about addition of device.
                    device_added(d); // Notify new device about existing device d.
                    if (d->is_suspended) // Notify new device if existing device d is suspended.
                    {
                        device_suspended(d);
                    }
                }
            }
            notify_added_device();
            post_added();
        }
        si32 libinput_device_config_rotation_is_available()
        {
            return config.rotation ? config.rotation->is_available(This()) : 0;
        }
        libinput_config_status libinput_device_config_rotation_set_angle(ui32 degrees_cw)
        {
            if (!libinput_device_config_rotation_is_available())
            {
                return degrees_cw ? LIBINPUT_CONFIG_STATUS_UNSUPPORTED : LIBINPUT_CONFIG_STATUS_SUCCESS;
            }
            else if (degrees_cw >= 360)
            {
                return LIBINPUT_CONFIG_STATUS_INVALID;
            }
            return config.rotation->set_angle(This(), degrees_cw);
        }
        ui32 libinput_device_config_rotation_get_angle()
        {
            if (!libinput_device_config_rotation_is_available()) return 0;
            return config.rotation->get_angle(This());
        }
        void evdev_tag_external_mouse()
        {
            auto bustype = libevdev_get_id_bustype();
            if (bustype == BUS_USB || bustype == BUS_BLUETOOTH)
            {
                device_tags |= EVDEV_TAG_EXTERNAL_MOUSE;
            }
        }
        void evdev_tag_trackpoint()
        {
            if (!libevdev_has_property(INPUT_PROP_POINTING_STICK)
             && !ud_device.parse_udev_flag("ID_INPUT_POINTINGSTICK"))
            {
                return;
            }
            device_tags |= EVDEV_TAG_TRACKPOINT;
            if (auto q = li.quirks_fetch_for_device(ud_device))
            {
                auto prop = text{};
                if (q->quirks_get(QUIRK_ATTR_TRACKPOINT_INTEGRATION, prop))
                {
                    if ("internal"sv == prop)
                    {
                        // Noop, this is the default anyway.
                    }
                    else if ("external"sv == prop)
                    {
                        device_tags |= EVDEV_TAG_EXTERNAL_MOUSE;
                        log("is an external pointing stick");
                    }
                    else
                    {
                        log("tagged with unknown value %s%", prop);
                    }
                }
            }
        }
        fp64 evdev_get_trackpoint_multiplier()
        {
            auto multiplier = 1.0;
            if (device_tags & EVDEV_TAG_TRACKPOINT)
            {
                if (auto q = li.quirks_fetch_for_device(ud_device))
                {
                    q->quirks_get(QUIRK_ATTR_TRACKPOINT_MULTIPLIER, multiplier);
                }
                if (multiplier <= 0.0)
                {
                    log("trackpoint multiplier %.2f% is invalid", multiplier);
                    multiplier = 1.0;
                }
                if (multiplier != 1.0)
                {
                    log("trackpoint multiplier is %.2f%", multiplier);
                }
            }
            return multiplier;
        }
            si32 parse_mouse_dpi_property(qiew prop_str)
            {
                // Helper function to parse the mouse DPI tag from udev.
                // The tag is of the form:
                // MOUSE_DPI=400 *1000 2000
                // or
                // MOUSE_DPI=400@125 *1000@125 2000@125
                // Where the * indicates the default value and @number indicates device poll rate.
                // Numbers should be in ascending order, and if rates are present they should be present for all entries.
                //
                // When parsing the mouse DPI property, if we find an error we just return 0
                // since it's obviously invalid, the caller will treat that as an error and
                // use a reasonable default instead. If the property contains multiple DPI
                // settings but none flagged as default, we return the last because we're
                // lazy and that's a silly way to set the property anyway.
                //
                // @param prop The value of the udev property (without the MOUSE_DPI=).
                // @return The default dpi value on success, 0 on error.
                if (!prop_str) return 0;
                auto is_default = faux;
                auto nread = 0;
                auto dpi = 0;
                auto rate = 0;
                auto prop = prop_str.begin();
                while (*prop != 0)
                {
                    if (*prop == ' ')
                    {
                        prop++;
                        continue;
                    }
                    if (*prop == '*')
                    {
                        prop++;
                        is_default = true;
                        if (!isdigit(prop[0])) return 0;
                    }
                    // While we don't do anything with the rate right now we will validate that, if it's present, it is non-zero and positive.
                    rate = 1;
                    nread = 0;
                    ::sscanf(prop, "%d@%d%n", &dpi, &rate, &nread);
                    if (!nread) ::sscanf(prop, "%d%n", &dpi, &nread);
                    if (!nread || dpi <= 0 || rate <= 0 || prop[nread] == '@') return 0;
                    if (is_default) break;
                    prop += nread;
                }
                return dpi;
            }
        si32 evdev_read_dpi_prop()
        {
            auto dpi = lixx::default_mouse_dpi;
            if (device_tags & EVDEV_TAG_TRACKPOINT) return lixx::default_mouse_dpi;
            auto mouse_dpi = ud_device.libinput_udev_prop("MOUSE_DPI");
            if (mouse_dpi)
            {
                dpi = parse_mouse_dpi_property(mouse_dpi);
                if (!dpi)
                {
                    log("mouse DPI property is present but invalid, using %d% DPI instead", lixx::default_mouse_dpi);
                    dpi = lixx::default_mouse_dpi;
                }
                log("device set to %d% DPI", dpi);
            }
            return dpi;
        }
        void evdev_tag_keyboard()
        {
            if (ud_device.libevdev_has_event_type<EV_KEY>())
            {
                for (auto code = KEY_Q; code <= KEY_P; code++)
                {
                    if (!libevdev_has_event_code<EV_KEY>(code))
                    {
                        return;
                    }
                }
                if (auto q = li.quirks_fetch_for_device(ud_device))
                {
                    auto prop = text{};
                    if (q->quirks_get(QUIRK_ATTR_KEYBOARD_INTEGRATION, prop))
                    {
                        if ("internal"sv == prop)
                        {
                            device_tags |= EVDEV_TAG_INTERNAL_KEYBOARD;
                            device_tags &= ~EVDEV_TAG_EXTERNAL_KEYBOARD;
                        }
                        else if ("external"sv == prop)
                        {
                            device_tags |= EVDEV_TAG_EXTERNAL_KEYBOARD;
                            device_tags &= ~EVDEV_TAG_INTERNAL_KEYBOARD;
                        }
                        else
                        {
                            log("tagged with unknown value %s%", prop);
                        }
                    }
                }
                device_tags |= EVDEV_TAG_KEYBOARD;
            }
        }
        bool evdev_need_velocity_averaging()
        {
            auto use_velocity_averaging = faux; // Default off unless we have quirk.
            if (auto q = li.quirks_fetch_for_device(ud_device))
            {
                q->quirks_get(QUIRK_ATTR_USE_VELOCITY_AVERAGING, use_velocity_averaging);
                if (use_velocity_averaging)
                {
                    log("velocity averaging is turned on");
                }
            }
            return use_velocity_averaging;
        }
        void evdev_tag_touchpad()
        {
            auto evdev_tag_touchpad_internal = [&]
            {
                device_tags |= EVDEV_TAG_INTERNAL_TOUCHPAD;
                device_tags &= ~EVDEV_TAG_EXTERNAL_TOUCHPAD;
            };
            auto evdev_tag_touchpad_external = [&]
            {
                device_tags |= EVDEV_TAG_EXTERNAL_TOUCHPAD;
                device_tags &= ~EVDEV_TAG_INTERNAL_TOUCHPAD;
            };
            if (auto prop = ud_device.libinput_udev_prop("ID_INPUT_TOUCHPAD_INTEGRATION"))
            {
                     if (prop == "internal") { evdev_tag_touchpad_internal(); return; }
                else if (prop == "external") { evdev_tag_touchpad_external(); return; }
                else                         log("Device is tagged with unknown value %s%", prop);
            }
            // The hwdb is the authority on integration, these heuristics are the fallback only (they precede the hwdb too).
            // Simple approach:
            //   Bluetooth touchpads are considered external, anything else is internal. Except the ones from some vendors that only make external touchpads.
            auto bustype = libevdev_get_id_bustype();
            auto vendor  = libevdev_get_id_vendor();
            if (bustype == BUS_BLUETOOTH) evdev_tag_touchpad_external();
            else                          evdev_tag_touchpad_internal();
            if (vendor == lixx::vendor_id_logitech // Logitech does not have internal touchpads.
                || ud_device.model_flags & EVDEV_MODEL_WACOM_TOUCHPAD) // Wacom makes touchpads, but not internal ones.
            {
                evdev_tag_touchpad_external();
            }
            if (!(device_tags & (EVDEV_TAG_EXTERNAL_TOUCHPAD | EVDEV_TAG_INTERNAL_TOUCHPAD)))
            {
                log("Internal or external? Please file a bug");
                evdev_tag_touchpad_external();
            }
        }
        si32 evdev_device_tablet_pad_get_num_dials()
        {
            auto ndials = 0;
            if (!(device_caps & EVDEV_DEVICE_TABLET_PAD)) return -1;
            if (libevdev_has_event_code<EV_REL>(REL_WHEEL)
             || libevdev_has_event_code<EV_REL>(REL_DIAL))
            {
                ndials++;
                if (libevdev_has_event_code<EV_REL>(REL_HWHEEL))
                {
                    ndials++;
                }
            }
            return ndials;
        }
        si32 evdev_device_tablet_pad_get_num_rings()
        {
            auto nrings = 0;
            if (!(device_caps & EVDEV_DEVICE_TABLET_PAD)) return -1;
            if (libevdev_has_event_code<EV_ABS>(ABS_WHEEL))
            {
                nrings++;
                if (libevdev_has_event_code<EV_ABS>(ABS_THROTTLE))
                {
                    nrings++;
                }
            }
            return nrings;
        }
        si32 evdev_device_tablet_pad_get_num_strips()
        {
            auto nstrips = 0;
            if (!(device_caps & EVDEV_DEVICE_TABLET_PAD)) return -1;
            if (libevdev_has_event_code<EV_ABS>(ABS_RX))
            {
                nstrips++;
                if (libevdev_has_event_code<EV_ABS>(ABS_RY))
                {
                    nstrips++;
                }
            }
            return nstrips;
        }
    };

    struct libinput_paired_keyboard
    {
        libinput_device_sptr         li_device;
        libinput_event_listener_sptr listener;
    };
            struct tp_history_t
            {
                struct tp_history_point
                {
                    time      stamp{};
                    si32_coor point;
                };

                tp_history_point samples[lixx::touchpad_history_length] = {};
                ui32             index{};
                ui32             count{};
            };
            struct tp_button_t
            {
                button_state_enum   state{};
                button_event        current{}; // We use button_event here so we can use == on events.
                libinput_timer_sptr timer;
                si32_coor           initial;
                bool                has_moved{}; // Has moved more than threshold.
                time                initial_time{};
            };
            struct tp_tap_t
            {
                tp_tap_touch_state state{};
                si32_coor          initial;
                bool               is_thumb{};
                bool               is_palm{};
            };
            struct tp_scroll_t
            {
                tp_edge_scroll_touch_state edge_state{};
                ui32                       edge{};
                si32                       direction{};
                libinput_timer_sptr        timer;
                si32_coor                  initial;
            };
        struct tp_touch
        {
            tp_dispatch_sptr tp;
            ui32             index{};
            si32             pressure{};
            touch_state      state;
            bool             has_ended{}; // TRACKING_ID == -1.
            bool             dirty{};
            bool             is_tool_palm{}; // MT_TOOL_PALM.
            bool             was_down{}; // If distance == 0, false for pure hovering touches.
            time             initial_time{};
            si32_coor        gesture_origin;
            si32_coor        point;
            si32_range       touch_limits;
            fp64             jumps_last_delta_mm{};
            si32_coor        hysteresis_center;
            byte             hysteresis_x_motion_history{};
            bool             pinned_state{};  // A pinned touchpoint is the one that pressed the physical button on a clickpad. After the release, it won't move until the center moves more than a threshold away from the original coordinates.
            si32_coor        pinned_center; //
            bool             quirks_reset_motion_history{}; // A quirk mostly used on Synaptics touchpads. In a transition to/from fake touches > num_slots, the current event data is likely garbage and the subsequent event is likely too. This marker tells us to reset the motion history again -> this effectively swallows any motion.
            tp_history_t     history;
            tp_button_t      button; // Software-button state and timeout if applicable.
            tp_tap_t         tap;
            tp_scroll_t      scroll;
            touch_palm_state palm_state{}; // Palm state.
            si32_coor        palm_first; // Palm detected there.
            time             palm_stamp{}; // Palm detection time.
            fp64             speed_last{}; // Speed in mm/s at last sample.
            ui32             speed_exceeded_count{};
        };
        struct tp_dispatch_arbitration_t
        {
            libinput_arbitration_state state{};
            libinput_timer_sptr        arbitration_timer;
        };
        struct tp_dispatch_jump_t
        {
            bool detection_disabled{};
        };
        struct tp_dispatch_pressure_t // If pressure goes above high -> touch down, if pressure then goes below low -> touch up.
        {
            bool use_pressure{};
            si32 high{};
            si32 low{};
        };
        struct tp_dispatch_touch_size_t // If touch size (either axis) goes above high -> touch down, if touch size (either axis) goes below low -> touch up.
        {
            bool use_touch_size{};
            si32 high{};
            si32 low{};
            fp64 orientation_to_angle{}; // Convert device units to angle.
        };
        struct tp_dispatch_hysteresis_t
        {
            bool      enabled{};
            si32_coor margin;
            ui32      other_event_count{};
            time      last_motion_time{};
        };
        struct tp_dispatch_gesture_t
        {
            libinput_device_config_gesture config;
            bool                           enabled{};
            ui32                           finger_count{};
            ui32                           finger_count_pending{};
            libinput_timer_sptr            finger_count_switch_timer;
            tp_gesture_state               state;
            tp_touch*                      two_touches[2] = {};
            time                           initial_time{};
            fp64                           initial_distance{};
            fp64                           prev_scale{};
            fp64                           angle{};
            fp64_coor                      center;
            libinput_timer_sptr            hold_timer;
            bool                           hold_enabled{};
            libinput_timer_sptr            drag_3fg_timer;
            time                           drag_3fg_release_time{};
        };
            struct tp_dispatch_bottom_area_t // Only used for clickpads. The software button areas are always 2 horizontal stripes across the touchpad. The buttons are split according to the edge settings.
            {
                si32 top_edge{};               // In device coordinates.
                si32 rightbutton_left_edge{};  // In device coordinates.
                si32 middlebutton_left_edge{}; // In device coordinates.
            };
            struct tp_dispatch_top_area_t
            {
                si32 bottom_edge{};           // In device coordinates.
                si32 rightbutton_left_edge{}; // In device coordinates.
                si32 leftbutton_right_edge{}; // In device coordinates.
            };
        struct tp_dispatch_buttons_t
        {
            bool                                is_clickpad{}; // True for clickpads.
            bool                                has_topbuttons{};
            bool                                use_clickfinger{};  // Number of fingers decides button number.
            bool                                click_pending{};
            ui32                                state{};
            ui32                                old_state{};
            fp64_coor                           motion_dist_scale_coeff; // For pinned touches.
            ui32                                active{};  // evdev_usage_t  Currently active button, for release event.
            bool                                active_is_topbutton{}; // Is active a top button?
            tp_dispatch_bottom_area_t           bottom_area;
            tp_dispatch_top_area_t              top_area;
            libinput_device_sptr                trackpoint_li_device;
            libinput_config_click_method        click_method{};
            libinput_device_config_click_method config_method{};
            bool                                use_lmr_map{};
            bool                                want_use_lmr_map{};
        };
            struct tp_dispatch_active_t
            {
                bool h{};
                bool v{};
            };
            struct tp_dispatch_duration_t
            {
                span h{};
                span v{};
            };
        struct tp_dispatch_scroll_t
        {
            libinput_device_config_scroll_method config_method;
            libinput_config_scroll_method        method{};
            si32                                 right_edge{};  // In device coordinates.
            si32                                 bottom_edge{}; // In device coordinates.
            tp_dispatch_active_t                 active;
            fp64_coor                            vector;
            time                                 stamp{};
            tp_dispatch_duration_t               duration;
        };
        struct tp_dispatch_tap_t
        {
            libinput_device_config_tap      config;
            bool                            tap_state_enabled{};
            bool                            suspended{};
            libinput_timer_sptr             timer;
            tp_tap_state                    state{};
            ui32                            buttons_pressed{};
            time                            press_stamp{};
            time                            release_stamp{};
            bool                            use_lmr_map{};
            bool                            want_use_lmr_map{};
            bool                            drag_enabled{};
            libinput_config_drag_lock_state drag_lock{};
            ui32                            nfingers_down{}; // Number of fingers down for tapping (excl. thumb/palm).
        };
        struct tp_dispatch_drag_3fg_t
        {
            libinput_device_config_3fg_drag config;
            ui64                            nfingers{};
            ui64                            want_nfingers{};
        };
        struct tp_dispatch_palm_t
        {
            libinput_device_config_dwtp  config;
            bool                         dwtp_enabled{};
            si32                         right_edge{}; // In device coordinates.
            si32                         left_edge{};  // In device coordinates.
            si32                         upper_edge{}; // In device coordinates.
            bool                         trackpoint_active{};
            libinput_event_listener_sptr trackpoint_listener;
            libinput_timer_sptr          trackpoint_timer;
            time                         trackpoint_last_event_time{};
            ui32                         trackpoint_event_count{};
            bool                         monitor_trackpoint{};
            bool                         use_mt_tool{};
            bool                         use_pressure{};
            si32                         pressure_threshold{};
            bool                         use_size{};
            si32                         size_threshold{};
        };
        struct tp_dispatch_dwt_t // We have to allow for more than one device node to be the internal dwt keyboard (Razer Blade). But they're the same physical device, so we don't care about per-keyboard key/modifier masks.
        {
            libinput_device_config_dwt               config;
            bool                                     dwt_enabled{};
            std::list<libinput_paired_keyboard_sptr> paired_keyboard_list;
            button_state_t                           key_mask;
            button_state_t                           mod_mask;
            bool                                     keyboard_active{};
            libinput_timer_sptr                      keyboard_timer;
            time                                     keyboard_last_press_time{};
        };
        struct tp_dispatch_thumb_t
        {
            bool           detect_thumbs{};
            si32           upper_thumb_line{};
            si32           lower_thumb_line{};
            bool           use_pressure{};
            si32           pressure_threshold{};
            bool           use_size{};
            si32           size_threshold{};
            tp_thumb_state state{};
            ui32           index{};
            bool           pinch_eligible{};
        };
            struct msc_timestamp_t
            {
                tp_jump_state state{};
                span          interval{};
                span          now{};
            };
        struct tp_dispatch_quirks_t // A quirk used on the T450 series Synaptics hardware. Slowly moving the finger causes multiple events with only ABS_MT_PRESSURE but no x/y information. When the x/y event comes, it will be a jump of ~20 units. We use the below to count non-motion events to discard that first event with the jump.
        {
            ui32            nonmotion_event_count{};
            msc_timestamp_t msc_timestamp;
        };
        struct tp_dispatch_lid_switch_t
        {
            libinput_device_sptr         lid_switch_li_device;
            libinput_event_listener_sptr listener;
        };
        struct tp_dispatch_tablet_mode_switch_t
        {
            libinput_device_sptr         tablet_mode_switch_li_device;
            libinput_event_listener_sptr listener;
        };
        struct tp_dispatch_left_handed_t
        {
            bool                 rotate{};
            bool                 want_rotate{};
            bool                 must_rotate{}; // True if we should rotate when applicable.
            libinput_device_sptr tablet_li_device;
            bool                 tablet_left_handed_state{};
        };

    struct tp_device : libinput_device_t
    {
        ui32                             nfingers_down{};     // Number of fingers down.
        ui32                             old_nfingers_down{}; // Previous no fingers down.
        ui32                             slot{};              // Current slot.
        bool                             has_mt{};            //
        bool                             semi_mt{};           //
        ui32                             suspend_reason{};
        tp_dispatch_arbitration_t        arbitration;
        ui32                             nactive_slots{}; // Number of active slots.
        ui32                             num_slots{};     // Number of slots.
        ui32                             ntouches{};      // No slots inc. fakes.
        std::vector<tp_touch>            touches;         // len == ntouches.
        ui32                             fake_touches{};  // Bit 0: BTN_TOUCH, bit 1: BTN_TOOL_FINGER,  bit 2: BTN_TOOL_DOUBLETAP.
        tp_dispatch_jump_t               jump;
        tp_dispatch_pressure_t           pressure;
        tp_dispatch_touch_size_t         touch_size;
        tp_dispatch_hysteresis_t         hysteresis;
        fp64_coor                        accel_scale_coeff;
        fp64                             accel_xy_scale_coeff{};
        tp_dispatch_gesture_t            gesture;
        tp_dispatch_buttons_t            buttons;
        tp_dispatch_scroll_t             tp_scroll;
        touchpad_event                   queued{};
        tp_dispatch_tap_t                tap;
        tp_dispatch_drag_3fg_t           drag_3fg;
        tp_dispatch_palm_t               palm;
        tp_dispatch_dwt_t                dwt;
        tp_dispatch_thumb_t              thumb;
        tp_dispatch_quirks_t             quirks;
        tp_dispatch_lid_switch_t         lid_switch;
        tp_dispatch_tablet_mode_switch_t tablet_mode_switch;
        tp_dispatch_left_handed_t        left_handed;

        tp_device(auto&... args)
            : libinput_device_t{ args... }
        { }
        ~tp_device()
        {
            arbitration.arbitration_timer.reset();
            palm.trackpoint_timer.reset();
            dwt.keyboard_timer.reset();
            tap.timer.reset();
            gesture.finger_count_switch_timer.reset();
            gesture.hold_timer.reset();
            gesture.drag_3fg_timer.reset();
        }

        struct tp_impl_t
        {
            tp_device& tp;
                            tp_touch& tp_current_touch()
                            {
                                return tp.touches[std::min(tp.slot, tp.ntouches - 1)];
                            }
                            si32 rotated(ui32 usage, si32 value)
                            {
                                if (tp.left_handed.rotate)
                                {
                                    auto x = usage == evdev::abs_x || usage == evdev::abs_mt_position_x;
                                    auto& absinfo = x ? tp.ud_device.abs.absinfo_x : tp.ud_device.abs.absinfo_y;
                                    value = absinfo.maximum - (value - absinfo.minimum);
                                }
                                return value;
                            }
                                void tp_motion_history_reset(tp_touch& t)
                                {
                                    t.history.count = 0;
                                }
                            void tp_new_touch(tp_touch& t)
                            {
                                if (t.state == TOUCH_BEGIN || t.state == TOUCH_UPDATE || t.state == TOUCH_HOVERING)
                                {
                                    return;
                                }
                                // Bug #161: touch ends in the same event frame where it restarts again. That's a kernel bug, so let's complain.
                                if (t.state == TOUCH_MAYBE_END)
                                {
                                    log("touch %d% ended and began in in same frame", t.index);
                                    tp.nfingers_down++;
                                    t.state = TOUCH_UPDATE;
                                    t.has_ended = faux;
                                    return;
                                }
                                // We begin the touch as hovering because until BTN_TOUCH happens we don't know if it's a touch down or not. And BTN_TOUCH may happen after ABS_MT_TRACKING_ID.
                                tp_motion_history_reset(t);
                                t.dirty                       = true;
                                t.has_ended                   = faux;
                                t.was_down                    = faux;
                                t.palm_state                  = TOUCH_PALM_NONE;
                                t.state                       = TOUCH_HOVERING;
                                t.pinned_state                = faux;
                                t.speed_last                  = 0;
                                t.speed_exceeded_count        = 0;
                                t.hysteresis_x_motion_history = 0;
                                tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                            }
                                void tp_maybe_end_touch(tp_touch& t)
                                {
                                    // Schedule a touch to be ended, based on either the events or some attributes of the touch (size, pressure). In some cases we need to resurrect a touch that has ended, so this doesn't actually end the touch yet.
                                    // All the TOUCH_MAYBE_END touches get properly ended once the device state has been processed once and we know how many zombie touches we need.
                                    switch (t.state)
                                    {
                                        case TOUCH_NONE:
                                        case TOUCH_MAYBE_END: return;
                                        case TOUCH_END: log("touch %d%: already in TOUCH_END", t.index); return;
                                        case TOUCH_HOVERING:
                                        case TOUCH_BEGIN:
                                        case TOUCH_UPDATE: break;
                                    }
                                    if (t.state != TOUCH_HOVERING)
                                    {
                                        assert(tp.nfingers_down >= 1);
                                        tp.nfingers_down--;
                                        t.state = TOUCH_MAYBE_END;
                                    }
                                    else
                                    {
                                        t.state = TOUCH_NONE;
                                    }
                                    t.dirty = true;
                                }
                            void tp_end_sequence(tp_touch& t)
                            {
                                t.has_ended = true;
                                tp_maybe_end_touch(t);
                            }
                        void tp_process_absolute(evdev_event const& ev)
                        {
                            auto& t = tp_current_touch();
                            switch (ev.usage)
                            {
                                case evdev::abs_mt_position_x:
                                    tp.evdev_device_check_abs_axis_range(ev.usage, ev.value);
                                    t.point.x = rotated(ev.usage, ev.value);
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                                    break;
                                case evdev::abs_mt_position_y:
                                    tp.evdev_device_check_abs_axis_range(ev.usage, ev.value);
                                    t.point.y = rotated(ev.usage, ev.value);
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                                    break;
                                case evdev::abs_mt_slot:
                                    tp.slot = ev.value;
                                    break;
                                case evdev::abs_mt_tracking_id:
                                    if (ev.value != -1)
                                    {
                                        tp.nactive_slots += 1;
                                        tp_new_touch(t);
                                    }
                                    else if (tp.nactive_slots >= 1)
                                    {
                                        tp.nactive_slots -= 1;
                                        tp_end_sequence(t);
                                    }
                                    break;
                                case evdev::abs_mt_pressure:
                                    t.pressure = ev.value;
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_OTHERAXIS);
                                    break;
                                case evdev::abs_mt_tool_type:
                                    t.is_tool_palm = ev.value == MT_TOOL_PALM;
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_OTHERAXIS);
                                    break;
                                case evdev::abs_mt_touch_major:
                                    t.touch_limits.max = ev.value;
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_OTHERAXIS);
                                    break;
                                case evdev::abs_mt_touch_minor:
                                    t.touch_limits.min = ev.value;
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_OTHERAXIS);
                                    break;
                                default:
                                    break;
                            }
                        }
                        void tp_process_absolute_st(evdev_event const& ev)
                        {
                            auto& t = tp_current_touch();
                            switch (ev.usage)
                            {
                                case evdev::abs_x:
                                    tp.evdev_device_check_abs_axis_range(ev.usage, ev.value);
                                    t.point.x = rotated(ev.usage, ev.value);
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                                    break;
                                case evdev::abs_y:
                                    tp.evdev_device_check_abs_axis_range(ev.usage, ev.value);
                                    t.point.y = rotated(ev.usage, ev.value);
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                                    break;
                                case evdev::abs_pressure:
                                    t.pressure = ev.value;
                                    t.dirty = true;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_OTHERAXIS);
                                    break;
                                default:
                                    break;
                            }
                        }
                            void tp_process_button(evdev_event const& ev)
                            {
                                auto mask = 1ul << (ev.usage - evdev::btn_left);
                                if (tp.buttons.is_clickpad && ev.usage != evdev::btn_left) // Ignore other buttons on clickpads.
                                {
                                    log("received 'type=%% code=%%' button event on a clickpad", evdev_usage_type(ev.usage), evdev_usage_code(ev.usage));
                                }
                                else
                                {
                                    if (ev.value)
                                    {
                                        tp.buttons.state |= mask;
                                        tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_BUTTON_PRESS);
                                    }
                                    else
                                    {
                                        tp.buttons.state &= ~mask;
                                        tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_BUTTON_RELEASE);
                                    }
                                }
                            }
                            void tp_fake_finger_set(ui32 usage, bool is_press)
                            {
                                auto shift = 0;
                                switch (usage)
                                {
                                    case evdev::btn_touch:
                                        if (!is_press) tp.fake_touches &= ~lixx::fake_finger_overflow;
                                        shift = 0;
                                        break;
                                    case evdev::btn_tool_finger:
                                        shift = 1;
                                        break;
                                    case evdev::btn_tool_doubletap:
                                    case evdev::btn_tool_tripletap:
                                    case evdev::btn_tool_quadtap:
                                        shift = usage - evdev::btn_tool_doubletap + 2;
                                        break;
                                    // When QUINTTAP is released we're either switching to 6 fingers (flag stays in place until BTN_TOUCH is released) or one of DOUBLE/TRIPLE/QUADTAP (will clear the flag on press).
                                    case evdev::btn_tool_quinttap:
                                        if (is_press) tp.fake_touches |= lixx::fake_finger_overflow;
                                        return;
                                    default:
                                        return;
                                }
                                if (is_press)
                                {
                                    tp.fake_touches &= ~lixx::fake_finger_overflow;
                                    tp.fake_touches |= (1ul << shift);
                                }
                                else
                                {
                                    tp.fake_touches &= ~(1ul << shift);
                                }
                            }
                            void tp_process_trackpoint_button(evdev_event const& ev, time stamp)
                            {
                                auto button = ui32{};
                                if (!tp.buttons.trackpoint_li_device) return;
                                switch (ev.usage)
                                {
                                    case evdev::btn_0: button = evdev::btn_left; break;
                                    case evdev::btn_1: button = evdev::btn_right; break;
                                    case evdev::btn_2: button = evdev::btn_middle; break;
                                    default: return;
                                }
                                auto event = evdev_event
                                {
                                    .usage = button,
                                    .value = ev.value
                                };
                                auto syn_report = evdev_event
                                {
                                    .usage = evdev::syn_report,
                                    .value = 0
                                };
                                tp.buttons.trackpoint_li_device->process(event, stamp);
                                tp.buttons.trackpoint_li_device->process(syn_report, stamp);
                            }
                        void tp_process_key(evdev_event const& ev, time stamp)
                        {
                            if (ev.value == 2) return; // Ignore kernel key repeat.
                            switch (ev.usage)
                            {
                                case evdev::btn_left:
                                case evdev::btn_middle:
                                case evdev::btn_right: tp_process_button(ev); break;
                                case evdev::btn_touch:
                                case evdev::btn_tool_finger:
                                case evdev::btn_tool_doubletap:
                                case evdev::btn_tool_tripletap:
                                case evdev::btn_tool_quadtap:
                                case evdev::btn_tool_quinttap: tp_fake_finger_set(ev.usage, !!ev.value); break;
                                case evdev::btn_0:
                                case evdev::btn_1:
                                case evdev::btn_2: tp_process_trackpoint_button(ev, stamp); break;
                                default: break;
                            }
                        }
                                auto& tp_motion_history_offset(tp_touch& t, si32 offset)
                                {
                                    auto offset_index = (t.history.index - offset + lixx::touchpad_history_length) % lixx::touchpad_history_length;
                                    return t.history.samples[offset_index];
                                }
                            void tp_motion_history_fix_last(tp_touch& t, span jumping_interval, span normal_interval, time stamp)
                            {
                                if (t.state != TOUCH_UPDATE) return;
                                // We know the coordinates are correct because the touchpad should
                                // get that bit right. But the timestamps we got from the kernel are
                                // messed up, so we go back in the history and fix them.
                                //
                                // This way the next delta is huge but it's over a large time, so
                                // the pointer accel code should do the right thing.
                                for (auto i = 0; i < (si32)t.history.count; i++)
                                {
                                    auto& p = tp_motion_history_offset(t, i);
                                    p.stamp = stamp - jumping_interval - normal_interval * i;
                                }
                            }
                        void tp_process_msc(evdev_event const& ev)
                        {
                            if (ev.usage != evdev::msc_timestamp)
                            {
                                tp.quirks.msc_timestamp.now = std::chrono::microseconds{ ev.value };
                                tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_TIMESTAMP);
                            }
                        }
                                    void filter_restart(motion_filter_sptr filter, time stamp)
                                    {
                                        filter->restart(stamp);
                                    }
                                void tp_process_msc_timestamp(time stamp)
                                {
                                    auto& m = tp.quirks.msc_timestamp;
                                    // Pointer jump detection based on MSC_TIMESTAMP.
                                    // MSC_TIMESTAMP gets reset after a kernel timeout (1s) and on some
                                    // devices (Dell XPS) the i2c controller sleeps after a timeout. On
                                    // wakeup, some events are swallowed, triggering a cursor jump. The
                                    // event sequence after a sleep is always:
                                    // initial finger down:
                                    //     ABS_X/Y x/y
                                    //     MSC_TIMESTAMP 0
                                    //     SYN_REPORT +2500ms
                                    // second event:
                                    //     ABS_X/Y x+n/y+n     # normal movement
                                    //     MSC_TIMESTAMP 7300  # the hw interval
                                    //     SYN_REPORT +2ms
                                    // third event:
                                    //     ABS_X/Y x+lots/y+lots  # pointer jump!
                                    //     MSC_TIMESTAMP 123456   # well above the hw interval
                                    //     SYN_REPORT +2ms
                                    // fourth event:
                                    //     ABS_X/Y x+lots+n/y+lots+n  # all normal again
                                    //     MSC_TIMESTAMP 123456 + 7300
                                    //     SYN_REPORT +8ms
                                    // Our approach is to detect the 0 timestamp, check the interval on
                                    // the next event and then calculate the movement for one fictitious
                                    // event instead, swallowing all other movements. So if the time
                                    // delta is equivalent to 10 events and the movement is x, we
                                    // instead pretend there was movement of x/10.
                                    if (m.now == span{})
                                    {
                                        m.state = JUMP_STATE_EXPECT_FIRST;
                                        m.interval = {};
                                        return;
                                    }
                                    switch (m.state)
                                    {
                                        case JUMP_STATE_EXPECT_FIRST:
                                            if (m.now > 20ms)
                                            {
                                                m.state = JUMP_STATE_IGNORE;
                                            }
                                            else
                                            {
                                                m.state = JUMP_STATE_EXPECT_DELAY;
                                                m.interval = m.now;
                                            }
                                            break;
                                        case JUMP_STATE_EXPECT_DELAY:
                                            if (m.now > m.interval * 2)
                                            {
                                                auto tdelta = m.now - m.interval; // µs, The current time is > 2 times the interval so we have a jump. Fix the motion history.
                                                for (auto& t : tp.touches)
                                                {
                                                    tp_motion_history_fix_last(t, tdelta, m.interval, stamp);
                                                }
                                                m.state = JUMP_STATE_IGNORE;
                                                // We need to restart the acceleration filter to forget its history. * The current point becomes the first point in the history there * (including timestamp) and that accelerates correctly. * This has a potential to be incorrect but since we only ever see * those jumps over the first three events it doesn't matter.
                                                filter_restart(tp.pointer_filter, stamp - tdelta);
                                            }
                                            break;
                                        case JUMP_STATE_IGNORE:
                                            break;
                                    }
                                }
                                    ui32 tp_fake_finger_count()
                                    {
                                        auto fake_touches = tp.fake_touches & ~(lixx::fake_finger_overflow | 0x1);
                                        if (fake_touches & (fake_touches - 1)) // Only one of BTN_TOOL_DOUBLETAP/TRIPLETAP/... may be set at any time.
                                        {
                                            log("Invalid fake finger state %#x%", tp.fake_touches);
                                        }
                                        if (tp.fake_touches & lixx::fake_finger_overflow)
                                        {
                                            return (ui32)lixx::fake_finger_overflow;
                                        }
                                        return (ui32)::ffs(tp.fake_touches >> 1); // Don't count BTN_TOUCH.
                                    }
                                        void tp_recover_ended_touch(tp_touch& t) // Inverse to tp_maybe_end_touch(), restores a touch back to its previous state.
                                        {
                                            t.dirty = true;
                                            t.state = TOUCH_UPDATE;
                                            tp.nfingers_down++;
                                        }
                                    void tp_restore_synaptics_touches()
                                    {
                                        auto nfake_touches = tp_fake_finger_count();
                                        if (nfake_touches < 3 || tp.nfingers_down >= nfake_touches
                                         || (tp.nfingers_down == tp.num_slots && nfake_touches == tp.num_slots))
                                        {
                                            return;
                                        }
                                        // Synaptics devices may end touch 2 on transition to/from BTN_TOOL_TRIPLETAP and start it again on the next frame with different coordinates (bz#91352, gitlab#434). We search the touches we have, if there is one that has just ended despite us being on tripletap, we move it back to update.
                                        // Note: we only handle the transition from 2 to 3 touches, not the other way round (see gitlab#434).
                                        for (auto i = 0u; i < tp.num_slots; i++)
                                        {
                                            auto& t = tp.touches[i];
                                            if (t.state == TOUCH_MAYBE_END) // New touch, move it through begin to update immediately.
                                            {
                                                tp_recover_ended_touch(t);
                                            }
                                        }
                                    }
                                void tp_process_fake_touches()
                                {
                                    auto nfake_touches = tp_fake_finger_count();
                                    if (nfake_touches == lixx::fake_finger_overflow) return;
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_SYNAPTICS_SERIAL_TOUCHPAD)
                                    {
                                        tp_restore_synaptics_touches();
                                    }
                                    // ALPS serial touchpads always set 3 slots in the kernel, even
                                    // where they support less than that. So we get BTN_TOOL_TRIPLETAP
                                    // but never slot 2 because our slot count is wrong.
                                    // This also means that the third touch falls through the cracks and
                                    // is ignored.
                                    //
                                    // See https://gitlab.freedesktop.org/libinput/libinput/issues/408
                                    //
                                    // All touchpad devices have at least one slot so we only do this
                                    // for 2 touches or higher.
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_ALPS_SERIAL_TOUCHPAD
                                     && nfake_touches > 1 && tp.has_mt
                                     && nfake_touches > tp.nactive_slots
                                     && tp.nactive_slots < tp.num_slots)
                                    {
                                        log("Wrong slot count (%d%), reducing to %d%", tp.num_slots, tp.nactive_slots);
                                        // This should be safe since we fill the slots from the first one so hiding the excessive slots shouldn't matter. There are sequences where we could accidentally lose an actual touch point but that requires specially crafted sequences and let's deal with that when it happens.
                                        tp.num_slots = tp.nactive_slots;
                                    }
                                    auto start = tp.has_mt ? tp.num_slots : 0;
                                    for (auto i = start; i < tp.ntouches; i++)
                                    {
                                        auto& t = tp.touches[i];
                                        if (i < nfake_touches) tp_new_touch(t);
                                        else                   tp_end_sequence(t);
                                    }
                                }
                                        void tp_begin_touch(tp_touch& t, time stamp)
                                        {
                                            t.dirty                = true;
                                            t.state                = TOUCH_BEGIN;
                                            t.initial_time         = stamp;
                                            t.was_down             = true;
                                            t.palm_stamp           = stamp;
                                            t.tap.is_thumb         = faux;
                                            t.tap.is_palm          = faux;
                                            t.speed_exceeded_count = 0;
                                            tp.nfingers_down++;
                                            assert(tp.nfingers_down >= 1);
                                            tp.hysteresis.last_motion_time = stamp;
                                        }
                                    void tp_unhover_pressure(time stamp)
                                    {
                                        auto real_fingers_down = 0u;
                                        auto nfake_touches = tp_fake_finger_count();
                                        if (nfake_touches == lixx::fake_finger_overflow)
                                        {
                                            nfake_touches = 0;
                                        }
                                        for (auto i = 0; i < (si32)tp.num_slots; i++)
                                        {
                                            auto& t = tp.touches[i];
                                            if (t.state == TOUCH_NONE) continue;
                                            if (t.dirty)
                                            {
                                                if (t.state == TOUCH_HOVERING)
                                                {
                                                    if (t.pressure >= tp.pressure.high)
                                                    {
                                                        log("pressure: begin touch %d%", t.index);
                                                        tp_motion_history_reset(t); // Avoid jumps when landing a finger.
                                                        tp_begin_touch(t, stamp);
                                                    }
                                                    // Don't unhover for pressure if we have too many fake fingers down, see comment below. Except for single-finger touches where the real touch decides for the rest.
                                                }
                                                else if (nfake_touches <= tp.num_slots || tp.num_slots == 1)
                                                {
                                                    if (t.pressure < tp.pressure.low)
                                                    {
                                                        log("pressure: end touch %d%", t.index);
                                                        tp_maybe_end_touch(t);
                                                    }
                                                }
                                            }
                                            if (t.state == TOUCH_BEGIN || t.state == TOUCH_UPDATE)
                                            {
                                                real_fingers_down++;
                                            }
                                        }
                                        if (nfake_touches <= tp.num_slots || tp.nfingers_down == 0)
                                        {
                                            return;
                                        }
                                        // If we have more fake fingers down than slots, we assume _all_ fingers have enough pressure, even if some of the slotted ones don't. Anything else gets insane quickly.
                                        if (real_fingers_down > 0)
                                        {
                                            for (auto& t : tp.touches)
                                            {
                                                if (t.state == TOUCH_HOVERING)
                                                {
                                                    tp_motion_history_reset(t); // Avoid jumps when landing a finger.
                                                    tp_begin_touch(t, stamp);
                                                    if (tp.nfingers_down >= nfake_touches)
                                                    {
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                        if (tp.nfingers_down > nfake_touches || real_fingers_down == 0)
                                        {
                                            for (auto i = (si32)tp.ntouches - 1; i >= 0; i--)
                                            {
                                                auto& t = tp.touches[i];
                                                if (t.state != TOUCH_HOVERING && t.state != TOUCH_NONE && t.state != TOUCH_MAYBE_END)
                                                {
                                                    tp_maybe_end_touch(t);
                                                    if (real_fingers_down > 0 && tp.nfingers_down == nfake_touches)
                                                    {
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    void tp_unhover_size(time stamp)
                                    {
                                        auto low = tp.touch_size.low;
                                        auto high = tp.touch_size.high;
                                        // We require 5 slots for size handling, so we don't need to care about fake touches here.
                                        for (auto i = 0u; i < tp.num_slots; i++)
                                        {
                                            auto& t = tp.touches[i];
                                            if (t.state == TOUCH_NONE || !t.dirty) continue;
                                            if (t.state == TOUCH_HOVERING)
                                            {
                                                if ((t.touch_limits.max > high && t.touch_limits.min > low) || (t.touch_limits.max > low && t.touch_limits.min > high))
                                                {
                                                    log("touch-size: begin touch %d%", t.index);
                                                    tp_motion_history_reset(t); // Avoid jumps when landing a finger.
                                                    tp_begin_touch(t, stamp);
                                                }
                                            }
                                            else
                                            {
                                                if (t.touch_limits.max < low || t.touch_limits.min < low)
                                                {
                                                    log("touch-size: end touch %d%", t.index);
                                                    tp_maybe_end_touch(t);
                                                }
                                            }
                                        }
                                    }
                                        bool tp_fake_finger_is_touching()
                                        {
                                            return tp.fake_touches & 0x1;
                                        }
                                    void tp_unhover_fake_touches(time stamp)
                                    {
                                        if (!tp.fake_touches && !tp.nfingers_down)
                                        {
                                            return;
                                        }
                                        auto nfake_touches = tp_fake_finger_count();
                                        if (nfake_touches == lixx::fake_finger_overflow)
                                        {
                                            return;
                                        }
                                        if (tp.nfingers_down == nfake_touches
                                         && ((tp.nfingers_down == 0 && !tp_fake_finger_is_touching())
                                          || (tp.nfingers_down > 0 && tp_fake_finger_is_touching())))
                                        {
                                            return;
                                        }
                                        // If BTN_TOUCH is set and we have less fingers down than fake touches, switch each hovering touch to BEGIN until nfingers_down matches nfake_touches.
                                        if (tp_fake_finger_is_touching() && tp.nfingers_down < nfake_touches)
                                        {
                                            for (auto& t : tp.touches)
                                            {
                                                if (t.state == TOUCH_HOVERING)
                                                {
                                                    tp_begin_touch(t, stamp);
                                                    if (tp.nfingers_down >= nfake_touches)
                                                    {
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                        // If BTN_TOUCH is unset end all touches, we're hovering now. If we have too many touches also end some of them. This is done in reverse order.
                                        if (tp.nfingers_down > nfake_touches || !tp_fake_finger_is_touching())
                                        {
                                            for (auto i = (si32)tp.ntouches - 1; i >= 0; i--)
                                            {
                                                auto& t = tp.touches[i];
                                                if (t.state != TOUCH_HOVERING && t.state != TOUCH_NONE)
                                                {
                                                    tp_maybe_end_touch(t);
                                                    if (tp_fake_finger_is_touching() && tp.nfingers_down == nfake_touches)
                                                    {
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                void tp_unhover_touches(time stamp)
                                {
                                         if (tp.pressure.use_pressure)     tp_unhover_pressure(stamp);
                                    else if (tp.touch_size.use_touch_size) tp_unhover_size(stamp);
                                    else                                   tp_unhover_fake_touches(stamp);
                                }
                                void tp_end_touch(tp_touch& t)
                                {
                                    if (t.state != TOUCH_MAYBE_END)
                                    {
                                        log("touch %d% should be MAYBE_END, is %d%", t.index, t.state);
                                        return;
                                    }
                                    t.dirty                = true;
                                    t.palm_state           = TOUCH_PALM_NONE;
                                    t.state                = TOUCH_END;
                                    t.pinned_state     = faux;
                                    t.palm_stamp           = {};
                                    t.speed_exceeded_count = 0;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_MOTION);
                                }
                            void tp_pre_process_state(time stamp)
                            {
                                if (tp.queued & TOUCHPAD_EVENT_TIMESTAMP) tp_process_msc_timestamp(stamp);
                                tp_process_fake_touches();
                                tp_unhover_touches(stamp);
                                for (auto& t : tp.touches)
                                {
                                    if (t.state == TOUCH_MAYBE_END) tp_end_touch(t);
                                    // Ignore motion when pressure/touch size fell below the threshold, thus ending the touch.
                                    if (t.state == TOUCH_END && t.history.count > 0)
                                    {
                                        t.point = tp_motion_history_offset(t, 0).point;
                                    }
                                }
                            }
                                void tp_position_fake_touches()
                                {
                                    if (tp_fake_finger_count() <= tp.num_slots || tp.nfingers_down == 0) return;
                                    // We have at least one fake touch down. Find the top-most real touch and copy its coordinates over to to all fake touches. This is more reliable than just taking the first touch.
                                    auto topmost = (tp_touch*)nullptr;
                                    for (auto i = 0u; i < tp.num_slots; i++)
                                    {
                                        auto& t = tp.touches[i];
                                        if (t.state != TOUCH_END && t.state != TOUCH_NONE)
                                        {
                                            if (topmost == nullptr || t.point.y < topmost->point.y)
                                            {
                                                topmost = &t;
                                            }
                                        }
                                    }
                                    if (!topmost)
                                    {
                                        log("Unable to find topmost touch");
                                        return;
                                    }
                                    auto start = tp.has_mt ? tp.num_slots : 1;
                                    for (auto i = start; i < tp.ntouches; i++)
                                    {
                                        auto& t = tp.touches[i];
                                        if (t.state != TOUCH_NONE)
                                        {
                                            t.point = topmost->point;
                                            t.pressure = topmost->pressure;
                                            if (!t.dirty)
                                            {
                                                t.dirty = topmost->dirty;
                                            }
                                        }
                                    }
                                }
                                bool tp_need_motion_history_reset()
                                {
                                    auto rc = faux;
                                    // Changing the numbers of fingers can cause a jump in the coordinates, always reset the motion history for all touches when that happens.
                                    if (tp.nfingers_down != tp.old_nfingers_down) return true;
                                    // Quirk: if we had multiple events without x/y axis information, the next x/y event is going to be a jump. So we reset that touch to non-dirty effectively swallowing that event and restarting with the next event again.
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_LENOVO_T450_TOUCHPAD)
                                    {
                                        if (tp.queued & TOUCHPAD_EVENT_MOTION)
                                        {
                                            if (tp.quirks.nonmotion_event_count > 10)
                                            {
                                                tp.queued = (touchpad_event)(tp.queued & ~TOUCHPAD_EVENT_MOTION);
                                                rc = true;
                                            }
                                            tp.quirks.nonmotion_event_count = 0;
                                        }
                                        if ((tp.queued & (TOUCHPAD_EVENT_OTHERAXIS | TOUCHPAD_EVENT_MOTION)) == TOUCHPAD_EVENT_OTHERAXIS)
                                        {
                                            tp.quirks.nonmotion_event_count++;
                                        }
                                    }
                                    return rc;
                                }
                                void tp_motion_history_push(tp_touch& t, time stamp)
                                {
                                    auto motion_index = (t.history.index + 1) % lixx::touchpad_history_length;
                                    if (t.history.count < lixx::touchpad_history_length) t.history.count++;
                                    t.history.samples[motion_index].point = t.point;
                                    t.history.samples[motion_index].stamp = stamp;
                                    t.history.index = motion_index;
                                }
                                    fp64_coor evdev_device_unit_delta_to_mm(si32_coor units)
                                    {
                                        auto mm = fp64_coor{};
                                        if (!tp.ud_device.abs.absinfo_x || !tp.ud_device.abs.absinfo_y)
                                        {
                                            log("%s%: is not an abs device (3)", tp.ud_device.devname);
                                            return mm;
                                        }
                                        auto& absx = tp.ud_device.abs.absinfo_x;
                                        auto& absy = tp.ud_device.abs.absinfo_y;
                                        mm.x = 1.0 * units.x / absx.resolution;
                                        mm.y = 1.0 * units.y / absy.resolution;
                                        return mm;
                                    }
                                    char const* thumb_state_to_str(tp_thumb_state state)
                                    {
                                        switch (state)
                                        {
                                            CASE_RETURN_STRING(THUMB_STATE_FINGER);
                                            CASE_RETURN_STRING(THUMB_STATE_JAILED);
                                            CASE_RETURN_STRING(THUMB_STATE_PINCH);
                                            CASE_RETURN_STRING(THUMB_STATE_SUPPRESSED);
                                            CASE_RETURN_STRING(THUMB_STATE_REVIVED);
                                            CASE_RETURN_STRING(THUMB_STATE_REVIVED_JAILED);
                                            CASE_RETURN_STRING(THUMB_STATE_DEAD);
                                        }
                                        return nullptr;
                                    }
                                bool tp_detect_jumps(tp_touch& t, time stamp)
                                {
                                    // Reference interval from the touchpad the various thresholds were measured from.
                                    auto reference_interval = span{ 12ms };
                                    // On some touchpads the firmware does funky stuff and we cannot have our own jump detection, e.g. Lenovo Carbon X1 Gen 6 (see issue #506).
                                    if (tp.jump.detection_disabled) return faux;
                                    // We haven't seen pointer jumps on Wacom tablets yet, so exclude those.
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_WACOM_TOUCHPAD) return faux;
                                    if (t.history.count == 0)
                                    {
                                        t.jumps_last_delta_mm = 0.0;
                                        return faux;
                                    }
                                    // Called before tp_motion_history_push, so offset 0 is the most recent coordinate.
                                    auto& last = tp_motion_history_offset(t, 0);
                                    auto tdelta = stamp - last.stamp;
                                    // For test devices we always force the time delta to 12, at least until the test suite actually does proper intervals.
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_TEST_DEVICE)
                                    {
                                        reference_interval = tdelta;
                                    }
                                    // If the last frame is more than 30ms ago, we have irregular frames, who knows what's a pointer jump here and what's legitimate movement...
                                    if (tdelta > 2.5 * reference_interval || tdelta == span{})
                                    {
                                        return faux;
                                    }
                                    // We historically expected ~12ms frame intervals, so the numbers below are normalized to that (and that's also where the measured data came from).
                                    auto delta = std::abs(t.point - last.point);
                                    auto mm = evdev_device_unit_delta_to_mm(delta);
                                    auto abs_distance = mm.hypot() * reference_interval / tdelta;
                                    auto rel_distance = abs_distance - t.jumps_last_delta_mm;
                                    // Special case for the ALPS devices in the Lenovo ThinkPad E465, E550. These devices send occasional 4095/0 events on two fingers before snapping back to the correct position.
                                    // https://gitlab.freedesktop.org/libinput/libinput/-/issues/492.
                                    // The specific values are hardcoded here, if this ever happens on any other device we can make it absmax/absmin instead.
                                    if (tp.ud_device.model_flags & EVDEV_MODEL_ALPS_SERIAL_TOUCHPAD
                                     && t.point.x == 4095 && t.point.y == 0)
                                    {
                                        t.point = last.point;
                                        return true;
                                    }
                                    // Cursor jump if:
                                    // - current single-event delta is >20mm, or
                                    // - we increased the delta by over 7mm within a 12ms frame (12ms simply because that's what I measured).
                                    auto is_jump = abs_distance > 20.0 || rel_distance > 7;
                                    t.jumps_last_delta_mm = abs_distance;
                                    return is_jump;
                                }
                                    void tp_thumb_set_state(tp_touch& t, tp_thumb_state state)
                                    {
                                        auto real_touch = !!t.tp;
                                        auto index = real_touch ? t.index : ui32max;
                                        if (tp.thumb.state != state || tp.thumb.index != index)
                                        {
                                            log("thumb: touch %d%, %s% → %s%", (si32)index, thumb_state_to_str(tp.thumb.state), thumb_state_to_str(state));
                                            tp.thumb.state = state;
                                            tp.thumb.index = index;
                                        }
                                    }
                                    void tp_thumb_lift()
                                    {
                                        tp.thumb.state = THUMB_STATE_FINGER;
                                        tp.thumb.index = ui32max;
                                    }
                                            bool tp_thumb_in_exclusion_area(tp_touch& t)
                                            {
                                                return (t.point.y > tp.thumb.lower_thumb_line && tp.tp_scroll.method != LIBINPUT_CONFIG_SCROLL_EDGE);
                                            }
                                            bool tp_thumb_detect_pressure_size(tp_touch& t)
                                            {
                                                auto is_thumb = faux;
                                                if (tp.thumb.use_pressure && t.pressure > tp.thumb.pressure_threshold && tp_thumb_in_exclusion_area(t))
                                                {
                                                    is_thumb = true;
                                                }
                                                if (tp.thumb.use_size && (t.touch_limits.max > tp.thumb.size_threshold) && (t.touch_limits.min < (tp.thumb.size_threshold * 0.6)))
                                                {
                                                    is_thumb = true;
                                                }
                                                return is_thumb;
                                            }
                                        bool tp_thumb_needs_jail(tp_touch& t)
                                        {
                                            if (t.point.y < tp.thumb.upper_thumb_line || tp.tp_scroll.method == LIBINPUT_CONFIG_SCROLL_EDGE)
                                            {
                                                return faux;
                                            }
                                            if (!tp_thumb_in_exclusion_area(t) && (tp.thumb.use_size || tp.thumb.use_pressure) && !tp_thumb_detect_pressure_size(t))
                                            {
                                                return faux;
                                            }
                                            if (t.speed_exceeded_count >= 10)
                                            {
                                                return faux;
                                            }
                                            else
                                            {
                                                return true;
                                            }
                                        }
                                    void tp_thumb_revive(tp_touch& t)
                                    {
                                        if ((tp.thumb.state != THUMB_STATE_SUPPRESSED && tp.thumb.state != THUMB_STATE_PINCH)
                                         || (tp.thumb.index != t.index))
                                        {
                                            return;
                                        }
                                        if (tp_thumb_needs_jail(t)) tp_thumb_set_state(t, THUMB_STATE_REVIVED_JAILED);
                                        else                        tp_thumb_set_state(t, THUMB_STATE_REVIVED);
                                    }
                                void tp_thumb_update_touch(tp_touch& t)
                                {
                                    if (!tp.thumb.detect_thumbs) return;
                                    // Once any active touch exceeds the speed threshold, don't try to detect pinches until all touches lift.
                                    if (t.speed_exceeded_count >= 10 && tp.thumb.pinch_eligible && tp.gesture.state == GESTURE_STATE_NONE)
                                    {
                                        tp.thumb.pinch_eligible = faux;
                                        if (tp.thumb.state == THUMB_STATE_PINCH)
                                        {
                                            for (auto& thumb : tp.touches)
                                            {
                                                if (thumb.index == tp.thumb.index)
                                                {
                                                    tp_thumb_set_state(thumb, THUMB_STATE_SUPPRESSED);
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                    // Handle the thumb lifting off the touchpad.
                                    if (t.state == TOUCH_END && t.index == tp.thumb.index)
                                    {
                                        tp_thumb_lift();
                                        return;
                                    }
                                    // If this touch is not the only one, thumb updates happen by context instead of here.
                                    if (tp.nfingers_down > 1)
                                        return;
                                    // If we arrived here by other fingers lifting off, revive current touch if appropriate.
                                    tp_thumb_revive(t);
                                    // First new touch below the lower_thumb_line, or below the upper_thumb_ line if hardware can't verify it's a finger, starts as JAILED.
                                    if (t.state == TOUCH_BEGIN && tp_thumb_needs_jail(t))
                                    {
                                        tp_thumb_set_state(t, THUMB_STATE_JAILED);
                                        return;
                                    }
                                    // If a touch breaks the speed threshold, or leaves the thumb area (upper or lower, depending on HW detection), it "escapes" jail.
                                    if (tp.thumb.state == THUMB_STATE_JAILED && !(tp_thumb_needs_jail(t)))
                                    {
                                        tp_thumb_set_state(t, THUMB_STATE_FINGER);
                                    }
                                    if (tp.thumb.state == THUMB_STATE_REVIVED_JAILED && !(tp_thumb_needs_jail(t)))
                                    {
                                        tp_thumb_set_state(t, THUMB_STATE_REVIVED);
                                    }
                                }
                                    bool tp_palm_detect_pressure_triggered(tp_touch& t)
                                    {
                                        if (!tp.palm.use_pressure) return faux;
                                        if (t.palm_state != TOUCH_PALM_NONE && t.palm_state != TOUCH_PALM_PRESSURE) return faux;
                                        if (t.pressure > tp.palm.pressure_threshold) t.palm_state = TOUCH_PALM_PRESSURE;
                                        return t.palm_state == TOUCH_PALM_PRESSURE;
                                    }
                                    bool tp_palm_detect_arbitration_triggered(tp_touch& t)
                                    {
                                        if (tp.arbitration.state == ARBITRATION_NOT_ACTIVE) return faux;
                                        t.palm_state = TOUCH_PALM_ARBITRATION;
                                        return true;
                                    }
                                    bool tp_palm_detect_dwt_triggered(tp_touch& t)
                                    {
                                        if (tp.dwt.dwt_enabled && tp.dwt.keyboard_active && t.state == TOUCH_BEGIN)
                                        {
                                            t.palm_state = TOUCH_PALM_TYPING;
                                            t.palm_first = t.point;
                                            return true;
                                        }
                                        if (!tp.dwt.keyboard_active && t.state == TOUCH_UPDATE && t.palm_state == TOUCH_PALM_TYPING)
                                        {
                                            // If a touch has started before the first or after the last key press, release it on timeout. Benefit: a palm rested while typing on the touchpad will be ignored, but a touch started once we stop typing will be able to control the pointer (alas not tap, etc).
                                            if (t.palm_stamp == time{} || t.palm_stamp > tp.dwt.keyboard_last_press_time)
                                            {
                                                t.palm_state = TOUCH_PALM_NONE;
                                                log("palm: touch %d% released, timeout after typing", t.index);
                                            }
                                        }
                                        return faux;
                                    }
                                    bool tp_palm_detect_trackpoint_triggered(tp_touch& t)
                                    {
                                        if (tp.palm.monitor_trackpoint)
                                        {
                                            if (t.palm_state == TOUCH_PALM_NONE && t.state == TOUCH_BEGIN && tp.palm.trackpoint_active)
                                            {
                                                t.palm_state = TOUCH_PALM_TRACKPOINT;
                                                return true;
                                            }
                                            if (t.palm_state == TOUCH_PALM_TRACKPOINT && t.state == TOUCH_UPDATE && !tp.palm.trackpoint_active)
                                            {
                                                if (t.palm_stamp == time{} || t.palm_stamp > tp.palm.trackpoint_last_event_time)
                                                {
                                                    t.palm_state = TOUCH_PALM_NONE;
                                                    log("palm: touch %d% released, timeout after trackpoint", t.index);
                                                }
                                            }
                                        }
                                        return faux;
                                    }
                                    bool tp_palm_detect_tool_triggered(tp_touch& t)
                                    {
                                        if (!tp.palm.use_mt_tool) return faux;
                                        if (t.palm_state != TOUCH_PALM_NONE && t.palm_state != TOUCH_PALM_TOOL_PALM) return faux;
                                             if (t.palm_state == TOUCH_PALM_NONE      && t.is_tool_palm)  t.palm_state = TOUCH_PALM_TOOL_PALM;
                                        else if (t.palm_state == TOUCH_PALM_TOOL_PALM && !t.is_tool_palm) t.palm_state = TOUCH_PALM_NONE;
                                        return t.palm_state == TOUCH_PALM_TOOL_PALM;
                                    }
                                    bool tp_palm_detect_touch_size_triggered(tp_touch& t)
                                    {
                                        if (!tp.palm.use_size) return faux;
                                        // If a finger size is large enough for palm, we stick with that and force the user to release and reset the finger.
                                        if (t.palm_state != TOUCH_PALM_NONE && t.palm_state != TOUCH_PALM_TOUCH_SIZE) return faux;
                                        if (t.touch_limits.max > tp.palm.size_threshold || t.touch_limits.min > tp.palm.size_threshold)
                                        {
                                            if (t.palm_state != TOUCH_PALM_TOUCH_SIZE) log("palm: touch %d% size exceeded", t.index);
                                            t.palm_state = TOUCH_PALM_TOUCH_SIZE;
                                            return true;
                                        }
                                        return faux;
                                    }
                                                bool tp_thumb_ignored(tp_touch& t)
                                                {
                                                    return (tp.thumb.detect_thumbs && tp.thumb.index == t.index
                                                        && (tp.thumb.state == THUMB_STATE_JAILED
                                                         || tp.thumb.state == THUMB_STATE_PINCH
                                                         || tp.thumb.state == THUMB_STATE_SUPPRESSED
                                                         || tp.thumb.state == THUMB_STATE_REVIVED_JAILED
                                                         || tp.thumb.state == THUMB_STATE_DEAD));
                                                }
                                                bool tp_button_touch_active(tp_touch& t)
                                                {
                                                    return t.button.state == BUTTON_STATE_AREA || t.button.has_moved;
                                                }
                                                si32 tp_edge_scroll_touch_active(tp_touch& t)
                                                {
                                                    return t.scroll.edge_state == EDGE_SCROLL_TOUCH_STATE_AREA;
                                                }
                                            bool tp_touch_active(tp_touch& t)
                                            {
                                                return (t.state == TOUCH_BEGIN || t.state == TOUCH_UPDATE)
                                                     && t.palm_state == TOUCH_PALM_NONE
                                                     && !t.pinned_state
                                                     && !tp_thumb_ignored(t)
                                                     && tp_button_touch_active(t)
                                                     && tp_edge_scroll_touch_active(t);
                                            }
                                        bool tp_palm_detect_multifinger(tp_touch& t)
                                        {
                                            if (tp.nfingers_down < 2) return faux;
                                            // If we have at least one other active non-palm touch make this touch non-palm too. This avoids palm detection during two-finger scrolling.
                                            // Note: if both touches start in the palm zone within the same frame the second touch will still be TOUCH_PALM_NONE and thus detected here as non-palm touch. This is too niche to worry about for now.
                                            for (auto& other : tp.touches)
                                            {
                                                if (&other != &t && tp_touch_active(other) && other.palm_state == TOUCH_PALM_NONE)
                                                {
                                                    return true;
                                                }
                                            }
                                            return faux;
                                        }
                                                bool tp_palm_in_side_edge(tp_touch& t)
                                                {
                                                    return t.point.x < tp.palm.left_edge || t.point.x > tp.palm.right_edge;
                                                }
                                                bool tp_palm_in_top_edge(tp_touch& t)
                                                {
                                                    return t.point.y < tp.palm.upper_edge;
                                                }
                                            bool tp_palm_in_edge(tp_touch& t)
                                            {
                                                return tp_palm_in_side_edge(t) || tp_palm_in_top_edge(t);
                                            }
                                            bool tp_palm_was_in_side_edge(tp_touch& t)
                                            {
                                                return t.palm_first.x < tp.palm.left_edge || t.palm_first.x > tp.palm.right_edge;
                                            }
                                            bool tp_palm_was_in_top_edge(tp_touch& t)
                                            {
                                                return t.palm_first.y < tp.palm.upper_edge;
                                            }
                                            fp64_coor tp_phys_delta(fp64_coor delta)
                                            {
                                                auto mm = delta / tp.ud_device.abs.absinfo_x.resolution;
                                                return mm;
                                            }
                                        bool tp_palm_detect_move_out_of_edge(tp_touch& t, time stamp)
                                        {
                                            if (stamp < t.palm_stamp + lixx::palm_timeout && !tp_palm_in_edge(t))
                                            {
                                                auto directions = 0;
                                                if (tp_palm_was_in_side_edge(t))
                                                {
                                                    directions = NE | E | SE | SW | W | NW;
                                                }
                                                else if (tp_palm_was_in_top_edge(t))
                                                {
                                                    directions = S | SE | SW;
                                                }
                                                if (directions)
                                                {
                                                    auto delta = t.point - t.palm_first;
                                                    auto mm = tp_phys_delta(delta);
                                                    auto dirs = pointer_tracker::get_direction(mm);
                                                    if ((dirs & directions) && !(dirs & ~directions))
                                                    {
                                                        return true;
                                                    }
                                                }
                                            }
                                            return faux;
                                        }
                                        ui32 tp_touch_get_edge(tp_touch& t)
                                        {
                                            auto edge = (ui32)EDGE_NONE;
                                            if (tp.tp_scroll.method == LIBINPUT_CONFIG_SCROLL_EDGE)
                                            {
                                                if (t.point.x > tp.tp_scroll.right_edge)  edge |= EDGE_RIGHT;
                                                if (t.point.y > tp.tp_scroll.bottom_edge) edge |= EDGE_BOTTOM;
                                            }
                                            return edge;
                                        }
                                    bool tp_palm_detect_edge(tp_touch& t, time stamp)
                                    {
                                        if (t.palm_state == TOUCH_PALM_EDGE)
                                        {
                                            if (tp_palm_detect_multifinger(t))
                                            {
                                                t.palm_state = TOUCH_PALM_NONE;
                                                log("palm: touch %d% released, multiple fingers", t.index);
                                                // If labelled a touch as palm, we unlabel as palm when we move out of the palm edge zone within the timeout, provided the direction is within 45 degrees of the horizontal.
                                            }
                                            else if (tp_palm_detect_move_out_of_edge(t, stamp))
                                            {
                                                t.palm_state = TOUCH_PALM_NONE;
                                                log("palm: touch %d% released, out of edge zone", t.index);
                                            }
                                            return faux;
                                        }
                                        if (tp_palm_detect_multifinger(t))
                                        {
                                            return faux;
                                        }
                                        // Palm must start in exclusion zone, it's ok to move into the zone without being a palm.
                                        if (t.state != TOUCH_BEGIN || !tp_palm_in_edge(t)) return faux;
                                        if (tp_touch_get_edge(t) & EDGE_RIGHT) return faux;
                                        t.palm_state = TOUCH_PALM_EDGE;
                                        t.palm_stamp = stamp;
                                        t.palm_first = t.point;
                                        return true;
                                    }
                                    char const* touch_state_to_str(touch_state state)
                                    {
                                        switch (state)
                                        {
                                            CASE_RETURN_STRING(TOUCH_NONE);
                                            CASE_RETURN_STRING(TOUCH_HOVERING);
                                            CASE_RETURN_STRING(TOUCH_BEGIN);
                                            CASE_RETURN_STRING(TOUCH_UPDATE);
                                            CASE_RETURN_STRING(TOUCH_MAYBE_END);
                                            CASE_RETURN_STRING(TOUCH_END);
                                        }
                                        return nullptr;
                                    }
                                void tp_palm_detect(tp_touch& t, time stamp)
                                {
                                    auto oldstate = t.palm_state;
                                    auto detected = tp_palm_detect_pressure_triggered(   t)
                                                 || tp_palm_detect_arbitration_triggered(t)
                                                 || tp_palm_detect_dwt_triggered(        t)
                                                 || tp_palm_detect_trackpoint_triggered( t)
                                                 || tp_palm_detect_tool_triggered(       t)
                                                 || tp_palm_detect_touch_size_triggered( t)
                                                 || tp_palm_detect_edge(                 t, stamp)
                                                 || tp_palm_detect_pressure_triggered(   t);
                                                 // Pressure is highest priority because it cannot be released and overrides all other checks. So we check once before anything else in case pressure triggers on a non-palm touch. And again after everything in case one of the others released but we have a pressure trigger now.
                                    if constexpr (debugmode)
                                    {
                                        if (detected && oldstate != t.palm_state)
                                        {
                                            [[maybe_unused]] auto palm_state = (char const*)nullptr;
                                            switch (t.palm_state)
                                            {
                                                case TOUCH_PALM_EDGE:        palm_state = "edge";        break;
                                                case TOUCH_PALM_TYPING:      palm_state = "typing";      break;
                                                case TOUCH_PALM_TRACKPOINT:  palm_state = "trackpoint";  break;
                                                case TOUCH_PALM_TOOL_PALM:   palm_state = "tool-palm";   break;
                                                case TOUCH_PALM_PRESSURE:    palm_state = "pressure";    break;
                                                case TOUCH_PALM_TOUCH_SIZE:  palm_state = "touch size";  break;
                                                case TOUCH_PALM_ARBITRATION: palm_state = "arbitration"; break;
                                                case TOUCH_PALM_NONE:
                                                default: ::abort();
                                            }
                                            log("palm: touch %d% (%s%), palm detected (%s%)", t.index, touch_state_to_str(t.state), palm_state);
                                        }
                                    }
                                }
                                void tp_detect_wobbling(tp_touch& t, time stamp)
                                {
                                    // Idea: if we got a tuple of *very* quick moves like {Left, Right,
                                    // Left}, or {Right, Left, Right}, it means touchpad jitters since no
                                    // human can move like that within thresholds.
                                    //
                                    // We encode left moves as zeroes, and right as ones. We also drop
                                    // the array to all zeroes when constraints are not satisfied. Then we
                                    // search for the pattern {1,0,1}. It can't match {Left, Right, Left},
                                    // but it does match {Left, Right, Left, Right}, so it's okay.
                                    //
                                    // This only looks at x changes, y changes are ignored.
                                    if (tp.nfingers_down != 1 || tp.nfingers_down != tp.old_nfingers_down) return;
                                    if (tp.hysteresis.enabled || t.history.count == 0) return;
                                    if (!(tp.queued & TOUCHPAD_EVENT_MOTION))
                                    {
                                        t.hysteresis_x_motion_history = 0;
                                        return;
                                    }
                                    auto prev_point = tp_motion_history_offset(t, 0).point;
                                    auto d = prev_point - t.point;
                                    auto dtime = stamp - tp.hysteresis.last_motion_time;
                                    tp.hysteresis.last_motion_time = stamp;
                                    if ((d.x == 0 && d.y != 0) || dtime > 40ms)
                                    {
                                        t.hysteresis_x_motion_history = 0;
                                        return;
                                    }
                                    t.hysteresis_x_motion_history >>= 1;
                                    if (d.x > 0) // Right move.
                                    {
                                        static const auto r_l_r = (byte)0x5; // {Right, Left, Right}.
                                        t.hysteresis_x_motion_history |= (1ul << 2);
                                        if (t.hysteresis_x_motion_history == r_l_r)
                                        {
                                            tp.hysteresis.enabled = true;
                                            log("hysteresis enabled");
                                        }
                                    }
                                }
                                void tp_motion_hysteresis(tp_touch& t)
                                {
                                    if (tp.hysteresis.enabled)
                                    {
                                        if (t.history.count > 0)
                                        {
                                            t.point = libinput_device_t::apply_hysteresis(t.point, t.hysteresis_center, tp.hysteresis.margin);
                                        }
                                        t.hysteresis_center = t.point;
                                    }
                                }
                                void tp_calculate_motion_speed(tp_touch& t, time stamp)
                                {
                                    // Don't do this on single-touch or semi-mt devices.
                                    if (!tp.has_mt || tp.semi_mt) return;
                                    if (t.state != TOUCH_UPDATE) return;
                                    // This doesn't kick in until we have at least 4 events in the
                                    // motion history. As a side-effect, this automatically handles the
                                    // 2fg scroll where a finger is down and moving fast before the
                                    // other finger comes down for the scroll.
                                    //
                                    // We do *not* reset the speed to 0 here though. The motion history
                                    // is reset whenever a new finger is down, so we'd be resetting the
                                    // speed and failing.
                                    if (t.history.count < 4) return;
                                    //todo: we probably need a speed history here so we can average across a few events
                                    auto& last = tp_motion_history_offset(t, 1);
                                    auto delta = std::abs(t.point - last.point);
                                    auto mm = evdev_device_unit_delta_to_mm(delta);
                                    auto distance = mm.hypot();
                                    auto speed = distance / datetime::round<si64, std::chrono::microseconds>(stamp - last.stamp); // mm/us.
                                    speed *= 1000000; // mm/s.
                                    t.speed_last = speed;
                                }
                                void tp_unpin_finger(tp_touch& t)
                                {
                                    if (t.pinned_state)
                                    {
                                        auto delta = std::abs(t.point - t.pinned_center);
                                        auto mm = evdev_device_unit_delta_to_mm(delta);
                                        if (mm.hypot() >= 1.5) // 1.5mm movement -> unpin.
                                        {
                                            t.pinned_state = faux;
                                        }
                                    }
                                }
                                    void tp_thumb_suppress(tp_touch& t)
                                    {
                                        if (tp.thumb.state == THUMB_STATE_FINGER
                                         || tp.thumb.state == THUMB_STATE_JAILED
                                         || tp.thumb.state == THUMB_STATE_PINCH
                                         || tp.thumb.index != t.index)
                                        {
                                            tp_thumb_set_state(t, THUMB_STATE_SUPPRESSED);
                                        }
                                        else
                                        {
                                            tp_thumb_set_state(t, THUMB_STATE_DEAD);
                                        }
                                    }
                                    void tp_thumb_pinch(tp_touch& t)
                                    {
                                        if (tp.thumb.state == THUMB_STATE_FINGER
                                         || tp.thumb.state == THUMB_STATE_JAILED
                                         || tp.thumb.index != t.index)
                                        {
                                            tp_thumb_set_state(t, THUMB_STATE_PINCH);
                                        }
                                        else if (tp.thumb.state != THUMB_STATE_PINCH)
                                        {
                                            tp_thumb_suppress(t);
                                        }
                                    }
                                void tp_thumb_update_multifinger()
                                {
                                    auto first = (tp_touch*)nullptr;
                                    auto second = (tp_touch*)nullptr;
                                    auto newest = (tp_touch*)nullptr;
                                    auto oldest = (tp_touch*)nullptr;
                                    // Get the first and second bottom-most touches, the max speed exceeded count overall, and the newest and oldest touches.
                                    auto speed_exceeded_count = 0u;
                                    for (auto& t : tp.touches)
                                    {
                                        if (t.state != TOUCH_NONE && t.state != TOUCH_HOVERING)
                                        {
                                            if (t.state == TOUCH_BEGIN)
                                            {
                                                newest = &t;
                                            }
                                            speed_exceeded_count = std::max(speed_exceeded_count, t.speed_exceeded_count);
                                            if (!oldest || t.initial_time < oldest->initial_time)
                                            {
                                                oldest = &t;
                                            }
                                            if (!first)
                                            {
                                                first = &t;
                                            }
                                            else if (t.point.y > first->point.y)
                                            {
                                                second = first;
                                                first = &t;
                                            }
                                            else if (!second || t.point.y > second->point.y )
                                            {
                                                second = &t;
                                            }
                                        }
                                    }
                                    if (!first || !second) return;
                                    auto distance = std::abs(first->point - second->point);
                                    auto mm = evdev_device_unit_delta_to_mm(distance);
                                    // Speed-based thumb detection: if an existing finger is moving, and a new touch arrives, mark it as a thumb if it doesn't qualify as a 2-finger scroll. Also account for a thumb dropping onto the touchpad while scrolling or swiping.
                                    // distance between fingers to assume it is not a scroll.
                                    static constexpr auto scroll_mm_x = 35;
                                    static constexpr auto scroll_mm_y = 25;
                                    if (newest && tp.thumb.state == THUMB_STATE_FINGER
                                     && tp.nfingers_down >= 2 && speed_exceeded_count > 5
                                     && (tp.tp_scroll.method != LIBINPUT_CONFIG_SCROLL_2FG || mm.x > scroll_mm_x || mm.y > scroll_mm_y))
                                    {
                                        log("touch %d% is speed-based thumb", newest->index);
                                        tp_thumb_suppress(*newest);
                                        return;
                                    }
                                    // Contextual thumb detection: When a new touch arrives, check the timing and position of the two lowest touches.
                                    // If both touches are very close, regardless of timing, and no matter their absolute position on the touchpad, count them both as live to support responsive two-finger scrolling.
                                    if (mm.x < scroll_mm_x && mm.y < scroll_mm_y)
                                    {
                                        tp_thumb_lift();
                                        return;
                                    }
                                    // If all the touches arrived within a very short time, and all of them are above the lower_thumb_line, assume the touches are all live to enable double, triple, and quadruple taps, clicks, and gestures. (If there is an actual resting thumb, it will be detected later based on the behavior of the other touches.)
                                    if (newest && (newest->initial_time - oldest->initial_time) < lixx::thumb_timeout
                                     && first->point.y < tp.thumb.lower_thumb_line)
                                    {
                                        tp_thumb_lift();
                                        return;
                                    }
                                    // If we're past the lixx::thumb_timeout, and the touches are relatively far apart, then the new touch is unlikely to be a tap or clickfinger. Proceed with pre-1.14.901 thumb detection.
                                    if (mm.y > scroll_mm_y)
                                    {
                                        if (tp.thumb.pinch_eligible) tp_thumb_pinch(*first);
                                        else                         tp_thumb_suppress(*first);
                                    }
                                    else
                                    {
                                        tp_thumb_lift();
                                    }
                                }
                                            void tp_button_set_state(tp_touch& t, button_state_enum new_state, button_event event, time stamp)
                                            {
                                                t.button.timer->cancel();
                                                t.button.state = new_state;
                                                switch (t.button.state)
                                                {
                                                    case BUTTON_STATE_AREA:          t.button.current = BUTTON_EVENT_IN_AREA; break;
                                                    case BUTTON_STATE_BOTTOM:        t.button.current = event; break;
                                                    case BUTTON_STATE_TOP_NEW:       t.button.current = event;
                                                                                     t.button.timer->start(stamp + lixx::default_button_enter_timeout); break;
                                                    case BUTTON_STATE_NONE:
                                                    case BUTTON_STATE_IGNORE:        t.button.current = {}; break;
                                                    case BUTTON_STATE_TOP_TO_IGNORE: t.button.timer->start(stamp + lixx::default_button_leave_timeout); break;
                                                    case BUTTON_STATE_TOP: break;
                                                }
                                            }
                                        void tp_button_none_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L: tp_button_set_state(t, BUTTON_STATE_BOTTOM, event, stamp); break;
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L: tp_button_set_state(t, BUTTON_STATE_TOP_NEW, event, stamp); break;
                                                case BUTTON_EVENT_IN_AREA:  tp_button_set_state(t, BUTTON_STATE_AREA, event, stamp); break;
                                                case BUTTON_EVENT_UP:       tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_PRESS:
                                                case BUTTON_EVENT_RELEASE:
                                                case BUTTON_EVENT_TIMEOUT:
                                                case BUTTON_EVENT_NONE: break;
                                            }
                                        }
                                        void tp_button_area_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L:
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                case BUTTON_EVENT_IN_AREA: break;
                                                case BUTTON_EVENT_UP: tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_PRESS:
                                                case BUTTON_EVENT_RELEASE:
                                                case BUTTON_EVENT_TIMEOUT:
                                                case BUTTON_EVENT_NONE: break;
                                            }
                                        }
                                            void tp_button_release_other_bottom_touches(time other_start_time) // Release any button in the bottom area, provided it started within a threshold around start_time (i.e. simultaneously with the other touch that triggered this call).
                                            {
                                                for (auto& t : tp.touches)
                                                {
                                                    if (t.button.state == BUTTON_STATE_BOTTOM && !t.button.has_moved)
                                                    {
                                                        auto gt = other_start_time > t.button.initial_time;
                                                        auto tdelta = gt ? other_start_time - t.button.initial_time
                                                                         : t.button.initial_time - other_start_time;
                                                        if (tdelta <= 80ms)
                                                        {
                                                            t.button.has_moved = true;
                                                        }
                                                    }
                                                }
                                            }
                                        void tp_button_bottom_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L:
                                                    if (event != t.button.current) tp_button_set_state(t, BUTTON_STATE_BOTTOM, event, stamp);
                                                    break;
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                case BUTTON_EVENT_IN_AREA:
                                                    tp_button_set_state(t, BUTTON_STATE_AREA, event, stamp);
                                                    // We just transitioned one finger from BOTTOM to AREA, if there are other fingers in BOTTOM that started simultaneously with this finger, release those fingers because they're part of a gesture.
                                                    tp_button_release_other_bottom_touches(t.button.initial_time);
                                                    break;
                                                case BUTTON_EVENT_UP:
                                                    tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp);
                                                    break;
                                                case BUTTON_EVENT_PRESS:
                                                case BUTTON_EVENT_RELEASE:
                                                case BUTTON_EVENT_TIMEOUT:
                                                case BUTTON_EVENT_NONE:
                                                    break;
                                            }
                                        }
                                        void tp_button_top_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L: tp_button_set_state(t, BUTTON_STATE_TOP_TO_IGNORE, event, stamp); break;
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                    if (event != t.button.current) tp_button_set_state(t, BUTTON_STATE_TOP_NEW, event, stamp);
                                                    break;
                                                case BUTTON_EVENT_IN_AREA: tp_button_set_state(t, BUTTON_STATE_TOP_TO_IGNORE, event, stamp); break;
                                                case BUTTON_EVENT_UP:      tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_PRESS:
                                                case BUTTON_EVENT_RELEASE:
                                                case BUTTON_EVENT_TIMEOUT:
                                                case BUTTON_EVENT_NONE: break;
                                            }
                                        }
                                        void tp_button_top_new_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L: tp_button_set_state(t, BUTTON_STATE_AREA, event, stamp); break;
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                    if (event != t.button.current) tp_button_set_state(t, BUTTON_STATE_TOP_NEW, event, stamp);
                                                    break;
                                                case BUTTON_EVENT_IN_AREA: tp_button_set_state(t, BUTTON_STATE_AREA, event, stamp); break;
                                                case BUTTON_EVENT_UP:      tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_PRESS:   tp_button_set_state(t, BUTTON_STATE_TOP, event, stamp); break;
                                                case BUTTON_EVENT_TIMEOUT: tp_button_set_state(t, BUTTON_STATE_TOP, event, stamp); break;
                                                case BUTTON_EVENT_RELEASE:
                                                case BUTTON_EVENT_NONE:
                                                    break;
                                            }
                                        }
                                        void tp_button_top_to_ignore_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                    if (event == t.button.current) tp_button_set_state(t, BUTTON_STATE_TOP, event, stamp);
                                                    else                           tp_button_set_state(t, BUTTON_STATE_TOP_NEW, event, stamp);
                                                    break;
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L:
                                                case BUTTON_EVENT_IN_AREA:
                                                case BUTTON_EVENT_NONE: break;
                                                case BUTTON_EVENT_PRESS:
                                                case BUTTON_EVENT_RELEASE: break;
                                                case BUTTON_EVENT_UP:      tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_TIMEOUT: tp_button_set_state(t, BUTTON_STATE_IGNORE, event, stamp); break;
                                            }
                                        }
                                        void tp_button_ignore_handle_event(tp_touch& t, button_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case BUTTON_EVENT_IN_BOTTOM_R:
                                                case BUTTON_EVENT_IN_BOTTOM_M:
                                                case BUTTON_EVENT_IN_BOTTOM_L:
                                                case BUTTON_EVENT_IN_TOP_R:
                                                case BUTTON_EVENT_IN_TOP_M:
                                                case BUTTON_EVENT_IN_TOP_L:
                                                case BUTTON_EVENT_IN_AREA: break;
                                                case BUTTON_EVENT_UP: tp_button_set_state(t, BUTTON_STATE_NONE, event, stamp); break;
                                                case BUTTON_EVENT_PRESS: t.button.current = BUTTON_EVENT_IN_AREA; break;
                                                case BUTTON_EVENT_RELEASE: break;
                                                case BUTTON_EVENT_TIMEOUT:
                                                case BUTTON_EVENT_NONE: break;
                                            }
                                        }
                                        const char* button_state_to_str(button_state_enum state)
                                        {
                                            switch (state)
                                            {
                                                CASE_RETURN_STRING(BUTTON_STATE_NONE);
                                                CASE_RETURN_STRING(BUTTON_STATE_AREA);
                                                CASE_RETURN_STRING(BUTTON_STATE_BOTTOM);
                                                CASE_RETURN_STRING(BUTTON_STATE_TOP);
                                                CASE_RETURN_STRING(BUTTON_STATE_TOP_NEW);
                                                CASE_RETURN_STRING(BUTTON_STATE_TOP_TO_IGNORE);
                                                CASE_RETURN_STRING(BUTTON_STATE_IGNORE);
                                            }
                                            return nullptr;
                                        }
                                        const char* button_event_to_str(button_event event)
                                        {
                                            switch (event)
                                            {
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_BOTTOM_R);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_BOTTOM_M);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_BOTTOM_L);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_TOP_R);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_TOP_M);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_TOP_L);
                                                CASE_RETURN_STRING(BUTTON_EVENT_IN_AREA);
                                                CASE_RETURN_STRING(BUTTON_EVENT_UP);
                                                CASE_RETURN_STRING(BUTTON_EVENT_PRESS);
                                                CASE_RETURN_STRING(BUTTON_EVENT_RELEASE);
                                                CASE_RETURN_STRING(BUTTON_EVENT_TIMEOUT);
                                                CASE_RETURN_STRING(BUTTON_EVENT_NONE);
                                            }
                                            return nullptr;
                                        }
                                    void tp_button_handle_event(tp_touch& t, button_event event, time stamp)
                                    {
                                        auto current = t.button.state;
                                        switch (current)
                                        {
                                            case BUTTON_STATE_NONE:           tp_button_none_handle_event(         t, event, stamp); break;
                                            case BUTTON_STATE_AREA:           tp_button_area_handle_event(         t, event, stamp); break;
                                            case BUTTON_STATE_BOTTOM:         tp_button_bottom_handle_event(       t, event, stamp); break;
                                            case BUTTON_STATE_TOP:            tp_button_top_handle_event(          t, event, stamp); break;
                                            case BUTTON_STATE_TOP_NEW:        tp_button_top_new_handle_event(      t, event, stamp); break;
                                            case BUTTON_STATE_TOP_TO_IGNORE:  tp_button_top_to_ignore_handle_event(t, event, stamp); break;
                                            case BUTTON_STATE_IGNORE:         tp_button_ignore_handle_event(       t, event, stamp); break;
                                        }
                                        if (current != t.button.state)
                                        {
                                            log("button state1: touch %d% from %-20s% event %-24s% to %-20s%", t.index, button_state_to_str(current), button_event_to_str(event), button_state_to_str(t.button.state));
                                        }
                                    }
                                    bool is_inside_bottom_button_area(tp_touch& t)
                                    {
                                        return t.point.y >= tp.buttons.bottom_area.top_edge;
                                    }
                                    bool is_inside_bottom_right_area(tp_touch& t)
                                    {
                                        return is_inside_bottom_button_area(t) && t.point.x > tp.buttons.bottom_area.rightbutton_left_edge;
                                    }
                                    bool is_inside_bottom_middle_area(tp_touch& t)
                                    {
                                        return is_inside_bottom_button_area(t)
                                            && !is_inside_bottom_right_area(t)
                                            && t.point.x > tp.buttons.bottom_area.middlebutton_left_edge;
                                    }
                                    bool is_inside_top_button_area(tp_touch& t)
                                    {
                                        return t.point.y <= tp.buttons.top_area.bottom_edge;
                                    }
                                    bool is_inside_top_right_area(tp_touch& t)
                                    {
                                        return is_inside_top_button_area(t) && t.point.x > tp.buttons.top_area.rightbutton_left_edge;
                                    }
                                    bool is_inside_top_middle_area(tp_touch& t)
                                    {
                                        return is_inside_top_button_area(t)
                                            && t.point.x >= tp.buttons.top_area.leftbutton_right_edge
                                            && t.point.x <= tp.buttons.top_area.rightbutton_left_edge;
                                    }
                                    void tp_button_check_for_movement(tp_touch& t)
                                    {
                                        if (t.button.has_moved) return;
                                        switch (t.button.state)
                                        {
                                            case BUTTON_STATE_NONE:
                                            case BUTTON_STATE_AREA:
                                            case BUTTON_STATE_TOP:
                                            case BUTTON_STATE_TOP_NEW:
                                            case BUTTON_STATE_TOP_TO_IGNORE:
                                            case BUTTON_STATE_IGNORE:
                                                return; // No point calculating if we're not going to use it.
                                            case BUTTON_STATE_BOTTOM:
                                                break;
                                        }
                                        auto delta = t.point - t.button.initial;
                                        auto mm = evdev_device_unit_delta_to_mm(delta);
                                        auto vector_length = mm.hypot();
                                        if (vector_length > 5.0) // mm.
                                        {
                                            t.button.has_moved = true;
                                            tp_button_release_other_bottom_touches(t.button.initial_time);
                                        }
                                    }
                                void tp_button_handle_state(time stamp)
                                {
                                    for (auto& t : tp.touches)
                                    {
                                        if (t.state == TOUCH_NONE || t.state == TOUCH_HOVERING) continue;
                                        if (t.state == TOUCH_BEGIN)
                                        {
                                            t.button.initial = t.point;
                                            t.button.initial_time = stamp;
                                            t.button.has_moved = faux;
                                        }
                                        if (t.state == TOUCH_END)
                                        {
                                            tp_button_handle_event(t, BUTTON_EVENT_UP, stamp);
                                        }
                                        else if (t.dirty)
                                        {
                                            button_event event;
                                            if (is_inside_bottom_button_area(t))
                                            {
                                                     if (is_inside_bottom_right_area(t))  event = BUTTON_EVENT_IN_BOTTOM_R;
                                                else if (is_inside_bottom_middle_area(t)) event = BUTTON_EVENT_IN_BOTTOM_M;
                                                else                                          event = BUTTON_EVENT_IN_BOTTOM_L;
                                                // In the bottom area we check for movement within the area. Top area - meh.
                                                tp_button_check_for_movement(t);
                                            }
                                            else if (is_inside_top_button_area(t))
                                            {
                                                     if (is_inside_top_right_area(t))  event = BUTTON_EVENT_IN_TOP_R;
                                                else if (is_inside_top_middle_area(t)) event = BUTTON_EVENT_IN_TOP_M;
                                                else                                       event = BUTTON_EVENT_IN_TOP_L;
                                            }
                                            else
                                            {
                                                event = BUTTON_EVENT_IN_AREA;
                                            }
                                            tp_button_handle_event(t, event, stamp);
                                        }
                                        if (tp.queued & TOUCHPAD_EVENT_BUTTON_RELEASE) tp_button_handle_event(t, BUTTON_EVENT_RELEASE, stamp);
                                        if (tp.queued & TOUCHPAD_EVENT_BUTTON_PRESS)   tp_button_handle_event(t, BUTTON_EVENT_PRESS, stamp);
                                    }
                                }
                                                void tp_edge_scroll_set_timer(tp_touch& t, time stamp)
                                                {
                                                    if (tp.buttons.click_method != LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS) // If we use software buttons, we disable timeout-based edge scrolling. A finger resting on the button areas is likely there to trigger a button event.
                                                    {
                                                        t.scroll.timer->start(stamp + lixx::default_scroll_lock_timeout);
                                                    }
                                                }
                                            void tp_edge_scroll_set_state(tp_touch& t, tp_edge_scroll_touch_state state, time stamp)
                                            {
                                                t.scroll.timer->cancel();
                                                t.scroll.edge_state = state;
                                                switch (state)
                                                {
                                                    case EDGE_SCROLL_TOUCH_STATE_NONE:     t.scroll.edge = EDGE_NONE; break;
                                                    case EDGE_SCROLL_TOUCH_STATE_AREA:     t.scroll.edge = EDGE_NONE; break;
                                                    case EDGE_SCROLL_TOUCH_STATE_EDGE_NEW: t.scroll.edge = tp_touch_get_edge(t); t.scroll.initial = t.point; tp_edge_scroll_set_timer(t, stamp); break;
                                                    case EDGE_SCROLL_TOUCH_STATE_EDGE: break;
                                                }
                                            }
                                        void tp_edge_scroll_handle_none(tp_touch& t, scroll_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case SCROLL_EVENT_TOUCH:
                                                    if (tp_touch_get_edge(t)) tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_EDGE_NEW, stamp);
                                                    else                          tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_AREA, stamp);
                                                    break;
                                                case SCROLL_EVENT_MOTION:
                                                case SCROLL_EVENT_RELEASE:
                                                case SCROLL_EVENT_TIMEOUT:
                                                case SCROLL_EVENT_POSTED:
                                                    log("edge-scroll: touch %d%: unexpected scroll event %d% in none state", t.index, event);
                                                    break;
                                            }
                                        }
                                        void tp_edge_scroll_handle_edge_new(tp_touch& t, scroll_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case SCROLL_EVENT_TOUCH:
                                                    log("edge-scroll: touch %d%: unexpected scroll event %d% in edge new state", t.index, event);
                                                    break;
                                                case SCROLL_EVENT_MOTION:
                                                    t.scroll.edge &= tp_touch_get_edge(t);
                                                    if (!t.scroll.edge) tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_AREA, stamp);
                                                    break;
                                                case SCROLL_EVENT_RELEASE:
                                                    tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_NONE, stamp);
                                                    break;
                                                case SCROLL_EVENT_TIMEOUT:
                                                case SCROLL_EVENT_POSTED:
                                                    tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_EDGE, stamp);
                                                    break;
                                            }
                                        }
                                        void tp_edge_scroll_handle_edge(tp_touch& t, scroll_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case SCROLL_EVENT_TOUCH:
                                                case SCROLL_EVENT_TIMEOUT:
                                                    log("edge-scroll: touch %d%: unexpected scroll event %d% in edge state", t.index, event);
                                                    break;
                                                case SCROLL_EVENT_MOTION:
                                                    if (t.scroll.edge == (EDGE_RIGHT | EDGE_BOTTOM)) // If started at the bottom right, decide in which dir to scroll.
                                                    {
                                                        t.scroll.edge &= tp_touch_get_edge(t);
                                                        if (!t.scroll.edge) tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_AREA, stamp);
                                                    }
                                                    break;
                                                case SCROLL_EVENT_RELEASE:
                                                    tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_NONE, stamp);
                                                    break;
                                                case SCROLL_EVENT_POSTED:
                                                    break;
                                            }
                                        }
                                        void tp_edge_scroll_handle_area(tp_touch& t, scroll_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case SCROLL_EVENT_TOUCH:
                                                case SCROLL_EVENT_TIMEOUT:
                                                case SCROLL_EVENT_POSTED: log("unexpected scroll event %d% in area state", event); break;
                                                case SCROLL_EVENT_MOTION: break;
                                                case SCROLL_EVENT_RELEASE: tp_edge_scroll_set_state(t, EDGE_SCROLL_TOUCH_STATE_NONE, stamp); break;
                                            }
                                        }
                                        const char* edge_state_to_str(tp_edge_scroll_touch_state state)
                                        {
                                            switch (state)
                                            {
                                                CASE_RETURN_STRING(EDGE_SCROLL_TOUCH_STATE_NONE);
                                                CASE_RETURN_STRING(EDGE_SCROLL_TOUCH_STATE_EDGE_NEW);
                                                CASE_RETURN_STRING(EDGE_SCROLL_TOUCH_STATE_EDGE);
                                                CASE_RETURN_STRING(EDGE_SCROLL_TOUCH_STATE_AREA);
                                            }
                                            return nullptr;
                                        }
                                        char const* edge_event_to_str(scroll_event event)
                                        {
                                            switch (event)
                                            {
                                                CASE_RETURN_STRING(SCROLL_EVENT_TOUCH);
                                                CASE_RETURN_STRING(SCROLL_EVENT_MOTION);
                                                CASE_RETURN_STRING(SCROLL_EVENT_RELEASE);
                                                CASE_RETURN_STRING(SCROLL_EVENT_TIMEOUT);
                                                CASE_RETURN_STRING(SCROLL_EVENT_POSTED);
                                            }
                                            return nullptr;
                                        }
                                    void tp_edge_scroll_handle_event(tp_touch& t, scroll_event event, time stamp)
                                    {
                                        auto current = t.scroll.edge_state;
                                        switch (current)
                                        {
                                            case EDGE_SCROLL_TOUCH_STATE_NONE:     tp_edge_scroll_handle_none(    t, event, stamp); break;
                                            case EDGE_SCROLL_TOUCH_STATE_EDGE_NEW: tp_edge_scroll_handle_edge_new(t, event, stamp); break;
                                            case EDGE_SCROLL_TOUCH_STATE_EDGE:     tp_edge_scroll_handle_edge(    t, event, stamp); break;
                                            case EDGE_SCROLL_TOUCH_STATE_AREA:     tp_edge_scroll_handle_area(    t, event, stamp); break;
                                        }
                                        if (current != t.scroll.edge_state)
                                        {
                                            log("edge-scroll: touch %d% state %s% → %s% → %s%", t.index, edge_state_to_str(current), edge_event_to_str(event), edge_state_to_str(t.scroll.edge_state));
                                        }
                                    }
                                void tp_edge_scroll_handle_state(time stamp)
                                {
                                    if (tp.tp_scroll.method != LIBINPUT_CONFIG_SCROLL_EDGE)
                                    {
                                        for (auto& t : tp.touches)
                                        {
                                                 if (t.state == TOUCH_BEGIN) t.scroll.edge_state = EDGE_SCROLL_TOUCH_STATE_AREA;
                                            else if (t.state == TOUCH_END)   t.scroll.edge_state = EDGE_SCROLL_TOUCH_STATE_NONE;
                                        }
                                        return;
                                    }
                                    for (auto& t : tp.touches)
                                    {
                                        if (t.dirty)
                                        {
                                            switch (t.state)
                                            {
                                                case TOUCH_NONE:
                                                case TOUCH_HOVERING: break;
                                                case TOUCH_MAYBE_END: // This shouldn't happen we transfer to TOUCH_END before processing state.
                                                    log("touch %d%: unexpected state %d%", t.index, t.state);
                                                    [[fallthrough]];
                                                case TOUCH_END:      tp_edge_scroll_handle_event(t, SCROLL_EVENT_RELEASE, stamp); break;
                                                case TOUCH_BEGIN:    tp_edge_scroll_handle_event(t, SCROLL_EVENT_TOUCH, stamp); break;
                                                case TOUCH_UPDATE:   tp_edge_scroll_handle_event(t, SCROLL_EVENT_MOTION, stamp); break;
                                            }
                                        }
                                    }
                                }
                                void tp_pin_fingers()
                                {
                                    for (auto& t : tp.touches)
                                    {
                                        t.pinned_state = true;
                                        t.pinned_center = t.point;
                                    }
                                }
                                        bool tp_thumb_ignored_for_gesture(tp_touch& t)
                                        {
                                            return tp.thumb.detect_thumbs && tp.thumb.index == t.index
                                               && (tp.thumb.state == THUMB_STATE_JAILED
                                                || tp.thumb.state == THUMB_STATE_SUPPRESSED
                                                || tp.thumb.state == THUMB_STATE_REVIVED_JAILED
                                                || tp.thumb.state == THUMB_STATE_DEAD);
                                        }
                                    bool tp_touch_active_for_gesture(tp_touch& t)
                                    {
                                        return (t.state == TOUCH_BEGIN || t.state == TOUCH_UPDATE)
                                             && t.palm_state == TOUCH_PALM_NONE
                                             && !t.pinned_state
                                             && !tp_thumb_ignored_for_gesture(t)
                                             && tp_button_touch_active(t)
                                             && tp_edge_scroll_touch_active(t);
                                    }
                                                            bool tp_gesture_is_quick_hold()
                                                            {
                                                                // When 1 or 2 fingers are used to hold, always use a "quick" hold to make the hold to stop kinetic scrolling user interaction feel more natural.
                                                                return (tp.gesture.finger_count == 1) || (tp.gesture.finger_count == 2);
                                                            }
                                                        bool tp_gesture_use_hold_timer()
                                                        {
                                                            // When tap is not enabled, always use the timer.
                                                            if (!tp.tap.tap_state_enabled) return true;
                                                            // Always use the timer if it is a quick hold.
                                                            if (tp_gesture_is_quick_hold()) return true;
                                                            // If the number of fingers on the touchpad exceeds the number of allowed fingers to tap, use the timer.
                                                            if (tp.gesture.finger_count > 3) return true;
                                                            // If the tap state machine is already in a hold status, for example when holding with 3 fingers and then holding with 2, use the timer.
                                                            if (tp.tap.state == TAP_STATE_HOLD
                                                             || tp.tap.state == TAP_STATE_TOUCH_2_HOLD
                                                             || tp.tap.state == TAP_STATE_TOUCH_3_HOLD)
                                                            {
                                                                return true;
                                                            }
                                                            // If the tap state machine is in dead status, use the timer. This happens when the user holds after cancelling a gesture/scroll.
                                                            if (tp.tap.state == TAP_STATE_DEAD) return true;
                                                            // Otherwise, sync the hold notification with the tap state machine.
                                                            return faux;
                                                        }
                                                    void tp_gesture_set_hold_timer(time stamp)
                                                    {
                                                        if (tp.gesture.hold_enabled)
                                                        {
                                                            if (tp_gesture_use_hold_timer())
                                                            {
                                                                auto timeout = tp_gesture_is_quick_hold() ? lixx::quick_gesture_hold_timeout : lixx::default_gesture_hold_timeout;
                                                                tp.gesture.hold_timer->start(stamp + timeout);
                                                            }
                                                        }
                                                    }
                                                void tp_gesture_handle_event_on_state_none(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL: tp.gesture.hold_timer->cancel(); break;
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                            // Note: this makes 3fg drag more responsive but disables 3fg pinch/hold. Those are niche enough to not worry about for now.
                                                            if (!tp.tap.tap_state_enabled && tp.drag_3fg.nfingers == tp.gesture.finger_count)
                                                            {
                                                                tp.gesture.state = GESTURE_STATE_3FG_DRAG_START;
                                                            }
                                                            else
                                                            {
                                                                tp_gesture_set_hold_timer(stamp);
                                                                tp.gesture.state = GESTURE_STATE_UNKNOWN;
                                                            }
                                                            break;
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT: break;
                                                        case GESTURE_EVENT_POINTER_MOTION_START: tp.gesture.state = GESTURE_STATE_POINTER_MOTION; break;
                                                        case GESTURE_EVENT_SCROLL_START:         tp.gesture.state = GESTURE_STATE_SCROLL_START; break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log gesture bug: tp ", (ui32)event); break;
                                                    }
                                                }
                                                        void gesture_notify(time stamp, libinput_event_type type, si32 finger_count, bool cancelled, fp64_coor delta, fp64_coor unaccel, fp64 scale, fp64 angle)
                                                        {
                                                            if (tp.device_has_cap(LIBINPUT_DEVICE_CAP_GESTURE))
                                                            {
                                                                auto& gesture_event = tp.li.libinput_emplace_event<libinput_event_gesture>();
                                                                gesture_event.finger_count  = finger_count;
                                                                gesture_event.cancelled     = cancelled;
                                                                gesture_event.delta         = delta;
                                                                gesture_event.delta_unaccel = unaccel;
                                                                gesture_event.scale         = scale;
                                                                gesture_event.angle         = angle;
                                                                tp.post_device_event(stamp, type, gesture_event);
                                                            }
                                                        }
                                                    void gesture_notify_hold_begin(time stamp, si32 finger_count)
                                                    {
                                                        gesture_notify(stamp, LIBINPUT_EVENT_GESTURE_HOLD_BEGIN, finger_count, 0, lixx::zero_coor, lixx::zero_coor, 0.0, 0.0);
                                                    }
                                                        fp64_coor device_float_average(fp64_coor a, fp64_coor b)
                                                        {
                                                            return (a + b) / 2.0;
                                                        }
                                                        fp64_coor tp_normalize_delta(fp64_coor delta)
                                                        {
                                                            return delta * tp.accel_scale_coeff.x;
                                                        }
                                                    void tp_gesture_set_scroll_buildup()
                                                    {
                                                        auto first = tp.gesture.two_touches[0];
                                                        auto second = tp.gesture.two_touches[1];
                                                        auto d0 = first->point - first->gesture_origin;
                                                        auto d1 = second->point - second->gesture_origin;
                                                        auto average = device_float_average(d0, d1);
                                                        tp.scroll.buildup = tp_normalize_delta(average);
                                                    }
                                                        void tp_gesture_get_pinch_info(fp64& distance, fp64& angle, fp64_coor& center)
                                                        {
                                                            auto first = tp.gesture.two_touches[0];
                                                            auto second = tp.gesture.two_touches[1];
                                                            auto delta = first->point - second->point;
                                                            auto normalized = tp_normalize_delta(delta);
                                                            distance = normalized.hypot();
                                                            angle = netxs::rad2deg(std::atan2(normalized.y, normalized.x));
                                                            center = fp64_coor{ first->point + second->point } / 2.0;
                                                        }
                                                    void tp_gesture_init_pinch()
                                                    {
                                                        tp_gesture_get_pinch_info(tp.gesture.initial_distance, tp.gesture.angle, tp.gesture.center);
                                                        tp.gesture.prev_scale = 1.0;
                                                    }
                                                    void tp_gesture_init_3fg_drag()
                                                    {
                                                        //not implemented
                                                    }
                                                void tp_gesture_handle_event_on_state_unknown(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                            break;
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                            tp.gesture.state = GESTURE_STATE_HOLD;
                                                            gesture_notify_hold_begin(stamp, tp.gesture.finger_count);
                                                            break;
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                            // Don't cancel the hold timer. This pointer motion can end up being recognised as hold and motion.
                                                            tp.gesture.state = GESTURE_STATE_POINTER_MOTION;
                                                            break;
                                                        case GESTURE_EVENT_SCROLL_START:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp_gesture_set_scroll_buildup();
                                                            tp.gesture.state = GESTURE_STATE_SCROLL_START;
                                                            break;
                                                        case GESTURE_EVENT_SWIPE_START:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_SWIPE_START;
                                                            break;
                                                        case GESTURE_EVENT_PINCH_START:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp_gesture_init_pinch();
                                                            tp.gesture.state = GESTURE_STATE_PINCH_START;
                                                            break;
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp_gesture_init_3fg_drag();
                                                            tp.gesture.state = GESTURE_STATE_3FG_DRAG_START;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT:
                                                            log("log_gesture_bug: tp", (ui32)event);
                                                            break;
                                                    }
                                                }
                                                    void gesture_notify_hold_end(time stamp, si32 finger_count, bool cancelled)
                                                    {
                                                        gesture_notify(stamp, LIBINPUT_EVENT_GESTURE_HOLD_END, finger_count, cancelled, lixx::zero_coor, lixx::zero_coor, 0, 0.0);
                                                    }
                                                        void tp_gesture_end(time stamp, bool gesture_cancelled)
                                                        {
                                                            switch (tp.gesture.state)
                                                            {
                                                                case GESTURE_STATE_NONE:
                                                                case GESTURE_STATE_UNKNOWN:
                                                                case GESTURE_STATE_SCROLL_START:
                                                                case GESTURE_STATE_PINCH_START:
                                                                case GESTURE_STATE_SWIPE_START:
                                                                case GESTURE_STATE_3FG_DRAG_START: tp_gesture_handle_event(GESTURE_EVENT_RESET, stamp); break;
                                                                case GESTURE_STATE_HOLD:
                                                                case GESTURE_STATE_HOLD_AND_MOTION:
                                                                case GESTURE_STATE_POINTER_MOTION:
                                                                case GESTURE_STATE_SCROLL:
                                                                case GESTURE_STATE_PINCH:
                                                                case GESTURE_STATE_SWIPE:
                                                                case GESTURE_STATE_3FG_DRAG:
                                                                case GESTURE_STATE_3FG_DRAG_RELEASED:
                                                                    if (gesture_cancelled) tp_gesture_handle_event(GESTURE_EVENT_CANCEL, stamp);
                                                                    else                   tp_gesture_handle_event(GESTURE_EVENT_END, stamp);
                                                                    break;
                                                                default: break;
                                                            }
                                                        }
                                                    void tp_gesture_cancel(time stamp)
                                                    {
                                                        tp_gesture_end(stamp, true);
                                                    }
                                                void tp_gesture_handle_event_on_state_hold(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                        {
                                                            auto cancelled = event == GESTURE_EVENT_CANCEL;
                                                            gesture_notify_hold_end(stamp, tp.gesture.finger_count, cancelled);
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        }
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                            tp_gesture_cancel(stamp);
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                            tp.gesture.state = GESTURE_STATE_HOLD_AND_MOTION;
                                                            break;
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_POINTER_MOTION;
                                                            break;
                                                        case GESTURE_EVENT_SCROLL_START:
                                                            tp_gesture_set_scroll_buildup();
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_SCROLL_START;
                                                            break;
                                                        case GESTURE_EVENT_SWIPE_START:
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_SWIPE_START;
                                                            break;
                                                        case GESTURE_EVENT_PINCH_START:
                                                            tp_gesture_cancel(stamp);
                                                            tp_gesture_init_pinch();
                                                            tp.gesture.state = GESTURE_STATE_PINCH_START;
                                                            break;
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                            tp_gesture_cancel(stamp);
                                                            tp_gesture_init_3fg_drag();
                                                            tp.gesture.state = GESTURE_STATE_3FG_DRAG_START;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT:
                                                            log("log_gesture_bug: tp", (ui32)event);
                                                            break;
                                                    }
                                                }
                                                void tp_gesture_handle_event_on_state_hold_and_motion(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                        {
                                                            auto cancelled = event == GESTURE_EVENT_CANCEL;
                                                            gesture_notify_hold_end(stamp, tp.gesture.finger_count, cancelled);
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        }
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT: tp_gesture_cancel(stamp); break;
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_POINTER_MOTION;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                    fp64_coor tp_gesture_mm_moved(tp_touch& t)
                                                    {
                                                        auto delta = std::abs(t.point - t.gesture_origin);
                                                        return evdev_device_unit_delta_to_mm(delta);
                                                    }
                                                void tp_gesture_handle_event_on_state_pointer_motion(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                            if (tp.gesture.finger_count == 1)
                                                            {
                                                                auto first = tp.gesture.two_touches[0];
                                                                auto first_moved = tp_gesture_mm_moved(*first);
                                                                auto first_mm = first_moved.hypot();
                                                                if (first_mm < lixx::hold_and_motion_threshold)
                                                                {
                                                                    tp.gesture.state = GESTURE_STATE_HOLD_AND_MOTION;
                                                                    gesture_notify_hold_begin(stamp, tp.gesture.finger_count);
                                                                }
                                                            }
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                void tp_gesture_handle_event_on_state_scroll_start(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                            break;
                                                        case GESTURE_EVENT_PINCH_START:
                                                            tp_gesture_init_pinch();
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_PINCH_START;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                    void tp_gesture_stop_twofinger_scroll(time stamp)
                                                    {
                                                        if (tp.tp_scroll.method == LIBINPUT_CONFIG_SCROLL_2FG)
                                                        {
                                                            tp.evdev_stop_scroll(stamp, LIBINPUT_POINTER_AXIS_SOURCE_FINGER);
                                                        }
                                                    }
                                                void tp_gesture_handle_event_on_state_scroll(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp_gesture_stop_twofinger_scroll(stamp);
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                            tp_gesture_cancel(stamp);
                                                            break;
                                                        case GESTURE_EVENT_PINCH_START:
                                                            tp_gesture_init_pinch();
                                                            tp_gesture_cancel(stamp);
                                                            tp.gesture.state = GESTURE_STATE_PINCH_START;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                void tp_gesture_handle_event_on_state_pinch_start(gesture_event event)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL: tp.gesture.hold_timer->cancel(); break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT: break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                    void gesture_notify_pinch_end(time stamp, si32 finger_count, fp64 scale, bool cancelled)
                                                    {
                                                        gesture_notify(stamp, LIBINPUT_EVENT_GESTURE_PINCH_END, finger_count, cancelled, lixx::zero_coor, lixx::zero_coor, scale, 0.0);
                                                    }
                                                void tp_gesture_handle_event_on_state_pinch(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                        {
                                                            auto cancelled = event == GESTURE_EVENT_CANCEL;
                                                            gesture_notify_pinch_end(stamp, tp.gesture.finger_count, tp.gesture.prev_scale, cancelled);
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        }
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT: tp_gesture_cancel(stamp); break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                void tp_gesture_handle_event_on_state_swipe_start(gesture_event event)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT: break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                    void gesture_notify_swipe_end(time stamp, si32 finger_count, bool cancelled)
                                                    {
                                                        gesture_notify(stamp, LIBINPUT_EVENT_GESTURE_SWIPE_END, finger_count, cancelled, lixx::zero_coor, lixx::zero_coor, 0.0, 0.0);
                                                    }
                                                void tp_gesture_handle_event_on_state_swipe(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                        {
                                                            auto cancelled = event == GESTURE_EVENT_CANCEL;
                                                            gesture_notify_swipe_end(stamp, tp.gesture.finger_count, cancelled);
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        }
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT: tp_gesture_cancel(stamp); break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                void tp_gesture_handle_event_on_state_3fg_drag_start(gesture_event event)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET:
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                            tp.gesture.hold_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:  break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT:  log("log_gesture_bug: tp", (ui32)event);  break;
                                                    }
                                                }
                                                void tp_gesture_set_3fg_drag_timer(time stamp)
                                                {
                                                    tp.gesture.drag_3fg_release_time = stamp;
                                                    tp.gesture.drag_3fg_timer->start(stamp + 700ms);
                                                }
                                                void tp_gesture_handle_event_on_state_3fg_drag(gesture_event event, time stamp)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_CANCEL:
                                                            // If the gesture is cancelled we release the button immediately.
                                                            tp.evdev_pointer_notify_button(tp.gesture.drag_3fg_release_time, evdev::btn_left, evdev::released);
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_END:
                                                            // If the gesture ends we start the timer so we can keep dragging.
                                                            tp_gesture_set_3fg_drag_timer(stamp);
                                                            tp.gesture.state = GESTURE_STATE_3FG_DRAG_RELEASED;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                            if (tp.gesture.finger_count_pending < 2)
                                                            {
                                                                tp.evdev_pointer_notify_button(tp.gesture.drag_3fg_release_time, evdev::btn_left, evdev::released);
                                                                tp.gesture.state = GESTURE_STATE_NONE;
                                                            }
                                                            break;
                                                        case GESTURE_EVENT_TAP_TIMEOUT: break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_FINGER_DETECTED:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT:
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                        case GESTURE_EVENT_SCROLL_START:
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                    }
                                                }
                                                    void tp_gesture_stop_3fg_drag()
                                                    {
                                                        //not inplemented
                                                    }
                                                void tp_gesture_handle_event_on_state_3fg_drag_released(gesture_event event)
                                                {
                                                    switch (event)
                                                    {
                                                        case GESTURE_EVENT_RESET: log("log_gesture_bug: tp", (ui32)event); break;
                                                        case GESTURE_EVENT_END:
                                                        case GESTURE_EVENT_CANCEL:
                                                        case GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT:
                                                            tp_gesture_stop_3fg_drag();
                                                            tp.gesture.drag_3fg_timer->cancel();
                                                            tp.gesture.finger_count_switch_timer->cancel();
                                                            tp.evdev_pointer_notify_button(tp.gesture.drag_3fg_release_time, evdev::btn_left, evdev::released);
                                                            tp.gesture.state = GESTURE_STATE_NONE;
                                                            break;
                                                        case GESTURE_EVENT_FINGER_SWITCH_TIMEOUT:
                                                        case GESTURE_EVENT_TAP_TIMEOUT:
                                                            if (tp.gesture.finger_count_pending == tp.drag_3fg.nfingers)
                                                            {
                                                                tp.gesture.drag_3fg_timer->cancel();
                                                                tp.gesture.state = GESTURE_STATE_3FG_DRAG;
                                                            }
                                                            break;
                                                        case GESTURE_EVENT_FINGER_DETECTED: break;
                                                        case GESTURE_EVENT_POINTER_MOTION_START:
                                                            tp_gesture_stop_3fg_drag();
                                                            tp.gesture.drag_3fg_timer->cancel();
                                                            tp.evdev_pointer_notify_button(tp.gesture.drag_3fg_release_time, evdev::btn_left, evdev::released);
                                                            tp.gesture.state = GESTURE_STATE_POINTER_MOTION;
                                                            break;
                                                        case GESTURE_EVENT_HOLD_AND_MOTION_START:
                                                        case GESTURE_EVENT_HOLD_TIMEOUT: log("log_gesture_bug: tp", (ui32)event); break;
                                                        // Anything that's detected as gesture in this state will be continue the current 3fg drag gesture.
                                                        case GESTURE_EVENT_SCROLL_START:
                                                            tp.gesture.drag_3fg_timer->cancel();
                                                            tp.evdev_pointer_notify_button(tp.gesture.drag_3fg_release_time, evdev::btn_left, evdev::released);
                                                            tp.gesture.state = GESTURE_STATE_SCROLL_START;
                                                            break;
                                                        case GESTURE_EVENT_SWIPE_START:
                                                        case GESTURE_EVENT_PINCH_START:
                                                        case GESTURE_EVENT_3FG_DRAG_START:
                                                            tp.gesture.drag_3fg_timer->cancel();
                                                            tp.gesture.state = GESTURE_STATE_3FG_DRAG;
                                                            break;
                                                    }
                                                }
                                            char const* gesture_state_to_str(tp_gesture_state state)
                                            {
                                                switch (state)
                                                {
                                                    CASE_RETURN_STRING(GESTURE_STATE_NONE);
                                                    CASE_RETURN_STRING(GESTURE_STATE_UNKNOWN);
                                                    CASE_RETURN_STRING(GESTURE_STATE_HOLD);
                                                    CASE_RETURN_STRING(GESTURE_STATE_HOLD_AND_MOTION);
                                                    CASE_RETURN_STRING(GESTURE_STATE_POINTER_MOTION);
                                                    CASE_RETURN_STRING(GESTURE_STATE_SCROLL_START);
                                                    CASE_RETURN_STRING(GESTURE_STATE_SCROLL);
                                                    CASE_RETURN_STRING(GESTURE_STATE_PINCH_START);
                                                    CASE_RETURN_STRING(GESTURE_STATE_PINCH);
                                                    CASE_RETURN_STRING(GESTURE_STATE_SWIPE_START);
                                                    CASE_RETURN_STRING(GESTURE_STATE_SWIPE);
                                                    CASE_RETURN_STRING(GESTURE_STATE_3FG_DRAG_START);
                                                    CASE_RETURN_STRING(GESTURE_STATE_3FG_DRAG);
                                                    CASE_RETURN_STRING(GESTURE_STATE_3FG_DRAG_RELEASED);
                                                    default: break;
                                                }
                                                return nullptr;
                                            }
                                            char const* gesture_event_to_str(gesture_event event)
                                            {
                                                switch (event)
                                                {
                                                    CASE_RETURN_STRING(GESTURE_EVENT_RESET);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_END);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_CANCEL);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_FINGER_DETECTED);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_FINGER_SWITCH_TIMEOUT);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_TAP_TIMEOUT);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_HOLD_TIMEOUT);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_HOLD_AND_MOTION_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_POINTER_MOTION_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_SCROLL_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_SWIPE_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_PINCH_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_3FG_DRAG_START);
                                                    CASE_RETURN_STRING(GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT);
                                                }
                                                return nullptr;
                                            }
                                        void tp_gesture_handle_event(gesture_event event, time stamp)
                                        {
                                            auto oldstate = tp.gesture.state;
                                            switch (tp.gesture.state)
                                            {
                                                case GESTURE_STATE_NONE:              tp_gesture_handle_event_on_state_none(             event, stamp); break;
                                                case GESTURE_STATE_UNKNOWN:           tp_gesture_handle_event_on_state_unknown(          event, stamp); break;
                                                case GESTURE_STATE_HOLD:              tp_gesture_handle_event_on_state_hold(             event, stamp); break;
                                                case GESTURE_STATE_HOLD_AND_MOTION:   tp_gesture_handle_event_on_state_hold_and_motion(  event, stamp); break;
                                                case GESTURE_STATE_POINTER_MOTION:    tp_gesture_handle_event_on_state_pointer_motion(   event, stamp); break;
                                                case GESTURE_STATE_SCROLL_START:      tp_gesture_handle_event_on_state_scroll_start(     event, stamp); break;
                                                case GESTURE_STATE_SCROLL:            tp_gesture_handle_event_on_state_scroll(           event, stamp); break;
                                                case GESTURE_STATE_PINCH_START:       tp_gesture_handle_event_on_state_pinch_start(      event); break;
                                                case GESTURE_STATE_PINCH:             tp_gesture_handle_event_on_state_pinch(            event, stamp); break;
                                                case GESTURE_STATE_SWIPE_START:       tp_gesture_handle_event_on_state_swipe_start(      event); break;
                                                case GESTURE_STATE_SWIPE:             tp_gesture_handle_event_on_state_swipe(            event, stamp); break;
                                                case GESTURE_STATE_3FG_DRAG_START:    tp_gesture_handle_event_on_state_3fg_drag_start(   event); break;
                                                case GESTURE_STATE_3FG_DRAG:          tp_gesture_handle_event_on_state_3fg_drag(         event, stamp); break;
                                                case GESTURE_STATE_3FG_DRAG_RELEASED: tp_gesture_handle_event_on_state_3fg_drag_released(event); break;
                                                default: break;
                                            }
                                            if (oldstate != tp.gesture.state)
                                            {
                                                log("gesture0: [%d%fg] event %s% → %s% → %s%", tp.gesture.finger_count, gesture_state_to_str(oldstate), gesture_event_to_str(event), gesture_state_to_str(tp.gesture.state));
                                            }
                                        }
                                    void tp_gesture_stop(time stamp)
                                    {
                                        tp_gesture_end(stamp, faux);
                                    }
                                    bool tp_gesture_debounce_finger_changes()
                                    {
                                        switch (tp.gesture.state)
                                        {
                                            case GESTURE_STATE_NONE:
                                            case GESTURE_STATE_UNKNOWN:
                                            case GESTURE_STATE_SCROLL_START:
                                            case GESTURE_STATE_PINCH_START:
                                            case GESTURE_STATE_SWIPE_START:
                                            case GESTURE_STATE_POINTER_MOTION:
                                            default:
                                                return faux;
                                            case GESTURE_STATE_HOLD:
                                            case GESTURE_STATE_HOLD_AND_MOTION:
                                            case GESTURE_STATE_SCROLL:
                                            case GESTURE_STATE_PINCH:
                                            case GESTURE_STATE_SWIPE:
                                            case GESTURE_STATE_3FG_DRAG_START:
                                            case GESTURE_STATE_3FG_DRAG_RELEASED:
                                            case GESTURE_STATE_3FG_DRAG:
                                                return true;
                                        }
                                        ::abort();
                                    }
                                void tp_gesture_update_finger_state(time stamp)
                                {
                                    auto active_touches = 0u;
                                    for (auto& t : tp.touches)
                                    {
                                        if (tp_touch_active_for_gesture(t))
                                        {
                                            active_touches++;
                                        }
                                    }
                                    if (active_touches != tp.gesture.finger_count)
                                    {
                                        // If all fingers are lifted immediately end the gesture.
                                        if (active_touches == 0)
                                        {
                                            tp_gesture_stop(stamp);
                                            tp.gesture.finger_count = 0;
                                            tp.gesture.finger_count_pending = 0;
                                        // Immediately switch to new mode to avoid initial latency.
                                        }
                                        else if (!tp_gesture_debounce_finger_changes())
                                        {
                                            tp.gesture.finger_count = active_touches;
                                            tp.gesture.finger_count_pending = 0;
                                            // If in UNKNOWN or POINTER_MOTION state, go back to NONE to re-evaluate leftmost and rightmost touches
                                            if (tp.gesture.state == GESTURE_STATE_UNKNOWN || tp.gesture.state == GESTURE_STATE_POINTER_MOTION)
                                            {
                                                tp_gesture_handle_event(GESTURE_EVENT_RESET, stamp);
                                            }
                                        }
                                        else if (active_touches != tp.gesture.finger_count_pending) // else debounce finger changes.
                                        {
                                            tp.gesture.finger_count_pending = active_touches;
                                            tp.gesture.finger_count_switch_timer->start(stamp + lixx::default_gesture_switch_timeout);
                                        }
                                    }
                                    else
                                    {
                                        tp.gesture.finger_count_pending = 0;
                                    }
                                }
                            void tp_process_state(time stamp)
                            {
                                auto restart_filter = faux;
                                auto have_new_touch = faux;
                                auto speed_exceeded_count = 0u;
                                tp_position_fake_touches();
                                auto want_motion_reset = tp_need_motion_history_reset();
                                for (auto& t : tp.touches)
                                {
                                    if (t.state == TOUCH_NONE) continue;
                                    if (want_motion_reset)
                                    {
                                        tp_motion_history_reset(t);
                                        t.quirks_reset_motion_history = true;
                                    }
                                    else if (t.quirks_reset_motion_history)
                                    {
                                        tp_motion_history_reset(t);
                                        t.quirks_reset_motion_history = faux;
                                    }
                                    if (!t.dirty)
                                    {
                                        // A non-dirty touch must be below the speed limit.
                                        if (t.speed_exceeded_count > 0) t.speed_exceeded_count--;
                                        speed_exceeded_count = std::max(speed_exceeded_count, t.speed_exceeded_count);
                                        // A touch that hasn't moved must be in the same position, so let's add this to the motion history.
                                        tp_motion_history_push(t, stamp);
                                        continue;
                                    }
                                    if (tp_detect_jumps(t, stamp))
                                    {
                                        if (!tp.semi_mt) log("Touch jump detected and discarded");
                                        tp_motion_history_reset(t);
                                    }
                                    tp_thumb_update_touch(t);
                                    tp_palm_detect(t, stamp);
                                    tp_detect_wobbling(t, stamp);
                                    tp_motion_hysteresis(t);
                                    tp_motion_history_push(t, stamp);
                                    // Touch speed handling: if we'are above the threshold,
                                    // count each event that we're over the threshold up to 10
                                    // events. Count down when we are below the speed.
                                    //
                                    // Take the touch with the highest speed excess, if it is
                                    // above a certain threshold (5, see below), assume a
                                    // dropped finger is a thumb.
                                    //
                                    // Yes, this relies on the touchpad to keep sending us
                                    // events even if the finger doesn't move, otherwise we
                                    // never count down. Let's see how far we get with that.
                                    if (t.speed_last > lixx::thumb_ignore_speed_threshold)
                                    {
                                        if (t.speed_exceeded_count < 15)
                                        {
                                            t.speed_exceeded_count++;
                                        }
                                    }
                                    else if (t.speed_exceeded_count > 0)
                                    {
                                        t.speed_exceeded_count--;
                                    }
                                    speed_exceeded_count = std::max(speed_exceeded_count, t.speed_exceeded_count);
                                    tp_calculate_motion_speed(t, stamp);
                                    tp_unpin_finger(t);
                                    if (t.state == TOUCH_BEGIN)
                                    {
                                        have_new_touch = true;
                                        restart_filter = true;
                                    }
                                }
                                if (tp.thumb.detect_thumbs && have_new_touch && tp.nfingers_down >= 2)
                                {
                                    tp_thumb_update_multifinger();
                                }
                                if (restart_filter) filter_restart(tp.pointer_filter, stamp);
                                tp_button_handle_state(stamp);
                                tp_edge_scroll_handle_state(stamp);
                                // We have a physical button down event on a clickpad. To avoid spurious pointer moves by the clicking finger we pin all fingers.
                                // We unpin fingers when they move more then a certain threshold to to allow drag and drop.
                                if (tp.queued & TOUCHPAD_EVENT_BUTTON_PRESS && tp.buttons.is_clickpad)
                                {
                                    tp_pin_fingers();
                                }
                                tp_gesture_update_finger_state(stamp);
                            }
                                void tp_thumb_reset()
                                {
                                    tp.thumb.state = THUMB_STATE_FINGER;
                                    tp.thumb.index = ui32max;
                                    tp.thumb.pinch_eligible = true;
                                }
                                    void tp_tap_update_map()
                                    {
                                        if (tp.tap.state == TAP_STATE_IDLE && tp.tap.use_lmr_map != tp.tap.want_use_lmr_map)
                                        {
                                            tp.tap.use_lmr_map = tp.tap.want_use_lmr_map;
                                        }
                                    }
                                void tp_tap_post_process_state()
                                {
                                    tp_tap_update_map();
                                }
                                    void tp_button_update_clickfinger_map()
                                    {
                                        if (tp.buttons.state == BUTTON_STATE_NONE)
                                        {
                                            if (tp.buttons.use_lmr_map != tp.buttons.want_use_lmr_map) tp.buttons.use_lmr_map = tp.buttons.want_use_lmr_map;
                                        }
                                    }
                                void tp_button_post_process_state()
                                {
                                    tp_button_update_clickfinger_map();
                                }
                            void tp_post_process_state()
                            {
                                for (auto& t : tp.touches)
                                {
                                    if (t.dirty)
                                    {
                                        if (t.state == TOUCH_END)
                                        {
                                            t.state = t.has_ended ? TOUCH_NONE : TOUCH_HOVERING;
                                        }
                                        else if (t.state == TOUCH_BEGIN)
                                        {
                                            t.state = TOUCH_UPDATE;
                                        }
                                        t.dirty = faux;
                                    }
                                }
                                tp.old_nfingers_down = tp.nfingers_down;
                                tp.buttons.old_state = tp.buttons.state;
                                tp.queued = TOUCHPAD_EVENT_NONE;
                                if (tp.nfingers_down == 0) tp_thumb_reset();
                                tp_tap_post_process_state();
                                tp_button_post_process_state();
                            }
                                            void tp_edge_scroll_stop_events(time stamp)
                                            {
                                                for (auto& t : tp.touches)
                                                {
                                                    if (t.scroll.direction != -1)
                                                    {
                                                        tp.evdev_notify_axis_finger(stamp, 1ul << t.scroll.direction, lixx::zero_coor);
                                                        t.scroll.direction = -1;
                                                        t.scroll.edge = EDGE_NONE; // Reset touch to area state, avoids loading the state machine with special case handling.
                                                        t.scroll.edge_state = EDGE_SCROLL_TOUCH_STATE_AREA;
                                                    }
                                                }
                                            }
                                                bool tp_clickfinger_within_distance(tp_touch& t1, tp_touch& t2)
                                                {
                                                    auto real_touch = t1.tp && t2.tp;
                                                    if (!real_touch || tp_thumb_ignored(t1) || tp_thumb_ignored(t2)) return 0;
                                                    auto x = (fp64)std::abs(t1.point.x - t2.point.x);
                                                    auto y = (fp64)std::abs(t1.point.y - t2.point.y);
                                                    auto xres = tp.ud_device.abs.absinfo_x.resolution;
                                                    auto yres = tp.ud_device.abs.absinfo_y.resolution;
                                                    x /= xres;
                                                    y /= yres;
                                                    auto within_distance = x <= 40 && y <= 30;
                                                    if (within_distance) // Maximum horiz spread is 40mm horiz, 30mm vert, anything wider than that is probably a gesture.
                                                    {
                                                        if (y > 20) // If y spread is <= 20mm, they're definitely together.
                                                        {
                                                            // If they're vertically spread between 20-40mm, they're not together if:
                                                            // - the touchpad's vertical size is >50mm, anything smaller is unlikely to have a thumb resting on it
                                                            // - and one of the touches is in the bottom 20mm of the touchpad and the other one isn't.
                                                            if (tp.ud_device.abs.dimensions.y / yres >= 50)
                                                            {
                                                                auto bottom_threshold = tp.ud_device.abs.absinfo_y.maximum - 20 * yres;
                                                                if ((t1.point.y > bottom_threshold) != (t2.point.y > bottom_threshold))
                                                                {
                                                                    within_distance = faux;
                                                                }
                                                            }
                                                        }
                                                    }
                                                    return within_distance;
                                                }
                                            ui32 tp_clickfinger_set_button()
                                            {
                                                auto button = ui32{};
                                                auto nfingers = 0u;
                                                auto first = (tp_touch*)nullptr;
                                                auto second = (tp_touch*)nullptr;
                                                for (auto& t : tp.touches)
                                                {
                                                    if (t.state != TOUCH_BEGIN && t.state != TOUCH_UPDATE) continue;
                                                    if (tp_thumb_ignored(t)) continue;
                                                    if (t.palm_state != TOUCH_PALM_NONE) continue;
                                                    nfingers++;
                                                         if (!first)  first  = &t;
                                                    else if (!second) second = &t;
                                                }
                                                if (nfingers == 2) // Only check for finger distance when there are 2 fingers on the touchpad.
                                                {
                                                    if (!tp_clickfinger_within_distance(*first, *second))
                                                    {
                                                        nfingers = 1;
                                                    }
                                                }
                                                nfingers = std::max(1u, nfingers);
                                                switch (nfingers)
                                                {
                                                    case 1:
                                                    case 2:
                                                    case 3:
                                                        button = lixx::tap_button_map[tp.buttons.use_lmr_map][nfingers - 1];
                                                        break;
                                                    default:
                                                        button = 0;
                                                        break;
                                                }
                                                return button;
                                            }
                                        si32 tp_notify_clickpadbutton(time stamp, ui32 button, ui32 is_topbutton, si32 state)
                                        {
                                            // If we've a trackpoint, send top buttons through the trackpoint.
                                            if (tp.buttons.trackpoint_li_device)
                                            {
                                                if (is_topbutton)
                                                {
                                                    auto event = evdev_event{ .usage = button,
                                                                              .value = state };
                                                    auto syn_report = evdev_event{ .usage = evdev::syn_report,
                                                                                   .value = 0 };
                                                    tp.buttons.trackpoint_li_device->process(event, stamp);
                                                    tp.buttons.trackpoint_li_device->process(syn_report, stamp);
                                                    return 1;
                                                }
                                                // Ignore button events not for the trackpoint while suspended.
                                                if (tp.is_suspended) return 0;
                                            }
                                            // A button click always terminates edge scrolling, even if we don't end up sending a button event.
                                            tp_edge_scroll_stop_events(stamp);
                                            // If the user has requested clickfinger replace the button chosen by the softbutton code with one based on the number of fingers.
                                            if (tp.buttons.click_method == LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER && state == evdev::pressed)
                                            {
                                                button = tp_clickfinger_set_button();
                                                tp.buttons.active = button;
                                                if (button == 0) return 0;
                                            }
                                            tp.evdev_pointer_notify_button(stamp, button, state);
                                            return 1;
                                        }
                                    si32 tp_post_clickpadbutton_buttons(time stamp)
                                    {
                                        auto button = ui32{};
                                        auto state = si32{};
                                        enum { AREA = 0x01, LEFT = 0x02, MIDDLE = 0x04, RIGHT = 0x08 };
                                        auto want_left_handed = true;
                                        auto current = tp.buttons.state;
                                        auto old = tp.buttons.old_state;
                                        auto is_top = 0u;
                                        if (!tp.buttons.click_pending && current == old) return 0;
                                        if (current)
                                        {
                                            auto area = 0u;
                                            if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_TOUCHPAD_PHANTOM_CLICKS) && tp.nactive_slots == 0)
                                            {
                                                // Some touchpads, notably those on the Dell XPS 15 9500, are prone to registering touchpad clicks when the case is sufficiently flexed. Ignore these by disregarding any clicks that are registered without touchpad touch.
                                                tp.buttons.click_pending = true;
                                                return 0;
                                            }
                                            for (auto& t : tp.touches)
                                            {
                                                switch (t.button.current)
                                                {
                                                    case BUTTON_EVENT_IN_AREA:     area |= AREA; break;
                                                    case BUTTON_EVENT_IN_TOP_L:    is_top = 1; [[fallthrough]];
                                                    case BUTTON_EVENT_IN_BOTTOM_L: area |= LEFT; break;
                                                    case BUTTON_EVENT_IN_TOP_M:    is_top = 1; [[fallthrough]];
                                                    case BUTTON_EVENT_IN_BOTTOM_M: area |= MIDDLE; break;
                                                    case BUTTON_EVENT_IN_TOP_R:    is_top = 1; [[fallthrough]];
                                                    case BUTTON_EVENT_IN_BOTTOM_R: area |= RIGHT; break;
                                                    default: break;
                                                }
                                            }
                                            if (area == 0 && tp.buttons.click_method != LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER)
                                            {
                                                // No touches, wait for a touch before processing.
                                                tp.buttons.click_pending = true;
                                                return 0;
                                            }
                                            if ((tp.middlebutton.enabled || is_top) && (area & LEFT) && (area & RIGHT))
                                            {
                                                button = evdev::btn_middle;
                                            }
                                            else if (area & MIDDLE) button = evdev::btn_middle;
                                            else if (area & RIGHT)  button = evdev::btn_right;
                                            else if (area & LEFT)   button = evdev::btn_left;
                                            else // Main or no area (for clickfinger) is always BTN_LEFT.
                                            {
                                                button = evdev::btn_left;
                                                want_left_handed = faux;
                                            }
                                            if (is_top) want_left_handed = faux;
                                            if (want_left_handed) button = tp.evdev_to_left_handed(button);
                                            tp.buttons.active = button;
                                            tp.buttons.active_is_topbutton = is_top;
                                            state = evdev::pressed;
                                        }
                                        else
                                        {
                                            button = tp.buttons.active;
                                            is_top = tp.buttons.active_is_topbutton;
                                            tp.buttons.active = {};
                                            tp.buttons.active_is_topbutton = 0;
                                            state = evdev::released;
                                        }
                                        tp.buttons.click_pending = faux;
                                        return button != 0 ? tp_notify_clickpadbutton(stamp, button, is_top, state) : 0;
                                    }
                                    si32 tp_post_physical_buttons(time stamp)
                                    {
                                        auto current = tp.buttons.state;
                                        auto old = tp.buttons.old_state;
                                        auto button = evdev::btn_left;
                                        while (current || old)
                                        {
                                            if ((current & 0x1) ^ (old & 0x1))
                                            {
                                                auto state = (current & 0x1) ? evdev::pressed : evdev::released;
                                                auto b = tp.evdev_to_left_handed(button);
                                                tp.evdev_pointer_notify_physical_button(stamp, b, state);
                                            }
                                            button++;
                                            current >>= 1;
                                            old >>= 1;
                                        }
                                        return 0;
                                    }
                                si32 tp_post_button_events(time stamp)
                                {
                                    auto clickpad = tp.buttons.is_clickpad || (tp.ud_device.model_flags & EVDEV_MODEL_APPLE_TOUCHPAD_ONEBUTTON);
                                    return clickpad ? tp_post_clickpadbutton_buttons(stamp)
                                                    : tp_post_physical_buttons(stamp);
                                }
                                    bool tp_tap_enabled()
                                    {
                                        return tp.tap.tap_state_enabled && !tp.tap.suspended;
                                    }
                                            void tp_tap_set_timer(time stamp)
                                            {
                                                tp.tap.timer->start(stamp + lixx::default_tap_timeout_period);
                                            }
                                        void tp_tap_idle_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE: break;
                                                case TAP_EVENT_TIMEOUT: break;
                                                case TAP_EVENT_MOTION:  log("log_tap_bug: tp", event);  break;
                                                case TAP_EVENT_THUMB:   log("log_tap_bug: tp", event);  break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_IDLE; break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                            void tp_tap_notify(time stamp, si32 nfingers, si32 state)
                                            {
                                                assert((si32)tp.tap.use_lmr_map < std::size(lixx::tap_button_map));
                                                if (nfingers < 1 || nfingers > 3) return;
                                                tp_gesture_cancel(stamp);
                                                auto button = lixx::tap_button_map[tp.tap.use_lmr_map][nfingers - 1];
                                                if (state == evdev::pressed) tp.tap.buttons_pressed |= (1ul << nfingers);
                                                else                         tp.tap.buttons_pressed &= ~(1ul << nfingers);
                                                tp.evdev_pointer_notify_button(stamp, button, state);
                                            }
                                            void tp_tap_set_drag_timer(time stamp, si32 nfingers_tapped)
                                            {
                                                tp.tap.timer->start(stamp + lixx::default_drag_timeout_period_base + (nfingers_tapped * lixx::default_drag_timeout_period_perfinger));
                                            }
                                                void tp_tap_clear_timer()
                                                {
                                                    tp.tap.timer->cancel();
                                                }
                                            void tp_tap_move_to_dead(tp_touch& t)
                                            {
                                                tp.tap.state = TAP_STATE_DEAD;
                                                t.tap.state = TAP_TOUCH_STATE_DEAD;
                                                tp_tap_clear_timer();
                                            }
                                            void tp_gesture_tap_timeout(time stamp)
                                            {
                                                if (tp.gesture.hold_enabled)
                                                {
                                                    if (!tp_gesture_is_quick_hold()) tp_gesture_handle_event(GESTURE_EVENT_TAP_TIMEOUT, stamp);
                                                }
                                            }
                                        void tp_tap_touch_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH_2;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp_tap_notify(tp.tap.press_stamp, 1, evdev::pressed);
                                                    if (tp.tap.drag_enabled)
                                                    {
                                                        tp.tap.state = TAP_STATE_1FGTAP_TAPPED;
                                                        tp.tap.release_stamp = stamp;
                                                        tp_tap_set_drag_timer(stamp, 1);
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(stamp, 1, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                    tp_tap_move_to_dead(t);
                                                    break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp.tap.state = TAP_STATE_HOLD;
                                                    tp_tap_clear_timer();
                                                    tp_gesture_tap_timeout(stamp);
                                                    break;
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    t.tap.is_thumb = true;
                                                    tp.tap.nfingers_down--;
                                                    t.tap.state = TAP_TOUCH_STATE_DEAD;
                                                    tp_tap_clear_timer();
                                                    break;
                                                case TAP_EVENT_PALM:
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    tp_tap_clear_timer();
                                                    break;
                                                case TAP_EVENT_PALM_UP:
                                                    break;
                                            }
                                        }
                                        void tp_tap_hold_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_PALM_UP:
                                                case TAP_EVENT_TIMEOUT: break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_IDLE; break;
                                                case TAP_EVENT_RELEASE: tp.tap.state = TAP_STATE_IDLE; break;
                                                case TAP_EVENT_MOTION:  tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH_2;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    t.tap.is_thumb = true;
                                                    tp.tap.nfingers_down--;
                                                    t.tap.state = TAP_TOUCH_STATE_DEAD;
                                                    break;
                                            }
                                        }
                                        void tp_tap_tapped_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_RELEASE: log("log_tap_bug: tp ", event); break;
                                                case TAP_EVENT_TOUCH:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP,
                                                        TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP,
                                                        TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                }
                                                case TAP_EVENT_TIMEOUT: tp.tap.state = TAP_STATE_IDLE; tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released); break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released); break;
                                                case TAP_EVENT_THUMB:   log("log_tap_bug: tp ", event); break;
                                                case TAP_EVENT_PALM:    log("log_tap_bug: tp ", event); break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch2_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH_3;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp.tap.state = TAP_STATE_TOUCH_2_RELEASE;
                                                    tp.tap.release_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp.tap.state = TAP_STATE_TOUCH_2_HOLD;
                                                    tp_gesture_tap_timeout(stamp);
                                                    break;
                                                case TAP_EVENT_MOTION:  tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_TOUCH; break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch2_hold_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH_3;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE: tp.tap.state = TAP_STATE_HOLD; break;
                                                case TAP_EVENT_MOTION:  tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_TIMEOUT: tp.tap.state = TAP_STATE_TOUCH_2_HOLD; break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_HOLD; break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch2_release_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_TOUCH_2_HOLD;
                                                    t.tap.state = TAP_TOUCH_STATE_DEAD;
                                                    tp_tap_clear_timer();
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp_tap_notify(tp.tap.press_stamp, 2, evdev::pressed);
                                                    if (tp.tap.drag_enabled)
                                                    {
                                                        tp.tap.state = TAP_STATE_2FGTAP_TAPPED;
                                                        tp_tap_set_drag_timer(stamp, 2);
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(tp.tap.release_stamp, 2, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                                case TAP_EVENT_MOTION:  tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_TIMEOUT: tp.tap.state = TAP_STATE_HOLD; break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM_UP: break;
                                                case TAP_EVENT_PALM:
                                                    // There's only one saved press time and it's overwritten by the last touch down. So in the case of finger down, palm down, finger up, palm detected, we use the palm touch's press time here instead of the finger's press time. Let's wait and see if that's an issue.
                                                    tp_tap_notify(tp.tap.press_stamp, 1, evdev::pressed);
                                                    if (tp.tap.drag_enabled) // For a single-finger tap the timer delay is the same as for the release of the finger that became a palm, no reset necessary.
                                                    {
                                                        tp.tap.state = TAP_STATE_1FGTAP_TAPPED;
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(tp.tap.release_stamp, 1, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                            }
                                        }
                                        void tp_tap_touch3_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:  tp.tap.state = TAP_STATE_DEAD; tp_tap_clear_timer(); break;
                                                case TAP_EVENT_MOTION: tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp.tap.state = TAP_STATE_TOUCH_3_HOLD;
                                                    tp_tap_clear_timer();
                                                    tp_gesture_tap_timeout(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp.tap.state = TAP_STATE_TOUCH_3_RELEASE;
                                                    tp.tap.release_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_TOUCH_2; break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch3_hold_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:   tp.tap.state = TAP_STATE_DEAD; tp_tap_set_timer(stamp); break;
                                                case TAP_EVENT_RELEASE: tp.tap.state = TAP_STATE_TOUCH_2_HOLD; break;
                                                case TAP_EVENT_MOTION:  tp_tap_move_to_dead(t); break;
                                                case TAP_EVENT_TIMEOUT: break;
                                                case TAP_EVENT_BUTTON:  tp.tap.state = TAP_STATE_DEAD; break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_TOUCH_2_HOLD; break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch3_release_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_TOUCH_3;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp.tap.state = TAP_STATE_TOUCH_3_RELEASE_2;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp_tap_move_to_dead(t);
                                                    break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_TOUCH_2_HOLD;
                                                    break;
                                                case TAP_EVENT_BUTTON:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    break;
                                                case TAP_EVENT_THUMB:   break;
                                                case TAP_EVENT_PALM:    tp.tap.state = TAP_STATE_TOUCH_2_RELEASE; break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_touch3_release2_handle_event(tp_touch& t, tap_event event, time stamp)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_TOUCH_2;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    if (tp.tap.drag_enabled)
                                                    {
                                                        tp.tap.state = TAP_STATE_3FGTAP_TAPPED;
                                                        tp_tap_set_drag_timer(stamp, 3);
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp_tap_move_to_dead(t);
                                                    break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_HOLD;
                                                    break;
                                                case TAP_EVENT_BUTTON:
                                                    tp_tap_notify(tp.tap.press_stamp, 3, evdev::pressed);
                                                    tp_tap_notify(tp.tap.release_stamp, 3, evdev::released);
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                    break;
                                                case TAP_EVENT_PALM:
                                                    tp_tap_notify(tp.tap.press_stamp, 2, evdev::pressed);
                                                    if (tp.tap.drag_enabled) // Resetting the timer to the appropriate delay for a two-finger tap would be ideal, but the timestamp of the last real finger release is lost, so the in-progress similar delay for release of the finger which became a palm instead will have to do.
                                                    {
                                                        tp.tap.state = TAP_STATE_2FGTAP_TAPPED;
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(tp.tap.release_stamp, 2, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                                case TAP_EVENT_PALM_UP:
                                                    break;
                                            }
                                        }
                                        void tp_tap_dragging_or_doubletap_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released);
                                                    tp.tap.state = TAP_STATE_TOUCH_2;
                                                    tp.tap.press_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp.tap.state = TAP_STATE_1FGTAP_TAPPED;
                                                    tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released);
                                                    tp_tap_notify(tp.tap.press_stamp, 1, evdev::pressed);
                                                    tp.tap.release_stamp = stamp;
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_TIMEOUT:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING,
                                                        TAP_STATE_2FGTAP_DRAGGING,
                                                        TAP_STATE_3FGTAP_DRAGGING,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                    break;
                                                case TAP_EVENT_PALM:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_TAPPED,
                                                        TAP_STATE_2FGTAP_TAPPED,
                                                        TAP_STATE_3FGTAP_TAPPED,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_PALM_UP:
                                                    break;
                                            }
                                        }
                                            void tp_tap_set_draglock_timer(time stamp)
                                            {
                                                tp.tap.timer->start(stamp + lixx::default_draglock_timeout_period);
                                            }
                                        void tp_tap_dragging_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING_2,
                                                        TAP_STATE_2FGTAP_DRAGGING_2,
                                                        TAP_STATE_3FGTAP_DRAGGING_2,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_RELEASE:
                                                    if (tp.tap.drag_lock != LIBINPUT_CONFIG_DRAG_LOCK_DISABLED)
                                                    {
                                                        tp_tap_state dest[3] =
                                                        {
                                                            TAP_STATE_1FGTAP_DRAGGING_WAIT,
                                                            TAP_STATE_2FGTAP_DRAGGING_WAIT,
                                                            TAP_STATE_3FGTAP_DRAGGING_WAIT,
                                                        };
                                                        assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                        tp.tap.state = dest[nfingers_tapped - 1];
                                                        if (tp.tap.drag_lock == LIBINPUT_CONFIG_DRAG_LOCK_ENABLED_TIMEOUT)
                                                        {
                                                            tp_tap_set_draglock_timer(stamp);
                                                        }
                                                    }
                                                    else
                                                    {
                                                        tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                        tp.tap.state = TAP_STATE_IDLE;
                                                    }
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_TIMEOUT:
                                                    // Noop.
                                                    break;
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                    break;
                                                case TAP_EVENT_PALM:
                                                    tp_tap_notify(tp.tap.release_stamp, nfingers_tapped, evdev::released);
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    break;
                                                case TAP_EVENT_PALM_UP:
                                                    break;
                                            }
                                        }
                                        void tp_tap_dragging_wait_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING_OR_TAP,
                                                        TAP_STATE_2FGTAP_DRAGGING_OR_TAP,
                                                        TAP_STATE_3FGTAP_DRAGGING_OR_TAP,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    tp_tap_set_timer(stamp);
                                                    break;
                                                }
                                                case TAP_EVENT_RELEASE:
                                                case TAP_EVENT_MOTION: log("log_tap_bug1: tp ", event); break;
                                                case TAP_EVENT_TIMEOUT:
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_THUMB:
                                                case TAP_EVENT_PALM: log("log_tap_bug2: tp ", event); break;
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_dragging_tap_handle_event(tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_TOUCH:
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    tp_tap_clear_timer();
                                                    tp_tap_move_to_dead(t);
                                                    break;
                                                case TAP_EVENT_RELEASE:
                                                    tp.tap.state = TAP_STATE_IDLE;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_TIMEOUT:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING,
                                                        TAP_STATE_2FGTAP_DRAGGING,
                                                        TAP_STATE_3FGTAP_DRAGGING,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_THUMB: break;
                                                case TAP_EVENT_PALM:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING_WAIT,
                                                        TAP_STATE_2FGTAP_DRAGGING_WAIT,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_dragging2_handle_event([[maybe_unused]] tp_touch& t, tap_event event, time stamp, si32 nfingers_tapped)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_RELEASE:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING,
                                                        TAP_STATE_2FGTAP_DRAGGING,
                                                        TAP_STATE_3FGTAP_DRAGGING,
                                                    };
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_TOUCH:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_TIMEOUT: break; // Noop.
                                                case TAP_EVENT_BUTTON:
                                                    tp.tap.state = TAP_STATE_DEAD;
                                                    tp_tap_notify(stamp, nfingers_tapped, evdev::released);
                                                    break;
                                                case TAP_EVENT_THUMB: break;
                                                case TAP_EVENT_PALM:
                                                {
                                                    tp_tap_state dest[3] =
                                                    {
                                                        TAP_STATE_1FGTAP_DRAGGING,
                                                        TAP_STATE_2FGTAP_DRAGGING, }; // Noop.
                                                    assert(nfingers_tapped >= 1 && nfingers_tapped <= 3);
                                                    tp.tap.state = dest[nfingers_tapped - 1];
                                                    break;
                                                }
                                                case TAP_EVENT_PALM_UP: break;
                                            }
                                        }
                                        void tp_tap_dead_handle_event(tap_event event)
                                        {
                                            switch (event)
                                            {
                                                case TAP_EVENT_RELEASE:
                                                    if (tp.tap.nfingers_down == 0) tp.tap.state = TAP_STATE_IDLE;
                                                    break;
                                                case TAP_EVENT_TOUCH:
                                                case TAP_EVENT_MOTION:
                                                case TAP_EVENT_TIMEOUT:
                                                case TAP_EVENT_BUTTON:
                                                case TAP_EVENT_THUMB: break;
                                                case TAP_EVENT_PALM:
                                                case TAP_EVENT_PALM_UP:
                                                    if (tp.tap.nfingers_down == 0) tp.tap.state = TAP_STATE_IDLE;
                                                    break;
                                            }
                                        }
                                        const char* tap_state_to_str(enum tp_tap_state state)
                                        {
                                            switch (state)
                                            {
                                                CASE_RETURN_STRING(TAP_STATE_IDLE);
                                                CASE_RETURN_STRING(TAP_STATE_HOLD);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_TAPPED);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_TAPPED);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_TAPPED);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_2);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_2_HOLD);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_2_RELEASE);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_3);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_3_HOLD);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_3_RELEASE);
                                                CASE_RETURN_STRING(TAP_STATE_TOUCH_3_RELEASE_2);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_DRAGGING);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_DRAGGING);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_DRAGGING);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_DRAGGING_WAIT);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_DRAGGING_WAIT);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_DRAGGING_WAIT);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_DRAGGING_OR_TAP);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_DRAGGING_OR_TAP);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_DRAGGING_OR_TAP);
                                                CASE_RETURN_STRING(TAP_STATE_1FGTAP_DRAGGING_2);
                                                CASE_RETURN_STRING(TAP_STATE_2FGTAP_DRAGGING_2);
                                                CASE_RETURN_STRING(TAP_STATE_3FGTAP_DRAGGING_2);
                                                CASE_RETURN_STRING(TAP_STATE_DEAD);
                                            }
                                            return nullptr;
                                        }
                                        char const* tap_event_to_str(tap_event event)
                                        {
                                            switch (event)
                                            {
                                                CASE_RETURN_STRING(TAP_EVENT_TOUCH);
                                                CASE_RETURN_STRING(TAP_EVENT_MOTION);
                                                CASE_RETURN_STRING(TAP_EVENT_RELEASE);
                                                CASE_RETURN_STRING(TAP_EVENT_TIMEOUT);
                                                CASE_RETURN_STRING(TAP_EVENT_BUTTON);
                                                CASE_RETURN_STRING(TAP_EVENT_THUMB);
                                                CASE_RETURN_STRING(TAP_EVENT_PALM);
                                                CASE_RETURN_STRING(TAP_EVENT_PALM_UP);
                                            }
                                            return nullptr;
                                        }
                                    void tp_tap_handle_event(tp_touch& t, tap_event event, time stamp)
                                    {
                                        auto current = tp.tap.state;
                                        switch (current)
                                        {
                                            case TAP_STATE_IDLE:                         tp_tap_idle_handle_event(                 t, event, stamp); break;
                                            case TAP_STATE_TOUCH:                        tp_tap_touch_handle_event(                t, event, stamp); break;
                                            case TAP_STATE_HOLD:                         tp_tap_hold_handle_event(                 t, event, stamp); break;
                                            case TAP_STATE_1FGTAP_TAPPED:                tp_tap_tapped_handle_event(               t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_TAPPED:                tp_tap_tapped_handle_event(               t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_TAPPED:                tp_tap_tapped_handle_event(               t, event, stamp, 3); break;
                                            case TAP_STATE_TOUCH_2:                      tp_tap_touch2_handle_event(               t, event, stamp); break;
                                            case TAP_STATE_TOUCH_2_HOLD:                 tp_tap_touch2_hold_handle_event(          t, event, stamp); break;
                                            case TAP_STATE_TOUCH_2_RELEASE:              tp_tap_touch2_release_handle_event(       t, event, stamp); break;
                                            case TAP_STATE_TOUCH_3:                      tp_tap_touch3_handle_event(               t, event, stamp); break;
                                            case TAP_STATE_TOUCH_3_HOLD:                 tp_tap_touch3_hold_handle_event(          t, event, stamp); break;
                                            case TAP_STATE_TOUCH_3_RELEASE:              tp_tap_touch3_release_handle_event(       t, event, stamp); break;
                                            case TAP_STATE_TOUCH_3_RELEASE_2:            tp_tap_touch3_release2_handle_event(      t, event, stamp); break;
                                            case TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP: tp_tap_dragging_or_doubletap_handle_event(t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP: tp_tap_dragging_or_doubletap_handle_event(t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP: tp_tap_dragging_or_doubletap_handle_event(t, event, stamp, 3); break;
                                            case TAP_STATE_1FGTAP_DRAGGING:              tp_tap_dragging_handle_event(             t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_DRAGGING:              tp_tap_dragging_handle_event(             t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_DRAGGING:              tp_tap_dragging_handle_event(             t, event, stamp, 3); break;
                                            case TAP_STATE_1FGTAP_DRAGGING_WAIT:         tp_tap_dragging_wait_handle_event(        t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_DRAGGING_WAIT:         tp_tap_dragging_wait_handle_event(        t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_DRAGGING_WAIT:         tp_tap_dragging_wait_handle_event(        t, event, stamp, 3); break;
                                            case TAP_STATE_1FGTAP_DRAGGING_OR_TAP:       tp_tap_dragging_tap_handle_event(         t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_DRAGGING_OR_TAP:       tp_tap_dragging_tap_handle_event(         t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_DRAGGING_OR_TAP:       tp_tap_dragging_tap_handle_event(         t, event, stamp, 3); break;
                                            case TAP_STATE_1FGTAP_DRAGGING_2:            tp_tap_dragging2_handle_event(            t, event, stamp, 1); break;
                                            case TAP_STATE_2FGTAP_DRAGGING_2:            tp_tap_dragging2_handle_event(            t, event, stamp, 2); break;
                                            case TAP_STATE_3FGTAP_DRAGGING_2:            tp_tap_dragging2_handle_event(            t, event, stamp, 3); break;
                                            case TAP_STATE_DEAD:                         tp_tap_dead_handle_event(                    event); break;
                                        }
                                        if (tp.tap.state == TAP_STATE_IDLE || tp.tap.state == TAP_STATE_DEAD)
                                        {
                                            tp_tap_clear_timer();
                                        }
                                        if (current != tp.tap.state)
                                        {
                                            log("tap: touch %d% (%s%), tap state %s% → %s% → %s%", t.tp ? (si32)t.index : -1, t.tp ? touch_state_to_str(t.state) : "TOUCH_UNKNOWN", tap_state_to_str(current), tap_event_to_str(event), tap_state_to_str(tp.tap.state));
                                        }
                                    }
                                    bool tp_thumb_ignored_for_tap(tp_touch& t)
                                    {
                                        return (tp.thumb.detect_thumbs && tp.thumb.index == t.index
                                             && (tp.thumb.state == THUMB_STATE_PINCH
                                             || tp.thumb.state == THUMB_STATE_SUPPRESSED
                                             || tp.thumb.state == THUMB_STATE_DEAD));
                                    }
                                    bool tp_tap_exceeds_motion_threshold(tp_touch& t)
                                    {
                                        auto mm = tp_phys_delta(t.point - t.tap.initial);
                                        // If we have more fingers down than slots, we know that synaptics touchpads are likely to give us pointer jumps. This triggers the movement threshold, making three-finger taps less reliable (#101435).
                                        // This uses the real nfingers_down, not the one for taps.
                                        if ((tp.ud_device.model_flags & EVDEV_MODEL_SYNAPTICS_SERIAL_TOUCHPAD)
                                         && (tp.nfingers_down > 2 || tp.old_nfingers_down > 2)
                                         && (tp.nfingers_down > tp.num_slots || tp.old_nfingers_down > tp.num_slots))
                                        {
                                            return faux;
                                        }
                                        // Semi-mt devices will give us large movements on finger release, depending which touch is released. Make sure we ignore any movement in the same frame as a finger change.
                                        if (tp.semi_mt && tp.nfingers_down != tp.old_nfingers_down) return faux;
                                        else                                                        return mm.hypot() > lixx::default_tap_move_threshold;
                                    }
                                si32 tp_tap_handle_state(time stamp)
                                {
                                    if (!tp_tap_enabled()) return 0;
                                    // Handle queued button pressed events from clickpads. For touchpads with separate physical buttons, ignore button pressed events so they don't interfere with tapping.
                                    if (tp.buttons.is_clickpad && tp.queued & TOUCHPAD_EVENT_BUTTON_PRESS)
                                    {
                                        //todo handle empty touch
                                        //tp_tap_handle_event(nullptr, TAP_EVENT_BUTTON, stamp);
                                    }
                                    for (auto& t : tp.touches)
                                    {
                                        if (!t.dirty || t.state == TOUCH_NONE) continue;
                                        if (tp.buttons.is_clickpad && tp.queued & TOUCHPAD_EVENT_BUTTON_PRESS)
                                        {
                                            t.tap.state = TAP_TOUCH_STATE_DEAD;
                                        }
                                        // If a touch was considered thumb for tapping once, we ignore it for the rest of lifetime.
                                        if (t.tap.is_thumb) continue;
                                        // A palm tap needs to be properly released because we might be who-knows-where in the state machine. Otherwise, we ignore any event from it.
                                        if (t.tap.is_palm)
                                        {
                                            if (t.state == TOUCH_END) tp_tap_handle_event(t, TAP_EVENT_PALM_UP, stamp);
                                            continue;
                                        }
                                        if (t.state == TOUCH_HOVERING) continue;
                                        if (t.palm_state != TOUCH_PALM_NONE)
                                        {
                                            assert(!t.tap.is_palm);
                                            t.tap.is_palm = true;
                                            t.tap.state = TAP_TOUCH_STATE_DEAD;
                                            if (t.state != TOUCH_BEGIN)
                                            {
                                                tp_tap_handle_event(t, TAP_EVENT_PALM, stamp);
                                                assert(tp.tap.nfingers_down > 0);
                                                tp.tap.nfingers_down--;
                                            }
                                        }
                                        else if (t.state == TOUCH_BEGIN)
                                        {
                                            // The simple version: if a touch is a thumb on begin we ignore it. All other thumb touches follow the normal tap state for now.
                                            if (tp_thumb_ignored_for_tap(t))
                                            {
                                                t.tap.is_thumb = true;
                                                continue;
                                            }
                                            t.tap.state = TAP_TOUCH_STATE_TOUCH;
                                            t.tap.initial = t.point;
                                            tp.tap.nfingers_down++;
                                            tp_tap_handle_event(t, TAP_EVENT_TOUCH, stamp);
                                        }
                                        else if (t.state == TOUCH_END)
                                        {
                                            if (t.was_down)
                                            {
                                                assert(tp.tap.nfingers_down >= 1);
                                                tp.tap.nfingers_down--;
                                                tp_tap_handle_event(t, TAP_EVENT_RELEASE, stamp);
                                            }
                                            t.tap.state = TAP_TOUCH_STATE_IDLE;
                                        }
                                        else if (tp.tap.state != TAP_STATE_IDLE && tp_thumb_ignored(t))
                                        {
                                            tp_tap_handle_event(t, TAP_EVENT_THUMB, stamp);
                                        }
                                        else if (tp.tap.state != TAP_STATE_IDLE && tp_tap_exceeds_motion_threshold(t))
                                        {
                                            for (auto& tmp : tp.touches) // Any touch exceeding the threshold turns all touches into DEAD.
                                            {
                                                if (tmp.tap.state == TAP_TOUCH_STATE_TOUCH)
                                                {
                                                    tmp.tap.state = TAP_TOUCH_STATE_DEAD;
                                                }
                                            }
                                            tp_tap_handle_event(t, TAP_EVENT_MOTION, stamp);
                                        }
                                    }
                                    // In any state where motion exceeding the move threshold would move to the next state, filter that motion until we actually exceed it. This prevents small motion events while we're waiting on a decision if a tap is a tap.
                                    auto filter_motion = 0;
                                    switch (tp.tap.state)
                                    {
                                        case TAP_STATE_TOUCH:
                                        case TAP_STATE_1FGTAP_TAPPED:
                                        case TAP_STATE_2FGTAP_TAPPED:
                                        case TAP_STATE_3FGTAP_TAPPED:
                                        case TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP:
                                        case TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP:
                                        case TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP:
                                        case TAP_STATE_1FGTAP_DRAGGING_OR_TAP:
                                        case TAP_STATE_2FGTAP_DRAGGING_OR_TAP:
                                        case TAP_STATE_3FGTAP_DRAGGING_OR_TAP:
                                        case TAP_STATE_TOUCH_2:
                                        case TAP_STATE_TOUCH_3:
                                            filter_motion = 1;
                                            break;
                                        default: break;
                                    }
                                    assert(tp.tap.nfingers_down <= tp.nfingers_down);
                                    assert(tp.nfingers_down != 0 || tp.tap.nfingers_down == 0);
                                    return filter_motion;
                                }
                                void tp_gesture_cancel_motion_gestures(time stamp)
                                {
                                    switch (tp.gesture.state)
                                    {
                                        case GESTURE_STATE_NONE:
                                        case GESTURE_STATE_UNKNOWN:
                                        case GESTURE_STATE_SCROLL_START:
                                        case GESTURE_STATE_PINCH_START:
                                        case GESTURE_STATE_SWIPE_START:
                                        case GESTURE_STATE_3FG_DRAG_START:
                                        case GESTURE_STATE_HOLD:
                                            break;
                                        case GESTURE_STATE_HOLD_AND_MOTION:
                                        case GESTURE_STATE_POINTER_MOTION:
                                        case GESTURE_STATE_SCROLL:
                                        case GESTURE_STATE_PINCH:
                                        case GESTURE_STATE_SWIPE:
                                            log("Cancelling motion gestures");
                                            tp_gesture_cancel(stamp);
                                            break;
                                        case GESTURE_STATE_3FG_DRAG:
                                            break;
                                        case GESTURE_STATE_3FG_DRAG_RELEASED:
                                        default:
                                            break;
                                    }
                                }
                                    bool tp_tap_dragging()
                                    {
                                        switch (tp.tap.state)
                                        {
                                            case TAP_STATE_1FGTAP_DRAGGING:
                                            case TAP_STATE_2FGTAP_DRAGGING:
                                            case TAP_STATE_3FGTAP_DRAGGING:
                                            case TAP_STATE_1FGTAP_DRAGGING_2:
                                            case TAP_STATE_2FGTAP_DRAGGING_2:
                                            case TAP_STATE_3FGTAP_DRAGGING_2:
                                            case TAP_STATE_1FGTAP_DRAGGING_WAIT:
                                            case TAP_STATE_2FGTAP_DRAGGING_WAIT:
                                            case TAP_STATE_3FGTAP_DRAGGING_WAIT:
                                            case TAP_STATE_1FGTAP_DRAGGING_OR_TAP:
                                            case TAP_STATE_2FGTAP_DRAGGING_OR_TAP:
                                            case TAP_STATE_3FGTAP_DRAGGING_OR_TAP:
                                                return true;
                                            default:
                                                return faux;
                                        }
                                    }
                                    bool tp_gesture_thumb_moved()
                                    {
                                        if (tp.thumb.index != ui32max)
                                        {
                                            for (auto& thumb : tp.touches)
                                            {
                                                if (thumb.index == tp.thumb.index)
                                                {
                                                    if (!tp_touch_active_for_gesture(thumb))
                                                    {
                                                        return faux;
                                                    }
                                                    else
                                                    {
                                                        auto thumb_moved = tp_gesture_mm_moved(thumb);
                                                        auto thumb_mm = thumb_moved.hypot();
                                                        return thumb_mm >= lixx::pinch_disambiguation_move_threshold;
                                                    }
                                                }
                                            }
                                        }
                                        return faux;
                                    }
                                        void tp_gesture_handle_state_none(time stamp)
                                        {
                                            auto touch_ptrs = std::array<tp_touch*, 4>{};
                                            auto ntouches = 0u;
                                            for (auto& t : tp.touches) // Get active touches.
                                            {
                                                if (tp_touch_active_for_gesture(t))
                                                {
                                                    touch_ptrs[ntouches++] = &t;
                                                    if (ntouches == touch_ptrs.size()) break;
                                                }
                                            }
                                            // This can happen when the user does .e.g:
                                            // 1) Put down 1st finger in center (so active).
                                            // 2) Put down 2nd finger in a button area (so inactive).
                                            // 3) Put down 3th finger somewhere, gets reported as a fake finger, so gets same coordinates as 1st -> active.
                                            //
                                            // We could avoid this by looking at all touches, be we really only want to look at real touches.
                                            if (ntouches == 0) return;
                                            auto first  = touch_ptrs[0];
                                            auto second = touch_ptrs[1];
                                            if (ntouches == 1)
                                            {
                                                first->gesture_origin = first->point;
                                                tp.gesture.two_touches[0] = first;
                                                tp_gesture_handle_event(GESTURE_EVENT_FINGER_DETECTED, stamp);
                                                return;
                                            }
                                            if (!tp.gesture.enabled && !tp.tap.tap_state_enabled && ntouches == 2)
                                            {
                                                tp_gesture_handle_event(GESTURE_EVENT_SCROLL_START, stamp);
                                                return;
                                            }
                                            // For 3+ finger gestures, we only really need to track two touches.
                                            // The human hand's finger arrangement means that for a pinch, the
                                            // bottom-most touch will always be the thumb, and the top-most touch
                                            // will always be one of the fingers.
                                            //
                                            // For 3+ finger swipes, the fingers will likely (but not necessarily)
                                            // be in a horizontal line. They all move together, regardless, so it
                                            // doesn't really matter which two of those touches we track.
                                            //
                                            // Tracking top and bottom is a change from previous versions, where
                                            // we tracked leftmost and rightmost. This change enables:
                                            //
                                            // - More accurate pinch detection if thumb is near the center
                                            // - Better resting-thumb detection while two-finger scrolling
                                            // - On capable hardware, allow 3- or 4-finger swipes with resting thumb or held-down clickpad
                                            if (ntouches > 2)
                                            {
                                                second = touch_ptrs[0];
                                                for (auto i = 1u; i < ntouches && i < tp.num_slots; i++)
                                                {
                                                         if (touch_ptrs[i]->point.y < first->point.y)   first  = touch_ptrs[i];
                                                    else if (touch_ptrs[i]->point.y >= second->point.y) second = touch_ptrs[i];
                                                }
                                                if (first == second) return;
                                            }
                                            tp.gesture.initial_time = stamp;
                                            first->gesture_origin = first->point;
                                            second->gesture_origin = second->point;
                                            tp.gesture.two_touches[0] = first;
                                            tp.gesture.two_touches[1] = second;
                                            tp_gesture_handle_event(GESTURE_EVENT_FINGER_DETECTED, stamp);
                                        }
                                                                si32_coor tp_get_delta(tp_touch& t)
                                                                {
                                                                    if (t.history.count <= 1) return {};
                                                                    auto& t0 = tp_motion_history_offset(t, 0);
                                                                    auto& t1 = tp_motion_history_offset(t, 1);
                                                                    auto delta = t0.point - t1.point;
                                                                    return delta;
                                                                }
                                                            fp64_coor tp_get_touches_delta(bool average)
                                                            {
                                                                auto nactive = 0u;
                                                                auto delta = fp64_coor{};
                                                                for (auto i = 0u; i < tp.num_slots; i++)
                                                                {
                                                                    auto& t = tp.touches[i];
                                                                    if (tp_touch_active_for_gesture(t))
                                                                    {
                                                                        nactive++;
                                                                        if (t.dirty)
                                                                        {
                                                                            delta += tp_get_delta(t);
                                                                        }
                                                                    }
                                                                }
                                                                if (average && nactive != 0)
                                                                {
                                                                    delta /= nactive;
                                                                }
                                                                return delta;
                                                            }
                                                        fp64_coor tp_get_combined_touches_delta()
                                                        {
                                                            return tp_get_touches_delta(faux);
                                                        }
                                                        fp64_coor tp_get_average_touches_delta()
                                                        {
                                                            return tp_get_touches_delta(true);
                                                        }
                                                    fp64_coor tp_get_raw_pointer_motion()
                                                    {
                                                        // When a clickpad is clicked, combine motion of all active touches.
                                                        auto clicked = tp.buttons.is_clickpad && tp.buttons.state;
                                                        auto raw = clicked ? tp_get_combined_touches_delta()
                                                                           : tp_get_average_touches_delta();
                                                        return raw;
                                                    }
                                                bool tp_has_pending_pointer_motion()
                                                {
                                                    if (tp.queued & TOUCHPAD_EVENT_MOTION)
                                                    {
                                                        // Checking for raw pointer motion is enough in this case.
                                                        // Calling tp_filter_motion is intentionally omitted to avoid calling it twice (here and in tp_gesture_post_pointer_motion) with the same event.
                                                        auto raw = tp_get_raw_pointer_motion();
                                                        return !!raw;
                                                    }
                                                    return faux;
                                                }
                                                ui32 tp_gesture_get_direction(tp_touch& touch)
                                                {
                                                    auto delta = touch.point - touch.gesture_origin;
                                                    auto mm = tp_phys_delta(delta);
                                                    return pointer_tracker::get_direction(mm);
                                                }
                                                si32 tp_gesture_same_directions(si32 dir1, si32 dir2)
                                                {
                                                    // In some cases (semi-mt touchpads) we may seen one finger move e.g. N/NE and the other W/NW so we not only check for overlapping directions, but also for neighboring bits being set.
                                                    // The ((dira & 0x80) && (dirb & 0x01)) checks are to check for bit 0 and 7 being set as they also represent neighboring directions.
                                                    return ((dir1 | (dir1 >> 1)) & dir2)
                                                        || ((dir2 | (dir2 >> 1)) & dir1)
                                                        || ((dir1 & 0x80) && (dir2 & 0x01))
                                                        || ((dir2 & 0x80) && (dir1 & 0x01));
                                                }
                                            void tp_gesture_detect_motion_gestures(time stamp)
                                            {
                                                auto first = tp.gesture.two_touches[0];
                                                auto second = tp.gesture.two_touches[1];
                                                auto thumb = (tp_touch*)nullptr;
                                                auto is_hold_and_motion = faux;
                                                auto first_moved = tp_gesture_mm_moved(*first);
                                                auto first_mm = first_moved.hypot();
                                                if (tp.gesture.finger_count == 1)
                                                {
                                                    if (!tp_has_pending_pointer_motion()) return;
                                                    is_hold_and_motion = (first_mm < lixx::hold_and_motion_threshold);
                                                    if (tp.gesture.state == GESTURE_STATE_HOLD && is_hold_and_motion)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_HOLD_AND_MOTION_START, stamp);
                                                        return;
                                                    }
                                                    if (tp.gesture.state == GESTURE_STATE_HOLD_AND_MOTION && is_hold_and_motion)
                                                    {
                                                        return;
                                                    }
                                                    tp_gesture_handle_event(GESTURE_EVENT_POINTER_MOTION_START, stamp);
                                                    return;
                                                }
                                                // If we have more fingers than slots, we don't know where the fingers are. Default to swipe/3fg drag .
                                                if (tp.gesture.enabled && tp.gesture.finger_count > 2 && tp.gesture.finger_count > tp.num_slots)
                                                {
                                                    auto drag_start = tp.drag_3fg.nfingers == tp.gesture.finger_count;
                                                    if (drag_start) tp_gesture_handle_event(GESTURE_EVENT_3FG_DRAG_START, stamp);
                                                    else            tp_gesture_handle_event(GESTURE_EVENT_SWIPE_START, stamp);
                                                    return;
                                                }
                                                // Need more margin for error when there are more fingers.
                                                auto min_move = 1.5; // Min movement threshold in mm - count this touch.
                                                auto max_move = 4.0; // Max movement threshold in mm - ignore other touch.
                                                max_move += 2.0 * (tp.gesture.finger_count - 2);
                                                min_move += 0.5 * (tp.gesture.finger_count - 2);
                                                auto second_moved = tp_gesture_mm_moved(*second);
                                                auto second_mm = second_moved.hypot(); // Movement since gesture start in mm.
                                                auto delta = std::abs(first->point - second->point);
                                                auto distance_mm = evdev_device_unit_delta_to_mm(delta);
                                                // If both touches moved less than a mm, we cannot decide yet.
                                                if (first_mm < 1 && second_mm < 1) return;
                                                // If both touches are within 7mm vertically and 40mm horizontally past the timeout, assume scroll/swipe.
                                                if ((!tp.gesture.enabled || (distance_mm.x < 40.0 && distance_mm.y < 7.0))
                                                 && stamp > (tp.gesture.initial_time + lixx::default_gesture_swipe_timeout))
                                                {
                                                    auto f = tp.gesture.finger_count;
                                                    auto nf = tp.drag_3fg.nfingers;
                                                         if (f == 2)  tp_gesture_handle_event(GESTURE_EVENT_SCROLL_START, stamp);
                                                    else if (f == nf) tp_gesture_handle_event(GESTURE_EVENT_3FG_DRAG_START, stamp);
                                                    else              tp_gesture_handle_event(GESTURE_EVENT_SWIPE_START, stamp);
                                                    return;
                                                }
                                                // If 3fg dragging touches are within a 60x10mm box, start dragging immediately.
                                                if (tp.gesture.finger_count == tp.drag_3fg.nfingers
                                                 && distance_mm.x < 60.0 && distance_mm.y < 10.0)
                                                {
                                                    tp_gesture_handle_event(GESTURE_EVENT_3FG_DRAG_START, stamp);
                                                    return;
                                                }
                                                // If one touch exceeds the max_move threshold while the other has not yet passed the min_move threshold, there is either a resting thumb, or the user is doing "one-finger-scroll," where one touch stays in place while the other moves.
                                                if (first_mm >= max_move || second_mm >= max_move)
                                                {
                                                    auto thumb_mm = 0.0;
                                                    auto finger_mm = 0.0;
                                                    if (first->point.y > second->point.y) // Pick the thumb as the lowest point on the touchpad.
                                                    {
                                                        thumb = first;
                                                        thumb_mm = first_mm;
                                                        finger_mm = second_mm;
                                                    }
                                                    else
                                                    {
                                                        thumb = second;
                                                        thumb_mm = second_mm;
                                                        finger_mm = first_mm;
                                                    }
                                                    // If thumb detection is enabled, and thumb is still while finger moves, cancel gestures and mark lower as thumb.
                                                    // This applies to all gestures (2, 3, 4+ fingers), but allows more thumb motion on >2 finger gestures during detection.
                                                    if (tp.thumb.detect_thumbs && thumb_mm < min_move)
                                                    {
                                                        tp_thumb_suppress(*thumb);
                                                        tp_gesture_cancel(stamp);
                                                        return;
                                                    }
                                                    // If gestures detection is disabled, or if finger is still while thumb moves, assume this is "one-finger scrolling." This applies only to 2-finger gestures.
                                                    if ((!tp.gesture.enabled || finger_mm < min_move) && tp.gesture.finger_count == 2)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_SCROLL_START, stamp);
                                                        return;
                                                    }
                                                    // If more than 2 fingers are involved, and the thumb moves while the fingers stay still, assume a pinch if eligible.
                                                    if (finger_mm < min_move
                                                     && tp.gesture.finger_count > 2
                                                     && tp.gesture.enabled
                                                     && tp.thumb.pinch_eligible)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_PINCH_START, stamp);
                                                        return;
                                                    }
                                                }
                                                // If either touch is still below the min_move threshold, we can't tell what kind of gesture this is.
                                                if ((first_mm < min_move) || (second_mm < min_move)) return;
                                                // Both touches have exceeded the min_move threshold, so we have a valid gesture. Update gesture initial time and get directions so we know if it's a pinch or swipe/scroll.
                                                auto dir1 = tp_gesture_get_direction(*first);
                                                auto dir2 = tp_gesture_get_direction(*second);
                                                // If we can't accurately detect pinches, or if the touches are moving the same way, this is a scroll or swipe.
                                                if (tp.gesture.finger_count > tp.num_slots || tp_gesture_same_directions(dir1, dir2))
                                                {
                                                    if (tp.gesture.finger_count == 2)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_SCROLL_START, stamp);
                                                        return;
                                                    }
                                                    if (tp.drag_3fg.nfingers == tp.gesture.finger_count)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_3FG_DRAG_START, stamp);
                                                        return;
                                                    }
                                                    if (tp.gesture.enabled)
                                                    {
                                                        tp_gesture_handle_event(GESTURE_EVENT_SWIPE_START, stamp);
                                                        return;
                                                    }
                                                }
                                                // If the touches are moving away from each other, this is a pinch.
                                                tp_gesture_handle_event(GESTURE_EVENT_PINCH_START, stamp);
                                            }
                                        void tp_gesture_handle_state_unknown(time stamp, bool ignore_motion)
                                        {
                                            if (!ignore_motion) tp_gesture_detect_motion_gestures(stamp);
                                        }
                                        void tp_gesture_handle_state_hold(time stamp, bool ignore_motion)
                                        {
                                            if (!ignore_motion) tp_gesture_detect_motion_gestures(stamp);
                                        }
                                                    fp64_coor tp_scale_to_xaxis(fp64_coor delta)
                                                    {
                                                        delta.y *= tp.accel_xy_scale_coeff;
                                                        return delta;
                                                    }
                                                fp64_coor tp_filter_motion(fp64_coor unaccelerated, time stamp)
                                                {
                                                    if (!unaccelerated) return fp64_coor{};
                                                    auto raw = tp_scale_to_xaxis(unaccelerated); // Convert to device units with x/y in the same resolution.
                                                    return tp.pointer_filter->filter_dispatch(raw, &tp, stamp);
                                                }
                                            void tp_gesture_post_pointer_motion(time stamp)
                                            {
                                                auto raw = tp_get_raw_pointer_motion();
                                                auto delta = tp_filter_motion(raw, stamp);
                                                if (delta || raw)
                                                {
                                                    auto unaccel = tp_scale_to_xaxis(raw);
                                                    tp.pointer_notify_motion(stamp, delta, unaccel);
                                                }
                                            }
                                        void tp_gesture_handle_state_hold_and_pointer_motion(time stamp)
                                        {
                                            if (tp.queued & TOUCHPAD_EVENT_MOTION) tp_gesture_post_pointer_motion(stamp);
                                            tp_gesture_detect_motion_gestures(stamp);
                                        }
                                        void tp_gesture_handle_state_pointer_motion(time stamp)
                                        {
                                            if (tp.queued & TOUCHPAD_EVENT_MOTION) tp_gesture_post_pointer_motion(stamp);
                                        }
                                            bool tp_gesture_is_pinch()
                                            {
                                                auto first = tp.gesture.two_touches[0];
                                                auto second = tp.gesture.two_touches[1];
                                                auto dir1 = tp_gesture_get_direction(*first);
                                                auto dir2 = tp_gesture_get_direction(*second);
                                                if (tp_gesture_same_directions(dir1, dir2))
                                                {
                                                    return faux;
                                                }
                                                auto first_mm = tp_gesture_mm_moved(*first).hypot();
                                                if (first_mm < lixx::pinch_disambiguation_move_threshold)
                                                {
                                                    return faux;
                                                }
                                                auto second_mm = tp_gesture_mm_moved(*second).hypot();
                                                if (second_mm < lixx::pinch_disambiguation_move_threshold)
                                                {
                                                    return faux;
                                                }
                                                return true;
                                            }
                                            fp64_coor tp_filter_scroll(fp64_coor unaccelerated, time stamp)
                                            {
                                                if (!unaccelerated) return fp64_coor{};
                                                auto raw = tp_scale_to_xaxis(unaccelerated); // Convert to device units with x/y in the same resolution.
                                                return tp.pointer_filter->filter_dispatch_scroll(raw, stamp);
                                            }
                                            void tp_gesture_init_scroll()
                                            {
                                                tp.tp_scroll.active   = {};
                                                tp.tp_scroll.duration = {};
                                                tp.tp_scroll.vector   = {};
                                                tp.tp_scroll.stamp    = {};
                                            }
                                        void tp_gesture_handle_state_scroll_start(time stamp)
                                        {
                                            if (tp.tp_scroll.method != LIBINPUT_CONFIG_SCROLL_2FG) return;
                                            // We may confuse a pinch for a scroll initially, allow ourselves to correct our guess.
                                            if (stamp < (tp.gesture.initial_time + lixx::default_gesture_pinch_timeout) && tp_gesture_is_pinch())
                                            {
                                                tp_gesture_handle_event(GESTURE_EVENT_PINCH_START, stamp);
                                            }
                                            else
                                            {
                                                auto raw = tp_get_average_touches_delta();
                                                if (auto delta = tp_filter_scroll(raw, stamp)) // Scroll is not accelerated by default.
                                                {
                                                    tp_gesture_init_scroll();
                                                    tp.gesture.state = GESTURE_STATE_SCROLL;
                                                }
                                            }
                                        }
                                            void tp_gesture_apply_scroll_constraints(fp64_coor& raw, fp64_coor& delta, time stamp)
                                            {
                                                if (tp.tp_scroll.active.h && tp.tp_scroll.active.v) // Both axes active == true means free scrolling is enabled.
                                                {
                                                    return;
                                                }
                                                // Determine time delta since last movement event.
                                                auto tdelta = span{};
                                                if (tp.tp_scroll.stamp != time{}) tdelta = stamp - tp.tp_scroll.stamp;
                                                if (tdelta > lixx::default_scroll_event_timeout) tdelta = {};
                                                tp.tp_scroll.stamp = stamp;
                                                auto delta_mm = tp_phys_delta(raw); // Delta since last movement event in mm.
                                                // Old vector data "fades" over time. This is a two-part linear approximation of an exponential function - for example, for lixx::default_scroll_event_timeout of 100, vector_decay = (0.97)^tdelta. This linear approximation allows easier tweaking of lixx::default_scroll_event_timeout and is faster.
                                                auto vector_decay = 0.0;
                                                //if (std::chrono::is_negative(tdelta))
                                                if (tdelta > span{})
                                                {
                                                    auto recent = ((lixx::default_scroll_event_timeout / 2.0) - tdelta) / (lixx::default_scroll_event_timeout / 2.0);
                                                    auto later = (lixx::default_scroll_event_timeout - tdelta) / (lixx::default_scroll_event_timeout * 2.0);
                                                    vector_decay = tdelta <= (0.33 * lixx::default_scroll_event_timeout) ? recent : later;
                                                }
                                                // Calculate windowed vector from delta + weighted historic data.
                                                auto vector = (tp.tp_scroll.vector * vector_decay) + delta_mm;
                                                auto vector_length = vector.hypot();
                                                tp.tp_scroll.vector = vector;
                                                // We care somewhat about distance and speed, but more about consistency of direction over time. Keep track of the time spent primarily along each axis. If one axis is active, time spent NOT moving much in the other axis is subtracted, allowing a switch of axes in a single scroll + ability to "break out" and go diagonal.
                                                // Slope to degree conversions (infinity = 90°, 0 = 0°):
                                                static constexpr auto degree_75 = 3.73;
                                                static constexpr auto degree_60 = 1.73;
                                                static constexpr auto degree_30 = 0.57;
                                                static constexpr auto degree_15 = 0.27;
                                                auto slope = (vector.x != 0) ? std::abs(vector.y / vector.x) : INFINITY;
                                                // Ensure vector is big enough (in mm per lixx::default_scroll_event_timeout) to be confident of direction. Larger = harder to enable diagonal/free scrolling.
                                                static constexpr auto MIN_VECTOR = 0.15;
                                                if (slope >= degree_30 && vector_length > MIN_VECTOR)
                                                {
                                                    tp.tp_scroll.duration.v += tdelta;
                                                    if (tp.tp_scroll.duration.v > lixx::active_threshold) tp.tp_scroll.duration.v = lixx::active_threshold;
                                                    if (slope >= degree_75)
                                                    {
                                                        auto out = tp.tp_scroll.duration.h > tdelta;
                                                        tp.tp_scroll.duration.h = out ? tp.tp_scroll.duration.h - tdelta : span{};
                                                    }
                                                }
                                                if (slope < degree_60  && vector_length > MIN_VECTOR)
                                                {
                                                    tp.tp_scroll.duration.h += tdelta;
                                                    if (tp.tp_scroll.duration.h > lixx::active_threshold) tp.tp_scroll.duration.h = lixx::active_threshold;
                                                    if (slope < degree_15)
                                                    {
                                                        auto out = tp.tp_scroll.duration.v > tdelta;
                                                        tp.tp_scroll.duration.v = out ? tp.tp_scroll.duration.v - tdelta : span{};
                                                    }
                                                }
                                                if (tp.tp_scroll.duration.h == lixx::active_threshold)
                                                {
                                                    tp.tp_scroll.active.h = true;
                                                    if (tp.tp_scroll.duration.v < lixx::inactive_threshold) tp.tp_scroll.active.v = faux;
                                                }
                                                if (tp.tp_scroll.duration.v == lixx::active_threshold)
                                                {
                                                    tp.tp_scroll.active.v = true;
                                                    if (tp.tp_scroll.duration.h < lixx::inactive_threshold) tp.tp_scroll.active.h = faux;
                                                }
                                                // If vector is big enough in a diagonal direction, always unlock both axes regardless of thresholds.
                                                if (vector_length > 5.0 && slope < 1.73 && slope >= 0.57)
                                                {
                                                    tp.tp_scroll.active.v = true;
                                                    tp.tp_scroll.active.h = true;
                                                }
                                                // If only one axis is active, constrain motion accordingly. If both are set, we've detected deliberate diagonal movement; enable free scrolling for the life of the gesture.
                                                if (!tp.tp_scroll.active.h && tp.tp_scroll.active.v) delta.x = 0.0;
                                                if (tp.tp_scroll.active.h && !tp.tp_scroll.active.v) delta.y = 0.0;
                                                // If we haven't determined an axis, use the slope in the meantime.
                                                if (!tp.tp_scroll.active.h && !tp.tp_scroll.active.v)
                                                {
                                                    delta.x = (slope >= degree_60) ? 0.0 : delta.x;
                                                    delta.y = (slope < degree_30) ? 0.0 : delta.y;
                                                }
                                            }
                                        void tp_gesture_handle_state_scroll(time stamp)
                                        {
                                            if (tp.tp_scroll.method != LIBINPUT_CONFIG_SCROLL_2FG) return;
                                            // We may confuse a pinch for a scroll initially, allow ourselves to correct our guess.
                                            if (stamp < (tp.gesture.initial_time + lixx::default_gesture_pinch_timeout) && tp_gesture_is_pinch())
                                            {
                                                tp_gesture_handle_event(GESTURE_EVENT_PINCH_START, stamp);
                                            }
                                            else
                                            {
                                                auto raw = tp_get_average_touches_delta();
                                                // Scroll is not accelerated by default.
                                                if (auto delta = tp_filter_scroll(raw, stamp))
                                                {
                                                    tp_gesture_apply_scroll_constraints(raw, delta, stamp);
                                                    tp.evdev_post_scroll(stamp, LIBINPUT_POINTER_AXIS_SOURCE_FINGER, delta);
                                                }
                                            }
                                        }
                                            void gesture_notify_swipe(time stamp, libinput_event_type type, si32 finger_count, fp64_coor delta, fp64_coor unaccel)
                                            {
                                                gesture_notify(stamp, type, finger_count, 0, delta, unaccel, 0.0, 0.0);
                                            }
                                        void tp_gesture_handle_state_swipe_start(time stamp)
                                        {
                                            auto raw = tp_get_average_touches_delta();
                                            auto delta = tp_filter_motion(raw, stamp);
                                            if (delta || raw)
                                            {
                                                gesture_notify_swipe(stamp, LIBINPUT_EVENT_GESTURE_SWIPE_BEGIN, tp.gesture.finger_count, lixx::zero_coor, lixx::zero_coor);
                                                tp.gesture.state = GESTURE_STATE_SWIPE;
                                            }
                                        }
                                            fp64_coor tp_filter_motion_unaccelerated(fp64_coor unaccelerated, time stamp)
                                            {
                                                if (!unaccelerated) return fp64_coor{};
                                                auto raw = tp_scale_to_xaxis(unaccelerated); // Convert to device units with x/y in the same resolution.
                                                return tp.pointer_filter->filter_constant(raw, stamp);
                                            }
                                        void tp_gesture_handle_state_swipe(time stamp)
                                        {
                                            auto raw = tp_get_average_touches_delta();
                                            auto delta = tp_filter_motion(raw, stamp);
                                            if (delta || raw)
                                            {
                                                auto unaccel = tp_filter_motion_unaccelerated(raw, stamp);
                                                gesture_notify_swipe(stamp, LIBINPUT_EVENT_GESTURE_SWIPE_UPDATE, tp.gesture.finger_count, delta, unaccel);
                                            }
                                        }
                                            fp64_coor device_float_delta(fp64_coor a, fp64_coor b)
                                            {
                                                return a - b;
                                            }
                                            void gesture_notify_pinch(time stamp, libinput_event_type type, si32 finger_count, fp64_coor delta, fp64_coor unaccel, fp64 scale, fp64 angle)
                                            {
                                                gesture_notify(stamp, type, finger_count, 0, delta, unaccel, scale, angle);
                                            }
                                        void tp_gesture_handle_state_pinch_start(time stamp)
                                        {
                                            auto angle = 0.0;
                                            auto distance = 0.0;
                                            auto center = fp64_coor{};
                                            tp_gesture_get_pinch_info(distance, angle, center);
                                            auto scale = distance / tp.gesture.initial_distance;
                                            auto angle_delta = angle - tp.gesture.angle;
                                            tp.gesture.angle = angle;
                                                 if (angle_delta >  180.0) angle_delta -= 360.0;
                                            else if (angle_delta < -180.0) angle_delta += 360.0;
                                            auto fdelta = device_float_delta(center, tp.gesture.center);
                                            tp.gesture.center = center;
                                            auto delta = tp_filter_motion(fdelta, stamp);
                                            if (delta || fdelta || scale != tp.gesture.prev_scale || angle_delta != 0.0)
                                            {
                                                gesture_notify_pinch(stamp, LIBINPUT_EVENT_GESTURE_PINCH_BEGIN, tp.gesture.finger_count, lixx::zero_coor, lixx::zero_coor, 1.0, 0.0);
                                                tp.gesture.prev_scale = scale;
                                                tp.gesture.state = GESTURE_STATE_PINCH;
                                            }
                                        }
                                        void tp_gesture_handle_state_pinch(time stamp)
                                        {
                                            auto angle = 0.0;
                                            auto distance = 0.0;
                                            auto center = fp64_coor{};
                                            tp_gesture_get_pinch_info(distance, angle, center);
                                            auto scale = distance / tp.gesture.initial_distance;
                                            auto angle_delta = angle - tp.gesture.angle;
                                            tp.gesture.angle = angle;
                                            if (angle_delta > 180.0)
                                            {
                                                angle_delta -= 360.0;
                                            }
                                            else if (angle_delta < -180.0)
                                            {
                                                angle_delta += 360.0;
                                            }
                                            auto fdelta = device_float_delta(center, tp.gesture.center);
                                            tp.gesture.center = center;
                                            auto delta = tp_filter_motion(fdelta, stamp);
                                            if (delta || fdelta || scale != tp.gesture.prev_scale || angle_delta != 0.0)
                                            {
                                                auto unaccel = tp_filter_motion_unaccelerated(fdelta, stamp);
                                                gesture_notify_pinch(stamp, LIBINPUT_EVENT_GESTURE_PINCH_UPDATE, tp.gesture.finger_count, delta, unaccel, scale, angle_delta);
                                                tp.gesture.prev_scale = scale;
                                            }
                                        }
                                        void tp_gesture_handle_state_3fg_drag_start(time stamp)
                                        {
                                            tp.evdev_pointer_notify_button(stamp, evdev::btn_left, evdev::pressed);
                                            tp.gesture.state = GESTURE_STATE_3FG_DRAG; //todo FIXME: immediately send a motion event?
                                        }
                                        void tp_gesture_handle_state_3fg_drag(time stamp)
                                        {
                                            if (tp.queued & TOUCHPAD_EVENT_MOTION) tp_gesture_post_pointer_motion(stamp);
                                        }
                                        void tp_gesture_handle_state_3fg_drag_released(time stamp)
                                        {
                                            tp_gesture_detect_motion_gestures(stamp);
                                        }
                                    void tp_gesture_handle_state(time stamp, bool ignore_motion)
                                    {
                                        static auto remember_transition = [](auto& ts, auto& state)
                                        {
                                            if (*ts != state) *(++ts) = state;
                                        };
                                        auto transitions = std::array<tp_gesture_state, 16>{};
                                        static_assert(transitions.size() >= GESTURE_STATE_COUNT);
                                        auto oldstate = tp.gesture.state;
                                        auto transition_state = transitions.begin();
                                        *transition_state = tp.gesture.state;
                                        if (tp.gesture.state == GESTURE_STATE_NONE             ) { tp_gesture_handle_state_none(                   stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_UNKNOWN          ) { tp_gesture_handle_state_unknown(                stamp, ignore_motion); remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_HOLD             ) { tp_gesture_handle_state_hold(                   stamp, ignore_motion); remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_POINTER_MOTION   ) { tp_gesture_handle_state_pointer_motion(         stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_HOLD_AND_MOTION  ) { tp_gesture_handle_state_hold_and_pointer_motion(stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_SCROLL           ) { tp_gesture_handle_state_scroll(                 stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_SCROLL_START     ) { tp_gesture_handle_state_scroll_start(           stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_SWIPE            ) { tp_gesture_handle_state_swipe(                  stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_SWIPE_START      ) { tp_gesture_handle_state_swipe_start(            stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_PINCH            ) { tp_gesture_handle_state_pinch(                  stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_PINCH_START      ) { tp_gesture_handle_state_pinch_start(            stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_3FG_DRAG         ) { tp_gesture_handle_state_3fg_drag(               stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_3FG_DRAG_START   ) { tp_gesture_handle_state_3fg_drag_start(         stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if (tp.gesture.state == GESTURE_STATE_3FG_DRAG_RELEASED) { tp_gesture_handle_state_3fg_drag_released(      stamp);                remember_transition(transition_state, tp.gesture.state); }
                                        if constexpr (debugmode)
                                        {
                                            if (oldstate != tp.gesture.state)
                                            {
                                                auto states = text{};
                                                for (auto s = transitions.begin(); s < transition_state; s++)
                                                {
                                                    states += gesture_state_to_str(*s);
                                                    states += " → ";
                                                }
                                                states += gesture_state_to_str(tp.gesture.state);
                                                log("gesture1: [%d%fg] state %s%", tp.gesture.finger_count, states);
                                            }
                                        }
                                    }
                                void tp_gesture_post_events(time stamp, bool ignore_motion)
                                {
                                    if (tp.gesture.finger_count == 0) return;
                                    // When tap-and-dragging, force 1fg mode. On clickpads, if the physical button is down, don't allow gestures unless the button is held down by a *thumb*, specifically.
                                    if (tp_tap_dragging() || (tp.buttons.is_clickpad && tp.buttons.state && tp.thumb.state == THUMB_STATE_FINGER))
                                    {
                                        if (tp.gesture.state != GESTURE_STATE_POINTER_MOTION)
                                        {
                                            tp_gesture_cancel(stamp);
                                            tp_gesture_handle_event(GESTURE_EVENT_POINTER_MOTION_START, stamp);
                                        }
                                        tp.gesture.finger_count = 1;
                                        tp.gesture.finger_count_pending = 0;
                                    }
                                    // Don't send events when we're unsure in which mode we are.
                                    if (tp.gesture.finger_count_pending) return;
                                    // When pinching, the thumb tends to move slower than the finger, so we may suppress it too early. Give it some time to move..
                                    if (stamp < (tp.gesture.initial_time + lixx::default_gesture_pinch_timeout) && tp_gesture_thumb_moved())
                                    {
                                        tp_thumb_reset();
                                    }
                                    if (tp.gesture.finger_count <= 4)
                                    {
                                        tp_gesture_handle_state(stamp, ignore_motion);
                                    }
                                }
                                si32 tp_edge_scroll_post_events(time stamp)
                                {
                                    auto axis = libinput_pointer_axis{};
                                    auto delta = (fp64*)nullptr;
                                    auto normalized = fp64_coor{};
                                    auto tmp = fp64_coor{};
                                    for (auto& t : tp.touches)
                                    {
                                        if (!t.dirty) continue;
                                        if (t.palm_state != TOUCH_PALM_NONE || tp_thumb_ignored(t)) continue;
                                        // Only scroll with the finger in the previous edge.
                                        if (t.scroll.edge && !(tp_touch_get_edge(t) & t.scroll.edge)) continue;
                                        switch (t.scroll.edge)
                                        {
                                            case EDGE_NONE:
                                                if (t.scroll.direction != -1)
                                                {
                                                    // Send stop scroll event.
                                                    tp.evdev_notify_axis_finger(stamp, (1ul << t.scroll.direction), lixx::zero_coor);
                                                    t.scroll.direction = -1;
                                                }
                                                continue;
                                            case EDGE_RIGHT:
                                                axis = LIBINPUT_POINTER_AXIS_SCROLL_VERTICAL;
                                                delta = &normalized.y;
                                                break;
                                            case EDGE_BOTTOM:
                                                axis = LIBINPUT_POINTER_AXIS_SCROLL_HORIZONTAL;
                                                delta = &normalized.x;
                                                break;
                                            default: // EDGE_RIGHT | EDGE_BOTTOM.
                                                continue; // Don't know direction yet, skip.
                                        }
                                        auto raw = tp_get_delta(t);
                                        auto fraw = fp64_coor{ raw };
                                        // Scroll is not accelerated.
                                        normalized = tp_filter_motion_unaccelerated(fraw, stamp);
                                        switch (t.scroll.edge_state)
                                        {
                                            case EDGE_SCROLL_TOUCH_STATE_NONE:
                                            case EDGE_SCROLL_TOUCH_STATE_AREA: log("unexpected scroll state %d%", t.scroll.edge_state); break;
                                            case EDGE_SCROLL_TOUCH_STATE_EDGE_NEW:
                                                tmp = normalized;
                                                normalized = tp_normalize_delta(t.point - t.scroll.initial);
                                                // Use a reasonably large threshold until locked into scrolling mode, to avoid accidentally locking in scrolling mode when trying to use the entire touchpad to move the pointer. The user can wait for the timeout to trigger to do a small scroll.
                                                // Convert mm to a distance normalized to lixx::default_mouse_dpi.
                                                if (std::abs(*delta) < lixx::default_scroll_threshold) normalized = lixx::zero_coor;
                                                else                                                   normalized = tmp;
                                                break;
                                            case EDGE_SCROLL_TOUCH_STATE_EDGE: break;
                                        }
                                        if (*delta != 0.0)
                                        {
                                            tp.evdev_notify_axis_finger(stamp, (1ul << axis), normalized);
                                            t.scroll.direction = axis;
                                            tp_edge_scroll_handle_event(t, SCROLL_EVENT_POSTED, stamp);
                                        }
                                    }
                                    return 0; // Edge touches are suppressed by edge_scroll_touch_active.
                                }
                            void tp_post_events(time stamp)
                            {
                                auto ignore_motion = faux;
                                if (tp.is_suspended) // Only post (top) button events while suspended.
                                {
                                    tp_post_button_events(stamp);
                                    return;
                                }
                                ignore_motion |= tp_tap_handle_state(stamp);
                                ignore_motion |= tp_post_button_events(stamp);
                                if (tp.palm.trackpoint_active || tp.dwt.keyboard_active)
                                {
                                    tp_edge_scroll_stop_events(stamp);
                                    tp_gesture_cancel(stamp);
                                    return;
                                }
                                if (ignore_motion)
                                {
                                    tp_edge_scroll_stop_events(stamp);
                                    tp_gesture_cancel_motion_gestures(stamp);
                                    tp_gesture_post_events(stamp, true);
                                    return;
                                }
                                if (tp_edge_scroll_post_events(stamp) != 0)
                                {
                                    return;
                                }
                                tp_gesture_post_events(stamp, faux);
                            }
                            void tp_apply_rotation()
                            {
                                if (tp.left_handed.want_rotate != tp.left_handed.rotate && !tp.nfingers_down)
                                {
                                    tp.left_handed.rotate = tp.left_handed.want_rotate;
                                    log("touchpad-rotation: rotation is %s%", tp.left_handed.rotate ? "on" : "off");
                                }
                            }
                                void tp_init_softbuttons()
                                {
                                    auto mb_le = 0; // Middle button left/right edge.
                                    auto mb_re = 0;
                                    auto mm = fp64_coor{};
                                    auto [w, h] = tp.evdev_device_get_size();
                                    // Button height: 10mm or 15% or the touchpad height, whichever is smaller.
                                    if (h * 0.15 > 10) mm.y = h - 10;
                                    else               mm.y = h * 0.85;
                                    mm.x = w * 0.5;
                                    auto edges = tp.evdev_device_mm_to_units(mm);
                                    tp.buttons.bottom_area.top_edge = edges.y;
                                    tp.buttons.bottom_area.rightbutton_left_edge = edges.x;
                                    tp.buttons.bottom_area.middlebutton_left_edge = si32max;
                                    // If middlebutton emulation is enabled, don't init a software area.
                                    if (tp.middlebutton.want_enabled) return;
                                    // The middle button is 25% of the touchpad and centered. Many touchpads don't have markings for the middle button at all so we need to make it big enough to reliably hit it but not too big so it takes away all the space.
                                    // On touchpads with visible markings we reduce the size of the middle button since users have a visual guide.
                                    if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_TOUCHPAD_VISIBLE_MARKER))
                                    {
                                        mm.x = w / 2 - 5; // 10mm wide.
                                        edges = tp.evdev_device_mm_to_units(mm);
                                        mb_le = edges.x;
                                        mm.x = w / 2 + 5; // 10mm wide.
                                        edges = tp.evdev_device_mm_to_units(mm);
                                        mb_re = edges.x;
                                    }
                                    else
                                    {
                                        mm.x = w * 0.375;
                                        edges = tp.evdev_device_mm_to_units(mm);
                                        mb_le = edges.x;
                                        mm.x = w * 0.625;
                                        edges = tp.evdev_device_mm_to_units(mm);
                                        mb_re = edges.x;
                                    }
                                    tp.buttons.bottom_area.middlebutton_left_edge = mb_le;
                                    tp.buttons.bottom_area.rightbutton_left_edge = mb_re;
                                }
                            void tp_clickpad_middlebutton_apply_config()
                            {
                                if (!tp.buttons.is_clickpad || tp.buttons.state != 0) return;
                                if (tp.middlebutton.want_enabled == tp.middlebutton.enabled) return;
                                tp.middlebutton.enabled = tp.middlebutton.want_enabled;
                                if (tp.buttons.click_method == LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS)
                                {
                                    tp_init_softbuttons();
                                }
                            }
                            void tp_3fg_drag_apply_config()
                            {
                                if (tp.drag_3fg.want_nfingers == tp.drag_3fg.nfingers) return;
                                if (tp.nfingers_down) return;
                                tp.drag_3fg.nfingers = tp.drag_3fg.want_nfingers;
                                log("touchpad-3fg-drag: drag is now for %zd% fingers", tp.drag_3fg.nfingers);
                            }
                        void tp_handle_state(time stamp)
                        {
                            tp_pre_process_state(stamp);
                            tp_process_state(stamp);
                            tp_post_events(stamp);
                            tp_post_process_state();
                            tp_clickpad_middlebutton_apply_config();
                            tp_apply_rotation();
                            tp_3fg_drag_apply_config();
                        }
            void tp_interface_process(evdev_event& ev, time stamp)
            {
                auto type = evdev_usage_type(ev.usage);
                switch (type)
                {
                    case EV_ABS:
                        if (tp.has_mt) tp_process_absolute(ev);
                        else           tp_process_absolute_st(ev);
                        break;
                    case EV_KEY: tp_process_key(ev, stamp); break;
                    case EV_MSC: tp_process_msc(ev); break;
                    case EV_SYN: tp_handle_state(stamp); break;
                }
            }
                            void tp_release_all_buttons()
                            {
                                if (tp.buttons.state)
                                {
                                    tp.buttons.state = 0;
                                    tp.queued = (touchpad_event)(tp.queued | TOUCHPAD_EVENT_BUTTON_RELEASE);
                                }
                            }
                            void tp_release_all_taps(time now)
                            {
                                for (auto i = 1; i <= 3; i++)
                                {
                                    if (tp.tap.buttons_pressed & (1ul << i))
                                    {
                                        tp_tap_notify(now, i, evdev::released);
                                    }
                                }
                                for (auto& t : tp.touches) // To neutralize all current touches, we make them all palms.
                                {
                                    if (t.state != TOUCH_NONE && !t.tap.is_palm)
                                    {
                                        t.tap.is_palm = true;
                                        t.tap.state = TAP_TOUCH_STATE_DEAD;
                                    }
                                }
                                tp.tap.state = TAP_STATE_IDLE;
                                tp.tap.nfingers_down = 0;
                            }
                            void tp_release_fake_touches()
                            {
                                tp.fake_touches = 0;
                            }
                        void tp_clear_state()
                        {
                            // Unroll the touchpad state.
                            // Release buttons first.
                            // If tp is a clickpad, the button event must come before the touch up.
                            // If it isn't, the order doesn't matter anyway.
                            // Then cancel all timeouts on the taps, triggering the last set of events.
                            // Then lift all touches so the touchpad is in a neutral state.
                            // Then reset thumb state.
                            auto now = datetime::now();
                            tp_release_all_buttons();
                            tp_release_all_taps(now);
                            for (auto& t : tp.touches)
                            {
                                tp_end_sequence(t);
                            }
                            tp_release_fake_touches();
                            tp_thumb_reset();
                            tp_handle_state(now);
                        }
                        void tp_remove_tap()
                        {
                            tp.tap.timer->cancel();
                        }
                        void tp_remove_buttons()
                        {
                            for (auto& t : tp.touches)
                            {
                                t.button.timer->cancel();
                                t.button.timer.reset();
                            }
                        }
                        void tp_remove_sendevents()
                        {
                            tp.palm.trackpoint_timer->cancel();
                            tp.dwt.keyboard_timer->cancel();
                            if (tp.buttons.trackpoint_li_device && tp.palm.monitor_trackpoint)
                            {
                                tp.buttons.trackpoint_li_device->libinput_device_remove_event_listener(tp.palm.trackpoint_listener);
                            }
                            for (auto kbd : tp.dwt.paired_keyboard_list)
                            {
                                kbd->li_device->libinput_device_remove_event_listener(kbd->listener);
                            }
                            tp.dwt.paired_keyboard_list.clear();
                            tp.dwt.keyboard_active = faux;
                            if (tp.lid_switch.lid_switch_li_device)
                            {
                                tp.lid_switch.lid_switch_li_device->libinput_device_remove_event_listener(tp.lid_switch.listener);
                            }
                            if (tp.tablet_mode_switch.tablet_mode_switch_li_device)
                            {
                                tp.tablet_mode_switch.tablet_mode_switch_li_device->libinput_device_remove_event_listener(tp.tablet_mode_switch.listener);
                            }
                        }
                        void tp_remove_edge_scroll()
                        {
                            for (auto& t : tp.touches)
                            {
                                if (auto& timer = t.scroll.timer) 
                                {
                                    timer->cancel();
                                    timer.reset();
                                }
                            }
                        }
                        void tp_remove_gesture()
                        {
                            tp.gesture.finger_count_switch_timer->cancel();
                            tp.gesture.hold_timer->cancel();
                            tp.gesture.drag_3fg_timer->cancel();
                        }
            void tp_interface_remove()
            {
                tp.arbitration.arbitration_timer->cancel();
                tp_remove_tap();
                tp_remove_buttons();
                tp_remove_sendevents();
                tp_remove_edge_scroll();
                tp_remove_gesture();
            }
                                        void tp_tap_enabled_update(bool suspended, bool enabled, time stamp)
                                        {
                                            auto was_enabled = tp_tap_enabled();
                                            tp.tap.suspended = suspended;
                                            tp.tap.tap_state_enabled = enabled;
                                            if (tp_tap_enabled() == was_enabled) return;
                                            if (tp_tap_enabled())
                                            {
                                                for (auto& t : tp.touches) // On resume, all touches are considered palms.
                                                {
                                                    if (t.state != TOUCH_NONE)
                                                    {
                                                        t.tap.is_palm = true;
                                                        t.tap.state = TAP_TOUCH_STATE_DEAD;
                                                    }
                                                }
                                                tp.tap.state = TAP_STATE_IDLE;
                                                tp.tap.nfingers_down = 0;
                                            }
                                            else
                                            {
                                                tp_release_all_taps(stamp);
                                            }
                                        }
                                    void tp_tap_suspend(time stamp)
                                    {
                                        tp_tap_enabled_update(true, tp.tap.tap_state_enabled, stamp);
                                    }
                                void tp_stop_actions(time stamp)
                                {
                                    tp_edge_scroll_stop_events(stamp);
                                    tp_gesture_cancel(stamp);
                                    tp_tap_suspend(stamp);
                                }
                            void tp_trackpoint_event(time stamp, libinput_event& event)
                            {
                                if (tp.palm.dwtp_enabled)
                                {
                                    if (event.type != LIBINPUT_EVENT_POINTER_BUTTON) // Buttons do not count as trackpad activity, as people may use the trackpoint buttons in combination with the touchpad.
                                    {
                                        tp.palm.trackpoint_last_event_time = stamp;
                                        tp.palm.trackpoint_event_count++;
                                        if (tp.palm.trackpoint_event_count < 3) // Require at least three events before enabling palm detection.
                                        {
                                            tp.palm.trackpoint_timer->start(stamp + lixx::default_trackpoint_event_timeout);
                                        }
                                        else
                                        {
                                            if (!tp.palm.trackpoint_active)
                                            {
                                                tp_stop_actions(stamp);
                                                tp.palm.trackpoint_active = true;
                                            }
                                            tp.palm.trackpoint_timer->start(stamp + lixx::default_trackpoint_activity_timeout);
                                        }
                                    }
                                }
                            }
                        void tp_pair_trackpoint(libinput_device_sptr trackpoint_li_device)
                        {
                            auto bus_trp = trackpoint_li_device->libevdev_get_id_bustype();
                            bool tp_is_internal, trp_is_internal;
                            if (trackpoint_li_device->device_tags & EVDEV_TAG_TRACKPOINT)
                            {
                                tp_is_internal = !!(tp.device_tags & EVDEV_TAG_INTERNAL_TOUCHPAD);
                                trp_is_internal = bus_trp != BUS_USB && bus_trp != BUS_BLUETOOTH;
                                if (!tp.buttons.trackpoint_li_device && tp_is_internal && trp_is_internal)
                                {
                                    // Don't send any pending releases to the new trackpoint.
                                    tp.buttons.active_is_topbutton = faux;
                                    tp.buttons.trackpoint_li_device = trackpoint_li_device;
                                    if (tp.palm.monitor_trackpoint)
                                    {
                                        if (!tp.palm.trackpoint_listener)
                                        {
                                            tp.palm.trackpoint_listener = ptr::shared<libinput_event_listener>([&](time stamp, libinput_event& event){ tp_trackpoint_event(stamp, event); });
                                        }
                                        trackpoint_li_device->libinput_device_add_event_listener(tp.palm.trackpoint_listener);
                                    }
                                }
                            }
                        }
                            bool tp_want_dwt(libinput_device_sptr keyboard_li_device)
                            {
                                auto vendor_tp   = tp.libevdev_get_id_vendor();
                                auto vendor_kbd  = keyboard_li_device->libevdev_get_id_vendor();
                                auto product_tp  = tp.libevdev_get_id_product();
                                auto product_kbd = keyboard_li_device->libevdev_get_id_product();
                                // External touchpads with the same vid/pid as the keyboard are considered a happy couple.
                                if (tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD) return vendor_tp == vendor_kbd && product_tp == product_kbd;
                                if (keyboard_li_device->device_tags & EVDEV_TAG_INTERNAL_KEYBOARD) return true;
                                // Keyboard is not tagged as internal keyboard and it's not part of a combo.
                                return faux;
                            }
                                    bool tp_key_is_modifier(ui32 keycode)
                                    {
                                        switch (keycode)
                                        {
                                            case KEY_LEFTCTRL: // Ignore modifiers to be responsive to ctrl-click, alt-tab, etc.
                                            case KEY_RIGHTCTRL:
                                            case KEY_LEFTALT:
                                            case KEY_RIGHTALT:
                                            case KEY_LEFTSHIFT:
                                            case KEY_RIGHTSHIFT:
                                            case KEY_FN:
                                            case KEY_CAPSLOCK:
                                            case KEY_TAB:
                                            case KEY_COMPOSE:
                                            case KEY_RIGHTMETA:
                                            case KEY_LEFTMETA:
                                                return true;
                                            default:
                                                return faux;
                                        }
                                    }
                                bool tp_key_ignore_for_dwt(ui32 keycode)
                                {
                                    if (tp_key_is_modifier(keycode))
                                    {
                                        return faux; // Ignore keys not part of the "typewriter set", i.e. F-keys, multimedia keys, numpad, etc.
                                    }
                                    else if (keycode == KEY_ESC || keycode == KEY_KPASTERISK)
                                    {
                                        return true;
                                    }
                                    else
                                    {
                                        return keycode >= KEY_F1;
                                    }
                                }
                                bool tp_key_is_shift(ui32 keycode)
                                {
                                    return keycode == KEY_LEFTSHIFT || keycode == KEY_RIGHTSHIFT;
                                }
                            void tp_keyboard_event(time stamp, libinput_event& event)
                            {
                                auto timeout = span{};
                                auto is_modifier = faux;
                                if (event.type != LIBINPUT_EVENT_KEYBOARD_KEY) return;
                                //auto& kbdev = event.libinput_event_get_keyboard_event();
                                auto key = event.libinput_event_keyboard_get_key();
                                // Only trigger the timer on key down.
                                if (event.libinput_event_keyboard_get_key_state() != evdev::pressed)
                                {
                                    tp.dwt.key_mask.reset(key);
                                    tp.dwt.mod_mask.reset(key);
                                    return;
                                }
                                if (!tp.dwt.dwt_enabled) return;
                                if (tp_key_ignore_for_dwt(key)) return;
                                // Modifier keys don't trigger disable-while-typing so things like ctrl+zoom or ctrl+click are possible.
                                // The exception is shift which we don't trigger DWT for on its own but we do trigger DWT for once we type some other key.
                                is_modifier = tp_key_is_modifier(key);
                                if (is_modifier)
                                {
                                    if (!tp_key_is_shift(key))
                                    {
                                        tp.dwt.mod_mask.set(key);
                                    }
                                }
                                else
                                {
                                    if (!tp.dwt.keyboard_active)
                                    {
                                        // This is the first non-modifier key press. Check if the modifier mask is set. If any modifier is down we don't trigger dwt because it's likely to be combination like Ctrl+S or similar.
                                        if (tp.dwt.mod_mask.any()) return;
                                        tp_stop_actions(stamp);
                                        tp.dwt.keyboard_active = true;
                                        timeout = lixx::default_keyboard_activity_timeout_1;
                                    }
                                    else
                                    {
                                        timeout = lixx::default_keyboard_activity_timeout_2;
                                    }
                                    tp.dwt.keyboard_last_press_time = stamp;
                                    tp.dwt.key_mask.set(key);
                                    tp.dwt.keyboard_timer->start(stamp + timeout);
                                }
                            }
                        void tp_dwt_pair_keyboard(libinput_device_sptr keyboard_li_device)
                        {
                            if (!(keyboard_li_device->device_tags & EVDEV_TAG_KEYBOARD)) return;
                            if (!tp_want_dwt(keyboard_li_device)) return;
                            if (tp.dwt.paired_keyboard_list.size() > 3)
                            {
                                log("too many internal keyboards for dwt");
                            }
                            auto kbd = ptr::shared<libinput_paired_keyboard>();
                            kbd->li_device = keyboard_li_device;
                            kbd->listener = ptr::shared<libinput_event_listener>([&](time stamp, libinput_event& event){ tp_keyboard_event(stamp, event); });
                            keyboard_li_device->libinput_device_add_event_listener(kbd->listener);
                            tp.dwt.paired_keyboard_list.push_back(kbd);
                            log("palm: dwt activated with %s%<->%s%", tp.ud_device.devname, keyboard_li_device->ud_device.devname);
                        }
                                    void tp_init_top_softbuttons(fp64 topbutton_size_mult)
                                    {
                                        if (tp.buttons.has_topbuttons)
                                        {
                                            // T440s has the top button line 5mm from the top, event analysis has shown events to start down to ~10mm from the top - which maps to 15%.  We allow the caller to enlarge the area using a multiplier for the touchpad disabled case.
                                            auto topsize_mm = 10 * topbutton_size_mult;
                                            auto [w, h] = tp.evdev_device_get_size();
                                            auto mm = fp64_coor{};
                                            mm.x = w * 0.60;
                                            mm.y = topsize_mm;
                                            auto edges = tp.evdev_device_mm_to_units(mm);
                                            tp.buttons.top_area.bottom_edge = edges.y;
                                            tp.buttons.top_area.rightbutton_left_edge = edges.x;
                                            mm.x = w * 0.40;
                                            edges = tp.evdev_device_mm_to_units(mm);
                                            tp.buttons.top_area.leftbutton_right_edge = edges.x;
                                        }
                                        else
                                        {
                                            tp.buttons.top_area.bottom_edge = si32min;
                                        }
                                    }
                                        void tp_sync_touch(tp_touch& t, si32 slot)
                                        {
                                            auto tracking_id = 0;
                                            if (!tp.libevdev_fetch_slot_value(slot, ABS_MT_POSITION_X, t.point.x))
                                            {
                                                t.point.x = tp.libevdev_get_event_value<EV_ABS>(ABS_X);
                                            }
                                            if (!tp.libevdev_fetch_slot_value(slot, ABS_MT_POSITION_Y, t.point.y))
                                            {
                                                t.point.y = tp.libevdev_get_event_value<EV_ABS>(ABS_Y);
                                            }
                                            if (!tp.libevdev_fetch_slot_value(slot, ABS_MT_PRESSURE, t.pressure))
                                            {
                                                t.pressure = tp.libevdev_get_event_value<EV_ABS>(ABS_PRESSURE);
                                            }
                                            tp.libevdev_fetch_slot_value(slot, ABS_MT_TOUCH_MAJOR, t.touch_limits.max);
                                            tp.libevdev_fetch_slot_value(slot, ABS_MT_TOUCH_MINOR, t.touch_limits.min);
                                            if (tp.libevdev_fetch_slot_value(slot, ABS_MT_TRACKING_ID, tracking_id) && tracking_id != -1)
                                            {
                                                tp.nactive_slots++;
                                            }
                                        }
                                    void tp_sync_slots()
                                    {
                                        // Always sync the first touch so we get ABS_X/Y synced on single-touch touchpads.
                                        tp_sync_touch(tp.touches[0], 0);
                                        for (auto i = 1u; i < tp.num_slots; i++)
                                        {
                                            tp_sync_touch(tp.touches[i], i);
                                        }
                                    }
                                void tp_resume(suspend_trigger trigger)
                                {
                                    tp.suspend_reason &= ~trigger;
                                    if (tp.suspend_reason == 0)
                                    {
                                        if (tp.buttons.has_topbuttons)
                                        {
                                            // Tap state-machine is offline while suspended, reset state.
                                            tp_clear_state();
                                            // Restore original topbutton area size.
                                            tp_init_top_softbuttons(1.0);
                                            tp.evdev_notify_resumed_device();
                                        }
                                        else
                                        {
                                            tp.evdev_device_resume();
                                        }
                                        tp_sync_slots();
                                    }
                                }
                                void tp_suspend(suspend_trigger trigger)
                                {
                                    if (tp.suspend_reason & trigger) return;
                                    if (tp.suspend_reason == 0)
                                    {
                                        tp_clear_state();
                                        // On devices with top softwarebuttons we don't actually suspend the device, to keep the "trackpoint" buttons working. tp_post_events() will only send events for the trackpoint while suspended.
                                        if (tp.buttons.has_topbuttons)
                                        {
                                            tp.evdev_notify_suspended_device();
                                            // Enlarge topbutton area while suspended.
                                            tp_init_top_softbuttons(3.0);
                                        }
                                        else
                                        {
                                            tp.evdev_device_suspend();
                                        }
                                    }
                                    tp.suspend_reason |= trigger;
                                }
                            void tp_lid_switch_event(libinput_event& event)
                            {
                                if (event.type == LIBINPUT_EVENT_SWITCH_TOGGLE)
                                {
                                    if (event.libinput_event_switch_get_switch() == LIBINPUT_SWITCH_LID)
                                    {
                                        auto state = event.libinput_event_switch_get_switch_state();
                                        if (state)
                                        {
                                            tp_suspend(SUSPEND_LID);
                                            log("lid: suspending touchpad");
                                        }
                                        else
                                        {
                                            tp_resume(SUSPEND_LID);
                                            log("lid: resume touchpad");
                                        }
                                    }
                                }
                            }
                        void tp_pair_lid_switch(libinput_device_sptr lid_switch_li_device)
                        {
                            if (lid_switch_li_device->device_tags & EVDEV_TAG_LID_SWITCH
                             && !(tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD)
                             && !tp.lid_switch.lid_switch_li_device)
                            {
                                log("lid: activated for %s%<->%s%", tp.ud_device.devname, lid_switch_li_device->ud_device.devname);
                                if (!tp.lid_switch.listener)
                                {
                                    tp.lid_switch.listener = ptr::shared<libinput_event_listener>([&](time, libinput_event& event){ tp_lid_switch_event(event); });
                                }
                                lid_switch_li_device->libinput_device_add_event_listener(tp.lid_switch.listener);
                                tp.lid_switch.lid_switch_li_device = lid_switch_li_device;
                            }
                        }
                            void tp_tablet_mode_switch_event(libinput_event& event)
                            {
                                if (event.type == LIBINPUT_EVENT_SWITCH_TOGGLE)
                                {
                                    if (event.libinput_event_switch_get_switch() == LIBINPUT_SWITCH_TABLET_MODE)
                                    {
                                        if (event.libinput_event_switch_get_switch_state())
                                        {
                                            tp_suspend(SUSPEND_TABLET_MODE);
                                            log("tablet-mode: suspending touchpad");
                                        }
                                        else
                                        {
                                            tp_resume(SUSPEND_TABLET_MODE);
                                            log("tablet-mode: resume touchpad");
                                        }
                                    }
                                }
                            }
                        void tp_pair_tablet_mode_switch(libinput_device_sptr tablet_mode_switch_li_device)
                        {
                            if (!(tablet_mode_switch_li_device->device_tags & EVDEV_TAG_TABLET_MODE_SWITCH)) return;
                            if (tp.tablet_mode_switch.tablet_mode_switch_li_device) return;
                            if (tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD) return;
                            if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_TABLET_MODE_NO_SUSPEND)) return;
                            log("tablet-mode: activated for %s%<->%s%", tp.ud_device.devname, tablet_mode_switch_li_device->ud_device.devname);
                            if (!tp.tablet_mode_switch.listener)
                            {
                                tp.tablet_mode_switch.listener = ptr::shared<libinput_event_listener>([&](time, libinput_event& event){ tp_tablet_mode_switch_event(event); });
                            }
                            tablet_mode_switch_li_device->libinput_device_add_event_listener(tp.tablet_mode_switch.listener);
                            tp.tablet_mode_switch.tablet_mode_switch_li_device = tablet_mode_switch_li_device;
                            if (tablet_mode_switch_li_device->get_switch_state(LIBINPUT_SWITCH_TABLET_MODE))
                            {
                                tp_suspend(SUSPEND_TABLET_MODE);
                            }
                        }
                            void tp_change_rotation(bool notify)
                            {
                                auto tablet_li_device = tp.left_handed.tablet_li_device;
                                bool tablet_is_left, touchpad_is_left;
                                if (tp.left_handed.must_rotate)
                                {
                                    touchpad_is_left = tp.dev_left_handed.enabled;
                                    tablet_is_left = tp.left_handed.tablet_left_handed_state;
                                    tp.left_handed.want_rotate = touchpad_is_left || tablet_is_left;
                                    tp_apply_rotation();
                                    if (notify && tablet_li_device)
                                    {
                                        tablet_li_device->left_handed_toggle(tp.left_handed.want_rotate);
                                    }
                                }
                            }
                        void tp_pair_tablet(libinput_device_sptr tablet_li_device)
                        {
                            if (tp.left_handed.must_rotate && (tablet_li_device->device_caps & EVDEV_DEVICE_TABLET))
                            if (tp.device_group.size() && tp.device_group == tablet_li_device->device_group)
                            {
                                tp.left_handed.tablet_li_device = tablet_li_device;
                                log("touchpad-rotation: %s% will rotate %s%", tp.ud_device.devname, tablet_li_device->ud_device.devname);
                                if (tablet_li_device->libinput_device_config_left_handed_get())
                                {
                                    tp.left_handed.want_rotate = true;
                                    tp.left_handed.tablet_left_handed_state = true;
                                    tp_change_rotation(faux);
                                }
                            }
                        }
            void tp_interface_device_added(libinput_device_sptr added_li_device)
            {
                tp_pair_trackpoint(        added_li_device);
                tp_dwt_pair_keyboard(      added_li_device);
                tp_pair_lid_switch(        added_li_device);
                tp_pair_tablet_mode_switch(added_li_device);
                tp_pair_tablet(            added_li_device);
                if (tp.sendevents_current_mode == LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE)
                {
                    if (added_li_device->device_tags & EVDEV_TAG_EXTERNAL_MOUSE)
                    {
                        tp_suspend(SUSPEND_EXTERNAL_MOUSE);
                    }
                }
            }
            void tp_interface_device_removed(libinput_device_sptr removed_li_device)
            {
                if (removed_li_device == tp.buttons.trackpoint_li_device) // Clear any pending releases for the trackpoint.
                {
                    if (tp.buttons.active != 0 && tp.buttons.active_is_topbutton)
                    {
                        tp.buttons.active = {};
                        tp.buttons.active_is_topbutton = faux;
                    }
                    if (tp.palm.monitor_trackpoint)
                    {
                        tp.buttons.trackpoint_li_device->libinput_device_remove_event_listener(tp.palm.trackpoint_listener);
                    }
                    tp.buttons.trackpoint_li_device = {};
                }
                std::erase_if(tp.dwt.paired_keyboard_list, [&](auto kbd)
                {
                    auto found = kbd->li_device == removed_li_device;
                    if (found)
                    {
                        kbd->li_device->libinput_device_remove_event_listener(kbd->listener);
                        tp.dwt.keyboard_active = faux;
                    }
                    return found;
                });
                if (removed_li_device == tp.lid_switch.lid_switch_li_device)
                {
                    tp.lid_switch.lid_switch_li_device->libinput_device_remove_event_listener(tp.lid_switch.listener);
                    tp.lid_switch.lid_switch_li_device.reset();
                    tp_resume(SUSPEND_LID);
                }
                if (removed_li_device == tp.tablet_mode_switch.tablet_mode_switch_li_device)
                {
                    tp.tablet_mode_switch.tablet_mode_switch_li_device->libinput_device_remove_event_listener(tp.tablet_mode_switch.listener);
                    tp.tablet_mode_switch.tablet_mode_switch_li_device.reset();
                    tp_resume(SUSPEND_TABLET_MODE);
                }
                if (tp.sendevents_current_mode == LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE)
                {
                    auto found = faux;
                    for (auto d : tp.li.device_list)
                    {
                        if (d != removed_li_device && (d->device_tags & EVDEV_TAG_EXTERNAL_MOUSE))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found) tp_resume(SUSPEND_EXTERNAL_MOUSE);
                }
                if (removed_li_device == tp.left_handed.tablet_li_device)
                {
                    tp.left_handed.tablet_li_device = {};
                    tp.left_handed.tablet_left_handed_state = faux;
                    tp_change_rotation(true); // Slight awkwardness: removing the tablet causes the touchpad to rotate back to normal if only the tablet was set to left-handed. Niche case, nothing to worry about.
                }
            }
            void tp_interface_toggle_touch(libinput_arbitration_state which, time stamp)
            {
                if (which == tp.arbitration.state) return;
                switch (which)
                {
                    case ARBITRATION_IGNORE_ALL:
                    case ARBITRATION_IGNORE_RECT:
                        tp.arbitration.arbitration_timer->cancel();
                        tp_clear_state();
                        tp.arbitration.state = which;
                        break;
                    case ARBITRATION_NOT_ACTIVE:
                        // If in-kernel arbitration is in use and there is a touch and a pen in proximity, lifting the pen out of proximity causes a touch begin for the touch. On a hand-lift the proximity out precedes the touch up by a few ms, so we get what looks like a tap. Fix this by delaying arbitration by just a little bit so that any touch in event is caught as palm touch.
                        tp.arbitration.arbitration_timer->start(stamp + 90ms);
                        break;
                }
            }
            void touchpad_left_handed_toggled(bool left_handed_enabled)
            {
                if (tp.left_handed.tablet_li_device) // Called when the tablet toggles to left-handed.
                {
                    log("touchpad-rotation: tablet is %s%", left_handed_enabled ? "left-handed" : "right-handed");
                    // Our left-handed config is independent even though rotation is locked. So we rotate when either device is left-handed. But it can only be actually changed when the device is in a neutral state, hence the want_rotate.
                    tp.left_handed.tablet_left_handed_state = left_handed_enabled;
                    tp_change_rotation(faux);
                }
            }
                bool tp_init_slots()
                {
                    auto n_btn_tool_touches = 1u;
                    auto& absinfo = tp.libevdev_get_abs_info(ABS_MT_SLOT);
                    if (absinfo)
                    {
                        tp.num_slots = absinfo.maximum + 1;
                        tp.slot = absinfo.value;
                        tp.has_mt = true;
                    }
                    else
                    {
                        tp.num_slots = 1;
                        tp.slot = 0;
                        tp.has_mt = faux;
                    }
                    tp.semi_mt = tp.libevdev_has_property(INPUT_PROP_SEMI_MT);
                    // Semi-mt devices are not reliable for true multitouch data, so we
                    // simply pretend they're single touch touchpads with BTN_TOOL bits.
                    // Synaptics:
                    // Terrible resolution when two fingers are down,
                    // causing scroll jumps. The single-touch emulation ABS_X/Y is
                    // accurate but the ABS_MT_POSITION touchpoints report the bounding
                    // box and that causes jumps. See https://bugzilla.redhat.com/1235175
                    // Elantech:
                    // On three-finger taps/clicks, one slot doesn't get a coordinate
                    // assigned. See https://bugs.freedesktop.org/show_bug.cgi?id=93583
                    // Alps:
                    // If three fingers are set down in the same frame, one slot has the
                    // coordinates 0/0 and may not get updated for several frames.
                    // See https://bugzilla.redhat.com/show_bug.cgi?id=1295073
                    //
                    // The HP Pavilion DM4 touchpad has random jumps in slots, including
                    // for single-finger movement. See fdo bug 91135.
                    if (tp.semi_mt || tp.evdev_device_has_model_quirk(QUIRK_MODEL_HP_PAVILION_DM4_TOUCHPAD))
                    {
                        tp.num_slots = 1;
                        tp.slot = 0;
                        tp.has_mt = faux;
                    }
                    if (!tp.has_mt)
                    {
                        tp.tp_disable_abs_mt();
                    }
                    static constexpr auto max_touches = std::to_array<std::pair<ui32, si32>>(
                    {
                        { BTN_TOOL_QUINTTAP , 5 },
                        { BTN_TOOL_QUADTAP  , 4 },
                        { BTN_TOOL_TRIPLETAP, 3 },
                        { BTN_TOOL_DOUBLETAP, 2 },
                    });
                    for (auto [code, ntouches] : max_touches)
                    {
                        if (tp.libevdev_has_event_code<EV_KEY>(code))
                        {
                            n_btn_tool_touches = ntouches;
                            break;
                        }
                    }
                    tp.ntouches = std::max(tp.num_slots, n_btn_tool_touches);
                    tp.touches.resize(tp.ntouches);
                    auto self = tp.This<tp_device>();
                    auto i = 0u;
                    for (auto& t : tp.touches)
                    {
                        t.tp        = self;
                        t.has_ended = true;
                        t.index     = i++;
                    }
                    tp_sync_slots();
                    // Some touchpads don't reset BTN_TOOL_FINGER on touch up and only change to/from it when BTN_TOOL_DOUBLETAP is set. This causes us to ignore the first touches events until a two-finger gesture is performed.
                    if (tp.libevdev_get_event_value<EV_KEY>(BTN_TOOL_FINGER))
                    {
                        tp_fake_finger_set(evdev::btn_tool_finger, true);
                    }
                    return true;
                }
                bool tp_init_touch_size()
                {
                    auto rc = faux;
                    if (!tp.libevdev_has_event_code<EV_ABS>(ABS_MT_TOUCH_MAJOR))
                    {
                        return faux;
                    }
                    if (auto q = tp.li.quirks_fetch_for_device(tp.ud_device))
                    {
                        auto r = si32_range{};
                        if (q->quirks_get(QUIRK_ATTR_TOUCH_SIZE_RANGE, r))
                        {
                            auto hi = r.max;
                            auto lo = r.min;
                            if (tp.ud_device.num_slots < 5)
                            {
                                log("Expected 5+ slots for touch size detection");
                            }
                            else
                            {
                                if (hi == 0 && lo == 0)
                                {
                                    log("touch size based touch detection disabled");
                                }
                                else // Thresholds apply for both major or minor.
                                {
                                    tp.touch_size.low = lo;
                                    tp.touch_size.high = hi;
                                    tp.touch_size.use_touch_size = true;
                                    log("using size-based touch detection (%d%:%d%)", hi, lo);
                                    rc = true;
                                }
                            }
                        }
                    }
                    return rc;
                }
                void tp_init_pressure()
                {
                    auto code = tp.has_mt ? ABS_MT_PRESSURE : ABS_PRESSURE;
                    if (!tp.libevdev_has_event_code<EV_ABS>(code))
                    {
                        tp.pressure.use_pressure = faux;
                        return;
                    }
                    auto& abs = tp.libevdev_get_abs_info(code);
                    assert(abs);
                    auto q = tp.li.quirks_fetch_for_device(tp.ud_device);
                    auto r = si32_range{};
                    auto hi = 0;
                    auto lo = 0;
                    if (q && q->quirks_get(QUIRK_ATTR_PRESSURE_RANGE, r))
                    {
                        hi = r.max;
                        lo = r.min;
                        if (hi == 0 && lo == 0)
                        {
                            log("pressure-based touch detection disabled");
                            return;
                        }
                    }
                    else // Approximately the synaptics defaults.
                    {
                        auto range = abs.absinfo_range();
                        hi = abs.minimum + 0.12 * range;
                        lo = abs.minimum + 0.10 * range;
                    }
                    if (hi > abs.maximum || hi < abs.minimum || lo > abs.maximum || lo < abs.minimum)
                    {
                        log("discarding out-of-bounds pressure range %d%:%d%", hi, lo);
                    }
                    else
                    {
                        tp.pressure.use_pressure = true;
                        tp.pressure.high = hi;
                        tp.pressure.low = lo;
                        log("using pressure-based touch detection (%d%:%d%)", lo, hi);
                    }
                }
                void tp_init_hysteresis()
                {
                    auto xmargin = 0;
                    auto ymargin = 0;
                    auto& ax = tp.ud_device.abs.absinfo_x;
                    auto& ay = tp.ud_device.abs.absinfo_y;
                    xmargin = ax.fuzz ? ax.fuzz : ax.resolution / 4;
                    ymargin = ay.fuzz ? ay.fuzz : ay.resolution / 4;
                    tp.hysteresis.margin.x = xmargin;
                    tp.hysteresis.margin.y = ymargin;
                    tp.hysteresis.enabled = (ax.fuzz || ay.fuzz);
                    if (tp.hysteresis.enabled) log("hysteresis enabled");
                }
                        libinput_config_status tp_accel_config_set_speed(fp64 speed)
                        {
                            auto ok = tp.pointer_filter->filter_set_speed(speed);
                            return ok ? LIBINPUT_CONFIG_STATUS_SUCCESS : LIBINPUT_CONFIG_STATUS_INVALID;
                        }
                    static libinput_config_status tp_accel_config_set_profile(libinput_device_sptr li_device, libinput_config_accel_profile profile)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        auto& filter = li_device->pointer_filter;
                        if (filter->filter_get_type() != profile)
                        {
                            auto speed = filter->filter_get_speed();
                            tp.tp_impl.tp_init_accel(profile);
                            tp.tp_impl.tp_accel_config_set_speed(speed);
                        }
                        //todo always true
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                void tp_init_accel(libinput_config_accel_profile which)
                {
                    auto filter = motion_filter_sptr{};
                    auto dpi = tp.dpi;
                    auto use_v_avg = tp.use_velocity_averaging;
                    auto res_x = tp.ud_device.abs.absinfo_x.resolution;
                    auto res_y = tp.ud_device.abs.absinfo_y.resolution;
                    // Not all touchpads report the same amount of units/mm (resolution).
                    // Normalize motion events to the default mouse DPI as base (unaccelerated) speed. This also evens out any differences in x and y resolution, so that a circle on the touchpad does not turn into an ellipse on the screen.
                    tp.accel_scale_coeff = { (lixx::default_mouse_dpi / 25.4) / res_x, (lixx::default_mouse_dpi / 25.4) / res_y };
                    tp.accel_xy_scale_coeff = 1.0 * res_x / res_y;
                    if (which == LIBINPUT_CONFIG_ACCEL_PROFILE_FLAT)
                    {
                        filter = ptr::shared<touchpad_accelerator_flat>(dpi);
                    }
                    else if (which == LIBINPUT_CONFIG_ACCEL_PROFILE_CUSTOM)
                    {
                        filter = ptr::shared<custom_accelerator>();
                    }
                    else if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_LENOVO_X230)
                          || tp.ud_device.model_flags & EVDEV_MODEL_LENOVO_X220_TOUCHPAD_FW81)
                    {
                        filter = ptr::shared<pointer_accelerator_x230>(dpi, use_v_avg);
                    }
                    else
                    {
                        auto eds_threshold = span{};
                        auto eds_value     = span{};
                        if (tp.libevdev_get_id_bustype() == BUS_BLUETOOTH)
                        {
                            eds_threshold = 50ms;
                            eds_value     = 10ms;
                        }
                        filter = ptr::shared<touchpad_accelerator>(dpi, eds_threshold, eds_value, use_v_avg);
                    }
                    tp.evdev_device_init_pointer_acceleration(filter);
                    tp.pointer_config.set_profile = tp_accel_config_set_profile;
                }
                    void tp_tap_handle_timeout(time now)
                    {
                        static auto empty_touch = tp_touch{}; // This touch is not dereferenced anywhere. This just is a placeholder.
                        tp.tp_impl.tp_tap_handle_event(empty_touch, TAP_EVENT_TIMEOUT, now);
                        for (auto& t : tp.touches)
                        {
                            if (t.state != TOUCH_NONE && t.tap.state != TAP_TOUCH_STATE_IDLE)
                            {
                                t.tap.state = TAP_TOUCH_STATE_DEAD;
                            }
                        }
                    }
                    static si32 tp_tap_config_count(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return std::min(tp.ntouches, 3U); // We only do up to 3 finger tap.
                    }
                    static libinput_config_status tp_tap_config_set_enabled(libinput_device_sptr li_device, bool tap_state_enabled)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.tp_impl.tp_tap_enabled_update(tp.tap.suspended, tap_state_enabled, datetime::now());
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_tap_config_is_enabled(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tap.tap_state_enabled;
                    }
                    static bool tp_tap_config_get_default(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tp_impl.tp_tap_default();
                    }
                    static libinput_config_status tp_tap_config_set_map(libinput_device_sptr li_device, bool use_lmr_map)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.tap.want_use_lmr_map = use_lmr_map;
                        tp.tp_impl.tp_tap_update_map();
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_tap_config_get_map(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tap.want_use_lmr_map;
                    }
                    static bool tp_tap_config_get_default_map([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return faux;
                    }
                    static libinput_config_status tp_tap_config_set_drag_enabled(libinput_device_sptr li_device, bool drag_enbled)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.tap.drag_enabled = drag_enbled;
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_tap_config_get_drag_enabled(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tap.drag_enabled;
                    }
                    static bool tp_drag_default([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return true;
                    }
                    static bool tp_tap_config_get_default_drag_enabled(libinput_device_sptr li_device)
                    {
                        return tp_drag_default(li_device);
                    }
                    static libinput_config_status tp_tap_config_set_draglock_enabled(libinput_device_sptr li_device, libinput_config_drag_lock_state enabled)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.tap.drag_lock = enabled;
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static libinput_config_drag_lock_state tp_tap_config_get_draglock_enabled(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tap.drag_lock;
                    }
                    static libinput_config_drag_lock_state tp_drag_lock_default([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return LIBINPUT_CONFIG_DRAG_LOCK_DISABLED;
                    }
                    static libinput_config_drag_lock_state tp_tap_config_get_default_draglock_enabled(libinput_device_sptr li_device)
                    {
                        return tp_drag_lock_default(li_device);
                    }
                void tp_init_tap()
                {
                    tp.tap.config.count                        = tp_tap_config_count;
                    tp.tap.config.set_enabled                  = tp_tap_config_set_enabled;
                    tp.tap.config.get_enabled                  = tp_tap_config_is_enabled;
                    tp.tap.config.get_default                  = tp_tap_config_get_default;
                    tp.tap.config.set_map                      = tp_tap_config_set_map;
                    tp.tap.config.get_map                      = tp_tap_config_get_map;
                    tp.tap.config.get_default_map              = tp_tap_config_get_default_map;
                    tp.tap.config.set_drag_enabled             = tp_tap_config_set_drag_enabled;
                    tp.tap.config.get_drag_enabled             = tp_tap_config_get_drag_enabled;
                    tp.tap.config.get_default_drag_enabled     = tp_tap_config_get_default_drag_enabled;
                    tp.tap.config.set_draglock_enabled         = tp_tap_config_set_draglock_enabled;
                    tp.tap.config.get_draglock_enabled         = tp_tap_config_get_draglock_enabled;
                    tp.tap.config.get_default_draglock_enabled = tp_tap_config_get_default_draglock_enabled;
                    tp.config.tap = &tp.tap.config;
                    tp.tap.state = TAP_STATE_IDLE;
                    tp.tap.tap_state_enabled = tp_tap_default();
                    tp.tap.use_lmr_map = faux;
                    tp.tap.want_use_lmr_map = tp.tap.use_lmr_map;
                    tp.tap.drag_enabled = tp_drag_default(tp.This());
                    tp.tap.drag_lock = tp_drag_lock_default(tp.This());
                    auto timer_name = utf::fprint("%s% tap", tp.ud_device.sysname);
                    tp.tap.timer = tp.li.timers.create(timer_name, [&](time now){ tp_tap_handle_timeout(now); });
                }
                    bool tp_guess_clickpad()
                    {
                        auto has_left    = tp.libevdev_has_event_code<EV_KEY>(BTN_LEFT);
                        auto has_middle  = tp.libevdev_has_event_code<EV_KEY>(BTN_MIDDLE);
                        auto has_right   = tp.libevdev_has_event_code<EV_KEY>(BTN_RIGHT);
                        auto is_clickpad = tp.libevdev_has_property(INPUT_PROP_BUTTONPAD);
                        // A non-clickpad without a right button is a clickpad, assume the kernel is wrong.
                        // Exceptions here:
                        // - The one-button Apple touchpad (discontinued in 2008) has a single physical button.
                        // - Wacom touch devices have neither left nor right buttons.
                        if (!is_clickpad && has_left && !has_right && !(tp.ud_device.model_flags & EVDEV_MODEL_APPLE_TOUCHPAD_ONEBUTTON))
                        {
                            log("missing right button, assuming it is a clickpad");
                            is_clickpad = true;
                        }
                        if (has_middle || has_right)
                        {
                            if (is_clickpad) log("clickpad advertising right button");
                        }
                        else if (has_left && !is_clickpad && tp.libevdev_get_id_vendor() != lixx::vendor_id_apple)
                        {
                                log("non clickpad without right button?");
                        }
                        return is_clickpad;
                    }
                        void tp_switch_click_method()
                        {
                            // All we need to do when switching click methods is to change the
                            // bottom_area.top_edge so that when in clickfinger mode the bottom
                            // touchpad area is not dead wrt finger movement starting there.
                            //
                            // We do not need to take any state into account, fingers which are
                            // already down will simply keep the state / area they have assigned
                            // until they are released, and the post_button_events path is state
                            // agnostic.
                            switch (tp.buttons.click_method)
                            {
                                case LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS: tp_init_softbuttons(); break;
                                case LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER:
                                case LIBINPUT_CONFIG_CLICK_METHOD_NONE: tp.buttons.bottom_area.top_edge = si32max; break;
                            }
                        }
                    libinput_config_click_method tp_click_get_default_method()
                    {
                        if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_CHROMEBOOK)
                         || tp.evdev_device_has_model_quirk(QUIRK_MODEL_SYSTEM76_BONOBO)
                         || tp.evdev_device_has_model_quirk(QUIRK_MODEL_SYSTEM76_GALAGO)
                         || tp.evdev_device_has_model_quirk(QUIRK_MODEL_SYSTEM76_KUDU)
                         || tp.evdev_device_has_model_quirk(QUIRK_MODEL_CLEVO_W740SU)
                         || tp.evdev_device_has_model_quirk(QUIRK_MODEL_APPLE_TOUCHPAD_ONEBUTTON))
                        {
                            return LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER;
                        }
                        if (!tp.buttons.is_clickpad)
                        {
                            return LIBINPUT_CONFIG_CLICK_METHOD_NONE;
                        }
                        if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_APPLE_TOUCHPAD))
                        {
                            return LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER;
                        }
                        return LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS;
                    }
                void tp_init_middlebutton_emulation()
                {
                    auto enable_by_default = faux;
                    auto want_config_option = faux;
                    // On clickpads we provide the config option but disable by default. When enabled, the middle software button disappears.
                    if (tp.buttons.is_clickpad)
                    {
                        tp_init_clickpad_middlebutton_emulation();
                        return;
                    }
                    // Init middle button emulation on non-clickpads, but only if we don't have a middle button. Exception: ALPS touchpads don't know if they have a middle button, so we always want the option there and enabled by default.
                    if (!tp.libevdev_has_event_code<EV_KEY>(BTN_MIDDLE))
                    {
                        enable_by_default = true;
                        want_config_option = faux;
                    }
                    else if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_ALPS_SERIAL_TOUCHPAD))
                    {
                        enable_by_default = true;
                        want_config_option = true;
                    }
                    else return;
                    tp.evdev_init_middlebutton(enable_by_default, want_config_option);
                }
                    static ui32 tp_button_config_click_get_methods(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        auto methods = (ui32)LIBINPUT_CONFIG_CLICK_METHOD_NONE;
                        if (tp.buttons.is_clickpad)
                        {
                            methods |= LIBINPUT_CONFIG_CLICK_METHOD_BUTTON_AREAS;
                            if (tp.has_mt) methods |= LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER;
                        }
                        if (li_device->ud_device.model_flags & EVDEV_MODEL_APPLE_TOUCHPAD_ONEBUTTON) methods |= LIBINPUT_CONFIG_CLICK_METHOD_CLICKFINGER;
                        return methods;
                    }
                    static libinput_config_status tp_button_config_click_set_method(libinput_device_sptr li_device, libinput_config_click_method method)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.buttons.click_method = method;
                        tp.tp_impl.tp_switch_click_method();
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static libinput_config_click_method tp_button_config_click_get_method(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.buttons.click_method;
                    }
                    static libinput_config_click_method tp_button_config_click_get_default_method(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tp_impl.tp_click_get_default_method();
                    }
                    static libinput_config_status tp_button_config_set_clickfinger_map(libinput_device_sptr li_device, bool use_lmr_map)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.buttons.want_use_lmr_map = use_lmr_map;
                        tp.tp_impl.tp_button_update_clickfinger_map();
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_button_config_get_clickfinger_map(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.buttons.want_use_lmr_map;
                    }
                    static bool tp_button_config_get_default_clickfinger_map([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return faux;
                    }
                void tp_init_buttons()
                {
                    tp.buttons.is_clickpad = tp_guess_clickpad();
                    tp.buttons.has_topbuttons = tp.libevdev_has_property(INPUT_PROP_TOPBUTTONPAD);
                    auto& absinfo_x = tp.ud_device.abs.absinfo_x;
                    auto& absinfo_y = tp.ud_device.abs.absinfo_y;
                    // Pinned-finger motion threshold, see tp_unpin_finger.
                    tp.buttons.motion_dist_scale_coeff                   = { 1.0 / absinfo_x.resolution, 1.0 / absinfo_y.resolution };
                    tp.buttons.config_method.get_methods                 = tp_button_config_click_get_methods;
                    tp.buttons.config_method.set_method                  = tp_button_config_click_set_method;
                    tp.buttons.config_method.get_method                  = tp_button_config_click_get_method;
                    tp.buttons.config_method.get_default_method          = tp_button_config_click_get_default_method;
                    tp.buttons.config_method.set_clickfinger_map         = tp_button_config_set_clickfinger_map;
                    tp.buttons.config_method.get_clickfinger_map         = tp_button_config_get_clickfinger_map;
                    tp.buttons.config_method.get_default_clickfinger_map = tp_button_config_get_default_clickfinger_map;
                    tp.config.click_method = &tp.buttons.config_method;
                    tp.buttons.use_lmr_map = faux;
                    tp.buttons.want_use_lmr_map = tp.buttons.use_lmr_map;
                    tp.buttons.click_method = tp_click_get_default_method();
                    tp_switch_click_method();
                    tp_init_top_softbuttons(1.0);
                    tp_init_middlebutton_emulation();
                    auto i = 0;
                    for (auto& t : tp.touches)
                    {
                        auto timer_name = utf::fprint("%s% (%d%) button", tp.ud_device.sysname, ++i);
                        t.button.state = BUTTON_STATE_NONE;
                        t.button.timer = tp.li.timers.create(timer_name, [&](time now){ tp_button_handle_event(t, BUTTON_EVENT_TIMEOUT, now); });
                    }
                }
                    static si32 tp_dwt_config_is_available([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return 1;
                    }
                    static libinput_config_status tp_dwt_config_set(libinput_device_sptr li_device, bool dwt_enabled)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.dwt.dwt_enabled = dwt_enabled;
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_dwt_config_get(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.dwt.dwt_enabled;
                    }
                    static bool tp_dwt_config_get_default(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.tp_impl.tp_dwt_default_enabled();
                    }
                    bool tp_dwt_default_enabled()
                    {
                        return true;
                    }
                    bool tp_is_tpkb_combo_below()
                    {
                        auto rc = faux;
                        auto prop = text{};
                        if (auto q = tp.li.quirks_fetch_for_device(tp.ud_device); q->quirks_get(QUIRK_ATTR_TPKBCOMBO_LAYOUT, prop))
                        {
                            rc = prop == "below"sv;
                        }
                        return rc;
                    }
                void tp_init_dwt()
                {
                    if (!(tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD) || tp_is_tpkb_combo_below())
                    {
                        tp.dwt.config.is_available        = tp_dwt_config_is_available;
                        tp.dwt.config.set_enabled         = tp_dwt_config_set;
                        tp.dwt.config.get_enabled         = tp_dwt_config_get;
                        tp.dwt.config.get_default_enabled = tp_dwt_config_get_default;
                        tp.dwt.dwt_enabled                = tp_dwt_default_enabled();
                        tp.config.dwt = &tp.dwt.config;
                    }
                }
                    bool tp_dwtp_default_enabled()
                    {
                        return true;
                    }
                    static si32 tp_dwtp_config_is_available([[maybe_unused]] libinput_device_sptr li_device)
                    {
                        return 1;
                    }
                    static libinput_config_status tp_dwtp_config_set(libinput_device_sptr li_device, bool dwtp_enabled)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        tp.palm.dwtp_enabled = dwtp_enabled;
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                    static bool tp_dwtp_config_get(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        return tp.palm.dwtp_enabled;
                    }
                    static bool tp_dwtp_config_get_default(libinput_device_sptr li_device)
                    {
                        auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                        auto dwtp = tp.tp_impl.tp_dwtp_default_enabled();
                        return dwtp;
                    }
                void tp_init_dwtp()
                {
                    tp.palm.dwtp_enabled = tp_dwtp_default_enabled();
                    if (!(tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD))
                    {
                        tp.palm.config.is_available        = tp_dwtp_config_is_available;
                        tp.palm.config.set_enabled         = tp_dwtp_config_set;
                        tp.palm.config.get_enabled         = tp_dwtp_config_get;
                        tp.palm.config.get_default_enabled = tp_dwtp_config_get_default;
                        tp.config.dwtp = &tp.palm.config;
                    }
                }
                        void tp_arbitration_timeout()
                        {
                            tp.arbitration.state = ARBITRATION_NOT_ACTIVE;
                        }
                    void tp_init_palmdetect_arbitration()
                    {
                        auto timer_name = utf::fprint("%s% arbitration", tp.ud_device.sysname);
                        tp.arbitration.arbitration_timer = tp.li.timers.create(timer_name, [&](time){ tp_arbitration_timeout(); });
                        tp.arbitration.state = ARBITRATION_NOT_ACTIVE;
                    }
                    void tp_init_palmdetect_edge()
                    {
                        if (tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD && !tp_is_tpkb_combo_below()) return;
                        // Edge palm detection hurts more than it helps on Apple touchpads.
                        if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_APPLE_TOUCHPAD)) return;
                        auto [w, h] = tp.evdev_device_get_size();
                        // Enable edge palm detection on touchpads >= 70 mm. Anything smaller probably won't need it, until we find out it does.
                        if (w < 70.0) return;
                        // Palm edges are 8% of the width on each side.
                        auto mm = fp64_coor{};
                        mm.x = std::min(8.0, w * 0.08);
                        auto edges = tp.evdev_device_mm_to_units(mm);
                        tp.palm.left_edge = edges.x;
                        mm.x = w - std::min(8.0, w * 0.08);
                        edges = tp.evdev_device_mm_to_units(mm);
                        tp.palm.right_edge = edges.x;
                        if (!tp.buttons.has_topbuttons && h > 55)
                        {
                            // Top edge is 5% of the height.
                            mm.y = h * 0.05;
                            edges = tp.evdev_device_mm_to_units(mm);
                            tp.palm.upper_edge = edges.y;
                        }
                    }
                        si32 tp_read_palm_pressure_prop()
                        {
                            static constexpr auto default_palm_threshold = ui32{ 130 };
                            auto threshold = default_palm_threshold;
                            if (auto q = tp.li.quirks_fetch_for_device(tp.ud_device))
                            {
                                q->quirks_get(QUIRK_ATTR_PALM_PRESSURE_THRESHOLD, threshold);
                            }
                            return threshold;
                        }
                    void tp_init_palmdetect_pressure()
                    {
                        if (!tp.libevdev_has_event_code<EV_ABS>(ABS_MT_PRESSURE))
                        {
                            tp.palm.use_pressure = faux;
                            return;
                        }
                        tp.palm.pressure_threshold = tp_read_palm_pressure_prop();
                        if (tp.palm.pressure_threshold != 0)
                        {
                            tp.palm.use_pressure = true;
                            log("palm: pressure threshold is %d%", tp.palm.pressure_threshold);
                        }
                    }
                    void tp_init_palmdetect_size()
                    {
                        if (auto q = tp.li.quirks_fetch_for_device(tp.ud_device))
                        {
                            auto threshold = ui32{};
                            if (q->quirks_get(QUIRK_ATTR_PALM_SIZE_THRESHOLD, threshold) && threshold != 0)
                            {
                                tp.palm.use_size = true;
                                tp.palm.size_threshold = threshold;
                            }
                        }
                    }
                    bool tp_is_tablet()
                    {
                        return tp.device_tags & EVDEV_TAG_TABLET_TOUCHPAD;
                    }
                void tp_init_palmdetect()
                {
                    tp.palm.right_edge = si32max;
                    tp.palm.left_edge = si32min;
                    tp.palm.upper_edge = si32min;
                    tp_init_palmdetect_arbitration();
                    if (tp.device_tags & EVDEV_TAG_EXTERNAL_TOUCHPAD
                     && !tp_is_tpkb_combo_below()
                     && !tp_is_tablet())
                    {
                        return;
                    }
                    if (!tp_is_tablet()) tp.palm.monitor_trackpoint = true;
                    if (tp.libevdev_has_event_code<EV_ABS>(ABS_MT_TOOL_TYPE)) tp.palm.use_mt_tool = true;
                    if (!tp_is_tablet()) tp_init_palmdetect_edge();
                    tp_init_palmdetect_pressure();
                    tp_init_palmdetect_size();
                }
                        void tp_tap_resume(time stamp)
                        {
                            tp_tap_enabled_update(faux, tp.tap.tap_state_enabled, stamp);
                        }
                    void tp_trackpoint_timeout(time now)
                    {
                        if (tp.palm.trackpoint_active)
                        {
                            tp.tp_impl.tp_tap_resume(now);
                            tp.palm.trackpoint_active = faux;
                        }
                        tp.palm.trackpoint_event_count = 0;
                    }
                    void tp_keyboard_timeout(time now)
                    {
                        if (tp.dwt.dwt_enabled && tp.dwt.key_mask.any())
                        {
                            tp.dwt.keyboard_timer->start(now + lixx::default_keyboard_activity_timeout_2);
                            tp.dwt.keyboard_last_press_time = now;
                            log("palm: keyboard timeout refresh");
                            return;
                        }
                        tp.tp_impl.tp_tap_resume(now);
                        tp.dwt.keyboard_active = faux;
                        log("palm: keyboard timeout");
                    }
                void tp_init_sendevents()
                {
                    auto tp_timer_name = utf::fprint("%s% trackpoint", tp.ud_device.sysname);
                    auto kb_timer_name = utf::fprint("%s% keyboard", tp.ud_device.sysname);
                    tp.palm.trackpoint_timer = tp.li.timers.create(tp_timer_name, [&](time now){ tp_trackpoint_timeout(now); });
                    tp.dwt.keyboard_timer = tp.li.timers.create(kb_timer_name, [&](time now){ tp_keyboard_timeout(now); });
                }
                    void tp_edge_scroll_init()
                    {
                        // Touchpads smaller than 40mm are not tall enough to have a horizontal scroll area, it takes too much space away. But clickpads have enough space here anyway because of the software button area (and all these tiny clickpads were built when software buttons were a thing, e.g. Lenovo *20 series).
                        auto [w, h] = tp.evdev_device_get_size();
                        auto want_horiz_scroll = true;
                        if (!tp.buttons.is_clickpad) want_horiz_scroll = (h >= 40);
                        auto mm = fp64_coor{};
                        mm.x = w - 7; // 7mm edge size.
                        mm.y = h - 7;
                        auto edges = tp.evdev_device_mm_to_units(mm);
                        tp.tp_scroll.right_edge = edges.x;
                        if (want_horiz_scroll) tp.tp_scroll.bottom_edge = edges.y;
                        else                   tp.tp_scroll.bottom_edge = si32max;
                        auto i = 0;
                        for (auto& t : tp.touches)
                        {
                            auto timer_name = utf::fprint("%s% (%d%) edgescroll", tp.ud_device.sysname, i++);
                            t.scroll.direction = -1;
                            t.scroll.timer = tp.li.timers.create(timer_name, [&](time now){ tp_edge_scroll_handle_event(t, SCROLL_EVENT_TIMEOUT, now); });
                        }
                    }
                bool tp_pass_sanity_check()
                {
                    if (tp.libevdev_has_event_code<EV_ABS>(ABS_X)
                     && tp.libevdev_has_event_code<EV_KEY>(BTN_TOUCH)
                     && tp.libevdev_has_event_code<EV_KEY>(BTN_TOOL_FINGER))
                    {
                        return true;
                    }
                    else
                    {
                        log("device failed touchpad sanity checks");
                        return faux;
                    }
                }
                void tp_init_default_resolution()
                {
                    static constexpr auto touchpad_width_mm = 69; // 1 under palm detection.
                    static constexpr auto touchpad_height_mm = 50;
                    if (!tp.ud_device.abs.is_fake_resolution) return;
                    // We only get here if
                    // - the touchpad provides no resolution
                    // - the udev hwdb didn't override the resolution
                    // - no ATTR_SIZE_HINT is set
                    // The majority of touchpads that triggers all these conditions are old ones, so let's assume a small touchpad size and assume that.
                    log("Device size is mx=%% my=%%. No resolution or size hints, assuming a size of %d%x%d%mm", tp.ud_device.abs.dimensions.x, tp.ud_device.abs.dimensions.y, touchpad_width_mm, touchpad_height_mm);
                    auto xres = tp.ud_device.abs.dimensions.x / touchpad_width_mm;
                    auto yres = tp.ud_device.abs.dimensions.y / touchpad_height_mm;
                    tp.ud_device.libevdev_set_abs_resolution(ABS_X, xres);
                    tp.ud_device.libevdev_set_abs_resolution(ABS_Y, yres);
                    tp.ud_device.libevdev_set_abs_resolution(ABS_MT_POSITION_X, xres);
                    tp.ud_device.libevdev_set_abs_resolution(ABS_MT_POSITION_Y, yres);
                    tp.ud_device.abs.is_fake_resolution = faux;
                }
                void tp_init_pressurepad()
                {
                    // On traditional touchpads, the pressure value equals contact
                    // size. On PressurePads, pressure is a real physical axis for the
                    // force down. So we disable it here because we don't do anything
                    // with it anyway and using it for touch size messes things up.
                    //
                    // The kernel/udev set the resolution to non-zero on those devices
                    // to indicate that the value is in a known axis space.
                    //
                    // See also #562.
                    if (tp.libevdev_get_abs_resolution(ABS_MT_PRESSURE) != 0 || tp.evdev_device_has_model_quirk(QUIRK_MODEL_PRESSURE_PAD))
                    {
                        tp.libevdev_disable_event_code<EV_ABS>(ABS_MT_PRESSURE);
                        tp.libevdev_disable_event_code<EV_ABS>(ABS_PRESSURE);
                    }
                }
                    bool tp_tap_default()
                    {
                        if (!tp.libevdev_has_event_code<EV_KEY>(BTN_LEFT)) // If we don't have a left button we must have tapping enabled by default.
                        {
                            return true;
                        }
                        else
                        {
                            // Tapping is disabled by default for two reasons:
                            // - If you don't know that tapping is a thing (or enabled by default), you get spurious mouse events that make the desktop feel buggy.
                            // - If you do know what tapping is and you want it, you usually know where to enable it, or at least you can search for it.
                            return faux;
                        }
                    }
                static si32 tp_clickpad_middlebutton_is_available(libinput_device_sptr li_device)
                {
                    return libinput_device_t::evdev_middlebutton_is_available(li_device);
                }
                static libinput_config_status tp_clickpad_middlebutton_set(libinput_device_sptr li_device, bool middle_emulation_enabled)
                {
                    li_device->middlebutton.want_enabled = middle_emulation_enabled;
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    tp.tp_impl.tp_clickpad_middlebutton_apply_config();
                    return LIBINPUT_CONFIG_STATUS_SUCCESS;
                }
                static bool tp_clickpad_middlebutton_get(libinput_device_sptr li_device)
                {
                    return libinput_device_t::evdev_middlebutton_get(li_device);
                }
                static bool tp_clickpad_middlebutton_get_default(libinput_device_sptr li_device)
                {
                    return libinput_device_t::evdev_middlebutton_get_default(li_device);
                }
            void tp_init_clickpad_middlebutton_emulation()
            {
                tp.middlebutton.enabled_default    = faux;
                tp.middlebutton.want_enabled       = faux;
                tp.middlebutton.enabled            = faux;
                tp.middlebutton.config.available   = tp_clickpad_middlebutton_is_available;
                tp.middlebutton.config.set         = tp_clickpad_middlebutton_set;
                tp.middlebutton.config.get         = tp_clickpad_middlebutton_get;
                tp.middlebutton.config.get_default = tp_clickpad_middlebutton_get_default;
                tp.config.middle_emulation = &tp.middlebutton.config;
            }
                static si32 tp_scroll_config_natural_get_default(libinput_device_sptr li_device)
                {
                    return (li_device->evdev_device_has_model_quirk(QUIRK_MODEL_APPLE_TOUCHPAD) ||
                            li_device->evdev_device_has_model_quirk(QUIRK_MODEL_APPLE_TOUCHPAD_ONEBUTTON));
                }
                    ui32 tp_scroll_get_methods()
                    {
                        auto methods = (ui32)LIBINPUT_CONFIG_SCROLL_EDGE;
                        // Any movement with more than one finger has random cursor jumps. Don't allow for 2fg scrolling on this device, see fdo bug 91135.
                        if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_HP_PAVILION_DM4_TOUCHPAD))
                        {
                            return LIBINPUT_CONFIG_SCROLL_EDGE;
                        }
                        else
                        {
                            if (tp.ntouches >= 2) methods |= LIBINPUT_CONFIG_SCROLL_2FG;
                            return methods;
                        }
                    }
                static ui32 tp_scroll_config_scroll_method_get_methods(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.tp_impl.tp_scroll_get_methods();
                }
                static libinput_config_status tp_scroll_config_scroll_method_set_method(libinput_device_sptr li_device, libinput_config_scroll_method method)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    auto stamp = datetime::now();
                    if (method != tp.tp_scroll.method)
                    {
                        tp.tp_impl.tp_edge_scroll_stop_events(stamp);
                        tp.tp_impl.tp_gesture_stop_twofinger_scroll(stamp);
                        tp.tp_scroll.method = method;
                    }
                    return LIBINPUT_CONFIG_STATUS_SUCCESS;
                }
                static libinput_config_scroll_method tp_scroll_config_scroll_method_get_method(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.tp_scroll.method;
                }
                libinput_config_scroll_method tp_scroll_get_default_method2()
                {
                    auto methods = tp_scroll_get_methods();
                    auto twofg = methods & LIBINPUT_CONFIG_SCROLL_2FG;
                    auto method = twofg ? LIBINPUT_CONFIG_SCROLL_2FG : LIBINPUT_CONFIG_SCROLL_EDGE;
                    if (!(methods & method))
                    {
                        log("invalid default scroll method %d%", method);
                    }
                    return method;
                }
                static libinput_config_scroll_method tp_scroll_get_default_method(tp_dispatch_sptr tp)
                {
                    return tp->tp_impl.tp_scroll_get_default_method2();
                }
                static libinput_config_scroll_method tp_scroll_config_scroll_method_get_default_method(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.tp_impl.tp_scroll_get_default_method2();
                }
            void tp_init_scroll()
            {
                tp_edge_scroll_init();
                tp.evdev_init_natural_scroll();
                tp.scroll.config_natural.get_default_enabled = tp_scroll_config_natural_get_default; // Override natural scroll config for Apple touchpads.
                tp.scroll.natural_scrolling_enabled           = tp_scroll_config_natural_get_default(tp.This());
                tp.tp_scroll.config_method.get_methods        = tp_scroll_config_scroll_method_get_methods;
                tp.tp_scroll.config_method.set_method         = tp_scroll_config_scroll_method_set_method;
                tp.tp_scroll.config_method.get_method         = tp_scroll_config_scroll_method_get_method;
                tp.tp_scroll.config_method.get_default_method = tp_scroll_config_scroll_method_get_default_method;
                tp.tp_scroll.method                           = tp_scroll_get_default_method2();
                tp.config.scroll_method = &tp.tp_scroll.config_method;
                // In mm for touchpads with valid resolution, see tp_init_accel().
                tp.scroll.threshold = 0.0;
                tp.scroll.direction_lock_threshold = 5.0;
            }
                    bool tp_gesture_are_gestures_enabled()
                    {
                        return (!tp.semi_mt && tp.num_slots > 1);
                    }
                static libinput_config_status tp_gesture_set_hold_enabled(libinput_device_sptr li_device, bool hold_enabled)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    if (!tp.tp_impl.tp_gesture_are_gestures_enabled())
                    {
                        return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
                    }
                    else
                    {
                        tp.gesture.hold_enabled = hold_enabled;
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                }
                static bool tp_gesture_is_hold_enabled(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.gesture.hold_enabled;
                }
                static bool tp_gesture_get_hold_default(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.tp_impl.tp_gesture_are_gestures_enabled();
                }
                static si32 tp_3fg_drag_count(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    if (!tp.tp_impl.tp_gesture_are_gestures_enabled()) // If we can't to gestures we can't do 3fg drag.
                    {
                        return 0;
                    }
                    else // For now return the number of MT slots until we need to figure out if we can implement this on a 2-finger BTN_TOOL_TRIPLETAP device.
                    {
                        return tp.num_slots;
                    }
                }
                static libinput_config_status tp_3fg_drag_set_enabled(libinput_device_sptr li_device, libinput_config_3fg_drag_state enabled)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    if (tp_3fg_drag_count(li_device) < 3)
                    {
                        return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
                    }
                    else
                    {
                        switch (enabled)
                        {
                            case LIBINPUT_CONFIG_3FG_DRAG_DISABLED:    tp.drag_3fg.want_nfingers = 0; break;
                            case LIBINPUT_CONFIG_3FG_DRAG_ENABLED_3FG: tp.drag_3fg.want_nfingers = 3; break;
                            case LIBINPUT_CONFIG_3FG_DRAG_ENABLED_4FG: tp.drag_3fg.want_nfingers = 4; break;
                        }
                        tp.tp_impl.tp_3fg_drag_apply_config();
                        return LIBINPUT_CONFIG_STATUS_SUCCESS;
                    }
                }
                static libinput_config_3fg_drag_state tp_3fg_drag_get_enabled(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    auto want_nfingers = tp.drag_3fg.want_nfingers;
                    if (want_nfingers == 3) return LIBINPUT_CONFIG_3FG_DRAG_ENABLED_3FG;
                    if (want_nfingers == 4) return LIBINPUT_CONFIG_3FG_DRAG_ENABLED_4FG;
                    else                    return LIBINPUT_CONFIG_3FG_DRAG_DISABLED;
                }
                    libinput_config_3fg_drag_state tp_3fg_drag_default()
                    {
                        return LIBINPUT_CONFIG_3FG_DRAG_DISABLED;
                    }
                static libinput_config_3fg_drag_state tp_3fg_drag_get_default_enabled(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    return tp.tp_impl.tp_3fg_drag_default();
                }
                void tp_gesture_finger_count_switch_timeout(time now)
                {
                    if (tp.gesture.finger_count_pending)
                    {
                        tp_gesture_handle_event(GESTURE_EVENT_FINGER_SWITCH_TIMEOUT, now);
                        tp.gesture.finger_count = tp.gesture.finger_count_pending;
                        tp.gesture.finger_count_pending = 0;
                    }
                }
                    bool tp_tap_dragging_or_double_tapping()
                    {
                        auto state = tp.tap.state;
                        return state == TAP_STATE_1FGTAP_DRAGGING_OR_DOUBLETAP
                            || state == TAP_STATE_2FGTAP_DRAGGING_OR_DOUBLETAP
                            || state == TAP_STATE_3FGTAP_DRAGGING_OR_DOUBLETAP;
                    }
                void tp_gesture_hold_timeout(time now)
                {
                    if (!tp_tap_dragging_or_double_tapping() && !tp_tap_dragging())
                    {
                        tp_gesture_handle_event(GESTURE_EVENT_HOLD_TIMEOUT, now);
                    }
                }
                void tp_gesture_3fg_drag_timeout(time now)
                {
                    tp_gesture_handle_event(GESTURE_EVENT_3FG_DRAG_RELEASE_TIMEOUT, now);
                }
            void tp_init_gesture()
            {
                tp.gesture.config.set_hold_enabled = tp_gesture_set_hold_enabled;
                tp.gesture.config.get_hold_enabled = tp_gesture_is_hold_enabled;
                tp.gesture.config.get_hold_default = tp_gesture_get_hold_default;
                tp.config.gesture  = &tp.gesture.config;
                tp.drag_3fg.config.count           = tp_3fg_drag_count;
                tp.drag_3fg.config.set_enabled     = tp_3fg_drag_set_enabled;
                tp.drag_3fg.config.get_enabled     = tp_3fg_drag_get_enabled;
                tp.drag_3fg.config.get_default     = tp_3fg_drag_get_default_enabled;
                tp.config.drag_3fg = &tp.drag_3fg.config;
                switch (tp_3fg_drag_default())
                {
                    case LIBINPUT_CONFIG_3FG_DRAG_DISABLED:    tp.drag_3fg.nfingers = 0; break;
                    case LIBINPUT_CONFIG_3FG_DRAG_ENABLED_3FG: tp.drag_3fg.nfingers = 3; break;
                    case LIBINPUT_CONFIG_3FG_DRAG_ENABLED_4FG: tp.drag_3fg.nfingers = 4; break;
                }
                tp.drag_3fg.want_nfingers = tp.drag_3fg.nfingers;
                // Two-finger scrolling is always enabled, this flag just decides whether we detect pinch. semi-mt devices are too unreliable to do pinch gestures.
                tp.gesture.state        = GESTURE_STATE_NONE;
                tp.gesture.enabled      = tp_gesture_are_gestures_enabled();
                tp.gesture.hold_enabled = tp_gesture_are_gestures_enabled();
                auto sysname = tp.ud_device.sysname;
                auto gestures_timer_name = utf::fprint("%s% gestures", sysname);
                auto hold_timer_name     = utf::fprint("%s% hold", sysname);
                auto drag_3fg_timer_name = utf::fprint("%s% drag_3fg", sysname);
                tp.gesture.finger_count_switch_timer = tp.li.timers.create(gestures_timer_name, [&](time now){ tp_gesture_finger_count_switch_timeout(now); });
                tp.gesture.hold_timer                = tp.li.timers.create(hold_timer_name    , [&](time now){ tp_gesture_hold_timeout(now);                });
                tp.gesture.drag_3fg_timer            = tp.li.timers.create(drag_3fg_timer_name, [&](time now){ tp_gesture_3fg_drag_timeout(now);            });
            }
            void tp_init_thumb()
            {
                tp.thumb.detect_thumbs = faux;
                if (!tp.buttons.is_clickpad) return;
                // If the touchpad is less than 50mm high, skip thumb detection. It's too small to meaningfully interact with a thumb on the touchpad.
                auto [w, h] = tp.evdev_device_get_size();
                if (h < 50) return;
                tp.thumb.detect_thumbs      = true;
                tp.thumb.use_pressure       = faux;
                tp.thumb.pressure_threshold = si32max;
                tp.thumb.size_threshold     = si32max;
                auto mm = fp64_coor{};
                mm.y = h * 0.85; // Detect thumbs by pressure in the bottom 15mm, detect thumbs by lingering in the bottom 8mm.
                auto edges = tp.evdev_device_mm_to_units(mm);
                tp.thumb.upper_thumb_line = edges.y;
                mm.y = h * 0.92;
                edges = tp.evdev_device_mm_to_units(mm);
                tp.thumb.lower_thumb_line = edges.y;
                if (auto q = tp.li.quirks_fetch_for_device(tp.ud_device))
                {
                    auto threshold = ui32{};
                    if (tp.libevdev_has_event_code<EV_ABS>(ABS_MT_PRESSURE))
                    {
                        if (q->quirks_get(QUIRK_ATTR_THUMB_PRESSURE_THRESHOLD, threshold))
                        {
                            tp.thumb.use_pressure = true;
                            tp.thumb.pressure_threshold = threshold;
                        }
                    }
                    if (tp.libevdev_has_event_code<EV_ABS>(ABS_MT_TOUCH_MAJOR))
                    {
                        if (q->quirks_get(QUIRK_ATTR_THUMB_SIZE_THRESHOLD, threshold))
                        {
                            tp.thumb.use_size = true;
                            tp.thumb.size_threshold = threshold;
                        }
                    }
                }
                tp_thumb_reset();
                log("thumb: enabled thumb detection (area%s%%s%)", tp.thumb.use_pressure ? ", pressure" : "", tp.thumb.use_size ? ", size" : "");
            }
            si32 tp_init()
            {
                auto use_touch_size = faux;
                if (!tp_pass_sanity_check())
                {
                    return faux;
                }
                tp_init_default_resolution();
                tp_init_pressurepad();
                if (!tp_init_slots())
                {
                    return faux;
                }
                tp.evdev_device_init_abs_range_warnings();
                use_touch_size = tp_init_touch_size();
                if (!use_touch_size)
                {
                    tp_init_pressure();
                }
                tp.dpi = tp.ud_device.abs.absinfo_x.resolution * 25.4; // Set the dpi to that of the x axis, because that's what we normalize to when needed.
                tp_init_hysteresis();
                tp_init_accel(LIBINPUT_CONFIG_ACCEL_PROFILE_ADAPTIVE);
                tp_init_tap();
                tp_init_buttons();
                tp_init_dwt();
                tp_init_dwtp();
                tp_init_palmdetect();
                tp_init_sendevents();
                tp_init_scroll();
                tp_init_gesture();
                tp_init_thumb();
                // Lenovo X1 Gen6 buffers the events in a weird way, making jump detection impossible.
                // See https://gitlab.freedesktop.org/libinput/libinput/-/issues/506.
                if (tp.evdev_device_has_model_quirk(QUIRK_MODEL_LENOVO_X1GEN6_TOUCHPAD))
                {
                    tp.jump.detection_disabled = true;
                }
                tp.device_caps |= EVDEV_DEVICE_POINTER;
                if (tp.gesture.enabled)
                {
                    tp.device_caps |= EVDEV_DEVICE_GESTURE;
                }
                return true;
            }
            void tp_suspend_conditional()
            {
                for (auto d : tp.li.device_list)
                {
                    if (d->device_tags & EVDEV_TAG_EXTERNAL_MOUSE)
                    {
                        tp_suspend(SUSPEND_EXTERNAL_MOUSE);
                        break;
                    }
                }
            }
                bool tp_requires_rotation()
                {
                    auto rotate = faux;
                    #if HAVE_LIBWACOM
                    if (li_device->tags & EVDEV_TAG_TABLET_TOUCHPAD)
                    {
                        auto db = tp.li.libinput_libwacom_ref();
                        if (db)
                        {
                            // Check if we have a device with the same vid/pid. If not, we need to loop through all devices and check their paired device.
                            auto vid = li_device->libevdev_get_id_vendor();
                            auto pid = li_device->libevdev_get_id_product();
                            auto dev = ::libwacom_new_from_usbid(db, vid, pid, nullptr);
                            if (dev)
                            {
                                rotate = ::libwacom_is_reversible(dev);
                                ::libwacom_destroy(dev);
                            }
                            else
                            {
                                auto devices = ::libwacom_list_devices_from_database(db, nullptr);
                                if (devices)
                                {
                                    auto d = devices;
                                    while (*d)
                                    {
                                        auto paired = ::libwacom_get_paired_device(*d);
                                        if (paired && ::libwacom_match_get_vendor_id(paired) == vid
                                                   && ::libwacom_match_get_product_id(paired) == pid)
                                        {
                                            rotate = ::libwacom_is_reversible(dev);
                                            break;
                                        }
                                        d++;
                                    }
                                    ::free(devices);
                                }
                            }
                            if (db) libinput_libwacom_unref(tp.li); // We don't need to keep it around for the touchpad, we're done with it until the device dies.
                        }
                    }
                    #endif
                    return rotate;
                }
                static void tp_change_to_left_handed(libinput_device_sptr li_device)
                {
                    auto& tp = *std::static_pointer_cast<tp_device>(li_device);
                    if (li_device->dev_left_handed.want_enabled == li_device->dev_left_handed.enabled)
                    {
                        return;
                    }
                    if (tp.buttons.state & 0x3) // BTN_LEFT | BTN_RIGHT.
                    {
                        return;
                    }
                    // Tapping and clickfinger aren't affected by left-handed config, so checking physical buttons is enough.
                    li_device->dev_left_handed.enabled = li_device->dev_left_handed.want_enabled;
                    tp.tp_impl.tp_change_rotation(true);
                }
            void tp_init_left_handed()
            {
                tp.left_handed.must_rotate = tp_requires_rotation();
                auto want_left_handed = !(tp.ud_device.model_flags & EVDEV_MODEL_APPLE_TOUCHPAD_ONEBUTTON);
                if (want_left_handed)
                {
                    tp.evdev_init_left_handed(tp_change_to_left_handed);
                }
            }
        };

        tp_impl_t tp_impl{ *this };
        void                  process(evdev_event& ev, time stamp)                           { tp_impl.        tp_interface_process(ev, stamp); }
        void                  suspend()                                                      { tp_impl.              tp_clear_state(); }
        void                   remove()                                                      { tp_impl.         tp_interface_remove(); }
        void             device_added(libinput_device_sptr added_li_device)                  { tp_impl.   tp_interface_device_added(added_li_device); }
        void           device_removed(libinput_device_sptr removed_li_device)                { tp_impl. tp_interface_device_removed(removed_li_device); }
        void       left_handed_toggle(bool left_handed_enabled)                              { tp_impl.touchpad_left_handed_toggled(left_handed_enabled); }
        void touch_arbitration_toggle(libinput_arbitration_state which, fp64_rect, time now) { tp_impl.   tp_interface_toggle_touch(which, now); }
        void         device_suspended(libinput_device_sptr suspended_li_device)              { device_removed(suspended_li_device); }
        void           device_resumed(libinput_device_sptr resumed_li_device)                { device_added(resumed_li_device); }
        virtual ui32 sendevents_get_modes() override
        {
            auto modes = (ui32)LIBINPUT_CONFIG_SEND_EVENTS_DISABLED;
            if (device_tags & EVDEV_TAG_INTERNAL_TOUCHPAD)
            {
                modes |= LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE;
            }
            return modes;
        }
        virtual libinput_config_status sendevents_set_mode(libinput_config_send_events_mode mode) override
        {
            // DISABLED overrides any DISABLED_ON_.
            if ((mode & LIBINPUT_CONFIG_SEND_EVENTS_DISABLED) && (mode & LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE))
            {
                mode = (libinput_config_send_events_mode)(mode & ~LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE);
            }
            if (mode == sendevents_current_mode) return LIBINPUT_CONFIG_STATUS_SUCCESS;
            switch (mode)
            {
                case LIBINPUT_CONFIG_SEND_EVENTS_ENABLED:
                    tp_impl.tp_resume(SUSPEND_SENDEVENTS);
                    tp_impl.tp_resume(SUSPEND_EXTERNAL_MOUSE);
                    break;
                case LIBINPUT_CONFIG_SEND_EVENTS_DISABLED:
                    tp_impl.tp_suspend(SUSPEND_SENDEVENTS);
                    tp_impl.tp_resume(SUSPEND_EXTERNAL_MOUSE);
                    break;
                case LIBINPUT_CONFIG_SEND_EVENTS_DISABLED_ON_EXTERNAL_MOUSE:
                    tp_impl.tp_suspend_conditional();
                    tp_impl.tp_resume(SUSPEND_SENDEVENTS);
                    break;
                default:
                    return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
            }
            sendevents_current_mode = mode;
            return LIBINPUT_CONFIG_STATUS_SUCCESS;
        }
        virtual libinput_config_send_events_mode sendevents_get_default_mode() override
        {
            return LIBINPUT_CONFIG_SEND_EVENTS_ENABLED;
        }
    };

    struct pad_mode_led
    {
        fd_t brightness_fd; // /sys/devices/..../input1235/input1235::wacom-led_0.1/brightness.
        si32 mode_idx;

        pad_mode_led()
            : brightness_fd{ os::invalid_fd },
              mode_idx{}
        { }
        ~pad_mode_led()
        {
            if (brightness_fd != os::invalid_fd)
            {
                os::close(brightness_fd);
            }
        }
    };
    struct pad_mode_toggle_button
    {
        ui32                          button_index{};
        pad_toggle_button_target_mode target_mode{};
    };
    struct pad_led_group : libinput_tablet_pad_mode_group
    {
        std::list<pad_mode_led>           led_list;
        std::list<pad_mode_toggle_button> toggle_button_list;
    };
    struct pad_device : libinput_device_t
    {
        struct dials_t
        {
            bool has_hires_dial{};
            fp64 dial1{};
            fp64 dial2{};
        };
        struct modes_t
        {
            std::list<libinput_tablet_pad_mode_group_sptr> mode_group_list;
        };

        pad_device(auto&... args)
            : libinput_device_t{ args... }
        { }

        byte           status{};
        ui32           changed_axes{};
        button_state_t next_button_state;
        button_state_t prev_button_state;
        ui32           button_map[KEY_CNT] = {};
        ui32           nbuttons{};
        bool           have_abs_misc_terminator{};
        dials_t        dials;
        modes_t        modes;

        struct pad_impl_t
        {
            pad_device& pad;
                        void pad_set_status(byte s) { pad.status |= s; }
                        bool pad_has_status(byte s) { return !!(pad.status & s); }
                        void pad_unset_status(byte s) { pad.status &= ~s; }
                        void pad_process_relative(evdev_event& ev)
                        {
                            switch (ev.usage)
                            {
                                case evdev::rel_dial:
                                    pad.dials.dial1 = ev.value * 120;
                                    pad.changed_axes |= PAD_AXIS_DIAL1;
                                    pad_set_status(PAD_AXES_UPDATED);
                                    break;
                                case evdev::rel_wheel:
                                    if (!pad.dials.has_hires_dial)
                                    {
                                        pad.dials.dial1 = -1 * ev.value * 120;
                                        pad.changed_axes |= PAD_AXIS_DIAL1;
                                        pad_set_status(PAD_AXES_UPDATED);
                                    }
                                    break;
                                case evdev::rel_hwheel:
                                    if (!pad.dials.has_hires_dial)
                                    {
                                        pad.dials.dial2 = ev.value * 120;
                                        pad.changed_axes |= PAD_AXIS_DIAL2;
                                        pad_set_status(PAD_AXES_UPDATED);
                                    }
                                    break;
                                case evdev::rel_wheel_hi_res:
                                    pad.dials.dial1 = -1 * ev.value;
                                    pad.changed_axes |= PAD_AXIS_DIAL1;
                                    pad_set_status(PAD_AXES_UPDATED);
                                    break;
                                case evdev::rel_hwheel_hi_res:
                                    pad.dials.dial2 = ev.value;
                                    pad.changed_axes |= PAD_AXIS_DIAL2;
                                    pad_set_status(PAD_AXES_UPDATED);
                                    break;
                                default:
                                    log("Unhandled EV_REL event code %#x%", ev.usage);
                                    break;
                            }
                        }
                            void pad_update_changed_axis(pad_axes axis, evdev_event const& ev)
                            {
                                if (pad.changed_axes & axis)
                                {
                                    log("Multiple EV_ABS 'type=%% code=%%' events in the same SYN_REPORT", evdev_usage_type(ev.usage), evdev_usage_code(ev.usage));
                                    // Special heuristics probably good enough: if we get multiple EV_ABS in the same SYN_REPORT and one of them is zero, assume they're all zero and unchanged. That's not perfectly correct but probably covers all cases.
                                    if (ev.value == 0)
                                    {
                                        pad.changed_axes &= ~axis;
                                        if (pad.changed_axes == 0)
                                        {
                                            pad_unset_status(PAD_AXES_UPDATED);
                                        }
                                        return;
                                    }
                                }
                                pad.changed_axes |= axis;
                                pad_set_status(PAD_AXES_UPDATED);
                            }
                        void pad_process_absolute(evdev_event& ev)
                        {
                            auto axis = PAD_AXIS_NONE;
                            switch (ev.usage)
                            {
                                case evdev::abs_wheel:    axis = PAD_AXIS_RING1; break;
                                case evdev::abs_throttle: axis = PAD_AXIS_RING2; break;
                                case evdev::abs_rx:       axis = PAD_AXIS_STRIP1; break;
                                case evdev::abs_ry:       axis = PAD_AXIS_STRIP2; break;
                                case evdev::abs_misc:
                                    // The wacom driver always sends a 0 axis event on finger up, but we also get an ABS_MISC 15 on touch down and ABS_MISC 0 on touch up, on top of the actual event. This is kernel behavior for xf86-input-wacom backwards compatibility after the 3.17 wacom HID move. We use that event to tell when we truly went a full rotation around the wheel vs. a finger release. FIXME: On the Intuos5 and later the kernel merges all states into that event, so if any finger is down on any button, the wheel release won't trigger the ABS_MISC 0 but still send a 0 event. We can't currently detect this.
                                    pad.have_abs_misc_terminator = true;
                                    break;
                                default:
                                    log("Unhandled EV_ABS event code %#x%", ev.usage);
                                    break;
                            }
                            if (axis != PAD_AXIS_NONE)
                            {
                                pad_update_changed_axis(axis, ev);
                            }
                        }
                            void pad_button_set_down(ui32 button, bool is_down)
                            {
                                auto code = evdev_usage_code(button);
                                if (is_down)
                                {
                                    pad.next_button_state.set(code);
                                    pad_set_status(PAD_BUTTONS_PRESSED);
                                }
                                else
                                {
                                    pad.next_button_state.reset(code);
                                    pad_set_status(PAD_BUTTONS_RELEASED);
                                }
                            }
                        void pad_process_key(evdev_event& ev)
                        {
                            auto is_press = (ui32)(ev.value != 0);
                            if (ev.value != 2) // Ignore kernel key repeat.
                            {
                                pad_button_set_down(ev.usage, is_press);
                            }
                        }
                                    si32 libinput_tablet_pad_mode_group_has_dial(libinput_tablet_pad_mode_group_sptr group, ui32 dial)
                                    {
                                        if ((si32)dial >= group->li_device->evdev_device_tablet_pad_get_num_dials())
                                        {
                                            return 0;
                                        }
                                        return !!(group->dial_mask & (1ul << dial));
                                    }
                                libinput_tablet_pad_mode_group_sptr pad_dial_get_mode_group(ui32 dial)
                                {
                                    for (auto group : pad.modes.mode_group_list)
                                    {
                                        if (libinput_tablet_pad_mode_group_has_dial(group, dial))
                                        {
                                            return group;
                                        }
                                    }
                                    assert(!"Unable to find dial mode group");
                                    return nullptr;
                                }
                                void tablet_pad_notify_dial(time stamp, ui32 number, fp64 value, libinput_tablet_pad_mode_group_sptr group)
                                {
                                    auto& dial_event = pad.li.libinput_emplace_event<libinput_event_tablet_pad>();
                                    dial_event.mode       = group->current_mode;
                                    dial_event.mode_group = group;
                                    dial_event.dial       = { .v120 = value, .number = (si32)number };
                                    pad.post_device_event(stamp, LIBINPUT_EVENT_TABLET_PAD_DIAL, dial_event);
                                }
                                    fp64 normalize_wacom_ring(abs_info_t const& absinfo)
                                    {
                                        // Libinput has 0 as the ring's northernmost point in the device's current logical rotation, increasing clockwise to 1. Wacom has 0 on the left-most wheel position.
                                        auto range = absinfo.absinfo_range();
                                        auto value = (absinfo.value - absinfo.minimum) / range - 0.25;
                                        if (value < 0.0) value += 1.0;
                                        return value;
                                    }
                                fp64 pad_handle_ring(ui32 code)
                                {
                                    auto& absinfo = pad.libevdev_get_abs_info(code);
                                    assert(absinfo);
                                    auto degrees = normalize_wacom_ring(absinfo) * 360;
                                    if (pad.dev_left_handed.enabled)
                                    {
                                        degrees = std::fmod(degrees + 180, 360);
                                    }
                                    return degrees;
                                }
                                    si32 libinput_tablet_pad_mode_group_has_ring(libinput_tablet_pad_mode_group_sptr group, ui32 ring)
                                    {
                                        if ((si32)ring >= group->li_device->evdev_device_tablet_pad_get_num_rings())
                                        {
                                            return 0;
                                        }
                                        return !!(group->ring_mask & (1ul << ring));
                                    }
                                libinput_tablet_pad_mode_group_sptr pad_ring_get_mode_group(ui32 ring)
                                {
                                    for (auto group : pad.modes.mode_group_list)
                                    {
                                        if (libinput_tablet_pad_mode_group_has_ring(group, ring))
                                        {
                                            return group;
                                        }
                                    }
                                    assert(!"Unable to find ring mode group");
                                    return {};
                                }
                                void tablet_pad_notify_ring(time stamp, ui32 number, fp64 value, libinput_tablet_pad_ring_axis_source source, libinput_tablet_pad_mode_group_sptr group)
                                {
                                    auto& ring_event = pad.li.libinput_emplace_event<libinput_event_tablet_pad>();
                                    ring_event.mode       = group->current_mode;
                                    ring_event.mode_group = group;
                                    ring_event.ring       = { .source = source,
                                                              .position = value,
                                                              .number = (si32)number };
                                    pad.post_device_event(stamp, LIBINPUT_EVENT_TABLET_PAD_RING, ring_event);
                                }
                                    fp64 normalize_wacom_strip(abs_info_t const& absinfo)
                                    {
                                        // Strip axes don't use a proper value, they just shift the bit left for each position. 0 isn't a real value either, it's only sent on finger release.
                                        auto min = fp64{};
                                        auto max = log2(absinfo.maximum);
                                        auto range = max - min;
                                        auto value = (log2(absinfo.value) - min) / range;
                                        return value;
                                    }
                                    fp64 normalize_strip(abs_info_t const& absinfo)
                                    {
                                        return absinfo.absinfo_normalize_value(absinfo.value);
                                    }
                                fp64 pad_handle_strip(ui32 code)
                                {
                                    auto pos = fp64{};
                                    auto& absinfo = pad.libevdev_get_abs_info(code);
                                    assert(absinfo);
                                    if (absinfo.value == 0) return 0.0;
                                    if (pad.libevdev_get_id_vendor() == lixx::vendor_id_wacom)
                                    {
                                        pos = normalize_wacom_strip(absinfo);
                                    }
                                    else
                                    {
                                        pos = normalize_strip(absinfo);
                                    }
                                    if (pad.dev_left_handed.enabled) pos = 1.0 - pos;
                                    return pos;
                                }
                                    si32 libinput_tablet_pad_mode_group_has_strip(libinput_tablet_pad_mode_group_sptr group, ui32 strip)
                                    {
                                        if ((si32)strip >= group->li_device->evdev_device_tablet_pad_get_num_strips()) return 0;
                                        return !!(group->strip_mask & (1ul << strip));
                                    }
                                libinput_tablet_pad_mode_group_sptr pad_strip_get_mode_group(ui32 strip)
                                {
                                    for (auto group : pad.modes.mode_group_list)
                                    {
                                        if (libinput_tablet_pad_mode_group_has_strip(group, strip))
                                        {
                                            return group;
                                        }
                                    }
                                    assert(!"Unable to find strip mode group");
                                    return {};
                                }
                                void tablet_pad_notify_strip(time stamp, ui32 number, fp64 value, libinput_tablet_pad_strip_axis_source source, libinput_tablet_pad_mode_group_sptr group)
                                {
                                    auto& strip_event = pad.li.libinput_emplace_event<libinput_event_tablet_pad>();
                                    strip_event.mode       = group->current_mode;
                                    strip_event.mode_group = group;
                                    strip_event.strip      = { .source = source,
                                                               .position = value,
                                                               .number = (si32)number };
                                    pad.post_device_event(stamp, LIBINPUT_EVENT_TABLET_PAD_STRIP, strip_event);
                                }
                            void pad_check_notify_axes(time stamp)
                            {
                                auto send_finger_up = faux;
                                if (pad.have_abs_misc_terminator // Suppress the reset to 0 on finger up. See the comment in pad_process_absolute.
                                 && pad.libevdev_get_event_value<EV_ABS>(ABS_MISC) == 0)
                                {
                                    send_finger_up = true;
                                }
                                if (pad.changed_axes & PAD_AXIS_DIAL1) // Unlike the ring axis we don't get an event when we release so we can't set a source.
                                {
                                    auto group = pad_dial_get_mode_group(0);
                                    tablet_pad_notify_dial(stamp, 0, pad.dials.dial1, group);
                                }
                                if (pad.changed_axes & PAD_AXIS_DIAL2)
                                {
                                    auto group = pad_dial_get_mode_group(1);
                                    tablet_pad_notify_dial(stamp, 1, pad.dials.dial2, group);
                                }
                                if (pad.changed_axes & PAD_AXIS_RING1)
                                {
                                    auto value = pad_handle_ring(ABS_WHEEL);
                                    if (send_finger_up) value = -1.0;
                                    auto group = pad_ring_get_mode_group(0);
                                    tablet_pad_notify_ring(stamp, 0, value, LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER, group);
                                }
                                if (pad.changed_axes & PAD_AXIS_RING2)
                                {
                                    auto value = pad_handle_ring(ABS_THROTTLE);
                                    if (send_finger_up) value = -1.0;
                                    auto group = pad_ring_get_mode_group(1);
                                    tablet_pad_notify_ring(stamp, 1, value, LIBINPUT_TABLET_PAD_RING_SOURCE_FINGER, group);
                                }
                                if (pad.changed_axes & PAD_AXIS_STRIP1)
                                {
                                    auto value = pad_handle_strip(ABS_RX);
                                    if (send_finger_up) value = -1.0;
                                    auto group = pad_strip_get_mode_group(0);
                                    tablet_pad_notify_strip(stamp, 0, value, LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER, group);
                                }
                                if (pad.changed_axes & PAD_AXIS_STRIP2)
                                {
                                    auto value = pad_handle_strip(ABS_RY);
                                    if (send_finger_up) value = -1.0;
                                    auto group = pad_strip_get_mode_group(1);
                                    tablet_pad_notify_strip(stamp, 1, value, LIBINPUT_TABLET_PAD_STRIP_SOURCE_FINGER, group);
                                }
                                pad.changed_axes = PAD_AXIS_NONE;
                                pad.have_abs_misc_terminator = faux;
                            }
                                                si32 evdev_device_tablet_pad_get_num_buttons(libinput_device_sptr li_device)
                                                {
                                                    auto is_pad = li_device->device_caps & EVDEV_DEVICE_TABLET_PAD;
                                                    return is_pad ? pad.nbuttons : -1;
                                                }
                                        si32 libinput_tablet_pad_mode_group_has_button(libinput_tablet_pad_mode_group_sptr group, ui32 button)
                                        {
                                            if ((si32)button >= evdev_device_tablet_pad_get_num_buttons(group->li_device))
                                            {
                                                return 0;
                                            }
                                            return !!(group->button_mask & (1ul << button));
                                        }
                                    libinput_tablet_pad_mode_group_sptr pad_button_get_mode_group(ui32 button)
                                    {
                                        for (auto group : pad.modes.mode_group_list)
                                        {
                                            if (libinput_tablet_pad_mode_group_has_button(group, button))
                                            {
                                                return group;
                                            }
                                        }
                                        assert(!"Unable to find button mode group");
                                        return {};
                                    }
                                        si32 libinput_tablet_pad_mode_group_button_is_toggle(libinput_tablet_pad_mode_group_sptr group, ui32 button)
                                        {
                                            if ((si32)button >= evdev_device_tablet_pad_get_num_buttons(group->li_device))
                                            {
                                                return 0;
                                            }
                                            return !!(group->toggle_button_mask & (1ul << button));
                                        }
                                        si32 pad_led_group_get_mode(pad_led_group_sptr group)
                                        {
                                            char buf[4] = {};
                                            auto rc = 0;
                                            auto brightness = 0u;
                                            //todo use os::fs
                                            for (auto& led : group->led_list)
                                            {
                                                rc = ::lseek(led.brightness_fd, 0, SEEK_SET);
                                                if (rc == -1) return -errno;
                                                rc = ::read(led.brightness_fd, buf, sizeof(buf) - 1);
                                                if (rc == -1) return -errno;
                                                rc = ::sscanf(buf, "%u\n", &brightness);
                                                if (rc != 1) return -EINVAL;
                                                // Assumption: only one LED lit up at any time.
                                                if (brightness != 0) return led.mode_idx;
                                            }
                                            // Wacom PTH-660 doesn't light up any LEDs until the button is pressed, so let's assume mode 0.
                                            return 0;
                                        }
                                    void pad_button_update_mode(libinput_tablet_pad_mode_group_sptr g, ui32 button_index, si32 state)
                                    {
                                        auto group = std::static_pointer_cast<pad_led_group>(g);
                                        auto rc = -ENODEV;
                                        if (state != evdev::pressed) return;
                                        if (!libinput_tablet_pad_mode_group_button_is_toggle(g, button_index)) return;
                                        if (group->led_list.empty())
                                        {
                                            for (auto& button : group->toggle_button_list)
                                            {
                                                if (button.button_index == button_index)
                                                {
                                                    if (button.target_mode == MODE_NEXT)
                                                    {
                                                        auto nmodes = group->num_modes;
                                                        rc = (group->current_mode + 1) % nmodes;
                                                    }
                                                    else
                                                    {
                                                        rc = button.target_mode;
                                                    }
                                                    break;
                                                }
                                            }
                                        }
                                        else
                                        {
                                            rc = pad_led_group_get_mode(group);
                                        }
                                        if (rc >= 0) group->current_mode = rc;
                                    }
                                    void tablet_pad_notify_button(time stamp, ui32 button, si32 state, libinput_tablet_pad_mode_group_sptr group)
                                    {
                                        auto& button_event = pad.li.libinput_emplace_event<libinput_event_tablet_pad>();
                                        button_event.mode       = group->current_mode;
                                        button_event.mode_group = group;
                                        button_event.button     = { .number = button, .state = state };
                                        pad.post_device_event(stamp, LIBINPUT_EVENT_TABLET_PAD_BUTTON, button_event);
                                    }
                                    void tablet_pad_notify_key(time stamp, si32 key, si32 state)
                                    {
                                        auto& key_event = pad.li.libinput_emplace_event<libinput_event_tablet_pad>();
                                        key_event.key.code  = (ui32)key;
                                        key_event.key.state = state;
                                        pad.post_device_event(stamp, LIBINPUT_EVENT_TABLET_PAD_KEY, key_event);
                                    }
                                void pad_notify_button_mask(time stamp, button_state_t& buttons, si32 state)
                                {
                                    for (auto code = 0u; code < buttons.size(); code++)
                                    {
                                        if (buttons[code])
                                        {
                                            auto map_value = pad.button_map[code];
                                            if (map_value != (ui32)-1)
                                            {
                                                auto button_or_key = (si32)(map_value & 0x00FFFFFF); // Map value.
                                                if (!(map_value & 0xFF000000)) // It is a button.
                                                {
                                                    auto group = pad_button_get_mode_group(button_or_key);
                                                    pad_button_update_mode(group, button_or_key, state);
                                                    tablet_pad_notify_button(stamp, (ui32)button_or_key, state, group);
                                                }
                                                else // It is a key.
                                                {
                                                    tablet_pad_notify_key(stamp, button_or_key, state);
                                                }
                                            }
                                        }
                                    }
                                }
                            void pad_notify_buttons(time stamp, si32 state)
                            {
                                auto buttons = state == evdev::pressed ? pad.next_button_state & ~pad.prev_button_state  // pad_get_buttons_pressed()
                                                                       : pad.prev_button_state & ~pad.next_button_state; // pad_get_buttons_released();
                                pad_notify_button_mask(stamp, buttons, state);
                            }
                            static void pad_change_to_left_handed(libinput_device_sptr li_device)
                            {
                                auto& pad = *std::static_pointer_cast<pad_device>(li_device);
                                if (li_device->dev_left_handed.enabled != li_device->dev_left_handed.want_enabled && !pad.next_button_state.any()) // If not pad_any_button_down.
                                {
                                    li_device->dev_left_handed.enabled = li_device->dev_left_handed.want_enabled;
                                }
                            }
                        void pad_flush(time stamp)
                        {
                            if (pad_has_status(PAD_AXES_UPDATED))
                            {
                                pad_check_notify_axes(stamp);
                                pad_unset_status(PAD_AXES_UPDATED);
                            }
                            if (pad_has_status(PAD_BUTTONS_RELEASED))
                            {
                                pad_notify_buttons(stamp, evdev::released);
                                pad_unset_status(PAD_BUTTONS_RELEASED);
                                pad_change_to_left_handed(pad.This());
                            }
                            if (pad_has_status(PAD_BUTTONS_PRESSED))
                            {
                                pad_notify_buttons(stamp, evdev::pressed);
                                pad_unset_status(PAD_BUTTONS_PRESSED);
                            }
                            pad.prev_button_state = pad.next_button_state;
                            pad.dials.dial1 = 0;
                            pad.dials.dial2 = 0;
                        }
                    void pad_process(evdev_event& ev, time stamp)
                    {
                        auto type = evdev_usage_type(ev.usage);
                        switch (type)
                        {
                            case EV_REL: pad_process_relative(ev); break;
                            case EV_ABS: pad_process_absolute(ev); break;
                            case EV_KEY: pad_process_key(     ev); break;
                            case EV_SYN: pad_flush(           stamp); break;
                            case EV_MSC: /* The EKR sends the serial as MSC_SERIAL, ignore this for now */ break;
                            default:
                                log("Unexpected event type %s% (%#x%)", libevdev_event_type_get_name(type), ev.usage);
                                break;
                        }
                    }
                    void pad_suspend()
                    {
                        for (auto usage = evdev::key_esc; usage <= evdev::key_max; usage++)
                        {
                            auto button = evdev_usage_code(usage);
                            if (pad.next_button_state[button])
                            {
                                pad_button_set_down(usage, faux);
                            }
                        }
                        pad_flush(datetime::now());
                    }
                    bool pad_init_buttons_from_libwacom([[maybe_unused]] WacomDevice* tablet)
                    {
                        auto rc = faux;
                        #if HAVE_LIBWACOM
                        if (tablet)
                        {
                            auto num_buttons = ::libwacom_get_num_buttons(tablet);
                            auto map = 0;
                            for (auto i = 0; i < num_buttons; i++)
                            {
                                auto code = ::libwacom_get_button_evdev_code(tablet, 'A' + i);
                                if (code == 0) continue;
                                map_set_button_map(pad.button_map[code], map++);
                            }
                            pad.nbuttons = map;
                            rc = true;
                        }
                        #endif
                        return rc;
                    }
                    void pad_init_buttons_from_kernel()
                    {
                        auto map = 0;
                        // We match wacom_report_numbered_buttons() from the kernel.
                        for (auto code = BTN_0; code < BTN_0 + 10; code++)
                        {
                            if (pad.libevdev_has_event_code<EV_KEY>(code))
                            {
                                pad.button_map[code] = map++;
                            }
                        }
                        for (auto code = BTN_BASE; code < BTN_BASE + 2; code++)
                        {
                            if (pad.libevdev_has_event_code<EV_KEY>(code))
                            {
                                pad.button_map[code] = map++;
                            }
                        }
                        for (auto code = BTN_A; code < BTN_A + 6; code++)
                        {
                            if (pad.libevdev_has_event_code<EV_KEY>(code))
                            {
                                pad.button_map[code] = map++;
                            }
                        }
                        for (auto code = BTN_LEFT; code < BTN_LEFT + 7; code++)
                        {
                            if (pad.libevdev_has_event_code<EV_KEY>(code))
                            {
                                pad.button_map[code] = map++;
                            }
                        }
                        pad.nbuttons = map;
                    }
                    void pad_init_keys()
                    {
                        static constexpr auto codes = std::to_array( // Wacom's keys are the only ones we know anything about.
                        {
                            KEY_BUTTONCONFIG,
                            KEY_ONSCREEN_KEYBOARD,
                            KEY_CONTROLPANEL,
                        });
                        if (pad.libevdev_get_id_vendor() == lixx::vendor_id_wacom)
                        {
                            for (auto code : codes)
                            {
                                if (pad.libevdev_has_event_code<EV_KEY>(code))
                                {
                                    pad.button_map[code] = code | 0xFF000000;
                                }
                            }
                        }
                    }
                void pad_init_buttons(WacomDevice* wacom)
                {
                    for (auto i = 0u; i < std::size(pad.button_map); i++)
                    {
                        pad.button_map[i] = (ui32)-1;
                    }
                    if (!pad_init_buttons_from_libwacom(wacom))
                    {
                        pad_init_buttons_from_kernel();
                    }
                    pad_init_keys();
                }
                void pad_init_left_handed([[maybe_unused]] WacomDevice* wacom)
                {
                    auto has_left_handed = true;
                    #if HAVE_LIBWACOM
                    has_left_handed = !wacom || ::libwacom_is_reversible(wacom);
                    #endif
                    if (has_left_handed)
                    {
                        pad.evdev_init_left_handed(pad_device::pad_impl_t::pad_change_to_left_handed);
                    }
                }
                    si32 pad_init_fallback_group()
                    {
                        auto group_index = 0u;
                        auto num_modes = 1;
                        auto group = ptr::shared<pad_led_group>();
                        group->li_device = pad.This();
                        group->index = group_index;
                        group->current_mode = 0;
                        group->num_modes = num_modes;
                        // If we only have one group, all buttons/strips/rings are part of that group. We rely on the other layers to filter out invalid indices.
                        group->button_mask = -1;
                        group->strip_mask = -1;
                        group->ring_mask = -1;
                        group->dial_mask = -1;
                        group->toggle_button_mask = 0;
                        pad.modes.mode_group_list.push_back(group);
                        return 0;
                    }
                si32 pad_init_leds([[maybe_unused]] WacomDevice* wacom)
                {
                    auto rc = 1;
                    if (pad.nbuttons > 32)
                    {
                        log("Too many pad buttons for modes %d%", pad.nbuttons);
                        return rc;
                    }
                    #if HAVE_LIBWACOM
                    rc = pad_init_leds_from_libwacom(device, wacom);
                    #endif
                    if (rc != 0) rc = pad_init_fallback_group(); // If libwacom fails, we init one fallback group anyway.
                    return rc;
                }
            si32 pad_init()
            {
                pad.status       = PAD_NONE;
                pad.changed_axes = PAD_AXIS_NONE;
                // We expect the kernel to either give us both axes as hires or neither. Getting one is a kernel bug we don't need to care about.
                pad.dials.has_hires_dial = pad.libevdev_has_event_code<EV_REL>(REL_WHEEL_HI_RES) || pad.libevdev_has_event_code<EV_REL>(REL_HWHEEL_HI_RES);
                if (pad.libevdev_has_event_code<EV_REL>(REL_WHEEL)
                 && pad.libevdev_has_event_code<EV_REL>(REL_DIAL))
                {
                    log("Unsupported combination REL_DIAL and REL_WHEEL");
                }
                auto wacom = (WacomDevice*)nullptr;
                pad_init_buttons(wacom);
                pad_init_left_handed(wacom);
                auto rc = pad_init_leds(wacom);
                return rc;
            }
        };

        pad_impl_t pad_impl{ *this };
        void process(evdev_event& ev, time stamp) { pad_impl.pad_process(ev, stamp); }
        void suspend()                            { pad_impl.pad_suspend(); }
        virtual ui32 sendevents_get_modes() override
        {
            return LIBINPUT_CONFIG_SEND_EVENTS_DISABLED;
        }
        virtual libinput_config_status sendevents_set_mode(libinput_config_send_events_mode mode) override
        {
                 if (mode == sendevents_current_mode)              return LIBINPUT_CONFIG_STATUS_SUCCESS;
            else if (mode == LIBINPUT_CONFIG_SEND_EVENTS_DISABLED) suspend();
            else if (mode != LIBINPUT_CONFIG_SEND_EVENTS_ENABLED)  return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
            sendevents_current_mode = mode;
            return LIBINPUT_CONFIG_STATUS_SUCCESS;
        }
        virtual libinput_config_send_events_mode sendevents_get_default_mode() override
        {
            return LIBINPUT_CONFIG_SEND_EVENTS_ENABLED;
        }
    };

        struct totem_slot
        {
            bool                      dirty{};
            ui32                      index{};
            slot_state_enum           state{};
            libinput_tablet_tool_sptr tool;
            tablet_axes               axes;
            tablet_axes_bitset        changed_axes_bits;
            si32_coor                 last_point;
        };
    struct totem_device : libinput_device_t
    {
        si32                       slot_index{}; // Current slot index.
        std::vector<totem_slot>    slots;
        libinput_device_sptr       touch_li_device;
        bool                       button_state_now{}; // We only have one button.
        bool                       button_state_previous{};
        libinput_arbitration_state arbitration_state{};

        totem_device(auto&... args)
            : libinput_device_t{ args... }
        { }

        struct totem_impl_t
        {
            totem_device& totem;
            libinput_tablet_tool_sptr totem_new_tool()
            {
                auto tool = ptr::shared<libinput_tablet_tool>();
                tool->serial = 0;
                tool->tool_id = 0;
                tool->type = LIBINPUT_TABLET_TOOL_TYPE_TOTEM;
                tool->pressure.tablet_id = 0;
                tool->pressure.threshold_offset = 0.0;
                tool->pressure.threshold_has_offset = faux;
                tool->pressure.threshold.min = 0;
                tool->pressure.threshold.max = 1;
                tool->axis_caps_bits.set(LIBINPUT_TABLET_TOOL_AXIS_X);
                tool->axis_caps_bits.set(LIBINPUT_TABLET_TOOL_AXIS_Y);
                tool->axis_caps_bits.set(LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                tool->axis_caps_bits.set(LIBINPUT_TABLET_TOOL_AXIS_SIZE_MAJOR);
                tool->axis_caps_bits.set(LIBINPUT_TABLET_TOOL_AXIS_SIZE_MINOR);
                tool->buttons_bits.set(BTN_0);
                totem.li.tool_list.push_back(tool);
                return tool;
            }
            void slot_axes_initialize(totem_slot& slot)
            {
                slot.axes.point.x = totem.libevdev_get_slot_value(slot.index, ABS_MT_POSITION_X);
                slot.axes.point.y = totem.libevdev_get_slot_value(slot.index, ABS_MT_POSITION_Y);
                slot.last_point.x = slot.axes.point.x;
                slot.last_point.y = slot.axes.point.y;
            }
            void totem_slot_mark_all_axes_changed(totem_slot& slot, libinput_tablet_tool_sptr tool)
            {
                slot.changed_axes_bits = tool->axis_caps_bits;
            }
            bool totem_slot_fetch_axes(totem_slot& slot, libinput_tablet_tool_sptr tool, tablet_axes& axes_out, time now)
            {
                auto axes = tablet_axes{};
                auto rc = faux;
                if (!slot.changed_axes_bits.any())
                {
                    axes = slot.axes;
                }
                else
                {
                    if (slot.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_X]
                     || slot.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_Y])
                    {
                        slot.axes.point.x = totem.libevdev_get_slot_value(slot.index, ABS_MT_POSITION_X);
                        slot.axes.point.y = totem.libevdev_get_slot_value(slot.index, ABS_MT_POSITION_Y);
                    }
                    if (slot.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z])
                    {
                        auto angle = totem.libevdev_get_slot_value(slot.index, ABS_MT_ORIENTATION);
                        slot.axes.rotation = (360 - angle) % 360; // The kernel gives us ±90 degrees off neutral.
                    }
                    if (slot.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_SIZE_MAJOR]
                     || slot.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_SIZE_MINOR])
                    {
                        auto smin = totem.libevdev_get_slot_value(slot.index, ABS_MT_TOUCH_MINOR);
                        auto smax = totem.libevdev_get_slot_value(slot.index, ABS_MT_TOUCH_MAJOR);
                        auto rmin = totem.libevdev_get_abs_resolution(ABS_MT_TOUCH_MINOR);
                        auto rmax = totem.libevdev_get_abs_resolution(ABS_MT_TOUCH_MAJOR);
                        slot.axes.size_limits.min = (fp64)smin / rmin;
                        slot.axes.size_limits.max = (fp64)smax / rmax;
                    }
                    auto delta = fp64_coor{ slot.axes.point - slot.last_point };
                    axes.point       = slot.axes.point;
                    axes.rotation    = slot.axes.rotation;
                    axes.size_limits = slot.axes.size_limits;
                    axes.delta       = totem.pointer_filter->filter_dispatch(delta, tool.get(), now);
                    rc = true;
                }
                axes_out = axes;
                return rc;
            }
            void totem_slot_reset_changed_axes(totem_slot& slot)
            {
                slot.changed_axes_bits.reset();
            }
            slot_state_enum totem_handle_slot_state(totem_slot& slot, time now)
            {
                switch (slot.state)
                {
                    case SLOT_STATE_BEGIN:
                        if (!slot.tool)
                        {
                            slot.tool = totem_new_tool();
                        }
                        slot_axes_initialize(slot);
                        totem_slot_mark_all_axes_changed(slot, slot.tool);
                        break;
                    case SLOT_STATE_UPDATE:
                    case SLOT_STATE_END:  assert(slot.tool); break;
                    case SLOT_STATE_NONE: return SLOT_STATE_NONE;
                }
                auto axes = tablet_axes{};
                auto tip_state = LIBINPUT_TABLET_TOOL_TIP_UP;
                auto updated = totem_slot_fetch_axes(slot, slot.tool, axes, now);
                switch (slot.state)
                {
                    case SLOT_STATE_BEGIN:
                        tip_state = LIBINPUT_TABLET_TOOL_TIP_DOWN;
                        totem.tablet_notify_proximity(now, slot.tool, LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                        totem_slot_reset_changed_axes(slot);
                        totem.tablet_notify_tip(now, slot.tool, tip_state, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                        slot.state = SLOT_STATE_UPDATE;
                        break;
                    case SLOT_STATE_UPDATE:
                        tip_state = LIBINPUT_TABLET_TOOL_TIP_DOWN;
                        if (updated)
                        {
                            totem.tablet_notify_axis(now, slot.tool, tip_state, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                        }
                        break;
                    case SLOT_STATE_END: /* prox out is handled after button events */ break;
                    case SLOT_STATE_NONE:
                        ::abort();
                        break;
                }
                // We only have one button but possibly multiple totems. It's not
                // clear how the firmware will work, so for now we just handle the
                // button state in the first slot.
                //
                // Due to the design of the totem we're also less fancy about
                // button handling than the tablet code. Worst case, you might get
                // tip up before button up but meh.
                if (totem.button_state_now != totem.button_state_previous)
                {
                    auto btn_state = totem.button_state_now ? evdev::pressed : evdev::released;
                    totem.tablet_notify_button(now, slot.tool, tip_state, axes, BTN_0, btn_state, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                    totem.button_state_previous = totem.button_state_now;
                }
                if (slot.state == SLOT_STATE_END)
                {
                    tip_state = LIBINPUT_TABLET_TOOL_TIP_UP;
                    totem.tablet_notify_tip(now, slot.tool, tip_state, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                    totem_slot_reset_changed_axes(slot);
                    totem.tablet_notify_proximity(now, slot.tool, LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_OUT, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                    slot.state = SLOT_STATE_NONE;
                }
                else if (slot.state == SLOT_STATE_NONE) ::abort();
                slot.last_point = slot.axes.point;
                totem_slot_reset_changed_axes(slot);
                return slot.state;
            }
            slot_state_enum totem_handle_state(time now)
            {
                auto global_state = SLOT_STATE_NONE;
                for (auto& slot : totem.slots)
                {
                    auto s = totem_handle_slot_state(slot, now);
                    if (s != SLOT_STATE_NONE) // If one slot is active, the totem is active.
                    {
                        global_state = SLOT_STATE_UPDATE;
                    }
                }
                return global_state;
            }
            void totem_set_touch_device_enabled(bool enable_touch_device, time now)
            {
                auto touch_device = totem.touch_li_device;
                if (!touch_device) return;
                auto r = fp64_rect{};
                auto state = ARBITRATION_NOT_ACTIVE;
                // We just pick the coordinates of the first touch we find. The totem only does one tool right now despite being nominally an MT device, so let's not go too hard on ourselves.
                for (auto i = 0u; !enable_touch_device && i < totem.slots.size(); i++)
                {
                    auto& slot = totem.slots[i];
                    if (slot.state != SLOT_STATE_NONE) // Totem size is ~70mm. We could calculate the real size but until we need that, hardcoding it is enough.
                    {
                        auto mm = totem.evdev_device_units_to_mm(slot.axes.point);
                        r.coor.x = mm.x - 30;
                        r.coor.y = mm.y - 30;
                        r.size.x = 100;
                        r.size.y = 100;
                        state = ARBITRATION_IGNORE_RECT;
                        break;
                    }
                }
                if (enable_touch_device)
                {
                    touch_device->touch_arbitration_toggle(state, r, now);
                }
                else
                {
                    switch (totem.arbitration_state)
                    {
                        case ARBITRATION_IGNORE_ALL: ::abort();
                        case ARBITRATION_NOT_ACTIVE:  touch_device->touch_arbitration_toggle(state, r, now); break;
                        case ARBITRATION_IGNORE_RECT: touch_device->touch_arbitration_update_rect(r, now); break;
                    }
                }
                totem.arbitration_state = state;
            }
            void totem_process_abs(evdev_event& ev)
            {
                auto& slot = totem.slots[totem.slot_index];
                switch (ev.usage)
                {
                    case evdev::abs_mt_slot:
                        if ((ui64)ev.value >= totem.slots.size())
                        {
                            log("exceeded slot count (%d% of max %zd%)", ev.value, totem.slots.size());
                            ev.value = totem.slots.size() - 1;
                        }
                        totem.slot_index = ev.value;
                        return;
                    case evdev::abs_mt_tracking_id:
                        if (ev.value >= 0) // If the totem is already down on init, we currently ignore it.
                        {
                            slot.state = SLOT_STATE_BEGIN;
                        }
                        else if (slot.state != SLOT_STATE_NONE)
                        {
                            slot.state = SLOT_STATE_END;
                        }
                        break;
                    case evdev::abs_mt_position_x:  slot.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_X); break;
                    case evdev::abs_mt_position_y:  slot.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_Y); break;
                    case evdev::abs_mt_touch_major: slot.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_SIZE_MAJOR); break;
                    case evdev::abs_mt_touch_minor: slot.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_SIZE_MINOR); break;
                    case evdev::abs_mt_orientation: slot.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z); break;
                    case evdev::abs_mt_tool_type: if (ev.value != MT_TOOL_DIAL) log("Unexpected tool type %#x%, changing to dial", ev.usage); break;
                    default: log("Unhandled ABS event code %#x%", ev.usage); break;
                }
            }
            void totem_process_key(evdev_event& ev)
            {
                if (ev.value == 2) return; // Ignore kernel key repeat.
                switch (ev.usage)
                {
                    case evdev::btn_0: totem.button_state_now = !!ev.value; break;
                    default: log("Unhandled KEY event code %#x%", ev.usage); break;
                }
            }
            void totem_interface_process(evdev_event& ev, time now)
            {
                auto global_state = slot_state_enum{};
                auto enable_touch = faux;
                auto type = evdev_usage_type(ev.usage);
                switch (type)
                {
                    case EV_ABS: totem_process_abs(ev); break;
                    case EV_KEY: totem_process_key(ev); break;
                    case EV_MSC: /* timestamp, ignore */ break;
                    case EV_SYN:
                        global_state = totem_handle_state(now);
                        enable_touch = (global_state == SLOT_STATE_NONE);
                        totem_set_touch_device_enabled(enable_touch, now);
                        break;
                    default: log("Unexpected event 'code=%% type=%%'", evdev_usage_type(ev.usage), evdev_usage_code(ev.usage)); break;
                }
            }
            void totem_interface_suspend()
            {
                auto now = datetime::now();
                for (auto& slot : totem.slots)
                {
                    auto axes = tablet_axes{};
                    auto tip_state = libinput_tablet_tool_tip_state{};
                    if (slot.tool) // If we never initialized a tool, we can skip everything.
                    {
                        totem_slot_fetch_axes(slot, slot.tool, axes, now);
                        totem_slot_reset_changed_axes(slot);
                        tip_state = slot.state == SLOT_STATE_NONE ? LIBINPUT_TABLET_TOOL_TIP_UP : LIBINPUT_TABLET_TOOL_TIP_DOWN;
                        if (totem.button_state_now)
                        {
                            totem.tablet_notify_button(now, slot.tool, tip_state, axes, BTN_0, evdev::released, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                            totem.button_state_now = faux;
                            totem.button_state_previous = faux;
                        }
                        if (slot.state != SLOT_STATE_NONE)
                        {
                            totem.tablet_notify_tip(now, slot.tool, LIBINPUT_TABLET_TOOL_TIP_UP, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                        }
                        totem.tablet_notify_proximity(now, slot.tool, LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_OUT, slot.changed_axes_bits, axes, totem.ud_device.abs.absinfo_x, totem.ud_device.abs.absinfo_y);
                    }
                }
                totem_set_touch_device_enabled(true, now);
            }
            void totem_interface_device_added(libinput_device_sptr added_li_device)
            {
                if ((added_li_device->libevdev_get_id_vendor() != totem.libevdev_get_id_vendor())
                 || (added_li_device->libevdev_get_id_product() != totem.libevdev_get_id_product()))
                {
                    return;
                }
                // Virtual devices don't have device groups, so check for that libinput replay.
                auto& g1 = totem.device_group;
                auto& g2 = added_li_device->device_group;
                if (g1.size() && g2.size() && g1 != g2)
                {
                    return;
                }
                if (totem.touch_li_device != nullptr)
                {
                    log("already has a paired touch device, ignoring (%s%)", added_li_device->ud_device.devname);
                    return;
                }
                totem.touch_li_device = added_li_device;
                log("%s%: is the totem touch device", added_li_device->ud_device.devname);
            }
            void totem_interface_device_removed(libinput_device_sptr removed_li_device)
            {
                if (totem.touch_li_device != removed_li_device) return;
                totem_set_touch_device_enabled(true, datetime::now());
                totem.touch_li_device = {};
            }
            void totem_interface_initial_proximity(libinput_device_sptr li_device)
            {
                auto now = datetime::now();
                auto enable_touch = true;
                auto i = 0;
                for (auto& slot : totem.slots)
                {
                    auto axes = tablet_axes{};
                    auto tracking_id = li_device->libevdev_get_slot_value(i++, ABS_MT_TRACKING_ID);
                    if (tracking_id != -1)
                    {
                        slot.tool = totem_new_tool();
                        slot_axes_initialize(slot);
                        totem_slot_mark_all_axes_changed(slot, slot.tool);
                        totem_slot_fetch_axes(slot, slot.tool, axes, now);
                        li_device->tablet_notify_proximity(now, slot.tool, LIBINPUT_TABLET_TOOL_PROXIMITY_STATE_IN, slot.changed_axes_bits, axes, li_device->ud_device.abs.absinfo_x, li_device->ud_device.abs.absinfo_y);
                        totem_slot_reset_changed_axes(slot);
                        li_device->tablet_notify_tip(now, slot.tool, LIBINPUT_TABLET_TOOL_TIP_DOWN, slot.changed_axes_bits, axes, li_device->ud_device.abs.absinfo_x, li_device->ud_device.abs.absinfo_y);
                        slot.state = SLOT_STATE_UPDATE;
                        enable_touch = faux;
                    }
                }
                totem_set_touch_device_enabled(enable_touch, now);
            }
                static ui32 totem_accel_config_get_profiles([[maybe_unused]] libinput_device_sptr li_device)
                {
                    return LIBINPUT_CONFIG_ACCEL_PROFILE_NONE;
                }
                static libinput_config_status totem_accel_config_set_profile([[maybe_unused]] libinput_device_sptr li_device, [[maybe_unused]] libinput_config_accel_profile profile)
                {
                    return LIBINPUT_CONFIG_STATUS_UNSUPPORTED;
                }
                static libinput_config_accel_profile totem_accel_config_get_profile([[maybe_unused]] libinput_device_sptr li_device)
                {
                    return LIBINPUT_CONFIG_ACCEL_PROFILE_NONE;
                }
                static libinput_config_accel_profile totem_accel_config_get_default_profile([[maybe_unused]] libinput_device_sptr li_device)
                {
                    return LIBINPUT_CONFIG_ACCEL_PROFILE_NONE;
                }
            si32 totem_init_accel()
            {
                auto resolution = si32_coor{ totem.ud_device.abs.absinfo_x.resolution, totem.ud_device.abs.absinfo_y.resolution };
                auto filter = ptr::shared<tablet_accelerator_flat>(resolution); // Same filter as the tablet.
                totem.evdev_device_init_pointer_acceleration(filter);
                // We override the profile hooks for accel configuration with hooks that don't allow selection of profiles.
                totem.pointer_config.get_profiles        = totem_accel_config_get_profiles;
                totem.pointer_config.set_profile         = totem_accel_config_set_profile;
                totem.pointer_config.get_profile         = totem_accel_config_get_profile;
                totem.pointer_config.get_default_profile = totem_accel_config_get_default_profile;
                return 0;
            }
        };

        totem_impl_t totem_impl{ *this };
        void          process(evdev_event& ev, time stamp)              { totem_impl.       totem_interface_process(ev, stamp); }
        void          suspend()                                         { totem_impl.       totem_interface_suspend(); }
        void     device_added(libinput_device_sptr added_li_device)     { totem_impl.  totem_interface_device_added(added_li_device); }
        void   device_removed(libinput_device_sptr removed_li_device)   { totem_impl.totem_interface_device_removed(removed_li_device); }
        void device_suspended(libinput_device_sptr suspended_li_device) { device_removed(suspended_li_device); }
        void   device_resumed(libinput_device_sptr resumed_li_device)   { device_added(resumed_li_device); }
    };

    struct tablet_device : libinput_device_t
    {
        struct history_t
        {
            ui32        index{};
            ui32        count{};
            tablet_axes samples[lixx::tablet_history_length] = {};
            ui64        size{};
        };
        struct current_tool_t
        {
            libinput_tablet_tool_type type{};
            ui32                      id{};
            ui32                      serial{};
        };
        struct area_t
        {
            libinput_device_config_area config;
            fp64_rect                   have_area;
            fp64_rect                   want_area;
            abs_info_t                  x;//todo unify
            abs_info_t                  y;//
        };
        struct rotation_t
        {
            libinput_device_sptr touch_li_device; // The device locked for rotation.
            bool                 touch_device_left_handed_state{}; // Last known left-handed state of the touchpad.
            bool                 rotate{};
            bool                 want_rotate{};
        };
        struct quirks_t
        {
            bool                need_to_force_prox_out{};
            libinput_timer_sptr prox_out_timer;
            bool                proximity_out_forced{};
            time                last_event_time{};
            bool                proximity_out_in_progress{}; // True while injecting BTN_TOOL_PEN events.
        };

        ui32                                 tablet_id{}; // Incremental ID.
        ui32                                 status{};
        tablet_axes_bitset                   changed_axes_bits;
        tablet_axes                          axes; // For assembling the current state.
        si32_coor                            last_smooth_point;
        history_t                            history;
        tablet_axes_bitset                   axis_caps_bits;
        si32                                 current_value[lixx::libinput_tablet_tool_axis_cnt] = {};
        si32                                 prev_value[lixx::libinput_tablet_tool_axis_cnt] = {};
        std::list<libinput_tablet_tool_sptr> tool_list; // Only used for tablets that don't report serial numbers.
        button_state_t                       next_button_state;
        button_state_t                       prev_button_state;
        ui32                                 tool_state{};
        ui32                                 prev_tool_state{};
        current_tool_t                       current_tool;
        ui32                                 cursor_proximity_threshold{};
        libinput_device_config_calibration   calibration;
        area_t                               area;
        libinput_device_sptr                 touch_li_device; // The paired touch device on devices with both pen & touch.
        libinput_arbitration_state           arbitration{};
        rotation_t                           rotation;
        quirks_t                             quirks;

        tablet_device(auto&... args)
            : libinput_device_t{ args... }
        { }
        ~tablet_device()
        {
            if (auto& timer = quirks.prox_out_timer)
            {
                timer->cancel();
                timer.reset();
            }
            tool_list.clear();
            li.libinput_libwacom_unref();
        }

        struct tablet_impl_t
        {
            tablet_device& tablet;
                    libinput_tablet_tool_axis evdev_usage_to_axis(ui32 usage)
                    {
                        static const auto axis_lut = std::to_array<std::pair<ui32, libinput_tablet_tool_axis>>(
                        {
                            { evdev::abs_x       , LIBINPUT_TABLET_TOOL_AXIS_X          },
                            { evdev::abs_y       , LIBINPUT_TABLET_TOOL_AXIS_Y          },
                            { evdev::abs_z       , LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z },
                            { evdev::abs_distance, LIBINPUT_TABLET_TOOL_AXIS_DISTANCE   },
                            { evdev::abs_pressure, LIBINPUT_TABLET_TOOL_AXIS_PRESSURE   },
                            { evdev::abs_tilt_x  , LIBINPUT_TABLET_TOOL_AXIS_TILT_X     },
                            { evdev::abs_tilt_y  , LIBINPUT_TABLET_TOOL_AXIS_TILT_Y     },
                            { evdev::abs_wheel   , LIBINPUT_TABLET_TOOL_AXIS_SLIDER     },
                            { evdev::rel_wheel   , LIBINPUT_TABLET_TOOL_AXIS_REL_WHEEL  },
                        });
                        auto iter = std::ranges::find_if(axis_lut, [&](auto a){ return a.first == usage; });
                        return iter != axis_lut.end() ? iter->second : LIBINPUT_TABLET_TOOL_AXIS_NONE;
                    }
                    bool tablet_filter_axis_fuzz(evdev_event const& ev, libinput_tablet_tool_axis axis)
                    {
                        auto previous = tablet.prev_value[axis];
                        auto current = ev.value;
                        auto delta = previous - current;
                        auto fuzz = tablet.ud_device.libevdev_get_abs_fuzz(evdev_usage_code(ev.usage));
                        if (ev.usage == evdev::abs_distance) // ABS_DISTANCE doesn't have have fuzz set and causes continuous updates for the cursor/lens tools.
                        {
                            fuzz = std::max(2, fuzz); // Add a minimum fuzz of 2, same as the xf86-input-wacom driver.
                        }
                        return std::abs(delta) <= fuzz;
                    }
                void tablet_process_absolute(evdev_event& ev)
                {
                    auto axis = libinput_tablet_tool_axis{};
                    switch (ev.usage)
                    {
                        case evdev::abs_x:
                        case evdev::abs_y:
                        case evdev::abs_z:
                        case evdev::abs_pressure:
                        case evdev::abs_tilt_x:
                        case evdev::abs_tilt_y:
                        case evdev::abs_distance:
                        case evdev::abs_wheel:
                            axis = evdev_usage_to_axis(ev.usage);
                            if (axis == LIBINPUT_TABLET_TOOL_AXIS_NONE)
                            {
                                log("Invalid ABS event code %#x%", ev.usage);
                            }
                            else
                            {
                                tablet.prev_value[axis] = tablet.current_value[axis];
                                if (tablet_filter_axis_fuzz(ev, axis)) break;
                                tablet.current_value[axis] = ev.value;
                                tablet.changed_axes_bits.set(axis);
                                tablet.status |= TABLET_AXES_UPDATED;
                            }
                            break;
                        case evdev::abs_misc: // tool_id is the identifier for the tool we can use in libwacom to identify it (if we have one anyway).
                            tablet.current_tool.id = ev.value;
                            break;
                        // Intuos 3 strip data. Should only happen on the Pad device, not on the Pen device.
                        case evdev::abs_rx:
                        case evdev::abs_ry:
                        case evdev::abs_rz: // Only on the 4D mouse (Intuos2), obsolete.
                        // Only on the 4D mouse (Intuos2), obsolete. The 24HD sends ABS_THROTTLE on the Pad device for the second wheel but we shouldn't get here on kernel >= 3.17.
                        case evdev::abs_throttle:
                        default: log("Unhandled ABS event code %#x%", ev.usage); break;
                    }
                }
                void tablet_process_relative(evdev_event& ev)
                {
                    if (ev.usage == evdev::rel_wheel)
                    {
                        auto axis = evdev_usage_to_axis(ev.usage);
                        if (axis == LIBINPUT_TABLET_TOOL_AXIS_NONE)
                        {
                            log("Invalid ABS event code %#x%", ev.usage);
                        }
                        else
                        {
                            tablet.changed_axes_bits.set(axis);
                            tablet.axes.wheel_discrete = -1 * ev.value;
                            tablet.status |= TABLET_AXES_UPDATED;
                        }
                    }
                    else
                    {
                        log("Unhandled relative axis 'type=%% code=%%'", evdev_usage_type(ev.usage), evdev_usage_code(ev.usage));
                    }
                }
                    void _tablet_set_state(evdev_event& ev, libinput_tablet_tool_type type)
                    {
                        tablet.status |= TABLET_TOOL_UPDATED;
                        if (ev.value) tablet.tool_state |= (1ul << type);
                        else          tablet.tool_state &= ~(1ul << type);
                    }
                void tablet_process_key(evdev_event& ev)
                {
                    if (ev.value == 2) return; // Ignore kernel key repeat.
                    auto usage = ev.usage;
                    switch (usage)
                    {
                        case evdev::btn_tool_finger: log("Invalid tool 'finger' on tablet interface"); break;
                        case evdev::btn_tool_pen:      _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_PEN);      break;
                        case evdev::btn_tool_rubber:   _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_ERASER);   break;
                        case evdev::btn_tool_brush:    _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_BRUSH);    break;
                        case evdev::btn_tool_pencil:   _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_PENCIL);   break;
                        case evdev::btn_tool_airbrush: _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH); break;
                        case evdev::btn_tool_mouse:    _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_MOUSE);    break;
                        case evdev::btn_tool_lens:     _tablet_set_state(ev, LIBINPUT_TABLET_TOOL_TYPE_LENS);     break;
                        case evdev::btn_touch:
                            if (!tablet.axis_caps_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE])
                            {
                                if (ev.value) tablet.status |= TABLET_TOOL_ENTERING_CONTACT;
                                else          tablet.status |= TABLET_TOOL_LEAVING_CONTACT;
                            }
                            break;
                        case evdev::btn_left:
                        case evdev::btn_right:
                        case evdev::btn_middle:
                        case evdev::btn_side:
                        case evdev::btn_extra:
                        case evdev::btn_forward:
                        case evdev::btn_back:
                        case evdev::btn_task:
                        case evdev::btn_stylus:
                        case evdev::btn_stylus2:
                        case evdev::btn_stylus3:
                            if (ev.value)
                            {
                                tablet.next_button_state.set(evdev_usage_code(usage));
                                tablet.status |= TABLET_BUTTONS_PRESSED;
                            }
                            else
                            {
                                tablet.next_button_state.reset(evdev_usage_code(usage));
                                tablet.status |= TABLET_BUTTONS_RELEASED;
                            }
                            break;
                        default: log("Unhandled button 'type=%% code='%%'", evdev_usage_type(usage), evdev_usage_code(usage)); break;
                    }
                }
                void tablet_process_misc(evdev_event& ev)
                {
                    switch (ev.usage)
                    {
                        case evdev::msc_serial: if (ev.value != -1) tablet.current_tool.serial = ev.value; break;
                        case evdev::msc_scan: break;
                        default: log("Unhandled MSC event code 'type=%% code='%%'", evdev_usage_type(ev.usage), evdev_usage_code(ev.usage)); break;
                    }
                }
                        void tablet_update_tool(libinput_tablet_tool_type tool, bool enabled)
                        {
                            assert(tool != LIBINPUT_TABLET_TOOL_TYPE_NONE);
                            if (enabled)
                            {
                                tablet.current_tool.type = tool;
                                tablet.status |= TABLET_TOOL_ENTERING_PROXIMITY;
                                tablet.status &= ~TABLET_TOOL_OUT_OF_PROXIMITY;
                            }
                            else if (tablet.status & TABLET_TOOL_OUT_OF_PROXIMITY)
                            {
                                tablet.status |= TABLET_TOOL_LEAVING_PROXIMITY;
                            }
                        }
                        void tablet_proximity_out_quirk_set_timer(time stamp)
                        {
                            // Handling for the proximity out workaround. Some tablets only send
                            // BTN_TOOL_PEN on the very first event, then leave it set even when the pen
                            // leaves the detectable range. To libinput this looks like we always have
                            // the pen in proximity.
                            //
                            // To avoid this, we set a timer on BTN_TOOL_PEN in. We expect the tablet to
                            // continuously send events, and while it's doing so we keep updating the
                            // timer. Once we go Xms without an event we assume proximity out and inject
                            // a BTN_TOOL_PEN event into the sequence through the timer func.
                            //
                            // We need to remember that we did that, on the first event after the
                            // timeout we need to emulate a BTN_TOOL_PEN event again to force proximity
                            // in.
                            //
                            // Other tools never send the BTN_TOOL_PEN event. For those tools, we
                            // piggyback along with the proximity out quirks by injecting
                            // the event during the first event frame.
                            if (tablet.quirks.need_to_force_prox_out)
                            {
                                tablet.quirks.prox_out_timer->start(stamp + lixx::forced_proxout_timeout);
                            }
                        }
                    bool tablet_update_tool_state(time stamp)
                    {
                        auto type = libinput_tablet_tool_type{};
                        auto changed = 0u;
                        auto state = 0;
                        auto doubled_up_new_tool_bit = 0u;
                        // We were already out of proximity but now got a tool update but our tool state is zero - i.e. we got a valid prox out from the device.
                        if (tablet.quirks.proximity_out_forced && (tablet.status & TABLET_TOOL_UPDATED) && !tablet.tool_state)
                        {
                            tablet.quirks.need_to_force_prox_out = faux;
                            tablet.quirks.proximity_out_forced = faux;
                        }
                        // We need to emulate a BTN_TOOL_PEN if we get an axis event (i.e. stylus is def. in proximity) and:
                        //  - we forced a proximity out before, or
                        //  - on the very first event after init, because if we didn't get a
                        //    BTN_TOOL_PEN and the state for the tool was 0, this device will never send the event.
                        // We don't do this for pure button events because we discard those.
                        // But: on some devices the proximity out is delayed by the kernel,
                        // so we get it after our forced prox-out has triggered. In that
                        // case we need to just ignore the change.
                        if (tablet.status & TABLET_AXES_UPDATED)
                        {
                            if (tablet.quirks.proximity_out_forced)
                            {
                                if (!(tablet.status & TABLET_TOOL_UPDATED) && !tablet.tool_state)
                                {
                                    tablet.tool_state = (1ul << LIBINPUT_TABLET_TOOL_TYPE_PEN);
                                }
                                tablet.quirks.proximity_out_forced = faux;
                            }
                            else if (tablet.tool_state == 0 && tablet.current_tool.type == LIBINPUT_TABLET_TOOL_TYPE_NONE)
                            {
                                tablet.tool_state = (1ul << LIBINPUT_TABLET_TOOL_TYPE_PEN);
                                tablet.quirks.proximity_out_forced = faux;
                            }
                        }
                        if (tablet.tool_state == tablet.prev_tool_state) return faux;
                        // Kernel tools are supposed to be mutually exclusive, but we may have
                        //   two bits set due to firmware/kernel bugs.
                        //   Two cases that have been seen in the wild:
                        //   - BTN_TOOL_PEN on proximity in, followed by
                        //     BTN_TOOL_RUBBER later, see #259
                        //     -> We force a prox-out of the pen, trigger prox-in for eraser
                        //   - BTN_TOOL_RUBBER on proximity in, but BTN_TOOL_PEN when
                        //     the tip is down, see #702.
                        //     -> We ignore BTN_TOOL_PEN
                        //   In both cases the eraser is what we want, so we bias
                        //   towards that.
                        if (tablet.tool_state & (tablet.tool_state - 1))
                        {
                            doubled_up_new_tool_bit = tablet.tool_state ^ tablet.prev_tool_state;
                            if (doubled_up_new_tool_bit == (1ul << LIBINPUT_TABLET_TOOL_TYPE_PEN)) // The new tool is the pen. Ignore it.
                            {
                                tablet.tool_state &= ~(1ul << LIBINPUT_TABLET_TOOL_TYPE_PEN);
                                return faux;
                            }
                            // The new tool is some tool other than pen (usually eraser).
                            // We set the current tool state to zero, thus setting
                            // everything up for a prox out on the tool. Once that is set
                            // up, we change the tool state to be the new one we just got.
                            // When we re-process this function we now get the new tool
                            // as prox in. Importantly, we basically rely on nothing else
                            // happening in the meantime.
                            tablet.tool_state = 0;
                        }
                        changed = tablet.tool_state ^ tablet.prev_tool_state;
                        type = (libinput_tablet_tool_type)(ffs(changed) - 1);
                        state = !!(tablet.tool_state & (1ul << type));
                        tablet_update_tool(type, state);
                        if (type == LIBINPUT_TABLET_TOOL_TYPE_PEN) // The proximity timeout is only needed for BTN_TOOL_PEN, devices that require it don't do erasers.
                        {
                            if (state)
                            {
                                tablet_proximity_out_quirk_set_timer(stamp);
                            }
                            else
                            {
                                if (!tablet.quirks.proximity_out_in_progress) // If we get a BTN_TOOL_PEN 0 when *not* injecting events it means the tablet will give us the right events after all and we can disable our timer-based proximity out.
                                {
                                    tablet.quirks.need_to_force_prox_out = faux;
                                }
                                tablet.quirks.prox_out_timer->cancel();
                            }
                        }
                        tablet.prev_tool_state = tablet.tool_state;
                        if (doubled_up_new_tool_bit)
                        {
                            tablet.tool_state = doubled_up_new_tool_bit;
                            return true; // Need to re-process.
                        }
                        return faux;
                    }
                                    void apply_pressure_range_configuration(libinput_tablet_tool_sptr tool, bool force_update)
                                    {
                                        if (!tablet.libevdev_has_event_code<EV_ABS>(ABS_PRESSURE)
                                         || (!force_update && tool->pressure.range.min == tool->pressure.wanted_range.min && tool->pressure.range.max == tool->pressure.wanted_range.max))
                                        {
                                            return;
                                        }
                                        tool->pressure.range.min = tool->pressure.wanted_range.min;
                                        tool->pressure.range.max = tool->pressure.wanted_range.max;
                                    }
                                void tool_init_pressure_thresholds(libinput_tablet_tool_sptr tool)
                                {
                                    auto& pressure = tool->pressure;
                                    pressure.tablet_id            = tablet.tablet_id;
                                    pressure.threshold_offset     = 0.0;
                                    pressure.threshold_has_offset = faux;
                                    pressure.threshold.min        = 0;
                                    pressure.threshold.max        = 1;
                                    auto& pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE);
                                    if (!pressure_abs_info) return;
                                    pressure.abs_info = pressure_abs_info;
                                    auto& distance_abs_info = tablet.libevdev_get_abs_info(ABS_DISTANCE);
                                    if (distance_abs_info)
                                    {
                                        pressure.threshold_offset = 0.0;
                                        pressure.heuristic_state = PRESSURE_HEURISTIC_STATE_DONE;
                                    }
                                    else
                                    {
                                        pressure.threshold_offset = 1.0;
                                        pressure.heuristic_state = PRESSURE_HEURISTIC_STATE_PROXIN1;
                                    }
                                    apply_pressure_range_configuration(tool, true);
                                }
                                    void copy_axis_cap(libinput_tablet_tool_sptr tool, libinput_tablet_tool_axis axis)
                                    {
                                        if (tablet.axis_caps_bits[axis])
                                        {
                                            tool->axis_caps_bits.set(axis);
                                        }
                                    }
                                    void copy_button_cap(libinput_tablet_tool_sptr tool, ui32 button)
                                    {
                                        if (tablet.libevdev_has_event_code<EV_KEY>(button))
                                        {
                                            tool->buttons_bits.set(button);
                                        }
                                    }
                                    bool tool_set_bits_from_libwacom([[maybe_unused]] libinput_tablet_tool_sptr tool)
                                    {
                                        auto rc = faux;
                                        #if HAVE_LIBWACOM
                                        auto db = tablet.li.libwacom.db;
                                        if (!db) return rc;
                                        #pragma GCC diagnostic push
                                        #pragma GCC diagnostic ignored "-Wdeprecated-declarations"
                                        auto s = ::libwacom_stylus_get_for_id(db, tool->tool_id);
                                        #pragma GCC diagnostic pop
                                        if (!s) return rc;
                                        auto type = ::libwacom_stylus_get_type(s);
                                        if (type == WSTYLUS_PUCK)
                                        {
                                            for (auto code = BTN_LEFT; code < BTN_LEFT + ::libwacom_stylus_get_num_buttons(s); code++)
                                            {
                                                copy_button_cap(tool, code);
                                            }
                                        }
                                        else
                                        {
                                            if (libwacom_stylus_get_num_buttons(s) >= 3) copy_button_cap(tool, BTN_STYLUS3);
                                            if (libwacom_stylus_get_num_buttons(s) >= 2) copy_button_cap(tool, BTN_STYLUS2);
                                            if (libwacom_stylus_get_num_buttons(s) >= 1) copy_button_cap(tool, BTN_STYLUS);
                                        }
                                        if (libwacom_stylus_has_wheel(s)) copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_REL_WHEEL);
                                        auto axes = ::libwacom_stylus_get_axes(s);
                                        if (axes & WACOM_AXIS_TYPE_TILT)
                                        {
                                            if (type == WSTYLUS_PUCK) // Tilt on the puck is converted to rotation.
                                            {
                                                set_bit(tool->axis_caps, LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                                            }
                                            else
                                            {
                                                copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_TILT_X);
                                                copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_TILT_Y);
                                            }
                                        }
                                        if (axes & WACOM_AXIS_TYPE_ROTATION_Z) copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                                        if (axes & WACOM_AXIS_TYPE_DISTANCE)   copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_DISTANCE);
                                        if (axes & WACOM_AXIS_TYPE_SLIDER)     copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_SLIDER);
                                        if (axes & WACOM_AXIS_TYPE_PRESSURE)   copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_PRESSURE);
                                        rc = true;
                                        #endif
                                        return rc;
                                    }
                                void tool_set_bits(libinput_tablet_tool_sptr tool)
                                {
                                    auto type = tool->type;
                                    copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_X);
                                    copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_Y);
                                    if (tool_set_bits_from_libwacom(tool)) return;
                                    // If we don't have libwacom, we simply copy any axis we have on the tablet onto the tool. Except we know that mice only have rotation anyway.
                                    switch (type)
                                    {
                                        case LIBINPUT_TABLET_TOOL_TYPE_PEN:
                                        case LIBINPUT_TABLET_TOOL_TYPE_ERASER:
                                        case LIBINPUT_TABLET_TOOL_TYPE_PENCIL:
                                        case LIBINPUT_TABLET_TOOL_TYPE_BRUSH:
                                        case LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH:
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_PRESSURE);
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_DISTANCE);
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_TILT_X);
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_TILT_Y);
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_SLIDER);
                                            // Rotation is special, it can be either ABS_Z or BTN_TOOL_MOUSE+ABS_TILT_X/Y. Aiptek tablets have mouse+tilt (and thus rotation), but they do not have ABS_Z. So let's not copy the axis bit if we don't have ABS_Z, otherwise we try to get the value from it later on proximity in and go boom because the absinfo isn't there.
                                            if (tablet.libevdev_has_event_code<EV_ABS>(ABS_Z))
                                            {
                                                copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                                            }
                                            break;
                                        case LIBINPUT_TABLET_TOOL_TYPE_MOUSE:
                                        case LIBINPUT_TABLET_TOOL_TYPE_LENS:
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                                            copy_axis_cap(tool, LIBINPUT_TABLET_TOOL_AXIS_REL_WHEEL);
                                            break;
                                        default: break;
                                    }
                                    switch (type) // If we don't have libwacom, copy all pen-related buttons from the tablet vs all mouse-related buttons.
                                    {
                                    case LIBINPUT_TABLET_TOOL_TYPE_PEN:
                                    case LIBINPUT_TABLET_TOOL_TYPE_BRUSH:
                                    case LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH:
                                    case LIBINPUT_TABLET_TOOL_TYPE_PENCIL:
                                    case LIBINPUT_TABLET_TOOL_TYPE_ERASER:
                                        copy_button_cap(tool, BTN_STYLUS);
                                        copy_button_cap(tool, BTN_STYLUS2);
                                        copy_button_cap(tool, BTN_STYLUS3);
                                        break;
                                    case LIBINPUT_TABLET_TOOL_TYPE_MOUSE:
                                    case LIBINPUT_TABLET_TOOL_TYPE_LENS:
                                        copy_button_cap(tool, BTN_LEFT);
                                        copy_button_cap(tool, BTN_MIDDLE);
                                        copy_button_cap(tool, BTN_RIGHT);
                                        copy_button_cap(tool, BTN_SIDE);
                                        copy_button_cap(tool, BTN_EXTRA);
                                        break;
                                    default: break;
                                    }
                                }
                            libinput_tablet_tool_sptr tablet_new_tool(libinput_tablet_tool_type type, ui32 tool_id, ui32 serial)
                            {
                                auto tool = ptr::shared<libinput_tablet_tool>();
                                tool->serial   = serial,
                                tool->tool_id  = tool_id,
                                tool->type     = type,
                                tool->pressure = { .range        = { .min = 0.0, .max = 0.0 }, // To trigger configuration.
                                                   .wanted_range = { .min = 0.0, .max = 1.0 }},
                                tool_init_pressure_thresholds(tool);
                                tool_set_bits(tool);
                                return tool;
                            }
                        libinput_tablet_tool_sptr tablet_get_tool(libinput_tablet_tool_type type, ui32 tool_id, ui32 serial)
                        {
                            auto& li = tablet.li;
                            auto tool = libinput_tablet_tool_sptr{};
                            if (serial)
                            {
                                for (auto t : li.tool_list) // Check if we already have the tool in our list of tools.
                                {
                                    if (type == t->type && serial == t->serial)
                                    {
                                        tool = t;
                                        break;
                                    }
                                }
                            }
                            if (!tool) // If we get a tool with a delayed serial number, we already created a 0-serial number tool for it earlier. Re-use that, even though it means we can't distinguish this tool from others. https://bugs.freedesktop.org/show_bug.cgi?id=97526.
                            {
                                for (auto t : tablet.tool_list) // We can't guarantee that tools without serial numbers are unique, so we keep them local to the tablet that they come into proximity of instead of storing them in the global tool list. Same as above, but don't bother checking the serial number.
                                {
                                    if (type == t->type)
                                    {
                                        tool = t;
                                        break;
                                    }
                                }
                            }
                            if (!tool) // If we didn't already have the new_tool in our list of tools, add it.
                            {
                                tool = tablet_new_tool(type, tool_id, serial);
                                auto& dest_list = serial ? li.tool_list : tablet.tool_list; // Select dest_tool so we create in the correct list.
                                dest_list.push_back(tool);
                            }
                            return tool;
                        }
                    libinput_tablet_tool_sptr tablet_get_current_tool()
                    {
                        auto ok = tablet.current_tool.type != LIBINPUT_TABLET_TOOL_TYPE_NONE;
                        return ok ? tablet_get_tool(tablet.current_tool.type, tablet.current_tool.id, tablet.current_tool.serial) : libinput_tablet_tool_sptr{};
                    }
                        void tablet_mark_all_axes_changed(libinput_tablet_tool_sptr tool)
                        {
                            tablet.changed_axes_bits = tool->axis_caps_bits;
                        }
                    void tablet_update_proximity_state(libinput_tablet_tool_sptr tool)
                    {
                        auto dist_max = tablet.cursor_proximity_threshold;
                        auto& distance = tablet.libevdev_get_abs_info(ABS_DISTANCE);
                        if (!distance || distance.value == 0) return;
                        auto dist = (ui32)distance.value;
                        if (dist < dist_max && (tablet.status & (TABLET_TOOL_OUT_OF_RANGE | TABLET_TOOL_OUT_OF_PROXIMITY))) // Tool got into permitted range.
                        {
                            tablet.status &= ~TABLET_TOOL_OUT_OF_RANGE;
                            tablet.status &= ~TABLET_TOOL_OUT_OF_PROXIMITY;
                            tablet.status |= TABLET_TOOL_ENTERING_PROXIMITY;
                            tablet_mark_all_axes_changed(tool);
                            tablet.status |= TABLET_BUTTONS_PRESSED;
                            tablet.next_button_state |= tablet.prev_button_state; // tablet_force_button_presses
                            tablet.prev_button_state.reset();                  //
                        }
                        else if (dist >= dist_max && !(tablet.status & TABLET_TOOL_OUT_OF_RANGE) && !(tablet.status & TABLET_TOOL_OUT_OF_PROXIMITY))
                        {
                            if (tablet.status & TABLET_TOOL_ENTERING_PROXIMITY) // Tool entered prox but is outside of permitted range.
                            {
                                tablet.status |= TABLET_TOOL_OUT_OF_RANGE;
                                tablet.status &= ~TABLET_TOOL_ENTERING_PROXIMITY;
                            }
                            else // Tool was in prox and is now outside of range. Set leaving proximity, on the next event it will be OUT_OF_PROXIMITY and thus caught by the above conditions.
                            {
                                tablet.status |= TABLET_TOOL_LEAVING_PROXIMITY;
                            }
                        }
                    }
                    bool is_inside_area(si32_coor point, fp64 normalized_margin)
                    {
                        if (tablet.area.have_area.coor.x == 0.0 && tablet.area.have_area.size.x == 1.0
                         && tablet.area.have_area.coor.y == 0.0 && tablet.area.have_area.size.y == 1.0)
                        {
                            return true;
                        }
                        assert(normalized_margin > 0.0);
                        assert(normalized_margin <= 1.0);
                        auto xmargin = (si32)((tablet.area.x.maximum - tablet.area.x.minimum) * normalized_margin);
                        auto ymargin = (si32)((tablet.area.y.maximum - tablet.area.y.minimum) * normalized_margin);
                        return (point.x >= tablet.area.x.minimum - xmargin
                             && point.x <= tablet.area.x.maximum + xmargin
                             && point.y >= tablet.area.y.minimum - ymargin
                             && point.y <= tablet.area.y.maximum + ymargin);
                    }
                        bool tablet_get_quirked_pressure_thresholds(si32* hi, si32* lo)
                        {
                            auto status = faux;
                            if (auto q = tablet.li.quirks_fetch_for_device(tablet.ud_device))
                            {
                                // Note: the quirk term "range" refers to the hi/lo settings, not the full available range for the pressure axis.
                                auto r = si32_range{};
                                if (q->quirks_get(QUIRK_ATTR_PRESSURE_RANGE, r))
                                {
                                    if (r.min < r.max)
                                    {
                                        *hi = r.min;
                                        *lo = r.max;
                                        status = true;
                                    }
                                    else
                                    {
                                        log("Invalid pressure range, using defaults");
                                    }
                                }
                            }
                            return status;
                        }
                    void update_pressure_range(libinput_tablet_tool_sptr tool)
                    {
                        auto min = tool->pressure.range.min;
                        auto max = tool->pressure.range.max;
                        auto pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE);
                        auto minimum = pressure_abs_info.axis_range_percentage(min * 100.0);
                        auto maximum = pressure_abs_info.axis_range_percentage(max * 100.0);
                        pressure_abs_info.minimum = minimum;
                        pressure_abs_info.maximum = maximum;
                        // Only use the quirk pressure range if we don't have a custom range.
                        auto hi = 0;
                        auto lo = 0;
                        if (tool->pressure.range.min != 0.0 || tool->pressure.range.max != 1.0
                            || !tablet_get_quirked_pressure_thresholds(&hi, &lo))
                        {
                            // 5 and 1% of the pressure range.
                            hi = pressure_abs_info.axis_range_percentage(5);
                            lo = pressure_abs_info.axis_range_percentage(1);
                        }
                        auto& pressure = tool->pressure;
                        pressure.abs_info      = pressure_abs_info;
                        pressure.threshold.min = lo;
                        pressure.threshold.max = hi;
                        if (pressure.threshold_has_offset) pressure.set_pressure_offset(pressure.threshold_offset);
                        if (tool->pressure.has_configured_range) // Disable any heuristics.
                        {
                            pressure.threshold_has_offset = true;
                            pressure.heuristic_state = PRESSURE_HEURISTIC_STATE_DONE;
                        }
                    }
                    void update_pressure_offset(libinput_tablet_tool_sptr tool)
                    {
                        auto& pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE);
                        if (pressure_abs_info && !tool->pressure.has_configured_range && tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE])
                        {
                            // If we have an event that falls below the current offset, adjust the offset downwards. A fast contact can start with a higher-than-needed pressure offset and then we'd be tied into a high pressure offset for the rest of the session.
                            // If we are still pending the offset decision, only update the observed offset value, don't actually set it to have an offset.
                            auto offset = pressure_abs_info.pressure_offset_from_absinfo(pressure_abs_info.value);
                            auto& pressure = tool->pressure;
                            if (pressure.threshold_has_offset)
                            {
                                if (offset < pressure.threshold_offset)
                                {
                                    pressure.set_pressure_offset(offset);
                                }
                            }
                            else if (pressure.heuristic_state != PRESSURE_HEURISTIC_STATE_DONE)
                            {
                                pressure.threshold_offset = std::min(offset, pressure.threshold_offset);
                            }
                        }
                    }
                        char const* tablet_tool_type_to_string(libinput_tablet_tool_type type)
                        {
                            switch (type)
                            {
                                case LIBINPUT_TABLET_TOOL_TYPE_PEN:      return "pen";
                                case LIBINPUT_TABLET_TOOL_TYPE_ERASER:   return "eraser";
                                case LIBINPUT_TABLET_TOOL_TYPE_BRUSH:    return "brush";
                                case LIBINPUT_TABLET_TOOL_TYPE_PENCIL:   return "pencil";
                                case LIBINPUT_TABLET_TOOL_TYPE_AIRBRUSH: return "airbrush";
                                case LIBINPUT_TABLET_TOOL_TYPE_MOUSE:    return "mouse";
                                case LIBINPUT_TABLET_TOOL_TYPE_LENS:     return "lens";
                                default:                                 return "unknown";
                            }
                        }
                    void detect_pressure_offset(libinput_tablet_tool_sptr tool)
                    {
                        if (tool->pressure.has_configured_range || !tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE])
                        {
                            return;
                        }
                        auto& pressure = tool->pressure;
                        if (pressure.threshold_has_offset) return;
                        auto& pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE);
                        auto& distance_abs_info = tablet.libevdev_get_abs_info(ABS_DISTANCE);
                        if (!pressure_abs_info || pressure_abs_info.value <= pressure_abs_info.minimum) return;
                        auto units = pressure_abs_info.value;
                        auto offset = pressure_abs_info.pressure_offset_from_absinfo(units);
                        if (distance_abs_info)
                        {
                            if (distance_abs_info.value < distance_abs_info.axis_range_percentage(50)) // If we're closer than 50% of the distance axis, skip pressure offset detection, too likely to be wrong.
                            {
                                return;
                            }
                        }
                        else
                        {
                            // A device without distance will always have some pressure on contact. Offset detection is delayed for a few proximity ins in the hope we'll find the minimum value until then. That offset is updated during motion events so by the time the deciding prox-in arrives we should know the minimum offset.
                            if (units > pressure_abs_info.minimum)
                            {
                                pressure.threshold_offset = std::min(offset, pressure.threshold_offset);
                            }
                            switch (pressure.heuristic_state)
                            {
                                case PRESSURE_HEURISTIC_STATE_PROXIN1:
                                case PRESSURE_HEURISTIC_STATE_PROXIN2:
                                    pressure.heuristic_state = (pressure_heuristic_enum)(pressure.heuristic_state + 1);
                                    return;
                                case PRESSURE_HEURISTIC_STATE_DECIDE:
                                    pressure.heuristic_state = (pressure_heuristic_enum)(pressure.heuristic_state + 1);
                                    units = pressure_abs_info.pressure_offset_to_absinfo(pressure.threshold_offset);
                                    offset = pressure.threshold_offset;
                                    break;
                                case PRESSURE_HEURISTIC_STATE_DONE:
                                    return;
                            }
                        }
                        if (units > pressure_abs_info.minimum)
                        {
                            if (offset > 0.5) log("Ignoring pressure offset greater than 50 percents detected on tool %s% (serial %#x%)", tablet_tool_type_to_string(tool->type), tool->serial);
                            else              log("Pressure offset detected on tool %s% (serial %#x%)", tablet_tool_type_to_string(tool->type), tool->serial);
                            pressure.set_pressure_offset(offset);
                        }
                    }
                    void detect_tool_contact(libinput_tablet_tool_sptr tool)
                    {
                        if (tool->axis_caps_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE])
                        {
                            if (tablet.status & TABLET_TOOL_ENTERING_CONTACT) // If we have pressure, always use that for contact, not BTN_TOUCH.
                            {
                                log("Invalid status: entering contact");
                            }
                            if ((tablet.status & TABLET_TOOL_LEAVING_CONTACT) && !(tablet.status & TABLET_TOOL_LEAVING_PROXIMITY))
                            {
                                log("Invalid status: leaving contact");
                            }
                            if (auto& pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE))
                            {
                                auto pressure_value = pressure_abs_info.value;
                                auto& pressure = tool->pressure;
                                if (pressure_value <= pressure.threshold.min && (tablet.status & TABLET_TOOL_IN_CONTACT))
                                {
                                    tablet.status |= TABLET_TOOL_LEAVING_CONTACT;
                                }
                                else if (pressure_value >= pressure.threshold.max && !(tablet.status & TABLET_TOOL_IN_CONTACT))
                                {
                                    tablet.status |= TABLET_TOOL_ENTERING_CONTACT;
                                }
                            }
                            else
                            {
                                log("Missing pressure axis");
                            }
                        }
                    }
                                void apply_tablet_area(si32_coor& point)
                                {
                                    if (tablet.area.have_area.coor.x == 0.0 && tablet.area.have_area.size.x == 1.0
                                     && tablet.area.have_area.coor.y == 0.0 && tablet.area.have_area.size.y == 1.0)
                                    {
                                        return;
                                    }
                                    // The point is somewhere on the tablet in device coordinates,
                                    // but we need it relative to the x/y offset.
                                    // So clip it first, then offset it to our area min/max.
                                    //
                                    // Right now we're just clipping, we don't completely
                                    // ignore events. What we should do is ignore events outside
                                    // altogether and generate prox in/out events when we actually
                                    // enter the area.
                                    point.x = std::min(point.x, tablet.area.x.maximum);
                                    point.y = std::min(point.y, tablet.area.y.maximum);
                                    point.x = std::max(point.x, tablet.area.x.minimum);
                                    point.y = std::max(point.y, tablet.area.y.minimum);
                                }
                            void tablet_update_xy()
                            {
                                if (tablet.libevdev_has_event_code<EV_ABS>(ABS_X)
                                 && tablet.libevdev_has_event_code<EV_ABS>(ABS_Y))
                                {
                                    if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_X]
                                     || tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_Y])
                                    {
                                        auto& absinfo_x = tablet.ud_device.abs.absinfo_x;
                                        auto& absinfo_y = tablet.ud_device.abs.absinfo_y;
                                        tablet.axes.point.x = tablet.rotation.rotate ? absinfo_x.invert_axis() : absinfo_x.value;
                                        tablet.axes.point.y = tablet.rotation.rotate ? absinfo_y.invert_axis() : absinfo_y.value;
                                        // Calibration and area are currently mutually exclusive so one of those is a noop.
                                        tablet.evdev_transform_absolute(tablet.axes.point);
                                        apply_tablet_area(tablet.axes.point);
                                    }
                                }
                            }
                            fp64_coor tablet_tool_process_delta(libinput_tablet_tool_sptr tool, tablet_axes& axes, time stamp)
                            {
                                auto delta = si32_coor{};
                                // When tool contact changes, we probably got a cursor jump. Don't try to calculate a delta for that event.
                                if (!(tablet.status & TABLET_TOOL_ENTERING_PROXIMITY)
                                 && !(tablet.status & TABLET_TOOL_ENTERING_CONTACT)
                                 && !(tablet.status & TABLET_TOOL_LEAVING_CONTACT)
                                 && (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_X] || tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_Y]))
                                {
                                    delta = axes.point - tablet.last_smooth_point;
                                }
                                if (axes.point.x != tablet.last_smooth_point.x) tablet.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_X);
                                if (axes.point.y != tablet.last_smooth_point.y) tablet.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_Y);
                                tablet.last_smooth_point = axes.point;
                                auto accel = fp64_coor{ delta };
                                if (!accel) return {};
                                else        return tablet.pointer_filter->filter_dispatch(accel, tool.get(), stamp);
                            }
                            void tablet_update_pressure(libinput_tablet_tool_sptr tool)
                            {
                                auto& pressure_abs_info = tablet.libevdev_get_abs_info(ABS_PRESSURE);
                                if (pressure_abs_info && tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_PRESSURE])
                                {
                                    auto& pressure = tool->pressure;
                                    tablet.axes.pressure = pressure.normalize_pressure(pressure_abs_info.value);
                                }
                            }
                            void tablet_update_distance()
                            {
                                if (!tablet.libevdev_has_event_code<EV_ABS>(ABS_DISTANCE)) return;
                                if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_DISTANCE])
                                {
                                    auto& absinfo = tablet.libevdev_get_abs_info(ABS_DISTANCE);
                                    tablet.axes.distance = absinfo.absinfo_normalize(); // Normalize distance.
                                }
                            }
                            void tablet_update_slider()
                            {
                                if (!tablet.libevdev_has_event_code<EV_ABS>(ABS_WHEEL)) return;
                                if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_SLIDER])
                                {
                                    auto& absinfo = tablet.libevdev_get_abs_info(ABS_WHEEL);
                                    tablet.axes.slider = absinfo.absinfo_normalize() * 2 - 1; // Normalize slider.
                                }
                            }
                                fp64 adjust_tilt(abs_info_t const& absinfo)
                                {
                                    auto value = absinfo.absinfo_normalize();
                                    auto WACOM_MAX_DEGREES = 64;
                                    // If resolution is nonzero, it's in units/radian. But require a min/max less/greater than zero so we can assume 0 is the center.
                                    if (absinfo.resolution != 0 && absinfo.maximum > 0 && absinfo.minimum < 0)
                                    {
                                        value = netxs::rad2deg((fp64)absinfo.value / absinfo.resolution);
                                    }
                                    else // Wacom supports physical [-64, 64] degrees, so map to that by default. If other tablets have a different physical range or nonzero physical offsets, they need extra treatment here.
                                    {
                                        value = (value * 2) - 1; // Map to the (-1, 1) range.
                                        value *= WACOM_MAX_DEGREES;
                                    }
                                    return value;
                                }
                            void tablet_update_tilt()
                            {
                                if (!tablet.libevdev_has_event_code<EV_ABS>(ABS_TILT_X)
                                 || !tablet.libevdev_has_event_code<EV_ABS>(ABS_TILT_Y))
                                {
                                    return;
                                }
                                // Mouse rotation resets tilt to 0 so always fetch both axes if either has changed.
                                if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_TILT_X]
                                 || tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_TILT_Y])
                                {
                                    auto& absinfo_x = tablet.libevdev_get_abs_info(ABS_TILT_X);
                                    auto& absinfo_y = tablet.libevdev_get_abs_info(ABS_TILT_Y);
                                    tablet.axes.tilt.x = adjust_tilt(absinfo_x);
                                    tablet.axes.tilt.y = adjust_tilt(absinfo_y);
                                    if (tablet.dev_left_handed.enabled)
                                    {
                                        tablet.axes.tilt = -tablet.axes.tilt;
                                    }
                                }
                            }
                                fp64 normalize_wheel(si32 value)
                                {
                                    return value * tablet.scroll.wheel_click_angle.x;
                                }
                            void tablet_update_wheel()
                            {
                                if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_REL_WHEEL]) // Tablet->axes.wheel_discrete is already set.
                                {
                                    tablet.axes.wheel = normalize_wheel(tablet.axes.wheel_discrete);
                                }
                                else
                                {
                                    tablet.axes.wheel = 0;
                                    tablet.axes.wheel_discrete = 0;
                                }
                            }
                                    fp64 convert_to_degrees(abs_info_t const& absinfo, fp64 offset)
                                    {
                                        auto value = (absinfo.value - absinfo.minimum) / absinfo.absinfo_range(); // Range is [0, 360[, i.e. range + 1.
                                        return std::fmod(value * 360.0 + offset, 360.0);
                                    }
                                void tablet_update_artpen_rotation()
                                {
                                    if (tablet.libevdev_has_event_code<EV_ABS>(ABS_Z)
                                        && tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z])
                                    {
                                        auto& absinfo = tablet.libevdev_get_abs_info(ABS_Z);
                                        tablet.axes.rotation = convert_to_degrees(absinfo, 90); // Artpen has 0 with buttons pointing east.
                                    }
                                }
                                    void convert_tilt_to_rotation()
                                    {
                                        // Wacom Intuos 4, 5, Pro mouse calculates rotation from the x/y tilt values. The device has a 175 degree CCW hardware offset but since we use std::atan2() the effective offset is just 5 degrees.
                                        auto angle = fp64{};
                                        auto x = tablet.axes.tilt.x;
                                        auto y = tablet.axes.tilt.y;
                                        if (x || y) angle = netxs::rad2deg(std::atan2(-x, y)); // std::atan2() is CCW, we want CW -> negate x.
                                        const auto offset = 5;
                                        angle = std::fmod(360 + angle - offset, 360);
                                        tablet.axes.rotation = angle;
                                        tablet.changed_axes_bits.set(LIBINPUT_TABLET_TOOL_AXIS_ROTATION_Z);
                                    }
                                void tablet_update_mouse_rotation()
                                {
                                    if (tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_TILT_X]
                                     || tablet.changed_axes_bits[LIBINPUT_TABLET_TOOL_AXIS_TILT_Y])
                                    {
                                        convert_tilt_to_rotation();
                                    }
                                }
                            void tablet_update_rotation()
                            {
                                // We must check ROTATION_Z after TILT_X/Y so that the tilt axes are already normalized and set if we have the mouse/lens tool.
                                if (tablet.current_tool.type == LIBINPUT_TABLET_TOOL_TYPE_MOUSE
                                 || tablet.current_tool.type == LIBINPUT_TABLET_TOOL_TYPE_LENS)
                                {
                                    tablet_update_mouse_rotation();
                                    tablet.changed_axes_bits.reset(LIBINPUT_TABLET_TOOL_AXIS_TILT_X);
                                    tablet.changed_axes_bits.reset(LIBINPUT_TABLET_TOOL_AXIS_TILT_Y);
                                    tablet.axes.tilt.x = 0;
                                    tablet.axes.tilt.y = 0;
                                    // Tilt is already converted to left-handed, so mouse rotation is converted to left-handed automatically.
                                }
                                else
                                {
                                    tablet_update_artpen_rotation();
                                    if (tablet.dev_left_handed.enabled)
                                    {
                                        auto r = tablet.axes.rotation;
                                        tablet.axes.rotation = std::fmod(180 + r, 360);
                                    }
                                }
                            }
                            void tablet_history_reset()
                            {
                                tablet.history.count = 0;
                            }
                                ui64 tablet_history_size()
                                {
                                    return tablet.history.size;
                                }
                            void tablet_history_push(tablet_axes const& axes)
                            {
                                a