// -*- mode: c++; indent-tabs-mode: nil -*-
//! @file XmlRpcHandler.qm @brief XML-RPC handler class definition for the HttpServer module

/*  XmlRpcHandler.qm Copyright (C) 2012 - 2022 Qore Technologies, s.r.o.

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

// minimum qore version

// requires the binary xml module

// need mime definitions

// need the Util module

// need HttpServerUtil definitions

// do not use $ for vars


/*  Version History
    * 2012-05-31 v1.0: David Nichols <david@qore.org>: updated to a user module
    * 2013-09-24 v1.1: David Nichols <david@qore.org>: added optional log closure to constructor
*/

/** @mainpage XmlRpcHandler Module

    @section xmlrpchandlerintro Introduction to the XmlRpcHandler Module

    This module implements server-side support for the <a href="http://en.wikipedia.org/wiki/XML-RPC">XML-RPC protocol</a> for serialization/deserialization of message data.

    This module provides the @ref XmlRpcHandler::XmlRpcHandler "XmlRpcHandler" class which can be used to provide an RPC handler for the HttpServer class provided by the HttpServer module.

    @section example Example Usage
    @code
%requires HttpServer
%requires XmlRpcHandler
%requires Mime

const ApiMethods = (
     ("name": "^sys\\.shutdown\$",
      "text": "sys.shutdown",
      "function": sub () { background http.stop();  return "OK"; },
      "help": "shuts down this server",
      "logopt": 0,
      ),
);

# a logging closure
code log = sub (string fmt) {printf("%y: %s\n", now_us(), vsprintf(fmt, argv));};

# our bind address
const Bind = 8888;

XmlRpcHandler xmlRpcHandler(new AbstractAuthenticator(), ApiMethods);
our HttpServer http(log, log);
http.addListener(Bind);
http.setHandler("xmlrpc", "", MimeTypeXmlRpc, xmlRpcHandler);
http.setDefaultHandler("xmlrpc", xmlRpcHandler);
log("now listening on %s\n", Bind);
http.waitStop();
    @endcode
*/

//! the %XmlRpcHandler namespace holds all public definitions in the %XmlRpcHandler module
namespace XmlRpcHandler {
    //! XML-RPC Handler implementation; to be registered as a request handler in the HttpServer class
    /**
     */
class XmlRpcHandler : public AbstractHttpRequestHandler  {

public:
        //! version of the implementation
        const Version = "1.0";

        //! internal methods (for introspection)
        const InternalMethods = ...;


        //! error code for invalid XML-RPC
        const XMLRPC_INVALID_XML = 2001;

        //! @cond nodoc
protected:
            list methods = ();
            hash mi = hash();
            int loglevel;

            // if True then verbose exception info will be logged
            bool debug;

            // prefix to add to derived methods with GET requests if no "." characters are in the path
            __7_ string get_prefix;

            // a closure/call reference to get the log message and/or process arguments in incoming requests
            __7_ code getLogMessage;

            // a closure/call reference for logging (when set this is used instead of the HTTP server's logfunc for logging)
            __7_ code logc;

public:
        //! @endcond

        //! creates the handler with the given method list
        /** @param auth an authentication object (use new AbstractAuthenticator() for no authentication)
            @param methods a list of hashes with the following keys:
            - \c name: a regular expression to use for matching the method name
            - \c function: a string (giving a function name to call), a call reference, or a closure to call with the deserialized arguments to the method; the return value will be serialized to XML-RPC and sent back to the caller
            - \c text: the human-readable name of the method
            - \c help: help text for the method
            - \c logopt: (optional - by convention) log options which can be used by a custom logger (see the getLogMessage parameter)
            @param n_getLogMessage an optional closure or call reference to be called when an incoming request is received; if this is set then it will
be called with the following arguments: a context hash (see HttpServer::AbstractHttpRequestHandler::handleRequest() for a description of the context h
ash), the method definition as passed in the methods argument to this constructor, and a reference to the arguments in this call
            @param dbg this parameter is set to @ref Qore::True "True", then additional information will be logged when errors occur
            @param n_get_prefix prefix to add to derived methods with GET requests if no "." characters are in the path
            @param log an optional closure or call reference to be called when logging

            @throw XML-RPC-CONSTRUCTOR-ERROR missing \c "name", \c "function", or \"text" key in method hash, \c "function" key not assigned to a callable value
         */
        constructor(HttpServer::AbstractAuthenticator auth, list methods, __7_ code n_getLogMessage, bool dbg = False, __7_ string n_get_prefix, __7_ code log) ;


        //! adds a method to the handler dynamically
        /** @param name a regular expression to use for matching the method name
            @param func a string (giving a function name to call), a call reference, or a closure to call with the deserialized arguments to the method; the return value will be serialized to XML-RPC and sent back to the caller
            @param text the human-readable name of the method
            @param help help text for the method
            @param logopt log options which can be used by a custom logger (see the getLogMessage parameter)
            @param cmark any value that will be added in the method hash under the \c "cmark" key
          */
        addMethod(string name, code func, string text, string help, int logopt, auto cmark);


        //! turns on or off debugging; when debugging is enabled more verbose error messages are reported
        setDebug(bool dbg = True);


        //! returns the current status of the debug flag
        bool getDebug();


        //! @cond nodoc
        // don't reimplement this method; fix/enhance it in the module
protected:
        final  addMethodInternal(hash h);
public:


protected:
         hash help();
public:


protected:
         list listMethods();
public:


protected:
         log(hash cx, string str);
public:


        // don't reimplement this method; fix/enhance it in the module
protected:
        final  hash<auto> callMethod(hash cx, auto params);
public:


        // method called by HttpServer to handle an XML-RPC request
        // don't reimplement this method; fix/enhance it in the module
        final hash handleRequest(hash cx, hash hdr, __7_ data body);

        //! @endcond
    };
};
