// -*- mode: c++; indent-tabs-mode: nil -*-
//! @file SoapHandler.qm SOAP handler module providing the SoapHandler class to be registered as a handler with the Qore HttpServer module

/*  SoapHandler.qm Copyright (C) 2012 - 2022 Qore Technologies, s.r.o.

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

// make sure we have the required qore version

// requires XML functionality

// requires the WSDL module

// need mime definitions

// need definitions from the HttpServerUtil

// need Util functions

// do not use $ for vars

// require type declarations everywhere

// do not ignore errors in call arguments


/** @mainpage SoapHandler Module

    @tableofcontents

    @section soaphandlerintro Introduction to the SoapHandler Module

    This module implements server-side support for the <a href="http://en.wikipedia.org/wiki/SOAP">SOAP protocol</a> for serialization/deserialization of message data.

    This module provides the @ref SoapHandler::SoapHandler "SoapHandler" class which can be used to provide an RPC handler for the HttpServer class provided by the HttpServer module.

    @section soaphandler_relnotes SoapHandler Release History

    @subsection soaphandler_0_3_2 SoapHandler 0.3.2
    - fixed bugs where invalid XML messages were returned with some SOAP fault responses
      (<a href="https://github.com/qorelanguage/qore/issues/4441">issue 4441</a>)
    - fixed errors propagating XSD types in WSDLs to data providers
      (<a href="https://github.com/qorelanguage/qore/issues/4436">issue 4436</a>)

    @subsection soaphandler_0_3_1 SoapHandler 0.3.1
    - fixed a type error processing messages on listeners with no "hostname" socket info value
      (<a href="https://github.com/qorelanguage/qore/issues/4307">issue 4307</a>)

    @subsection soaphandler_0_3 SoapHandler 0.3
    - do not run SOAP operations in the SoapHandler lock
      (<a href="https://github.com/qorelanguage/qore/issues/4178">issue 4178</a>)

    @subsection soaphandler_0_2_7 SoapHandler 0.2.7
    - fixed a bug where HTTP request headers were not reported correctly in log callbacks
      (<a href="https://github.com/qorelanguage/qore/issues/2874">issue 2874</a>)
    - fixed a bug where duplicate \c soapAction values would cause an invalid error
      to be thrown even if there were unique paths for each operation
      (<a href="https://github.com/qorelanguage/qore/issues/2871">issue 2871</a>)

    @subsection soaphandler_0_2_6 SoapHandler 0.2.6
    - <b>Compatibility Warnings</b>
      - Because the SoapHandler module now supports HTTP bindings, it will only return
        WSDLs from GET requests if the URI path includes <tt>?wsdl</tt> at the end of the string
    - reimplemented operation to support multiple SOAP bindings, SOAP operations can be assigned to multiple bindings
    - HTTP binding support
    - implemented supoprt for handling SOAP faults based on the exception \c err string (must correspond to the fault name)
      (<a href="https://github.com/qorelanguage/qore/issues/2804">issue 2804</a>)

    @subsection soaphandler_0_2_5 SoapHandler 0.2.5
    - added support for matching requests with soap action values
    - added the err_func argument so SOAP response serialization errors can be logged at the source
    - added support for debugging (verbose exception info/logging)
    - fixes to service creation and GET WSDL handling
    - Content-type in exceptional cases follows Soap version
    - added support for logging content of SOAP messages

    @subsection soaphandler_0_2_4 SoapHandler 0.2.4
    - updated to a user module

    @subsection soaphandler_0_2_3 SoapHandler 0.2.3
    - return correct soap fault structure according to request

    @subsection soaphandler_0_2_2 SoapHandler 0.2.2
    - soap 1.2 improvements

    @subsection soaphandler_0_2_1 SoapHandler 0.2.1
    - matched to new WSDL and HttpServer implementation

    @subsection soaphandler_0_2_0 SoapHandler 0.2.0
    - better WSDL support

    @subsection soaphandler_0_1_0 SoapHandler 0.1.0
    - initial WSDL-based SOAP support (still incomplete)
*/

//! main SoapHandler namespace
namespace SoapHandler {
    //! SoapHandler implementation; to be registered as a request handler in the HttpServer class
class SoapHandler : public AbstractHttpRequestHandler {

public:
        //! version of the SoapHandler implementation
        const Version = "0.2.6";

        //! @cond nodoc
protected:
            // method mappings: method name -> method hash
            hash<auto> methods;

            // reverse mappings for methods: unique_id -> method -> True
            hash<string, hash<string, bool>> rmethods;

            // URI path -> method name -> method mappings
            hash<string, hash> uri_methods;

            // reverse URI path method mappings: unique_id -> URI path -> method name -> true
            hash<string, hash<string, hash<string, bool>>> ruri_methods;

            // service name -> WebService objects
            hash<string, WebService> wsh;

            // reverse mapping to remove wsh mappings; unique_id -> path -> True
            hash<string, hash<string, bool>> rwsh;

            // operation name -> WebService object
            hash<string, WebService> owsh;

            // reverse operation mappings; unique_id -> op name -> true
            hash<string, hash<string, bool>> rowsh;

            // lookup URI path -> WebService object
            hash<string, WebService> wsph;

            // reverse URI path lookups: unique_id -> URI path -> true
            hash<string, hash<string, bool>> rwsph;

            // URI path -> SOAP Action -> method mappings
            hash<string, hash<string, hash>> uri_sam;

            // reverse URI path -> SOAP Action mappings: unique_id -> URI path -> SOAPAction -> true
            hash<string, hash<string, hash<string, bool>>> ruri_sam;

            // soapaction map: action -> binding -> operation
            hash<string, hash> sam;

            // reverse mapping: unique_id -> soapaction -> True
            hash<string, hash<string, bool>> rsam;

            int loglevel;

            // if True then verbose exception info will be logged
            bool debug;

            // a closure/call reference to get the log message and/or process arguments in incoming requests
            __7_ code getLogMessage;

            // for atomicity when adding / removing methods
            RWLock rwl();

            // path TreeMap for each HTTP method
            hash<string, TreeMap> mapMethodPathToOperation;

            // soapaction -> method -> binding

public:
        //! @endcond

        //! creates the handler with the given method list
/** @param auth an authentication object (use new AbstractAuthenticator() for no authentication)
            @param n_getLogMessage an optional closure or call reference to be called when an incoming request is
            received; if this is set then it will be called with the following arguments: a context hash (see
            HttpServer::AbstractHttpRequestHandler::handleRequest() for a description of the context hash), the method
            definition as passed in the methods argument to this constructor, and a reference to the arguments in this
            call
            @param dbg this parameter is set to @ref Qore::True "True", then additional information will be logged
            when errors occur
         */
        constructor(AbstractAuthenticator auth, __7_ code n_getLogMessage, bool dbg = False)
 ;


        //! adds a method to the handler dynamically
/** @param ws the WebService object for the method
            @param op the WSOperation object for the web service operation the method corresponds to
            @param func a call reference, a closure, or a string function name to call with the deserialized
            arguments to the method; the return value will be serialized to SOAP according to the WSDL and sent back
            to the caller
            @param help optional help text for the method
            @param logopt log options which can be used by a custom logger (see the \a getLogMessage parameter in the
            constructor)
            @param cmark an optional "context marker" for the method
            @param path an optional path for the method (assumed to be the name of the service)
            @param err_func a call reference, a closure, or a string function name to call with error information if
            an exception is thrown with SOAP data
            @param altpath an alternate path for the service / WSDL
            @param binding SOAP binding name, leave empty to use the first assigned binding
            @param unique_id the unique ID for the backend service implementation for all operations on this WSDL;
            necessary for any removeService() call
         */
addMethod(WebService ws, WSOperation op, auto func, __7_ string help, __7_ int logopt, auto cmark, __7_ string path, auto err_func, *string altpath, *string binding, *string unique_id);


        //! turns on or off debugging; when debugging is enabled more verbose error messages are reported
        setDebug(bool dbg = True);


        //! returns the current status of the debug flag
        bool getDebug();


        //! call to remove the given service
/** @param unique_id must correspond to the \a unique_id in any addMethod() calls
        */
        removeService(string unique_id);


        //! @cond nodoc
        // don't reimplement this method; fix/enhance it in the module
protected:
        final  addMethodInternal(WebService ws, hash<auto> method);
public:


protected:
         __7_ hash<auto> help(hash<auto> cx);
public:


        // don't reimplement this method; fix/enhance it in the module
protected:
        final  log(hash<auto> cx, string str);
public:


/** Method is to be overriden to implement custom logging HTTP messages handled by this object.
            @par cx context passed from @ref HttpServer::HttpServer "HttpServer"
            @par msg information about message to be logged. There are keys 'reason' with value 'request', 'response'
            or 'error', 'header' (HTTP header, in casew ) and 'body' containing XML data in readable form, i.e.
            uncompressed and in case of multipart message only the related part is passed. The method is executed
            before the message is sent or after it has been received.  This method must not throw exceptions.

            @par Example:
            @code{.py}
private nothing msglog(hash<auto> cx, hash<auto> msg) {
    log(sprintf("%s:\n%s\n", msg.reason, msg.body));
}
            @endcode

            @since %SoapHandler 0.2.5
        */
protected:
         nothing msglog(hash<auto> cx, hash<auto> msg);
public:


        // don't reimplement this method; fix/enhance it in the module
protected:
 hash<auto> makeSoapFaultResponse(hash<auto> cx, string errLog, bool soap12, string err, string desc, *bool fmt);
public:


        // don't reimplement this method; fix/enhance it in the module
protected:
        final  __7_ hash<auto> callOperation(hash<auto> cx, auto args, hash<auto> method, bool reqsoap12);
public:


        //! Dispatches the SOAP operation call and returns the result for validation and serialization
protected:
         auto dispatchSoapOperationCall(hash<auto> method, hash<auto> cx, auto args);
public:


protected:
         __7_ hash<auto> matchMethod(TreeMap tm, string path, reference unmatched);
public:


        // method called by HttpServer
        // don't reimplement this method; fix/enhance it in the module
        final hash<auto> handleRequest(hash<auto> cx, hash<auto> hdr, __7_ data body);


protected:
         __7_ WebService tryMatch(string path);
public:

        //! @endcond
    };
};
