import { EditorState, Extension, StateCommand, Text } from '@codemirror/state';
import { Command, EditorView } from '@codemirror/view';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { Signal } from '@lumino/signaling';
import { IEditorExtensionRegistry, IEditorLanguageRegistry, IExtensionsHandler } from './token';
/**
 * CodeMirror editor.
 */
export declare class CodeMirrorEditor implements CodeEditor.IEditor {
    /**
     * Construct a CodeMirror editor.
     */
    constructor(options: CodeMirrorEditor.IOptions);
    /**
     * A signal emitted when either the top or bottom edge is requested.
     */
    readonly edgeRequested: Signal<this, CodeEditor.EdgeLocation>;
    /**
     * The DOM node that hosts the editor.
     */
    readonly host: HTMLElement;
    /**
     * The uuid of this editor;
     */
    get uuid(): string;
    set uuid(value: string);
    /**
     * Get the codemirror editor wrapped by the editor.
     */
    get editor(): EditorView;
    /**
     * Get the codemirror doc wrapped by the widget.
     */
    get doc(): Text;
    /**
     * Get the number of lines in the editor.
     */
    get lineCount(): number;
    /**
     * Returns a model for this editor.
     */
    get model(): CodeEditor.IModel;
    /**
     * The height of a line in the editor in pixels.
     */
    get lineHeight(): number;
    /**
     * The widget of a character in the editor in pixels.
     */
    get charWidth(): number;
    /**
     * Tests whether the editor is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the widget.
     */
    dispose(): void;
    /**
     * Get a config option for the editor.
     */
    getOption(option: string): unknown;
    /**
     * Whether the option exists or not.
     */
    hasOption(option: string): boolean;
    /**
     * Set a config option for the editor.
     */
    setOption(option: string, value: unknown): void;
    /**
     * Set config options for the editor.
     *
     * This method is preferred when setting several options. The
     * options are set within an operation, which only performs
     * the costly update at the end, and not after every option
     * is set.
     */
    setOptions(options: Record<string, any>): void;
    /**
     * Inject an extension into the editor
     *
     * @alpha
     * @experimental
     * @param ext CodeMirror 6 extension
     */
    injectExtension(ext: Extension): void;
    /**
     * Returns the content for the given line number.
     */
    getLine(line: number): string | undefined;
    /**
     * Find an offset for the given position.
     */
    getOffsetAt(position: CodeEditor.IPosition): number;
    /**
     * Find a position for the given offset.
     */
    getPositionAt(offset: number): CodeEditor.IPosition;
    /**
     * Undo one edit (if any undo events are stored).
     */
    undo(): void;
    /**
     * Redo one undone edit.
     */
    redo(): void;
    /**
     * Clear the undo history.
     */
    clearHistory(): void;
    /**
     * Brings browser focus to this editor text.
     */
    focus(): void;
    /**
     * Test whether the editor has keyboard focus.
     */
    hasFocus(): boolean;
    /**
     * Explicitly blur the editor.
     */
    blur(): void;
    get state(): EditorState;
    firstLine(): number;
    lastLine(): number;
    cursorCoords(where: boolean, mode?: 'window' | 'page' | 'local'): {
        left: number;
        top: number;
        bottom: number;
    };
    getRange(from: {
        line: number;
        ch: number;
    }, to: {
        line: number;
        ch: number;
    }, separator?: string): string;
    /**
     * Reveal the given position in the editor.
     */
    revealPosition(position: CodeEditor.IPosition): void;
    /**
     * Reveal the given selection in the editor.
     */
    revealSelection(selection: CodeEditor.IRange): void;
    /**
     * Get the window coordinates given a cursor position.
     */
    getCoordinateForPosition(position: CodeEditor.IPosition): CodeEditor.ICoordinate;
    /**
     * Get the cursor position given window coordinates.
     *
     * @param coordinate - The desired coordinate.
     *
     * @returns The position of the coordinates, or null if not
     *   contained in the editor.
     */
    getPositionForCoordinate(coordinate: CodeEditor.ICoordinate): CodeEditor.IPosition | null;
    /**
     * Returns the primary position of the cursor, never `null`.
     */
    getCursorPosition(): CodeEditor.IPosition;
    /**
     * Set the primary position of the cursor.
     *
     * #### Notes
     * This will remove any secondary cursors.
     *
     * @deprecated options bias and origin are not used
     */
    setCursorPosition(position: CodeEditor.IPosition, options?: {
        bias?: number;
        origin?: string;
        scroll?: boolean;
    }): void;
    /**
     * Returns the primary selection, never `null`.
     */
    getSelection(): CodeEditor.ITextSelection;
    /**
     * Set the primary selection. This will remove any secondary cursors.
     */
    setSelection(selection: CodeEditor.IRange): void;
    /**
     * Gets the selections for all the cursors, never `null` or empty.
     */
    getSelections(): CodeEditor.ITextSelection[];
    /**
     * Sets the selections for all the cursors, should not be empty.
     * Cursors will be removed or added, as necessary.
     * Passing an empty array resets a cursor position to the start of a document.
     */
    setSelections(selections: CodeEditor.IRange[]): void;
    /**
     * Replaces the current selection with the given text.
     *
     * Behaviour for multiple selections is undefined.
     *
     * @param text The text to be inserted.
     */
    replaceSelection(text: string): void;
    /**
     * Get a list of tokens for the current editor text content.
     */
    getTokens(): CodeEditor.IToken[];
    /**
     * Get the token at a given editor position.
     */
    getTokenAt(offset: number): CodeEditor.IToken;
    /**
     * Get the token a the cursor position.
     */
    getTokenAtCursor(): CodeEditor.IToken;
    /**
     * Insert a new indented line at the current cursor position.
     */
    newIndentedLine(): void;
    /**
     * Execute a codemirror command on the editor.
     *
     * @param command - The name of the command to execute.
     */
    execCommand(command: Command | StateCommand): void;
    protected onConfigChanged(configurator: IExtensionsHandler, changes: Record<string, any>): void;
    /**
     * Handle keydown events from the editor.
     */
    protected onKeydown(event: KeyboardEvent): boolean;
    /**
     * Handles a mime type change.
     */
    private _onMimeTypeChanged;
    /**
     * Handles a cursor activity event.
     */
    private _onCursorActivity;
    /**
     * Converts a code mirror selection to an editor selection.
     */
    private _toSelection;
    /**
     * Convert a code mirror position to an editor position.
     */
    private _toPosition;
    /**
     * Convert an editor position to a code mirror position.
     */
    private _toCodeMirrorPosition;
    /**
     * Handles document changes.
     */
    private _onDocChanged;
    /**
     * Handle the DOM events for the editor.
     *
     * @param event - The DOM event sent to the editor.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the editor's DOM node. It should
     * not be called directly by user code.
     */
    handleEvent(event: Event): void;
    /**
     * Handle `focus` events for the editor.
     */
    private _evtFocus;
    /**
     * Handle `blur` events for the editor.
     */
    private _evtBlur;
    private _configurator;
    private _editor;
    private _isDisposed;
    private _language;
    private _languages;
    private _model;
    private _uuid;
}
/**
 * The namespace for `CodeMirrorEditor` statics.
 */
export declare namespace CodeMirrorEditor {
    /**
     * The options used to initialize a code mirror editor.
     */
    interface IOptions extends CodeEditor.IOptions {
        /**
         * CodeMirror extensions registry
         */
        extensionsRegistry?: IEditorExtensionRegistry;
        /**
         * CodeMirror languages registry
         */
        languages?: IEditorLanguageRegistry;
    }
}
