/* manage prefixes and mapping them to/from lat/lng.
 * 
 * we actually use two different lists. a short one in memory for LL->prefix and a much longer one
 * based on AC1D's cty file for prefix->LL. TODO: use cty for both but needs some clever sort for speed.
 */

#include "HamClock.h"


/* prefixes are cached with fast radix access
 */

static char cty_page[] = "/cty/cty_wt_mod-ll-dxcc.txt";         // web page to download
static char cty_fn[] = "cty-ll-dxcc.txt";                       // local cache copy name

typedef struct {
    char call[MAX_SPOTCALL_LEN];                // mostly prefixes, a few calls; sorted ala strcmp
    float lat_d, lng_d;                         // +N +E degrees
    int call_len;                               // handy strlen(call)
    int dxcc;                                   // DXCC number
} CtyLoc;
static CtyLoc *cty_list;                        // malloced list
static int n_cty, n_malloc;                     // n entries used, n malloced
#define _N_RADIX ('Z' - '0' + 1)                // radix range, 
static int cty_radix[_N_RADIX];                 // table of cty_list index from first character
static char prev_radix;                         // used to detect change in radis index
static time_t next_refresh;                     // time of next download
#define MAX_CTY_AGE     (1*24*3600)             // normally update city file this often, secs
#define MIN_CTY_SIZ     800000                  // min believable file size
#define RETRY_DT        60                      // retry interval if trouble, secs
#define MAX_DIST        12                      // max dist from target, degrees


/* table of common prefixes and their rough center location.
 */

#define SMALL_PREF_LEN 4
typedef struct {
    char pref[SMALL_PREF_LEN];                  // prefix, left justified, \0 padded but none if full
    int16_t lat, lng;                           // rough center, degs*100 +E +N
} SmallPrefix;

static const SmallPrefix small_prefs[] = {
     {{'1','A','\0','\0'},        4154,   1230},
     {{'1','S','\0','\0'},         839,  11129},
     {{'3','A','\0','\0'},        4345,    725},
     {{'3','B','6','\0'},        -1021,   5636},
     {{'3','B','8','\0'},        -2010,   5730},
     {{'3','B','9','\0'},        -1941,   6328},
     {{'3','C','\0','\0'},         345,    847},
     {{'3','C','0','\0'},          128,    540},
     {{'3','D','2','\0'},        -1807,  17825},
     {{'3','D','2','C'},         -2144,  17438},
     {{'3','D','2','X'},         -1248,  17701},
     {{'3','D','A','\0'},        -2718,   3160},
     {{'3','V','\0','\0'},        3647,   1011},
     {{'3','W','\0','\0'},        1048,  10642},
     {{'3','X','\0','\0'},         931,  -1343},
     {{'3','Y','\0','\0'},       -5418,    322},
     {{'3','Y','0','\0'},        -6840,  -9040},
     {{'4','J','\0','\0'},        4036,   4989},
     {{'4','L','\0','\0'},        4143,   4449},
     {{'4','O','\0','\0'},        4244,   1926},
     {{'4','S','\0','\0'},         656,   7951},
     {{'4','U','1','I'},          4612,    609},
     {{'4','U','1','U'},          4043,  -7401},
     {{'4','W','\0','\0'},        -855,  12557},
     {{'4','X','\0','\0'},        3146,   3514},
     {{'5','A','\0','\0'},        3000,   1400},
     {{'5','A','\0','\0'},        3000,   2100},
     {{'5','A','\0','\0'},        2300,   2100},
     {{'5','A','\0','\0'},        2600,   1200},
     {{'5','B','\0','\0'},        3510,   3322},
     {{'5','H','\0','\0'},        -400,   3300},
     {{'5','H','\0','\0'},        -700,   3800},
     {{'5','N','\0','\0'},         627,    324},
     {{'5','R','\0','\0'},       -1700,   4600},
     {{'5','R','\0','\0'},       -2200,   4500},
     {{'5','T','\0','\0'},        2400,   -800},
     {{'5','T','\0','\0'},        1800,   -800},
     {{'5','T','\0','\0'},        1800,  -1500},
     {{'5','U','\0','\0'},        1900,   1100},
     {{'5','U','\0','\0'},        1500,    900},
     {{'5','U','\0','\0'},        1400,    200},
     {{'5','V','\0','\0'},         608,    112},
     {{'5','W','\0','\0'},       -1350, -17144},
     {{'5','X','\0','\0'},          19,   3125},
     {{'5','Z','\0','\0'},        -117,   3649},
     {{'6','W','\0','\0'},        1440,  -1726},
     {{'6','Y','\0','\0'},        1800,  -7648},
     {{'7','O','\0','\0'},        1500,   4700},
     {{'7','P','\0','\0'},       -2919,   2729},
     {{'7','Q','\0','\0'},       -1359,   3344},
     {{'7','X','\0','\0'},        3400,    300},
     {{'7','X','\0','\0'},        2300,    600},
     {{'7','X','\0','\0'},        2700,    400},
     {{'8','P','\0','\0'},        1306,  -5937},
     {{'8','Q','\0','\0'},         373,   7300},
     {{'8','R','\0','\0'},         648,  -5810},
     {{'9','A','\0','\0'},        4548,   1558},
     {{'9','G','\0','\0'},         533,    -13},
     {{'9','H','\0','\0'},        3554,   1431},
     {{'9','J','\0','\0'},       -1400,   2600},
     {{'9','J','\0','\0'},       -1000,   3100},
     {{'9','K','\0','\0'},        2920,   4759},
     {{'9','L','\0','\0'},         830,  -1315},
     {{'9','M','2','\0'},          310,  10142},
     {{'9','M','6','\0'},          601,  11607},
     {{'9','M','8','\0'},          200,  11300},
     {{'9','N','\0','\0'},        2743,   8519},
     {{'9','Q','\0','\0'},         200,   2700},
     {{'9','Q','\0','\0'},           0,   1900},
     {{'9','Q','\0','\0'},        -700,   2500},
     {{'9','U','\0','\0'},        -323,   2922},
     {{'9','V','\0','\0'},         117,  10351},
     {{'9','X','\0','\0'},        -157,   3004},
     {{'9','Y','\0','\0'},        1039,  -6131},
     {{'A','2','\0','\0'},       -2200,   2400},
     {{'A','3','\0','\0'},       -2110, -17515},
     {{'A','4','\0','\0'},        2337,   5835},
     {{'A','5','\0','\0'},        2728,   8939},
     {{'A','6','\0','\0'},        2428,   5422},
     {{'A','7','\0','\0'},        2517,   5132},
     {{'A','9','\0','\0'},        2613,   5035},
     {{'A','P','\0','\0'},        3100,   7100},
     {{'A','P','\0','\0'},        2700,   6700},
     {{'B','S','7','\0'},         1518,  11776},
     {{'B','V','\0','\0'},        2503,  12130},
     {{'B','V','9','P'},          2071,  11669},
     {{'B','Y','0','\0'},         2400,   8400},
     {{'B','Y','0','\0'},         3900,   8300},
     {{'B','Y','1','\0'},         3954,  11625},
     {{'B','Y','2','\0'},         4330,  12700},
     {{'B','Y','3','\0'},         3900,  10000},
     {{'B','Y','3','\0'},         4100,  11200},
     {{'B','Y','4','\0'},         3114,  12130},
     {{'B','Y','5','\0'},         2839,  11558},
     {{'B','Y','6','\0'},         3440,  11338},
     {{'B','Y','7','\0'},         2809,  11300},
     {{'B','Y','8','\0'},         3039,  10404},
     {{'B','Y','9','\0'},         3600,   9400},
     {{'C','2','\0','\0'},         -30,  16655},
     {{'C','3','\0','\0'},        4230,    131},
     {{'C','5','\0','\0'},        1328,  -1639},
     {{'C','6','\0','\0'},        2505,  -7720},
     {{'C','9','\0','\0'},       -1400,   3800},
     {{'C','9','\0','\0'},       -1500,   3300},
     {{'C','9','\0','\0'},       -2200,   3400},
     {{'C','E','0','\0'},        -2706, -10921},
     {{'C','E','0','X'},         -2504,  -8025},
     {{'C','E','0','Z'},         -3326,  -7841},
     {{'C','E','1','\0'},        -2339,  -7023},
     {{'C','E','2','\0'},        -3304,  -7140},
     {{'C','E','3','\0'},        -3327,  -7040},
     {{'C','E','4','\0'},        -3636,  -7209},
     {{'C','E','5','\0'},        -3747,  -7245},
     {{'C','E','6','\0'},        -3845,  -7240},
     {{'C','E','7','\0'},        -4128,  -7300},
     {{'C','E','8','\0'},        -5310,  -7056},
     {{'C','E','9','\0'},        -8500,  -6700},
     {{'C','N','\0','\0'},        3300,    500},
     {{'C','N','\0','\0'},        2900,  -1000},
     {{'C','O','\0','\0'},        2308,  -8222},
     {{'C','P','1','\0'},        -1630,  -6809},
     {{'C','P','6','\0'},        -1748,  -6310},
     {{'C','T','\0','\0'},        3844,   -908},
     {{'C','T','3','\0'},         3240,  -1645},
     {{'C','U','\0','\0'},        3739,  -2534},
     {{'C','X','\0','\0'},       -3453,  -5611},
     {{'C','Y','0','\0'},         4402,  -6000},
     {{'C','Y','9','\0'},         4713,  -6008},
     {{'D','2','\0','\0'},       -1200,   1700},
     {{'D','4','\0','\0'},        1500,  -2351},
     {{'D','6','\0','\0'},       -1141,   4316},
     {{'D','L','\0','\0'},        5231,   1324},
     {{'D','U','\0','\0'},        1435,  12100},
     {{'E','3','\0','\0'},        1520,   3900},
     {{'E','4','\0','\0'},        3177,   3521},
     {{'E','5','\0','\0'},       -2123, -15977},
     {{'E','6','\0','\0'},       -1857, -17006},
     {{'E','7','\0','\0'},        4352,   1825},
     {{'E','A','\0','\0'},        4024,   -341},
     {{'E','A','6','\0'},         3934,    239},
     {{'E','A','8','\0'},         2806,  -1524},
     {{'E','A','9','\0'},         3519,   -258},
     {{'E','I','\0','\0'},        5320,   -615},
     {{'E','K','\0','\0'},        4011,   4434},
     {{'E','L','\0','\0'},         618,  -1047},
     {{'E','P','\0','\0'},        3500,   4800},
     {{'E','P','\0','\0'},        3000,   5700},
     {{'E','P','\0','\0'},        3500,   5700},
     {{'E','R','\0','\0'},        4700,   2850},
     {{'E','S','\0','\0'},        5922,   2448},
     {{'E','T','\0','\0'},        1200,   3900},
     {{'E','T','\0','\0'},         700,   4200},
     {{'E','T','\0','\0'},         700,   3600},
     {{'E','U','\0','\0'},        5354,   2734},
     {{'E','X','\0','\0'},        4254,   7436},
     {{'E','Y','\0','\0'},        3835,   6848},
     {{'E','Z','\0','\0'},        3757,   5823},
     {{'F','\0','\0','\0'},       4852,    220},
     {{'F','G','\0','\0'},        1610,  -6140},
     {{'F','H','\0','\0'},       -1247,   4517},
     {{'F','J','\0','\0'},        1791,  -6283},
     {{'F','K','\0','\0'},       -2216,  16627},
     {{'F','K','\0','\0'},       -1900,  15800},
     {{'F','M','\0','\0'},        1436,  -6105},
     {{'F','O','\0','\0'},       -1753, -14956},
     {{'F','O','A','\0'},        -2300, -15000},
     {{'F','O','C','\0'},         1010, -10850},
     {{'F','O','M','\0'},         -944, -13938},
     {{'F','P','\0','\0'},        4655,  -5610},
     {{'F','R','\0','\0'},       -2052,   5528},
     {{'F','S','\0','\0'},        1809,  -6300},
     {{'F','T','G','\0'},        -1131,   4718},
     {{'F','T','J','\0'},        -1821,   4136},
     {{'F','T','T','\0'},        -1557,   5421},
     {{'F','T','W','\0'},        -4630,   5200},
     {{'F','T','X','\0'},        -4912,   7000},
     {{'F','T','Z','\0'},        -3731,   7730},
     {{'F','W','\0','\0'},       -1311, -17643},
     {{'F','Y','\0','\0'},         455,  -5220},
     {{'G','\0','\0','\0'},       5130,     10},
     {{'G','D','\0','\0'},        5460,   -428},
     {{'G','I','\0','\0'},        5435,   -555},
     {{'G','J','\0','\0'},        4912,   -237},
     {{'G','M','\0','\0'},        5557,   -313},
     {{'G','M','S','\0'},         6000,   -313},
     {{'G','U','\0','\0'},        4927,   -231},
     {{'G','W','\0','\0'},        5129,   -313},
     {{'H','4','\0','\0'},        -926,  15957},
     {{'H','4','0','\0'},        -1068,  16606},
     {{'H','A','\0','\0'},        4730,   1905},
     {{'H','B','\0','\0'},        4657,    726},
     {{'H','B','0','\0'},         4790,    932},
     {{'H','C','\0','\0'},         -13,  -7830},
     {{'H','C','8','\0'},          -36,  -9018},
     {{'H','H','\0','\0'},        1832,  -7220},
     {{'H','I','\0','\0'},        1828,  -6954},
     {{'H','K','\0','\0'},           0,  -7200},
     {{'H','K','\0','\0'},         400,  -7000},
     {{'H','K','\0','\0'},         700,  -7500},
     {{'H','K','0','\0'},         1232,  -8142},
     {{'H','K','0','M'},           409,  -8138},
     {{'H','L','\0','\0'},        3733,  12658},
     {{'H','P','\0','\0'},         858,  -7931},
     {{'H','R','\0','\0'},        1406,  -8713},
     {{'H','S','\0','\0'},        1800,   9900},
     {{'H','S','\0','\0'},        1500,  10200},
     {{'H','S','\0','\0'},        1100,  10000},
     {{'H','V','\0','\0'},        4154,   1229},
     {{'H','Z','\0','\0'},        2800,   4000},
     {{'H','Z','\0','\0'},        2400,   4800},
     {{'H','Z','\0','\0'},        2000,   4200},
     {{'I','0','\0','\0'},        4154,   1229},
     {{'I','1','\0','\0'},        4424,    856},
     {{'I','2','\0','\0'},        4528,    911},
     {{'I','3','\0','\0'},        4526,   1220},
     {{'I','4','\0','\0'},        4430,   1120},
     {{'I','5','\0','\0'},        4347,   1115},
     {{'I','6','\0','\0'},        4240,   1343},
     {{'I','7','\0','\0'},        4107,   1652},
     {{'I','8','\0','\0'},        4050,   1415},
     {{'I','A','5','\0'},         4245,   1015},
     {{'I','C','2','\0'},         4549,    906},
     {{'I','C','8','\0'},         4033,   1413},
     {{'I','G','9','\0'},         3515,   1230},
     {{'I','H','9','\0'},         3645,   1200},
     {{'I','J','7','\0'},         4000,   1714},
     {{'I','L','3','\0'},         4526,   1220},
     {{'I','N','3','\0'},         4604,   1108},
     {{'I','P','1','\0'},         4418,    827},
     {{'I','S','\0','\0'},        3920,    900},
     {{'I','T','\0','\0'},        3729,   1400},
     {{'I','V','3','\0'},         4539,   1347},
     {{'J','2','\0','\0'},        1136,   4309},
     {{'J','3','\0','\0'},        1203,  -6145},
     {{'J','5','\0','\0'},        1151,  -1535},
     {{'J','6','\0','\0'},        1401,  -6100},
     {{'J','7','\0','\0'},        1518,  -6124},
     {{'J','8','\0','\0'},        1309,  -6114},
     {{'J','A','0','\0'},         3754,  13903},
     {{'J','A','1','\0'},         3542,  13946},
     {{'J','A','2','\0'},         3458,  13822},
     {{'J','A','3','\0'},         3500,  13544},
     {{'J','A','4','\0'},         3439,  13355},
     {{'J','A','5','\0'},         3420,  13403},
     {{'J','A','6','\0'},         3300,  13100},
     {{'J','A','7','\0'},         3942,  14109},
     {{'J','A','8','\0'},         4330,  14300},
     {{'J','A','9','\0'},         3640,  13713},
     {{'J','D','1','\0'},         2430,  15358},
     {{'J','D','1','O'},          2648,  14219},
     {{'J','T','\0','\0'},        4700,   9700},
     {{'J','T','\0','\0'},        4600,  10900},
     {{'J','W','\0','\0'},        7440,   1807},
     {{'J','X','\0','\0'},        7111,   -800},
     {{'J','Y','\0','\0'},        3157,   3556},
     {{'K','0','\0','\0',},       3836,  -9246},     // Missouri
     {{'K','0','\0','\0',},       3849,  -9838},     // Kansas
     {{'K','0','\0','\0',},       3900, -10555},     // Colorado
     {{'K','0','\0','\0',},       4154,  -9980},     // Nebraska
     {{'K','0','\0','\0',},       4208,  -9350},     // Iowa
     {{'K','0','\0','\0',},       4444, -10023},     // South Dakota
     {{'K','0','\0','\0',},       4628,  -9431},     // Minnesota
     {{'K','0','\0','\0',},       4745, -10047},     // North Dakota
     {{'K','1','\0','\0',},       4162,  -7273},     // Connecticut
     {{'K','1','\0','\0',},       4168,  -7156},     // Rhode Island
     {{'K','1','\0','\0',},       4226,  -7181},     // Massachusetts
     {{'K','1','\0','\0',},       4368,  -7158},     // New Hampshire
     {{'K','1','\0','\0',},       4407,  -7267},     // Vermont
     {{'K','1','\0','\0',},       4537,  -6924},     // Maine
     {{'K','2','\0','\0',},       4019,  -7467},     // New Jersey
     {{'K','2','\0','\0',},       4295,  -7553},     // New York
     {{'K','3','\0','\0',},       3899,  -7551},     // Delaware
     {{'K','3','\0','\0',},       3906,  -7679},     // Maryland
     {{'K','3','\0','\0',},       4088,  -7780},     // Pennsylvania
     {{'K','4','\0','\0',},       2863,  -8245},     // Florida
     {{'K','4','\0','\0',},       3264,  -8344},     // Georgia
     {{'K','4','\0','\0',},       3278,  -8683},     // Alabama
     {{'K','4','\0','\0',},       3392,  -8090},     // South Carolina
     {{'K','4','\0','\0',},       3556,  -7939},     // North Carolina
     {{'K','4','\0','\0',},       3586,  -8635},     // Tennessee
     {{'K','4','\0','\0',},       3752,  -7885},     // Virginia
     {{'K','4','\0','\0',},       3753,  -8530},     // Kentucky
     {{'K','5','\0','\0',},       3107,  -9200},     // Louisiana
     {{'K','5','\0','\0',},       3148,  -9933},     // Texas
     {{'K','5','\0','\0',},       3274,  -8967},     // Mississippi
     {{'K','5','\0','\0',},       3441, -10611},     // New Mexico
     {{'K','5','\0','\0',},       3489,  -9244},     // Arkansas
     {{'K','5','\0','\0',},       3559,  -9749},     // Oklahoma
     {{'K','6','\0','\0',},       3718, -11947},     // California
     {{'K','7','\0','\0',},       3427, -11166},     // Arizona
     {{'K','7','\0','\0',},       3931, -11167},     // Utah
     {{'K','7','\0','\0',},       3933, -11663},     // Nevada
     {{'K','7','\0','\0',},       4300, -10755},     // Wyoming
     {{'K','7','\0','\0',},       4393, -12056},     // Oregon
     {{'K','7','\0','\0',},       4435, -11461},     // Idaho
     {{'K','7','\0','\0',},       4705, -10963},     // Montana
     {{'K','7','\0','\0',},       4738, -12045},     // Washington
     {{'K','8','\0','\0',},       3864,  -8062},     // West Virginia
     {{'K','8','\0','\0',},       4029,  -8279},     // Ohio
     {{'K','8','\0','\0',},       4435,  -8541},     // Michigan
     {{'K','9','\0','\0',},       3989,  -8628},     // Indiana
     {{'K','9','\0','\0',},       4004,  -8920},     // Illinois
     {{'K','9','\0','\0',},       4462,  -8999},     // Wisconsin
     {{'K','H','6','\0',},        2029, -15637},     // Hawaii
     {{'K','L','7','\0',},        6407, -15228},     // Alaska
     {{'K','G','4','\0'},         1955,  -7506},
     {{'K','H','0','\0'},         1512,  14544},
     {{'K','H','1','\0'},            8, -17628},
     {{'K','H','2','\0'},         1328,  14444},
     {{'K','H','3','\0'},         1619, -16833},
     {{'K','H','4','\0'},         2814, -17548},
     {{'K','H','5','\0'},          549, -16213},
     {{'K','H','6','\0'},         2118, -15751},
     {{'K','H','7','K'},          2825, -17820},
     {{'K','H','8','\0'},        -1416, -17042},
     {{'K','H','8','\0'},        -1150, -17108},
     {{'K','H','9','\0'},         1921,  16640},
     {{'K','L','7','\0'},         6110, -15000},
     {{'K','P','1','\0'},         1823,  -7458},
     {{'K','P','2','\0'},         1821,  -6456},
     {{'K','P','4','\0'},         1829,  -6608},
     {{'K','P','5','\0'},         1823,  -6729},
     {{'L','A','\0','\0'},        6100,    900},
     {{'L','A','\0','\0'},        6600,   1100},
     {{'L','A','\0','\0'},        7000,   2300},
     {{'L','U','\0','\0'},       -3400,  -5600},
     {{'L','U','\0','\0'},       -3200,  -6400},
     {{'L','U','\0','\0'},       -3800,  -6600},
     {{'L','U','\0','\0'},       -4600,  -6700},
     {{'L','U','\0','\0'},       -5400,  -6600},
     {{'L','X','\0','\0'},        4936,    609},
     {{'L','Y','\0','\0'},        5440,   2519},
     {{'L','Z','\0','\0'},        4241,   2319},
     {{'O','A','\0','\0'},        -400,  -7500},
     {{'O','A','\0','\0'},       -1000,  -7500},
     {{'O','A','\0','\0'},       -1300,  -7200},
     {{'O','D','\0','\0'},        3353,   3529},
     {{'O','E','\0','\0'},        4813,   1620},
     {{'O','H','9','\0'},         6200,   2200},
     {{'O','H','8','\0'},         6300,   2400},
     {{'O','H','\0','\0'},        6300,   2400},
     {{'O','H','0','\0'},         6010,   1955},
     {{'O','J','0','\0'},         6018,   1907},
     {{'O','K','\0','\0'},        5005,   1426},
     {{'O','M','\0','\0'},        4809,   1709},
     {{'O','N','\0','\0'},        5050,    420},
     {{'O','X','\0','\0'},        7400,  -4000},
     {{'O','Y','\0','\0'},        6200,   -642},
     {{'O','Z','\0','\0'},        5540,   1235},
     {{'P','2','\0','\0'},        -930,  14710},
     {{'P','4','\0','\0'},        1233,  -7006},
     {{'P','5','\0','\0'},        3901,  12545},
     {{'P','A','\0','\0'},        5220,    454},
     {{'P','J','2','\0'},         1206,  -6856},
     {{'P','J','4','\0'},         1220,  -6826},
     {{'P','J','5','\0'},         1748,   6297},
     {{'P','J','7','\0'},         1804,  -6301},
     {{'P','P','1','\0'},        -2019,  -4021},
     {{'P','P','2','\0'},        -1639,  -4916},
     {{'P','P','5','\0'},        -2735,  -4831},
     {{'P','P','6','\0'},        -1055,  -3704},
     {{'P','P','7','\0'},         -940,  -3543},
     {{'P','P','8','\0'},         -300,  -6300},
     {{'P','Q','8','\0'},            2,  -5103},
     {{'P','R','7','\0'},         -707,  -3452},
     {{'P','S','7','\0'},         -547,  -3513},
     {{'P','S','8','\0'},         -505,  -4249},
     {{'P','T','2','\0'},        -1547,  -4755},
     {{'P','T','7','\0'},         -343,  -3879},
     {{'P','T','8','\0'},         -958,  -6748},
     {{'P','T','9','\0'},        -2027,  -5437},
     {{'P','V','8','\0'},          250,  -6043},
     {{'P','W','8','\0'},         -846,  -6354},
     {{'P','Y','0','F'},           351,  -3225},
     {{'P','Y','0','S'},            56,  -2934},
     {{'P','Y','0','T'},         -2025,  -2924},
     {{'P','Y','1','\0'},        -2253,  -4317},
     {{'P','Y','2','\0'},        -2333,  -4639},
     {{'P','Y','3','\0'},        -3003,  -5110},
     {{'P','Y','4','\0'},        -1955,  -4356},
     {{'P','Y','5','\0'},        -2524,  -4916},
     {{'P','Y','6','\0'},        -1259,  -3831},
     {{'P','Y','7','\0'},         -802,  -3454},
     {{'P','Y','8','\0'},         -300,  -5300},
     {{'P','Y','9','\0'},        -1000,  -5900},
     {{'P','Z','\0','\0'},         552,  -5514},
     {{'R','0','A','\0'},         5601,   9250},
     {{'R','0','B','\0'},         6925,   8615},
     {{'R','0','C','\0'},         4827,  13506},
     {{'R','0','D','\0'},         4848,  13257},
     {{'R','0','H','\0'},         6417,  10015},
     {{'R','0','I','\0'},         5934,  15047},
     {{'R','0','J','\0'},         5017,  12732},
     {{'R','0','L','\0'},         4310,  13156},
     {{'R','0','O','\0'},         5150,  10737},
     {{'R','0','Q','\0'},         6900,  11200},
     {{'R','0','S','\0'},         5216,  10420},
     {{'R','0','S','\0'},         5250,  10500},
     {{'R','0','U','\0'},         5203,  11330},
     {{'R','0','W','\0'},         5343,   9126},
     {{'R','0','X','\0'},         5600,  15900},
     {{'R','0','Y','\0'},         5142,   9427},
     {{'R','0','Z','\0'},         5454,   6906},
     {{'R','1','C','\0'},         5955,   3015},
     {{'R','1','F','\0'},         8100,   5800},
     {{'R','1','M','\0'},         6030,   2860},
     {{'R','1','N','\0'},         6147,   3420},
     {{'R','1','O','\0'},         6450,   4040},
     {{'R','1','P','\0'},         6740,   5300},
     {{'R','1','Q','\0'},         5912,   3954},
     {{'R','1','T','\0'},         5831,   3117},
     {{'R','1','W','\0'},         5750,   2820},
     {{'R','1','Z','\0'},         6859,   3308},
     {{'R','2','F','\0'},         5443,   2030},
     {{'R','3','A','\0'},         5545,   3735},
     {{'R','3','E','\0'},         5259,   3604},
     {{'R','3','G','\0'},         5237,   3935},
     {{'R','3','I','\0'},         5652,   3554},
     {{'R','3','K','\0'},         5140,   3910},
     {{'R','3','L','\0'},         5447,   3203},
     {{'R','3','M','\0'},         5737,   3952},
     {{'R','3','N','\0'},         5746,   4054},
     {{'R','3','P','\0'},         5412,   3736},
     {{'R','3','R','\0'},         5243,   4125},
     {{'R','3','S','\0'},         5438,   3944},
     {{'R','3','T','\0'},         5620,   4400},
     {{'R','3','U','\0'},         5700,   4059},
     {{'R','3','V','\0'},         5610,   4025},
     {{'R','3','W','\0'},         5142,   3611},
     {{'R','3','X','\0'},         5431,   3615},
     {{'R','3','Z','\0'},         5036,   3635},
     {{'R','4','A','\0'},         4844,   4425},
     {{'R','4','C','\0'},         5134,   4602},
     {{'R','4','F','\0'},         5313,   4500},
     {{'R','4','H','\0'},         5300,   5145},
     {{'R','4','L','\0'},         5420,   4824},
     {{'R','4','N','\0'},         5405,   3420},
     {{'R','4','P','\0'},         5545,   4908},
     {{'R','4','S','\0'},         5638,   4752},
     {{'R','4','U','\0'},         5411,   4511},
     {{'R','4','W','\0'},         5651,   5314},
     {{'R','4','Y','\0'},         5609,   4715},
     {{'R','6','A','\0'},         4502,   3900},
     {{'R','6','E','\0'},         4414,   4204},
     {{'R','6','H','\0'},         4502,   4159},
     {{'R','6','I','\0'},         4410,   4650},
     {{'R','6','J','\0'},         4440,   4310},
     {{'R','6','L','\0'},         4714,   3942},
     {{'R','6','P','\0'},         4320,   4542},
     {{'R','6','Q','\0'},         4322,   4477},
     {{'R','6','U','\0'},         4621,   4803},
     {{'R','6','W','\0'},         4258,   4730},
     {{'R','6','X','\0'},         4329,   4337},
     {{'R','6','Y','\0'},         4435,   4007},
     {{'R','9','A','\0'},         5510,   6124},
     {{'R','9','D','\0'},         5651,   6036},
     {{'R','9','F','\0'},         5800,   5615},
     {{'R','9','H','\0'},         5630,   8458},
     {{'R','9','K','\0'},         6633,   6634},
     {{'R','9','L','\0'},         5709,   6531},
     {{'R','9','M','\0'},         5500,   7323},
     {{'R','9','O','\0'},         5502,   8255},
     {{'R','9','Q','\0'},         5526,   6518},
     {{'R','9','S','\0'},         5154,   5506},
     {{'R','9','U','\0'},         5520,   8605},
     {{'R','9','W','\0'},         5444,   5556},
     {{'R','9','X','\0'},         6140,   5046},
     {{'R','9','Y','\0'},         5322,   8345},
     {{'R','9','Z','\0'},         5146,   8600},
     {{'S','0','\0','\0'},        2600,  -1100},
     {{'S','0','\0','\0'},        2300,  -1500},
     {{'S','2','\0','\0'},        2343,   9025},
     {{'S','5','\0','\0'},        4603,   1431},
     {{'S','7','\0','\0'},         438,   5527},
     {{'S','9','\0','\0'},          20,    644},
     {{'S','M','\0','\0'},        5900,   1500},
     {{'S','M','1','\0'},         5700,   1800},
     {{'S','M','2','\0'},         6600,   1900},
     {{'S','M','3','\0'},         6300,   1600},
     {{'S','P','\0','\0'},        5215,   2100},
     {{'S','T','\0','\0'},        1900,   3500},
     {{'S','T','\0','\0'},        1800,   2700},
     {{'S','T','\0','\0'},        1300,   2900},
     {{'S','U','\0','\0'},        3000,   2900},
     {{'S','U','\0','\0'},        2400,   2800},
     {{'S','U','\0','\0'},        2400,   3300},
     {{'S','V','\0','\0'},        3759,   2343},
     {{'S','V','5','\0'},         3626,   2813},
     {{'S','V','9','\0'},         3520,   2509},
     {{'S','V','A','\0'},         4009,   2421},
     {{'T','2','\0','\0'},        -922,  17943},
     {{'T','3','0','\0'},          141,  17313},
     {{'T','3','1','\0'},         -505, -17043},
     {{'T','3','2','\0'},          350, -15916},
     {{'T','3','3','\0'},          -52, -16935},
     {{'T','5','\0','\0'},         200,   4520},
     {{'T','7','\0','\0'},        4355,   1228},
     {{'T','8','\0','\0'},         751,  13458},
     {{'T','A','1','\0'},         4101,   2858},
     {{'T','A','2','\0'},         3954,   3250},
     {{'T','F','\0','\0'},        6409,  -2151},
     {{'T','G','\0','\0'},        1438,  -9031},
     {{'T','I','\0','\0'},         956,  -8405},
     {{'T','I','9','\0'},          548,  -8700},
     {{'T','J','\0','\0'},         352,   1131},
     {{'T','K','\0','\0'},        4155,    844},
     {{'T','L','\0','\0'},         422,   1835},
     {{'T','N','\0','\0'},        -416,   1517},
     {{'T','R','\0','\0'},          23,    927},
     {{'T','T','\0','\0'},        1800,   1900},
     {{'T','T','\0','\0'},        1200,   1700},
     {{'T','U','\0','\0'},         519,   -401},
     {{'T','Y','\0','\0'},         629,    237},
     {{'T','Z','\0','\0'},        2100,   -300},
     {{'T','Z','\0','\0'},        1700,      0},
     {{'T','Z','\0','\0'},        1300,   -700},
     {{'U','K','\0','\0'},        4120,   6918},
     {{'U','N','\0','\0'},        4315,   7656},
     {{'U','R','\0','\0'},        5030,   3100},
     {{'V','2','\0','\0'},        1703,  -6148},
     {{'V','3','\0','\0'},        1714,  -8845},
     {{'V','4','\0','\0'},        1610,  -6140},
     {{'V','5','\0','\0'},       -2100,   1600},
     {{'V','6','\0','\0'},         742,  15055},
     {{'V','7','\0','\0'},         700,  17100},
     {{'V','8','\0','\0'},         455,  11455},
     {{'V','E','1','\0'},         4438,  -6335},
     {{'V','E','2','\0'},         4650,  -7115},
     {{'V','E','3','\0'},         4342,  -7925},
     {{'V','E','4','\0'},         4953,  -9710},
     {{'V','E','5','\0'},         5030, -10438},
     {{'V','E','6','\0'},         5334, -11325},
     {{'V','E','7','\0'},         4825, -12322},
     {{'V','E','8','\0'},         6230, -11429},
     {{'V','E','9','\0'},         4557,  -6640},
     {{'V','K','0','H'},         -5303,   7330},
     {{'V','K','0','M'},         -5437,  15900},
     {{'V','K','1','\0'},        -3518,  14908},
     {{'V','K','2','\0'},        -3100,  14800},
     {{'V','K','3','\0'},        -3745,  14458},
     {{'V','K','4','\0'},        -2300,  14300},
     {{'V','K','5','\0'},        -3000,  13400},
     {{'V','K','6','\0'},        -2500,  12200},
     {{'V','K','7','\0'},        -4254,  14718},
     {{'V','K','8','\0'},        -1800,  13200},
     {{'V','K','9','C'},         -1140,   9651},
     {{'V','K','9','L'},         -3133,  15905},
     {{'V','K','9','M'},         -1700,  15500},
     {{'V','K','9','N'},         -2852,  16756},
     {{'V','K','9','W'},         -1614,  15000},
     {{'V','K','9','X'},         -1030,  10540},
     {{'V','O','1','\0'},         4734,  -5241},
     {{'V','O','2','\0'},         5254,  -6650},
     {{'V','P','2','E'},          1814,  -6305},
     {{'V','P','2','M'},          1644,  -6214},
     {{'V','P','2','V'},          1826,  -6437},
     {{'V','P','5','\0'},         2128,  -7108},
     {{'V','P','6','\0'},        -2440, -12447},
     {{'V','P','8','F'},         -5145,  -5900},
     {{'V','P','0','\0'},        -5406,  -3658},
     {{'V','P','0','\0'},        -6255,  -6034},
     {{'V','P','0','\0'},        -6058,  -4558},
     {{'V','P','0','\0'},        -5741,  -2806},
     {{'V','P','9','\0'},         3211,  -6445},
     {{'V','Q','9','\0'},         -519,   7228},
     {{'V','R','\0','\0'},        2217,  11409},
     {{'V','R','6','\0'},        -2504, -13100},
     {{'V','U','\0','\0'},        2100,   7800},
     {{'V','U','\0','\0'},        2800,   7600},
     {{'V','U','\0','\0'},        2300,   8500},
     {{'V','U','4','\0'},         1143,   9244},
     {{'V','U','7','\0'},         1028,   7237},
     {{'V','Y','0','\0'},         7000,  -7400},
     {{'V','Y','1','\0'},         6041, -13508},
     {{'V','Y','2','\0'},         4614,  -6309},
     {{'X','E','1','\0'},         2300, -10300},
     {{'X','E','2','\0'},         2700, -10600},
     {{'X','E','3','\0'},         1700,  -9800},
     {{'X','E','3','\0'},         2000,  -8900},
     {{'X','F','4','\0'},         1900, -11100},
     {{'X','T','\0','\0'},        1222,   -131},
     {{'X','U','\0','\0'},        1133,  10455},
     {{'X','W','\0','\0'},        1757,  10236},
     {{'X','X','9','\0'},         2214,  11335},
     {{'X','Z','\0','\0'},        1647,   9610},
     {{'Y','A','\0','\0'},        3400,   6600},
     {{'Y','B','0','\0'},         -608,  10645},
     {{'Y','B','1','\0'},         -657,  10734},
     {{'Y','B','2','\0'},         -658,  11029},
     {{'Y','B','3','\0'},         -714,  11245},
     {{'Y','B','4','\0'},         -259,  10445},
     {{'Y','B','5','\0'},         -100,  10021},
     {{'Y','B','6','\0'},          335,   9839},
     {{'Y','B','7','\0'},           -5,  10916},
     {{'Y','B','8','\0'},         -507,  11924},
     {{'Y','B','9','\0'},         -840,  11514},
     {{'Y','I','\0','\0'},        3300,   4300},
     {{'Y','J','\0','\0'},       -1744,  16819},
     {{'Y','K','\0','\0'},        3329,   3618},
     {{'Y','L','\0','\0'},        5700,   2400},
     {{'Y','N','\0','\0'},        1209,  -8617},
     {{'Y','O','\0','\0'},        4426,   2606},
     {{'Y','S','\0','\0'},        1341,  -8917},
     {{'Y','U','\0','\0'},        4455,   2030},
     {{'Y','V','\0','\0'},         300,  -6500},
     {{'Y','V','\0','\0'},         700,  -6300},
     {{'Y','V','\0','\0'},         900,  -7000},
     {{'Y','V','0','\0'},         1542,  -6338},
     {{'Z','2','\0','\0'},       -1750,   3103},
     {{'Z','3','\0','\0'},        4159,   2126},
     {{'Z','6','\0','\0'},        4266,   2116},
     {{'Z','8','\0','\0'},        1534,   3232},
     {{'Z','A','\0','\0'},        4120,   1950},
     {{'Z','B','2','\0'},         3557,   -536},
     {{'Z','C','4','\0'},         3510,   3322},
     {{'Z','D','7','\0'},        -1451,   -415},
     {{'Z','D','8','\0'},         -806,  -1418},
     {{'Z','D','9','\0'},        -3700,  -1203},
     {{'Z','F','\0','\0'},        1918,  -8123},
     {{'Z','K','3','\0'},         -922, -16951},
     {{'Z','L','\0','\0'},       -4118,  17447},
     {{'Z','L','7','\0'},        -4408, -17635},
     {{'Z','L','8','\0'},        -3059, -17850},
     {{'Z','L','9','\0'},        -5146,  16608},
     {{'Z','P','\0','\0'},       -2400,  -5700},
     {{'Z','S','\0','\0'},       -3100,   2300},
     {{'Z','S','\0','\0'},       -2600,   2900},
     {{'Z','S','8','\0'},        -4649,   3947},
};

#define N_SMALLPREFS NARRAY(small_prefs)


/* find nearest small_prefs to the given LL, if within allowed max
 * N.B. tried cty but it has way too many weird ones, eg it finds AX? instead of VK? and lots of
 *   parochial US calls. Don't try it!
 */
bool ll2Prefix (const LatLong &ll, char prefix[MAX_PREF_LEN])
{
    // scan for closest location -- use simple linear approx
    float mind = 1e10;                                  // min linear degree separation so far
    float coslat = cosf(ll.lat);                        // handy
    uint16_t closest_smpref = 0;                        // small_prefs index of closest entry
    for (int i = 0; i < N_SMALLPREFS; i++) {
        float dlat = fabsf (ll.lat_d - 0.01F * small_prefs[i].lat);
        if (dlat < mind) {                              // dont bother adding dlng if already > mind
            float dlng = coslat * fabsf (lngDiff (ll.lng_d - 0.01F * small_prefs[i].lng));
            float d = dlat + dlng;
            if (d < mind) {
                mind = d;
                closest_smpref = i;
            }
        }
    }

    // fail if too far away
    if (mind > MAX_DIST)
        return (false);

    // save in prefix[] as legitimate string
    memset (prefix, 0, MAX_PREF_LEN);
    memcpy (prefix, small_prefs[closest_smpref].pref, SMALL_PREF_LEN);

    // good
    return (true);
}

/* split a call into home and dx portions.
 * if nothing looks like a dx portion then copy call to both.
 * N.B. we assume call has already been cleaned with strTrimAll() and strtoupper()
 */
void splitCallSign (const char *raw_call, char home_call[NV_CALLSIGN_LEN], char dx_call[NV_CALLSIGN_LEN])
{
    // work with copy so we can remove any trailing -# from RBN
    char call[NV_CALLSIGN_LEN];
    quietStrncpy (call, raw_call, sizeof(call));
    char *dash_pound = strstr (call, "-#");
    if (dash_pound)
        *dash_pound = '\0';

    // split at slash, if any
    const char *slash = strchr (call, '/');

    if (slash) {

        // find left and right boundaries
        const char *left = call;                        // just handy
        const char *right = slash+1;                    // beginning of right portion
        int l_len = slash - left;                       // length of left portion
        int r_len = strlen (right);                     // length of right portion
        const char *slash2 = strchr (right, '/');
        if (slash2)
            r_len = slash2 - right;                      // don't count past a 2nd slash

        // dx is the shorter side unless it looks like a common shorthand or seems suspicious

        if (r_len <= 1 || !strcmp(right,"MM") || !strcmp(right,"AM")
                       || (r_len > 2 && (int)strcspn(right,"0123456789") == r_len)      // 3+ all alpha
                       || (int)strspn(right,"0123456789") > 1) {                        // 2+ leading digits

            // disregard suspicious right side
            snprintf (home_call, NV_CALLSIGN_LEN, "%.*s", l_len, left);
            snprintf (dx_call, NV_CALLSIGN_LEN, "%.*s", l_len, left);

        } else if (l_len <= r_len) {

            // left is shorter, consider it the dx
            snprintf (home_call, NV_CALLSIGN_LEN, "%.*s", r_len, right);
            snprintf (dx_call, NV_CALLSIGN_LEN, "%.*s", l_len, left);

        } else {

            // right is shorter, consider it the dx
            snprintf (home_call, NV_CALLSIGN_LEN, "%.*s", l_len, left);
            snprintf (dx_call, NV_CALLSIGN_LEN, "%.*s", r_len, right);

        }

    } else {

        // call is home, no dx
        snprintf (home_call, NV_CALLSIGN_LEN, "%s", call);
        snprintf (dx_call, NV_CALLSIGN_LEN, "%s", call);

    }
}

/* extract the most likely portion from the given call that appears to be the dx prefix
 */
void findCallPrefix (const char *call, char prefix[MAX_PREF_LEN])
{
    // init prefix
    memset (prefix, 0, MAX_PREF_LEN);

    // protect from empty call
    if (call[0] == '\0')
        return;

    // always work with the dx-end
    char home_call[NV_CALLSIGN_LEN];
    char dx_call[NV_CALLSIGN_LEN];
    splitCallSign (call, home_call, dx_call);

    // find rmd "right-most-digit" in dx_call
    const char *rmd = NULL;
    for (const char *cp = dx_call; *cp != '\0'; cp++)
        if (isdigit(*cp))
            rmd = cp;

    // if no rmd:            use all; will likely fail call2LL()
    // if rmd is first char: use first two chars
    // else:                 use up to and including rmd
    if (!rmd)
        snprintf (prefix, MAX_PREF_LEN, "%.*s", MAX_PREF_LEN-1, dx_call);
    else if (rmd == dx_call)
        snprintf (prefix, MAX_PREF_LEN, "%.2s", dx_call);
    else
        snprintf (prefix, MAX_PREF_LEN, "%.*s", (int)(rmd - dx_call + 1), dx_call);

    // printf ("************************ %10s %s\n", call, prefix);          // RBF
}


/***********************************************************************************
 *
 * cty file support
 * 
 ***********************************************************************************/

/* init cty values for a fresh start
 */
static void initCty(void)
{
    if (cty_list)
        free (cty_list);
    cty_list = NULL;
    n_cty = 0;
    n_malloc = 0;
}

/* crack and add another line to cty_list
 */
static void addCtyLine (char *line)
{
    // skip blank and comment lines
    if (line[0] == '\n' || line[0] == '#')
        return;

    // crack
    CtyLoc cl;
    if (sscanf (line, "%10s %f %f %d", cl.call, &cl.lat_d, &cl.lng_d, &cl.dxcc) != 4) {
        Serial.printf ("CTY: %s bad format: %s\n", cty_page, line);
        return;
    }

    // add to list, expanding as needed
    if (n_cty + 1 > n_malloc) {
        cty_list = (CtyLoc *) realloc (cty_list, (n_malloc += 1000) * sizeof(CtyLoc));
        if (!cty_list)
            fatalError ("No memory for cty location list %d\n", n_malloc);
    }
    cl.call_len = strlen(cl.call);
    cty_list[n_cty++] = cl;

    // update radix index when first char changes
    if (cl.call[0] != prev_radix) {
        int radix_index = cl.call[0] - '0';
        if (radix_index < 0 || radix_index >= _N_RADIX)
            fatalError ("cty radix out of range %d %d", radix_index, _N_RADIX);
        cty_radix[radix_index] = n_cty - 1;
        prev_radix = cl.call[0];
        // printf ("********* radix %c %5d %5d\n",cl.call[0], radix_index, radix[radix_index]);
    }
}

/* insure cty_lst and its supporting radix index are ready to use, even if stale if no other way.
 * use local file but if absent or too old try to download.
 * return whether cty_list is ready.
 */
static bool loadCtyFile(void)
{
    // out fast until next refresh
    if (myNow() < next_refresh)
        return (cty_list != NULL);

    // open cached file
    FILE *fp = openCachedFile (cty_fn, cty_page, MAX_CTY_AGE, MIN_CTY_SIZ);
    if (!fp) {
        next_refresh = myNow() + RETRY_DT;
        return (false);
    }

    // (re)build cty_list
    initCty();
    char line[100];
    while (fgets (line, sizeof(line), fp)) {
        chompString (line);
        addCtyLine (line);
    }

    // done
    fclose (fp);
    next_refresh = myNow() + MAX_CTY_AGE;
    Serial.printf ("CTY: loaded %d locations from %s\n", n_cty, cty_fn);

    // real question is whether cty_list exists
    return (cty_list != NULL);
}

/* search for best CtyLoc for the given prefix.
 * return pointer else NULL
 */
static const CtyLoc *searchCty (const char *prefix)
{
    // start at radix then find longest cty_list call entry that starts with prefix.
    const CtyLoc *candidate = NULL;
    int radix_index = prefix[0] - '0';
    if (radix_index >= 0 && radix_index < _N_RADIX) {
        int start = cty_radix[radix_index];
        int len_match = 0;                              // find longest match
        for (int i = start; i < n_cty; i++) {           // start at this radix, go to end or next radix
            const CtyLoc *cp = &cty_list[i];
            if (cp->call[0] != prefix[0])
                break;                                  // end of this radix
            if (strncmp (cp->call, prefix, cp->call_len) == 0) {
                int cc_len = strlen(cp->call);
                if (cc_len > len_match) {
                    len_match = cc_len;
                    candidate = cp;
                    if (debugLevel (DEBUG_CTY, 1))
                        Serial.printf ("CTY: match for %s now %s length %d\n", prefix, cp->call, len_match);
                }
            }
        }
    }

    return (candidate);
}


/* given a call sign or prefix find its lat/long by querying the cty table.
 * return whether successful.
 */
bool call2LL (const char *call, LatLong &ll)
{
    // check cty_list
    if (!loadCtyFile())
        return (false);

    // use the dx end of a portable call
    char home_call[NV_CALLSIGN_LEN];
    char dx_call[NV_CALLSIGN_LEN];
    splitCallSign (call, home_call, dx_call);

    // require a digit if 3 or more chars
    if (strlen(dx_call) >= 3 && !strHasDigit(dx_call)) {
        Serial.printf ("CTY: no digit in %s\n", dx_call);
        return (false);
    }

    const CtyLoc *candidate = searchCty (dx_call);

    if (candidate) {
        ll.lat_d = candidate->lat_d;
        ll.lng_d = candidate->lng_d;
        ll.normalize();
        return (true);
    } else {
        // darn
        if (strcmp (call, dx_call))
            Serial.printf ("CTY: No location for %s AKA %s\n", dx_call, call);
        else
            Serial.printf ("CTY: No location for %s\n", call);
        return (false);
    }
}

/* given a call sign or prefix find its DXCC number by querying the cty table.
 * return whether successful.
 */
bool call2DXCC (const char *call, int &dxcc)
{
    // check cty_list
    if (!loadCtyFile())
        return (false);

    // use the dx end of a portable call
    char home_call[NV_CALLSIGN_LEN];
    char dx_call[NV_CALLSIGN_LEN];
    splitCallSign (call, home_call, dx_call);

    const CtyLoc *candidate = searchCty (dx_call);

    if (candidate) {
        dxcc = candidate->dxcc;
        return (true);
    } else {
        // darn
        if (strcmp (call, dx_call))
            Serial.printf ("CTY: No DXCC for %s AKA %s\n", dx_call, call);
        else
            Serial.printf ("CTY: No DXCC for %s\n", call);
        return (false);
    }
}
