# -*- coding: utf-8 -*- #
# Copyright 2022 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Describe policy command."""


from googlecloudsdk.api_lib.container.binauthz import platform_policy
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.container.binauthz import flags


@base.DefaultUniverseOnly
@base.ReleaseTracks(base.ReleaseTrack.BETA, base.ReleaseTrack.ALPHA)
class Describe(base.DescribeCommand):
  """Describe a Binary Authorization platform policy.

  ## EXAMPLES

  To describe an existing policy using its resource name:

    $ {command} projects/my_proj/platforms/gke/policies/policy1

  To describe the same policy using flags:

    $ {command} policy1 --platform=gke --project=my_proj
  """

  @staticmethod
  def Args(parser):
    flags.AddPlatformPolicyResourceArg(parser, 'to describe')

  def Run(self, args):
    policy_ref = args.CONCEPTS.policy_resource_name.Parse().RelativeName()
    # The API is only available in v1.
    return platform_policy.Client('v1').Describe(policy_ref)
