# -*- coding: utf-8 -*- #
# Copyright 2014 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Command for listing networks."""


from googlecloudsdk.api_lib.compute import base_classes
from googlecloudsdk.api_lib.compute import lister
from googlecloudsdk.api_lib.compute import networks_utils
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.compute.networks import flags


@base.UniverseCompatible
@base.ReleaseTracks(base.ReleaseTrack.GA, base.ReleaseTrack.BETA)
class List(base.ListCommand):
  """List Compute Engine networks."""

  @staticmethod
  def Args(parser):
    lister.AddBaseListerArgs(parser)
    parser.display_info.AddFormat(flags.LIST_FORMAT_WITH_ULA_IPV6)
    parser.display_info.AddCacheUpdater(flags.NetworksCompleter)

  def Run(self, args):
    holder = base_classes.ComputeApiHolder(self.ReleaseTrack())
    client = holder.client

    request_data = lister.ParseNamesAndRegexpFlags(args, holder.resources)

    list_implementation = lister.GlobalLister(
        client, client.apitools_client.networks)

    return (networks_utils.AddModesForListFormat(resource)
            for resource in lister.Invoke(request_data, list_implementation))


@base.UniverseCompatible
@base.ReleaseTracks(base.ReleaseTrack.ALPHA)
class ListAlpha(List):
  """List Compute Engine networks."""

  @staticmethod
  def Args(parser):
    lister.AddBaseListerArgs(parser)
    parser.display_info.AddFormat(flags.LIST_FORMAT_WITH_ULA_IPV6)
    parser.display_info.AddCacheUpdater(flags.NetworksCompleter)


List.detailed_help = base_classes.GetGlobalListerHelp('networks')
