package roundtripperreverserserver

import (
	"bytes"
	"context"
	"math"
	"strings"
	"testing"

	"github.com/v2fly/struc"
)

func TestGenerateTokenLength(t *testing.T) {
	p, err := NewPasswordAccessChecker("test-password")
	if err != nil {
		t.Fatalf("failed to create checker: %v", err)
	}

	tok, err := p.GenerateToken(12345)
	if err != nil {
		t.Fatalf("GenerateToken failed: %v", err)
	}

	if len(tok) != 16 {
		t.Fatalf("expected token length 16, got %d", len(tok))
	}
}

func TestGenerateAndCheckRoundTrip(t *testing.T) {
	p, err := NewPasswordAccessChecker("another-secret")
	if err != nil {
		t.Fatalf("failed to create checker: %v", err)
	}

	userID := int64(42)
	serverKey, err := p.GenerateToken(userID)
	if err != nil {
		t.Fatalf("GenerateToken failed: %v", err)
	}

	clientKey, err := p.CheckReverserAccess(context.Background(), serverKey)
	if err != nil {
		t.Fatalf("CheckReverserAccess failed: %v", err)
	}

	if len(clientKey) != 16 {
		t.Fatalf("expected client key length 16, got %d", len(clientKey))
	}

	// decrypt clientKey using internal block to inspect contents
	decrypted := make([]byte, 16)
	p.block.Decrypt(decrypted, clientKey)

	token := &tokenUnpacked{}
	if err := struc.Unpack(bytes.NewReader(decrypted), token); err != nil {
		t.Fatalf("failed to unpack client token: %v", err)
	}

	if token.Check != 0 {
		t.Fatalf("expected check 0, got %d", token.Check)
	}
	if token.UserID != -userID {
		t.Fatalf("expected userID %d, got %d", -userID, token.UserID)
	}
}

func TestPasswordAccessChecker_ValidRoundTrip(t *testing.T) {
	p, err := NewPasswordAccessChecker("test-password")
	if err != nil {
		t.Fatalf("NewPasswordAccessChecker failed: %v", err)
	}

	userID := int64(42)
	token, err := p.GenerateToken(userID)
	if err != nil {
		t.Fatalf("GenerateToken failed: %v", err)
	}

	clientKey, err := p.CheckReverserAccess(context.TODO(), token)
	if err != nil {
		t.Fatalf("CheckReverserAccess failed: %v", err)
	}

	// decrypt clientKey and verify contents
	buf := make([]byte, 16)
	p.block.Decrypt(buf, clientKey)
	tok := &tokenUnpacked{}
	if err := struc.Unpack(bytes.NewReader(buf), tok); err != nil {
		t.Fatalf("failed to unpack client token: %v", err)
	}

	if tok.Check != 0 {
		t.Fatalf("unexpected Check value: got %d want 0", tok.Check)
	}
	if tok.UserID != -userID {
		t.Fatalf("unexpected UserID: got %d want %d", tok.UserID, -userID)
	}
}

func TestPasswordAccessChecker_InvalidKeyLength(t *testing.T) {
	p, err := NewPasswordAccessChecker("test-password")
	if err != nil {
		t.Fatalf("NewPasswordAccessChecker failed: %v", err)
	}

	_, err = p.CheckReverserAccess(context.TODO(), []byte{1, 2, 3})
	if err == nil {
		t.Fatalf("expected error for invalid key length, got nil")
	}
	if !strings.Contains(err.Error(), "invalid server key length") {
		t.Fatalf("unexpected error: %v", err)
	}
}

func TestPasswordAccessChecker_InvalidTokenCheckValue(t *testing.T) {
	p, err := NewPasswordAccessChecker("test-password")
	if err != nil {
		t.Fatalf("NewPasswordAccessChecker failed: %v", err)
	}

	// craft a token with non-zero Check
	buf := bytes.NewBuffer(make([]byte, 0, 16))
	if err := struc.Pack(buf, &tokenUnpacked{UserID: 7, Check: 1}); err != nil {
		t.Fatalf("failed to pack token: %v", err)
	}
	plain := buf.Bytes()
	if len(plain) != 16 {
		t.Fatalf("unexpected packed length: %d", len(plain))
	}
	encrypted := make([]byte, 16)
	p.block.Encrypt(encrypted, plain)

	_, err = p.CheckReverserAccess(context.TODO(), encrypted)
	if err == nil {
		t.Fatalf("expected error for invalid token check value, got nil")
	}
	if !strings.Contains(err.Error(), "invalid token check value") {
		t.Fatalf("unexpected error: %v", err)
	}
}

func TestTokensNotInterchangeable(t *testing.T) {
	p1, err := NewPasswordAccessChecker("password-one")
	if err != nil {
		t.Fatalf("failed to create checker p1: %v", err)
	}
	p2, err := NewPasswordAccessChecker("password-two")
	if err != nil {
		t.Fatalf("failed to create checker p2: %v", err)
	}

	userID := int64(7)
	// token created by p1 should not be accepted by p2
	tok1, err := p1.GenerateToken(userID)
	if err != nil {
		t.Fatalf("p1.GenerateToken failed: %v", err)
	}
	if _, err := p2.CheckReverserAccess(context.Background(), tok1); err == nil {
		t.Fatalf("expected p2 to reject token generated by p1, but it accepted it")
	}

	// token created by p2 should not be accepted by p1
	tok2, err := p2.GenerateToken(userID)
	if err != nil {
		t.Fatalf("p2.GenerateToken failed: %v", err)
	}
	if _, err := p1.CheckReverserAccess(context.Background(), tok2); err == nil {
		t.Fatalf("expected p1 to reject token generated by p2, but it accepted it")
	}
}

func TestCheckReverserAccess_NegativeUserID(t *testing.T) {
	p, err := NewPasswordAccessChecker("neg-test")
	if err != nil {
		t.Fatalf("failed to create checker: %v", err)
	}

	// craft a token with negative UserID and Check == 0
	buf := bytes.NewBuffer(make([]byte, 0, 16))
	if err := struc.Pack(buf, &tokenUnpacked{UserID: -5, Check: 0}); err != nil {
		t.Fatalf("failed to pack token: %v", err)
	}
	plain := buf.Bytes()
	if len(plain) != 16 {
		t.Fatalf("unexpected packed length: %d", len(plain))
	}
	serverKey := make([]byte, 16)
	p.block.Encrypt(serverKey, plain)

	if _, err := p.CheckReverserAccess(context.Background(), serverKey); err == nil {
		t.Fatalf("expected error for negative userID token, got nil")
	} else if !strings.Contains(err.Error(), "invalid token userID for server") {
		t.Fatalf("unexpected error for negative userID token: %v", err)
	}
}

func TestCheckReverserAccess_MinInt64Token(t *testing.T) {
	p, err := NewPasswordAccessChecker("minint-test")
	if err != nil {
		t.Fatalf("failed to create checker: %v", err)
	}

	// craft a token with MinInt64 and Check == 0
	buf := bytes.NewBuffer(make([]byte, 0, 16))
	if err := struc.Pack(buf, &tokenUnpacked{UserID: math.MinInt64, Check: 0}); err != nil {
		t.Fatalf("failed to pack token: %v", err)
	}
	plain := buf.Bytes()
	if len(plain) != 16 {
		t.Fatalf("unexpected packed length: %d", len(plain))
	}
	serverKey := make([]byte, 16)
	p.block.Encrypt(serverKey, plain)

	if _, err := p.CheckReverserAccess(context.Background(), serverKey); err == nil {
		t.Fatalf("expected error for MinInt64 token, got nil")
	} else if !strings.Contains(err.Error(), "invalid token userID: MinInt64") {
		t.Fatalf("unexpected error for MinInt64 token: %v", err)
	}
}

func TestGenerateTokensDifferentPasswords(t *testing.T) {
	p1, err := NewPasswordAccessChecker("pw-a")
	if err != nil {
		t.Fatalf("failed to create checker p1: %v", err)
	}
	p2, err := NewPasswordAccessChecker("pw-b")
	if err != nil {
		t.Fatalf("failed to create checker p2: %v", err)
	}

	userID := int64(100)
	tok1, err := p1.GenerateToken(userID)
	if err != nil {
		t.Fatalf("p1.GenerateToken failed: %v", err)
	}
	tok2, err := p2.GenerateToken(userID)
	if err != nil {
		t.Fatalf("p2.GenerateToken failed: %v", err)
	}

	if bytes.Equal(tok1, tok2) {
		t.Fatalf("expected tokens from different passwords to differ, but they are equal")
	}
}

func TestNewPasswordAccessChecker_EmptyPassword(t *testing.T) {
	// ensure creating a checker with empty password succeeds and block is set
	p, err := NewPasswordAccessChecker("")
	if err != nil {
		t.Fatalf("failed to create checker with empty password: %v", err)
	}
	if p.block == nil {
		t.Fatalf("expected block to be initialized for empty password")
	}
}
