!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module uv_filter
  use gbl_message
  !
  public :: uv_filter_comm
  public :: uv_filter_channel_parsing
  private
  !
  character(len=*), parameter :: rname='UV_FILTER'
  !
contains
  !
  subroutine uv_filter_comm(line,error)
    use gildas_def
    !----------------------------------------------------------
    ! Front end for UV_FILTER command
    !----------------------------------------------------------
    character(len=*), intent(inout) :: line  ! Command line
    logical,          intent(inout) :: error ! Error flag
    !
    integer(kind=4), allocatable :: chanlist(:,:)
    integer(kind=size_length) :: nchanlist
    integer(kind=4), allocatable :: antlist(:)
    integer(kind=4) :: nantlist
    integer(kind=4), allocatable :: baselist(:,:)
    integer(kind=4) :: nbaselist
    logical :: dozero
    !
    call uv_filter_parsing(line,chanlist,nchanlist,  &
                                antlist,nantlist, &
                                baselist,nbaselist, &
                                dozero,error)
    if (error) return
    !
    ! Only go to filtering if anything to filter
    if (nchanlist.gt.0) then
       call uv_filter_sub(chanlist,nchanlist, &
                          antlist,nantlist, &
                          baselist,nbaselist, &
                          dozero,error)
       if (error) return
    endif
  end subroutine uv_filter_comm
  !
  subroutine uv_filter_parsing(line,chanlist,nchanlist,  &
                                    antlist,nantlist,  &
                                    baselist,nbaselist,  &
                                    dozero,error)
    use gildas_def
    use gkernel_interfaces
    !----------------------------------------------------------
    ! Command line parsing for UV_FILTER
    !----------------------------------------------------------
    character(len=*),             intent(inout) :: line           ! Command line
    integer(kind=4), allocatable, intent(out)   :: chanlist(:,:)  ! Array containing the flagging ranges
    integer(kind=size_length),    intent(out)   :: nchanlist      ! Number of elements in descriptor
    integer(kind=4), allocatable, intent(out)   :: antlist(:)     ! List of antennas to flag
    integer(kind=4),              intent(out)   :: nantlist       ! Number of antennas in list
    integer(kind=4), allocatable, intent(out)   :: baselist(:,:)  ! List of baselines to flag
    integer(kind=4),              intent(out)   :: nbaselist      ! Number of baselines in list
    logical,                      intent(out)   :: dozero         ! Zero option is present
    logical,                      intent(inout) :: error          ! Error flag
    !
    integer(kind=4), parameter :: opt_zero=5 ! /ZERO
    integer(kind=4), parameter :: opt_ante=6 ! /ANTENNA
    integer(kind=4), parameter :: opt_base=7 ! /BASELINE
    integer(kind=4) :: iant,ibase,ier,larg,idashe
    character(len=8) :: arg
    !
    ! Options for channel lists
    call uv_filter_channel_parsing(rname,line,chanlist,nchanlist,error)
    if (error) return
    !
    ! Other options
    dozero = sic_present(opt_zero,0)
    !
    nantlist = sic_narg(opt_ante)
    allocate(antlist(nantlist),stat=ier)
    if (failed_allocate(rname,'antenna list',ier,error))  return
    do iant=1, nantlist
       call sic_i4(line,opt_ante,iant,antlist(iant),.true.,error)
       if (error) return
    enddo
    !
    nbaselist = sic_narg(opt_base)
    allocate(baselist(2,nbaselist),stat=ier)
    if (failed_allocate(rname,'baseline list',ier,error))  return
    do ibase=1,nbaselist
       call sic_ch(line,opt_base,ibase,arg,larg,.true.,error)
       if (error) return
       if (arg(2:2).eq.'-') then
          idashe = 2
       elseif (arg(3:3).eq.'-') then
          idashe = 3
       else
          goto 100
       endif
       read(arg(1:idashe-1),*,iostat=ier) baselist(1,ibase)
       if (ier.ne.0)  goto 100
       read(arg(idashe+1:larg),*,iostat=ier) baselist(2,ibase)
       if (ier.ne.0)  goto 100
    enddo
    return
    !
    ! Error decoding baseline list
100 continue
    call map_message(seve%e,rname,'Malformed baseline, syntax is AntI-AntJ with e.g. 1, 01, 10 etc...')
    error = .true.
    return
  end subroutine uv_filter_parsing
  !
  subroutine uv_filter_channel_parsing(rname,line,chanlist,nchanlist,error)
    use gildas_def
    use gkernel_types
    use gkernel_interfaces
    use uv_buffers
    !--------------------------------------------------------------------
    ! Command line parsing for options:
    !   /CHANNELS list
    !   /FREQUENCY list [/WIDTH width]
    !   /RANGE list
    !-------------------------------------------------------------------
    character(len=*),             intent(in)    :: rname          ! Who is calling?
    character(len=*),             intent(in)    :: line           ! Command line
    integer(kind=4), allocatable, intent(out)   :: chanlist(:,:)  ! Array containing the flagging ranges
    integer(kind=size_length),    intent(out)   :: nchanlist      ! Number of elements in descriptor
    logical,                      intent(inout) :: error          ! Error flag
    !
    integer, parameter                    :: opt_chan=1 ! option for channel list
    integer, parameter                    :: opt_freq=2 ! option for frequency list
    integer, parameter                    :: opt_rang=3 ! option for range list !!! Not implemented
    integer, parameter                    :: opt_widt=4 ! option for width
    integer, parameter                    :: nunits =3
    integer(kind=index_length), parameter :: one=1      ! 1 in i8    
    character(len=10)                     :: units(3)
    data units/'CHANNEL','FREQUENCY','VELOCITY'/
    !
    real(kind=4)                :: width            ! Width to extract around frequency in unit
    character(len=10)           :: unitin           ! Input unit
    character(len=10)           :: unit             ! Treated unit
    integer                     :: chlength         ! Argument length
    integer                     :: narg             ! Number of arguments of an option
    integer                     :: ier              ! error flag
    real(kind=8)                :: freq             ! Input frequency
    integer(kind=4)             :: channel          ! Input Channel
    integer(kind=4)             :: ilist            ! Input list index
    type(sic_descriptor_t)      :: desc             ! Sic Variable descriptor
    character(len=64)           :: listname         ! Sic variable name
    integer                     :: nopt,iopt        ! to handle options
    real(kind=4)                :: drange(2)        ! First and last values of range
    !
    ! Initialization
    !
    width = 1
    unitin = 'CHANNEL'
    !
    ! Interface
    !
    if (huv%loca%size.eq.0) then
       call map_message(seve%e,rname,'No UV data loaded')
       error = .true.
       return
    endif
    nopt = 0 
    do iopt=1, 3
       if (sic_present(iopt,0)) nopt = nopt+1
    enddo
    ! Making nchanlist 0 just in case no option is given
    nchanlist = 0
    ! Checking for conflicting options
    if (nopt.gt.1) then
       call map_message(seve%e,rname,"Options /CHANNELS, /RANGE and /FREQUENCY are mutually exclusive")
       error = .true.
       return
       ! Option range is present
    else if (sic_present(opt_rang,0)) then
       narg = sic_narg(opt_rang)
       if (narg.eq.1.or.narg.eq.0) then
          call map_message(seve%e,rname,"At least two values must be given")
          error = .true.
          return
       endif
       nchanlist = narg/2
       allocate(chanlist(2,nchanlist),stat=ier)
       if (failed_allocate(rname,'channel list',ier,error))  return
       !
       if (.not.((narg/2)*2.eq.narg)) then ! Odd, means last arg is UNIT!
          call sic_ke(line, opt_rang, narg, unitin, chlength, .true., error)
       endif
       call sic_ambigs (rname,unitin,unit,chlength,units,nunits,error)
       if (error) return
       do ilist=1,nchanlist
          call sic_r4(line,opt_rang,2*(ilist-1)+1,drange(1),.true.,error)
          call sic_r4(line,opt_rang,2*(ilist-1)+2,drange(2),.true.,error)
          if (error) return
          call uv_spectral_range_sel(drange,unit,chanlist(:,ilist),error)
       enddo
       ! Option frequency is present
    else if (sic_present(opt_freq,0)) then
       ! Getting frequency list
       narg = sic_narg(opt_freq)
       if (narg.eq.0) then
          call map_message(seve%e,rname,"At least one frequency must be given.")
          error = .true.
          return
       endif
       ! Getting width
       if (sic_present(opt_widt,0)) then
          call sic_r4(line, opt_widt, 1, width, .true., error)
          call sic_ke(line, opt_widt, 2, unitin, chlength, .false., error)
       endif
       call sic_ambigs (rname,unitin,unit,chlength,units,nunits,error)
       if (error) return
       allocate(chanlist(2,narg),stat=ier)
       if (failed_allocate(rname,'channel list',ier,error))  return
       do ilist=1,narg
          call sic_r8(line,opt_freq,ilist,freq,.true.,error)
          if (error) return
          call uv_spectral_frequency_sel(freq,width,unit,chanlist(:,ilist),error)
       enddo
       nchanlist = narg
       ! Option channels is present
    else if (sic_present(opt_chan,0)) then
       narg = sic_narg(opt_chan)
       if (narg.eq.0) then
          call map_message(seve%e,rname,"At least one channel must be given.")
          error = .true.
          return
       endif
       if (narg.gt.1) then
          allocate(chanlist(2,narg),stat=ier)
          if (failed_allocate(rname,'channel list',ier,error))  return
          do ilist=1,narg
             call sic_i4(line,opt_chan,ilist,channel,.true.,error)
             if (error) return
             chanlist(:,ilist) = channel
          enddo
          nchanlist = narg
       else
          call sic_de(line,opt_chan,1,desc,.true.,error)
          if (error) then
             call map_message(seve%e,rname,'Variable does not exists.')
             return
          endif
          if (.not.(desc%type.eq.fmt_i8.or.desc%type.eq.fmt_i4)) then
             call map_message(seve%e,rname,'Variable '//trim(listname)//' must be Integer ')
             error = .true.
             return
          endif
          if (desc%ndim.gt.1) then
             call map_message(seve%e,rname,'Variable '//trim(listname)//' must have rank 1')
             error = .true.
             return
          endif
          nchanlist = desc_nelem(desc)
          if (nchanlist.lt.2) then
             allocate(chanlist(2,1),stat=ier)
             if (failed_allocate(rname,'channel list',ier,error))  return
             call sic_descriptor_getval(desc,one,channel,error)
             if (error) then
                call map_message(seve%e,rname,"Can't retrieve value from SIC variable")
                return
             end if
             chanlist = channel
          else
             allocate(chanlist(2,nchanlist),stat=ier)
             if (failed_allocate(rname,'channel list',ier,error))  return
             do ilist=1, nchanlist
                call sic_descriptor_getval(desc,one*ilist,channel,error)
                if (error) then
                   call map_message(seve%e,rname,"Can't retrieve value from SIC variable")
                   return
                end if
                chanlist(:,ilist) = channel
             enddo
          endif
          call sic_volatile(desc)
       endif
    else
       if (trim(rname).eq."UV_FILTER") call map_message(seve%w,rname,"No options given skipping filtering")
       if (trim(rname).eq."UV_BASELINE") call map_message(seve%w,rname,"No windows given, fitting baseline to all channels")
    end if
  end subroutine uv_filter_channel_parsing
  !
  subroutine uv_filter_sub(chanlist,nchanlist, &
                           antlist,nantlist, &
                           baselist,nbaselist, &
                           dozero,error)
    use gildas_def, only: index_length,size_length
    use mapping_interfaces
    use uv_buffers
    !-------------------------------------------------------------------
    ! Apply flagging to a list of channels given as input
    !-------------------------------------------------------------------
    integer(kind=4),           intent(in)    :: chanlist(:,:)
    integer(kind=size_length), intent(in)    :: nchanlist
    integer(kind=4),           intent(in)    :: antlist(:)
    integer(kind=4),           intent(in)    :: nantlist
    integer(kind=4),           intent(in)    :: baselist(:,:)
    integer(kind=4),           intent(in)    :: nbaselist
    logical,                   intent(in)    :: dozero
    logical,                   intent(inout) :: error
    !
    integer(kind=size_length) :: iranges
    integer(kind=index_length) :: fchan,lchan,ichan
    character(len=50) :: warning
    logical :: doantenna,dobaseline,doallvisi
    !
    doantenna  = nantlist.gt.0
    dobaseline = nbaselist.gt.0
    doallvisi  = nantlist.le.0 .and. nbaselist.le.0
     !
    do iranges=1,nchanlist
       fchan = chanlist(1,iranges)
       lchan = chanlist(2,iranges)
       if (lchan.lt.1.or.fchan.gt.huv%gil%nchan) then
          write(warning,'(i0,a)') iranges,"-th spectral range is outside UV table"
          call map_message(seve%w,rname,warning)
          cycle
       endif
       if (fchan.lt.1)             fchan = 1
       if (lchan.gt.huv%gil%nchan) lchan = huv%gil%nchan
       !
       do ichan=fchan,lchan
          if (dozero) then
             if (doantenna)  call uv_spectral_zero_by_antenna(ichan,antlist,nantlist,error)
             if (dobaseline) call uv_spectral_zero_by_baseline(ichan,baselist,nbaselist,error)
             if (doallvisi)  call uv_spectral_zero(ichan,error)
          else
             if (doantenna)  call uv_spectral_flag_by_antenna(ichan,antlist,nantlist,error)
             if (dobaseline) call uv_spectral_flag_by_baseline(ichan,baselist,nbaselist,error)
             if (doallvisi)  call uv_spectral_flag(ichan,error)
          endif
       enddo
    enddo
  end subroutine uv_filter_sub
end module uv_filter
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
