
#include "gtk-local.h"

#include "gtk-graph.h"
#include "gtk/gtk.h"

#include "magnify-tool.h"
#include "gtk-toolbar.h"
#include "gtv/event-stack.h"
#include "zoom-tool.h"
#include "message-c.h"

#ifndef darwin
#include "malloc.h"
#else
#include "stdlib.h"
#endif

typedef gushort ggtk_colormap_index;

#define NPEN 16

/*
 * Maximum number of primitives rendered per expose cycle.
 * Lower values improve UI responsiveness on heavy plots.
 * Higher values reduce total number of redraw cycles.
 */
#define GGTK_DRAW_CHUNK 32768U

static int Width_default;
static int Height_default;

static int pencil_colors[NPEN];
static int pix_colors[NCOL];

typedef struct {
    gboolean valid;
    double width;
    gboolean dashed;
    gboolean invert;
    double dash_list[_DASH_SIZE];
    double red;
    double green;
    double blue;
    cairo_line_cap_t cap_style;
    cairo_line_join_t join_style;
} PolylineStyleState;

static void _check_task(char *proc)
{
    if (gtv_called_from_main()) {
        ggtk_c_message(seve_e, "GTK", "%s: call from main thread", proc);
    }
}

static void polyline_init(
    Polyline *pl, 
    size_t reserve, 
    ggtk_env_t *env,
    gboolean is_filled)
{
    pl->n = 0;
    pl->cap = reserve ? reserve : 0;
    pl->pts = reserve ? (Point*)malloc(reserve * sizeof(Point)) : NULL;
    pl->width = env->line_width;
    pl->dashed = env->line_is_dashed;
    if(pl->dashed) {
        int i;
        for (i = 0; i < _DASH_SIZE; i++)
           pl->dash_list[i] = env->dash_list[i];
    }
    pl->red = env->red;
    pl->green = env->green;
    pl->blue = env->blue;
    pl->filled = is_filled;
    pl->invert = env->invert;
    pl->cap_style = env->cap_style;
    pl->join_style = env->join_style;
}

static void polyline_free(Polyline *pl)
{
    free(pl->pts);
    pl->pts = NULL; pl->n = pl->cap = 0;
}

static void polystore_init(PolyStore *S, size_t reserve_lines)
{
    S->n = 0;
    S->cap = reserve_lines ? reserve_lines : 0;
    S->lines = reserve_lines ? (Polyline*)malloc(reserve_lines * sizeof(Polyline)) : NULL;
}

static void polystore_free(PolyStore *S)
{
    size_t i;
    for (i = 0; i < S->n; ++i)
        polyline_free(&S->lines[i]);
    free(S->lines);
    S->lines = NULL; S->n = S->cap = 0;
}

static void polyline_add_point(Polyline *pl, int x, int y)
{
    if (pl->n == pl->cap) {
        size_t new_cap = pl->cap ? pl->cap * 2 : 64;
        Point *nb = (Point*)realloc(pl->pts, new_cap * sizeof(Point));
        if (!nb) return;
        pl->pts = nb; pl->cap = new_cap;
    }
    pl->pts[pl->n++] = (Point){ x, y };
}

static Polyline* polystore_add_polyline(PolyStore *S, size_t reserve_points, ggtk_env_t *env, gboolean is_polygon)
{
    if (S->n == S->cap) {
        size_t new_cap = S->cap ? S->cap * 2 : 16;
        Polyline *nb = (Polyline*)realloc(S->lines, new_cap * sizeof(Polyline));
        if (!nb) return NULL;
        S->lines = nb; S->cap = new_cap;
    }
    Polyline *pl = &S->lines[S->n++];
    polyline_init(pl, reserve_points, env, is_polygon);
    return pl;
}

static void imagestore_init(ImageStore *S, size_t reserve)
{
    S->n = 0;
    S->cap = reserve ? reserve : 0;
    S->positions = reserve ? (Point*)malloc(reserve * sizeof(Point)) : NULL;
    S->surfaces = NULL;
}

static void imagestore_free(ImageStore *S) 
{
    GList *l;
    for (l = S->surfaces; l; l = l->next)
        cairo_surface_destroy((cairo_surface_t*)l->data);
    g_list_free(S->surfaces);
    S->surfaces = NULL;
    S->n = 0;
    S->cap = 0;
    free(S->positions);
    S->positions = NULL;
}

static void store_surface(ImageStore *S, cairo_surface_t *s, int x, int y) 
{
    S->surfaces = g_list_append(S->surfaces, s);
    if (S->n == S->cap) {
        size_t new_cap = S->cap ? S->cap * 2 : 64;
        Point *new_positions = (Point*)realloc(S->positions, new_cap * sizeof(Point));
        if (!new_positions) return;
        S->positions = new_positions; S->cap = new_cap;
    }
    S->positions[S->n++] = (Point){ x, y };
}

/* Reset incremental rendering counters for the cached scene. */
static void _ggtk_reset_scene_progress(ggtk_env_t *genv)
{
    genv->rendered_primitive_count = 0;
    genv->rendered_poly_count = 0;
    genv->rendered_image_count = 0;
}

/* Clear the cached scene surface with the current background color. */
static void _ggtk_clear_scene_surface(ggtk_env_t *genv, int white_back)
{
    cairo_t *scene_cr;

    if (genv->scene_surface == NULL)
        return;
    scene_cr = cairo_create(genv->scene_surface);
    if (white_back == 1)
        cairo_set_source_rgb(scene_cr, 1.0, 1.0, 1.0);
    else
        cairo_set_source_rgb(scene_cr, 0.0, 0.0, 0.0);
    cairo_paint(scene_cr);
    cairo_destroy(scene_cr);
    genv->scene_background = white_back;
    _ggtk_reset_scene_progress(genv);
}

/* Drop cached surface and restart incremental rendering state. */
static void _ggtk_reset_scene_cache(ggtk_env_t *genv)
{
    if (genv->scene_surface != NULL) {
        cairo_surface_destroy(genv->scene_surface);
        genv->scene_surface = NULL;
    }
    genv->scene_surface_width = 0;
    genv->scene_surface_height = 0;
    genv->scene_background = -1;
    _ggtk_reset_scene_progress(genv);
}

/* Ensure the cached scene surface exists and matches widget geometry. */
static void _ggtk_prepare_scene_surface(ggtk_env_t *genv, int width, int height,
 int white_back)
{
    if (width <= 0 || height <= 0)
        return;
    if (genv->scene_surface == NULL
     || genv->scene_surface_width != width
     || genv->scene_surface_height != height) {
        _ggtk_reset_scene_cache(genv);
        genv->scene_surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,
         width, height);
        if (cairo_surface_status(genv->scene_surface) != CAIRO_STATUS_SUCCESS) {
            cairo_surface_destroy(genv->scene_surface);
            genv->scene_surface = NULL;
            return;
        }
        genv->scene_surface_width = width;
        genv->scene_surface_height = height;
        genv->scene_background = -1;
    }
    if (genv->scene_background != white_back)
        _ggtk_clear_scene_surface(genv, white_back);
}

/* Queue at most one pending redraw for this drawing area. */
static void _ggtk_queue_draw(ggtk_env_t *genv)
{
    if (genv == NULL || genv->drawing_area == NULL)
        return;
    if (genv->draw_pending)
        return;
    genv->draw_pending = TRUE;
    gtk_widget_queue_draw(genv->drawing_area);
}

/* Compare two dash arrays used by Cairo stroke styles. */
static gboolean _same_dash_pattern(const double *a, const double *b)
{
    int i;
    for (i = 0; i < _DASH_SIZE; i++) {
        if (a[i] != b[i])
            return FALSE;
    }
    return TRUE;
}

/* Return TRUE when cached style state already matches this polyline. */
static gboolean _polyline_style_matches(PolylineStyleState *state, Polyline *pl)
{
    if (!state->valid)
        return FALSE;
    if (state->width != pl->width || state->dashed != pl->dashed
     || state->invert != pl->invert || state->cap_style != pl->cap_style
     || state->join_style != pl->join_style)
        return FALSE;
    if (!state->invert
     && (state->red != pl->red || state->green != pl->green
      || state->blue != pl->blue))
        return FALSE;
    if (state->dashed && !_same_dash_pattern(state->dash_list, pl->dash_list))
        return FALSE;
    return TRUE;
}

/* Apply polyline style only when it differs from the cached one. */
static void _apply_polyline_style(cairo_t *cr, Polyline *pl,
 PolylineStyleState *state)
{
    int i;
    if (_polyline_style_matches(state, pl))
        return;

    cairo_set_line_cap(cr, pl->cap_style);
    cairo_set_line_join(cr, pl->join_style);
    cairo_set_line_width(cr, pl->width);
    if (pl->dashed)
        cairo_set_dash(cr, pl->dash_list, _DASH_SIZE, 0.0);
    else
        cairo_set_dash(cr, NULL, 0, 0);
    if (pl->invert) {
        cairo_set_operator(cr, CAIRO_OPERATOR_DIFFERENCE);
        cairo_set_source_rgb(cr, 1, 1, 1);
    } else {
        cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
        cairo_set_source_rgb(cr, pl->red, pl->green, pl->blue);
    }

    state->valid = TRUE;
    state->width = pl->width;
    state->dashed = pl->dashed;
    state->invert = pl->invert;
    if (pl->dashed) {
        for (i = 0; i < _DASH_SIZE; i++)
            state->dash_list[i] = pl->dash_list[i];
    }
    state->red = pl->red;
    state->green = pl->green;
    state->blue = pl->blue;
    state->cap_style = pl->cap_style;
    state->join_style = pl->join_style;
}

static void _ggtk_init_genv(ggtk_env_t *env)
{
    init_genv( (G_env *)env);
    env->post_refresh_handler = NULL;
    env->post_refresh_handler_data = NULL;
    env->line_width = 0.5;
    env->line_is_dashed = FALSE;
    env->cap_style      = CAIRO_LINE_CAP_BUTT;
    env->join_style     = CAIRO_LINE_JOIN_MITER;
    env->red = 0.0;
    env->green = 0.0;
    env->blue = 0.0;
    env->invert = FALSE;
    env->antialiasing = CAIRO_ANTIALIAS_NONE;
    polystore_init(&env->polylines_store, 8);
    imagestore_init(&env->images_store, 8);
    env->poly_or_image_store = g_array_new(FALSE, FALSE, sizeof(gboolean));
    env->draw_pending = FALSE;
    env->scene_surface = NULL;
    env->scene_surface_width = 0;
    env->scene_surface_height = 0;
    env->scene_background = -1;
    env->rendered_primitive_count = 0;
    env->rendered_poly_count = 0;
    env->rendered_image_count = 0;
    CursorData cursor;
    cursor.display_cursor = FALSE;
    env->cursor = cursor;
}

static void internal_draw_polyline_or_polygon(cairo_t *cr, Polyline *pl,
 PolylineStyleState *style_state)
{
    if (!pl || !cr || pl->n < 1) return;
    _apply_polyline_style(cr, pl, style_state);
    cairo_new_path(cr);
    if(pl->n > 1) {
        cairo_move_to(cr, pl->pts[0].x, pl->pts[0].y);
        size_t k;
        for (k = 1; k < pl->n; ++k)
            cairo_line_to(cr, pl->pts[k].x, pl->pts[k].y);
    } else {
        cairo_rectangle(cr, pl->pts[0].x-0.5, pl->pts[0].y-0.5, 1, 1);
    }
    if(pl->filled || pl->n == 1) {
        cairo_fill(cr);
    } else {
        cairo_stroke(cr);
    }
}

static void internal_draw_image(cairo_t *cr, cairo_surface_t *s, Point *position) {
    if (!cr || !s || cairo_surface_status(s) != CAIRO_STATUS_SUCCESS) return;
    cairo_save(cr);
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    cairo_set_source_surface(
        cr, 
        s, 
        position->x-1, 
        position->y-1
    );
    cairo_paint(cr);
    cairo_restore(cr);
}

void draw_scene_from_genv(
    GtkWidget *widget, 
    cairo_t *cr, 
    ggtk_env_t *genv, 
    int white_back, 
    gboolean draw_border)
{
    if(!genv || !widget) return;
    genv->draw_pending = FALSE;
    cairo_set_antialias(cr, genv->antialiasing);
    int width  = gtk_widget_get_allocated_width(widget);
    int height = gtk_widget_get_allocated_height(widget);
    _ggtk_prepare_scene_surface(genv, width, height, white_back);
    if (genv->scene_surface == NULL)
        return;

    PolyStore *poly_store = &genv->polylines_store;
    ImageStore *images_store = &genv->images_store;
    GArray *poly_or_image = genv->poly_or_image_store;
    guint nb_primitives = poly_or_image->len;
    guint start_primitive;
    guint end_primitive;
    if (genv->rendered_primitive_count > nb_primitives
     || genv->rendered_poly_count > poly_store->n
     || genv->rendered_image_count > images_store->n)
        _ggtk_clear_scene_surface(genv, white_back);

    start_primitive = genv->rendered_primitive_count;
    end_primitive = start_primitive;
    if (nb_primitives - start_primitive > GGTK_DRAW_CHUNK)
        end_primitive += GGTK_DRAW_CHUNK;
    else
        end_primitive = nb_primitives;

    if (start_primitive < end_primitive) {
        cairo_t *scene_cr = cairo_create(genv->scene_surface);
        PolylineStyleState style_state;
        guint i;

        style_state.valid = FALSE;
        cairo_set_antialias(scene_cr, genv->antialiasing);
        for (i = start_primitive; i < end_primitive; i++) {
            gboolean is_poly = g_array_index(poly_or_image, gboolean, i);
            if (is_poly == TRUE) {
                if (genv->rendered_poly_count < poly_store->n) {
                    internal_draw_polyline_or_polygon(scene_cr,
                     &poly_store->lines[genv->rendered_poly_count],
                     &style_state);
                    genv->rendered_poly_count++;
                }
            } else {
                if (genv->rendered_image_count < images_store->n) {
                    cairo_surface_t *s = g_list_nth_data(images_store->surfaces,
                     genv->rendered_image_count);
                    internal_draw_image(scene_cr, s,
                     &images_store->positions[genv->rendered_image_count]);
                    genv->rendered_image_count++;
                }
            }
            genv->rendered_primitive_count = i + 1;
        }
        cairo_destroy(scene_cr);
    }

    cairo_set_source_surface(cr, genv->scene_surface, 0, 0);
    cairo_paint(cr);
    if (genv->rendered_primitive_count < nb_primitives)
        _ggtk_queue_draw(genv);

    if(genv->cursor.display_cursor) {
        cairo_save(cr);
        cairo_set_operator(cr, CAIRO_OPERATOR_DIFFERENCE);
        cairo_set_source_rgb(cr, 1, 1, 1);
        cairo_set_line_width(cr, 1);
        cairo_move_to(cr, genv->cursor.x, 0);
        cairo_line_to (cr, genv->cursor.x, height);
        cairo_stroke(cr);
        cairo_move_to(cr, 0, genv->cursor.y);
        cairo_line_to (cr, width, genv->cursor.y);
        cairo_stroke(cr);
        if (genv->cursor.width && genv->cursor.height) {
            cairo_rectangle(cr, genv->cursor.x - genv->cursor.width, genv->cursor.y - genv->cursor.height, 
            genv->cursor.width * 2, genv->cursor.height * 2);
            cairo_stroke(cr);
        }
        cairo_restore(cr);
    }

    if(draw_border) {
        cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
        cairo_set_dash(cr, NULL, 0, 0);
        cairo_set_line_width(cr, 1);
        cairo_rectangle(cr, 0.5, 0.5, width - 1.0, height - 1.0);
        cairo_stroke(cr);
    }
}

static G_env *ggtk_creer_genv(void)
{
    ggtk_env_t *env;

    if ((env = (ggtk_env_t *)calloc(1, sizeof(ggtk_env_t))) != NULL) {
        _ggtk_init_genv( env);
    } else {
        ggtk_c_message(seve_e, "GTK", "Fail to allocate new genv");
    }
    return (G_env *)env;
}

void set_cursor(ggtk_env_t *genv, int x, int y, gboolean display_cursor, int width, int height)
{
    if(genv->cursor.display_cursor != display_cursor ||
        genv->cursor.x != x ||
        genv->cursor.y != y ||
        genv->cursor.width != width ||
        genv->cursor.height != height) 
    {
        genv->cursor.display_cursor = display_cursor;
        genv->cursor.x = x;
        genv->cursor.y = y;
        genv->cursor.width = width;
        genv->cursor.height = height;
        _ggtk_queue_draw(genv);
    }
}

static void ggtk_get_corners( G_env *env, int x[3], int y[3])
{
    int width, height;
    ggtk_screen_size(&width, &height);
    width = width - env->Width_graphique;
    height = height - env->Height_graphique;
    x[0] = 0;
    x[1] = width >> 1;
    x[2] = width;
    y[2] = 0;
    y[1] = height >> 1;
    y[0] = height;
}

static void ggtk_move_window( G_env *_env, int _x, int _y)
{
    if (((ggtk_env_t *)_env)->main_widget == NULL) return;
    GdkWindow *window = gtk_widget_get_window(((ggtk_env_t*)_env)->main_widget);
    if (window == NULL) return;
    gdk_window_move(window, _x, _y);
}

static void ggtk_resize_window( G_env *_env, int _width, int _height)
{
    if (((ggtk_env_t *)_env)->main_widget == NULL) return;
    GdkWindow *window = gtk_widget_get_window(((ggtk_env_t*)_env)->main_widget);
    if (window == NULL) return;
    int width, height;
    ggtk_screen_size(&width, &height);
    if (_width < 2 || _height < 2 || _width > width || _height > height) {
        ggtk_c_message(seve_w, "GTK",
         "ggtk_resize_window: invalid values (%d %d)", _width, _height);
        return;
    }
    gdk_window_resize(window, _width, _height);
}

static void ggtk_destroy_window( G_env *env)
{
    clear_graphic_primitives((ggtk_env_t *)env);
    gtk_widget_destroy(((ggtk_env_t *)env)->main_widget);
}

void ggtk_set_post_refresh_handler( ggtk_env_t *genv, ggtk_gpointer_handler_t h, gpointer user_data)
{
    genv->post_refresh_handler = h;
    genv->post_refresh_handler_data = user_data;
}

void ggtk_call_post_refresh_handler( ggtk_env_t *genv)
{
    if (genv->post_refresh_handler != NULL)
        genv->post_refresh_handler( genv->post_refresh_handler_data);
}

/*****************************************************************************/

static void ggtk_refresh_window( fortran_type adr_dir, int mode, G_env *env)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    gtv_refresh_win( adr_dir, env, mode);
}

static void ggtk_flush_window( )
{
    GdkDisplay *display = gdk_display_get_default();
    if (!display) return;
    gdk_display_flush(display);
}

static void _ggtk_release_graphic_primitives(ggtk_env_t *genv,
 gboolean recreate_primitive_list)
{
    if(!genv) return;
    polystore_free(&genv->polylines_store);
    imagestore_free(&genv->images_store);
    if(genv->poly_or_image_store) {
        g_array_free(genv->poly_or_image_store, TRUE);
        if (recreate_primitive_list)
            genv->poly_or_image_store = g_array_new(FALSE, FALSE,
             sizeof(gboolean));
        else
            genv->poly_or_image_store = NULL;
    }
    _ggtk_reset_scene_cache(genv);
    genv->draw_pending = FALSE;
}

void clear_graphic_primitives(ggtk_env_t *genv) {
    _ggtk_release_graphic_primitives(genv, TRUE);
}

void ggtk_release_genv(ggtk_env_t *genv)
{
    _ggtk_release_graphic_primitives(genv, FALSE);
    free(genv);
}

static void ggtk_clear_window( G_env *env)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    clear_graphic_primitives(genv);
    _ggtk_queue_draw(genv);
}

static void ggtk_invalidate_window( G_env *env)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    clear_graphic_primitives(genv);
    _ggtk_queue_draw(genv);
}

static int ggtk_get_image_width( int w)
{
    return w;
}

static void _ggtk_set_rgb( ggtk_env_t *genv, GdkRGBA *c)
{
    genv->red = (float)c->red / MAX_INTENSITY;
    genv->green = (float)c->green / MAX_INTENSITY;
    genv->blue = (float)c->blue / MAX_INTENSITY;
}

static void ggtk_pen_invert( G_env *env)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    genv->invert = !genv->invert;
}

static void ggtk_pen_rgb( G_env *env, int r, int g, int b)
{
    GdkRGBA c;
    c.red = r;
    c.green = g;
    c.blue = b;
    _ggtk_set_rgb((ggtk_env_t *)env, &c);
}

static void ggtk_pen_color( G_env *env, char *color)
{
    GdkRGBA c;
    if (!gdk_rgba_parse(&c, color)) {
        ggtk_c_message(seve_e, "GTK", "Unknown color: %s", color);
        return;
    }
    _ggtk_set_rgb((ggtk_env_t *)env, &c);
}

static void ggtk_fill_poly( G_env *env, int nb_pts, int *x, int *y)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    int i;
    if (nb_pts) {
        Polyline *a = polystore_add_polyline(&genv->polylines_store, 128, genv, TRUE);
        for (i = 0; i < nb_pts; i++) {
            polyline_add_point(a, x[i], y[i]);
        }
        gboolean draw_poly = TRUE;
        g_array_append_val(genv->poly_or_image_store, draw_poly);
        _ggtk_queue_draw(genv);
    }
}

static void ggtk_ask_for_corners( int *ax, int *bx, int *ay, int *by)
{
}

static void ggtk_other_curs( G_env *env, char *cde)
{
}

static void ggtk_weigh( G_env *env, int weight)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    _check_task("ggtk_weigh");
    genv->line_width = weight;
    if (!genv->line_width)
        genv->line_width = 1.0;
}

static void ggtk_dash( G_env *env, int flag_dash, int ds[_DASH_SIZE])
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    _check_task("ggtk_dash");
    genv->line_is_dashed = flag_dash;
    if (flag_dash) {
        int i;
        for (i = 0; i < _DASH_SIZE; i++)
           genv->dash_list[i] = ds[i];
        genv->cap_style = CAIRO_LINE_CAP_BUTT;
    } else {
        genv->cap_style = CAIRO_LINE_CAP_SQUARE;
    }
}

static void ggtk_ximage_loadrgb( int red[], int green[], int blue[], int n,
 int dir)
{
}

static GdkColor *_default_colormap = NULL;
static int _default_colormap_size = 0;

GdkColor *ggtk_default_colormap()
{
    return _default_colormap;
}

int ggtk_default_colormap_size()
{
    return _default_colormap_size;
}

void ggtk_xcolormap_set_default( size_t colormap)
{
    _default_colormap = (GdkColor *)colormap;
    _default_colormap_size = *(int *)((GdkColor *)colormap - 1);
}

size_t ggtk_xcolormap_create( float red[], float green[], float blue[], int n, int is_default)
{
    GdkColor *colormap;
    GdkColor *color;
    int i;

    if ((colormap = (GdkColor *)calloc(n + 1, sizeof(GdkColor))) == NULL) {
        ggtk_c_message(seve_e, "GTK", "Fail to allocate  colormap");
        return 0;
    }
    *(int *)colormap = n;
    colormap++;

    for (i = 0, color = colormap; i < n; i++, color++) {
        color->red   = (int)  (red[i] * MAX_INTENSITY + 0.5);
        color->green = (int)(green[i] * MAX_INTENSITY + 0.5);
        color->blue  = (int) (blue[i] * MAX_INTENSITY + 0.5);
    }
    if (is_default) {
        ggtk_xcolormap_set_default( (size_t)colormap);
    }
    return (size_t)colormap;
}

void ggtk_xcolormap_delete( size_t colormap)
{
    free( (GdkColor *)colormap - 1);
}

static int ggtk_ximage_inquire( int *lcol, int *ocol, int *nx,
    int *ny, int *is_color, int *is_static)
{
    *lcol = 65536; /* The most we can encode on 2-byte integers */
    *ocol = 0;
    *is_color = 1;
    *is_static = 1;
    *nx = Width_default;
    *ny = Height_default;
    return 1;
}

void ggtk_draw_image(ggtk_env_t *genv, size_t *adr_data,
 int n_x0, int n_y0, int n_larg, int n_haut, size_t colormap)
{
    ggtk_colormap_index *data = (ggtk_colormap_index *)adr_data;
    GdkColor *color;
    int i, j, k;
    cairo_surface_t *s;
    cairo_format_t format;
    int stride;
    unsigned char *cbuf;

    color = (GdkColor *)colormap;
    format = CAIRO_FORMAT_RGB24;
    stride = cairo_format_stride_for_width( format, n_larg);
    if (stride <= 0) {
        ggtk_c_message(seve_e,"GTK","Invalid stride");
        return;
    }
    s = cairo_image_surface_create(format, n_larg, n_haut);
    cbuf = cairo_image_surface_get_data(s);
    k = 0;
    for (j = 0; j < n_haut; j++) {
        for (i = 0; i < n_larg; i++) {
            cbuf[j*stride+i*4] = (unsigned char)(color[data[k]].blue >> 8);
            cbuf[j*stride+i*4+1] = (unsigned char)(color[data[k]].green >> 8);
            cbuf[j*stride+i*4+2] = (unsigned char)(color[data[k]].red >> 8);
            cbuf[j*stride+i*4+3] = 123; // alpha channel. unused with CAIRO_FORMAT_RGB24.
            k++;
        }
    }
    cairo_surface_mark_dirty(s);
    store_surface(&genv->images_store, s, n_x0, n_y0);
    gboolean draw_poly = FALSE;
    g_array_append_val(genv->poly_or_image_store, draw_poly);
    _ggtk_queue_draw(genv);
}

void ggtk_draw_rgb( G_env *env, size_t *r, size_t *g, size_t *b, int n_x0, int n_y0, int n_larg, int n_haut)
{
    int i, j,k;
    cairo_surface_t *s;
    cairo_format_t format;
    int stride;
    unsigned char *cbuf;

    ggtk_env_t *genv = (ggtk_env_t *)env;
    format = CAIRO_FORMAT_RGB24;
    stride = cairo_format_stride_for_width( format, n_larg);
    if (stride <= 0) {
        ggtk_c_message(seve_e,"GTK","Invalid stride");
        return;
    }
    s = cairo_image_surface_create(format, n_larg, n_haut);
    cbuf = cairo_image_surface_get_data(s);
    k = 0;
    for (j = 0; j < n_haut; j++) {
        for (i = 0; i < n_larg; i++) {
            cbuf[j*stride+i*4] = (unsigned char)(((gushort*)b)[k] >> 8);
            cbuf[j*stride+i*4+1] = (unsigned char)(((gushort*)g)[k] >> 8);
            cbuf[j*stride+i*4+2] = (unsigned char)(((gushort*)r)[k] >> 8);
            cbuf[j*stride+i*4+3] = 123; // alpha channel. unused with CAIRO_FORMAT_RGB24.
            k++;
        }
    }
    cairo_surface_mark_dirty(s);
    store_surface(&genv->images_store, s, n_x0, n_y0);
    gboolean draw_poly = FALSE;
    g_array_append_val(genv->poly_or_image_store, draw_poly);
    _ggtk_queue_draw(genv);
}

/* ggtk_affiche_image1 invert red and blue colors */
#define ggtk_affiche_image2 ggtk_affiche_image

static void ggtk_affiche_image2( G_env *env, size_t *adr_data, int n_x0,
    int n_y0, int n_larg, int n_haut, int val_trou, size_t colormap)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    _check_task("ggtk_affiche_image2");
    ggtk_draw_image(genv, adr_data, n_x0, n_y0, n_larg,
     n_haut, colormap);
}

static void ggtk_clal( G_env *env)
{
    if (((ggtk_env_t *)env)->main_widget == NULL) return;
    GdkWindow *win = gtk_widget_get_window(((ggtk_env_t*)env)->main_widget);
    if (win == NULL) return;
    gdk_window_raise(win);
}

static void ggtk_clpl( G_env *env)
{
    if (((ggtk_env_t *)env)->main_widget == NULL) return;
    GdkWindow *win = gtk_widget_get_window(((ggtk_env_t*)env)->main_widget);
    if (win == NULL) return;
    gdk_window_lower(win);
}

static void ggtk_cmdwin()
{
}

static void ggtk_size( G_env *env, int *width, int *height)
{
    *width = env->Width_graphique;
    *height = env->Height_graphique;
}

void ggtk_screen_size( int *width, int *height)
{
    GdkDisplay *display = gdk_display_get_default();
    if (!display) return;

    GdkMonitor *monitor = gdk_display_get_primary_monitor(display);
    if (!monitor) {
        /* Fallback: first monitor */
        if (gdk_display_get_n_monitors(display) > 0)
            monitor = gdk_display_get_monitor(display, 0);
        else
            return;
    }

    GdkRectangle geometry;
    gdk_monitor_get_geometry(monitor, &geometry);

    /* geometry is in application pixels; for device pixels multiply by scale */
    int scale = gdk_monitor_get_scale_factor(monitor);
    *width  = geometry.width  * scale;
    *height = geometry.height * scale;
}

static void ggtk_flush_points( G_env *env, struct _point tabpts[], int npts)
{
    ggtk_env_t *genv = (ggtk_env_t *)env;
    int i;
    if (npts) {
        Polyline *a = polystore_add_polyline(&genv->polylines_store, 128, genv, FALSE);
        for (i = 0; i < npts; i++) {
            polyline_add_point(a, tabpts[i].x, tabpts[i].y);
        }
        gboolean draw_poly = TRUE;
        g_array_append_val(genv->poly_or_image_store, draw_poly);
        _ggtk_queue_draw(genv);
    }
}

static void ggtk_close( void)
{
}

static void ggtk_quit( void)
{
    gtk_main_quit();
}

static void ggtk_new_graph(
    int backg,
    G_env *env,
    char *dirname,
    int width, int height,
    fortran_type dir_cour,
    int reuse)
{
    /* check values */
#define GTV_MINIMAL_WIDTH 100
    if (width  < 0)  width = GTV_MINIMAL_WIDTH;
    if (height < 0)  height = GTV_MINIMAL_WIDTH;

    /* Gardons ces dimensions pour les autres */
    if (width > 1) {
        Width_default = width;
    } else {
        width = Width_default;
    }
    if (height > 1) {
        Height_default = height;
    } else {
        height = Height_default;
    }

    env->adr_dir = dir_cour;

    if (reuse) {
        gtv_push_clear( env);
    } else {
        static ggtk_toolbar_args_t args;
        ggtk_env_t *genv = (ggtk_env_t *)env;

        env->win_backg = backg;
        env->Width_graphique = width;
        env->Height_graphique = height;
        env->update_gtv_lut = 1;
        //args.herit.cmap = cmap;
        strcpy( args.herit.lut_path, sic_s_get_logical_path( "GAG_LUT:"));
        strcpy( args.herit.tmp_path, sic_s_get_logical_path( "GAG_TMP:"));
        args.herit.pix_colors = pix_colors;
        args.herit.named_colors = NULL;
        args.herit.pencil_colors = pencil_colors;
        args.herit.ncells = 0;
        args.herit.ncells_pen = 0;

        args.herit.win_graph = 0;
        args.herit.win_width = width;
        args.herit.win_height = height;
        args.herit.win_main = 0;
        args.env = genv;
        args.window_name = dirname;

        launch_gtv_main_loop( NULL);
        gtv_push_create_window( (gtv_toolbar_args_t *)&args);
        // sic_wait_widget_created( ) must be called by caller (createwindow_wait)

        //ggtk_c_message(seve_t,"GTK","cmap %d",          (int)cmap);
        ggtk_c_message(seve_t,"GTK","pix_colors %d",    pix_colors[0]);
        //ggtk_c_message(seve_t,"GTK","ncells %d",        ncells);
        ggtk_c_message(seve_t,"GTK","pencil_colors %d", pencil_colors[0]);
        //ggtk_c_message(seve_t,"GTK","ncells_pen %d",    ncells_pen);
    }
}

static int ggtk_open_x(
    int cross               /* !=0 if the user wants crosshair cursor  */
)
{
    void ggtk_init();
    ggtk_init();
    return 1;
}

G_env *ggtk_new_genv_from( G_env *ref_genv, GtkWidget *drawing_area, int width,
 int height, gboolean connect_expose)
{
    G_env *genv;

    genv = ggtk_creer_genv();
    genv->adr_dir = ref_genv->adr_dir;
    genv->win_backg = ref_genv->win_backg;
    ggtk_attach_window_genv( (ggtk_env_t *)genv, drawing_area, drawing_area,
     connect_expose);

    return genv;
}

static int _on_event( void *event)
{
    //printf( " on_event: %s\n", gtv_event_name(event));
    gtv_on_event( event);
    return FALSE; // do not call again
}

static void ggtk_push_event( void *event)
{
    //printf( " push_event: %s\n", gtv_event_name(event));
    g_idle_add(_on_event, (gpointer)event);
}

static void ggtk_activate_lens( G_env *genv)
{
    ggtk_activate_magnify( NULL, genv);
}

static graph_api_t s_ggtk_graph_api = {
    ggtk_creer_genv,
    ggtk_destroy_window,
    ggtk_refresh_window,
    NULL,
    NULL,
    ggtk_flush_window,
    ggtk_move_window,
    ggtk_resize_window,
    ggtk_invalidate_window,
    ggtk_clear_window,
    ggtk_get_corners,
    ggtk_get_image_width,
    ggtk_new_graph,
    NULL,
    ggtk_open_x,
    ggtk_pen_invert,
    ggtk_pen_rgb,
    ggtk_pen_color,
    ggtk_fill_poly,
    ggtk_activate_zoom,
    ggtk_ask_for_corners,
    ggtk_other_curs,
    ggtk_weigh,
    ggtk_dash,
    NULL,
    ggtk_ximage_loadrgb,
    ggtk_xcolormap_create,
    ggtk_xcolormap_set_default,
    ggtk_xcolormap_delete,
    ggtk_ximage_inquire,
    ggtk_affiche_image,
    ggtk_draw_rgb,
    ggtk_clal,
    ggtk_clpl,
    ggtk_cmdwin,
    ggtk_size,
    ggtk_screen_size,
    ggtk_flush_points,
    ggtk_close,
    ggtk_create_drawing_area,
    ggtk_push_event,
    ggtk_quit,
    ggtk_activate_lens,
};

#ifdef GAG_USE_STATICLINK
#define init_graph_api CFC_EXPORT_NAME( init_graph_api)

void CFC_API init_graph_api()
#elif defined(WIN32)
// to force link of ggtk into greg
void ggtk_entry( )
{
}
BOOL APIENTRY DllMain( HANDLE hInstance, 
                      DWORD ul_reason_being_called, LPVOID lpReserved)
#else
__attribute__((constructor))
static void ggtk_on_init( )
#endif
{
#if defined(WIN32) && !defined(GAG_USE_STATICLINK)
    if (ul_reason_being_called == DLL_PROCESS_ATTACH) {
#endif
#if !defined(GAG_USE_STATICLINK)
    void ggtk_init_gui();
    ggtk_init_gui();
#endif
    set_graph_api( &s_ggtk_graph_api);
#if defined(WIN32) && !defined(GAG_USE_STATICLINK)
    }
    return TRUE;
#endif
}
