#include "gtk-local.h"

#include "gtk-toolbar.h"

#include "hsv-control.h"

#include "magnify-tool.h"
#include "gtk-graph.h"
#include "gtv/event-stack.h"
#include <stdio.h>
#include <string.h>
#ifdef WIN32
#include <gdk/gdkwin32.h>
#endif

static void post_command( const char *args, ...)
{
    va_list l;

    va_start( l, args);
    if (sic_post_command_va( args, l) == -1)
        gdk_display_beep( gdk_display_get_default());
    va_end( l);
}

static void click_item( GtkWidget *w, void *data)
{
}

static void set_pen_number( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\PENCIL %d", (size_t)data);
}

static void set_pen_color( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\PENCIL /COLOUR %d", (size_t)data);
}

static void set_pen_dash( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\PENCIL /DASHED %d", (size_t)data);
}

static void set_pen_weight( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\PENCIL /WEIGHT %d", (size_t)data);
}

static void set_marker_nsides( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\SET MARKER %d * * *", (size_t)data);
}

static void set_marker_mstyle( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\SET MARKER * %d * *", (size_t)data);
}

static void set_marker_size( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\SET MARKER * * %f *", (size_t)data * 0.2);
}

static void set_marker_orientation( GtkWidget *widget, void *data)
{
    post_command( "GREG1\\SET MARKER * * * %d", (size_t)data * 15);
}


static void send_command(
GtkWidget *widget,
void *data)
{
    post_command( (char *)data);
}

static void ggtk_hardcopy(
GtkWidget *widget,
G_env *genv)
{
    char dir_name[1024];
    gtv_get_dir_name( genv, dir_name, sizeof(dir_name));
    post_command( "GTVL\\HARDCOPY /PRINT /DIRECTORY %s", dir_name);
}

static void ggtk_clear(
GtkWidget *widget,
G_env *genv)
{
    char dir_name[1024];
    gtv_get_dir_name( genv, dir_name, sizeof(dir_name));
    post_command( "GTVL\\CLEAR DIRECTORY %s", dir_name);
}

static void ggtk_continue(
GtkWidget *widget,
G_env *genv)
{
    post_command( "SIC\\CONTINUE");
}

static void ggtk_edit_lut(
GtkWidget *widget,
G_env *genv)
{
    run_hsv_control( genv);
}

/* ---------- draw/resize handlers (GTK3) ---------- */

static gboolean draw_callback(GtkWidget *widget, cairo_t *cr, gpointer data)
{
    ggtk_env_t *genv = (ggtk_env_t *)data;
    G_env *env = (G_env *)data;
    draw_scene_from_genv(widget, cr, genv, env->win_backg, FALSE);
    if (genv->post_refresh_handler != NULL) {
        ggtk_call_post_refresh_handler(genv);
    }
    return TRUE; // don't continue propagation
}

static gboolean configure_event_callback( GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
    ggtk_env_t *genv = (ggtk_env_t *)data;
    clear_graphic_primitives(genv);
    gtv_on_resize((G_env *)data, event->width, event->height);
    gtv_refresh( (G_env *)data, GTV_REFRESH_MODE_REWIND);
    return TRUE; // don't continue propagation
}

/* ---------- pointer/event dispatch ---------- */

static _event_handler_t _pointer_event_handler;
static gpointer _pointer_user_data;

void ggtk_set_pointer_event_handler( _event_handler_t h, gpointer user_data)
{
    _pointer_event_handler = h;
    _pointer_user_data = user_data;
}

int ggtk_call_pointer_event_handler( GdkEvent *event, gpointer event_data)
{
    if (_pointer_event_handler == NULL)
        return 0;

    return _pointer_event_handler( event, event_data, _pointer_user_data);
}

/* ---------- input callbacks ---------- */

static gboolean button_press_callback( GtkWidget *widget, GdkEventButton *event, gpointer data)
{
     //printf("button_press_callback: %d\n", event->button);
    if (ggtk_call_pointer_event_handler( (GdkEvent *)event, data))
        return TRUE; // don't continue propagation

    if (event->button == 2)
        ggtk_activate_magnify( event, data);
    return TRUE; // don't continue propagation
}

static gboolean button_release_callback( GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    //printf("button_release_callback: %d\n", event->button);
    ggtk_call_pointer_event_handler( (GdkEvent *)event, data);
    return TRUE; // don't continue propagation
}

static gboolean motion_notify_callback( GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
    //printf("motion_notify_callback\n");
    ggtk_call_pointer_event_handler( (GdkEvent *)event, data);
    return TRUE; // don't continue propagation
}

static gboolean scroll_notify_callback( GtkWidget *widget, GdkEventScroll *event, gpointer data)
{
    //printf("scroll_notify_callback: %d %f %f\n", event->type, event->x, event->y);
    return ggtk_call_pointer_event_handler( (GdkEvent *)event, data) == 1;
}

static gboolean key_press_callback( GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    // printf("key_press_callback\n");
    ggtk_call_pointer_event_handler( (GdkEvent *)event, data);
    return TRUE; // don't continue propagation
}

static gboolean delete_event_callback( GtkWidget *widget, GdkEvent *event, gpointer data)
{
    /* return FALSE for emitting "destroy" */
    return FALSE;
}

static int _gtk_nb_windows = 0;

static void destroy_event_callback( GtkWidget *widget, gpointer data)
{
    //printf( "gtk-toolbar: destroy_event_callback: %d\n", _gtk_nb_windows);
    _gtk_nb_windows--;
    gtv_on_destroy_genv( (G_env *)data);
}

static void _on_tool_item_clicked( GtkWidget *toggle)
{
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), TRUE);
}

static void _find_toggle_callback( GtkWidget *widget, gpointer data)
{
    if (GTK_IS_TOGGLE_BUTTON(widget)) {
        *(GtkWidget**)data = widget;
    } else if (GTK_IS_CONTAINER(widget)) {
        gtk_container_forall(GTK_CONTAINER(widget),  _find_toggle_callback,
        data);
    }
}

GtkWidget *ggtk_menu_new( const char* text, GtkWidget *parent, gboolean parent_is_menu_bar)
{
    GtkWidget *menu_pane;
    GtkWidget *item;

    menu_pane = gtk_menu_new( );
    if (parent_is_menu_bar) {
        GtkWidget *toggle;
        item =  (GtkWidget *)gtk_menu_tool_button_new( NULL, text);
        gtk_container_forall(GTK_CONTAINER(item),  _find_toggle_callback,
         &toggle);
        g_signal_connect_swapped( item, "clicked",
         G_CALLBACK(_on_tool_item_clicked), toggle);
        gtk_menu_tool_button_set_menu( GTK_MENU_TOOL_BUTTON(item), menu_pane);
        gtk_toolbar_insert( GTK_TOOLBAR(parent), GTK_TOOL_ITEM(item), -1);
    } else {
        item = gtk_menu_item_new_with_label( text);
        gtk_menu_item_set_submenu( GTK_MENU_ITEM(item), menu_pane);
        gtk_widget_show(item);
        if (parent_is_menu_bar) {
            gtk_menu_shell_append( GTK_MENU_SHELL(parent), item);
        } else {
            gtk_menu_shell_append( GTK_MENU_SHELL(parent), item);
        }
    }

    return menu_pane;
}

GtkWidget *ggtk_menu_item_on_menu_bar_new( const char* text, GtkWidget *menu_bar, GCallback click_item, gpointer data)
{
    GtkToolItem *item;

    item =  gtk_tool_button_new( NULL, text);
    gtk_toolbar_insert( GTK_TOOLBAR(menu_bar), item, -1);
    g_signal_connect( item, "clicked", click_item, data);

    return (GtkWidget *)item;
}

GtkWidget *ggtk_menu_item_new( const char* text, GtkWidget *parent, GCallback click_item, gpointer data)
{
    GtkWidget *item;

    item = gtk_menu_item_new_with_label( text);
    gtk_menu_shell_append( GTK_MENU_SHELL(parent), item);
    g_signal_connect( item, "activate", click_item, data);
    gtk_widget_show(item);
    return item;
}

static GtkWidget *ggtk_image_menu_item_from_pixbuf(const char* text, GdkPixbuf *pixbuf,
                                                   GtkWidget *parent, GCallback click_item, gpointer data)
{
    GtkWidget *item = gtk_menu_item_new();
    GtkWidget *box  = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 6);
    GtkWidget *img  = gtk_image_new_from_pixbuf(pixbuf);
    GtkWidget *lbl  = gtk_label_new(text);

    gtk_container_add(GTK_CONTAINER(item), box);
    gtk_box_pack_start(GTK_BOX(box), img, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(box), lbl, FALSE, FALSE, 0);

    gtk_menu_shell_append(GTK_MENU_SHELL(parent), item);
    g_signal_connect(item, "activate", click_item, data);
    gtk_widget_show_all(item);
    return item;
}

static GdkPixbuf* _make_line_preview_pixbuf(double width, const double *dashes, int ndashes)
{
    const int W = 64, H = 32;
    cairo_surface_t *surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, W, H);
    cairo_t *cr = cairo_create(surface);

    /* white background */
    cairo_set_source_rgb(cr, 1, 1, 1);
    cairo_paint(cr);

    /* black stroke */
    cairo_set_source_rgb(cr, 0, 0, 0);
    cairo_set_line_width(cr, width);
    if (dashes && ndashes > 0) {
        cairo_set_dash(cr, dashes, ndashes, 0.0);
    } else {
        cairo_set_dash(cr, NULL, 0, 0.0);
    }
    cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);
    cairo_set_line_join(cr, CAIRO_LINE_JOIN_ROUND);
    cairo_move_to(cr, 0, H/2.0);
    cairo_line_to(cr, W, H/2.0);
    cairo_stroke(cr);

    GdkPixbuf *pix = gdk_pixbuf_get_from_surface(surface, 0, 0, W, H);
    cairo_destroy(cr);
    cairo_surface_destroy(surface);
    return pix;
}

static void _fill_menu_bar(GtkWidget *menu_bar, ggtk_env_t *env)
{
    GtkWidget *menu_pane;
    GtkWidget *menu_pane2;
    GtkWidget *item;
    int i;
    char dum[32];

    menu_pane = ggtk_menu_new( "Pencil", menu_bar, 1);

    menu_pane2 = ggtk_menu_new( "Number", menu_pane, 0);
    for (i = 0; i <= 15; i++) {
        sprintf( dum, "%d", i);
        item = ggtk_menu_item_new( dum, menu_pane2, G_CALLBACK( set_pen_number), (gpointer)(size_t)i);
    }

    menu_pane2 = ggtk_menu_new( "Dash", menu_pane, 0);
    static double patterns[][4] = {
        { },                  /* solid (handled as 0 length)        */
        {10,10},              /* 2 entries                          */
        {2,6},
        {10,10,2,10},
        {22,18},
        {20,8,1,8},
        {20,8,8,8}
    };
    static int plen[] = {0,2,2,4,2,4,4};

    for (i = 1; i <= 7; i++) {
        GdkPixbuf *pix = _make_line_preview_pixbuf(5.0,
                                plen[i-1] ? patterns[i-1] : NULL,
                                plen[i-1]);
        sprintf( dum, "%d", i);
        ggtk_image_menu_item_from_pixbuf( dum, pix, menu_pane2,
                                          G_CALLBACK( set_pen_dash),
                                          (gpointer)(size_t)i);
        g_object_unref(pix);
    }

    menu_pane2 = ggtk_menu_new( "Weight", menu_pane, 0);
    for (i = 1; i <= 5; i++) {
        GdkPixbuf *pix = _make_line_preview_pixbuf((double)i, NULL, 0);
        sprintf( dum, "%d", i);
        ggtk_image_menu_item_from_pixbuf( dum, pix, menu_pane2,
                                          G_CALLBACK( set_pen_weight),
                                          (gpointer)(size_t)i);
        g_object_unref(pix);
    }

    menu_pane = ggtk_menu_new( "Marker", menu_bar, 1);
    menu_pane2 = ggtk_menu_new( "Nsides", menu_pane, 0);
    for (i = 1; i <= 10; i++) {
        sprintf( dum, "%d", i);
        item = ggtk_menu_item_new( dum, menu_pane2, G_CALLBACK( set_marker_nsides), (gpointer)(size_t)i);
    }

    menu_pane2 = ggtk_menu_new( "Mstyle", menu_pane, 0);
    static const char *mstyle_list[] = {
        "Convex polygon","Vertex connected","Starred polygon",
        "Filled convex polygon","Filled starred polygon"
    };
    for (i = 0; i <= 4; i++) {
        item = ggtk_menu_item_new( mstyle_list[i], menu_pane2, G_CALLBACK( set_marker_mstyle), (gpointer)(size_t)i);
    }

    menu_pane2 = ggtk_menu_new( "Size", menu_pane, 0);
    for (i = 1; i <= 10; i++) {
        sprintf( dum, "%0.1f cm", i*0.2);
        item = ggtk_menu_item_new( dum, menu_pane2, G_CALLBACK( set_marker_size), (gpointer)(size_t)i);
    }

    menu_pane2 = ggtk_menu_new( "Orientation", menu_pane, 0);
    for (i = 0; i <= 12; i++) {
        sprintf( dum, "%d degrees", i*15);
        item = ggtk_menu_item_new( dum, menu_pane2, G_CALLBACK( set_marker_orientation), (gpointer)(size_t)i);
    }

    ggtk_menu_item_on_menu_bar_new( "Hardcopy", menu_bar, G_CALLBACK(
     ggtk_hardcopy), (gpointer)env);
    ggtk_menu_item_on_menu_bar_new( "Draw", menu_bar, G_CALLBACK(
     send_command), (gpointer)"DRAW");
    ggtk_menu_item_on_menu_bar_new( "Clear", menu_bar, G_CALLBACK(
     ggtk_clear), (gpointer)env);
    ggtk_menu_item_on_menu_bar_new( "Select LUT", menu_bar, G_CALLBACK(
     ggtk_edit_lut), (gpointer)env);
    ggtk_menu_item_on_menu_bar_new( "Continue", menu_bar, G_CALLBACK(
     ggtk_continue), (gpointer)env);
}

void ggtk_attach_window_genv(ggtk_env_t *env, GtkWidget *window, GtkWidget
 *drawing_area, gboolean connect_expose)
{
    _gtk_nb_windows++;
    env->main_widget = window;
    env->drawing_area = drawing_area;

    if (connect_expose) {
        g_signal_connect( drawing_area, "draw",
         G_CALLBACK(draw_callback), env);
    }
    g_signal_connect( drawing_area, "configure_event",
     G_CALLBACK(configure_event_callback), env);
    g_signal_connect( drawing_area, "button_press_event",
     G_CALLBACK(button_press_callback), env);
    g_signal_connect( drawing_area, "button_release_event",
     G_CALLBACK(button_release_callback), env);
    g_signal_connect( drawing_area, "motion_notify_event",
     G_CALLBACK(motion_notify_callback), env);
    g_signal_connect( drawing_area, "scroll_event",
    G_CALLBACK(scroll_notify_callback), env);
    g_signal_connect( window, "key_press_event",
     G_CALLBACK(key_press_callback), env);
    g_signal_connect( window, "key_release_event",
     G_CALLBACK(key_press_callback), env);
    g_signal_connect( window, "delete_event",
     G_CALLBACK(delete_event_callback), env);
    g_signal_connect( window, "destroy",
     G_CALLBACK (destroy_event_callback), env);
}

void ggtk_create_drawing_area(gtv_toolbar_args_t *args)
{
    ggtk_toolbar_args_t *gargs = (ggtk_toolbar_args_t *)args;
    GtkWidget *window;
    GtkWidget *main_box;
    GtkWidget *menu_bar;
    GtkWidget *drawing_area;

    /* Create a new window */
    window = gtk_window_new( GTK_WINDOW_TOPLEVEL);
    gtk_window_set_default_size( GTK_WINDOW(window), args->win_width,
     args->win_height);
    gtk_window_set_focus_on_map( GTK_WINDOW(window), FALSE);

    /* Set the window title */
    gtk_window_set_title( GTK_WINDOW(window), gargs->window_name);

    main_box = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gtk_container_add( GTK_CONTAINER(window), main_box);

    menu_bar = gtk_toolbar_new( );
    gtk_toolbar_set_style( GTK_TOOLBAR(menu_bar), GTK_TOOLBAR_TEXT);
    if (menu_bar != NULL) {
        _fill_menu_bar(menu_bar, gargs->env);
        gtk_box_pack_start( GTK_BOX(main_box), menu_bar, FALSE, FALSE, 0);
    }

    drawing_area = gtk_drawing_area_new();
    gtk_box_pack_start(GTK_BOX(main_box), drawing_area, TRUE, TRUE, 0);
    gtk_widget_set_hexpand(drawing_area, TRUE);
    gtk_widget_set_vexpand(drawing_area, TRUE);
    gtk_widget_set_can_focus(drawing_area, TRUE);

    gtk_widget_set_events( drawing_area,
        GDK_BUTTON_MOTION_MASK
        | GDK_BUTTON_PRESS_MASK
        | GDK_BUTTON_RELEASE_MASK
        | GDK_KEY_PRESS_MASK
        | GDK_KEY_RELEASE_MASK
    );
    ggtk_attach_window_genv(gargs->env, window, drawing_area, TRUE);

    gtk_widget_show_all(window);
    gtk_window_present(GTK_WINDOW(window));
    sic_post_widget_created();
}

int run_gtk_main_loop( void *data)
{
    static int active = 0;
    void ggtk_init();
    ggtk_init();
    if (active) return 1;
    active = 1;
    gtk_main();
    active = 0;
    return 0;
}
