/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Command enum for all available keyboard shortcuts
 */
export var Command;
(function (Command) {
    // Basic Controls
    Command["RETURN"] = "basic.confirm";
    Command["ESCAPE"] = "basic.cancel";
    Command["QUIT"] = "basic.quit";
    Command["EXIT"] = "basic.exit";
    // Cursor Movement
    Command["HOME"] = "cursor.home";
    Command["END"] = "cursor.end";
    Command["MOVE_UP"] = "cursor.up";
    Command["MOVE_DOWN"] = "cursor.down";
    Command["MOVE_LEFT"] = "cursor.left";
    Command["MOVE_RIGHT"] = "cursor.right";
    Command["MOVE_WORD_LEFT"] = "cursor.wordLeft";
    Command["MOVE_WORD_RIGHT"] = "cursor.wordRight";
    // Editing
    Command["KILL_LINE_RIGHT"] = "edit.deleteRightAll";
    Command["KILL_LINE_LEFT"] = "edit.deleteLeftAll";
    Command["CLEAR_INPUT"] = "edit.clear";
    Command["DELETE_WORD_BACKWARD"] = "edit.deleteWordLeft";
    Command["DELETE_WORD_FORWARD"] = "edit.deleteWordRight";
    Command["DELETE_CHAR_LEFT"] = "edit.deleteLeft";
    Command["DELETE_CHAR_RIGHT"] = "edit.deleteRight";
    Command["UNDO"] = "edit.undo";
    Command["REDO"] = "edit.redo";
    // Scrolling
    Command["SCROLL_UP"] = "scroll.up";
    Command["SCROLL_DOWN"] = "scroll.down";
    Command["SCROLL_HOME"] = "scroll.home";
    Command["SCROLL_END"] = "scroll.end";
    Command["PAGE_UP"] = "scroll.pageUp";
    Command["PAGE_DOWN"] = "scroll.pageDown";
    // History & Search
    Command["HISTORY_UP"] = "history.previous";
    Command["HISTORY_DOWN"] = "history.next";
    Command["REVERSE_SEARCH"] = "history.search.start";
    Command["SUBMIT_REVERSE_SEARCH"] = "history.search.submit";
    Command["ACCEPT_SUGGESTION_REVERSE_SEARCH"] = "history.search.accept";
    Command["REWIND"] = "history.rewind";
    // Navigation
    Command["NAVIGATION_UP"] = "nav.up";
    Command["NAVIGATION_DOWN"] = "nav.down";
    Command["DIALOG_NAVIGATION_UP"] = "nav.dialog.up";
    Command["DIALOG_NAVIGATION_DOWN"] = "nav.dialog.down";
    Command["DIALOG_NEXT"] = "nav.dialog.next";
    Command["DIALOG_PREV"] = "nav.dialog.previous";
    // Suggestions & Completions
    Command["ACCEPT_SUGGESTION"] = "suggest.accept";
    Command["COMPLETION_UP"] = "suggest.focusPrevious";
    Command["COMPLETION_DOWN"] = "suggest.focusNext";
    Command["EXPAND_SUGGESTION"] = "suggest.expand";
    Command["COLLAPSE_SUGGESTION"] = "suggest.collapse";
    // Text Input
    Command["SUBMIT"] = "input.submit";
    Command["NEWLINE"] = "input.newline";
    Command["OPEN_EXTERNAL_EDITOR"] = "input.openExternalEditor";
    Command["PASTE_CLIPBOARD"] = "input.paste";
    Command["BACKGROUND_SHELL_ESCAPE"] = "backgroundShellEscape";
    Command["BACKGROUND_SHELL_SELECT"] = "backgroundShellSelect";
    Command["TOGGLE_BACKGROUND_SHELL"] = "toggleBackgroundShell";
    Command["TOGGLE_BACKGROUND_SHELL_LIST"] = "toggleBackgroundShellList";
    Command["KILL_BACKGROUND_SHELL"] = "backgroundShell.kill";
    Command["UNFOCUS_BACKGROUND_SHELL"] = "backgroundShell.unfocus";
    Command["UNFOCUS_BACKGROUND_SHELL_LIST"] = "backgroundShell.listUnfocus";
    Command["SHOW_BACKGROUND_SHELL_UNFOCUS_WARNING"] = "backgroundShell.unfocusWarning";
    Command["SHOW_SHELL_INPUT_UNFOCUS_WARNING"] = "shellInput.unfocusWarning";
    // App Controls
    Command["SHOW_ERROR_DETAILS"] = "app.showErrorDetails";
    Command["SHOW_FULL_TODOS"] = "app.showFullTodos";
    Command["SHOW_IDE_CONTEXT_DETAIL"] = "app.showIdeContextDetail";
    Command["TOGGLE_MARKDOWN"] = "app.toggleMarkdown";
    Command["TOGGLE_COPY_MODE"] = "app.toggleCopyMode";
    Command["TOGGLE_YOLO"] = "app.toggleYolo";
    Command["CYCLE_APPROVAL_MODE"] = "app.cycleApprovalMode";
    Command["SHOW_MORE_LINES"] = "app.showMoreLines";
    Command["EXPAND_PASTE"] = "app.expandPaste";
    Command["FOCUS_SHELL_INPUT"] = "app.focusShellInput";
    Command["UNFOCUS_SHELL_INPUT"] = "app.unfocusShellInput";
    Command["CLEAR_SCREEN"] = "app.clearScreen";
    Command["RESTART_APP"] = "app.restart";
    Command["SUSPEND_APP"] = "app.suspend";
})(Command || (Command = {}));
/**
 * Default key binding configuration
 * Matches the original hard-coded logic exactly
 */
export const defaultKeyBindings = {
    // Basic Controls
    [Command.RETURN]: [{ key: 'return' }],
    [Command.ESCAPE]: [{ key: 'escape' }, { key: '[', ctrl: true }],
    [Command.QUIT]: [{ key: 'c', ctrl: true }],
    [Command.EXIT]: [{ key: 'd', ctrl: true }],
    // Cursor Movement
    [Command.HOME]: [
        { key: 'a', ctrl: true },
        { key: 'home', shift: false, ctrl: false },
    ],
    [Command.END]: [
        { key: 'e', ctrl: true },
        { key: 'end', shift: false, ctrl: false },
    ],
    [Command.MOVE_UP]: [
        { key: 'up', shift: false, alt: false, ctrl: false, cmd: false },
    ],
    [Command.MOVE_DOWN]: [
        { key: 'down', shift: false, alt: false, ctrl: false, cmd: false },
    ],
    [Command.MOVE_LEFT]: [
        { key: 'left', shift: false, alt: false, ctrl: false, cmd: false },
    ],
    [Command.MOVE_RIGHT]: [
        { key: 'right', shift: false, alt: false, ctrl: false, cmd: false },
        { key: 'f', ctrl: true },
    ],
    [Command.MOVE_WORD_LEFT]: [
        { key: 'left', ctrl: true },
        { key: 'left', alt: true },
        { key: 'b', alt: true },
    ],
    [Command.MOVE_WORD_RIGHT]: [
        { key: 'right', ctrl: true },
        { key: 'right', alt: true },
        { key: 'f', alt: true },
    ],
    // Editing
    [Command.KILL_LINE_RIGHT]: [{ key: 'k', ctrl: true }],
    [Command.KILL_LINE_LEFT]: [{ key: 'u', ctrl: true }],
    [Command.CLEAR_INPUT]: [{ key: 'c', ctrl: true }],
    [Command.DELETE_WORD_BACKWARD]: [
        { key: 'backspace', ctrl: true },
        { key: 'backspace', alt: true },
        { key: 'w', ctrl: true },
    ],
    [Command.DELETE_WORD_FORWARD]: [
        { key: 'delete', ctrl: true },
        { key: 'delete', alt: true },
    ],
    [Command.DELETE_CHAR_LEFT]: [{ key: 'backspace' }, { key: 'h', ctrl: true }],
    [Command.DELETE_CHAR_RIGHT]: [{ key: 'delete' }, { key: 'd', ctrl: true }],
    [Command.UNDO]: [
        { key: 'z', cmd: true, shift: false },
        { key: 'z', alt: true, shift: false },
    ],
    [Command.REDO]: [
        { key: 'z', ctrl: true, shift: true },
        { key: 'z', cmd: true, shift: true },
        { key: 'z', alt: true, shift: true },
    ],
    // Scrolling
    [Command.SCROLL_UP]: [{ key: 'up', shift: true }],
    [Command.SCROLL_DOWN]: [{ key: 'down', shift: true }],
    [Command.SCROLL_HOME]: [
        { key: 'home', ctrl: true },
        { key: 'home', shift: true },
    ],
    [Command.SCROLL_END]: [
        { key: 'end', ctrl: true },
        { key: 'end', shift: true },
    ],
    [Command.PAGE_UP]: [{ key: 'pageup' }],
    [Command.PAGE_DOWN]: [{ key: 'pagedown' }],
    // History & Search
    [Command.HISTORY_UP]: [{ key: 'p', shift: false, ctrl: true }],
    [Command.HISTORY_DOWN]: [{ key: 'n', shift: false, ctrl: true }],
    [Command.REVERSE_SEARCH]: [{ key: 'r', ctrl: true }],
    [Command.REWIND]: [{ key: 'double escape' }],
    [Command.SUBMIT_REVERSE_SEARCH]: [{ key: 'return', ctrl: false }],
    [Command.ACCEPT_SUGGESTION_REVERSE_SEARCH]: [{ key: 'tab' }],
    // Navigation
    [Command.NAVIGATION_UP]: [{ key: 'up', shift: false }],
    [Command.NAVIGATION_DOWN]: [{ key: 'down', shift: false }],
    // Navigation shortcuts appropriate for dialogs where we do not need to accept
    // text input.
    [Command.DIALOG_NAVIGATION_UP]: [
        { key: 'up', shift: false },
        { key: 'k', shift: false },
    ],
    [Command.DIALOG_NAVIGATION_DOWN]: [
        { key: 'down', shift: false },
        { key: 'j', shift: false },
    ],
    [Command.DIALOG_NEXT]: [{ key: 'tab', shift: false }],
    [Command.DIALOG_PREV]: [{ key: 'tab', shift: true }],
    // Suggestions & Completions
    [Command.ACCEPT_SUGGESTION]: [{ key: 'tab' }, { key: 'return', ctrl: false }],
    [Command.COMPLETION_UP]: [
        { key: 'up', shift: false },
        { key: 'p', shift: false, ctrl: true },
    ],
    [Command.COMPLETION_DOWN]: [
        { key: 'down', shift: false },
        { key: 'n', shift: false, ctrl: true },
    ],
    [Command.EXPAND_SUGGESTION]: [{ key: 'right' }],
    [Command.COLLAPSE_SUGGESTION]: [{ key: 'left' }],
    // Text Input
    // Must also exclude shift to allow shift+enter for newline
    [Command.SUBMIT]: [
        {
            key: 'return',
            shift: false,
            alt: false,
            ctrl: false,
            cmd: false,
        },
    ],
    [Command.NEWLINE]: [
        { key: 'return', ctrl: true },
        { key: 'return', cmd: true },
        { key: 'return', alt: true },
        { key: 'return', shift: true },
        { key: 'j', ctrl: true },
    ],
    [Command.OPEN_EXTERNAL_EDITOR]: [{ key: 'x', ctrl: true }],
    [Command.PASTE_CLIPBOARD]: [
        { key: 'v', ctrl: true },
        { key: 'v', cmd: true },
        { key: 'v', alt: true },
    ],
    // App Controls
    [Command.SHOW_ERROR_DETAILS]: [{ key: 'f12' }],
    [Command.SHOW_FULL_TODOS]: [{ key: 't', ctrl: true }],
    [Command.SHOW_IDE_CONTEXT_DETAIL]: [{ key: 'g', ctrl: true }],
    [Command.TOGGLE_MARKDOWN]: [{ key: 'm', alt: true }],
    [Command.TOGGLE_COPY_MODE]: [{ key: 's', ctrl: true }],
    [Command.TOGGLE_YOLO]: [{ key: 'y', ctrl: true }],
    [Command.CYCLE_APPROVAL_MODE]: [{ key: 'tab', shift: true }],
    [Command.TOGGLE_BACKGROUND_SHELL]: [{ key: 'b', ctrl: true }],
    [Command.TOGGLE_BACKGROUND_SHELL_LIST]: [{ key: 'l', ctrl: true }],
    [Command.KILL_BACKGROUND_SHELL]: [{ key: 'k', ctrl: true }],
    [Command.UNFOCUS_BACKGROUND_SHELL]: [{ key: 'tab', shift: true }],
    [Command.UNFOCUS_BACKGROUND_SHELL_LIST]: [{ key: 'tab', shift: false }],
    [Command.SHOW_BACKGROUND_SHELL_UNFOCUS_WARNING]: [
        { key: 'tab', shift: false },
    ],
    [Command.SHOW_SHELL_INPUT_UNFOCUS_WARNING]: [{ key: 'tab', shift: false }],
    [Command.BACKGROUND_SHELL_SELECT]: [{ key: 'return' }],
    [Command.BACKGROUND_SHELL_ESCAPE]: [{ key: 'escape' }],
    [Command.SHOW_MORE_LINES]: [{ key: 'o', ctrl: true }],
    [Command.EXPAND_PASTE]: [{ key: 'o', ctrl: true }],
    [Command.FOCUS_SHELL_INPUT]: [{ key: 'tab', shift: false }],
    [Command.UNFOCUS_SHELL_INPUT]: [{ key: 'tab', shift: true }],
    [Command.CLEAR_SCREEN]: [{ key: 'l', ctrl: true }],
    [Command.RESTART_APP]: [{ key: 'r' }],
    [Command.SUSPEND_APP]: [{ key: 'z', ctrl: true }],
};
/**
 * Presentation metadata for grouping commands in documentation or UI.
 */
export const commandCategories = [
    {
        title: 'Basic Controls',
        commands: [Command.RETURN, Command.ESCAPE, Command.QUIT, Command.EXIT],
    },
    {
        title: 'Cursor Movement',
        commands: [
            Command.HOME,
            Command.END,
            Command.MOVE_UP,
            Command.MOVE_DOWN,
            Command.MOVE_LEFT,
            Command.MOVE_RIGHT,
            Command.MOVE_WORD_LEFT,
            Command.MOVE_WORD_RIGHT,
        ],
    },
    {
        title: 'Editing',
        commands: [
            Command.KILL_LINE_RIGHT,
            Command.KILL_LINE_LEFT,
            Command.CLEAR_INPUT,
            Command.DELETE_WORD_BACKWARD,
            Command.DELETE_WORD_FORWARD,
            Command.DELETE_CHAR_LEFT,
            Command.DELETE_CHAR_RIGHT,
            Command.UNDO,
            Command.REDO,
        ],
    },
    {
        title: 'Scrolling',
        commands: [
            Command.SCROLL_UP,
            Command.SCROLL_DOWN,
            Command.SCROLL_HOME,
            Command.SCROLL_END,
            Command.PAGE_UP,
            Command.PAGE_DOWN,
        ],
    },
    {
        title: 'History & Search',
        commands: [
            Command.HISTORY_UP,
            Command.HISTORY_DOWN,
            Command.REVERSE_SEARCH,
            Command.SUBMIT_REVERSE_SEARCH,
            Command.ACCEPT_SUGGESTION_REVERSE_SEARCH,
            Command.REWIND,
        ],
    },
    {
        title: 'Navigation',
        commands: [
            Command.NAVIGATION_UP,
            Command.NAVIGATION_DOWN,
            Command.DIALOG_NAVIGATION_UP,
            Command.DIALOG_NAVIGATION_DOWN,
            Command.DIALOG_NEXT,
            Command.DIALOG_PREV,
        ],
    },
    {
        title: 'Suggestions & Completions',
        commands: [
            Command.ACCEPT_SUGGESTION,
            Command.COMPLETION_UP,
            Command.COMPLETION_DOWN,
            Command.EXPAND_SUGGESTION,
            Command.COLLAPSE_SUGGESTION,
        ],
    },
    {
        title: 'Text Input',
        commands: [
            Command.SUBMIT,
            Command.NEWLINE,
            Command.OPEN_EXTERNAL_EDITOR,
            Command.PASTE_CLIPBOARD,
        ],
    },
    {
        title: 'App Controls',
        commands: [
            Command.SHOW_ERROR_DETAILS,
            Command.SHOW_FULL_TODOS,
            Command.SHOW_IDE_CONTEXT_DETAIL,
            Command.TOGGLE_MARKDOWN,
            Command.TOGGLE_COPY_MODE,
            Command.TOGGLE_YOLO,
            Command.CYCLE_APPROVAL_MODE,
            Command.SHOW_MORE_LINES,
            Command.EXPAND_PASTE,
            Command.TOGGLE_BACKGROUND_SHELL,
            Command.TOGGLE_BACKGROUND_SHELL_LIST,
            Command.KILL_BACKGROUND_SHELL,
            Command.BACKGROUND_SHELL_SELECT,
            Command.BACKGROUND_SHELL_ESCAPE,
            Command.UNFOCUS_BACKGROUND_SHELL,
            Command.UNFOCUS_BACKGROUND_SHELL_LIST,
            Command.SHOW_BACKGROUND_SHELL_UNFOCUS_WARNING,
            Command.SHOW_SHELL_INPUT_UNFOCUS_WARNING,
            Command.FOCUS_SHELL_INPUT,
            Command.UNFOCUS_SHELL_INPUT,
            Command.CLEAR_SCREEN,
            Command.RESTART_APP,
            Command.SUSPEND_APP,
        ],
    },
];
/**
 * Human-readable descriptions for each command, used in docs/tooling.
 */
export const commandDescriptions = {
    // Basic Controls
    [Command.RETURN]: 'Confirm the current selection or choice.',
    [Command.ESCAPE]: 'Dismiss dialogs or cancel the current focus.',
    [Command.QUIT]: 'Cancel the current request or quit the CLI when input is empty.',
    [Command.EXIT]: 'Exit the CLI when the input buffer is empty.',
    // Cursor Movement
    [Command.HOME]: 'Move the cursor to the start of the line.',
    [Command.END]: 'Move the cursor to the end of the line.',
    [Command.MOVE_UP]: 'Move the cursor up one line.',
    [Command.MOVE_DOWN]: 'Move the cursor down one line.',
    [Command.MOVE_LEFT]: 'Move the cursor one character to the left.',
    [Command.MOVE_RIGHT]: 'Move the cursor one character to the right.',
    [Command.MOVE_WORD_LEFT]: 'Move the cursor one word to the left.',
    [Command.MOVE_WORD_RIGHT]: 'Move the cursor one word to the right.',
    // Editing
    [Command.KILL_LINE_RIGHT]: 'Delete from the cursor to the end of the line.',
    [Command.KILL_LINE_LEFT]: 'Delete from the cursor to the start of the line.',
    [Command.CLEAR_INPUT]: 'Clear all text in the input field.',
    [Command.DELETE_WORD_BACKWARD]: 'Delete the previous word.',
    [Command.DELETE_WORD_FORWARD]: 'Delete the next word.',
    [Command.DELETE_CHAR_LEFT]: 'Delete the character to the left.',
    [Command.DELETE_CHAR_RIGHT]: 'Delete the character to the right.',
    [Command.UNDO]: 'Undo the most recent text edit.',
    [Command.REDO]: 'Redo the most recent undone text edit.',
    // Scrolling
    [Command.SCROLL_UP]: 'Scroll content up.',
    [Command.SCROLL_DOWN]: 'Scroll content down.',
    [Command.SCROLL_HOME]: 'Scroll to the top.',
    [Command.SCROLL_END]: 'Scroll to the bottom.',
    [Command.PAGE_UP]: 'Scroll up by one page.',
    [Command.PAGE_DOWN]: 'Scroll down by one page.',
    // History & Search
    [Command.HISTORY_UP]: 'Show the previous entry in history.',
    [Command.HISTORY_DOWN]: 'Show the next entry in history.',
    [Command.REVERSE_SEARCH]: 'Start reverse search through history.',
    [Command.SUBMIT_REVERSE_SEARCH]: 'Submit the selected reverse-search match.',
    [Command.ACCEPT_SUGGESTION_REVERSE_SEARCH]: 'Accept a suggestion while reverse searching.',
    [Command.REWIND]: 'Browse and rewind previous interactions.',
    // Navigation
    [Command.NAVIGATION_UP]: 'Move selection up in lists.',
    [Command.NAVIGATION_DOWN]: 'Move selection down in lists.',
    [Command.DIALOG_NAVIGATION_UP]: 'Move up within dialog options.',
    [Command.DIALOG_NAVIGATION_DOWN]: 'Move down within dialog options.',
    [Command.DIALOG_NEXT]: 'Move to the next item or question in a dialog.',
    [Command.DIALOG_PREV]: 'Move to the previous item or question in a dialog.',
    // Suggestions & Completions
    [Command.ACCEPT_SUGGESTION]: 'Accept the inline suggestion.',
    [Command.COMPLETION_UP]: 'Move to the previous completion option.',
    [Command.COMPLETION_DOWN]: 'Move to the next completion option.',
    [Command.EXPAND_SUGGESTION]: 'Expand an inline suggestion.',
    [Command.COLLAPSE_SUGGESTION]: 'Collapse an inline suggestion.',
    // Text Input
    [Command.SUBMIT]: 'Submit the current prompt.',
    [Command.NEWLINE]: 'Insert a newline without submitting.',
    [Command.OPEN_EXTERNAL_EDITOR]: 'Open the current prompt in an external editor.',
    [Command.PASTE_CLIPBOARD]: 'Paste from the clipboard.',
    // App Controls
    [Command.SHOW_ERROR_DETAILS]: 'Toggle detailed error information.',
    [Command.SHOW_FULL_TODOS]: 'Toggle the full TODO list.',
    [Command.SHOW_IDE_CONTEXT_DETAIL]: 'Show IDE context details.',
    [Command.TOGGLE_MARKDOWN]: 'Toggle Markdown rendering.',
    [Command.TOGGLE_COPY_MODE]: 'Toggle copy mode when in alternate buffer mode.',
    [Command.TOGGLE_YOLO]: 'Toggle YOLO (auto-approval) mode for tool calls.',
    [Command.CYCLE_APPROVAL_MODE]: 'Cycle through approval modes: default (prompt), auto_edit (auto-approve edits), and plan (read-only). Plan mode is skipped when the agent is busy.',
    [Command.SHOW_MORE_LINES]: 'Expand and collapse blocks of content when not in alternate buffer mode.',
    [Command.EXPAND_PASTE]: 'Expand or collapse a paste placeholder when cursor is over placeholder.',
    [Command.BACKGROUND_SHELL_SELECT]: 'Confirm selection in background shell list.',
    [Command.BACKGROUND_SHELL_ESCAPE]: 'Dismiss background shell list.',
    [Command.TOGGLE_BACKGROUND_SHELL]: 'Toggle current background shell visibility.',
    [Command.TOGGLE_BACKGROUND_SHELL_LIST]: 'Toggle background shell list.',
    [Command.KILL_BACKGROUND_SHELL]: 'Kill the active background shell.',
    [Command.UNFOCUS_BACKGROUND_SHELL]: 'Move focus from background shell to Gemini.',
    [Command.UNFOCUS_BACKGROUND_SHELL_LIST]: 'Move focus from background shell list to Gemini.',
    [Command.SHOW_BACKGROUND_SHELL_UNFOCUS_WARNING]: 'Show warning when trying to move focus away from background shell.',
    [Command.SHOW_SHELL_INPUT_UNFOCUS_WARNING]: 'Show warning when trying to move focus away from shell input.',
    [Command.FOCUS_SHELL_INPUT]: 'Move focus from Gemini to the active shell.',
    [Command.UNFOCUS_SHELL_INPUT]: 'Move focus from the shell back to Gemini.',
    [Command.CLEAR_SCREEN]: 'Clear the terminal screen and redraw the UI.',
    [Command.RESTART_APP]: 'Restart the application.',
    [Command.SUSPEND_APP]: 'Suspend the CLI and move it to the background.',
};
//# sourceMappingURL=keyBindings.js.map