
//    SPDX-License-Identifier: LGPL-3.0-or-later
/*
    This file is part of FlexiBLAS, a BLAS/LAPACK interface wrapper library.
    Copyright (C) 2013-2025 Martin Koehler

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 3 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/



#include <stdio.h>
#include <stdlib.h>
#include <complex.h>
#include <dlfcn.h>
#include "flexiblas_backend.h"

#include "profile_hook.h"
#include "cscutils/table.h"

extern void flexiblas_chain_clarf (char* side, Int * m, Int * n, float complex* v, Int * incv, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side);
void hook_clarf(char* side, Int * m, Int * n, float complex* v, Int * incv, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarf((void*) side, (void*) m, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->clarf.timings[0] += (helpTimeStop - helpTime);
    data->clarf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zstegr (char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_zstegr(char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zstegr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->zstegr.timings[0] += (helpTimeStop - helpTime);
    data->zstegr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sppsvx (char* fact, char* uplo, Int * n, Int * nrhs, float* ap, float* afp, char* equed, float* s, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed);
void hook_sppsvx(char* fact, char* uplo, Int * n, Int * nrhs, float* ap, float* afp, char* equed, float* s, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sppsvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) equed, (void*) s, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->sppsvx.timings[0] += (helpTimeStop - helpTime);
    data->sppsvx.calls[0]++;

    return ;
}



extern float flexiblas_chain_clantb (char* norm, char* uplo, char* diag, Int * n, Int * k, float complex* ab, Int * ldab, float* work, int len_norm, int len_uplo, int len_diag);
float hook_clantb(char* norm, char* uplo, char* diag, Int * n, Int * k, float complex* ab, Int * ldab, float* work, int len_norm, int len_uplo, int len_diag)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clantb((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) k, (void*) ab, (void*) ldab, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->clantb.timings[0] += (helpTimeStop - helpTime);
    data->clantb.calls[0]++;

    return v;
}



extern void flexiblas_chain_dorglq (Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dorglq(Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorglq((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorglq.timings[0] += (helpTimeStop - helpTime);
    data->dorglq.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbev_2stage (char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo);
void hook_chbev_2stage(char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbev_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbev_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb6 (Int * m1, Int * m2, Int * n, float complex* x1, Int * incx1, float complex* x2, Int * incx2, float complex* q1, Int * ldq1, float complex* q2, Int * ldq2, float complex* work, Int * lwork, Int * info);
void hook_cunbdb6(Int * m1, Int * m2, Int * n, float complex* x1, Int * incx1, float complex* x2, Int * incx2, float complex* q1, Int * ldq1, float complex* q2, Int * ldq2, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb6((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb6.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb6.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbgvx (char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsbgvx(char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbgvx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbgvx.timings[0] += (helpTimeStop - helpTime);
    data->dsbgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctpqrt2 (Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, Int * info);
void hook_ctpqrt2(Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpqrt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->ctpqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_strexc (char* compq, Int * n, float* t, Int * ldt, float* q, Int * ldq, Int * ifst, Int * ilst, float* work, Int * info, int len_compq);
void hook_strexc(char* compq, Int * n, float* t, Int * ldt, float* q, Int * ldq, Int * ifst, Int * ilst, float* work, Int * info, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strexc((void*) compq, (void*) n, (void*) t, (void*) ldt, (void*) q, (void*) ldq, (void*) ifst, (void*) ilst, (void*) work, (void*) info, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->strexc.timings[0] += (helpTimeStop - helpTime);
    data->strexc.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgesvdx (char* jobu, char* jobvt, char* range, Int * m, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double* u, Int * ldu, double* vt, Int * ldvt, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobvt, int len_range);
void hook_dgesvdx(char* jobu, char* jobvt, char* range, Int * m, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double* u, Int * ldu, double* vt, Int * ldvt, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobvt, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgesvdx((void*) jobu, (void*) jobvt, (void*) range, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) ns, (void*) s, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu, (int) len_jobvt, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->dgesvdx.timings[0] += (helpTimeStop - helpTime);
    data->dgesvdx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpbequ (char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, Int * info, int len_uplo);
void hook_cpbequ(char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpbequ((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpbequ.timings[0] += (helpTimeStop - helpTime);
    data->cpbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgetc2 (Int * n, float* a, Int * lda, Int * ipiv, Int * jpiv, Int * info);
void hook_sgetc2(Int * n, float* a, Int * lda, Int * ipiv, Int * jpiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgetc2((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) jpiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetc2.timings[0] += (helpTimeStop - helpTime);
    data->sgetc2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorm2l (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans);
void hook_sorm2l(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorm2l((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sorm2l.timings[0] += (helpTimeStop - helpTime);
    data->sorm2l.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunhr_col (Int * m, Int * n, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* d, Int * info);
void hook_cunhr_col(Int * m, Int * n, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunhr_col((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunhr_col.timings[0] += (helpTimeStop - helpTime);
    data->cunhr_col.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormlq (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dormlq(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormlq.timings[0] += (helpTimeStop - helpTime);
    data->dormlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarzt (char* direct, char* storev, Int * n, Int * k, float complex* v, Int * ldv, float complex* tau, float complex* t, Int * ldt, int len_direct, int len_storev);
void hook_clarzt(char* direct, char* storev, Int * n, Int * k, float complex* v, Int * ldv, float complex* tau, float complex* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarzt((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->clarzt.timings[0] += (helpTimeStop - helpTime);
    data->clarzt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqgb (Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed);
void hook_dlaqgb(Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqgb((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dlaqgb.timings[0] += (helpTimeStop - helpTime);
    data->dlaqgb.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyev (char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_ssyev(char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyev((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyev.timings[0] += (helpTimeStop - helpTime);
    data->ssyev.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpbcon (char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_cpbcon(char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpbcon((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpbcon.timings[0] += (helpTimeStop - helpTime);
    data->cpbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_cptsv (Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb, Int * info);
void hook_cptsv(Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cptsv((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cptsv.timings[0] += (helpTimeStop - helpTime);
    data->cptsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dggev3 (char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr);
void hook_dggev3(char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggev3((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->dggev3.timings[0] += (helpTimeStop - helpTime);
    data->dggev3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgsja (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double* tola, double* tolb, double* alpha, double* beta, double complex* u, Int * ldu, double complex* v, Int * ldv, double complex* q, Int * ldq, double complex* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_ztgsja(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double* tola, double* tolb, double* alpha, double* beta, double complex* u, Int * ldu, double complex* v, Int * ldv, double complex* q, Int * ldq, double complex* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgsja((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) ncycle, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->ztgsja.timings[0] += (helpTimeStop - helpTime);
    data->ztgsja.calls[0]++;

    return ;
}



extern void flexiblas_chain_clagtm (char* trans, Int * n, Int * nrhs, float* alpha, float complex* dl, float complex* d, float complex* du, float complex* x, Int * ldx, float* beta, float complex* b, Int * ldb, int len_trans);
void hook_clagtm(char* trans, Int * n, Int * nrhs, float* alpha, float complex* dl, float complex* d, float complex* du, float complex* x, Int * ldx, float* beta, float complex* b, Int * ldb, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clagtm((void*) trans, (void*) n, (void*) nrhs, (void*) alpha, (void*) dl, (void*) d, (void*) du, (void*) x, (void*) ldx, (void*) beta, (void*) b, (void*) ldb, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->clagtm.timings[0] += (helpTimeStop - helpTime);
    data->clagtm.calls[0]++;

    return ;
}



extern void flexiblas_chain_spttrs (Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb, Int * info);
void hook_spttrs(Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spttrs((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->spttrs.timings[0] += (helpTimeStop - helpTime);
    data->spttrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sppcon (char* uplo, Int * n, float* ap, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo);
void hook_sppcon(char* uplo, Int * n, float* ap, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sppcon((void*) uplo, (void*) n, (void*) ap, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sppcon.timings[0] += (helpTimeStop - helpTime);
    data->sppcon.calls[0]++;

    return ;
}



extern float flexiblas_chain_clantp (char* norm, char* uplo, char* diag, Int * n, float complex* ap, float* work, int len_norm, int len_uplo, int len_diag);
float hook_clantp(char* norm, char* uplo, char* diag, Int * n, float complex* ap, float* work, int len_norm, int len_uplo, int len_diag)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clantp((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->clantp.timings[0] += (helpTimeStop - helpTime);
    data->clantp.calls[0]++;

    return v;
}



extern void flexiblas_chain_zlaqr0 (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info);
void hook_zlaqr0(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqr0((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqr0.timings[0] += (helpTimeStop - helpTime);
    data->zlaqr0.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztpqrt (Int * m, Int * n, Int * l, Int * nb, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, double complex* work, Int * info);
void hook_ztpqrt(Int * m, Int * n, Int * l, Int * nb, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztpqrt((void*) m, (void*) n, (void*) l, (void*) nb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ztpqrt.timings[0] += (helpTimeStop - helpTime);
    data->ztpqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgerfs (char* trans, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_trans);
void hook_cgerfs(char* trans, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgerfs((void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgerfs.timings[0] += (helpTimeStop - helpTime);
    data->cgerfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlagv2 (double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* csl, double* snl, double* csr, double* snr);
void hook_dlagv2(double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* csl, double* snl, double* csr, double* snr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlagv2((void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) csl, (void*) snl, (void*) csr, (void*) snr);

    helpTimeStop = flexiblas_wtime();

    data->dlagv2.timings[0] += (helpTimeStop - helpTime);
    data->dlagv2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgexc (Int * wantq, Int * wantz, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * ifst, Int * ilst, Int * info);
void hook_ctgexc(Int * wantq, Int * wantz, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * ifst, Int * ilst, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgexc((void*) wantq, (void*) wantz, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ifst, (void*) ilst, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctgexc.timings[0] += (helpTimeStop - helpTime);
    data->ctgexc.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspgv (Int * itype, char* jobz, char* uplo, Int * n, double* ap, double* bp, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo);
void hook_dspgv(Int * itype, char* jobz, char* uplo, Int * n, double* ap, double* bp, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspgv((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspgv.timings[0] += (helpTimeStop - helpTime);
    data->dspgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqr3 (Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double* h, Int * ldh, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * ns, Int * nd, double* sr, double* si, double* v, Int * ldv, Int * nh, double* t, Int * ldt, Int * nv, double* wv, Int * ldwv, double* work, Int * lwork);
void hook_dlaqr3(Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double* h, Int * ldh, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * ns, Int * nd, double* sr, double* si, double* v, Int * ldv, Int * nh, double* t, Int * ldt, Int * nv, double* wv, Int * ldwv, double* work, Int * lwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqr3((void*) wantt, (void*) wantz, (void*) n, (void*) ktop, (void*) kbot, (void*) nw, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) sr, (void*) si, (void*) v, (void*) ldv, (void*) nh, (void*) t, (void*) ldt, (void*) nv, (void*) wv, (void*) ldwv, (void*) work, (void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr3.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrtri (char* uplo, char* diag, Int * n, double* a, Int * lda, Int * info, int len_uplo, int len_diag);
void hook_dtrtri(char* uplo, char* diag, Int * n, double* a, Int * lda, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrtri((void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtrtri.timings[0] += (helpTimeStop - helpTime);
    data->dtrtri.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilaenv2stage (Int * ispec, char* name, char* opts, Int * n1, Int * n2, Int * n3, Int * n4, int len_name, int len_opts);
int hook_ilaenv2stage(Int * ispec, char* name, char* opts, Int * n1, Int * n2, Int * n3, Int * n4, int len_name, int len_opts)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilaenv2stage((void*) ispec, (void*) name, (void*) opts, (void*) n1, (void*) n2, (void*) n3, (void*) n4, (int) len_name, (int) len_opts);

    helpTimeStop = flexiblas_wtime();

    data->ilaenv2stage.timings[0] += (helpTimeStop - helpTime);
    data->ilaenv2stage.calls[0]++;

    return v;
}



extern void flexiblas_chain_dgbsv (Int * n, Int * kl, Int * ku, Int * nrhs, double* ab, Int * ldab, Int * ipiv, double* b, Int * ldb, Int * info);
void hook_dgbsv(Int * n, Int * kl, Int * ku, Int * nrhs, double* ab, Int * ldab, Int * ipiv, double* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbsv((void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbsv.timings[0] += (helpTimeStop - helpTime);
    data->dgbsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgtrfs (char* trans, Int * n, Int * nrhs, double complex* dl, double complex* d, double complex* du, double complex* dlf, double complex* df, double complex* duf, double complex* du2, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_trans);
void hook_zgtrfs(char* trans, Int * n, Int * nrhs, double complex* dl, double complex* d, double complex* du, double complex* dlf, double complex* df, double complex* duf, double complex* du2, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgtrfs((void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->zgtrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbtrf (char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, Int * info, int len_uplo);
void hook_zpbtrf(char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbtrf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbtrf.timings[0] += (helpTimeStop - helpTime);
    data->zpbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasr (char* side, char* pivot, char* direct, Int * m, Int * n, double* c, double* s, double* a, Int * lda, int len_side, int len_pivot, int len_direct);
void hook_dlasr(char* side, char* pivot, char* direct, Int * m, Int * n, double* c, double* s, double* a, Int * lda, int len_side, int len_pivot, int len_direct)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasr((void*) side, (void*) pivot, (void*) direct, (void*) m, (void*) n, (void*) c, (void*) s, (void*) a, (void*) lda, (int) len_side, (int) len_pivot, (int) len_direct);

    helpTimeStop = flexiblas_wtime();

    data->dlasr.timings[0] += (helpTimeStop - helpTime);
    data->dlasr.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaswlq (Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info);
void hook_slaswlq(Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaswlq((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaswlq.timings[0] += (helpTimeStop - helpTime);
    data->slaswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrs_aa (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytrs_aa(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrs_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_csyequb (char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, float complex* work, Int * info, int len_uplo);
void hook_csyequb(char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csyequb((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csyequb.timings[0] += (helpTimeStop - helpTime);
    data->csyequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlahef_rk (char* uplo, Int * n, Int * nb, Int * kb, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* w, Int * ldw, Int * info, int len_uplo);
void hook_zlahef_rk(char* uplo, Int * n, Int * nb, Int * kb, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlahef_rk((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlahef_rk.timings[0] += (helpTimeStop - helpTime);
    data->zlahef_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrd_2stage (char* vect, char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, float complex* hous2, Int * lhous2, float complex* work, Int * lwork, Int * info, int len_vect, int len_uplo);
void hook_chetrd_2stage(char* vect, char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, float complex* hous2, Int * lhous2, float complex* work, Int * lwork, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrd_2stage((void*) vect, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) hous2, (void*) lhous2, (void*) work, (void*) lwork, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlasr (char* side, char* pivot, char* direct, Int * m, Int * n, double* c, double* s, double complex* a, Int * lda, int len_side, int len_pivot, int len_direct);
void hook_zlasr(char* side, char* pivot, char* direct, Int * m, Int * n, double* c, double* s, double complex* a, Int * lda, int len_side, int len_pivot, int len_direct)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlasr((void*) side, (void*) pivot, (void*) direct, (void*) m, (void*) n, (void*) c, (void*) s, (void*) a, (void*) lda, (int) len_side, (int) len_pivot, (int) len_direct);

    helpTimeStop = flexiblas_wtime();

    data->zlasr.timings[0] += (helpTimeStop - helpTime);
    data->zlasr.calls[0]++;

    return ;
}



extern void flexiblas_chain_strtrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_strtrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strtrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->strtrs.timings[0] += (helpTimeStop - helpTime);
    data->strtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsptrf (char* uplo, Int * n, double* ap, Int * ipiv, Int * info, int len_uplo);
void hook_dsptrf(char* uplo, Int * n, double* ap, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsptrf((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsptrf.timings[0] += (helpTimeStop - helpTime);
    data->dsptrf.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlantr (char* norm, char* uplo, char* diag, Int * m, Int * n, double* a, Int * lda, double* work, int len_norm, int len_uplo, int len_diag);
double hook_dlantr(char* norm, char* uplo, char* diag, Int * m, Int * n, double* a, Int * lda, double* work, int len_norm, int len_uplo, int len_diag)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlantr((void*) norm, (void*) uplo, (void*) diag, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dlantr.timings[0] += (helpTimeStop - helpTime);
    data->dlantr.calls[0]++;

    return v;
}



extern void flexiblas_chain_chetrs2 (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * info, int len_uplo);
void hook_chetrs2(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrs2((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrs2.timings[0] += (helpTimeStop - helpTime);
    data->chetrs2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgtsvx (char* fact, char* trans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* dlf, float* df, float* duf, float* du2, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_trans);
void hook_sgtsvx(char* fact, char* trans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* dlf, float* df, float* duf, float* du2, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgtsvx((void*) fact, (void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->sgtsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgetc2 (Int * n, double complex* a, Int * lda, Int * ipiv, Int * jpiv, Int * info);
void hook_zgetc2(Int * n, double complex* a, Int * lda, Int * ipiv, Int * jpiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetc2((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) jpiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgetc2.timings[0] += (helpTimeStop - helpTime);
    data->zgetc2.calls[0]++;

    return ;
}



extern void flexiblas_chain_stgsen (Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* q, Int * ldq, float* z, Int * ldz, Int * m, float* pl, float* pr, float* dif, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info);
void hook_stgsen(Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* q, Int * ldq, float* z, Int * ldz, Int * m, float* pl, float* pr, float* dif, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stgsen((void*) ijob, (void*) wantq, (void*) wantz, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) m, (void*) pl, (void*) pr, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stgsen.timings[0] += (helpTimeStop - helpTime);
    data->stgsen.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztptrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ztptrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztptrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztptrs.timings[0] += (helpTimeStop - helpTime);
    data->ztptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytf2 (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_zsytf2(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytf2.timings[0] += (helpTimeStop - helpTime);
    data->zsytf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqrt2 (Int * m, Int * n, float* a, Int * lda, float* t, Int * ldt, Int * info);
void hook_sgeqrt2(Int * m, Int * n, float* a, Int * lda, float* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqrt2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtpqrt2 (Int * m, Int * n, Int * l, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, Int * info);
void hook_dtpqrt2(Int * m, Int * n, Int * l, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtpqrt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->dtpqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbgvx (char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_ssbgvx(char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbgvx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbgvx.timings[0] += (helpTimeStop - helpTime);
    data->ssbgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqsy (char* uplo, Int * n, double* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_dlaqsy(char* uplo, Int * n, double* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqsy((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsy.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorcsd2by1 (char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* u1, Int * ldu1, float* u2, Int * ldu2, float* v1t, Int * ldv1t, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t);
void hook_sorcsd2by1(char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* u1, Int * ldu1, float* u2, Int * ldu2, float* v1t, Int * ldv1t, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorcsd2by1((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t);

    helpTimeStop = flexiblas_wtime();

    data->sorcsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->sorcsd2by1.calls[0]++;

    return ;
}



extern void flexiblas_chain_sptsvx (char* fact, Int * n, Int * nrhs, float* d, float* e, float* df, float* ef, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * info, int len_fact);
void hook_sptsvx(char* fact, Int * n, Int * nrhs, float* d, float* e, float* df, float* ef, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * info, int len_fact)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sptsvx((void*) fact, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) info, (int) len_fact);

    helpTimeStop = flexiblas_wtime();

    data->sptsvx.timings[0] += (helpTimeStop - helpTime);
    data->sptsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmbr (char* vect, char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_vect, int len_side, int len_trans);
void hook_zunmbr(char* vect, char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_vect, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmbr((void*) vect, (void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_vect, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmbr.timings[0] += (helpTimeStop - helpTime);
    data->zunmbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbtf2 (char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, Int * info, int len_uplo);
void hook_zpbtf2(char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbtf2((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbtf2.timings[0] += (helpTimeStop - helpTime);
    data->zpbtf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgtcon (char* norm, Int * n, double complex* dl, double complex* d, double complex* du, double complex* du2, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_norm);
void hook_zgtcon(char* norm, Int * n, double complex* dl, double complex* d, double complex* du, double complex* du2, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgtcon((void*) norm, (void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->zgtcon.timings[0] += (helpTimeStop - helpTime);
    data->zgtcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_spftrs (char* transr, char* uplo, Int * n, Int * nrhs, float* a, float* b, Int * ldb, Int * info, int len_transr, int len_uplo);
void hook_spftrs(char* transr, char* uplo, Int * n, Int * nrhs, float* a, float* b, Int * ldb, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spftrs((void*) transr, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) b, (void*) ldb, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spftrs.timings[0] += (helpTimeStop - helpTime);
    data->spftrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqr2 (Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double* h, Int * ldh, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * ns, Int * nd, double* sr, double* si, double* v, Int * ldv, Int * nh, double* t, Int * ldt, Int * nv, double* wv, Int * ldwv, double* work, Int * lwork);
void hook_dlaqr2(Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double* h, Int * ldh, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * ns, Int * nd, double* sr, double* si, double* v, Int * ldv, Int * nh, double* t, Int * ldt, Int * nv, double* wv, Int * ldwv, double* work, Int * lwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqr2((void*) wantt, (void*) wantz, (void*) n, (void*) ktop, (void*) kbot, (void*) nw, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) sr, (void*) si, (void*) v, (void*) ldv, (void*) nh, (void*) t, (void*) ldt, (void*) nv, (void*) wv, (void*) ldwv, (void*) work, (void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr2.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr2.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlanhe (char* norm, char* uplo, Int * n, double complex* a, Int * lda, double* work, int len_norm, int len_uplo);
double hook_zlanhe(char* norm, char* uplo, Int * n, double complex* a, Int * lda, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlanhe((void*) norm, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlanhe.timings[0] += (helpTimeStop - helpTime);
    data->zlanhe.calls[0]++;

    return v;
}



extern void flexiblas_chain_dtpmqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, double* v, Int * ldv, double* t, Int * ldt, double* a, Int * lda, double* b, Int * ldb, double* work, Int * info, int len_side, int len_trans);
void hook_dtpmqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, double* v, Int * ldv, double* t, Int * ldt, double* a, Int * lda, double* b, Int * ldb, double* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtpmqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dtpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->dtpmqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbtrf (Int * m, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, Int * ipiv, Int * info);
void hook_sgbtrf(Int * m, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbtrf((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->sgbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgehrd (Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgehrd(Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgehrd((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgehrd.timings[0] += (helpTimeStop - helpTime);
    data->sgehrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgerq2 (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info);
void hook_cgerq2(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgerq2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgerq2.timings[0] += (helpTimeStop - helpTime);
    data->cgerq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspsvx (char* fact, char* uplo, Int * n, Int * nrhs, double* ap, double* afp, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo);
void hook_dspsvx(char* fact, char* uplo, Int * n, Int * nrhs, double* ap, double* afp, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspsvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspsvx.timings[0] += (helpTimeStop - helpTime);
    data->dspsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpotf2 (char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo);
void hook_zpotf2(char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpotf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpotf2.timings[0] += (helpTimeStop - helpTime);
    data->zpotf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbequb (Int * m, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_sgbequb(Int * m, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbequb((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbequb.timings[0] += (helpTimeStop - helpTime);
    data->sgbequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgsvj0 (char* jobv, Int * m, Int * n, float* a, Int * lda, float* d, float* sva, Int * mv, float* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float* work, Int * lwork, Int * info, int len_jobv);
void hook_sgsvj0(char* jobv, Int * m, Int * n, float* a, Int * lda, float* d, float* sva, Int * mv, float* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float* work, Int * lwork, Int * info, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgsvj0((void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) eps, (void*) sfmin, (void*) tol, (void*) nsweep, (void*) work, (void*) lwork, (void*) info, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->sgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->sgsvj0.calls[0]++;

    return ;
}



extern float flexiblas_chain_slantp (char* norm, char* uplo, char* diag, Int * n, float* ap, float* work, int len_norm, int len_uplo, int len_diag);
float hook_slantp(char* norm, char* uplo, char* diag, Int * n, float* ap, float* work, int len_norm, int len_uplo, int len_diag)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slantp((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->slantp.timings[0] += (helpTimeStop - helpTime);
    data->slantp.calls[0]++;

    return v;
}



extern void flexiblas_chain_cgghd3 (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* work, Int * lwork, Int * info, int len_compq, int len_compz);
void hook_cgghd3(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* work, Int * lwork, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgghd3((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->cgghd3.timings[0] += (helpTimeStop - helpTime);
    data->cgghd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrf_aa_2stage (char* uplo, Int * n, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetrf_aa_2stage(char* uplo, Int * n, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrf_aa_2stage((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhetrf_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztptri (char* uplo, char* diag, Int * n, double complex* ap, Int * info, int len_uplo, int len_diag);
void hook_ztptri(char* uplo, char* diag, Int * n, double complex* ap, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztptri((void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztptri.timings[0] += (helpTimeStop - helpTime);
    data->ztptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorglq (Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sorglq(Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorglq((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorglq.timings[0] += (helpTimeStop - helpTime);
    data->sorglq.calls[0]++;

    return ;
}



extern void flexiblas_chain_csrscl (Int * n, float* sa, float complex* sx, Int * incx);
void hook_csrscl(Int * n, float* sa, float complex* sx, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csrscl((void*) n, (void*) sa, (void*) sx, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->csrscl.timings[0] += (helpTimeStop - helpTime);
    data->csrscl.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctprfb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_ctprfb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctprfb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) l, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->ctprfb.timings[0] += (helpTimeStop - helpTime);
    data->ctprfb.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheequb (char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, float complex* work, Int * info, int len_uplo);
void hook_cheequb(char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheequb((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheequb.timings[0] += (helpTimeStop - helpTime);
    data->cheequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_stpttf (char* transr, char* uplo, Int * n, float* ap, float* arf, Int * info, int len_transr, int len_uplo);
void hook_stpttf(char* transr, char* uplo, Int * n, float* ap, float* arf, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stpttf((void*) transr, (void*) uplo, (void*) n, (void*) ap, (void*) arf, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->stpttf.timings[0] += (helpTimeStop - helpTime);
    data->stpttf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorbdb2 (Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* work, Int * lwork, Int * info);
void hook_dorbdb2(Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorbdb2((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb2.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb2.calls[0]++;

    return ;
}



extern void flexiblas_chain_strsyl3 (char* trana, char* tranb, Int * isgn, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* scale, Int * iwork, Int * liwork, float* swork, Int * ldswork, Int * info, int len_trana, int len_tranb);
void hook_strsyl3(char* trana, char* tranb, Int * isgn, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* scale, Int * iwork, Int * liwork, float* swork, Int * ldswork, Int * info, int len_trana, int len_tranb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strsyl3((void*) trana, (void*) tranb, (void*) isgn, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) scale, (void*) iwork, (void*) liwork, (void*) swork, (void*) ldswork, (void*) info, (int) len_trana, (int) len_tranb);

    helpTimeStop = flexiblas_wtime();

    data->strsyl3.timings[0] += (helpTimeStop - helpTime);
    data->strsyl3.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeevx (char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, float* a, Int * lda, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * ilo, Int * ihi, float* scale, float* abnrm, float* rconde, float* rcondv, float* work, Int * lwork, Int * iwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense);
void hook_sgeevx(char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, float* a, Int * lda, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * ilo, Int * ihi, float* scale, float* abnrm, float* rconde, float* rcondv, float* work, Int * lwork, Int * iwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeevx((void*) balanc, (void*) jobvl, (void*) jobvr, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) ilo, (void*) ihi, (void*) scale, (void*) abnrm, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_balanc, (int) len_jobvl, (int) len_jobvr, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->sgeevx.timings[0] += (helpTimeStop - helpTime);
    data->sgeevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlargv (Int * n, double complex* x, Int * incx, double complex* y, Int * incy, double* c, Int * incc);
void hook_zlargv(Int * n, double complex* x, Int * incx, double complex* y, Int * incy, double* c, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlargv((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) c, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->zlargv.timings[0] += (helpTimeStop - helpTime);
    data->zlargv.calls[0]++;

    return ;
}



extern float flexiblas_chain_clanhe (char* norm, char* uplo, Int * n, float complex* a, Int * lda, float* work, int len_norm, int len_uplo);
float hook_clanhe(char* norm, char* uplo, Int * n, float complex* a, Int * lda, float* work, int len_norm, int len_uplo)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clanhe((void*) norm, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clanhe.timings[0] += (helpTimeStop - helpTime);
    data->clanhe.calls[0]++;

    return v;
}



extern void flexiblas_chain_dsyequb (char* uplo, Int * n, double* a, Int * lda, double* s, double* scond, double* amax, double* work, Int * info, int len_uplo);
void hook_dsyequb(char* uplo, Int * n, double* a, Int * lda, double* s, double* scond, double* amax, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyequb((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyequb.timings[0] += (helpTimeStop - helpTime);
    data->dsyequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlahr2 (Int * n, Int * k, Int * nb, double* a, Int * lda, double* tau, double* t, Int * ldt, double* y, Int * ldy);
void hook_dlahr2(Int * n, Int * k, Int * nb, double* a, Int * lda, double* tau, double* t, Int * ldt, double* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlahr2((void*) n, (void*) k, (void*) nb, (void*) a, (void*) lda, (void*) tau, (void*) t, (void*) ldt, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->dlahr2.timings[0] += (helpTimeStop - helpTime);
    data->dlahr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sspgvd (Int * itype, char* jobz, char* uplo, Int * n, float* ap, float* bp, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_sspgvd(Int * itype, char* jobz, char* uplo, Int * n, float* ap, float* bp, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sspgvd((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sspgvd.timings[0] += (helpTimeStop - helpTime);
    data->sspgvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_clatsqr (Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info);
void hook_clatsqr(Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clatsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->clatsqr.timings[0] += (helpTimeStop - helpTime);
    data->clatsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zspmv (char* uplo, Int * n, double complex* alpha, double complex* ap, double complex* x, Int * incx, double complex* beta, double complex* y, Int * incy, int len_uplo);
void hook_zspmv(char* uplo, Int * n, double complex* alpha, double complex* ap, double complex* x, Int * incx, double complex* beta, double complex* y, Int * incy, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zspmv((void*) uplo, (void*) n, (void*) alpha, (void*) ap, (void*) x, (void*) incx, (void*) beta, (void*) y, (void*) incy, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zspmv.timings[0] += (helpTimeStop - helpTime);
    data->zspmv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpptri (char* uplo, Int * n, double* ap, Int * info, int len_uplo);
void hook_dpptri(char* uplo, Int * n, double* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpptri((void*) uplo, (void*) n, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpptri.timings[0] += (helpTimeStop - helpTime);
    data->dpptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgebrd (Int * m, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tauq, float complex* taup, float complex* work, Int * lwork, Int * info);
void hook_cgebrd(Int * m, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tauq, float complex* taup, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgebrd((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgebrd.timings[0] += (helpTimeStop - helpTime);
    data->cgebrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelqt3 (Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info);
void hook_cgelqt3(Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelqt3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->cgelqt3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbevd_2stage (char* jobz, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_ssbevd_2stage(char* jobz, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssbevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarre (char* range, Int * n, float* vl, float* vu, Int * il, Int * iu, float* d, float* e, float* e2, float* rtol1, float* rtol2, float* spltol, Int * nsplit, Int * isplit, Int * m, float* w, float* werr, float* wgap, Int * iblock, Int * indexw, float* gers, float* pivmin, float* work, Int * iwork, Int * info, int len_range);
void hook_slarre(char* range, Int * n, float* vl, float* vu, Int * il, Int * iu, float* d, float* e, float* e2, float* rtol1, float* rtol2, float* spltol, Int * nsplit, Int * isplit, Int * m, float* w, float* werr, float* wgap, Int * iblock, Int * indexw, float* gers, float* pivmin, float* work, Int * iwork, Int * info, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarre((void*) range, (void*) n, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) d, (void*) e, (void*) e2, (void*) rtol1, (void*) rtol2, (void*) spltol, (void*) nsplit, (void*) isplit, (void*) m, (void*) w, (void*) werr, (void*) wgap, (void*) iblock, (void*) indexw, (void*) gers, (void*) pivmin, (void*) work, (void*) iwork, (void*) info, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->slarre.timings[0] += (helpTimeStop - helpTime);
    data->slarre.calls[0]++;

    return ;
}



extern void flexiblas_chain_spbstf (char* uplo, Int * n, Int * kd, float* ab, Int * ldab, Int * info, int len_uplo);
void hook_spbstf(char* uplo, Int * n, Int * kd, float* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbstf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spbstf.timings[0] += (helpTimeStop - helpTime);
    data->spbstf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhegv_2stage (Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zhegv_2stage(Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhegv_2stage((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhegv_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhegv_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgesvj (char* joba, char* jobu, char* jobv, Int * m, Int * n, double complex* a, Int * lda, double* sva, Int * mv, double complex* v, Int * ldv, double complex* cwork, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobu, int len_jobv);
void hook_zgesvj(char* joba, char* jobu, char* jobv, Int * m, Int * n, double complex* a, Int * lda, double* sva, Int * mv, double complex* v, Int * ldv, double complex* cwork, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobu, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgesvj((void*) joba, (void*) jobu, (void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) cwork, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_joba, (int) len_jobu, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->zgesvj.timings[0] += (helpTimeStop - helpTime);
    data->zgesvj.calls[0]++;

    return ;
}



extern void flexiblas_chain_claed7 (Int * n, Int * cutpnt, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, float* d, float complex* q, Int * ldq, float* rho, Int * indxq, float* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float complex* work, float* rwork, Int * iwork, Int * info);
void hook_claed7(Int * n, Int * cutpnt, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, float* d, float complex* q, Int * ldq, float* rho, Int * indxq, float* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float complex* work, float* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claed7((void*) n, (void*) cutpnt, (void*) qsiz, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) d, (void*) q, (void*) ldq, (void*) rho, (void*) indxq, (void*) qstore, (void*) qptr, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) work, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claed7.timings[0] += (helpTimeStop - helpTime);
    data->claed7.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqps (Int * m, Int * n, Int * offset, Int * nb, Int * kb, float* a, Int * lda, Int * jpvt, float* tau, float* vn1, float* vn2, float* auxv, float* f, Int * ldf);
void hook_slaqps(Int * m, Int * n, Int * offset, Int * nb, Int * kb, float* a, Int * lda, Int * jpvt, float* tau, float* vn1, float* vn2, float* auxv, float* f, Int * ldf)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqps((void*) m, (void*) n, (void*) offset, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf);

    helpTimeStop = flexiblas_wtime();

    data->slaqps.timings[0] += (helpTimeStop - helpTime);
    data->slaqps.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrs_rook (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zsytrs_rook(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->zsytrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlags2 (Int * upper, double* a1, double* a2, double* a3, double* b1, double* b2, double* b3, double* csu, double* snu, double* csv, double* snv, double* csq, double* snq);
void hook_dlags2(Int * upper, double* a1, double* a2, double* a3, double* b1, double* b2, double* b3, double* csu, double* snu, double* csv, double* snv, double* csq, double* snq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlags2((void*) upper, (void*) a1, (void*) a2, (void*) a3, (void*) b1, (void*) b2, (void*) b3, (void*) csu, (void*) snu, (void*) csv, (void*) snv, (void*) csq, (void*) snq);

    helpTimeStop = flexiblas_wtime();

    data->dlags2.timings[0] += (helpTimeStop - helpTime);
    data->dlags2.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilazlc (Int * m, Int * n, double complex* a, Int * lda);
int hook_ilazlc(Int * m, Int * n, double complex* a, Int * lda)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilazlc((void*) m, (void*) n, (void*) a, (void*) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilazlc.timings[0] += (helpTimeStop - helpTime);
    data->ilazlc.calls[0]++;

    return v;
}



extern void flexiblas_chain_zgetrf (Int * m, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info);
void hook_zgetrf(Int * m, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetrf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgetrf.timings[0] += (helpTimeStop - helpTime);
    data->zgetrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zptrfs (char* uplo, Int * n, Int * nrhs, double* d, double complex* e, double* df, double complex* ef, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zptrfs(char* uplo, Int * n, Int * nrhs, double* d, double complex* e, double* df, double complex* ef, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zptrfs((void*) uplo, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zptrfs.timings[0] += (helpTimeStop - helpTime);
    data->zptrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqps (Int * m, Int * n, Int * offset, Int * nb, Int * kb, double* a, Int * lda, Int * jpvt, double* tau, double* vn1, double* vn2, double* auxv, double* f, Int * ldf);
void hook_dlaqps(Int * m, Int * n, Int * offset, Int * nb, Int * kb, double* a, Int * lda, Int * jpvt, double* tau, double* vn1, double* vn2, double* auxv, double* f, Int * ldf)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqps((void*) m, (void*) n, (void*) offset, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf);

    helpTimeStop = flexiblas_wtime();

    data->dlaqps.timings[0] += (helpTimeStop - helpTime);
    data->dlaqps.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgecon (char* norm, Int * n, double* a, Int * lda, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_norm);
void hook_dgecon(char* norm, Int * n, double* a, Int * lda, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgecon((void*) norm, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->dgecon.timings[0] += (helpTimeStop - helpTime);
    data->dgecon.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqrt2 (Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info);
void hook_cgeqrt2(Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqrt2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrttf (char* transr, char* uplo, Int * n, double* a, Int * lda, double* arf, Int * info, int len_transr, int len_uplo);
void hook_dtrttf(char* transr, char* uplo, Int * n, double* a, Int * lda, double* arf, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrttf((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) arf, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dtrttf.timings[0] += (helpTimeStop - helpTime);
    data->dtrttf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqz1 (Int * ilq, Int * ilz, Int * k, Int * istartm, Int * istopm, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, Int * nq, Int * qstart, double complex* q, Int * ldq, Int * nz, Int * zstart, double complex* z, Int * ldz);
void hook_zlaqz1(Int * ilq, Int * ilz, Int * k, Int * istartm, Int * istopm, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, Int * nq, Int * qstart, double complex* q, Int * ldq, Int * nz, Int * zstart, double complex* z, Int * ldz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqz1((void*) ilq, (void*) ilz, (void*) k, (void*) istartm, (void*) istopm, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) nq, (void*) qstart, (void*) q, (void*) ldq, (void*) nz, (void*) zstart, (void*) z, (void*) ldz);

    helpTimeStop = flexiblas_wtime();

    data->zlaqz1.timings[0] += (helpTimeStop - helpTime);
    data->zlaqz1.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarfb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_clarfb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarfb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->clarfb.timings[0] += (helpTimeStop - helpTime);
    data->clarfb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsysv_aa (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsysv_aa(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsysv_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_spftrf (char* transr, char* uplo, Int * n, float* a, Int * info, int len_transr, int len_uplo);
void hook_spftrf(char* transr, char* uplo, Int * n, float* a, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spftrf((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spftrf.timings[0] += (helpTimeStop - helpTime);
    data->spftrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytri_3 (char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytri_3(char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytri_3.timings[0] += (helpTimeStop - helpTime);
    data->dsytri_3.calls[0]++;

    return ;
}



extern int flexiblas_chain_sisnan (float* sin);
int hook_sisnan(float* sin)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_sisnan((void*) sin);

    helpTimeStop = flexiblas_wtime();

    data->sisnan.timings[0] += (helpTimeStop - helpTime);
    data->sisnan.calls[0]++;

    return v;
}



extern void flexiblas_chain_cpotri (char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_cpotri(char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpotri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpotri.timings[0] += (helpTimeStop - helpTime);
    data->cpotri.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqrt2 (Int * m, Int * n, double complex* a, Int * lda, double complex* t, Int * ldt, Int * info);
void hook_zgeqrt2(Int * m, Int * n, double complex* a, Int * lda, double complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqrt2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->zgeqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarrv (Int * n, double* vl, double* vu, double* d, double* l, double* pivmin, Int * isplit, Int * m, Int * dol, Int * dou, double* minrgp, double* rtol1, double* rtol2, double* w, double* werr, double* wgap, Int * iblock, Int * indexw, double* gers, double complex* z, Int * ldz, Int * isuppz, double* work, Int * iwork, Int * info);
void hook_zlarrv(Int * n, double* vl, double* vu, double* d, double* l, double* pivmin, Int * isplit, Int * m, Int * dol, Int * dou, double* minrgp, double* rtol1, double* rtol2, double* w, double* werr, double* wgap, Int * iblock, Int * indexw, double* gers, double complex* z, Int * ldz, Int * isuppz, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarrv((void*) n, (void*) vl, (void*) vu, (void*) d, (void*) l, (void*) pivmin, (void*) isplit, (void*) m, (void*) dol, (void*) dou, (void*) minrgp, (void*) rtol1, (void*) rtol2, (void*) w, (void*) werr, (void*) wgap, (void*) iblock, (void*) indexw, (void*) gers, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlarrv.timings[0] += (helpTimeStop - helpTime);
    data->zlarrv.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlamch (char* cmach, int len_cmach);
double hook_dlamch(char* cmach, int len_cmach)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlamch((void*) cmach, (int) len_cmach);

    helpTimeStop = flexiblas_wtime();

    data->dlamch.timings[0] += (helpTimeStop - helpTime);
    data->dlamch.calls[0]++;

    return v;
}



extern double flexiblas_chain_dlamc3 (double* a, double* b);
double hook_dlamc3(double* a, double* b)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlamc3((void*) a, (void*) b);

    helpTimeStop = flexiblas_wtime();

    data->dlamc3.timings[0] += (helpTimeStop - helpTime);
    data->dlamc3.calls[0]++;

    return v;
}



extern void flexiblas_chain_ssytri_3x (char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * nb, Int * info, int len_uplo);
void hook_ssytri_3x(char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri_3x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_3x.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_3x.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbgvx (char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float complex* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_chbgvx(char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float complex* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbgvx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbgvx.timings[0] += (helpTimeStop - helpTime);
    data->chbgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssygvd (Int * itype, char* jobz, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, float* w, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_ssygvd(Int * itype, char* jobz, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, float* w, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssygvd((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssygvd.timings[0] += (helpTimeStop - helpTime);
    data->ssygvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasdq (char* uplo, Int * sqre, Int * n, Int * ncvt, Int * nru, Int * ncc, double* d, double* e, double* vt, Int * ldvt, double* u, Int * ldu, double* c, Int * ldc, double* work, Int * info, int len_uplo);
void hook_dlasdq(char* uplo, Int * sqre, Int * n, Int * ncvt, Int * nru, Int * ncc, double* d, double* e, double* vt, Int * ldvt, double* u, Int * ldu, double* c, Int * ldc, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasdq((void*) uplo, (void*) sqre, (void*) n, (void*) ncvt, (void*) nru, (void*) ncc, (void*) d, (void*) e, (void*) vt, (void*) ldvt, (void*) u, (void*) ldu, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlasdq.timings[0] += (helpTimeStop - helpTime);
    data->dlasdq.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrexc (char* compq, Int * n, double complex* t, Int * ldt, double complex* q, Int * ldq, Int * ifst, Int * ilst, Int * info, int len_compq);
void hook_ztrexc(char* compq, Int * n, double complex* t, Int * ldt, double complex* q, Int * ldq, Int * ifst, Int * ilst, Int * info, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrexc((void*) compq, (void*) n, (void*) t, (void*) ldt, (void*) q, (void*) ldq, (void*) ifst, (void*) ilst, (void*) info, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->ztrexc.timings[0] += (helpTimeStop - helpTime);
    data->ztrexc.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztpmqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * info, int len_side, int len_trans);
void hook_ztpmqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztpmqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ztpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->ztpmqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbcon (char* norm, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_norm);
void hook_dgbcon(char* norm, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbcon((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->dgbcon.timings[0] += (helpTimeStop - helpTime);
    data->dgbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_sstedc (char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_compz);
void hook_sstedc(char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sstedc((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->sstedc.timings[0] += (helpTimeStop - helpTime);
    data->sstedc.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspsv (char* uplo, Int * n, Int * nrhs, double* ap, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dspsv(char* uplo, Int * n, Int * nrhs, double* ap, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspsv.timings[0] += (helpTimeStop - helpTime);
    data->dspsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed3 (Int * k, Int * n, Int * n1, float* d, float* q, Int * ldq, float* rho, float* dlambda, float* q2, Int * indx, Int * ctot, float* w, float* s, Int * info);
void hook_slaed3(Int * k, Int * n, Int * n1, float* d, float* q, Int * ldq, float* rho, float* dlambda, float* q2, Int * indx, Int * ctot, float* w, float* s, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed3((void*) k, (void*) n, (void*) n1, (void*) d, (void*) q, (void*) ldq, (void*) rho, (void*) dlambda, (void*) q2, (void*) indx, (void*) ctot, (void*) w, (void*) s, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed3.timings[0] += (helpTimeStop - helpTime);
    data->slaed3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd1 (Int * nl, Int * nr, Int * sqre, double* d, double* alpha, double* beta, double* u, Int * ldu, double* vt, Int * ldvt, Int * idxq, Int * iwork, double* work, Int * info);
void hook_dlasd1(Int * nl, Int * nr, Int * sqre, double* d, double* alpha, double* beta, double* u, Int * ldu, double* vt, Int * ldvt, Int * idxq, Int * iwork, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd1((void*) nl, (void*) nr, (void*) sqre, (void*) d, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) idxq, (void*) iwork, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd1.timings[0] += (helpTimeStop - helpTime);
    data->dlasd1.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggsvd3 (char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float* alpha, float* beta, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* q, Int * ldq, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_cggsvd3(char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float* alpha, float* beta, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* q, Int * ldq, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggsvd3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) n, (void*) p, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->cggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->cggsvd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dptts2 (Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb);
void hook_dptts2(Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dptts2((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->dptts2.timings[0] += (helpTimeStop - helpTime);
    data->dptts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarra (Int * n, float* d, float* e, float* e2, float* spltol, float* tnrm, Int * nsplit, Int * isplit, Int * info);
void hook_slarra(Int * n, float* d, float* e, float* e2, float* spltol, float* tnrm, Int * nsplit, Int * isplit, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarra((void*) n, (void*) d, (void*) e, (void*) e2, (void*) spltol, (void*) tnrm, (void*) nsplit, (void*) isplit, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slarra.timings[0] += (helpTimeStop - helpTime);
    data->slarra.calls[0]++;

    return ;
}



extern void flexiblas_chain_cstegr (char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_cstegr(char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cstegr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->cstegr.timings[0] += (helpTimeStop - helpTime);
    data->cstegr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbgv (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo);
void hook_ssbgv(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbgv((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbgv.timings[0] += (helpTimeStop - helpTime);
    data->ssbgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyevr_2stage (char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsyevr_2stage(char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyevr_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyevr_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_sggev3 (char* jobvl, char* jobvr, Int * n, float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* vl, Int * ldvl, float* vr, Int * ldvr, float* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr);
void hook_sggev3(char* jobvl, char* jobvr, Int * n, float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* vl, Int * ldvl, float* vr, Int * ldvr, float* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sggev3((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->sggev3.timings[0] += (helpTimeStop - helpTime);
    data->sggev3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlagtf (Int * n, double* a, double* lambda, double* b, double* c, double* tol, double* d, Int * in, Int * info);
void hook_dlagtf(Int * n, double* a, double* lambda, double* b, double* c, double* tol, double* d, Int * in, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlagtf((void*) n, (void*) a, (void*) lambda, (void*) b, (void*) c, (void*) tol, (void*) d, (void*) in, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlagtf.timings[0] += (helpTimeStop - helpTime);
    data->dlagtf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaein (Int * rightv, Int * noinit, Int * n, double complex* h, Int * ldh, double complex* w, double complex* v, double complex* b, Int * ldb, double* rwork, double* eps3, double* smlnum, Int * info);
void hook_zlaein(Int * rightv, Int * noinit, Int * n, double complex* h, Int * ldh, double complex* w, double complex* v, double complex* b, Int * ldb, double* rwork, double* eps3, double* smlnum, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaein((void*) rightv, (void*) noinit, (void*) n, (void*) h, (void*) ldh, (void*) w, (void*) v, (void*) b, (void*) ldb, (void*) rwork, (void*) eps3, (void*) smlnum, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaein.timings[0] += (helpTimeStop - helpTime);
    data->zlaein.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsygvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsygvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsygvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsygvx.timings[0] += (helpTimeStop - helpTime);
    data->dsygvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgbtrf (Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, Int * ipiv, Int * info);
void hook_zgbtrf(Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgbtrf((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->zgbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd5 (Int * i, double* d, double* z, double* delta, double* rho, double* dsigma, double* work);
void hook_dlasd5(Int * i, double* d, double* z, double* delta, double* rho, double* dsigma, double* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd5((void*) i, (void*) d, (void*) z, (void*) delta, (void*) rho, (void*) dsigma, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->dlasd5.timings[0] += (helpTimeStop - helpTime);
    data->dlasd5.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmr2 (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans);
void hook_cunmr2(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmr2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmr2.timings[0] += (helpTimeStop - helpTime);
    data->cunmr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgees (char* jobvs, char* sort, Int * select, Int * n, float* a, Int * lda, Int * sdim, float* wr, float* wi, float* vs, Int * ldvs, float* work, Int * lwork, Int * bwork, Int * info, int len_jobvs, int len_sort);
void hook_sgees(char* jobvs, char* sort, Int * select, Int * n, float* a, Int * lda, Int * sdim, float* wr, float* wi, float* vs, Int * ldvs, float* work, Int * lwork, Int * bwork, Int * info, int len_jobvs, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgees((void*) jobvs, (void*) sort, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) wr, (void*) wi, (void*) vs, (void*) ldvs, (void*) work, (void*) lwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->sgees.timings[0] += (helpTimeStop - helpTime);
    data->sgees.calls[0]++;

    return ;
}



extern void flexiblas_chain_stfsm (char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, float* alpha, float* a, float* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag);
void hook_stfsm(char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, float* alpha, float* a, float* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stfsm((void*) transr, (void*) side, (void*) uplo, (void*) trans, (void*) diag, (void*) m, (void*) n, (void*) alpha, (void*) a, (void*) b, (void*) ldb, (int) len_transr, (int) len_side, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stfsm.timings[0] += (helpTimeStop - helpTime);
    data->stfsm.calls[0]++;

    return ;
}



extern void flexiblas_chain_clahef_rook (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clahef_rook(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahef_rook((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clahef_rook.timings[0] += (helpTimeStop - helpTime);
    data->clahef_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeequ (Int * m, Int * n, double complex* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info);
void hook_zgeequ(Int * m, Int * n, double complex* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeequ((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeequ.timings[0] += (helpTimeStop - helpTime);
    data->zgeequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqsp (char* uplo, Int * n, float* ap, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_slaqsp(char* uplo, Int * n, float* ap, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqsp((void*) uplo, (void*) n, (void*) ap, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->slaqsp.timings[0] += (helpTimeStop - helpTime);
    data->slaqsp.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbsvx (char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, char* equed, double* s, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed);
void hook_dpbsvx(char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, char* equed, double* s, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbsvx((void*) fact, (void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) equed, (void*) s, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dpbsvx.timings[0] += (helpTimeStop - helpTime);
    data->dpbsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgemqr (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* t, Int * tsize, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_sgemqr(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* t, Int * tsize, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgemqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgemqr.timings[0] += (helpTimeStop - helpTime);
    data->sgemqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetrf2 (Int * m, Int * n, double* a, Int * lda, Int * ipiv, Int * info);
void hook_dgetrf2(Int * m, Int * n, double* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetrf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->dgetrf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasq1 (Int * n, float* d, float* e, float* work, Int * info);
void hook_slasq1(Int * n, float* d, float* e, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasq1((void*) n, (void*) d, (void*) e, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasq1.timings[0] += (helpTimeStop - helpTime);
    data->slasq1.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgttrs (char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_trans);
void hook_cgttrs(char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgttrs((void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgttrs.timings[0] += (helpTimeStop - helpTime);
    data->cgttrs.calls[0]++;

    return ;
}



extern float flexiblas_chain_slamch (char* cmach, int len_cmach);
float hook_slamch(char* cmach, int len_cmach)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slamch((void*) cmach, (int) len_cmach);

    helpTimeStop = flexiblas_wtime();

    data->slamch.timings[0] += (helpTimeStop - helpTime);
    data->slamch.calls[0]++;

    return v;
}



extern float flexiblas_chain_slamc3 (float* a, float* b);
float hook_slamc3(float* a, float* b)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slamc3((void*) a, (void*) b);

    helpTimeStop = flexiblas_wtime();

    data->slamc3.timings[0] += (helpTimeStop - helpTime);
    data->slamc3.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlartv (Int * n, double* x, Int * incx, double* y, Int * incy, double* c, double* s, Int * incc);
void hook_dlartv(Int * n, double* x, Int * incx, double* y, Int * incy, double* c, double* s, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlartv((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) c, (void*) s, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->dlartv.timings[0] += (helpTimeStop - helpTime);
    data->dlartv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrf_aa (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsytrf_aa(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrf_aa((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->zsytrf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlas2 (double* f, double* g, double* h, double* ssmin, double* ssmax);
void hook_dlas2(double* f, double* g, double* h, double* ssmin, double* ssmax)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlas2((void*) f, (void*) g, (void*) h, (void*) ssmin, (void*) ssmax);

    helpTimeStop = flexiblas_wtime();

    data->dlas2.timings[0] += (helpTimeStop - helpTime);
    data->dlas2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctpcon (char* norm, char* uplo, char* diag, Int * n, float complex* ap, float* rcond, float complex* work, float* rwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_ctpcon(char* norm, char* uplo, char* diag, Int * n, float complex* ap, float* rcond, float complex* work, float* rwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctpcon.timings[0] += (helpTimeStop - helpTime);
    data->ctpcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_sspgv (Int * itype, char* jobz, char* uplo, Int * n, float* ap, float* bp, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo);
void hook_sspgv(Int * itype, char* jobz, char* uplo, Int * n, float* ap, float* bp, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sspgv((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sspgv.timings[0] += (helpTimeStop - helpTime);
    data->sspgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqp3rk (Int * m, Int * n, Int * nrhs, Int * kmax, double* abstol, double* reltol, double* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double* tau, double* work, Int * lwork, Int * iwork, Int * info);
void hook_dgeqp3rk(Int * m, Int * n, Int * nrhs, Int * kmax, double* abstol, double* reltol, double* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double* tau, double* work, Int * lwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) kmax, (void*) abstol, (void*) reltol, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) work, (void*) lwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->dgeqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasy2 (Int * ltranl, Int * ltranr, Int * isgn, Int * n1, Int * n2, float* tl, Int * ldtl, float* tr, Int * ldtr, float* b, Int * ldb, float* scale, float* x, Int * ldx, float* xnorm, Int * info);
void hook_slasy2(Int * ltranl, Int * ltranr, Int * isgn, Int * n1, Int * n2, float* tl, Int * ldtl, float* tr, Int * ldtr, float* b, Int * ldb, float* scale, float* x, Int * ldx, float* xnorm, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasy2((void*) ltranl, (void*) ltranr, (void*) isgn, (void*) n1, (void*) n2, (void*) tl, (void*) ldtl, (void*) tr, (void*) ldtr, (void*) b, (void*) ldb, (void*) scale, (void*) x, (void*) ldx, (void*) xnorm, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasy2.timings[0] += (helpTimeStop - helpTime);
    data->slasy2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhesv (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhesv(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhesv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhesv.timings[0] += (helpTimeStop - helpTime);
    data->zhesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssysv_aa (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssysv_aa(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssysv_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_aa.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpotrf (char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo);
void hook_dpotrf(char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpotrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpotrf.timings[0] += (helpTimeStop - helpTime);
    data->dpotrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqrfp (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cgeqrfp(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqrfp((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrfp.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorcsd (char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, char* signs, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x12, Int * ldx12, double* x21, Int * ldx21, double* x22, Int * ldx22, double* theta, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* v2t, Int * ldv2t, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans, int len_signs);
void hook_dorcsd(char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, char* signs, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x12, Int * ldx12, double* x21, Int * ldx21, double* x22, Int * ldx22, double* theta, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* v2t, Int * ldv2t, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorcsd((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) jobv2t, (void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) v2t, (void*) ldv2t, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t, (int) len_jobv2t, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->dorcsd.timings[0] += (helpTimeStop - helpTime);
    data->dorcsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungrq (Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zungrq(Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungrq((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungrq.timings[0] += (helpTimeStop - helpTime);
    data->zungrq.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrs_aa (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrs_aa(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrs_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbsv (Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, Int * ipiv, float complex* b, Int * ldb, Int * info);
void hook_cgbsv(Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, Int * ipiv, float complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbsv((void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbsv.timings[0] += (helpTimeStop - helpTime);
    data->cgbsv.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlangb (char* norm, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* work, int len_norm);
double hook_zlangb(char* norm, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* work, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlangb((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->zlangb.timings[0] += (helpTimeStop - helpTime);
    data->zlangb.calls[0]++;

    return v;
}



extern void flexiblas_chain_ztrsyl3 (char* trana, char* tranb, Int * isgn, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double* scale, double* swork, Int * ldswork, Int * info, int len_trana, int len_tranb);
void hook_ztrsyl3(char* trana, char* tranb, Int * isgn, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double* scale, double* swork, Int * ldswork, Int * info, int len_trana, int len_tranb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrsyl3((void*) trana, (void*) tranb, (void*) isgn, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) scale, (void*) swork, (void*) ldswork, (void*) info, (int) len_trana, (int) len_tranb);

    helpTimeStop = flexiblas_wtime();

    data->ztrsyl3.timings[0] += (helpTimeStop - helpTime);
    data->ztrsyl3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhseqr (char* job, char* compz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info, int len_job, int len_compz);
void hook_zhseqr(char* job, char* compz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info, int len_job, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhseqr((void*) job, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_job, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->zhseqr.timings[0] += (helpTimeStop - helpTime);
    data->zhseqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasyf_rk (char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, float* e, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo);
void hook_slasyf_rk(char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, float* e, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasyf_rk((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_rk.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgtsqr (Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info);
void hook_sorgtsqr(Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgtsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgtsqr.timings[0] += (helpTimeStop - helpTime);
    data->sorgtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpotrf (char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_cpotrf(char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpotrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpotrf.timings[0] += (helpTimeStop - helpTime);
    data->cpotrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunhr_col (Int * m, Int * n, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* d, Int * info);
void hook_zunhr_col(Int * m, Int * n, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunhr_col((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zunhr_col.timings[0] += (helpTimeStop - helpTime);
    data->zunhr_col.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrf_aa_2stage (char* uplo, Int * n, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetrf_aa_2stage(char* uplo, Int * n, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrf_aa_2stage((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasv2 (double* f, double* g, double* h, double* ssmin, double* ssmax, double* snr, double* csr, double* snl, double* csl);
void hook_dlasv2(double* f, double* g, double* h, double* ssmin, double* ssmax, double* snr, double* csr, double* snl, double* csl)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasv2((void*) f, (void*) g, (void*) h, (void*) ssmin, (void*) ssmax, (void*) snr, (void*) csr, (void*) snl, (void*) csl);

    helpTimeStop = flexiblas_wtime();

    data->dlasv2.timings[0] += (helpTimeStop - helpTime);
    data->dlasv2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slatps (char* uplo, char* trans, char* diag, char* normin, Int * n, float* ap, float* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_slatps(char* uplo, char* trans, char* diag, char* normin, Int * n, float* ap, float* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slatps((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) ap, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->slatps.timings[0] += (helpTimeStop - helpTime);
    data->slatps.calls[0]++;

    return ;
}



extern void flexiblas_chain_checon (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo);
void hook_checon(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_checon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->checon.timings[0] += (helpTimeStop - helpTime);
    data->checon.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqrt (Int * m, Int * n, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * info);
void hook_zgeqrt(Int * m, Int * n, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqrt((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqrt.timings[0] += (helpTimeStop - helpTime);
    data->zgeqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zporfs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zporfs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zporfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zporfs.timings[0] += (helpTimeStop - helpTime);
    data->zporfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrs (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chetrs(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrs.timings[0] += (helpTimeStop - helpTime);
    data->chetrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormr2 (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_trans);
void hook_dormr2(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormr2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormr2.timings[0] += (helpTimeStop - helpTime);
    data->dormr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dopmtr (char* side, char* uplo, char* trans, Int * m, Int * n, double* ap, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_uplo, int len_trans);
void hook_dopmtr(char* side, char* uplo, char* trans, Int * m, Int * n, double* ap, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dopmtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) ap, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dopmtr.timings[0] += (helpTimeStop - helpTime);
    data->dopmtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbgv (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double complex* ab, Int * ldab, double complex* bb, Int * ldbb, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zhbgv(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double complex* ab, Int * ldab, double complex* bb, Int * ldbb, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbgv((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbgv.timings[0] += (helpTimeStop - helpTime);
    data->zhbgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmrz (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zunmrz(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmrz((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmrz.timings[0] += (helpTimeStop - helpTime);
    data->zunmrz.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhecon_3 (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo);
void hook_zhecon_3(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhecon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhecon_3.timings[0] += (helpTimeStop - helpTime);
    data->zhecon_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelqt3 (Int * m, Int * n, double* a, Int * lda, double* t, Int * ldt, Int * info);
void hook_dgelqt3(Int * m, Int * n, double* a, Int * lda, double* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelqt3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->dgelqt3.calls[0]++;

    return ;
}



extern void flexiblas_chain_stgsja (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, float* a, Int * lda, float* b, Int * ldb, float* tola, float* tolb, float* alpha, float* beta, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, float* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_stgsja(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, float* a, Int * lda, float* b, Int * ldb, float* tola, float* tolb, float* alpha, float* beta, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, float* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stgsja((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) ncycle, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->stgsja.timings[0] += (helpTimeStop - helpTime);
    data->stgsja.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlar2v (Int * n, double complex* x, double complex* y, double complex* z, Int * incx, double* c, double complex* s, Int * incc);
void hook_zlar2v(Int * n, double complex* x, double complex* y, double complex* z, Int * incx, double* c, double complex* s, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlar2v((void*) n, (void*) x, (void*) y, (void*) z, (void*) incx, (void*) c, (void*) s, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->zlar2v.timings[0] += (helpTimeStop - helpTime);
    data->zlar2v.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpotrs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zpotrs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpotrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpotrs.timings[0] += (helpTimeStop - helpTime);
    data->zpotrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgetrf (Int * m, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info);
void hook_cgetrf(Int * m, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgetrf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetrf.timings[0] += (helpTimeStop - helpTime);
    data->cgetrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormqr (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dormqr(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormqr.timings[0] += (helpTimeStop - helpTime);
    data->dormqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrtrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ztrtrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrtrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztrtrs.timings[0] += (helpTimeStop - helpTime);
    data->ztrtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpftrs (char* transr, char* uplo, Int * n, Int * nrhs, double complex* a, double complex* b, Int * ldb, Int * info, int len_transr, int len_uplo);
void hook_zpftrs(char* transr, char* uplo, Int * n, Int * nrhs, double complex* a, double complex* b, Int * ldb, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpftrs((void*) transr, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) b, (void*) ldb, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpftrs.timings[0] += (helpTimeStop - helpTime);
    data->zpftrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspev (char* jobz, char* uplo, Int * n, double* ap, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo);
void hook_dspev(char* jobz, char* uplo, Int * n, double* ap, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspev((void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspev.timings[0] += (helpTimeStop - helpTime);
    data->dspev.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqz3 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * ns, Int * nd, double* alphar, double* alphai, double* beta, double* qc, Int * ldqc, double* zc, Int * ldzc, double* work, Int * lwork, Int * rec, Int * info);
void hook_dlaqz3(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * ns, Int * nd, double* alphar, double* alphai, double* beta, double* qc, Int * ldqc, double* zc, Int * ldzc, double* work, Int * lwork, Int * rec, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqz3((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nw, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) alphar, (void*) alphai, (void*) beta, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) rec, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqz3.timings[0] += (helpTimeStop - helpTime);
    data->dlaqz3.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbrfs (char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_trans);
void hook_sgbrfs(char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbrfs((void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->sgbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrevc (char* side, char* howmny, Int * select, Int * n, double complex* t, Int * ldt, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * info, int len_side, int len_howmny);
void hook_ztrevc(char* side, char* howmny, Int * select, Int * n, double complex* t, Int * ldt, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrevc((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) rwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ztrevc.timings[0] += (helpTimeStop - helpTime);
    data->ztrevc.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctplqt (Int * m, Int * n, Int * l, Int * mb, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, float complex* work, Int * info);
void hook_ctplqt(Int * m, Int * n, Int * l, Int * mb, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctplqt((void*) m, (void*) n, (void*) l, (void*) mb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctplqt.timings[0] += (helpTimeStop - helpTime);
    data->ctplqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrs_aa_2stage (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zsytrs_aa_2stage(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrs_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zsytrs_aa_2stage.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlanhb (char* norm, char* uplo, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo);
double hook_zlanhb(char* norm, char* uplo, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlanhb((void*) norm, (void*) uplo, (void*) n, (void*) k, (void*) ab, (void*) ldab, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlanhb.timings[0] += (helpTimeStop - helpTime);
    data->zlanhb.calls[0]++;

    return v;
}



extern void flexiblas_chain_zgeesx (char* jobvs, char* sort, Int * select, char* sense, Int * n, double complex* a, Int * lda, Int * sdim, double complex* w, double complex* vs, Int * ldvs, double* rconde, double* rcondv, double complex* work, Int * lwork, double* rwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense);
void hook_zgeesx(char* jobvs, char* sort, Int * select, char* sense, Int * n, double complex* a, Int * lda, Int * sdim, double complex* w, double complex* vs, Int * ldvs, double* rconde, double* rcondv, double complex* work, Int * lwork, double* rwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeesx((void*) jobvs, (void*) sort, (void*) select, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) w, (void*) vs, (void*) ldvs, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) rwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->zgeesx.timings[0] += (helpTimeStop - helpTime);
    data->zgeesx.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo);
void hook_csytri(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri.timings[0] += (helpTimeStop - helpTime);
    data->csytri.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqp3rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float complex* a, Int * lda, Int * done, Int * kb, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float complex* tau, float* vn1, float* vn2, float complex* auxv, float complex* f, Int * ldf, Int * iwork, Int * info);
void hook_claqp3rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float complex* a, Int * lda, Int * done, Int * kb, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float complex* tau, float* vn1, float* vn2, float complex* auxv, float complex* f, Int * ldf, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) nb, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) done, (void*) kb, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->claqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssysvx (char* fact, char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * lwork, Int * iwork, Int * info, int len_fact, int len_uplo);
void hook_ssysvx(char* fact, char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * lwork, Int * iwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssysvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssysvx.timings[0] += (helpTimeStop - helpTime);
    data->ssysvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasy2 (Int * ltranl, Int * ltranr, Int * isgn, Int * n1, Int * n2, double* tl, Int * ldtl, double* tr, Int * ldtr, double* b, Int * ldb, double* scale, double* x, Int * ldx, double* xnorm, Int * info);
void hook_dlasy2(Int * ltranl, Int * ltranr, Int * isgn, Int * n1, Int * n2, double* tl, Int * ldtl, double* tr, Int * ldtr, double* b, Int * ldb, double* scale, double* x, Int * ldx, double* xnorm, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasy2((void*) ltranl, (void*) ltranr, (void*) isgn, (void*) n1, (void*) n2, (void*) tl, (void*) ldtl, (void*) tr, (void*) ldtr, (void*) b, (void*) ldb, (void*) scale, (void*) x, (void*) ldx, (void*) xnorm, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasy2.timings[0] += (helpTimeStop - helpTime);
    data->dlasy2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbrfs (char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* afb, Int * ldafb, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zpbrfs(char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* afb, Int * ldafb, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbrfs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbrfs.timings[0] += (helpTimeStop - helpTime);
    data->zpbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyevx (char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsyevx(char* jobz, char* range, char* uplo, Int * n, double* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyevx.timings[0] += (helpTimeStop - helpTime);
    data->dsyevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slanv2 (float* a, float* b, float* c, float* d, float* rt1r, float* rt1i, float* rt2r, float* rt2i, float* cs, float* sn);
void hook_slanv2(float* a, float* b, float* c, float* d, float* rt1r, float* rt1i, float* rt2r, float* rt2i, float* cs, float* sn)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slanv2((void*) a, (void*) b, (void*) c, (void*) d, (void*) rt1r, (void*) rt1i, (void*) rt2r, (void*) rt2i, (void*) cs, (void*) sn);

    helpTimeStop = flexiblas_wtime();

    data->slanv2.timings[0] += (helpTimeStop - helpTime);
    data->slanv2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpotrf2 (char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo);
void hook_dpotrf2(char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpotrf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpotrf2.timings[0] += (helpTimeStop - helpTime);
    data->dpotrf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarzb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_slarzb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarzb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) l, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->slarzb.timings[0] += (helpTimeStop - helpTime);
    data->slarzb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dbdsdc (char* uplo, char* compq, Int * n, double* d, double* e, double* u, Int * ldu, double* vt, Int * ldvt, double* q, Int * iq, double* work, Int * iwork, Int * info, int len_uplo, int len_compq);
void hook_dbdsdc(char* uplo, char* compq, Int * n, double* d, double* e, double* u, Int * ldu, double* vt, Int * ldvt, double* q, Int * iq, double* work, Int * iwork, Int * info, int len_uplo, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dbdsdc((void*) uplo, (void*) compq, (void*) n, (void*) d, (void*) e, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) q, (void*) iq, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->dbdsdc.timings[0] += (helpTimeStop - helpTime);
    data->dbdsdc.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasyf_rook (char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo);
void hook_slasyf_rook(char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasyf_rook((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgesvdq (char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, float* a, Int * lda, float* s, float* u, Int * ldu, float* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, float* work, Int * lwork, float* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv);
void hook_sgesvdq(char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, float* a, Int * lda, float* s, float* u, Int * ldu, float* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, float* work, Int * lwork, float* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgesvdq((void*) joba, (void*) jobp, (void*) jobr, (void*) jobu, (void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) numrank, (void*) iwork, (void*) liwork, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_joba, (int) len_jobp, (int) len_jobr, (int) len_jobu, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->sgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->sgesvdq.calls[0]++;

    return ;
}



extern void flexiblas_chain_stplqt2 (Int * m, Int * n, Int * l, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, Int * info);
void hook_stplqt2(Int * m, Int * n, Int * l, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stplqt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stplqt2.timings[0] += (helpTimeStop - helpTime);
    data->stplqt2.calls[0]++;

    return ;
}



extern double complex flexiblas_chain_zladiv (void *retvalue, double complex* x, double complex* y);
double complex hook_zladiv(double complex* x, double complex* y)
{
    double complex v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zladiv( (void*) &v , (void*) x, (void*) y);

    helpTimeStop = flexiblas_wtime();

    data->zladiv.timings[0] += (helpTimeStop - helpTime);
    data->zladiv.calls[0]++;

    return v;
}



extern void flexiblas_chain_ctpmlqt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * mb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * info, int len_side, int len_trans);
void hook_ctpmlqt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * mb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpmlqt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) mb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ctpmlqt.timings[0] += (helpTimeStop - helpTime);
    data->ctpmlqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasq6 (Int * i0, Int * n0, float* z, Int * pp, float* dmin, float* dmin1, float* dmin2, float* dn, float* dnm1, float* dnm2);
void hook_slasq6(Int * i0, Int * n0, float* z, Int * pp, float* dmin, float* dmin1, float* dmin2, float* dn, float* dnm1, float* dnm2)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasq6((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dnm1, (void*) dnm2);

    helpTimeStop = flexiblas_wtime();

    data->slasq6.timings[0] += (helpTimeStop - helpTime);
    data->slasq6.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgglse (Int * m, Int * n, Int * p, double* a, Int * lda, double* b, Int * ldb, double* c, double* d, double* x, double* work, Int * lwork, Int * info);
void hook_dgglse(Int * m, Int * n, Int * p, double* a, Int * lda, double* b, Int * ldb, double* c, double* d, double* x, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgglse((void*) m, (void*) n, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) d, (void*) x, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgglse.timings[0] += (helpTimeStop - helpTime);
    data->dgglse.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunml2 (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans);
void hook_cunml2(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunml2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunml2.timings[0] += (helpTimeStop - helpTime);
    data->cunml2.calls[0]++;

    return ;
}



extern int flexiblas_chain_iparmq (Int * ispec, char* name, char* opts, Int * n, Int * ilo, Int * ihi, Int * lwork, int len_name, int len_opts);
int hook_iparmq(Int * ispec, char* name, char* opts, Int * n, Int * ilo, Int * ihi, Int * lwork, int len_name, int len_opts)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_iparmq((void*) ispec, (void*) name, (void*) opts, (void*) n, (void*) ilo, (void*) ihi, (void*) lwork, (int) len_name, (int) len_opts);

    helpTimeStop = flexiblas_wtime();

    data->iparmq.timings[0] += (helpTimeStop - helpTime);
    data->iparmq.calls[0]++;

    return v;
}



extern void flexiblas_chain_sgebrd (Int * m, Int * n, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* work, Int * lwork, Int * info);
void hook_sgebrd(Int * m, Int * n, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgebrd((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgebrd.timings[0] += (helpTimeStop - helpTime);
    data->sgebrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggbak (char* job, char* side, Int * n, Int * ilo, Int * ihi, double* lscale, double* rscale, Int * m, double complex* v, Int * ldv, Int * info, int len_job, int len_side);
void hook_zggbak(char* job, char* side, Int * n, Int * ilo, Int * ihi, double* lscale, double* rscale, Int * m, double complex* v, Int * ldv, Int * info, int len_job, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggbak((void*) job, (void*) side, (void*) n, (void*) ilo, (void*) ihi, (void*) lscale, (void*) rscale, (void*) m, (void*) v, (void*) ldv, (void*) info, (int) len_job, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->zggbak.timings[0] += (helpTimeStop - helpTime);
    data->zggbak.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlanhs (char* norm, Int * n, double* a, Int * lda, double* work, int len_norm);
double hook_dlanhs(char* norm, Int * n, double* a, Int * lda, double* work, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlanhs((void*) norm, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->dlanhs.timings[0] += (helpTimeStop - helpTime);
    data->dlanhs.calls[0]++;

    return v;
}



extern void flexiblas_chain_dgghd3 (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_compq, int len_compz);
void hook_dgghd3(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgghd3((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->dgghd3.timings[0] += (helpTimeStop - helpTime);
    data->dgghd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunbdb4 (Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* phantom, double complex* work, Int * lwork, Int * info);
void hook_zunbdb4(Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* phantom, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunbdb4((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) phantom, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zunbdb4.timings[0] += (helpTimeStop - helpTime);
    data->zunbdb4.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpprfs (char* uplo, Int * n, Int * nrhs, double* ap, double* afp, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dpprfs(char* uplo, Int * n, Int * nrhs, double* ap, double* afp, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpprfs.timings[0] += (helpTimeStop - helpTime);
    data->dpprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqrf (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgeqrf(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqrf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrf.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slag2 (float* a, Int * lda, float* b, Int * ldb, float* safmin, float* scale1, float* scale2, float* wr1, float* wr2, float* wi);
void hook_slag2(float* a, Int * lda, float* b, Int * ldb, float* safmin, float* scale1, float* scale2, float* wr1, float* wr2, float* wi)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slag2((void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) safmin, (void*) scale1, (void*) scale2, (void*) wr1, (void*) wr2, (void*) wi);

    helpTimeStop = flexiblas_wtime();

    data->slag2.timings[0] += (helpTimeStop - helpTime);
    data->slag2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelqf (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dgelqf(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelqf.timings[0] += (helpTimeStop - helpTime);
    data->dgelqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrttp (char* uplo, Int * n, float complex* a, Int * lda, float complex* ap, Int * info, int len_uplo);
void hook_ctrttp(char* uplo, Int * n, float complex* a, Int * lda, float complex* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrttp((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ctrttp.timings[0] += (helpTimeStop - helpTime);
    data->ctrttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrf (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrf(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlansb (char* norm, char* uplo, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo);
double hook_zlansb(char* norm, char* uplo, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlansb((void*) norm, (void*) uplo, (void*) n, (void*) k, (void*) ab, (void*) ldab, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlansb.timings[0] += (helpTimeStop - helpTime);
    data->zlansb.calls[0]++;

    return v;
}



extern void flexiblas_chain_zlarnv (Int * idist, Int * iseed, Int * n, double complex* x);
void hook_zlarnv(Int * idist, Int * iseed, Int * n, double complex* x)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarnv((void*) idist, (void*) iseed, (void*) n, (void*) x);

    helpTimeStop = flexiblas_wtime();

    data->zlarnv.timings[0] += (helpTimeStop - helpTime);
    data->zlarnv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpocon (char* uplo, Int * n, double complex* a, Int * lda, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zpocon(char* uplo, Int * n, double complex* a, Int * lda, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpocon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpocon.timings[0] += (helpTimeStop - helpTime);
    data->zpocon.calls[0]++;

    return ;
}



extern void flexiblas_chain_spbrfs (char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo);
void hook_spbrfs(char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbrfs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spbrfs.timings[0] += (helpTimeStop - helpTime);
    data->spbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_strsyl (char* trana, char* tranb, Int * isgn, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* scale, Int * info, int len_trana, int len_tranb);
void hook_strsyl(char* trana, char* tranb, Int * isgn, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* scale, Int * info, int len_trana, int len_tranb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strsyl((void*) trana, (void*) tranb, (void*) isgn, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) scale, (void*) info, (int) len_trana, (int) len_tranb);

    helpTimeStop = flexiblas_wtime();

    data->strsyl.timings[0] += (helpTimeStop - helpTime);
    data->strsyl.calls[0]++;

    return ;
}



extern int flexiblas_chain_izmax1 (Int * n, double complex* zx, Int * incx);
int hook_izmax1(Int * n, double complex* zx, Int * incx)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_izmax1((void*) n, (void*) zx, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->izmax1.timings[0] += (helpTimeStop - helpTime);
    data->izmax1.calls[0]++;

    return v;
}



extern void flexiblas_chain_dsfrk (char* transr, char* uplo, char* trans, Int * n, Int * k, double* alpha, double* a, Int * lda, double* beta, double* c, int len_transr, int len_uplo, int len_trans);
void hook_dsfrk(char* transr, char* uplo, char* trans, Int * n, Int * k, double* alpha, double* a, Int * lda, double* beta, double* c, int len_transr, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsfrk((void*) transr, (void*) uplo, (void*) trans, (void*) n, (void*) k, (void*) alpha, (void*) a, (void*) lda, (void*) beta, (void*) c, (int) len_transr, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dsfrk.timings[0] += (helpTimeStop - helpTime);
    data->dsfrk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaed0 (Int * qsiz, Int * n, double* d, double* e, double complex* q, Int * ldq, double complex* qstore, Int * ldqs, double* rwork, Int * iwork, Int * info);
void hook_zlaed0(Int * qsiz, Int * n, double* d, double* e, double complex* q, Int * ldq, double complex* qstore, Int * ldqs, double* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaed0((void*) qsiz, (void*) n, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) qstore, (void*) ldqs, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaed0.timings[0] += (helpTimeStop - helpTime);
    data->zlaed0.calls[0]++;

    return ;
}



extern void flexiblas_chain_slagts (Int * job, Int * n, float* a, float* b, float* c, float* d, Int * in, float* y, float* tol, Int * info);
void hook_slagts(Int * job, Int * n, float* a, float* b, float* c, float* d, Int * in, float* y, float* tol, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slagts((void*) job, (void*) n, (void*) a, (void*) b, (void*) c, (void*) d, (void*) in, (void*) y, (void*) tol, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slagts.timings[0] += (helpTimeStop - helpTime);
    data->slagts.calls[0]++;

    return ;
}



extern void flexiblas_chain_sppsv (char* uplo, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, Int * info, int len_uplo);
void hook_sppsv(char* uplo, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sppsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sppsv.timings[0] += (helpTimeStop - helpTime);
    data->sppsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dbdsvdx (char* uplo, char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double* z, Int * ldz, double* work, Int * iwork, Int * info, int len_uplo, int len_jobz, int len_range);
void hook_dbdsvdx(char* uplo, char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double* z, Int * ldz, double* work, Int * iwork, Int * info, int len_uplo, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dbdsvdx((void*) uplo, (void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) ns, (void*) s, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->dbdsvdx.timings[0] += (helpTimeStop - helpTime);
    data->dbdsvdx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpttrs (char* uplo, Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_cpttrs(char* uplo, Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpttrs((void*) uplo, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpttrs.timings[0] += (helpTimeStop - helpTime);
    data->cpttrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlacrt (Int * n, double complex* cx, Int * incx, double complex* cy, Int * incy, double complex* c, double complex* s);
void hook_zlacrt(Int * n, double complex* cx, Int * incx, double complex* cy, Int * incy, double complex* c, double complex* s)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlacrt((void*) n, (void*) cx, (void*) incx, (void*) cy, (void*) incy, (void*) c, (void*) s);

    helpTimeStop = flexiblas_wtime();

    data->zlacrt.timings[0] += (helpTimeStop - helpTime);
    data->zlacrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpttrf (Int * n, double* d, double* e, Int * info);
void hook_dpttrf(Int * n, double* d, double* e, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpttrf((void*) n, (void*) d, (void*) e, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dpttrf.timings[0] += (helpTimeStop - helpTime);
    data->dpttrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytri (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * info, int len_uplo);
void hook_dsytri(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytri.timings[0] += (helpTimeStop - helpTime);
    data->dsytri.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgetrf2 (Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info);
void hook_sgetrf2(Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgetrf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->sgetrf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cstemr (char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, Int * m, float* w, float complex* z, Int * ldz, Int * nzc, Int * isuppz, Int * tryrac, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_cstemr(char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, Int * m, float* w, float complex* z, Int * ldz, Int * nzc, Int * isuppz, Int * tryrac, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cstemr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) nzc, (void*) isuppz, (void*) tryrac, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->cstemr.timings[0] += (helpTimeStop - helpTime);
    data->cstemr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sggsvd3 (char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, float* a, Int * lda, float* b, Int * ldb, float* alpha, float* beta, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_sggsvd3(char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, float* a, Int * lda, float* b, Int * ldb, float* alpha, float* beta, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sggsvd3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) n, (void*) p, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->sggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->sggsvd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelq2 (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zgelq2(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelq2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelq2.timings[0] += (helpTimeStop - helpTime);
    data->zgelq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgerqf (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zgerqf(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgerqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgerqf.timings[0] += (helpTimeStop - helpTime);
    data->zgerqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slauu2 (char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo);
void hook_slauu2(char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slauu2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slauu2.timings[0] += (helpTimeStop - helpTime);
    data->slauu2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbrfs (char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans);
void hook_dgbrfs(char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbrfs((void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dgbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpftrs (char* transr, char* uplo, Int * n, Int * nrhs, double* a, double* b, Int * ldb, Int * info, int len_transr, int len_uplo);
void hook_dpftrs(char* transr, char* uplo, Int * n, Int * nrhs, double* a, double* b, Int * ldb, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpftrs((void*) transr, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) b, (void*) ldb, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpftrs.timings[0] += (helpTimeStop - helpTime);
    data->dpftrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormtr (char* side, char* uplo, char* trans, Int * m, Int * n, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans);
void hook_dormtr(char* side, char* uplo, char* trans, Int * m, Int * n, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormtr.timings[0] += (helpTimeStop - helpTime);
    data->dormtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgglse (Int * m, Int * n, Int * p, float* a, Int * lda, float* b, Int * ldb, float* c, float* d, float* x, float* work, Int * lwork, Int * info);
void hook_sgglse(Int * m, Int * n, Int * p, float* a, Int * lda, float* b, Int * ldb, float* c, float* d, float* x, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgglse((void*) m, (void*) n, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) d, (void*) x, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgglse.timings[0] += (helpTimeStop - helpTime);
    data->sgglse.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasq2 (Int * n, float* z, Int * info);
void hook_slasq2(Int * n, float* z, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasq2((void*) n, (void*) z, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasq2.timings[0] += (helpTimeStop - helpTime);
    data->slasq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dggev (char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr);
void hook_dggev(char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggev((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->dggev.timings[0] += (helpTimeStop - helpTime);
    data->dggev.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsysv_aa_2stage (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsysv_aa_2stage(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsysv_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zsysv_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgges3 (char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * sdim, double* alphar, double* alphai, double* beta, double* vsl, Int * ldvsl, double* vsr, Int * ldvsr, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort);
void hook_dgges3(char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * sdim, double* alphar, double* alphai, double* beta, double* vsl, Int * ldvsl, double* vsr, Int * ldvsr, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgges3((void*) jobvsl, (void*) jobvsr, (void*) sort, (void*) selctg, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sdim, (void*) alphar, (void*) alphai, (void*) beta, (void*) vsl, (void*) ldvsl, (void*) vsr, (void*) ldvsr, (void*) work, (void*) lwork, (void*) bwork, (void*) info, (int) len_jobvsl, (int) len_jobvsr, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->dgges3.timings[0] += (helpTimeStop - helpTime);
    data->dgges3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyevd_2stage (char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_ssyevd_2stage(char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqr4 (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float complex* h, Int * ldh, float complex* w, Int * iloz, Int * ihiz, float complex* z, Int * ldz, float complex* work, Int * lwork, Int * info);
void hook_claqr4(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float complex* h, Int * ldh, float complex* w, Int * iloz, Int * ihiz, float complex* z, Int * ldz, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqr4((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claqr4.timings[0] += (helpTimeStop - helpTime);
    data->claqr4.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgsen (Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * m, float* pl, float* pr, float* dif, float complex* work, Int * lwork, Int * iwork, Int * liwork, Int * info);
void hook_ctgsen(Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * m, float* pl, float* pr, float* dif, float complex* work, Int * lwork, Int * iwork, Int * liwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgsen((void*) ijob, (void*) wantq, (void*) wantz, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) m, (void*) pl, (void*) pr, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctgsen.timings[0] += (helpTimeStop - helpTime);
    data->ctgsen.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqz3 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, double complex* alpha, double complex* beta, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* qc, Int * ldqc, double complex* zc, Int * ldzc, double complex* work, Int * lwork, Int * info);
void hook_zlaqz3(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, double complex* alpha, double complex* beta, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* qc, Int * ldqc, double complex* zc, Int * ldzc, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqz3((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nshifts, (void*) nblock_desired, (void*) alpha, (void*) beta, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqz3.timings[0] += (helpTimeStop - helpTime);
    data->zlaqz3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctprfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ctprfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctprfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctprfs.timings[0] += (helpTimeStop - helpTime);
    data->ctprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaed7 (Int * n, Int * cutpnt, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, double* d, double complex* q, Int * ldq, double* rho, Int * indxq, double* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double complex* work, double* rwork, Int * iwork, Int * info);
void hook_zlaed7(Int * n, Int * cutpnt, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, double* d, double complex* q, Int * ldq, double* rho, Int * indxq, double* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double complex* work, double* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaed7((void*) n, (void*) cutpnt, (void*) qsiz, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) d, (void*) q, (void*) ldq, (void*) rho, (void*) indxq, (void*) qstore, (void*) qptr, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) work, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaed7.timings[0] += (helpTimeStop - helpTime);
    data->zlaed7.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqr3 (Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, float complex* h, Int * ldh, Int * iloz, Int * ihiz, float complex* z, Int * ldz, Int * ns, Int * nd, float complex* sh, float complex* v, Int * ldv, Int * nh, float complex* t, Int * ldt, Int * nv, float complex* wv, Int * ldwv, float complex* work, Int * lwork);
void hook_claqr3(Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, float complex* h, Int * ldh, Int * iloz, Int * ihiz, float complex* z, Int * ldz, Int * ns, Int * nd, float complex* sh, float complex* v, Int * ldv, Int * nh, float complex* t, Int * ldt, Int * nv, float complex* wv, Int * ldwv, float complex* work, Int * lwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqr3((void*) wantt, (void*) wantz, (void*) n, (void*) ktop, (void*) kbot, (void*) nw, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) sh, (void*) v, (void*) ldv, (void*) nh, (void*) t, (void*) ldt, (void*) nv, (void*) wv, (void*) ldwv, (void*) work, (void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->claqr3.timings[0] += (helpTimeStop - helpTime);
    data->claqr3.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorbdb4 (Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* phantom, float* work, Int * lwork, Int * info);
void hook_sorbdb4(Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* phantom, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorbdb4((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) phantom, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb4.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb4.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarrk (Int * n, Int * iw, double* gl, double* gu, double* d, double* e2, double* pivmin, double* reltol, double* w, double* werr, Int * info);
void hook_dlarrk(Int * n, Int * iw, double* gl, double* gu, double* d, double* e2, double* pivmin, double* reltol, double* w, double* werr, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarrk((void*) n, (void*) iw, (void*) gl, (void*) gu, (void*) d, (void*) e2, (void*) pivmin, (void*) reltol, (void*) w, (void*) werr, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrk.timings[0] += (helpTimeStop - helpTime);
    data->dlarrk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqrf (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dgeqrf(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqrf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrf.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cppequ (char* uplo, Int * n, float complex* ap, float* s, float* scond, float* amax, Int * info, int len_uplo);
void hook_cppequ(char* uplo, Int * n, float complex* ap, float* s, float* scond, float* amax, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cppequ((void*) uplo, (void*) n, (void*) ap, (void*) s, (void*) scond, (void*) amax, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cppequ.timings[0] += (helpTimeStop - helpTime);
    data->cppequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_chesv_aa_2stage (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chesv_aa_2stage(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chesv_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chesv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chesv_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbsv (Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, Int * ipiv, float* b, Int * ldb, Int * info);
void hook_sgbsv(Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, Int * ipiv, float* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbsv((void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbsv.timings[0] += (helpTimeStop - helpTime);
    data->sgbsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztpqrt2 (Int * m, Int * n, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, Int * info);
void hook_ztpqrt2(Int * m, Int * n, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztpqrt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ztpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->ztpqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_claic1 (Int * job, Int * j, float complex* x, float* sest, float complex* w, float complex* gamma, float* sestpr, float complex* s, float complex* c);
void hook_claic1(Int * job, Int * j, float complex* x, float* sest, float complex* w, float complex* gamma, float* sestpr, float complex* s, float complex* c)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claic1((void*) job, (void*) j, (void*) x, (void*) sest, (void*) w, (void*) gamma, (void*) sestpr, (void*) s, (void*) c);

    helpTimeStop = flexiblas_wtime();

    data->claic1.timings[0] += (helpTimeStop - helpTime);
    data->claic1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbevd_2stage (char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_dsbevd_2stage(char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgsyl (char* trans, Int * ijob, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float complex* d, Int * ldd, float complex* e, Int * lde, float complex* f, Int * ldf, float* scale, float* dif, float complex* work, Int * lwork, Int * iwork, Int * info, int len_trans);
void hook_ctgsyl(char* trans, Int * ijob, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float complex* d, Int * ldd, float complex* e, Int * lde, float complex* f, Int * ldf, float* scale, float* dif, float complex* work, Int * lwork, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgsyl((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ctgsyl.timings[0] += (helpTimeStop - helpTime);
    data->ctgsyl.calls[0]++;

    return ;
}



extern void flexiblas_chain_zheevr_2stage (char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, Int * isuppz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zheevr_2stage(char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, Int * isuppz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheevr_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zheevr_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgtrfs (char* trans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* dlf, float* df, float* duf, float* du2, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_trans);
void hook_sgtrfs(char* trans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* dlf, float* df, float* duf, float* du2, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgtrfs((void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->sgtrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqp3rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double* a, Int * lda, Int * done, Int * kb, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double* tau, double* vn1, double* vn2, double* auxv, double* f, Int * ldf, Int * iwork, Int * info);
void hook_dlaqp3rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double* a, Int * lda, Int * done, Int * kb, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double* tau, double* vn1, double* vn2, double* auxv, double* f, Int * ldf, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) nb, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) done, (void*) kb, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->dlaqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgesv (Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info);
void hook_zgesv(Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgesv((void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgesv.timings[0] += (helpTimeStop - helpTime);
    data->zgesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtgex2 (Int * wantq, Int * wantz, Int * n, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * j1, Int * n1, Int * n2, double* work, Int * lwork, Int * info);
void hook_dtgex2(Int * wantq, Int * wantz, Int * n, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * j1, Int * n1, Int * n2, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtgex2((void*) wantq, (void*) wantz, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) j1, (void*) n1, (void*) n2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtgex2.timings[0] += (helpTimeStop - helpTime);
    data->dtgex2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cherfs (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_cherfs(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cherfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cherfs.timings[0] += (helpTimeStop - helpTime);
    data->cherfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zppcon (char* uplo, Int * n, double complex* ap, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zppcon(char* uplo, Int * n, double complex* ap, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zppcon((void*) uplo, (void*) n, (void*) ap, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zppcon.timings[0] += (helpTimeStop - helpTime);
    data->zppcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasd7 (Int * icompq, Int * nl, Int * nr, Int * sqre, Int * k, float* d, float* z, float* zw, float* vf, float* vfw, float* vl, float* vlw, float* alpha, float* beta, float* dsigma, Int * idx, Int * idxp, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* c, float* s, Int * info);
void hook_slasd7(Int * icompq, Int * nl, Int * nr, Int * sqre, Int * k, float* d, float* z, float* zw, float* vf, float* vfw, float* vl, float* vlw, float* alpha, float* beta, float* dsigma, Int * idx, Int * idxp, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* c, float* s, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasd7((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) k, (void*) d, (void*) z, (void*) zw, (void*) vf, (void*) vfw, (void*) vl, (void*) vlw, (void*) alpha, (void*) beta, (void*) dsigma, (void*) idx, (void*) idxp, (void*) idxq, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) c, (void*) s, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd7.timings[0] += (helpTimeStop - helpTime);
    data->slasd7.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaesy (double complex* a, double complex* b, double complex* c, double complex* rt1, double complex* rt2, double complex* evscal, double complex* cs1, double complex* sn1);
void hook_zlaesy(double complex* a, double complex* b, double complex* c, double complex* rt1, double complex* rt2, double complex* evscal, double complex* cs1, double complex* sn1)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaesy((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2, (void*) evscal, (void*) cs1, (void*) sn1);

    helpTimeStop = flexiblas_wtime();

    data->zlaesy.timings[0] += (helpTimeStop - helpTime);
    data->zlaesy.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgemlq (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* t, Int * tsize, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dgemlq(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* t, Int * tsize, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgemlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgemlq.timings[0] += (helpTimeStop - helpTime);
    data->dgemlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbbrd (char* vect, Int * m, Int * n, Int * ncc, Int * kl, Int * ku, double* ab, Int * ldab, double* d, double* e, double* q, Int * ldq, double* pt, Int * ldpt, double* c, Int * ldc, double* work, Int * info, int len_vect);
void hook_dgbbrd(char* vect, Int * m, Int * n, Int * ncc, Int * kl, Int * ku, double* ab, Int * ldab, double* d, double* e, double* q, Int * ldq, double* pt, Int * ldpt, double* c, Int * ldc, double* work, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbbrd((void*) vect, (void*) m, (void*) n, (void*) ncc, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) pt, (void*) ldpt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->dgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->dgbbrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqrfp (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zgeqrfp(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqrfp((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->zgeqrfp.calls[0]++;

    return ;
}



extern void flexiblas_chain_slacn2 (Int * n, float* v, float* x, Int * isgn, float* est, Int * kase, Int * isave);
void hook_slacn2(Int * n, float* v, float* x, Int * isgn, float* est, Int * kase, Int * isave)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slacn2((void*) n, (void*) v, (void*) x, (void*) isgn, (void*) est, (void*) kase, (void*) isave);

    helpTimeStop = flexiblas_wtime();

    data->slacn2.timings[0] += (helpTimeStop - helpTime);
    data->slacn2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slacpy (char* uplo, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, int len_uplo);
void hook_slacpy(char* uplo, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slacpy((void*) uplo, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slacpy.timings[0] += (helpTimeStop - helpTime);
    data->slacpy.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormrq (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dormrq(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormrq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormrq.timings[0] += (helpTimeStop - helpTime);
    data->dormrq.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggbal (char* job, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * ilo, Int * ihi, float* lscale, float* rscale, float* work, Int * info, int len_job);
void hook_cggbal(char* job, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * ilo, Int * ihi, float* lscale, float* rscale, float* work, Int * info, int len_job)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggbal((void*) job, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) ilo, (void*) ihi, (void*) lscale, (void*) rscale, (void*) work, (void*) info, (int) len_job);

    helpTimeStop = flexiblas_wtime();

    data->cggbal.timings[0] += (helpTimeStop - helpTime);
    data->cggbal.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrs_3 (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhetrs_3(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrs_3((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrs_3.timings[0] += (helpTimeStop - helpTime);
    data->zhetrs_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrf_aa (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytrf_aa(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrf_aa((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaorhr_col_getrfnp (Int * m, Int * n, float* a, Int * lda, float* d, Int * info);
void hook_slaorhr_col_getrfnp(Int * m, Int * n, float* a, Int * lda, float* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaorhr_col_getrfnp((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaorhr_col_getrfnp.timings[0] += (helpTimeStop - helpTime);
    data->slaorhr_col_getrfnp.calls[0]++;

    return ;
}



extern void flexiblas_chain_csysv_aa_2stage (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csysv_aa_2stage(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csysv_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->csysv_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbsvx (char* fact, char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, Int * ipiv, char* equed, float* r, float* c, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_trans, int len_equed);
void hook_sgbsvx(char* fact, char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, Int * ipiv, char* equed, float* r, float* c, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_trans, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbsvx((void*) fact, (void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) ipiv, (void*) equed, (void*) r, (void*) c, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_trans, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->sgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->sgbsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarfg (Int * n, float* alpha, float* x, Int * incx, float* tau);
void hook_slarfg(Int * n, float* alpha, float* x, Int * incx, float* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarfg((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->slarfg.timings[0] += (helpTimeStop - helpTime);
    data->slarfg.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgtsv (Int * n, Int * nrhs, double complex* dl, double complex* d, double complex* du, double complex* b, Int * ldb, Int * info);
void hook_zgtsv(Int * n, Int * nrhs, double complex* dl, double complex* d, double complex* du, double complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgtsv((void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgtsv.timings[0] += (helpTimeStop - helpTime);
    data->zgtsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunml2 (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zunml2(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunml2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunml2.timings[0] += (helpTimeStop - helpTime);
    data->zunml2.calls[0]++;

    return ;
}



extern void flexiblas_chain_claswlq (Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info);
void hook_claswlq(Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claswlq((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claswlq.timings[0] += (helpTimeStop - helpTime);
    data->claswlq.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlansp (char* norm, char* uplo, Int * n, double complex* ap, double* work, int len_norm, int len_uplo);
double hook_zlansp(char* norm, char* uplo, Int * n, double complex* ap, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlansp((void*) norm, (void*) uplo, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlansp.timings[0] += (helpTimeStop - helpTime);
    data->zlansp.calls[0]++;

    return v;
}



extern void flexiblas_chain_zsyconv (char* uplo, char* way, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* e, Int * info, int len_uplo, int len_way);
void hook_zsyconv(char* uplo, char* way, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* e, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsyconv((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) e, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->zsyconv.timings[0] += (helpTimeStop - helpTime);
    data->zsyconv.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelqt3 (Int * m, Int * n, float* a, Int * lda, float* t, Int * ldt, Int * info);
void hook_sgelqt3(Int * m, Int * n, float* a, Int * lda, float* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelqt3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->sgelqt3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbcon (char* norm, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_norm);
void hook_cgbcon(char* norm, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbcon((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->cgbcon.timings[0] += (helpTimeStop - helpTime);
    data->cgbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarnv (Int * idist, Int * iseed, Int * n, double* x);
void hook_dlarnv(Int * idist, Int * iseed, Int * n, double* x)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarnv((void*) idist, (void*) iseed, (void*) n, (void*) x);

    helpTimeStop = flexiblas_wtime();

    data->dlarnv.timings[0] += (helpTimeStop - helpTime);
    data->dlarnv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpoequb (Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, Int * info);
void hook_zpoequb(Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpoequb((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zpoequb.timings[0] += (helpTimeStop - helpTime);
    data->zpoequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlahqr (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * info);
void hook_dlahqr(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, Int * iloz, Int * ihiz, double* z, Int * ldz, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlahqr((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlahqr.timings[0] += (helpTimeStop - helpTime);
    data->dlahqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeequb (Int * m, Int * n, double* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info);
void hook_dgeequb(Int * m, Int * n, double* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeequb((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeequb.timings[0] += (helpTimeStop - helpTime);
    data->dgeequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_slabrd (Int * m, Int * n, Int * nb, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* x, Int * ldx, float* y, Int * ldy);
void hook_slabrd(Int * m, Int * n, Int * nb, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* x, Int * ldx, float* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slabrd((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) x, (void*) ldx, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->slabrd.timings[0] += (helpTimeStop - helpTime);
    data->slabrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtptrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* ap, double* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_dtptrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* ap, double* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtptrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtptrs.timings[0] += (helpTimeStop - helpTime);
    data->dtptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrs (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_csytrs(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrs.timings[0] += (helpTimeStop - helpTime);
    data->csytrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqr4 (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, Int * iloz, Int * ihiz, double* z, Int * ldz, double* work, Int * lwork, Int * info);
void hook_dlaqr4(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, Int * iloz, Int * ihiz, double* z, Int * ldz, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqr4((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr4.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr4.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgees (char* jobvs, char* sort, Int * select, Int * n, double* a, Int * lda, Int * sdim, double* wr, double* wi, double* vs, Int * ldvs, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvs, int len_sort);
void hook_dgees(char* jobvs, char* sort, Int * select, Int * n, double* a, Int * lda, Int * sdim, double* wr, double* wi, double* vs, Int * ldvs, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvs, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgees((void*) jobvs, (void*) sort, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) wr, (void*) wi, (void*) vs, (void*) ldvs, (void*) work, (void*) lwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->dgees.timings[0] += (helpTimeStop - helpTime);
    data->dgees.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbev_2stage (char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_dsbev_2stage(char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbev_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbev_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelss (Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * info);
void hook_zgelss(Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelss((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelss.timings[0] += (helpTimeStop - helpTime);
    data->zgelss.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrs_rook (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhetrs_rook(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->zhetrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpstrf (char* uplo, Int * n, double* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo);
void hook_dpstrf(char* uplo, Int * n, double* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpstrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpstrf.timings[0] += (helpTimeStop - helpTime);
    data->dpstrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb1 (Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info);
void hook_cunbdb1(Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb1((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb1.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasda (Int * icompq, Int * smlsiz, Int * n, Int * sqre, double* d, double* e, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* work, Int * iwork, Int * info);
void hook_dlasda(Int * icompq, Int * smlsiz, Int * n, Int * sqre, double* d, double* e, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasda((void*) icompq, (void*) smlsiz, (void*) n, (void*) sqre, (void*) d, (void*) e, (void*) u, (void*) ldu, (void*) vt, (void*) k, (void*) difl, (void*) difr, (void*) z, (void*) poles, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) perm, (void*) givnum, (void*) c, (void*) s, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasda.timings[0] += (helpTimeStop - helpTime);
    data->dlasda.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarrr (Int * n, float* d, float* e, Int * info);
void hook_slarrr(Int * n, float* d, float* e, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarrr((void*) n, (void*) d, (void*) e, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrr.timings[0] += (helpTimeStop - helpTime);
    data->slarrr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhesvx (char* fact, char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, Int * lwork, double* rwork, Int * info, int len_fact, int len_uplo);
void hook_zhesvx(char* fact, char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, Int * lwork, double* rwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhesvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhesvx.timings[0] += (helpTimeStop - helpTime);
    data->zhesvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrs_3 (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* e, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dsytrs_3(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* e, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrs_3((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_3.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasq6 (Int * i0, Int * n0, double* z, Int * pp, double* dmin, double* dmin1, double* dmin2, double* dn, double* dnm1, double* dnm2);
void hook_dlasq6(Int * i0, Int * n0, double* z, Int * pp, double* dmin, double* dmin1, double* dmin2, double* dn, double* dnm1, double* dnm2)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasq6((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dnm1, (void*) dnm2);

    helpTimeStop = flexiblas_wtime();

    data->dlasq6.timings[0] += (helpTimeStop - helpTime);
    data->dlasq6.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqp2 (Int * m, Int * n, Int * offset, double* a, Int * lda, Int * jpvt, double* tau, double* vn1, double* vn2, double* work);
void hook_dlaqp2(Int * m, Int * n, Int * offset, double* a, Int * lda, Int * jpvt, double* tau, double* vn1, double* vn2, double* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqp2((void*) m, (void*) n, (void*) offset, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->dlaqp2.timings[0] += (helpTimeStop - helpTime);
    data->dlaqp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spotri (char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo);
void hook_spotri(char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spotri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spotri.timings[0] += (helpTimeStop - helpTime);
    data->spotri.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrs2 (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * info, int len_uplo);
void hook_ssytrs2(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrs2((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs2.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqr2p (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info);
void hook_dgeqr2p(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqr2p((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr2p.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrtri (char* uplo, char* diag, Int * n, float complex* a, Int * lda, Int * info, int len_uplo, int len_diag);
void hook_ctrtri(char* uplo, char* diag, Int * n, float complex* a, Int * lda, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrtri((void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctrtri.timings[0] += (helpTimeStop - helpTime);
    data->ctrtri.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbequ (char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* s, double* scond, double* amax, Int * info, int len_uplo);
void hook_dpbequ(char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* s, double* scond, double* amax, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbequ((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpbequ.timings[0] += (helpTimeStop - helpTime);
    data->dpbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgebak (char* job, char* side, Int * n, Int * ilo, Int * ihi, float* scale, Int * m, float* v, Int * ldv, Int * info, int len_job, int len_side);
void hook_sgebak(char* job, char* side, Int * n, Int * ilo, Int * ihi, float* scale, Int * m, float* v, Int * ldv, Int * info, int len_job, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgebak((void*) job, (void*) side, (void*) n, (void*) ilo, (void*) ihi, (void*) scale, (void*) m, (void*) v, (void*) ldv, (void*) info, (int) len_job, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->sgebak.timings[0] += (helpTimeStop - helpTime);
    data->sgebak.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrd (char* uplo, Int * n, double* a, Int * lda, double* d, double* e, double* tau, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrd(char* uplo, Int * n, double* a, Int * lda, double* d, double* e, double* tau, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrd((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_sdisna (char* job, Int * m, Int * n, float* d, float* sep, Int * info, int len_job);
void hook_sdisna(char* job, Int * m, Int * n, float* d, float* sep, Int * info, int len_job)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sdisna((void*) job, (void*) m, (void*) n, (void*) d, (void*) sep, (void*) info, (int) len_job);

    helpTimeStop = flexiblas_wtime();

    data->sdisna.timings[0] += (helpTimeStop - helpTime);
    data->sdisna.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgerqf (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgerqf(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgerqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgerqf.timings[0] += (helpTimeStop - helpTime);
    data->sgerqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zuncsd2by1 (char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double complex* u1, Int * ldu1, double complex* u2, Int * ldu2, double complex* v1t, Int * ldv1t, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t);
void hook_zuncsd2by1(char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double complex* u1, Int * ldu1, double complex* u2, Int * ldu2, double complex* v1t, Int * ldv1t, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zuncsd2by1((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t);

    helpTimeStop = flexiblas_wtime();

    data->zuncsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->zuncsd2by1.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgehrd (Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cgehrd(Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgehrd((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgehrd.timings[0] += (helpTimeStop - helpTime);
    data->cgehrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpotrf (char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo);
void hook_zpotrf(char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpotrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpotrf.timings[0] += (helpTimeStop - helpTime);
    data->zpotrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slals0 (Int * icompq, Int * nl, Int * nr, Int * sqre, Int * nrhs, float* b, Int * ldb, float* bx, Int * ldbx, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* poles, float* difl, float* difr, float* z, Int * k, float* c, float* s, float* work, Int * info);
void hook_slals0(Int * icompq, Int * nl, Int * nr, Int * sqre, Int * nrhs, float* b, Int * ldb, float* bx, Int * ldbx, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* poles, float* difl, float* difr, float* z, Int * k, float* c, float* s, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slals0((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) nrhs, (void*) b, (void*) ldb, (void*) bx, (void*) ldbx, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) poles, (void*) difl, (void*) difr, (void*) z, (void*) k, (void*) c, (void*) s, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slals0.timings[0] += (helpTimeStop - helpTime);
    data->slals0.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgerqf (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dgerqf(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgerqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgerqf.timings[0] += (helpTimeStop - helpTime);
    data->dgerqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaebz (Int * ijob, Int * nitmax, Int * n, Int * mmax, Int * minp, Int * nbmin, double* abstol, double* reltol, double* pivmin, double* d, double* e, double* e2, Int * nval, double* ab, double* c, Int * mout, Int * nab, double* work, Int * iwork, Int * info);
void hook_dlaebz(Int * ijob, Int * nitmax, Int * n, Int * mmax, Int * minp, Int * nbmin, double* abstol, double* reltol, double* pivmin, double* d, double* e, double* e2, Int * nval, double* ab, double* c, Int * mout, Int * nab, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaebz((void*) ijob, (void*) nitmax, (void*) n, (void*) mmax, (void*) minp, (void*) nbmin, (void*) abstol, (void*) reltol, (void*) pivmin, (void*) d, (void*) e, (void*) e2, (void*) nval, (void*) ab, (void*) c, (void*) mout, (void*) nab, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaebz.timings[0] += (helpTimeStop - helpTime);
    data->dlaebz.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilaslc (Int * m, Int * n, float* a, Int * lda);
int hook_ilaslc(Int * m, Int * n, float* a, Int * lda)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilaslc((void*) m, (void*) n, (void*) a, (void*) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaslc.timings[0] += (helpTimeStop - helpTime);
    data->ilaslc.calls[0]++;

    return v;
}



extern void flexiblas_chain_strtri (char* uplo, char* diag, Int * n, float* a, Int * lda, Int * info, int len_uplo, int len_diag);
void hook_strtri(char* uplo, char* diag, Int * n, float* a, Int * lda, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strtri((void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->strtri.timings[0] += (helpTimeStop - helpTime);
    data->strtri.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgetri (Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info);
void hook_zgetri(Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetri((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgetri.timings[0] += (helpTimeStop - helpTime);
    data->zgetri.calls[0]++;

    return ;
}



extern void flexiblas_chain_strcon (char* norm, char* uplo, char* diag, Int * n, float* a, Int * lda, float* rcond, float* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_strcon(char* norm, char* uplo, char* diag, Int * n, float* a, Int * lda, float* rcond, float* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->strcon.timings[0] += (helpTimeStop - helpTime);
    data->strcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarfgp (Int * n, float complex* alpha, float complex* x, Int * incx, float complex* tau);
void hook_clarfgp(Int * n, float complex* alpha, float complex* x, Int * incx, float complex* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarfgp((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->clarfgp.timings[0] += (helpTimeStop - helpTime);
    data->clarfgp.calls[0]++;

    return ;
}



extern float flexiblas_chain_clanhp (char* norm, char* uplo, Int * n, float complex* ap, float* work, int len_norm, int len_uplo);
float hook_clanhp(char* norm, char* uplo, Int * n, float complex* ap, float* work, int len_norm, int len_uplo)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clanhp((void*) norm, (void*) uplo, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clanhp.timings[0] += (helpTimeStop - helpTime);
    data->clanhp.calls[0]++;

    return v;
}



extern void flexiblas_chain_zlarcm (Int * m, Int * n, double* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double* rwork);
void hook_zlarcm(Int * m, Int * n, double* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double* rwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarcm((void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) rwork);

    helpTimeStop = flexiblas_wtime();

    data->zlarcm.timings[0] += (helpTimeStop - helpTime);
    data->zlarcm.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaruv (Int * iseed, Int * n, float* x);
void hook_slaruv(Int * iseed, Int * n, float* x)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaruv((void*) iseed, (void*) n, (void*) x);

    helpTimeStop = flexiblas_wtime();

    data->slaruv.timings[0] += (helpTimeStop - helpTime);
    data->slaruv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgemqr (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* t, Int * tsize, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dgemqr(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* t, Int * tsize, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgemqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgemqr.timings[0] += (helpTimeStop - helpTime);
    data->dgemqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgghrd (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * info, int len_compq, int len_compz);
void hook_cgghrd(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgghrd((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->cgghrd.timings[0] += (helpTimeStop - helpTime);
    data->cgghrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgttrf (Int * n, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, Int * info);
void hook_cgttrf(Int * n, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgttrf((void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgttrf.timings[0] += (helpTimeStop - helpTime);
    data->cgttrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeequ (Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_sgeequ(Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeequ((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeequ.timings[0] += (helpTimeStop - helpTime);
    data->sgeequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunbdb3 (Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* work, Int * lwork, Int * info);
void hook_zunbdb3(Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x21, Int * ldx21, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunbdb3((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zunbdb3.timings[0] += (helpTimeStop - helpTime);
    data->zunbdb3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd7 (Int * icompq, Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* z, double* zw, double* vf, double* vfw, double* vl, double* vlw, double* alpha, double* beta, double* dsigma, Int * idx, Int * idxp, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* c, double* s, Int * info);
void hook_dlasd7(Int * icompq, Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* z, double* zw, double* vf, double* vfw, double* vl, double* vlw, double* alpha, double* beta, double* dsigma, Int * idx, Int * idxp, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* c, double* s, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd7((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) k, (void*) d, (void*) z, (void*) zw, (void*) vf, (void*) vfw, (void*) vl, (void*) vlw, (void*) alpha, (void*) beta, (void*) dsigma, (void*) idx, (void*) idxp, (void*) idxq, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) c, (void*) s, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd7.timings[0] += (helpTimeStop - helpTime);
    data->dlasd7.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsycon_3 (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo);
void hook_zsycon_3(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsycon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsycon_3.timings[0] += (helpTimeStop - helpTime);
    data->zsycon_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgtsqr_row (Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info);
void hook_dorgtsqr_row(Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgtsqr_row((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->dorgtsqr_row.calls[0]++;

    return ;
}



extern void flexiblas_chain_dhseqr (char* job, char* compz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_job, int len_compz);
void hook_dhseqr(char* job, char* compz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* wr, double* wi, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_job, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dhseqr((void*) job, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_job, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->dhseqr.timings[0] += (helpTimeStop - helpTime);
    data->dhseqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssptri (char* uplo, Int * n, float* ap, Int * ipiv, float* work, Int * info, int len_uplo);
void hook_ssptri(char* uplo, Int * n, float* ap, Int * ipiv, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssptri((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssptri.timings[0] += (helpTimeStop - helpTime);
    data->ssptri.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlanhp (char* norm, char* uplo, Int * n, double complex* ap, double* work, int len_norm, int len_uplo);
double hook_zlanhp(char* norm, char* uplo, Int * n, double complex* ap, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlanhp((void*) norm, (void*) uplo, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlanhp.timings[0] += (helpTimeStop - helpTime);
    data->zlanhp.calls[0]++;

    return v;
}



extern void flexiblas_chain_cgbequb (Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_cgbequb(Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbequb((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbequb.timings[0] += (helpTimeStop - helpTime);
    data->cgbequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungbr (char* vect, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info, int len_vect);
void hook_zungbr(char* vect, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungbr((void*) vect, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->zungbr.timings[0] += (helpTimeStop - helpTime);
    data->zungbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssycon_3 (char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo);
void hook_ssycon_3(char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssycon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssycon_3.timings[0] += (helpTimeStop - helpTime);
    data->ssycon_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgemlq (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cgemlq(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgemlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgemlq.timings[0] += (helpTimeStop - helpTime);
    data->cgemlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgebal (char* job, Int * n, double complex* a, Int * lda, Int * ilo, Int * ihi, double* scale, Int * info, int len_job);
void hook_zgebal(char* job, Int * n, double complex* a, Int * lda, Int * ilo, Int * ihi, double* scale, Int * info, int len_job)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgebal((void*) job, (void*) n, (void*) a, (void*) lda, (void*) ilo, (void*) ihi, (void*) scale, (void*) info, (int) len_job);

    helpTimeStop = flexiblas_wtime();

    data->zgebal.timings[0] += (helpTimeStop - helpTime);
    data->zgebal.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgtcon (char* norm, Int * n, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_norm);
void hook_cgtcon(char* norm, Int * n, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgtcon((void*) norm, (void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->cgtcon.timings[0] += (helpTimeStop - helpTime);
    data->cgtcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeev (char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr);
void hook_dgeev(char* jobvl, char* jobvr, Int * n, double* a, Int * lda, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, double* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeev((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->dgeev.timings[0] += (helpTimeStop - helpTime);
    data->dgeev.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgesvx (char* fact, char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, char* equed, double* r, double* c, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact, int len_trans, int len_equed);
void hook_zgesvx(char* fact, char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, char* equed, double* r, double* c, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact, int len_trans, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgesvx((void*) fact, (void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) equed, (void*) r, (void*) c, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_trans, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zgesvx.timings[0] += (helpTimeStop - helpTime);
    data->zgesvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_shsein (char* side, char* eigsrc, char* initv, Int * select, Int * n, float* h, Int * ldh, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv);
void hook_shsein(char* side, char* eigsrc, char* initv, Int * select, Int * n, float* h, Int * ldh, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_shsein((void*) side, (void*) eigsrc, (void*) initv, (void*) select, (void*) n, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) ifaill, (void*) ifailr, (void*) info, (int) len_side, (int) len_eigsrc, (int) len_initv);

    helpTimeStop = flexiblas_wtime();

    data->shsein.timings[0] += (helpTimeStop - helpTime);
    data->shsein.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgebd2 (Int * m, Int * n, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* work, Int * info);
void hook_sgebd2(Int * m, Int * n, float* a, Int * lda, float* d, float* e, float* tauq, float* taup, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgebd2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgebd2.timings[0] += (helpTimeStop - helpTime);
    data->sgebd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqrt2 (Int * m, Int * n, double* a, Int * lda, double* t, Int * ldt, Int * info);
void hook_dgeqrt2(Int * m, Int * n, double* a, Int * lda, double* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqrt2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbequb (Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info);
void hook_dgbequb(Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbequb((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbequb.timings[0] += (helpTimeStop - helpTime);
    data->dgbequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgtsv (Int * n, Int * nrhs, double* dl, double* d, double* du, double* b, Int * ldb, Int * info);
void hook_dgtsv(Int * n, Int * nrhs, double* dl, double* d, double* du, double* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgtsv((void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgtsv.timings[0] += (helpTimeStop - helpTime);
    data->dgtsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_spbsvx (char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, char* equed, float* s, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed);
void hook_spbsvx(char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* afb, Int * ldafb, char* equed, float* s, float* b, Int * ldb, float* x, Int * ldx, float* rcond, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbsvx((void*) fact, (void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) equed, (void*) s, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->spbsvx.timings[0] += (helpTimeStop - helpTime);
    data->spbsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_spbequ (char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* s, float* scond, float* amax, Int * info, int len_uplo);
void hook_spbequ(char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* s, float* scond, float* amax, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbequ((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spbequ.timings[0] += (helpTimeStop - helpTime);
    data->spbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytri_rook (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * info, int len_uplo);
void hook_dsytri_rook(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytri_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytri_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbevx (char* jobz, char* range, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_ssbevx(char* jobz, char* range, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbevx.timings[0] += (helpTimeStop - helpTime);
    data->ssbevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqrt3 (Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info);
void hook_cgeqrt3(Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqrt3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrt3.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrt3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrf_rk (char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssytrf_rk(char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrf_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_claset (char* uplo, Int * m, Int * n, float complex* alpha, float complex* beta, float complex* a, Int * lda, int len_uplo);
void hook_claset(char* uplo, Int * m, Int * n, float complex* alpha, float complex* beta, float complex* a, Int * lda, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claset((void*) uplo, (void*) m, (void*) n, (void*) alpha, (void*) beta, (void*) a, (void*) lda, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->claset.timings[0] += (helpTimeStop - helpTime);
    data->claset.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelsd (Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float* work, Int * lwork, Int * iwork, Int * info);
void hook_sgelsd(Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float* work, Int * lwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelsd((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelsd.timings[0] += (helpTimeStop - helpTime);
    data->sgelsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpptrf (char* uplo, Int * n, double complex* ap, Int * info, int len_uplo);
void hook_zpptrf(char* uplo, Int * n, double complex* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpptrf((void*) uplo, (void*) n, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpptrf.timings[0] += (helpTimeStop - helpTime);
    data->zpptrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_claed8 (Int * k, Int * n, Int * qsiz, float complex* q, Int * ldq, float* d, float* rho, Int * cutpnt, float* z, float* dlambda, float complex* q2, Int * ldq2, float* w, Int * indxp, Int * indx, Int * indxq, Int * perm, Int * givptr, Int * givcol, float* givnum, Int * info);
void hook_claed8(Int * k, Int * n, Int * qsiz, float complex* q, Int * ldq, float* d, float* rho, Int * cutpnt, float* z, float* dlambda, float complex* q2, Int * ldq2, float* w, Int * indxp, Int * indx, Int * indxq, Int * perm, Int * givptr, Int * givcol, float* givnum, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claed8((void*) k, (void*) n, (void*) qsiz, (void*) q, (void*) ldq, (void*) d, (void*) rho, (void*) cutpnt, (void*) z, (void*) dlambda, (void*) q2, (void*) ldq2, (void*) w, (void*) indxp, (void*) indx, (void*) indxq, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claed8.timings[0] += (helpTimeStop - helpTime);
    data->claed8.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrs_aa_2stage (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_csytrs_aa_2stage(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrs_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaswlq (Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info);
void hook_zlaswlq(Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaswlq((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaswlq.timings[0] += (helpTimeStop - helpTime);
    data->zlaswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlalsd (char* uplo, Int * smlsiz, Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, double* rcond, Int * rank_bn, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dlalsd(char* uplo, Int * smlsiz, Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, double* rcond, Int * rank_bn, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlalsd((void*) uplo, (void*) smlsiz, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) rcond, (void*) rank_bn, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlalsd.timings[0] += (helpTimeStop - helpTime);
    data->dlalsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_sspev (char* jobz, char* uplo, Int * n, float* ap, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo);
void hook_sspev(char* jobz, char* uplo, Int * n, float* ap, float* w, float* z, Int * ldz, float* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sspev((void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sspev.timings[0] += (helpTimeStop - helpTime);
    data->sspev.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasr (char* side, char* pivot, char* direct, Int * m, Int * n, float* c, float* s, float* a, Int * lda, int len_side, int len_pivot, int len_direct);
void hook_slasr(char* side, char* pivot, char* direct, Int * m, Int * n, float* c, float* s, float* a, Int * lda, int len_side, int len_pivot, int len_direct)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasr((void*) side, (void*) pivot, (void*) direct, (void*) m, (void*) n, (void*) c, (void*) s, (void*) a, (void*) lda, (int) len_side, (int) len_pivot, (int) len_direct);

    helpTimeStop = flexiblas_wtime();

    data->slasr.timings[0] += (helpTimeStop - helpTime);
    data->slasr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunm22 (char* side, char* trans, Int * m, Int * n, Int * n1, Int * n2, double complex* q, Int * ldq, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zunm22(char* side, char* trans, Int * m, Int * n, Int * n1, Int * n2, double complex* q, Int * ldq, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunm22((void*) side, (void*) trans, (void*) m, (void*) n, (void*) n1, (void*) n2, (void*) q, (void*) ldq, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunm22.timings[0] += (helpTimeStop - helpTime);
    data->zunm22.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqp3 (Int * m, Int * n, double* a, Int * lda, Int * jpvt, double* tau, double* work, Int * lwork, Int * info);
void hook_dgeqp3(Int * m, Int * n, double* a, Int * lda, Int * jpvt, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqp3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->dgeqp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrf_rook (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetrf_rook(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrf_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->zhetrf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb3 (Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info);
void hook_cunbdb3(Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb3((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb3.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cungqr (Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cungqr(Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungqr((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cungqr.timings[0] += (helpTimeStop - helpTime);
    data->cungqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbequ (Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info);
void hook_dgbequ(Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbequ((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbequ.timings[0] += (helpTimeStop - helpTime);
    data->dgbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpstrf (char* uplo, Int * n, float complex* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo);
void hook_cpstrf(char* uplo, Int * n, float complex* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpstrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpstrf.timings[0] += (helpTimeStop - helpTime);
    data->cpstrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasd6 (Int * icompq, Int * nl, Int * nr, Int * sqre, float* d, float* vf, float* vl, float* alpha, float* beta, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* poles, float* difl, float* difr, float* z, Int * k, float* c, float* s, float* work, Int * iwork, Int * info);
void hook_slasd6(Int * icompq, Int * nl, Int * nr, Int * sqre, float* d, float* vf, float* vl, float* alpha, float* beta, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, float* givnum, Int * ldgnum, float* poles, float* difl, float* difr, float* z, Int * k, float* c, float* s, float* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasd6((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) d, (void*) vf, (void*) vl, (void*) alpha, (void*) beta, (void*) idxq, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) poles, (void*) difl, (void*) difr, (void*) z, (void*) k, (void*) c, (void*) s, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd6.timings[0] += (helpTimeStop - helpTime);
    data->slasd6.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmql (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cunmql(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmql((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmql.timings[0] += (helpTimeStop - helpTime);
    data->cunmql.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqz2 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * ns, Int * nd, float complex* alpha, float complex* beta, float complex* qc, Int * ldqc, float complex* zc, Int * ldzc, float complex* work, Int * lwork, float* rwork, Int * rec, Int * info);
void hook_claqz2(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, Int * ns, Int * nd, float complex* alpha, float complex* beta, float complex* qc, Int * ldqc, float complex* zc, Int * ldzc, float complex* work, Int * lwork, float* rwork, Int * rec, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqz2((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nw, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) alpha, (void*) beta, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) rwork, (void*) rec, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claqz2.timings[0] += (helpTimeStop - helpTime);
    data->claqz2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dhgeqz (char* job, char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* t, Int * ldt, double* alphar, double* alphai, double* beta, double* q, Int * ldq, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_job, int len_compq, int len_compz);
void hook_dhgeqz(char* job, char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* h, Int * ldh, double* t, Int * ldt, double* alphar, double* alphai, double* beta, double* q, Int * ldq, double* z, Int * ldz, double* work, Int * lwork, Int * info, int len_job, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dhgeqz((void*) job, (void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) t, (void*) ldt, (void*) alphar, (void*) alphai, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_job, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->dhgeqz.timings[0] += (helpTimeStop - helpTime);
    data->dhgeqz.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarfgp (Int * n, double complex* alpha, double complex* x, Int * incx, double complex* tau);
void hook_zlarfgp(Int * n, double complex* alpha, double complex* x, Int * incx, double complex* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarfgp((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->zlarfgp.timings[0] += (helpTimeStop - helpTime);
    data->zlarfgp.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgels (char* trans, Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_trans);
void hook_zgels(char* trans, Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgels((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgels.timings[0] += (helpTimeStop - helpTime);
    data->zgels.calls[0]++;

    return ;
}



extern void flexiblas_chain_stprfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_stprfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stprfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stprfs.timings[0] += (helpTimeStop - helpTime);
    data->stprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_clatps (char* uplo, char* trans, char* diag, char* normin, Int * n, float complex* ap, float complex* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_clatps(char* uplo, char* trans, char* diag, char* normin, Int * n, float complex* ap, float complex* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clatps((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) ap, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->clatps.timings[0] += (helpTimeStop - helpTime);
    data->clatps.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetf2_rk (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, Int * info, int len_uplo);
void hook_chetf2_rk(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetf2_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->chetf2_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrs_rook (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dsytrs_rook(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrf_aa (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetrf_aa(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrf_aa((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqhb (char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_claqhb(char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqhb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->claqhb.timings[0] += (helpTimeStop - helpTime);
    data->claqhb.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgetrf (Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info);
void hook_sgetrf(Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgetrf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetrf.timings[0] += (helpTimeStop - helpTime);
    data->sgetrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbevx_2stage (char* jobz, char* range, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsbevx_2stage(char* jobz, char* range, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbevx_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbevx_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssygs2 (Int * itype, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo);
void hook_ssygs2(Int * itype, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssygs2((void*) itype, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssygs2.timings[0] += (helpTimeStop - helpTime);
    data->ssygs2.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqz0 (char* wants, char* wantq, char* wantz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * rec, Int * info, int len_wants, int len_wantq, int len_wantz);
void hook_claqz0(char* wants, char* wantq, char* wantz, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * rec, Int * info, int len_wants, int len_wantq, int len_wantz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqz0((void*) wants, (void*) wantq, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) rec, (void*) info, (int) len_wants, (int) len_wantq, (int) len_wantz);

    helpTimeStop = flexiblas_wtime();

    data->claqz0.timings[0] += (helpTimeStop - helpTime);
    data->claqz0.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlanst (char* norm, Int * n, double* d, double* e, int len_norm);
double hook_dlanst(char* norm, Int * n, double* d, double* e, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlanst((void*) norm, (void*) n, (void*) d, (void*) e, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->dlanst.timings[0] += (helpTimeStop - helpTime);
    data->dlanst.calls[0]++;

    return v;
}



extern void flexiblas_chain_zupmtr (char* side, char* uplo, char* trans, Int * m, Int * n, double complex* ap, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_uplo, int len_trans);
void hook_zupmtr(char* side, char* uplo, char* trans, Int * m, Int * n, double complex* ap, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zupmtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) ap, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zupmtr.timings[0] += (helpTimeStop - helpTime);
    data->zupmtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyev_2stage (char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_ssyev_2stage(char* jobz, char* uplo, Int * n, float* a, Int * lda, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyev_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyev_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_chegst (Int * itype, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chegst(Int * itype, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chegst((void*) itype, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chegst.timings[0] += (helpTimeStop - helpTime);
    data->chegst.calls[0]++;

    return ;
}



extern void flexiblas_chain_spteqr (char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_compz);
void hook_spteqr(char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spteqr((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->spteqr.timings[0] += (helpTimeStop - helpTime);
    data->spteqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtbcon (char* norm, char* uplo, char* diag, Int * n, Int * kd, double* ab, Int * ldab, double* rcond, double* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_dtbcon(char* norm, char* uplo, char* diag, Int * n, Int * kd, double* ab, Int * ldab, double* rcond, double* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtbcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtbcon.timings[0] += (helpTimeStop - helpTime);
    data->dtbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbcon (char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dpbcon(char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbcon((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpbcon.timings[0] += (helpTimeStop - helpTime);
    data->dpbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlasyf_rook (char* uplo, Int * n, Int * nb, Int * kb, double complex* a, Int * lda, Int * ipiv, double complex* w, Int * ldw, Int * info, int len_uplo);
void hook_zlasyf_rook(char* uplo, Int * n, Int * nb, Int * kb, double complex* a, Int * lda, Int * ipiv, double complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlasyf_rook((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->zlasyf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_chpgst (Int * itype, char* uplo, Int * n, float complex* ap, float complex* bp, Int * info, int len_uplo);
void hook_chpgst(Int * itype, char* uplo, Int * n, float complex* ap, float complex* bp, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chpgst((void*) itype, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chpgst.timings[0] += (helpTimeStop - helpTime);
    data->chpgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlabrd (Int * m, Int * n, Int * nb, double complex* a, Int * lda, double* d, double* e, double complex* tauq, double complex* taup, double complex* x, Int * ldx, double complex* y, Int * ldy);
void hook_zlabrd(Int * m, Int * n, Int * nb, double complex* a, Int * lda, double* d, double* e, double complex* tauq, double complex* taup, double complex* x, Int * ldx, double complex* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlabrd((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) x, (void*) ldx, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->zlabrd.timings[0] += (helpTimeStop - helpTime);
    data->zlabrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgemlq (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zgemlq(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgemlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgemlq.timings[0] += (helpTimeStop - helpTime);
    data->zgemlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgetf2 (Int * m, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info);
void hook_cgetf2(Int * m, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgetf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetf2.timings[0] += (helpTimeStop - helpTime);
    data->cgetf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhegst (Int * itype, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhegst(Int * itype, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhegst((void*) itype, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhegst.timings[0] += (helpTimeStop - helpTime);
    data->zhegst.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheevd_2stage (char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_cheevd_2stage(char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqp3 (Int * m, Int * n, double complex* a, Int * lda, Int * jpvt, double complex* tau, double complex* work, Int * lwork, double* rwork, Int * info);
void hook_zgeqp3(Int * m, Int * n, double complex* a, Int * lda, Int * jpvt, double complex* tau, double complex* work, Int * lwork, double* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqp3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) work, (void*) lwork, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->zgeqp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunghr (Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cunghr(Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunghr((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunghr.timings[0] += (helpTimeStop - helpTime);
    data->cunghr.calls[0]++;

    return ;
}



extern void flexiblas_chain_strsna (char* job, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, float* s, float* sep, Int * mm, Int * m, float* work, Int * ldwork, Int * iwork, Int * info, int len_job, int len_howmny);
void hook_strsna(char* job, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, float* s, float* sep, Int * mm, Int * m, float* work, Int * ldwork, Int * iwork, Int * info, int len_job, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strsna((void*) job, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) s, (void*) sep, (void*) mm, (void*) m, (void*) work, (void*) ldwork, (void*) iwork, (void*) info, (int) len_job, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->strsna.timings[0] += (helpTimeStop - helpTime);
    data->strsna.calls[0]++;

    return ;
}



extern void flexiblas_chain_dggsvp3 (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, double* a, Int * lda, double* b, Int * ldb, double* tola, double* tolb, Int * k, Int * l, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, Int * iwork, double* tau, double* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_dggsvp3(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, double* a, Int * lda, double* b, Int * ldb, double* tola, double* tolb, Int * k, Int * l, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, Int * iwork, double* tau, double* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggsvp3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) k, (void*) l, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) iwork, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->dggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->dggsvp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_clahqr (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float complex* h, Int * ldh, float complex* w, Int * iloz, Int * ihiz, float complex* z, Int * ldz, Int * info);
void hook_clahqr(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float complex* h, Int * ldh, float complex* w, Int * iloz, Int * ihiz, float complex* z, Int * ldz, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahqr((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->clahqr.timings[0] += (helpTimeStop - helpTime);
    data->clahqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtzrzf (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dtzrzf(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtzrzf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtzrzf.timings[0] += (helpTimeStop - helpTime);
    data->dtzrzf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsyr (char* uplo, Int * n, double complex* alpha, double complex* x, Int * incx, double complex* a, Int * lda, int len_uplo);
void hook_zsyr(char* uplo, Int * n, double complex* alpha, double complex* x, Int * incx, double complex* a, Int * lda, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsyr((void*) uplo, (void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) a, (void*) lda, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsyr.timings[0] += (helpTimeStop - helpTime);
    data->zsyr.calls[0]++;

    return ;
}



extern void flexiblas_chain_checon_3 (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo);
void hook_checon_3(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_checon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->checon_3.timings[0] += (helpTimeStop - helpTime);
    data->checon_3.calls[0]++;

    return ;
}



extern int flexiblas_chain_slaisnan (float* sin1, float* sin2);
int hook_slaisnan(float* sin1, float* sin2)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slaisnan((void*) sin1, (void*) sin2);

    helpTimeStop = flexiblas_wtime();

    data->slaisnan.timings[0] += (helpTimeStop - helpTime);
    data->slaisnan.calls[0]++;

    return v;
}



extern void flexiblas_chain_slarrf (Int * n, float* d, float* l, float* ld, Int * clstrt, Int * clend, float* w, float* wgap, float* werr, float* spdiam, float* clgapl, float* clgapr, float* pivmin, float* sigma, float* dplus, float* lplus, float* work, Int * info);
void hook_slarrf(Int * n, float* d, float* l, float* ld, Int * clstrt, Int * clend, float* w, float* wgap, float* werr, float* spdiam, float* clgapl, float* clgapr, float* pivmin, float* sigma, float* dplus, float* lplus, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarrf((void*) n, (void*) d, (void*) l, (void*) ld, (void*) clstrt, (void*) clend, (void*) w, (void*) wgap, (void*) werr, (void*) spdiam, (void*) clgapl, (void*) clgapr, (void*) pivmin, (void*) sigma, (void*) dplus, (void*) lplus, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrf.timings[0] += (helpTimeStop - helpTime);
    data->slarrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_clapll (Int * n, float complex* x, Int * incx, float complex* y, Int * incy, float* ssmin);
void hook_clapll(Int * n, float complex* x, Int * incx, float complex* y, Int * incy, float* ssmin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clapll((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->clapll.timings[0] += (helpTimeStop - helpTime);
    data->clapll.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed6 (Int * kniter, Int * orgati, float* rho, float* d, float* z, float* finit, float* tau, Int * info);
void hook_slaed6(Int * kniter, Int * orgati, float* rho, float* d, float* z, float* finit, float* tau, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed6((void*) kniter, (void*) orgati, (void*) rho, (void*) d, (void*) z, (void*) finit, (void*) tau, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed6.timings[0] += (helpTimeStop - helpTime);
    data->slaed6.calls[0]++;

    return ;
}



extern void flexiblas_chain_zspcon (char* uplo, Int * n, double complex* ap, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo);
void hook_zspcon(char* uplo, Int * n, double complex* ap, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zspcon((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zspcon.timings[0] += (helpTimeStop - helpTime);
    data->zspcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpev (char* jobz, char* uplo, Int * n, double complex* ap, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zhpev(char* jobz, char* uplo, Int * n, double complex* ap, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpev((void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpev.timings[0] += (helpTimeStop - helpTime);
    data->zhpev.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqsb (char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_claqsb(char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqsb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->claqsb.timings[0] += (helpTimeStop - helpTime);
    data->claqsb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmlq (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zunmlq(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmlq.timings[0] += (helpTimeStop - helpTime);
    data->zunmlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssb2st_kernels (char* uplo, Int * wantz, Int * ttype, Int * st, Int * ed, Int * sweep, Int * n, Int * nb, Int * ib, float* a, Int * lda, float* v, float* tau, Int * ldvt, float* work, int len_uplo);
void hook_ssb2st_kernels(char* uplo, Int * wantz, Int * ttype, Int * st, Int * ed, Int * sweep, Int * n, Int * nb, Int * ib, float* a, Int * lda, float* v, float* tau, Int * ldvt, float* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssb2st_kernels((void*) uplo, (void*) wantz, (void*) ttype, (void*) st, (void*) ed, (void*) sweep, (void*) n, (void*) nb, (void*) ib, (void*) a, (void*) lda, (void*) v, (void*) tau, (void*) ldvt, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssb2st_kernels.timings[0] += (helpTimeStop - helpTime);
    data->ssb2st_kernels.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbtrs (char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zpbtrs(char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbtrs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbtrs.timings[0] += (helpTimeStop - helpTime);
    data->zpbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqge (Int * m, Int * n, double* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed);
void hook_dlaqge(Int * m, Int * n, double* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqge((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dlaqge.timings[0] += (helpTimeStop - helpTime);
    data->dlaqge.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarft (char* direct, char* storev, Int * n, Int * k, double complex* v, Int * ldv, double complex* tau, double complex* t, Int * ldt, int len_direct, int len_storev);
void hook_zlarft(char* direct, char* storev, Int * n, Int * k, double complex* v, Int * ldv, double complex* tau, double complex* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarft((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->zlarft.timings[0] += (helpTimeStop - helpTime);
    data->zlarft.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbequ (char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, Int * info, int len_uplo);
void hook_zpbequ(char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbequ((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbequ.timings[0] += (helpTimeStop - helpTime);
    data->zpbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgl2 (Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * info);
void hook_dorgl2(Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgl2((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgl2.timings[0] += (helpTimeStop - helpTime);
    data->dorgl2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlar1v (Int * n, Int * b1, Int * bn, double* lambda, double* d, double* l, double* ld, double* lld, double* pivmin, double* gaptol, double* z, Int * wantnc, Int * negcnt, double* ztz, double* mingma, Int * r, Int * isuppz, double* nrminv, double* resid, double* rqcorr, double* work);
void hook_dlar1v(Int * n, Int * b1, Int * bn, double* lambda, double* d, double* l, double* ld, double* lld, double* pivmin, double* gaptol, double* z, Int * wantnc, Int * negcnt, double* ztz, double* mingma, Int * r, Int * isuppz, double* nrminv, double* resid, double* rqcorr, double* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlar1v((void*) n, (void*) b1, (void*) bn, (void*) lambda, (void*) d, (void*) l, (void*) ld, (void*) lld, (void*) pivmin, (void*) gaptol, (void*) z, (void*) wantnc, (void*) negcnt, (void*) ztz, (void*) mingma, (void*) r, (void*) isuppz, (void*) nrminv, (void*) resid, (void*) rqcorr, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->dlar1v.timings[0] += (helpTimeStop - helpTime);
    data->dlar1v.calls[0]++;

    return ;
}



extern void flexiblas_chain_clacrt (Int * n, float complex* cx, Int * incx, float complex* cy, Int * incy, float complex* c, float complex* s);
void hook_clacrt(Int * n, float complex* cx, Int * incx, float complex* cy, Int * incy, float complex* c, float complex* s)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clacrt((void*) n, (void*) cx, (void*) incx, (void*) cy, (void*) incy, (void*) c, (void*) s);

    helpTimeStop = flexiblas_wtime();

    data->clacrt.timings[0] += (helpTimeStop - helpTime);
    data->clacrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsprfs (char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zsprfs(char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsprfs.timings[0] += (helpTimeStop - helpTime);
    data->zsprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyconv (char* uplo, char* way, Int * n, double* a, Int * lda, Int * ipiv, double* e, Int * info, int len_uplo, int len_way);
void hook_dsyconv(char* uplo, char* way, Int * n, double* a, Int * lda, Int * ipiv, double* e, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyconv((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) e, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->dsyconv.timings[0] += (helpTimeStop - helpTime);
    data->dsyconv.calls[0]++;

    return ;
}



extern void flexiblas_chain_clacrm (Int * m, Int * n, float complex* a, Int * lda, float* b, Int * ldb, float complex* c, Int * ldc, float* rwork);
void hook_clacrm(Int * m, Int * n, float complex* a, Int * lda, float* b, Int * ldb, float complex* c, Int * ldc, float* rwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clacrm((void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) rwork);

    helpTimeStop = flexiblas_wtime();

    data->clacrm.timings[0] += (helpTimeStop - helpTime);
    data->clacrm.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggrqf (Int * m, Int * p, Int * n, double complex* a, Int * lda, double complex* taua, double complex* b, Int * ldb, double complex* taub, double complex* work, Int * lwork, Int * info);
void hook_zggrqf(Int * m, Int * p, Int * n, double complex* a, Int * lda, double complex* taua, double complex* b, Int * ldb, double complex* taub, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggrqf((void*) m, (void*) p, (void*) n, (void*) a, (void*) lda, (void*) taua, (void*) b, (void*) ldb, (void*) taub, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zggrqf.timings[0] += (helpTimeStop - helpTime);
    data->zggrqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytf2_rk (char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo);
void hook_ssytf2_rk(char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytf2_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssytf2_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungqr (Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zungqr(Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungqr((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungqr.timings[0] += (helpTimeStop - helpTime);
    data->zungqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dptsvx (char* fact, Int * n, Int * nrhs, double* d, double* e, double* df, double* ef, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * info, int len_fact);
void hook_dptsvx(char* fact, Int * n, Int * nrhs, double* d, double* e, double* df, double* ef, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * info, int len_fact)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dptsvx((void*) fact, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) info, (int) len_fact);

    helpTimeStop = flexiblas_wtime();

    data->dptsvx.timings[0] += (helpTimeStop - helpTime);
    data->dptsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slar2v (Int * n, float* x, float* y, float* z, Int * incx, float* c, float* s, Int * incc);
void hook_slar2v(Int * n, float* x, float* y, float* z, Int * incx, float* c, float* s, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slar2v((void*) n, (void*) x, (void*) y, (void*) z, (void*) incx, (void*) c, (void*) s, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->slar2v.timings[0] += (helpTimeStop - helpTime);
    data->slar2v.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrevc3 (char* side, char* howmny, Int * select, Int * n, double complex* t, Int * ldt, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_side, int len_howmny);
void hook_ztrevc3(char* side, char* howmny, Int * select, Int * n, double complex* t, Int * ldt, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrevc3((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ztrevc3.timings[0] += (helpTimeStop - helpTime);
    data->ztrevc3.calls[0]++;

    return ;
}



extern void flexiblas_chain_stgevc (char* side, char* howmny, Int * select, Int * n, float* s, Int * lds, float* p, Int * ldp, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * info, int len_side, int len_howmny);
void hook_stgevc(char* side, char* howmny, Int * select, Int * n, float* s, Int * lds, float* p, Int * ldp, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stgevc((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) s, (void*) lds, (void*) p, (void*) ldp, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->stgevc.timings[0] += (helpTimeStop - helpTime);
    data->stgevc.calls[0]++;

    return ;
}



extern void flexiblas_chain_claesy (float complex* a, float complex* b, float complex* c, float complex* rt1, float complex* rt2, float complex* evscal, float complex* cs1, float complex* sn1);
void hook_claesy(float complex* a, float complex* b, float complex* c, float complex* rt1, float complex* rt2, float complex* evscal, float complex* cs1, float complex* sn1)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claesy((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2, (void*) evscal, (void*) cs1, (void*) sn1);

    helpTimeStop = flexiblas_wtime();

    data->claesy.timings[0] += (helpTimeStop - helpTime);
    data->claesy.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorbdb2 (Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* work, Int * lwork, Int * info);
void hook_sorbdb2(Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x21, Int * ldx21, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorbdb2((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb2.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhsein (char* side, char* eigsrc, char* initv, Int * select, Int * n, double complex* h, Int * ldh, double complex* w, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv);
void hook_zhsein(char* side, char* eigsrc, char* initv, Int * select, Int * n, double complex* h, Int * ldh, double complex* w, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhsein((void*) side, (void*) eigsrc, (void*) initv, (void*) select, (void*) n, (void*) h, (void*) ldh, (void*) w, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) rwork, (void*) ifaill, (void*) ifailr, (void*) info, (int) len_side, (int) len_eigsrc, (int) len_initv);

    helpTimeStop = flexiblas_wtime();

    data->zhsein.timings[0] += (helpTimeStop - helpTime);
    data->zhsein.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgesv (Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info);
void hook_cgesv(Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgesv((void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgesv.timings[0] += (helpTimeStop - helpTime);
    data->cgesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyconvf_rook (char* uplo, char* way, Int * n, double* a, Int * lda, double* e, Int * ipiv, Int * info, int len_uplo, int len_way);
void hook_dsyconvf_rook(char* uplo, char* way, Int * n, double* a, Int * lda, double* e, Int * ipiv, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyconvf_rook((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->dsyconvf_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsyconvf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_csysv_rook (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csysv_rook(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csysv_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->csysv_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsyrfs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zsyrfs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsyrfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsyrfs.timings[0] += (helpTimeStop - helpTime);
    data->zsyrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarf (char* side, Int * m, Int * n, double* v, Int * incv, double* tau, double* c, Int * ldc, double* work, int len_side);
void hook_dlarf(char* side, Int * m, Int * n, double* v, Int * incv, double* tau, double* c, Int * ldc, double* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarf((void*) side, (void*) m, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->dlarf.timings[0] += (helpTimeStop - helpTime);
    data->dlarf.calls[0]++;

    return ;
}



extern float complex flexiblas_chain_cladiv (void *retvalue, float complex* x, float complex* y);
float complex hook_cladiv(float complex* x, float complex* y)
{
    float complex v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cladiv( (void*) &v , (void*) x, (void*) y);

    helpTimeStop = flexiblas_wtime();

    data->cladiv.timings[0] += (helpTimeStop - helpTime);
    data->cladiv.calls[0]++;

    return v;
}



extern void flexiblas_chain_sorghr (Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sorghr(Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorghr((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorghr.timings[0] += (helpTimeStop - helpTime);
    data->sorghr.calls[0]++;

    return ;
}



extern void flexiblas_chain_stftri (char* transr, char* uplo, char* diag, Int * n, float* a, Int * info, int len_transr, int len_uplo, int len_diag);
void hook_stftri(char* transr, char* uplo, char* diag, Int * n, float* a, Int * info, int len_transr, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stftri((void*) transr, (void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stftri.timings[0] += (helpTimeStop - helpTime);
    data->stftri.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasyf (char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo);
void hook_slasyf(char* uplo, Int * n, Int * nb, Int * kb, float* a, Int * lda, Int * ipiv, float* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasyf((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slasyf.timings[0] += (helpTimeStop - helpTime);
    data->slasyf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormql (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dormql(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormql((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormql.timings[0] += (helpTimeStop - helpTime);
    data->dormql.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetri_3 (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetri_3(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetri_3.timings[0] += (helpTimeStop - helpTime);
    data->zhetri_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cungtr (char* uplo, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_cungtr(char* uplo, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungtr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cungtr.timings[0] += (helpTimeStop - helpTime);
    data->cungtr.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlantp (char* norm, char* uplo, char* diag, Int * n, double* ap, double* work, int len_norm, int len_uplo, int len_diag);
double hook_dlantp(char* norm, char* uplo, char* diag, Int * n, double* ap, double* work, int len_norm, int len_uplo, int len_diag)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlantp((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dlantp.timings[0] += (helpTimeStop - helpTime);
    data->dlantp.calls[0]++;

    return v;
}



extern void flexiblas_chain_clahr2 (Int * n, Int * k, Int * nb, float complex* a, Int * lda, float complex* tau, float complex* t, Int * ldt, float complex* y, Int * ldy);
void hook_clahr2(Int * n, Int * k, Int * nb, float complex* a, Int * lda, float complex* tau, float complex* t, Int * ldt, float complex* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahr2((void*) n, (void*) k, (void*) nb, (void*) a, (void*) lda, (void*) tau, (void*) t, (void*) ldt, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->clahr2.timings[0] += (helpTimeStop - helpTime);
    data->clahr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_csyswapr (char* uplo, Int * n, float complex* a, Int * lda, Int * i1, Int * i2, int len_uplo);
void hook_csyswapr(char* uplo, Int * n, float complex* a, Int * lda, Int * i1, Int * i2, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csyswapr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) i1, (void*) i2, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csyswapr.timings[0] += (helpTimeStop - helpTime);
    data->csyswapr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpotrs (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_cpotrs(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpotrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpotrs.timings[0] += (helpTimeStop - helpTime);
    data->cpotrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sormr2 (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans);
void hook_sormr2(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sormr2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sormr2.timings[0] += (helpTimeStop - helpTime);
    data->sormr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlartgp (double* f, double* g, double* cs, double* sn, double* r);
void hook_dlartgp(double* f, double* g, double* cs, double* sn, double* r)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlartgp((void*) f, (void*) g, (void*) cs, (void*) sn, (void*) r);

    helpTimeStop = flexiblas_wtime();

    data->dlartgp.timings[0] += (helpTimeStop - helpTime);
    data->dlartgp.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgerq2 (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sgerq2(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgerq2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgerq2.timings[0] += (helpTimeStop - helpTime);
    data->sgerq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrttp (char* uplo, Int * n, double complex* a, Int * lda, double complex* ap, Int * info, int len_uplo);
void hook_ztrttp(char* uplo, Int * n, double complex* a, Int * lda, double complex* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrttp((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ztrttp.timings[0] += (helpTimeStop - helpTime);
    data->ztrttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_slae2 (float* a, float* b, float* c, float* rt1, float* rt2);
void hook_slae2(float* a, float* b, float* c, float* rt1, float* rt2)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slae2((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2);

    helpTimeStop = flexiblas_wtime();

    data->slae2.timings[0] += (helpTimeStop - helpTime);
    data->slae2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlar1v (Int * n, Int * b1, Int * bn, double* lambda, double* d, double* l, double* ld, double* lld, double* pivmin, double* gaptol, double complex* z, Int * wantnc, Int * negcnt, double* ztz, double* mingma, Int * r, Int * isuppz, double* nrminv, double* resid, double* rqcorr, double* work);
void hook_zlar1v(Int * n, Int * b1, Int * bn, double* lambda, double* d, double* l, double* ld, double* lld, double* pivmin, double* gaptol, double complex* z, Int * wantnc, Int * negcnt, double* ztz, double* mingma, Int * r, Int * isuppz, double* nrminv, double* resid, double* rqcorr, double* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlar1v((void*) n, (void*) b1, (void*) bn, (void*) lambda, (void*) d, (void*) l, (void*) ld, (void*) lld, (void*) pivmin, (void*) gaptol, (void*) z, (void*) wantnc, (void*) negcnt, (void*) ztz, (void*) mingma, (void*) r, (void*) isuppz, (void*) nrminv, (void*) resid, (void*) rqcorr, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->zlar1v.timings[0] += (helpTimeStop - helpTime);
    data->zlar1v.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgr2 (Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sorgr2(Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgr2((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgr2.timings[0] += (helpTimeStop - helpTime);
    data->sorgr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpftri (char* transr, char* uplo, Int * n, float complex* a, Int * info, int len_transr, int len_uplo);
void hook_cpftri(char* transr, char* uplo, Int * n, float complex* a, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpftri((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpftri.timings[0] += (helpTimeStop - helpTime);
    data->cpftri.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrs_aa_2stage (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chetrs_aa_2stage(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrs_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_claunhr_col_getrfnp2 (Int * m, Int * n, float complex* a, Int * lda, float complex* d, Int * info);
void hook_claunhr_col_getrfnp2(Int * m, Int * n, float complex* a, Int * lda, float complex* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claunhr_col_getrfnp2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claunhr_col_getrfnp2.timings[0] += (helpTimeStop - helpTime);
    data->claunhr_col_getrfnp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpptrs (char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zpptrs(char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpptrs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpptrs.timings[0] += (helpTimeStop - helpTime);
    data->zpptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgtr (char* uplo, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info, int len_uplo);
void hook_sorgtr(char* uplo, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgtr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sorgtr.timings[0] += (helpTimeStop - helpTime);
    data->sorgtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqr (Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* work, Int * lwork, Int * info);
void hook_cgeqr(Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqr((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqr.timings[0] += (helpTimeStop - helpTime);
    data->cgeqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgqr (Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sorgqr(Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgqr((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgqr.timings[0] += (helpTimeStop - helpTime);
    data->sorgqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgesvx (char* fact, char* trans, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, char* equed, float* r, float* c, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans, int len_equed);
void hook_cgesvx(char* fact, char* trans, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, char* equed, float* r, float* c, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgesvx((void*) fact, (void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) equed, (void*) r, (void*) c, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_trans, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->cgesvx.timings[0] += (helpTimeStop - helpTime);
    data->cgesvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cptts2 (Int * iuplo, Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb);
void hook_cptts2(Int * iuplo, Int * n, Int * nrhs, float* d, float complex* e, float complex* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cptts2((void*) iuplo, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->cptts2.timings[0] += (helpTimeStop - helpTime);
    data->cptts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgsy2 (char* trans, Int * ijob, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double complex* d, Int * ldd, double complex* e, Int * lde, double complex* f, Int * ldf, double* scale, double* rdsum, double* rdscal, Int * info, int len_trans);
void hook_ztgsy2(char* trans, Int * ijob, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double complex* d, Int * ldd, double complex* e, Int * lde, double complex* f, Int * ldf, double* scale, double* rdsum, double* rdscal, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgsy2((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) rdsum, (void*) rdscal, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ztgsy2.timings[0] += (helpTimeStop - helpTime);
    data->ztgsy2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqsb (char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_zlaqsb(char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqsb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqsb.timings[0] += (helpTimeStop - helpTime);
    data->zlaqsb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgghrd (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * info, int len_compq, int len_compz);
void hook_dgghrd(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgghrd((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->dgghrd.timings[0] += (helpTimeStop - helpTime);
    data->dgghrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgesc2 (Int * n, float complex* a, Int * lda, float complex* rhs, Int * ipiv, Int * jpiv, float* scale);
void hook_cgesc2(Int * n, float complex* a, Int * lda, float complex* rhs, Int * ipiv, Int * jpiv, float* scale)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgesc2((void*) n, (void*) a, (void*) lda, (void*) rhs, (void*) ipiv, (void*) jpiv, (void*) scale);

    helpTimeStop = flexiblas_wtime();

    data->cgesc2.timings[0] += (helpTimeStop - helpTime);
    data->cgesc2.calls[0]++;

    return ;
}



extern void flexiblas_chain_clatrs3 (char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* x, Int * ldx, float* scale, float* cnorm, float* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_clatrs3(char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* x, Int * ldx, float* scale, float* cnorm, float* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clatrs3((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) x, (void*) ldx, (void*) scale, (void*) cnorm, (void*) work, (void*) lwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->clatrs3.timings[0] += (helpTimeStop - helpTime);
    data->clatrs3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrti2 (char* uplo, char* diag, Int * n, double complex* a, Int * lda, Int * info, int len_uplo, int len_diag);
void hook_ztrti2(char* uplo, char* diag, Int * n, double complex* a, Int * lda, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrti2((void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztrti2.timings[0] += (helpTimeStop - helpTime);
    data->ztrti2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggqrf (Int * n, Int * m, Int * p, float complex* a, Int * lda, float complex* taua, float complex* b, Int * ldb, float complex* taub, float complex* work, Int * lwork, Int * info);
void hook_cggqrf(Int * n, Int * m, Int * p, float complex* a, Int * lda, float complex* taua, float complex* b, Int * ldb, float complex* taub, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggqrf((void*) n, (void*) m, (void*) p, (void*) a, (void*) lda, (void*) taua, (void*) b, (void*) ldb, (void*) taub, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cggqrf.timings[0] += (helpTimeStop - helpTime);
    data->cggqrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorm2l (char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_trans);
void hook_dorm2l(char* side, char* trans, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorm2l((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dorm2l.timings[0] += (helpTimeStop - helpTime);
    data->dorm2l.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgebrd (Int * m, Int * n, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* work, Int * lwork, Int * info);
void hook_dgebrd(Int * m, Int * n, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgebrd((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgebrd.timings[0] += (helpTimeStop - helpTime);
    data->dgebrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungtsqr (Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info);
void hook_zungtsqr(Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungtsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungtsqr.timings[0] += (helpTimeStop - helpTime);
    data->zungtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgecon (char* norm, Int * n, float* a, Int * lda, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_norm);
void hook_sgecon(char* norm, Int * n, float* a, Int * lda, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgecon((void*) norm, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->sgecon.timings[0] += (helpTimeStop - helpTime);
    data->sgecon.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbtf2 (Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, Int * info);
void hook_cgbtf2(Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbtf2((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbtf2.timings[0] += (helpTimeStop - helpTime);
    data->cgbtf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeequb (Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_sgeequb(Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeequb((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeequb.timings[0] += (helpTimeStop - helpTime);
    data->sgeequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbevd (char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_chbevd(char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbevd((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbevd.timings[0] += (helpTimeStop - helpTime);
    data->chbevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorbdb (char* trans, char* signs, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x12, Int * ldx12, float* x21, Int * ldx21, float* x22, Int * ldx22, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* tauq2, float* work, Int * lwork, Int * info, int len_trans, int len_signs);
void hook_sorbdb(char* trans, char* signs, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x12, Int * ldx12, float* x21, Int * ldx21, float* x22, Int * ldx22, float* theta, float* phi, float* taup1, float* taup2, float* tauq1, float* tauq2, float* work, Int * lwork, Int * info, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorbdb((void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) tauq2, (void*) work, (void*) lwork, (void*) info, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb.calls[0]++;

    return ;
}



extern void flexiblas_chain_sstein (Int * n, float* d, float* e, Int * m, float* w, Int * iblock, Int * isplit, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info);
void hook_sstein(Int * n, float* d, float* e, Int * m, float* w, Int * iblock, Int * isplit, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sstein((void*) n, (void*) d, (void*) e, (void*) m, (void*) w, (void*) iblock, (void*) isplit, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sstein.timings[0] += (helpTimeStop - helpTime);
    data->sstein.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgesc2 (Int * n, float* a, Int * lda, float* rhs, Int * ipiv, Int * jpiv, float* scale);
void hook_sgesc2(Int * n, float* a, Int * lda, float* rhs, Int * ipiv, Int * jpiv, float* scale)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgesc2((void*) n, (void*) a, (void*) lda, (void*) rhs, (void*) ipiv, (void*) jpiv, (void*) scale);

    helpTimeStop = flexiblas_wtime();

    data->sgesc2.timings[0] += (helpTimeStop - helpTime);
    data->sgesc2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgerfs (char* trans, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans);
void hook_dgerfs(char* trans, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgerfs((void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgerfs.timings[0] += (helpTimeStop - helpTime);
    data->dgerfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqr2 (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zgeqr2(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqr2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqr2.timings[0] += (helpTimeStop - helpTime);
    data->zgeqr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_clahef (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clahef(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahef((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clahef.timings[0] += (helpTimeStop - helpTime);
    data->clahef.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqr (Int * m, Int * n, double* a, Int * lda, double* t, Int * tsize, double* work, Int * lwork, Int * info);
void hook_dgeqr(Int * m, Int * n, double* a, Int * lda, double* t, Int * tsize, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqr((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_chprfs (char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* afp, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_chprfs(char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* afp, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chprfs.timings[0] += (helpTimeStop - helpTime);
    data->chprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyevx_2stage (char* jobz, char* range, char* uplo, Int * n, float* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_ssyevx_2stage(char* jobz, char* range, char* uplo, Int * n, float* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyevx_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevx_2stage.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlangb (char* norm, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* work, int len_norm);
double hook_dlangb(char* norm, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, double* work, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlangb((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->dlangb.timings[0] += (helpTimeStop - helpTime);
    data->dlangb.calls[0]++;

    return v;
}



extern void flexiblas_chain_dsytrf_rook (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrf_rook(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrf_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgerqf (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cgerqf(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgerqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgerqf.timings[0] += (helpTimeStop - helpTime);
    data->cgerqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slagv2 (float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* csl, float* snl, float* csr, float* snr);
void hook_slagv2(float* a, Int * lda, float* b, Int * ldb, float* alphar, float* alphai, float* beta, float* csl, float* snl, float* csr, float* snr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slagv2((void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) csl, (void*) snl, (void*) csr, (void*) snr);

    helpTimeStop = flexiblas_wtime();

    data->slagv2.timings[0] += (helpTimeStop - helpTime);
    data->slagv2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgtsvx (char* fact, char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* dlf, float complex* df, float complex* duf, float complex* du2, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans);
void hook_cgtsvx(char* fact, char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* dlf, float complex* df, float complex* duf, float complex* du2, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgtsvx((void*) fact, (void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->cgtsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyev (char* jobz, char* uplo, Int * n, double* a, Int * lda, double* w, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_dsyev(char* jobz, char* uplo, Int * n, double* a, Int * lda, double* w, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyev((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyev.timings[0] += (helpTimeStop - helpTime);
    data->dsyev.calls[0]++;

    return ;
}



extern void flexiblas_chain_slags2 (Int * upper, float* a1, float* a2, float* a3, float* b1, float* b2, float* b3, float* csu, float* snu, float* csv, float* snv, float* csq, float* snq);
void hook_slags2(Int * upper, float* a1, float* a2, float* a3, float* b1, float* b2, float* b3, float* csu, float* snu, float* csv, float* snv, float* csq, float* snq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slags2((void*) upper, (void*) a1, (void*) a2, (void*) a3, (void*) b1, (void*) b2, (void*) b3, (void*) csu, (void*) snu, (void*) csv, (void*) snv, (void*) csq, (void*) snq);

    helpTimeStop = flexiblas_wtime();

    data->slags2.timings[0] += (helpTimeStop - helpTime);
    data->slags2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarfy (char* uplo, Int * n, double* v, Int * incv, double* tau, double* c, Int * ldc, double* work, int len_uplo);
void hook_dlarfy(char* uplo, Int * n, double* v, Int * incv, double* tau, double* c, Int * ldc, double* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarfy((void*) uplo, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlarfy.timings[0] += (helpTimeStop - helpTime);
    data->dlarfy.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqz2 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * ns, Int * nd, double complex* alpha, double complex* beta, double complex* qc, Int * ldqc, double complex* zc, Int * ldzc, double complex* work, Int * lwork, double* rwork, Int * rec, Int * info);
void hook_zlaqz2(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * ns, Int * nd, double complex* alpha, double complex* beta, double complex* qc, Int * ldqc, double complex* zc, Int * ldzc, double complex* work, Int * lwork, double* rwork, Int * rec, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqz2((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nw, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) alpha, (void*) beta, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) rwork, (void*) rec, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqz2.timings[0] += (helpTimeStop - helpTime);
    data->zlaqz2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqhe (char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_zlaqhe(char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqhe((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqhe.timings[0] += (helpTimeStop - helpTime);
    data->zlaqhe.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgges (char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * sdim, double* alphar, double* alphai, double* beta, double* vsl, Int * ldvsl, double* vsr, Int * ldvsr, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort);
void hook_dgges(char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * sdim, double* alphar, double* alphai, double* beta, double* vsl, Int * ldvsl, double* vsr, Int * ldvsr, double* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgges((void*) jobvsl, (void*) jobvsr, (void*) sort, (void*) selctg, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sdim, (void*) alphar, (void*) alphai, (void*) beta, (void*) vsl, (void*) ldvsl, (void*) vsr, (void*) ldvsr, (void*) work, (void*) lwork, (void*) bwork, (void*) info, (int) len_jobvsl, (int) len_jobvsr, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->dgges.timings[0] += (helpTimeStop - helpTime);
    data->dgges.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlarmm (double* anorm, double* bnorm, double* cnorm);
double hook_dlarmm(double* anorm, double* bnorm, double* cnorm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlarmm((void*) anorm, (void*) bnorm, (void*) cnorm);

    helpTimeStop = flexiblas_wtime();

    data->dlarmm.timings[0] += (helpTimeStop - helpTime);
    data->dlarmm.calls[0]++;

    return v;
}



extern void flexiblas_chain_sorbdb6 (Int * m1, Int * m2, Int * n, float* x1, Int * incx1, float* x2, Int * incx2, float* q1, Int * ldq1, float* q2, Int * ldq2, float* work, Int * lwork, Int * info);
void hook_sorbdb6(Int * m1, Int * m2, Int * n, float* x1, Int * incx1, float* x2, Int * incx2, float* q1, Int * ldq1, float* q2, Int * ldq2, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorbdb6((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb6.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb6.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyevr_2stage (char* jobz, char* range, char* uplo, Int * n, float* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_ssyevr_2stage(char* jobz, char* range, char* uplo, Int * n, float* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyevr_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevr_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri2 (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytri2(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri2.timings[0] += (helpTimeStop - helpTime);
    data->csytri2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlacon (Int * n, double complex* v, double complex* x, double* est, Int * kase);
void hook_zlacon(Int * n, double complex* v, double complex* x, double* est, Int * kase)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlacon((void*) n, (void*) v, (void*) x, (void*) est, (void*) kase);

    helpTimeStop = flexiblas_wtime();

    data->zlacon.timings[0] += (helpTimeStop - helpTime);
    data->zlacon.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytf2_rk (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, Int * info, int len_uplo);
void hook_csytf2_rk(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytf2_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->csytf2_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqsb (char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_dlaqsb(char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqsb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsb.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsb.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgesv (Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info);
void hook_sgesv(Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgesv((void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgesv.timings[0] += (helpTimeStop - helpTime);
    data->sgesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarfy (char* uplo, Int * n, double complex* v, Int * incv, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_uplo);
void hook_zlarfy(char* uplo, Int * n, double complex* v, Int * incv, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarfy((void*) uplo, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlarfy.timings[0] += (helpTimeStop - helpTime);
    data->zlarfy.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbevx (char* jobz, char* range, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zhbevx(char* jobz, char* range, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbevx.timings[0] += (helpTimeStop - helpTime);
    data->zhbevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorm22 (char* side, char* trans, Int * m, Int * n, Int * n1, Int * n2, float* q, Int * ldq, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_sorm22(char* side, char* trans, Int * m, Int * n, Int * n1, Int * n2, float* q, Int * ldq, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorm22((void*) side, (void*) trans, (void*) m, (void*) n, (void*) n1, (void*) n2, (void*) q, (void*) ldq, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sorm22.timings[0] += (helpTimeStop - helpTime);
    data->sorm22.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpsv (char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhpsv(char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpsv.timings[0] += (helpTimeStop - helpTime);
    data->zhpsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpstrf (char* uplo, Int * n, double complex* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo);
void hook_zpstrf(char* uplo, Int * n, double complex* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpstrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpstrf.timings[0] += (helpTimeStop - helpTime);
    data->zpstrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungr2 (Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zungr2(Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungr2((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungr2.timings[0] += (helpTimeStop - helpTime);
    data->zungr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cstein (Int * n, float* d, float* e, Int * m, float* w, Int * iblock, Int * isplit, float complex* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info);
void hook_cstein(Int * n, float* d, float* e, Int * m, float* w, Int * iblock, Int * isplit, float complex* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cstein((void*) n, (void*) d, (void*) e, (void*) m, (void*) w, (void*) iblock, (void*) isplit, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cstein.timings[0] += (helpTimeStop - helpTime);
    data->cstein.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmr3 (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zunmr3(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmr3((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmr3.timings[0] += (helpTimeStop - helpTime);
    data->zunmr3.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqsb (char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_slaqsb(char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqsb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->slaqsb.timings[0] += (helpTimeStop - helpTime);
    data->slaqsb.calls[0]++;

    return ;
}



extern void flexiblas_chain_slauum (char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo);
void hook_slauum(char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slauum((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slauum.timings[0] += (helpTimeStop - helpTime);
    data->slauum.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunghr (Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zunghr(Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunghr((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zunghr.timings[0] += (helpTimeStop - helpTime);
    data->zunghr.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlantr (char* norm, char* uplo, char* diag, Int * m, Int * n, double complex* a, Int * lda, double* work, int len_norm, int len_uplo, int len_diag);
double hook_zlantr(char* norm, char* uplo, char* diag, Int * m, Int * n, double complex* a, Int * lda, double* work, int len_norm, int len_uplo, int len_diag)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlantr((void*) norm, (void*) uplo, (void*) diag, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->zlantr.timings[0] += (helpTimeStop - helpTime);
    data->zlantr.calls[0]++;

    return v;
}



extern void flexiblas_chain_slaqz4 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, float* sr, float* si, float* ss, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, float* qc, Int * ldqc, float* zc, Int * ldzc, float* work, Int * lwork, Int * info);
void hook_slaqz4(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, float* sr, float* si, float* ss, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, float* qc, Int * ldqc, float* zc, Int * ldzc, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqz4((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nshifts, (void*) nblock_desired, (void*) sr, (void*) si, (void*) ss, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqz4.timings[0] += (helpTimeStop - helpTime);
    data->slaqz4.calls[0]++;

    return ;
}



extern void flexiblas_chain_stpmlqt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * mb, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * info, int len_side, int len_trans);
void hook_stpmlqt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * mb, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stpmlqt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) mb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->stpmlqt.timings[0] += (helpTimeStop - helpTime);
    data->stpmlqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgels (char* trans, Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_trans);
void hook_sgels(char* trans, Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgels((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgels.timings[0] += (helpTimeStop - helpTime);
    data->sgels.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpocon (char* uplo, Int * n, double* a, Int * lda, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dpocon(char* uplo, Int * n, double* a, Int * lda, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpocon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpocon.timings[0] += (helpTimeStop - helpTime);
    data->dpocon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytri2 (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytri2(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytri2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytri2.timings[0] += (helpTimeStop - helpTime);
    data->dsytri2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd2 (Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* z, double* alpha, double* beta, double* u, Int * ldu, double* vt, Int * ldvt, double* dsigma, double* u2, Int * ldu2, double* vt2, Int * ldvt2, Int * idxp, Int * idx, Int * idxc, Int * idxq, Int * coltyp, Int * info);
void hook_dlasd2(Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* z, double* alpha, double* beta, double* u, Int * ldu, double* vt, Int * ldvt, double* dsigma, double* u2, Int * ldu2, double* vt2, Int * ldvt2, Int * idxp, Int * idx, Int * idxc, Int * idxq, Int * coltyp, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd2((void*) nl, (void*) nr, (void*) sqre, (void*) k, (void*) d, (void*) z, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) dsigma, (void*) u2, (void*) ldu2, (void*) vt2, (void*) ldvt2, (void*) idxp, (void*) idx, (void*) idxc, (void*) idxq, (void*) coltyp, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd2.timings[0] += (helpTimeStop - helpTime);
    data->dlasd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgehd2 (Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info);
void hook_cgehd2(Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgehd2((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgehd2.timings[0] += (helpTimeStop - helpTime);
    data->cgehd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtgsja (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, double* a, Int * lda, double* b, Int * ldb, double* tola, double* tolb, double* alpha, double* beta, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, double* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_dtgsja(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, double* a, Int * lda, double* b, Int * ldb, double* tola, double* tolb, double* alpha, double* beta, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, double* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtgsja((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) ncycle, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->dtgsja.timings[0] += (helpTimeStop - helpTime);
    data->dtgsja.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarrb (Int * n, float* d, float* lld, Int * ifirst, Int * ilast, float* rtol1, float* rtol2, Int * offset, float* w, float* wgap, float* werr, float* work, Int * iwork, float* pivmin, float* spdiam, Int * twist, Int * info);
void hook_slarrb(Int * n, float* d, float* lld, Int * ifirst, Int * ilast, float* rtol1, float* rtol2, Int * offset, float* w, float* wgap, float* werr, float* work, Int * iwork, float* pivmin, float* spdiam, Int * twist, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarrb((void*) n, (void*) d, (void*) lld, (void*) ifirst, (void*) ilast, (void*) rtol1, (void*) rtol2, (void*) offset, (void*) w, (void*) wgap, (void*) werr, (void*) work, (void*) iwork, (void*) pivmin, (void*) spdiam, (void*) twist, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrb.timings[0] += (helpTimeStop - helpTime);
    data->slarrb.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasyf_aa (char* uplo, Int * j1, Int * m, Int * nb, float* a, Int * lda, Int * ipiv, float* h, Int * ldh, float* work, int len_uplo);
void hook_slasyf_aa(char* uplo, Int * j1, Int * m, Int * nb, float* a, Int * lda, Int * ipiv, float* h, Int * ldh, float* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasyf_aa((void*) uplo, (void*) j1, (void*) m, (void*) nb, (void*) a, (void*) lda, (void*) ipiv, (void*) h, (void*) ldh, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_aa.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_aa.calls[0]++;

    return ;
}



extern float flexiblas_chain_scsum1 (Int * n, float complex* cx, Int * incx);
float hook_scsum1(Int * n, float complex* cx, Int * incx)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_scsum1((void*) n, (void*) cx, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->scsum1.timings[0] += (helpTimeStop - helpTime);
    data->scsum1.calls[0]++;

    return v;
}



extern void flexiblas_chain_zheev_2stage (char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zheev_2stage(char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheev_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zheev_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqr0 (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float* h, Int * ldh, float* wr, float* wi, Int * iloz, Int * ihiz, float* z, Int * ldz, float* work, Int * lwork, Int * info);
void hook_slaqr0(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, float* h, Int * ldh, float* wr, float* wi, Int * iloz, Int * ihiz, float* z, Int * ldz, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqr0((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqr0.timings[0] += (helpTimeStop - helpTime);
    data->slaqr0.calls[0]++;

    return ;
}



extern double flexiblas_chain_dsecnd (void);
double hook_dsecnd(void)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dsecnd();

    helpTimeStop = flexiblas_wtime();

    data->dsecnd.timings[0] += (helpTimeStop - helpTime);
    data->dsecnd.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlamtsqr (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dlamtsqr(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlamtsqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dlamtsqr.timings[0] += (helpTimeStop - helpTime);
    data->dlamtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbevd (char* jobz, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_ssbevd(char* jobz, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbevd((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbevd.timings[0] += (helpTimeStop - helpTime);
    data->ssbevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztplqt2 (Int * m, Int * n, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, Int * info);
void hook_ztplqt2(Int * m, Int * n, Int * l, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztplqt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ztplqt2.timings[0] += (helpTimeStop - helpTime);
    data->ztplqt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytf2 (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_dsytf2(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytf2.timings[0] += (helpTimeStop - helpTime);
    data->dsytf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrs (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo);
void hook_ssytrs(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgglse (Int * m, Int * n, Int * p, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, float complex* d, float complex* x, float complex* work, Int * lwork, Int * info);
void hook_cgglse(Int * m, Int * n, Int * p, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, float complex* d, float complex* x, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgglse((void*) m, (void*) n, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) d, (void*) x, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgglse.timings[0] += (helpTimeStop - helpTime);
    data->cgglse.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrs_3 (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chetrs_3(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrs_3((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_3.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cungtsqr (Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info);
void hook_cungtsqr(Int * m, Int * n, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungtsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cungtsqr.timings[0] += (helpTimeStop - helpTime);
    data->cungtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgtsqr (Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info);
void hook_dorgtsqr(Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgtsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgtsqr.timings[0] += (helpTimeStop - helpTime);
    data->dorgtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbgst (char* vect, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* x, Int * ldx, float* work, Int * info, int len_vect, int len_uplo);
void hook_ssbgst(char* vect, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* x, Int * ldx, float* work, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbgst((void*) vect, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) x, (void*) ldx, (void*) work, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbgst.timings[0] += (helpTimeStop - helpTime);
    data->ssbgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqr2 (Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, float* h, Int * ldh, Int * iloz, Int * ihiz, float* z, Int * ldz, Int * ns, Int * nd, float* sr, float* si, float* v, Int * ldv, Int * nh, float* t, Int * ldt, Int * nv, float* wv, Int * ldwv, float* work, Int * lwork);
void hook_slaqr2(Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, float* h, Int * ldh, Int * iloz, Int * ihiz, float* z, Int * ldz, Int * ns, Int * nd, float* sr, float* si, float* v, Int * ldv, Int * nh, float* t, Int * ldt, Int * nv, float* wv, Int * ldwv, float* work, Int * lwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqr2((void*) wantt, (void*) wantz, (void*) n, (void*) ktop, (void*) kbot, (void*) nw, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) sr, (void*) si, (void*) v, (void*) ldv, (void*) nh, (void*) t, (void*) ldt, (void*) nv, (void*) wv, (void*) ldwv, (void*) work, (void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->slaqr2.timings[0] += (helpTimeStop - helpTime);
    data->slaqr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slargv (Int * n, float* x, Int * incx, float* y, Int * incy, float* c, Int * incc);
void hook_slargv(Int * n, float* x, Int * incx, float* y, Int * incy, float* c, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slargv((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) c, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->slargv.timings[0] += (helpTimeStop - helpTime);
    data->slargv.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrexc (char* compq, Int * n, float complex* t, Int * ldt, float complex* q, Int * ldq, Int * ifst, Int * ilst, Int * info, int len_compq);
void hook_ctrexc(char* compq, Int * n, float complex* t, Int * ldt, float complex* q, Int * ldq, Int * ifst, Int * ilst, Int * info, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrexc((void*) compq, (void*) n, (void*) t, (void*) ldt, (void*) q, (void*) ldq, (void*) ifst, (void*) ilst, (void*) info, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->ctrexc.timings[0] += (helpTimeStop - helpTime);
    data->ctrexc.calls[0]++;

    return ;
}



extern float flexiblas_chain_slansy (char* norm, char* uplo, Int * n, float* a, Int * lda, float* work, int len_norm, int len_uplo);
float hook_slansy(char* norm, char* uplo, Int * n, float* a, Int * lda, float* work, int len_norm, int len_uplo)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slansy((void*) norm, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slansy.timings[0] += (helpTimeStop - helpTime);
    data->slansy.calls[0]++;

    return v;
}



extern void flexiblas_chain_cungr2 (Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info);
void hook_cungr2(Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungr2((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cungr2.timings[0] += (helpTimeStop - helpTime);
    data->cungr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaic1 (Int * job, Int * j, double* x, double* sest, double* w, double* gamma, double* sestpr, double* s, double* c);
void hook_dlaic1(Int * job, Int * j, double* x, double* sest, double* w, double* gamma, double* sestpr, double* s, double* c)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaic1((void*) job, (void*) j, (void*) x, (void*) sest, (void*) w, (void*) gamma, (void*) sestpr, (void*) s, (void*) c);

    helpTimeStop = flexiblas_wtime();

    data->dlaic1.timings[0] += (helpTimeStop - helpTime);
    data->dlaic1.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctftri (char* transr, char* uplo, char* diag, Int * n, float complex* a, Int * info, int len_transr, int len_uplo, int len_diag);
void hook_ctftri(char* transr, char* uplo, char* diag, Int * n, float complex* a, Int * info, int len_transr, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctftri((void*) transr, (void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctftri.timings[0] += (helpTimeStop - helpTime);
    data->ctftri.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgghrd (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * info, int len_compq, int len_compz);
void hook_zgghrd(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgghrd((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->zgghrd.timings[0] += (helpTimeStop - helpTime);
    data->zgghrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmrq (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zunmrq(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmrq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmrq.timings[0] += (helpTimeStop - helpTime);
    data->zunmrq.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggqrf (Int * n, Int * m, Int * p, double complex* a, Int * lda, double complex* taua, double complex* b, Int * ldb, double complex* taub, double complex* work, Int * lwork, Int * info);
void hook_zggqrf(Int * n, Int * m, Int * p, double complex* a, Int * lda, double complex* taua, double complex* b, Int * ldb, double complex* taub, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggqrf((void*) n, (void*) m, (void*) p, (void*) a, (void*) lda, (void*) taua, (void*) b, (void*) ldb, (void*) taub, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zggqrf.timings[0] += (helpTimeStop - helpTime);
    data->zggqrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_sormql (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_sormql(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sormql((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sormql.timings[0] += (helpTimeStop - helpTime);
    data->sormql.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelqt (Int * m, Int * n, Int * mb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * info);
void hook_sgelqt(Int * m, Int * n, Int * mb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelqt((void*) m, (void*) n, (void*) mb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqt.timings[0] += (helpTimeStop - helpTime);
    data->sgelqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetsqrhrt (Int * m, Int * n, Int * mb1, Int * nb1, Int * nb2, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info);
void hook_dgetsqrhrt(Int * m, Int * n, Int * mb1, Int * nb1, Int * nb2, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetsqrhrt((void*) m, (void*) n, (void*) mb1, (void*) nb1, (void*) nb2, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetsqrhrt.timings[0] += (helpTimeStop - helpTime);
    data->dgetsqrhrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbevx_2stage (char* jobz, char* range, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float complex* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_chbevx_2stage(char* jobz, char* range, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float complex* q, Int * ldq, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbevx_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbevx_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbequ (Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_cgbequ(Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbequ((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbequ.timings[0] += (helpTimeStop - helpTime);
    data->cgbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasd1 (Int * nl, Int * nr, Int * sqre, float* d, float* alpha, float* beta, float* u, Int * ldu, float* vt, Int * ldvt, Int * idxq, Int * iwork, float* work, Int * info);
void hook_slasd1(Int * nl, Int * nr, Int * sqre, float* d, float* alpha, float* beta, float* u, Int * ldu, float* vt, Int * ldvt, Int * idxq, Int * iwork, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasd1((void*) nl, (void*) nr, (void*) sqre, (void*) d, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) idxq, (void*) iwork, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd1.timings[0] += (helpTimeStop - helpTime);
    data->slasd1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlapll (Int * n, double* x, Int * incx, double* y, Int * incy, double* ssmin);
void hook_dlapll(Int * n, double* x, Int * incx, double* y, Int * incy, double* ssmin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlapll((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->dlapll.timings[0] += (helpTimeStop - helpTime);
    data->dlapll.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtftri (char* transr, char* uplo, char* diag, Int * n, double* a, Int * info, int len_transr, int len_uplo, int len_diag);
void hook_dtftri(char* transr, char* uplo, char* diag, Int * n, double* a, Int * info, int len_transr, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtftri((void*) transr, (void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtftri.timings[0] += (helpTimeStop - helpTime);
    data->dtftri.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaorhr_col_getrfnp (Int * m, Int * n, double* a, Int * lda, double* d, Int * info);
void hook_dlaorhr_col_getrfnp(Int * m, Int * n, double* a, Int * lda, double* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaorhr_col_getrfnp((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaorhr_col_getrfnp.timings[0] += (helpTimeStop - helpTime);
    data->dlaorhr_col_getrfnp.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed1 (Int * n, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* work, Int * iwork, Int * info);
void hook_dlaed1(Int * n, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed1((void*) n, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed1.timings[0] += (helpTimeStop - helpTime);
    data->dlaed1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlamswlq (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dlamswlq(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlamswlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dlamswlq.timings[0] += (helpTimeStop - helpTime);
    data->dlamswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytri2x (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * nb, Int * info, int len_uplo);
void hook_ssytri2x(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri2x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri2x.timings[0] += (helpTimeStop - helpTime);
    data->ssytri2x.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlacp2 (char* uplo, Int * m, Int * n, double* a, Int * lda, double complex* b, Int * ldb, int len_uplo);
void hook_zlacp2(char* uplo, Int * m, Int * n, double* a, Int * lda, double complex* b, Int * ldb, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlacp2((void*) uplo, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlacp2.timings[0] += (helpTimeStop - helpTime);
    data->zlacp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaev2 (float* a, float* b, float* c, float* rt1, float* rt2, float* cs1, float* sn1);
void hook_slaev2(float* a, float* b, float* c, float* rt1, float* rt2, float* cs1, float* sn1)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaev2((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2, (void*) cs1, (void*) sn1);

    helpTimeStop = flexiblas_wtime();

    data->slaev2.timings[0] += (helpTimeStop - helpTime);
    data->slaev2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmhr (char* side, char* trans, Int * m, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cunmhr(char* side, char* trans, Int * m, Int * n, Int * ilo, Int * ihi, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmhr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmhr.timings[0] += (helpTimeStop - helpTime);
    data->cunmhr.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrf (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetrf(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrf.timings[0] += (helpTimeStop - helpTime);
    data->chetrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed8 (Int * icompq, Int * k, Int * n, Int * qsiz, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* z, double* dlambda, double* q2, Int * ldq2, double* w, Int * perm, Int * givptr, Int * givcol, double* givnum, Int * indxp, Int * indx, Int * info);
void hook_dlaed8(Int * icompq, Int * k, Int * n, Int * qsiz, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* z, double* dlambda, double* q2, Int * ldq2, double* w, Int * perm, Int * givptr, Int * givcol, double* givnum, Int * indxp, Int * indx, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed8((void*) icompq, (void*) k, (void*) n, (void*) qsiz, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) z, (void*) dlambda, (void*) q2, (void*) ldq2, (void*) w, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) indxp, (void*) indx, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed8.timings[0] += (helpTimeStop - helpTime);
    data->dlaed8.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd6 (Int * icompq, Int * nl, Int * nr, Int * sqre, double* d, double* vf, double* vl, double* alpha, double* beta, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* poles, double* difl, double* difr, double* z, Int * k, double* c, double* s, double* work, Int * iwork, Int * info);
void hook_dlasd6(Int * icompq, Int * nl, Int * nr, Int * sqre, double* d, double* vf, double* vl, double* alpha, double* beta, Int * idxq, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* poles, double* difl, double* difr, double* z, Int * k, double* c, double* s, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd6((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) d, (void*) vf, (void*) vl, (void*) alpha, (void*) beta, (void*) idxq, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) poles, (void*) difl, (void*) difr, (void*) z, (void*) k, (void*) c, (void*) s, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd6.timings[0] += (helpTimeStop - helpTime);
    data->dlasd6.calls[0]++;

    return ;
}



extern float flexiblas_chain_slangb (char* norm, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, float* work, int len_norm);
float hook_slangb(char* norm, Int * n, Int * kl, Int * ku, float* ab, Int * ldab, float* work, int len_norm)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slangb((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->slangb.timings[0] += (helpTimeStop - helpTime);
    data->slangb.calls[0]++;

    return v;
}



extern void flexiblas_chain_ctpttr (char* uplo, Int * n, float complex* ap, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_ctpttr(char* uplo, Int * n, float complex* ap, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpttr((void*) uplo, (void*) n, (void*) ap, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ctpttr.timings[0] += (helpTimeStop - helpTime);
    data->ctpttr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrexc (char* compq, Int * n, double* t, Int * ldt, double* q, Int * ldq, Int * ifst, Int * ilst, double* work, Int * info, int len_compq);
void hook_dtrexc(char* compq, Int * n, double* t, Int * ldt, double* q, Int * ldq, Int * ifst, Int * ilst, double* work, Int * info, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrexc((void*) compq, (void*) n, (void*) t, (void*) ldt, (void*) q, (void*) ldq, (void*) ifst, (void*) ilst, (void*) work, (void*) info, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->dtrexc.timings[0] += (helpTimeStop - helpTime);
    data->dtrexc.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssycon (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo);
void hook_ssycon(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssycon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssycon.timings[0] += (helpTimeStop - helpTime);
    data->ssycon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasq5 (Int * i0, Int * n0, double* z, Int * pp, double* tau, double* sigma, double* dmin, double* dmin1, double* dmin2, double* dn, double* dnm1, double* dnm2, Int * ieee, double* eps);
void hook_dlasq5(Int * i0, Int * n0, double* z, Int * pp, double* tau, double* sigma, double* dmin, double* dmin1, double* dmin2, double* dn, double* dnm1, double* dnm2, Int * ieee, double* eps)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasq5((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) tau, (void*) sigma, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dnm1, (void*) dnm2, (void*) ieee, (void*) eps);

    helpTimeStop = flexiblas_wtime();

    data->dlasq5.timings[0] += (helpTimeStop - helpTime);
    data->dlasq5.calls[0]++;

    return ;
}



extern void flexiblas_chain_strttp (char* uplo, Int * n, float* a, Int * lda, float* ap, Int * info, int len_uplo);
void hook_strttp(char* uplo, Int * n, float* a, Int * lda, float* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strttp((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->strttp.timings[0] += (helpTimeStop - helpTime);
    data->strttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqp2rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float complex* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float complex* tau, float* vn1, float* vn2, float complex* work, Int * info);
void hook_claqp2rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float complex* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float complex* tau, float* vn1, float* vn2, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqp2rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) kmax, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claqp2rk.timings[0] += (helpTimeStop - helpTime);
    data->claqp2rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgemqr (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cgemqr(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgemqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgemqr.timings[0] += (helpTimeStop - helpTime);
    data->cgemqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtpqrt (Int * m, Int * n, Int * l, Int * nb, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, double* work, Int * info);
void hook_dtpqrt(Int * m, Int * n, Int * l, Int * nb, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtpqrt((void*) m, (void*) n, (void*) l, (void*) nb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtpqrt.timings[0] += (helpTimeStop - helpTime);
    data->dtpqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlagtm (char* trans, Int * n, Int * nrhs, double* alpha, double complex* dl, double complex* d, double complex* du, double complex* x, Int * ldx, double* beta, double complex* b, Int * ldb, int len_trans);
void hook_zlagtm(char* trans, Int * n, Int * nrhs, double* alpha, double complex* dl, double complex* d, double complex* du, double complex* x, Int * ldx, double* beta, double complex* b, Int * ldb, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlagtm((void*) trans, (void*) n, (void*) nrhs, (void*) alpha, (void*) dl, (void*) d, (void*) du, (void*) x, (void*) ldx, (void*) beta, (void*) b, (void*) ldb, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zlagtm.timings[0] += (helpTimeStop - helpTime);
    data->zlagtm.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarzt (char* direct, char* storev, Int * n, Int * k, float* v, Int * ldv, float* tau, float* t, Int * ldt, int len_direct, int len_storev);
void hook_slarzt(char* direct, char* storev, Int * n, Int * k, float* v, Int * ldv, float* tau, float* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarzt((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->slarzt.timings[0] += (helpTimeStop - helpTime);
    data->slarzt.calls[0]++;

    return ;
}



extern void flexiblas_chain_stpqrt (Int * m, Int * n, Int * l, Int * nb, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, float* work, Int * info);
void hook_stpqrt(Int * m, Int * n, Int * l, Int * nb, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stpqrt((void*) m, (void*) n, (void*) l, (void*) nb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stpqrt.timings[0] += (helpTimeStop - helpTime);
    data->stpqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgemqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans);
void hook_cgemqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgemqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->cgemqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgrq (Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dorgrq(Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgrq((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgrq.timings[0] += (helpTimeStop - helpTime);
    data->dorgrq.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgtsvx (char* fact, char* trans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* dlf, double* df, double* duf, double* du2, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_trans);
void hook_dgtsvx(char* fact, char* trans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* dlf, double* df, double* duf, double* du2, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgtsvx((void*) fact, (void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->dgtsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbgvx (char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, double complex* ab, Int * ldab, double complex* bb, Int * ldbb, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zhbgvx(char* jobz, char* range, char* uplo, Int * n, Int * ka, Int * kb, double complex* ab, Int * ldab, double complex* bb, Int * ldbb, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbgvx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbgvx.timings[0] += (helpTimeStop - helpTime);
    data->zhbgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarnv (Int * idist, Int * iseed, Int * n, float* x);
void hook_slarnv(Int * idist, Int * iseed, Int * n, float* x)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarnv((void*) idist, (void*) iseed, (void*) n, (void*) x);

    helpTimeStop = flexiblas_wtime();

    data->slarnv.timings[0] += (helpTimeStop - helpTime);
    data->slarnv.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlantb (char* norm, char* uplo, char* diag, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo, int len_diag);
double hook_zlantb(char* norm, char* uplo, char* diag, Int * n, Int * k, double complex* ab, Int * ldab, double* work, int len_norm, int len_uplo, int len_diag)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlantb((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) k, (void*) ab, (void*) ldab, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->zlantb.timings[0] += (helpTimeStop - helpTime);
    data->zlantb.calls[0]++;

    return v;
}



extern void flexiblas_chain_stpmqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * info, int len_side, int len_trans);
void hook_stpmqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stpmqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->stpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->stpmqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_claev2 (float complex* a, float complex* b, float complex* c, float* rt1, float* rt2, float* cs1, float complex* sn1);
void hook_claev2(float complex* a, float complex* b, float complex* c, float* rt1, float* rt2, float* cs1, float complex* sn1)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claev2((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2, (void*) cs1, (void*) sn1);

    helpTimeStop = flexiblas_wtime();

    data->claev2.timings[0] += (helpTimeStop - helpTime);
    data->claev2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spptri (char* uplo, Int * n, float* ap, Int * info, int len_uplo);
void hook_spptri(char* uplo, Int * n, float* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spptri((void*) uplo, (void*) n, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spptri.timings[0] += (helpTimeStop - helpTime);
    data->spptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqr5 (Int * wantt, Int * wantz, Int * kacc22, Int * n, Int * ktop, Int * kbot, Int * nshfts, double complex* s, double complex* h, Int * ldh, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* v, Int * ldv, double complex* u, Int * ldu, Int * nv, double complex* wv, Int * ldwv, Int * nh, double complex* wh, Int * ldwh);
void hook_zlaqr5(Int * wantt, Int * wantz, Int * kacc22, Int * n, Int * ktop, Int * kbot, Int * nshfts, double complex* s, double complex* h, Int * ldh, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* v, Int * ldv, double complex* u, Int * ldu, Int * nv, double complex* wv, Int * ldwv, Int * nh, double complex* wh, Int * ldwh)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqr5((void*) wantt, (void*) wantz, (void*) kacc22, (void*) n, (void*) ktop, (void*) kbot, (void*) nshfts, (void*) s, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) v, (void*) ldv, (void*) u, (void*) ldu, (void*) nv, (void*) wv, (void*) ldwv, (void*) nh, (void*) wh, (void*) ldwh);

    helpTimeStop = flexiblas_wtime();

    data->zlaqr5.timings[0] += (helpTimeStop - helpTime);
    data->zlaqr5.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrf_aa_2stage (char* uplo, Int * n, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsytrf_aa_2stage(char* uplo, Int * n, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrf_aa_2stage((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zsytrf_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelqt (Int * m, Int * n, Int * mb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * info);
void hook_zgelqt(Int * m, Int * n, Int * mb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelqt((void*) m, (void*) n, (void*) mb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelqt.timings[0] += (helpTimeStop - helpTime);
    data->zgelqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctptrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ctptrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctptrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctptrs.timings[0] += (helpTimeStop - helpTime);
    data->ctptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssprfs (char* uplo, Int * n, Int * nrhs, float* ap, float* afp, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo);
void hook_ssprfs(char* uplo, Int * n, Int * nrhs, float* ap, float* afp, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssprfs.timings[0] += (helpTimeStop - helpTime);
    data->ssprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqr2p (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sgeqr2p(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqr2p((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->sgeqr2p.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunbdb5 (Int * m1, Int * m2, Int * n, double complex* x1, Int * incx1, double complex* x2, Int * incx2, double complex* q1, Int * ldq1, double complex* q2, Int * ldq2, double complex* work, Int * lwork, Int * info);
void hook_zunbdb5(Int * m1, Int * m2, Int * n, double complex* x1, Int * incx1, double complex* x2, Int * incx2, double complex* q1, Int * ldq1, double complex* q2, Int * ldq2, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunbdb5((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zunbdb5.timings[0] += (helpTimeStop - helpTime);
    data->zunbdb5.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlahef_aa (char* uplo, Int * j1, Int * m, Int * nb, double complex* a, Int * lda, Int * ipiv, double complex* h, Int * ldh, double complex* work, int len_uplo);
void hook_zlahef_aa(char* uplo, Int * j1, Int * m, Int * nb, double complex* a, Int * lda, Int * ipiv, double complex* h, Int * ldh, double complex* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlahef_aa((void*) uplo, (void*) j1, (void*) m, (void*) nb, (void*) a, (void*) lda, (void*) ipiv, (void*) h, (void*) ldh, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlahef_aa.timings[0] += (helpTimeStop - helpTime);
    data->zlahef_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaein (Int * rightv, Int * noinit, Int * n, double* h, Int * ldh, double* wr, double* wi, double* vr, double* vi, double* b, Int * ldb, double* work, double* eps3, double* smlnum, double* bignum, Int * info);
void hook_dlaein(Int * rightv, Int * noinit, Int * n, double* h, Int * ldh, double* wr, double* wi, double* vr, double* vi, double* b, Int * ldb, double* work, double* eps3, double* smlnum, double* bignum, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaein((void*) rightv, (void*) noinit, (void*) n, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) vr, (void*) vi, (void*) b, (void*) ldb, (void*) work, (void*) eps3, (void*) smlnum, (void*) bignum, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaein.timings[0] += (helpTimeStop - helpTime);
    data->dlaein.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssysv_rk (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* e, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssysv_rk(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* e, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssysv_rk((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctbtrs (char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ctbtrs(char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctbtrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctbtrs.timings[0] += (helpTimeStop - helpTime);
    data->ctbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtfttp (char* transr, char* uplo, Int * n, double* arf, double* ap, Int * info, int len_transr, int len_uplo);
void hook_dtfttp(char* transr, char* uplo, Int * n, double* arf, double* ap, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtfttp((void*) transr, (void*) uplo, (void*) n, (void*) arf, (void*) ap, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dtfttp.timings[0] += (helpTimeStop - helpTime);
    data->dtfttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarnv (Int * idist, Int * iseed, Int * n, float complex* x);
void hook_clarnv(Int * idist, Int * iseed, Int * n, float complex* x)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarnv((void*) idist, (void*) iseed, (void*) n, (void*) x);

    helpTimeStop = flexiblas_wtime();

    data->clarnv.timings[0] += (helpTimeStop - helpTime);
    data->clarnv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrs (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dsytrs(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sggglm (Int * n, Int * m, Int * p, float* a, Int * lda, float* b, Int * ldb, float* d, float* x, float* y, float* work, Int * lwork, Int * info);
void hook_sggglm(Int * n, Int * m, Int * p, float* a, Int * lda, float* b, Int * ldb, float* d, float* x, float* y, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sggglm((void*) n, (void*) m, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) d, (void*) x, (void*) y, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sggglm.timings[0] += (helpTimeStop - helpTime);
    data->sggglm.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeesx (char* jobvs, char* sort, Int * select, char* sense, Int * n, float* a, Int * lda, Int * sdim, float* wr, float* wi, float* vs, Int * ldvs, float* rconde, float* rcondv, float* work, Int * lwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense);
void hook_sgeesx(char* jobvs, char* sort, Int * select, char* sense, Int * n, float* a, Int * lda, Int * sdim, float* wr, float* wi, float* vs, Int * ldvs, float* rconde, float* rcondv, float* work, Int * lwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeesx((void*) jobvs, (void*) sort, (void*) select, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) wr, (void*) wi, (void*) vs, (void*) ldvs, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->sgeesx.timings[0] += (helpTimeStop - helpTime);
    data->sgeesx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctbcon (char* norm, char* uplo, char* diag, Int * n, Int * kd, float complex* ab, Int * ldab, float* rcond, float complex* work, float* rwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_ctbcon(char* norm, char* uplo, char* diag, Int * n, Int * kd, float complex* ab, Int * ldab, float* rcond, float complex* work, float* rwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctbcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctbcon.timings[0] += (helpTimeStop - helpTime);
    data->ctbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgesv (Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info);
void hook_dgesv(Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgesv((void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgesv.timings[0] += (helpTimeStop - helpTime);
    data->dgesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlapmt (Int * forwrd, Int * m, Int * n, double complex* x, Int * ldx, Int * k);
void hook_zlapmt(Int * forwrd, Int * m, Int * n, double complex* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlapmt((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->zlapmt.timings[0] += (helpTimeStop - helpTime);
    data->zlapmt.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctplqt2 (Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, Int * info);
void hook_ctplqt2(Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctplqt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctplqt2.timings[0] += (helpTimeStop - helpTime);
    data->ctplqt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgerq2 (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zgerq2(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgerq2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgerq2.timings[0] += (helpTimeStop - helpTime);
    data->zgerq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_stbtrs (char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_stbtrs(char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stbtrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stbtrs.timings[0] += (helpTimeStop - helpTime);
    data->stbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarfg (Int * n, float complex* alpha, float complex* x, Int * incx, float complex* tau);
void hook_clarfg(Int * n, float complex* alpha, float complex* x, Int * incx, float complex* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarfg((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->clarfg.timings[0] += (helpTimeStop - helpTime);
    data->clarfg.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytri2 (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsytri2(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytri2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytri2.timings[0] += (helpTimeStop - helpTime);
    data->zsytri2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorcsd2by1 (char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t);
void hook_dorcsd2by1(char* jobu1, char* jobu2, char* jobv1t, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorcsd2by1((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t);

    helpTimeStop = flexiblas_wtime();

    data->dorcsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->dorcsd2by1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrttp (char* uplo, Int * n, double* a, Int * lda, double* ap, Int * info, int len_uplo);
void hook_dtrttp(char* uplo, Int * n, double* a, Int * lda, double* ap, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrttp((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ap, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dtrttp.timings[0] += (helpTimeStop - helpTime);
    data->dtrttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhesv_rook (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhesv_rook(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhesv_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhesv_rook.timings[0] += (helpTimeStop - helpTime);
    data->zhesv_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmql (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zunmql(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmql((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmql.timings[0] += (helpTimeStop - helpTime);
    data->zunmql.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggevx (char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * ilo, Int * ihi, float* lscale, float* rscale, float* abnrm, float* bbnrm, float* rconde, float* rcondv, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * bwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense);
void hook_cggevx(char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * ilo, Int * ihi, float* lscale, float* rscale, float* abnrm, float* bbnrm, float* rconde, float* rcondv, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * bwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggevx((void*) balanc, (void*) jobvl, (void*) jobvr, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) ilo, (void*) ihi, (void*) lscale, (void*) rscale, (void*) abnrm, (void*) bbnrm, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) bwork, (void*) info, (int) len_balanc, (int) len_jobvl, (int) len_jobvr, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->cggevx.timings[0] += (helpTimeStop - helpTime);
    data->cggevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsysv (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsysv(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsysv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsysv.timings[0] += (helpTimeStop - helpTime);
    data->zsysv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dggrqf (Int * m, Int * p, Int * n, double* a, Int * lda, double* taua, double* b, Int * ldb, double* taub, double* work, Int * lwork, Int * info);
void hook_dggrqf(Int * m, Int * p, Int * n, double* a, Int * lda, double* taua, double* b, Int * ldb, double* taub, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggrqf((void*) m, (void*) p, (void*) n, (void*) a, (void*) lda, (void*) taua, (void*) b, (void*) ldb, (void*) taub, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dggrqf.timings[0] += (helpTimeStop - helpTime);
    data->dggrqf.calls[0]++;

    return ;
}



extern float flexiblas_chain_slapy2 (float* x, float* y);
float hook_slapy2(float* x, float* y)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slapy2((void*) x, (void*) y);

    helpTimeStop = flexiblas_wtime();

    data->slapy2.timings[0] += (helpTimeStop - helpTime);
    data->slapy2.calls[0]++;

    return v;
}



extern void flexiblas_chain_cgecon (char* norm, Int * n, float complex* a, Int * lda, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_norm);
void hook_cgecon(char* norm, Int * n, float complex* a, Int * lda, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgecon((void*) norm, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->cgecon.timings[0] += (helpTimeStop - helpTime);
    data->cgecon.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytri_rook (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * info, int len_uplo);
void hook_ssytri_rook(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgemlqt (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zgemlqt(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgemlqt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgemlqt.timings[0] += (helpTimeStop - helpTime);
    data->zgemlqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dporfs (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dporfs(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dporfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dporfs.timings[0] += (helpTimeStop - helpTime);
    data->dporfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtbrfs (char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_dtbrfs(char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtbrfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dtbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarfx (char* side, Int * m, Int * n, double* v, double* tau, double* c, Int * ldc, double* work, int len_side);
void hook_dlarfx(char* side, Int * m, Int * n, double* v, double* tau, double* c, Int * ldc, double* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarfx((void*) side, (void*) m, (void*) n, (void*) v, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->dlarfx.timings[0] += (helpTimeStop - helpTime);
    data->dlarfx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlagts (Int * job, Int * n, double* a, double* b, double* c, double* d, Int * in, double* y, double* tol, Int * info);
void hook_dlagts(Int * job, Int * n, double* a, double* b, double* c, double* d, Int * in, double* y, double* tol, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlagts((void*) job, (void*) n, (void*) a, (void*) b, (void*) c, (void*) d, (void*) in, (void*) y, (void*) tol, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlagts.timings[0] += (helpTimeStop - helpTime);
    data->dlagts.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlauum (char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo);
void hook_zlauum(char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlauum((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlauum.timings[0] += (helpTimeStop - helpTime);
    data->zlauum.calls[0]++;

    return ;
}



extern void flexiblas_chain_zptts2 (Int * iuplo, Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb);
void hook_zptts2(Int * iuplo, Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zptts2((void*) iuplo, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->zptts2.timings[0] += (helpTimeStop - helpTime);
    data->zptts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpgvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, double complex* ap, double complex* bp, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zhpgvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, double complex* ap, double complex* bp, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpgvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpgvx.timings[0] += (helpTimeStop - helpTime);
    data->zhpgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgbr (char* vect, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info, int len_vect);
void hook_sorgbr(char* vect, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgbr((void*) vect, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->sorgbr.timings[0] += (helpTimeStop - helpTime);
    data->sorgbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetf2_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_chetf2_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetf2_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetf2_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssbtrd (char* vect, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* d, float* e, float* q, Int * ldq, float* work, Int * info, int len_vect, int len_uplo);
void hook_ssbtrd(char* vect, char* uplo, Int * n, Int * kd, float* ab, Int * ldab, float* d, float* e, float* q, Int * ldq, float* work, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbtrd((void*) vect, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbtrd.timings[0] += (helpTimeStop - helpTime);
    data->ssbtrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrs_aa_2stage (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhetrs_aa_2stage(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrs_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhetrs_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrs_3 (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_csytrs_3(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrs_3((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_3.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpttrs (char* uplo, Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zpttrs(char* uplo, Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpttrs((void*) uplo, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpttrs.timings[0] += (helpTimeStop - helpTime);
    data->zpttrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_clapmt (Int * forwrd, Int * m, Int * n, float complex* x, Int * ldx, Int * k);
void hook_clapmt(Int * forwrd, Int * m, Int * n, float complex* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clapmt((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->clapmt.timings[0] += (helpTimeStop - helpTime);
    data->clapmt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarzb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_zlarzb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarzb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) l, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->zlarzb.timings[0] += (helpTimeStop - helpTime);
    data->zlarzb.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyconvf (char* uplo, char* way, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo, int len_way);
void hook_ssyconvf(char* uplo, char* way, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyconvf((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->ssyconvf.timings[0] += (helpTimeStop - helpTime);
    data->ssyconvf.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlange (char* norm, Int * m, Int * n, double complex* a, Int * lda, double* work, int len_norm);
double hook_zlange(char* norm, Int * m, Int * n, double complex* a, Int * lda, double* work, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlange((void*) norm, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->zlange.timings[0] += (helpTimeStop - helpTime);
    data->zlange.calls[0]++;

    return v;
}



extern void flexiblas_chain_dpttrs (Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, Int * info);
void hook_dpttrs(Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpttrs((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dpttrs.timings[0] += (helpTimeStop - helpTime);
    data->dpttrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbstf (char* uplo, Int * n, Int * kd, double* ab, Int * ldab, Int * info, int len_uplo);
void hook_dpbstf(char* uplo, Int * n, Int * kd, double* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbstf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpbstf.timings[0] += (helpTimeStop - helpTime);
    data->dpbstf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgbbrd (char* vect, Int * m, Int * n, Int * ncc, Int * kl, Int * ku, double complex* ab, Int * ldab, double* d, double* e, double complex* q, Int * ldq, double complex* pt, Int * ldpt, double complex* c, Int * ldc, double complex* work, double* rwork, Int * info, int len_vect);
void hook_zgbbrd(char* vect, Int * m, Int * n, Int * ncc, Int * kl, Int * ku, double complex* ab, Int * ldab, double* d, double* e, double complex* q, Int * ldq, double complex* pt, Int * ldpt, double complex* c, Int * ldc, double complex* work, double* rwork, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgbbrd((void*) vect, (void*) m, (void*) n, (void*) ncc, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) pt, (void*) ldpt, (void*) c, (void*) ldc, (void*) work, (void*) rwork, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->zgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->zgbbrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_clasr (char* side, char* pivot, char* direct, Int * m, Int * n, float* c, float* s, float complex* a, Int * lda, int len_side, int len_pivot, int len_direct);
void hook_clasr(char* side, char* pivot, char* direct, Int * m, Int * n, float* c, float* s, float complex* a, Int * lda, int len_side, int len_pivot, int len_direct)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clasr((void*) side, (void*) pivot, (void*) direct, (void*) m, (void*) n, (void*) c, (void*) s, (void*) a, (void*) lda, (int) len_side, (int) len_pivot, (int) len_direct);

    helpTimeStop = flexiblas_wtime();

    data->clasr.timings[0] += (helpTimeStop - helpTime);
    data->clasr.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqp2 (Int * m, Int * n, Int * offset, float* a, Int * lda, Int * jpvt, float* tau, float* vn1, float* vn2, float* work);
void hook_slaqp2(Int * m, Int * n, Int * offset, float* a, Int * lda, Int * jpvt, float* tau, float* vn1, float* vn2, float* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqp2((void*) m, (void*) n, (void*) offset, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->slaqp2.timings[0] += (helpTimeStop - helpTime);
    data->slaqp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spotrs (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo);
void hook_spotrs(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spotrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spotrs.timings[0] += (helpTimeStop - helpTime);
    data->spotrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggesx (char* jobvsl, char* jobvsr, char* sort, Int * selctg, char* sense, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * sdim, double complex* alpha, double complex* beta, double complex* vsl, Int * ldvsl, double complex* vsr, Int * ldvsr, double* rconde, double* rcondv, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort, int len_sense);
void hook_zggesx(char* jobvsl, char* jobvsr, char* sort, Int * selctg, char* sense, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * sdim, double complex* alpha, double complex* beta, double complex* vsl, Int * ldvsl, double complex* vsr, Int * ldvsr, double* rconde, double* rcondv, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggesx((void*) jobvsl, (void*) jobvsr, (void*) sort, (void*) selctg, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sdim, (void*) alpha, (void*) beta, (void*) vsl, (void*) ldvsl, (void*) vsr, (void*) ldvsr, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) liwork, (void*) bwork, (void*) info, (int) len_jobvsl, (int) len_jobvsr, (int) len_sort, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->zggesx.timings[0] += (helpTimeStop - helpTime);
    data->zggesx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sggsvp3 (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, float* a, Int * lda, float* b, Int * ldb, float* tola, float* tolb, Int * k, Int * l, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, Int * iwork, float* tau, float* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_sggsvp3(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, float* a, Int * lda, float* b, Int * ldb, float* tola, float* tolb, Int * k, Int * l, float* u, Int * ldu, float* v, Int * ldv, float* q, Int * ldq, Int * iwork, float* tau, float* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sggsvp3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) k, (void*) l, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) iwork, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->sggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->sggsvp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_strttf (char* transr, char* uplo, Int * n, float* a, Int * lda, float* arf, Int * info, int len_transr, int len_uplo);
void hook_strttf(char* transr, char* uplo, Int * n, float* a, Int * lda, float* arf, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strttf((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) arf, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->strttf.timings[0] += (helpTimeStop - helpTime);
    data->strttf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhesv_aa_2stage (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhesv_aa_2stage(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* tb, Int * ltb, Int * ipiv, Int * ipiv2, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhesv_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhesv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhesv_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_clahef_aa (char* uplo, Int * j1, Int * m, Int * nb, float complex* a, Int * lda, Int * ipiv, float complex* h, Int * ldh, float complex* work, int len_uplo);
void hook_clahef_aa(char* uplo, Int * j1, Int * m, Int * nb, float complex* a, Int * lda, Int * ipiv, float complex* h, Int * ldh, float complex* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahef_aa((void*) uplo, (void*) j1, (void*) m, (void*) nb, (void*) a, (void*) lda, (void*) ipiv, (void*) h, (void*) ldh, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clahef_aa.timings[0] += (helpTimeStop - helpTime);
    data->clahef_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbtrd (char* vect, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* d, double* e, double complex* q, Int * ldq, double complex* work, Int * info, int len_vect, int len_uplo);
void hook_zhbtrd(char* vect, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* d, double* e, double complex* q, Int * ldq, double complex* work, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbtrd((void*) vect, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbtrd.timings[0] += (helpTimeStop - helpTime);
    data->zhbtrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_stgsyl (char* trans, Int * ijob, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* d, Int * ldd, float* e, Int * lde, float* f, Int * ldf, float* scale, float* dif, float* work, Int * lwork, Int * iwork, Int * info, int len_trans);
void hook_stgsyl(char* trans, Int * ijob, Int * m, Int * n, float* a, Int * lda, float* b, Int * ldb, float* c, Int * ldc, float* d, Int * ldd, float* e, Int * lde, float* f, Int * ldf, float* scale, float* dif, float* work, Int * lwork, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stgsyl((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->stgsyl.timings[0] += (helpTimeStop - helpTime);
    data->stgsyl.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarft (char* direct, char* storev, Int * n, Int * k, float complex* v, Int * ldv, float complex* tau, float complex* t, Int * ldt, int len_direct, int len_storev);
void hook_clarft(char* direct, char* storev, Int * n, Int * k, float complex* v, Int * ldv, float complex* tau, float complex* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarft((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->clarft.timings[0] += (helpTimeStop - helpTime);
    data->clarft.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqz3 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, float complex* alpha, float complex* beta, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* qc, Int * ldqc, float complex* zc, Int * ldzc, float complex* work, Int * lwork, Int * info);
void hook_claqz3(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, float complex* alpha, float complex* beta, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* q, Int * ldq, float complex* z, Int * ldz, float complex* qc, Int * ldqc, float complex* zc, Int * ldzc, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqz3((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nshifts, (void*) nblock_desired, (void*) alpha, (void*) beta, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claqz3.timings[0] += (helpTimeStop - helpTime);
    data->claqz3.calls[0]++;

    return ;
}



extern float flexiblas_chain_slangt (char* norm, Int * n, float* dl, float* d, float* du, int len_norm);
float hook_slangt(char* norm, Int * n, float* dl, float* d, float* du, int len_norm)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slangt((void*) norm, (void*) n, (void*) dl, (void*) d, (void*) du, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->slangt.timings[0] += (helpTimeStop - helpTime);
    data->slangt.calls[0]++;

    return v;
}



extern void flexiblas_chain_claein (Int * rightv, Int * noinit, Int * n, float complex* h, Int * ldh, float complex* w, float complex* v, float complex* b, Int * ldb, float* rwork, float* eps3, float* smlnum, Int * info);
void hook_claein(Int * rightv, Int * noinit, Int * n, float complex* h, Int * ldh, float complex* w, float complex* v, float complex* b, Int * ldb, float* rwork, float* eps3, float* smlnum, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claein((void*) rightv, (void*) noinit, (void*) n, (void*) h, (void*) ldh, (void*) w, (void*) v, (void*) b, (void*) ldb, (void*) rwork, (void*) eps3, (void*) smlnum, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->claein.timings[0] += (helpTimeStop - helpTime);
    data->claein.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbevd_2stage (char* jobz, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_zhbevd_2stage(char* jobz, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhbevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqsp (char* uplo, Int * n, double* ap, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_dlaqsp(char* uplo, Int * n, double* ap, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqsp((void*) uplo, (void*) n, (void*) ap, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsp.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsp.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqr2p (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info);
void hook_cgeqr2p(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqr2p((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->cgeqr2p.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarfgp (Int * n, float* alpha, float* x, Int * incx, float* tau);
void hook_slarfgp(Int * n, float* alpha, float* x, Int * incx, float* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarfgp((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->slarfgp.timings[0] += (helpTimeStop - helpTime);
    data->slarfgp.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasda (Int * icompq, Int * smlsiz, Int * n, Int * sqre, float* d, float* e, float* u, Int * ldu, float* vt, Int * k, float* difl, float* difr, float* z, float* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, float* givnum, float* c, float* s, float* work, Int * iwork, Int * info);
void hook_slasda(Int * icompq, Int * smlsiz, Int * n, Int * sqre, float* d, float* e, float* u, Int * ldu, float* vt, Int * k, float* difl, float* difr, float* z, float* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, float* givnum, float* c, float* s, float* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasda((void*) icompq, (void*) smlsiz, (void*) n, (void*) sqre, (void*) d, (void*) e, (void*) u, (void*) ldu, (void*) vt, (void*) k, (void*) difl, (void*) difr, (void*) z, (void*) poles, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) perm, (void*) givnum, (void*) c, (void*) s, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasda.timings[0] += (helpTimeStop - helpTime);
    data->slasda.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpoequ (Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, Int * info);
void hook_zpoequ(Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpoequ((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zpoequ.timings[0] += (helpTimeStop - helpTime);
    data->zpoequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssptrf (char* uplo, Int * n, float* ap, Int * ipiv, Int * info, int len_uplo);
void hook_ssptrf(char* uplo, Int * n, float* ap, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssptrf((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssptrf.timings[0] += (helpTimeStop - helpTime);
    data->ssptrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeequb (Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_cgeequb(Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeequb((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeequb.timings[0] += (helpTimeStop - helpTime);
    data->cgeequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqtr (Int * ltran, Int * lreal, Int * n, double* t, Int * ldt, double* b, double* w, double* scale, double* x, double* work, Int * info);
void hook_dlaqtr(Int * ltran, Int * lreal, Int * n, double* t, Int * ldt, double* b, double* w, double* scale, double* x, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqtr((void*) ltran, (void*) lreal, (void*) n, (void*) t, (void*) ldt, (void*) b, (void*) w, (void*) scale, (void*) x, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqtr.timings[0] += (helpTimeStop - helpTime);
    data->dlaqtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlamswlq (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zlamswlq(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlamswlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zlamswlq.timings[0] += (helpTimeStop - helpTime);
    data->zlamswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqps (Int * m, Int * n, Int * offset, Int * nb, Int * kb, float complex* a, Int * lda, Int * jpvt, float complex* tau, float* vn1, float* vn2, float complex* auxv, float complex* f, Int * ldf);
void hook_claqps(Int * m, Int * n, Int * offset, Int * nb, Int * kb, float complex* a, Int * lda, Int * jpvt, float complex* tau, float* vn1, float* vn2, float complex* auxv, float complex* f, Int * ldf)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqps((void*) m, (void*) n, (void*) offset, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf);

    helpTimeStop = flexiblas_wtime();

    data->claqps.timings[0] += (helpTimeStop - helpTime);
    data->claqps.calls[0]++;

    return ;
}



extern void flexiblas_chain_stfttp (char* transr, char* uplo, Int * n, float* arf, float* ap, Int * info, int len_transr, int len_uplo);
void hook_stfttp(char* transr, char* uplo, Int * n, float* arf, float* ap, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stfttp((void*) transr, (void*) uplo, (void*) n, (void*) arf, (void*) ap, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->stfttp.timings[0] += (helpTimeStop - helpTime);
    data->stfttp.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarfb_gett (char* ident, Int * m, Int * n, Int * k, double complex* t, Int * ldt, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * ldwork, int len_ident);
void hook_zlarfb_gett(char* ident, Int * m, Int * n, Int * k, double complex* t, Int * ldt, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * ldwork, int len_ident)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarfb_gett((void*) ident, (void*) m, (void*) n, (void*) k, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) ldwork, (int) len_ident);

    helpTimeStop = flexiblas_wtime();

    data->zlarfb_gett.timings[0] += (helpTimeStop - helpTime);
    data->zlarfb_gett.calls[0]++;

    return ;
}



extern void flexiblas_chain_slatsqr (Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info);
void hook_slatsqr(Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slatsqr((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slatsqr.timings[0] += (helpTimeStop - helpTime);
    data->slatsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgsja (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float* tola, float* tolb, float* alpha, float* beta, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* q, Int * ldq, float complex* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_ctgsja(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* b, Int * ldb, float* tola, float* tolb, float* alpha, float* beta, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* q, Int * ldq, float complex* work, Int * ncycle, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgsja((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) ncycle, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->ctgsja.timings[0] += (helpTimeStop - helpTime);
    data->ctgsja.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpstf2 (char* uplo, Int * n, float complex* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo);
void hook_cpstf2(char* uplo, Int * n, float complex* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpstf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpstf2.timings[0] += (helpTimeStop - helpTime);
    data->cpstf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dggglm (Int * n, Int * m, Int * p, double* a, Int * lda, double* b, Int * ldb, double* d, double* x, double* y, double* work, Int * lwork, Int * info);
void hook_dggglm(Int * n, Int * m, Int * p, double* a, Int * lda, double* b, Int * ldb, double* d, double* x, double* y, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggglm((void*) n, (void*) m, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) d, (void*) x, (void*) y, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dggglm.timings[0] += (helpTimeStop - helpTime);
    data->dggglm.calls[0]++;

    return ;
}



extern void flexiblas_chain_cptsvx (char* fact, Int * n, Int * nrhs, float* d, float complex* e, float* df, float complex* ef, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact);
void hook_cptsvx(char* fact, Int * n, Int * nrhs, float* d, float complex* e, float* df, float complex* ef, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cptsvx((void*) fact, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact);

    helpTimeStop = flexiblas_wtime();

    data->cptsvx.timings[0] += (helpTimeStop - helpTime);
    data->cptsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zhetrs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrs.timings[0] += (helpTimeStop - helpTime);
    data->zhetrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssterf (Int * n, float* d, float* e, Int * info);
void hook_ssterf(Int * n, float* d, float* e, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssterf((void*) n, (void*) d, (void*) e, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ssterf.timings[0] += (helpTimeStop - helpTime);
    data->ssterf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlahr2 (Int * n, Int * k, Int * nb, double complex* a, Int * lda, double complex* tau, double complex* t, Int * ldt, double complex* y, Int * ldy);
void hook_zlahr2(Int * n, Int * k, Int * nb, double complex* a, Int * lda, double complex* tau, double complex* t, Int * ldt, double complex* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlahr2((void*) n, (void*) k, (void*) nb, (void*) a, (void*) lda, (void*) tau, (void*) t, (void*) ldt, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->zlahr2.timings[0] += (helpTimeStop - helpTime);
    data->zlahr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sormtr (char* side, char* uplo, char* trans, Int * m, Int * n, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans);
void hook_sormtr(char* side, char* uplo, char* trans, Int * m, Int * n, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sormtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sormtr.timings[0] += (helpTimeStop - helpTime);
    data->sormtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrf_rk (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsytrf_rk(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrf_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->zsytrf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_strsen (char* job, char* compq, Int * select, Int * n, float* t, Int * ldt, float* q, Int * ldq, float* wr, float* wi, Int * m, float* s, float* sep, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_job, int len_compq);
void hook_strsen(char* job, char* compq, Int * select, Int * n, float* t, Int * ldt, float* q, Int * ldq, float* wr, float* wi, Int * m, float* s, float* sep, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_job, int len_compq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strsen((void*) job, (void*) compq, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) q, (void*) ldq, (void*) wr, (void*) wi, (void*) m, (void*) s, (void*) sep, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_job, (int) len_compq);

    helpTimeStop = flexiblas_wtime();

    data->strsen.timings[0] += (helpTimeStop - helpTime);
    data->strsen.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhegvd (Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_zhegvd(Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhegvd((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhegvd.timings[0] += (helpTimeStop - helpTime);
    data->zhegvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zsytrs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrs.timings[0] += (helpTimeStop - helpTime);
    data->zsytrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chesv_rook (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chesv_rook(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chesv_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chesv_rook.timings[0] += (helpTimeStop - helpTime);
    data->chesv_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_clar1v (Int * n, Int * b1, Int * bn, float* lambda, float* d, float* l, float* ld, float* lld, float* pivmin, float* gaptol, float complex* z, Int * wantnc, Int * negcnt, float* ztz, float* mingma, Int * r, Int * isuppz, float* nrminv, float* resid, float* rqcorr, float* work);
void hook_clar1v(Int * n, Int * b1, Int * bn, float* lambda, float* d, float* l, float* ld, float* lld, float* pivmin, float* gaptol, float complex* z, Int * wantnc, Int * negcnt, float* ztz, float* mingma, Int * r, Int * isuppz, float* nrminv, float* resid, float* rqcorr, float* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clar1v((void*) n, (void*) b1, (void*) bn, (void*) lambda, (void*) d, (void*) l, (void*) ld, (void*) lld, (void*) pivmin, (void*) gaptol, (void*) z, (void*) wantnc, (void*) negcnt, (void*) ztz, (void*) mingma, (void*) r, (void*) isuppz, (void*) nrminv, (void*) resid, (void*) rqcorr, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->clar1v.timings[0] += (helpTimeStop - helpTime);
    data->clar1v.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorg2l (Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sorg2l(Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorg2l((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorg2l.timings[0] += (helpTimeStop - helpTime);
    data->sorg2l.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssteqr (char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_compz);
void hook_ssteqr(char* compz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssteqr((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->ssteqr.timings[0] += (helpTimeStop - helpTime);
    data->ssteqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sstev (char* jobz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_jobz);
void hook_sstev(char* jobz, Int * n, float* d, float* e, float* z, Int * ldz, float* work, Int * info, int len_jobz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sstev((void*) jobz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz);

    helpTimeStop = flexiblas_wtime();

    data->sstev.timings[0] += (helpTimeStop - helpTime);
    data->sstev.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgetsls (char* trans, Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_trans);
void hook_zgetsls(char* trans, Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetsls((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgetsls.timings[0] += (helpTimeStop - helpTime);
    data->zgetsls.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarrd (char* range, char* order, Int * n, double* vl, double* vu, Int * il, Int * iu, double* gers, double* reltol, double* d, double* e, double* e2, double* pivmin, Int * nsplit, Int * isplit, Int * m, double* w, double* werr, double* wl, double* wu, Int * iblock, Int * indexw, double* work, Int * iwork, Int * info, int len_range, int len_order);
void hook_dlarrd(char* range, char* order, Int * n, double* vl, double* vu, Int * il, Int * iu, double* gers, double* reltol, double* d, double* e, double* e2, double* pivmin, Int * nsplit, Int * isplit, Int * m, double* w, double* werr, double* wl, double* wu, Int * iblock, Int * indexw, double* work, Int * iwork, Int * info, int len_range, int len_order)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarrd((void*) range, (void*) order, (void*) n, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) gers, (void*) reltol, (void*) d, (void*) e, (void*) e2, (void*) pivmin, (void*) nsplit, (void*) isplit, (void*) m, (void*) w, (void*) werr, (void*) wl, (void*) wu, (void*) iblock, (void*) indexw, (void*) work, (void*) iwork, (void*) info, (int) len_range, (int) len_order);

    helpTimeStop = flexiblas_wtime();

    data->dlarrd.timings[0] += (helpTimeStop - helpTime);
    data->dlarrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed2 (Int * k, Int * n, Int * n1, double* d, double* q, Int * ldq, Int * indxq, double* rho, double* z, double* dlambda, double* w, double* q2, Int * indx, Int * indxc, Int * indxp, Int * coltyp, Int * info);
void hook_dlaed2(Int * k, Int * n, Int * n1, double* d, double* q, Int * ldq, Int * indxq, double* rho, double* z, double* dlambda, double* w, double* q2, Int * indx, Int * indxc, Int * indxp, Int * coltyp, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed2((void*) k, (void*) n, (void*) n1, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) z, (void*) dlambda, (void*) w, (void*) q2, (void*) indx, (void*) indxc, (void*) indxp, (void*) coltyp, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed2.timings[0] += (helpTimeStop - helpTime);
    data->dlaed2.calls[0]++;

    return ;
}



extern char flexiblas_chain_chla_transtype (Int * trans);
char hook_chla_transtype(Int * trans)
{
    char v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_chla_transtype((void*) trans);

    helpTimeStop = flexiblas_wtime();

    data->chla_transtype.timings[0] += (helpTimeStop - helpTime);
    data->chla_transtype.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlaswp (Int * n, double* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx);
void hook_dlaswp(Int * n, double* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaswp((void*) n, (void*) a, (void*) lda, (void*) k1, (void*) k2, (void*) ipiv, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->dlaswp.timings[0] += (helpTimeStop - helpTime);
    data->dlaswp.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeqr2 (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info);
void hook_dgeqr2(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeqr2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr2.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasdq (char* uplo, Int * sqre, Int * n, Int * ncvt, Int * nru, Int * ncc, float* d, float* e, float* vt, Int * ldvt, float* u, Int * ldu, float* c, Int * ldc, float* work, Int * info, int len_uplo);
void hook_slasdq(char* uplo, Int * sqre, Int * n, Int * ncvt, Int * nru, Int * ncc, float* d, float* e, float* vt, Int * ldvt, float* u, Int * ldu, float* c, Int * ldc, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasdq((void*) uplo, (void*) sqre, (void*) n, (void*) ncvt, (void*) nru, (void*) ncc, (void*) d, (void*) e, (void*) vt, (void*) ldvt, (void*) u, (void*) ldu, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slasdq.timings[0] += (helpTimeStop - helpTime);
    data->slasdq.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpbrfs (char* uplo, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_cpbrfs(char* uplo, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpbrfs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpbrfs.timings[0] += (helpTimeStop - helpTime);
    data->cpbrfs.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilaslr (Int * m, Int * n, float* a, Int * lda);
int hook_ilaslr(Int * m, Int * n, float* a, Int * lda)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilaslr((void*) m, (void*) n, (void*) a, (void*) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaslr.timings[0] += (helpTimeStop - helpTime);
    data->ilaslr.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlatrs3 (char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, double* a, Int * lda, double* x, Int * ldx, double* scale, double* cnorm, double* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_dlatrs3(char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, double* a, Int * lda, double* x, Int * ldx, double* scale, double* cnorm, double* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatrs3((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) x, (void*) ldx, (void*) scale, (void*) cnorm, (void*) work, (void*) lwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->dlatrs3.timings[0] += (helpTimeStop - helpTime);
    data->dlatrs3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgtrfs (char* trans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* dlf, double* df, double* duf, double* du2, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans);
void hook_dgtrfs(char* trans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* dlf, double* df, double* duf, double* du2, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgtrfs((void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->dgtrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpstf2 (char* uplo, Int * n, double complex* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo);
void hook_zpstf2(char* uplo, Int * n, double complex* a, Int * lda, Int * piv, Int * rank_bn, double* tol, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpstf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpstf2.timings[0] += (helpTimeStop - helpTime);
    data->zpstf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sporfs (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo);
void hook_sporfs(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sporfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sporfs.timings[0] += (helpTimeStop - helpTime);
    data->sporfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbsvx (char* fact, char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, Int * ipiv, char* equed, float* r, float* c, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans, int len_equed);
void hook_cgbsvx(char* fact, char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, Int * ipiv, char* equed, float* r, float* c, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_trans, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbsvx((void*) fact, (void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) ipiv, (void*) equed, (void*) r, (void*) c, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_trans, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->cgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->cgbsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqp2rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double complex* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double* vn1, double* vn2, double complex* work, Int * info);
void hook_zlaqp2rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double complex* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double* vn1, double* vn2, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqp2rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) kmax, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqp2rk.timings[0] += (helpTimeStop - helpTime);
    data->zlaqp2rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zheswapr (char* uplo, Int * n, double complex* a, Int * lda, Int * i1, Int * i2, int len_uplo);
void hook_zheswapr(char* uplo, Int * n, double complex* a, Int * lda, Int * i1, Int * i2, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheswapr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) i1, (void*) i2, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheswapr.timings[0] += (helpTimeStop - helpTime);
    data->zheswapr.calls[0]++;

    return ;
}



extern void flexiblas_chain_slatdf (Int * ijob, Int * n, float* z, Int * ldz, float* rhs, float* rdsum, float* rdscal, Int * ipiv, Int * jpiv);
void hook_slatdf(Int * ijob, Int * n, float* z, Int * ldz, float* rhs, float* rdsum, float* rdscal, Int * ipiv, Int * jpiv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slatdf((void*) ijob, (void*) n, (void*) z, (void*) ldz, (void*) rhs, (void*) rdsum, (void*) rdscal, (void*) ipiv, (void*) jpiv);

    helpTimeStop = flexiblas_wtime();

    data->slatdf.timings[0] += (helpTimeStop - helpTime);
    data->slatdf.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrs_aa (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssytrs_aa(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrs_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgsvj1 (char* jobv, Int * m, Int * n, Int * n1, double complex* a, Int * lda, double complex* d, double* sva, Int * mv, double complex* v, Int * ldv, double* eps, double* sfmin, double* tol, Int * nsweep, double complex* work, Int * lwork, Int * info, int len_jobv);
void hook_zgsvj1(char* jobv, Int * m, Int * n, Int * n1, double complex* a, Int * lda, double complex* d, double* sva, Int * mv, double complex* v, Int * ldv, double* eps, double* sfmin, double* tol, Int * nsweep, double complex* work, Int * lwork, Int * info, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgsvj1((void*) jobv, (void*) m, (void*) n, (void*) n1, (void*) a, (void*) lda, (void*) d, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) eps, (void*) sfmin, (void*) tol, (void*) nsweep, (void*) work, (void*) lwork, (void*) info, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->zgsvj1.timings[0] += (helpTimeStop - helpTime);
    data->zgsvj1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd3 (Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* q, Int * ldq, double* dsigma, double* u, Int * ldu, double* u2, Int * ldu2, double* vt, Int * ldvt, double* vt2, Int * ldvt2, Int * idxc, Int * ctot, double* z, Int * info);
void hook_dlasd3(Int * nl, Int * nr, Int * sqre, Int * k, double* d, double* q, Int * ldq, double* dsigma, double* u, Int * ldu, double* u2, Int * ldu2, double* vt, Int * ldvt, double* vt2, Int * ldvt2, Int * idxc, Int * ctot, double* z, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd3((void*) nl, (void*) nr, (void*) sqre, (void*) k, (void*) d, (void*) q, (void*) ldq, (void*) dsigma, (void*) u, (void*) ldu, (void*) u2, (void*) ldu2, (void*) vt, (void*) ldvt, (void*) vt2, (void*) ldvt2, (void*) idxc, (void*) ctot, (void*) z, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd3.timings[0] += (helpTimeStop - helpTime);
    data->dlasd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_shseqr (char* job, char* compz, Int * n, Int * ilo, Int * ihi, float* h, Int * ldh, float* wr, float* wi, float* z, Int * ldz, float* work, Int * lwork, Int * info, int len_job, int len_compz);
void hook_shseqr(char* job, char* compz, Int * n, Int * ilo, Int * ihi, float* h, Int * ldh, float* wr, float* wi, float* z, Int * ldz, float* work, Int * lwork, Int * info, int len_job, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_shseqr((void*) job, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_job, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->shseqr.timings[0] += (helpTimeStop - helpTime);
    data->shseqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgemqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zgemqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgemqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->zgemqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqp2rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float* tau, float* vn1, float* vn2, float* work, Int * info);
void hook_slaqp2rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * kmax, float* abstol, float* reltol, Int * kp1, float* maxc2nrm, float* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float* tau, float* vn1, float* vn2, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqp2rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) kmax, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqp2rk.timings[0] += (helpTimeStop - helpTime);
    data->slaqp2rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmtr (char* side, char* uplo, char* trans, Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans);
void hook_zunmtr(char* side, char* uplo, char* trans, Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmtr.timings[0] += (helpTimeStop - helpTime);
    data->zunmtr.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlanht (char* norm, Int * n, double* d, double complex* e, int len_norm);
double hook_zlanht(char* norm, Int * n, double* d, double complex* e, int len_norm)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlanht((void*) norm, (void*) n, (void*) d, (void*) e, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->zlanht.timings[0] += (helpTimeStop - helpTime);
    data->zlanht.calls[0]++;

    return v;
}



extern void flexiblas_chain_csptrs (char* uplo, Int * n, Int * nrhs, float complex* ap, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_csptrs(char* uplo, Int * n, Int * nrhs, float complex* ap, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csptrs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csptrs.timings[0] += (helpTimeStop - helpTime);
    data->csptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgerfs (char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_trans);
void hook_zgerfs(char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgerfs((void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgerfs.timings[0] += (helpTimeStop - helpTime);
    data->zgerfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgttrf (Int * n, float* dl, float* d, float* du, float* du2, Int * ipiv, Int * info);
void hook_sgttrf(Int * n, float* dl, float* d, float* du, float* du2, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgttrf((void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgttrf.timings[0] += (helpTimeStop - helpTime);
    data->sgttrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsycon (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo);
void hook_zsycon(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double* anorm, double* rcond, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsycon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsycon.timings[0] += (helpTimeStop - helpTime);
    data->zsycon.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed5 (Int * i, float* d, float* z, float* delta, float* rho, float* dlam);
void hook_slaed5(Int * i, float* d, float* z, float* delta, float* rho, float* dlam)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed5((void*) i, (void*) d, (void*) z, (void*) delta, (void*) rho, (void*) dlam);

    helpTimeStop = flexiblas_wtime();

    data->slaed5.timings[0] += (helpTimeStop - helpTime);
    data->slaed5.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmrz (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cunmrz(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmrz((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmrz.timings[0] += (helpTimeStop - helpTime);
    data->cunmrz.calls[0]++;

    return ;
}



extern void flexiblas_chain_spstrf (char* uplo, Int * n, float* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo);
void hook_spstrf(char* uplo, Int * n, float* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spstrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spstrf.timings[0] += (helpTimeStop - helpTime);
    data->spstrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarra (Int * n, double* d, double* e, double* e2, double* spltol, double* tnrm, Int * nsplit, Int * isplit, Int * info);
void hook_dlarra(Int * n, double* d, double* e, double* e2, double* spltol, double* tnrm, Int * nsplit, Int * isplit, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarra((void*) n, (void*) d, (void*) e, (void*) e2, (void*) spltol, (void*) tnrm, (void*) nsplit, (void*) isplit, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarra.timings[0] += (helpTimeStop - helpTime);
    data->dlarra.calls[0]++;

    return ;
}



extern void flexiblas_chain_clamtsqr (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_clamtsqr(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clamtsqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->clamtsqr.timings[0] += (helpTimeStop - helpTime);
    data->clamtsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrevc (char* side, char* howmny, Int * select, Int * n, float complex* t, Int * ldt, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, float* rwork, Int * info, int len_side, int len_howmny);
void hook_ctrevc(char* side, char* howmny, Int * select, Int * n, float complex* t, Int * ldt, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, float* rwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrevc((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) rwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ctrevc.timings[0] += (helpTimeStop - helpTime);
    data->ctrevc.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgtrfs (char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* dlf, float complex* df, float complex* duf, float complex* du2, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_trans);
void hook_cgtrfs(char* trans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* dlf, float complex* df, float complex* duf, float complex* du2, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgtrfs((void*) trans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) dlf, (void*) df, (void*) duf, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->cgtrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelq (Int * m, Int * n, float* a, Int * lda, float* t, Int * tsize, float* work, Int * lwork, Int * info);
void hook_sgelq(Int * m, Int * n, float* a, Int * lda, float* t, Int * tsize, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelq((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelq.timings[0] += (helpTimeStop - helpTime);
    data->sgelq.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyswapr (char* uplo, Int * n, float* a, Int * lda, Int * i1, Int * i2, int len_uplo);
void hook_ssyswapr(char* uplo, Int * n, float* a, Int * lda, Int * i1, Int * i2, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyswapr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) i1, (void*) i2, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyswapr.timings[0] += (helpTimeStop - helpTime);
    data->ssyswapr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlauum (char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo);
void hook_dlauum(char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlauum((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlauum.timings[0] += (helpTimeStop - helpTime);
    data->dlauum.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgetsls (char* trans, Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_trans);
void hook_cgetsls(char* trans, Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgetsls((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgetsls.timings[0] += (helpTimeStop - helpTime);
    data->cgetsls.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqge (Int * m, Int * n, double complex* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed);
void hook_zlaqge(Int * m, Int * n, double complex* a, Int * lda, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqge((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqge.timings[0] += (helpTimeStop - helpTime);
    data->zlaqge.calls[0]++;

    return ;
}



extern void flexiblas_chain_chpsv (char* uplo, Int * n, Int * nrhs, float complex* ap, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chpsv(char* uplo, Int * n, Int * nrhs, float complex* ap, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chpsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chpsv.timings[0] += (helpTimeStop - helpTime);
    data->chpsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgetri (Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * lwork, Int * info);
void hook_sgetri(Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgetri((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetri.timings[0] += (helpTimeStop - helpTime);
    data->sgetri.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasq2 (Int * n, double* z, Int * info);
void hook_dlasq2(Int * n, double* z, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasq2((void*) n, (void*) z, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasq2.timings[0] += (helpTimeStop - helpTime);
    data->dlasq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_strevc3 (char* side, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * lwork, Int * info, int len_side, int len_howmny);
void hook_strevc3(char* side, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * lwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strevc3((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->strevc3.timings[0] += (helpTimeStop - helpTime);
    data->strevc3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeequ (Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info);
void hook_cgeequ(Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeequ((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeequ.timings[0] += (helpTimeStop - helpTime);
    data->cgeequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_dstevx (char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range);
void hook_dstevx(char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dstevx((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->dstevx.timings[0] += (helpTimeStop - helpTime);
    data->dstevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgexc (Int * wantq, Int * wantz, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * ifst, Int * ilst, Int * info);
void hook_ztgexc(Int * wantq, Int * wantz, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, Int * ifst, Int * ilst, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgexc((void*) wantq, (void*) wantz, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ifst, (void*) ilst, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ztgexc.timings[0] += (helpTimeStop - helpTime);
    data->ztgexc.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelsy (Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * jpvt, double* rcond, Int * rank_bn, double* work, Int * lwork, Int * info);
void hook_dgelsy(Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * jpvt, double* rcond, Int * rank_bn, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelsy((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) jpvt, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelsy.timings[0] += (helpTimeStop - helpTime);
    data->dgelsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelst (char* trans, Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_trans);
void hook_dgelst(char* trans, Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelst((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgelst.timings[0] += (helpTimeStop - helpTime);
    data->dgelst.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelq2 (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sgelq2(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelq2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelq2.timings[0] += (helpTimeStop - helpTime);
    data->sgelq2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spstf2 (char* uplo, Int * n, float* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo);
void hook_spstf2(char* uplo, Int * n, float* a, Int * lda, Int * piv, Int * rank_bn, float* tol, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spstf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) piv, (void*) rank_bn, (void*) tol, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spstf2.timings[0] += (helpTimeStop - helpTime);
    data->spstf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_clargv (Int * n, float complex* x, Int * incx, float complex* y, Int * incy, float* c, Int * incc);
void hook_clargv(Int * n, float complex* x, Int * incx, float complex* y, Int * incy, float* c, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clargv((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) c, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->clargv.timings[0] += (helpTimeStop - helpTime);
    data->clargv.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed8 (Int * icompq, Int * k, Int * n, Int * qsiz, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* z, float* dlambda, float* q2, Int * ldq2, float* w, Int * perm, Int * givptr, Int * givcol, float* givnum, Int * indxp, Int * indx, Int * info);
void hook_slaed8(Int * icompq, Int * k, Int * n, Int * qsiz, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* z, float* dlambda, float* q2, Int * ldq2, float* w, Int * perm, Int * givptr, Int * givcol, float* givnum, Int * indxp, Int * indx, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed8((void*) icompq, (void*) k, (void*) n, (void*) qsiz, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) z, (void*) dlambda, (void*) q2, (void*) ldq2, (void*) w, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) indxp, (void*) indx, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed8.timings[0] += (helpTimeStop - helpTime);
    data->slaed8.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelsd (Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info);
void hook_zgelsd(Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelsd((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelsd.timings[0] += (helpTimeStop - helpTime);
    data->zgelsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zheevx (char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zheevx(char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheevx.timings[0] += (helpTimeStop - helpTime);
    data->zheevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssysv_rook (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssysv_rook(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssysv_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_rook.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlapy3 (double* x, double* y, double* z);
double hook_dlapy3(double* x, double* y, double* z)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlapy3((void*) x, (void*) y, (void*) z);

    helpTimeStop = flexiblas_wtime();

    data->dlapy3.timings[0] += (helpTimeStop - helpTime);
    data->dlapy3.calls[0]++;

    return v;
}



extern void flexiblas_chain_cspsvx (char* fact, char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* afp, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_uplo);
void hook_cspsvx(char* fact, char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* afp, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cspsvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cspsvx.timings[0] += (helpTimeStop - helpTime);
    data->cspsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaswp (Int * n, float* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx);
void hook_slaswp(Int * n, float* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaswp((void*) n, (void*) a, (void*) lda, (void*) k1, (void*) k2, (void*) ipiv, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->slaswp.timings[0] += (helpTimeStop - helpTime);
    data->slaswp.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyequb (char* uplo, Int * n, float* a, Int * lda, float* s, float* scond, float* amax, float* work, Int * info, int len_uplo);
void hook_ssyequb(char* uplo, Int * n, float* a, Int * lda, float* s, float* scond, float* amax, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyequb((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyequb.timings[0] += (helpTimeStop - helpTime);
    data->ssyequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspevx (char* jobz, char* range, char* uplo, Int * n, double* ap, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dspevx(char* jobz, char* range, char* uplo, Int * n, double* ap, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ap, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspevx.timings[0] += (helpTimeStop - helpTime);
    data->dspevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaset (char* uplo, Int * m, Int * n, double* alpha, double* beta, double* a, Int * lda, int len_uplo);
void hook_dlaset(char* uplo, Int * m, Int * n, double* alpha, double* beta, double* a, Int * lda, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaset((void*) uplo, (void*) m, (void*) n, (void*) alpha, (void*) beta, (void*) a, (void*) lda, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlaset.timings[0] += (helpTimeStop - helpTime);
    data->dlaset.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlalsa (Int * icompq, Int * smlsiz, Int * n, Int * nrhs, double* b, Int * ldb, double* bx, Int * ldbx, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* work, Int * iwork, Int * info);
void hook_dlalsa(Int * icompq, Int * smlsiz, Int * n, Int * nrhs, double* b, Int * ldb, double* bx, Int * ldbx, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlalsa((void*) icompq, (void*) smlsiz, (void*) n, (void*) nrhs, (void*) b, (void*) ldb, (void*) bx, (void*) ldbx, (void*) u, (void*) ldu, (void*) vt, (void*) k, (void*) difl, (void*) difr, (void*) z, (void*) poles, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) perm, (void*) givnum, (void*) c, (void*) s, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlalsa.timings[0] += (helpTimeStop - helpTime);
    data->dlalsa.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtfsm (char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, double* alpha, double* a, double* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag);
void hook_dtfsm(char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, double* alpha, double* a, double* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtfsm((void*) transr, (void*) side, (void*) uplo, (void*) trans, (void*) diag, (void*) m, (void*) n, (void*) alpha, (void*) a, (void*) b, (void*) ldb, (int) len_transr, (int) len_side, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtfsm.timings[0] += (helpTimeStop - helpTime);
    data->dtfsm.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytri_3 (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zsytri_3(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytri_3.timings[0] += (helpTimeStop - helpTime);
    data->zsytri_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_slar1v (Int * n, Int * b1, Int * bn, float* lambda, float* d, float* l, float* ld, float* lld, float* pivmin, float* gaptol, float* z, Int * wantnc, Int * negcnt, float* ztz, float* mingma, Int * r, Int * isuppz, float* nrminv, float* resid, float* rqcorr, float* work);
void hook_slar1v(Int * n, Int * b1, Int * bn, float* lambda, float* d, float* l, float* ld, float* lld, float* pivmin, float* gaptol, float* z, Int * wantnc, Int * negcnt, float* ztz, float* mingma, Int * r, Int * isuppz, float* nrminv, float* resid, float* rqcorr, float* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slar1v((void*) n, (void*) b1, (void*) bn, (void*) lambda, (void*) d, (void*) l, (void*) ld, (void*) lld, (void*) pivmin, (void*) gaptol, (void*) z, (void*) wantnc, (void*) negcnt, (void*) ztz, (void*) mingma, (void*) r, (void*) isuppz, (void*) nrminv, (void*) resid, (void*) rqcorr, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->slar1v.timings[0] += (helpTimeStop - helpTime);
    data->slar1v.calls[0]++;

    return ;
}



extern void flexiblas_chain_clatrs (char* uplo, char* trans, char* diag, char* normin, Int * n, float complex* a, Int * lda, float complex* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_clatrs(char* uplo, char* trans, char* diag, char* normin, Int * n, float complex* a, Int * lda, float complex* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clatrs((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) a, (void*) lda, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->clatrs.timings[0] += (helpTimeStop - helpTime);
    data->clatrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqp3rk (Int * m, Int * n, Int * nrhs, Int * kmax, double* abstol, double* reltol, double complex* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info);
void hook_zgeqp3rk(Int * m, Int * n, Int * nrhs, Int * kmax, double* abstol, double* reltol, double complex* a, Int * lda, Int * k, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) kmax, (void*) abstol, (void*) reltol, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->zgeqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarrk (Int * n, Int * iw, float* gl, float* gu, float* d, float* e2, float* pivmin, float* reltol, float* w, float* werr, Int * info);
void hook_slarrk(Int * n, Int * iw, float* gl, float* gu, float* d, float* e2, float* pivmin, float* reltol, float* w, float* werr, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarrk((void*) n, (void*) iw, (void*) gl, (void*) gu, (void*) d, (void*) e2, (void*) pivmin, (void*) reltol, (void*) w, (void*) werr, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrk.timings[0] += (helpTimeStop - helpTime);
    data->slarrk.calls[0]++;

    return ;
}



extern void flexiblas_chain_slapll (Int * n, float* x, Int * incx, float* y, Int * incy, float* ssmin);
void hook_slapll(Int * n, float* x, Int * incx, float* y, Int * incy, float* ssmin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slapll((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->slapll.timings[0] += (helpTimeStop - helpTime);
    data->slapll.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctpmqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * info, int len_side, int len_trans);
void hook_ctpmqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, Int * nb, float complex* v, Int * ldv, float complex* t, Int * ldt, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpmqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ctpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->ctpmqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheevx_2stage (char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_cheevx_2stage(char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheevx_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevx_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgehd2 (Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sgehd2(Int * n, Int * ilo, Int * ihi, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgehd2((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgehd2.timings[0] += (helpTimeStop - helpTime);
    data->sgehd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_checon_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo);
void hook_checon_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_checon_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->checon_rook.timings[0] += (helpTimeStop - helpTime);
    data->checon_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri2x (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo);
void hook_csytri2x(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri2x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri2x.timings[0] += (helpTimeStop - helpTime);
    data->csytri2x.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhptrd (char* uplo, Int * n, double complex* ap, double* d, double* e, double complex* tau, Int * info, int len_uplo);
void hook_zhptrd(char* uplo, Int * n, double complex* ap, double* d, double* e, double complex* tau, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhptrd((void*) uplo, (void*) n, (void*) ap, (void*) d, (void*) e, (void*) tau, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhptrd.timings[0] += (helpTimeStop - helpTime);
    data->zhptrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgsna (char* job, char* howmny, Int * select, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double* s, double* dif, Int * mm, Int * m, double complex* work, Int * lwork, Int * iwork, Int * info, int len_job, int len_howmny);
void hook_ztgsna(char* job, char* howmny, Int * select, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double* s, double* dif, Int * mm, Int * m, double complex* work, Int * lwork, Int * iwork, Int * info, int len_job, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgsna((void*) job, (void*) howmny, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) s, (void*) dif, (void*) mm, (void*) m, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_job, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ztgsna.timings[0] += (helpTimeStop - helpTime);
    data->ztgsna.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgghd3 (char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info, int len_compq, int len_compz);
void hook_zgghd3(char* compq, char* compz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info, int len_compq, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgghd3((void*) compq, (void*) compz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info, (int) len_compq, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->zgghd3.timings[0] += (helpTimeStop - helpTime);
    data->zgghd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_clasyf_rk (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clasyf_rk(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clasyf_rk((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clasyf_rk.timings[0] += (helpTimeStop - helpTime);
    data->clasyf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrrfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_dtrrfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrrfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtrrfs.timings[0] += (helpTimeStop - helpTime);
    data->dtrrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunm2l (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zunm2l(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunm2l((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunm2l.timings[0] += (helpTimeStop - helpTime);
    data->zunm2l.calls[0]++;

    return ;
}



extern void flexiblas_chain_zrscl (Int * n, double complex* a, double complex* x, Int * incx);
void hook_zrscl(Int * n, double complex* a, double complex* x, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zrscl((void*) n, (void*) a, (void*) x, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->zrscl.timings[0] += (helpTimeStop - helpTime);
    data->zrscl.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsysv_aa_2stage (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* tb, Int * ltb, Int * ipiv, Int * ipiv2, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsysv_aa_2stage(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* tb, Int * ltb, Int * ipiv, Int * ipiv2, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsysv_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeqlf (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cgeqlf(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeqlf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->cgeqlf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhptri (char* uplo, Int * n, double complex* ap, Int * ipiv, double complex* work, Int * info, int len_uplo);
void hook_zhptri(char* uplo, Int * n, double complex* ap, Int * ipiv, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhptri((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhptri.timings[0] += (helpTimeStop - helpTime);
    data->zhptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlargv (Int * n, double* x, Int * incx, double* y, Int * incy, double* c, Int * incc);
void hook_dlargv(Int * n, double* x, Int * incx, double* y, Int * incy, double* c, Int * incc)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlargv((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) c, (void*) incc);

    helpTimeStop = flexiblas_wtime();

    data->dlargv.timings[0] += (helpTimeStop - helpTime);
    data->dlargv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zspr (char* uplo, Int * n, double complex* alpha, double complex* x, Int * incx, double complex* ap, int len_uplo);
void hook_zspr(char* uplo, Int * n, double complex* alpha, double complex* x, Int * incx, double complex* ap, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zspr((void*) uplo, (void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) ap, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zspr.timings[0] += (helpTimeStop - helpTime);
    data->zspr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunmr2 (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zunmr2(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunmr2((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunmr2.timings[0] += (helpTimeStop - helpTime);
    data->zunmr2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqp3rk (Int * m, Int * n, Int * nrhs, Int * kmax, float* abstol, float* reltol, float* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float* tau, float* work, Int * lwork, Int * iwork, Int * info);
void hook_sgeqp3rk(Int * m, Int * n, Int * nrhs, Int * kmax, float* abstol, float* reltol, float* a, Int * lda, Int * k, float* maxc2nrmk, float* relmaxc2nrmk, Int * jpiv, float* tau, float* work, Int * lwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) kmax, (void*) abstol, (void*) reltol, (void*) a, (void*) lda, (void*) k, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) work, (void*) lwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->sgeqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpotrf2 (char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_cpotrf2(char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpotrf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpotrf2.timings[0] += (helpTimeStop - helpTime);
    data->cpotrf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqrfp (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgeqrfp(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqrfp((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrfp.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaeda (Int * n, Int * tlvls, Int * curlvl, Int * curpbm, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float* q, Int * qptr, float* z, float* ztemp, Int * info);
void hook_slaeda(Int * n, Int * tlvls, Int * curlvl, Int * curpbm, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float* q, Int * qptr, float* z, float* ztemp, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaeda((void*) n, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) q, (void*) qptr, (void*) z, (void*) ztemp, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaeda.timings[0] += (helpTimeStop - helpTime);
    data->slaeda.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelss (Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float complex* work, Int * lwork, float* rwork, Int * info);
void hook_cgelss(Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float complex* work, Int * lwork, float* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelss((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelss.timings[0] += (helpTimeStop - helpTime);
    data->cgelss.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorbdb6 (Int * m1, Int * m2, Int * n, double* x1, Int * incx1, double* x2, Int * incx2, double* q1, Int * ldq1, double* q2, Int * ldq2, double* work, Int * lwork, Int * info);
void hook_dorbdb6(Int * m1, Int * m2, Int * n, double* x1, Int * incx1, double* x2, Int * incx2, double* q1, Int * ldq1, double* q2, Int * ldq2, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorbdb6((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb6.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb6.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgbr (char* vect, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info, int len_vect);
void hook_dorgbr(char* vect, Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgbr((void*) vect, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->dorgbr.timings[0] += (helpTimeStop - helpTime);
    data->dorgbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsptrd (char* uplo, Int * n, double* ap, double* d, double* e, double* tau, Int * info, int len_uplo);
void hook_dsptrd(char* uplo, Int * n, double* ap, double* d, double* e, double* tau, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsptrd((void*) uplo, (void*) n, (void*) ap, (void*) d, (void*) e, (void*) tau, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsptrd.timings[0] += (helpTimeStop - helpTime);
    data->dsptrd.calls[0]++;

    return ;
}



extern float flexiblas_chain_sroundup_lwork (Int * lwork);
float hook_sroundup_lwork(Int * lwork)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_sroundup_lwork((void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->sroundup_lwork.timings[0] += (helpTimeStop - helpTime);
    data->sroundup_lwork.calls[0]++;

    return v;
}



extern void flexiblas_chain_zptsvx (char* fact, Int * n, Int * nrhs, double* d, double complex* e, double* df, double complex* ef, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact);
void hook_zptsvx(char* fact, Int * n, Int * nrhs, double* d, double complex* e, double* df, double complex* ef, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zptsvx((void*) fact, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact);

    helpTimeStop = flexiblas_wtime();

    data->zptsvx.timings[0] += (helpTimeStop - helpTime);
    data->zptsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_clalsd (char* uplo, Int * smlsiz, Int * n, Int * nrhs, float* d, float* e, float complex* b, Int * ldb, float* rcond, Int * rank_bn, float complex* work, float* rwork, Int * iwork, Int * info, int len_uplo);
void hook_clalsd(char* uplo, Int * smlsiz, Int * n, Int * nrhs, float* d, float* e, float complex* b, Int * ldb, float* rcond, Int * rank_bn, float complex* work, float* rwork, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clalsd((void*) uplo, (void*) smlsiz, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) rcond, (void*) rank_bn, (void*) work, (void*) rwork, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clalsd.timings[0] += (helpTimeStop - helpTime);
    data->clalsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarf (char* side, Int * m, Int * n, double complex* v, Int * incv, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_side);
void hook_zlarf(char* side, Int * m, Int * n, double complex* v, Int * incv, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarf((void*) side, (void*) m, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->zlarf.timings[0] += (helpTimeStop - helpTime);
    data->zlarf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlalsa (Int * icompq, Int * smlsiz, Int * n, Int * nrhs, double complex* b, Int * ldb, double complex* bx, Int * ldbx, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* rwork, Int * iwork, Int * info);
void hook_zlalsa(Int * icompq, Int * smlsiz, Int * n, Int * nrhs, double complex* b, Int * ldb, double complex* bx, Int * ldbx, double* u, Int * ldu, double* vt, Int * k, double* difl, double* difr, double* z, double* poles, Int * givptr, Int * givcol, Int * ldgcol, Int * perm, double* givnum, double* c, double* s, double* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlalsa((void*) icompq, (void*) smlsiz, (void*) n, (void*) nrhs, (void*) b, (void*) ldb, (void*) bx, (void*) ldbx, (void*) u, (void*) ldu, (void*) vt, (void*) k, (void*) difl, (void*) difr, (void*) z, (void*) poles, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) perm, (void*) givnum, (void*) c, (void*) s, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlalsa.timings[0] += (helpTimeStop - helpTime);
    data->zlalsa.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaic1 (Int * job, Int * j, float* x, float* sest, float* w, float* gamma, float* sestpr, float* s, float* c);
void hook_slaic1(Int * job, Int * j, float* x, float* sest, float* w, float* gamma, float* sestpr, float* s, float* c)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaic1((void*) job, (void*) j, (void*) x, (void*) sest, (void*) w, (void*) gamma, (void*) sestpr, (void*) s, (void*) c);

    helpTimeStop = flexiblas_wtime();

    data->slaic1.timings[0] += (helpTimeStop - helpTime);
    data->slaic1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpotrs (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dpotrs(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpotrs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpotrs.timings[0] += (helpTimeStop - helpTime);
    data->dpotrs.calls[0]++;

    return ;
}



extern float flexiblas_chain_clangb (char* norm, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* work, int len_norm);
float hook_clangb(char* norm, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, float* work, int len_norm)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clangb((void*) norm, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->clangb.timings[0] += (helpTimeStop - helpTime);
    data->clangb.calls[0]++;

    return v;
}



extern void flexiblas_chain_zcgesv (Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double complex* work, float complex* swork, double* rwork, Int * iter, Int * info);
void hook_zcgesv(Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double complex* work, float complex* swork, double* rwork, Int * iter, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zcgesv((void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) work, (void*) swork, (void*) rwork, (void*) iter, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zcgesv.timings[0] += (helpTimeStop - helpTime);
    data->zcgesv.calls[0]++;

    return ;
}



extern void flexiblas_chain_clasyf_rook (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clasyf_rook(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clasyf_rook((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->clasyf_rook.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlansy (char* norm, char* uplo, Int * n, double* a, Int * lda, double* work, int len_norm, int len_uplo);
double hook_dlansy(char* norm, char* uplo, Int * n, double* a, Int * lda, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlansy((void*) norm, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlansy.timings[0] += (helpTimeStop - helpTime);
    data->dlansy.calls[0]++;

    return v;
}



extern void flexiblas_chain_stbrfs (char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_stbrfs(char* uplo, char* trans, char* diag, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stbrfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stbrfs.timings[0] += (helpTimeStop - helpTime);
    data->stbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetri2x (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo);
void hook_chetri2x(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetri2x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetri2x.timings[0] += (helpTimeStop - helpTime);
    data->chetri2x.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgesvj (char* joba, char* jobu, char* jobv, Int * m, Int * n, double* a, Int * lda, double* sva, Int * mv, double* v, Int * ldv, double* work, Int * lwork, Int * info, int len_joba, int len_jobu, int len_jobv);
void hook_dgesvj(char* joba, char* jobu, char* jobv, Int * m, Int * n, double* a, Int * lda, double* sva, Int * mv, double* v, Int * ldv, double* work, Int * lwork, Int * info, int len_joba, int len_jobu, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgesvj((void*) joba, (void*) jobu, (void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) work, (void*) lwork, (void*) info, (int) len_joba, (int) len_jobu, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->dgesvj.timings[0] += (helpTimeStop - helpTime);
    data->dgesvj.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgetrs (char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_trans);
void hook_zgetrs(char* trans, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetrs((void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgetrs.timings[0] += (helpTimeStop - helpTime);
    data->zgetrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chpgvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, float complex* ap, float complex* bp, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_chpgvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, float complex* ap, float complex* bp, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chpgvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chpgvx.timings[0] += (helpTimeStop - helpTime);
    data->chpgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarfy (char* uplo, Int * n, float* v, Int * incv, float* tau, float* c, Int * ldc, float* work, int len_uplo);
void hook_slarfy(char* uplo, Int * n, float* v, Int * incv, float* tau, float* c, Int * ldc, float* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarfy((void*) uplo, (void*) n, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slarfy.timings[0] += (helpTimeStop - helpTime);
    data->slarfy.calls[0]++;

    return ;
}



extern void flexiblas_chain_sspgst (Int * itype, char* uplo, Int * n, float* ap, float* bp, Int * info, int len_uplo);
void hook_sspgst(Int * itype, char* uplo, Int * n, float* ap, float* bp, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sspgst((void*) itype, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sspgst.timings[0] += (helpTimeStop - helpTime);
    data->sspgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorgtsqr_row (Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info);
void hook_sorgtsqr_row(Int * m, Int * n, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorgtsqr_row((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->sorgtsqr_row.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelq (Int * m, Int * n, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* work, Int * lwork, Int * info);
void hook_zgelq(Int * m, Int * n, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelq((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelq.timings[0] += (helpTimeStop - helpTime);
    data->zgelq.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetri_3x (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * nb, Int * info, int len_uplo);
void hook_zhetri_3x(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetri_3x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetri_3x.timings[0] += (helpTimeStop - helpTime);
    data->zhetri_3x.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpotri (char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo);
void hook_dpotri(char* uplo, Int * n, double* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpotri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpotri.timings[0] += (helpTimeStop - helpTime);
    data->dpotri.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgsna (char* job, char* howmny, Int * select, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float* s, float* dif, Int * mm, Int * m, float complex* work, Int * lwork, Int * iwork, Int * info, int len_job, int len_howmny);
void hook_ctgsna(char* job, char* howmny, Int * select, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float* s, float* dif, Int * mm, Int * m, float complex* work, Int * lwork, Int * iwork, Int * info, int len_job, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgsna((void*) job, (void*) howmny, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) s, (void*) dif, (void*) mm, (void*) m, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_job, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ctgsna.timings[0] += (helpTimeStop - helpTime);
    data->ctgsna.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorcsd (char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, char* signs, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x12, Int * ldx12, float* x21, Int * ldx21, float* x22, Int * ldx22, float* theta, float* u1, Int * ldu1, float* u2, Int * ldu2, float* v1t, Int * ldv1t, float* v2t, Int * ldv2t, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans, int len_signs);
void hook_sorcsd(char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, char* signs, Int * m, Int * p, Int * q, float* x11, Int * ldx11, float* x12, Int * ldx12, float* x21, Int * ldx21, float* x22, Int * ldx22, float* theta, float* u1, Int * ldu1, float* u2, Int * ldu2, float* v1t, Int * ldv1t, float* v2t, Int * ldv2t, float* work, Int * lwork, Int * iwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorcsd((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) jobv2t, (void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) v2t, (void*) ldv2t, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t, (int) len_jobv2t, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->sorcsd.timings[0] += (helpTimeStop - helpTime);
    data->sorcsd.calls[0]++;

    return ;
}



extern float flexiblas_chain_slansp (char* norm, char* uplo, Int * n, float* ap, float* work, int len_norm, int len_uplo);
float hook_slansp(char* norm, char* uplo, Int * n, float* ap, float* work, int len_norm, int len_uplo)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_slansp((void*) norm, (void*) uplo, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->slansp.timings[0] += (helpTimeStop - helpTime);
    data->slansp.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlaed5 (Int * i, double* d, double* z, double* delta, double* rho, double* dlam);
void hook_dlaed5(Int * i, double* d, double* z, double* delta, double* rho, double* dlam)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed5((void*) i, (void*) d, (void*) z, (void*) delta, (void*) rho, (void*) dlam);

    helpTimeStop = flexiblas_wtime();

    data->dlaed5.timings[0] += (helpTimeStop - helpTime);
    data->dlaed5.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetri_3 (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetri_3(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetri_3.timings[0] += (helpTimeStop - helpTime);
    data->chetri_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctgsy2 (char* trans, Int * ijob, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float complex* d, Int * ldd, float complex* e, Int * lde, float complex* f, Int * ldf, float* scale, float* rdsum, float* rdscal, Int * info, int len_trans);
void hook_ctgsy2(char* trans, Int * ijob, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float complex* d, Int * ldd, float complex* e, Int * lde, float complex* f, Int * ldf, float* scale, float* rdsum, float* rdscal, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctgsy2((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) rdsum, (void*) rdscal, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ctgsy2.timings[0] += (helpTimeStop - helpTime);
    data->ctgsy2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarft (char* direct, char* storev, Int * n, Int * k, float* v, Int * ldv, float* tau, float* t, Int * ldt, int len_direct, int len_storev);
void hook_slarft(char* direct, char* storev, Int * n, Int * k, float* v, Int * ldv, float* tau, float* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarft((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->slarft.timings[0] += (helpTimeStop - helpTime);
    data->slarft.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgtcon (char* norm, Int * n, float* dl, float* d, float* du, float* du2, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_norm);
void hook_sgtcon(char* norm, Int * n, float* dl, float* d, float* du, float* du2, Int * ipiv, float* anorm, float* rcond, float* work, Int * iwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgtcon((void*) norm, (void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->sgtcon.timings[0] += (helpTimeStop - helpTime);
    data->sgtcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorghr (Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dorghr(Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorghr((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorghr.timings[0] += (helpTimeStop - helpTime);
    data->dorghr.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilaclr (Int * m, Int * n, float complex* a, Int * lda);
int hook_ilaclr(Int * m, Int * n, float complex* a, Int * lda)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilaclr((void*) m, (void*) n, (void*) a, (void*) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaclr.timings[0] += (helpTimeStop - helpTime);
    data->ilaclr.calls[0]++;

    return v;
}



extern void flexiblas_chain_spbtrf (char* uplo, Int * n, Int * kd, float* ab, Int * ldab, Int * info, int len_uplo);
void hook_spbtrf(char* uplo, Int * n, Int * kd, float* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbtrf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spbtrf.timings[0] += (helpTimeStop - helpTime);
    data->spbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsprfs (char* uplo, Int * n, Int * nrhs, double* ap, double* afp, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dsprfs(char* uplo, Int * n, Int * nrhs, double* ap, double* afp, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsprfs.timings[0] += (helpTimeStop - helpTime);
    data->dsprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlat2c (char* uplo, Int * n, double complex* a, Int * lda, float complex* sa, Int * ldsa, Int * info, int len_uplo);
void hook_zlat2c(char* uplo, Int * n, double complex* a, Int * lda, float complex* sa, Int * ldsa, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlat2c((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) sa, (void*) ldsa, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlat2c.timings[0] += (helpTimeStop - helpTime);
    data->zlat2c.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgehd2 (Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * info);
void hook_dgehd2(Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgehd2((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgehd2.timings[0] += (helpTimeStop - helpTime);
    data->dgehd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dstegr (char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_dstegr(char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, Int * isuppz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dstegr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->dstegr.timings[0] += (helpTimeStop - helpTime);
    data->dstegr.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrf (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytrf(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrf((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrf.timings[0] += (helpTimeStop - helpTime);
    data->csytrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlasyf_aa (char* uplo, Int * j1, Int * m, Int * nb, double complex* a, Int * lda, Int * ipiv, double complex* h, Int * ldh, double complex* work, int len_uplo);
void hook_zlasyf_aa(char* uplo, Int * j1, Int * m, Int * nb, double complex* a, Int * lda, Int * ipiv, double complex* h, Int * ldh, double complex* work, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlasyf_aa((void*) uplo, (void*) j1, (void*) m, (void*) nb, (void*) a, (void*) lda, (void*) ipiv, (void*) h, (void*) ldh, (void*) work, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlasyf_aa.timings[0] += (helpTimeStop - helpTime);
    data->zlasyf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorbdb5 (Int * m1, Int * m2, Int * n, float* x1, Int * incx1, float* x2, Int * incx2, float* q1, Int * ldq1, float* q2, Int * ldq2, float* work, Int * lwork, Int * info);
void hook_sorbdb5(Int * m1, Int * m2, Int * n, float* x1, Int * incx1, float* x2, Int * incx2, float* q1, Int * ldq1, float* q2, Int * ldq2, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorbdb5((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb5.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb5.calls[0]++;

    return ;
}



extern void flexiblas_chain_clauum (char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_clauum(char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clauum((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clauum.timings[0] += (helpTimeStop - helpTime);
    data->clauum.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgglse (Int * m, Int * n, Int * p, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, double complex* d, double complex* x, double complex* work, Int * lwork, Int * info);
void hook_zgglse(Int * m, Int * n, Int * p, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, double complex* d, double complex* x, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgglse((void*) m, (void*) n, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) d, (void*) x, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgglse.timings[0] += (helpTimeStop - helpTime);
    data->zgglse.calls[0]++;

    return ;
}



extern int flexiblas_chain_dlaisnan (double* din1, double* din2);
int hook_dlaisnan(double* din1, double* din2)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlaisnan((void*) din1, (void*) din2);

    helpTimeStop = flexiblas_wtime();

    data->dlaisnan.timings[0] += (helpTimeStop - helpTime);
    data->dlaisnan.calls[0]++;

    return v;
}



extern void flexiblas_chain_zlahqr (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, Int * info);
void hook_zlahqr(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlahqr((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlahqr.timings[0] += (helpTimeStop - helpTime);
    data->zlahqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunbdb (char* trans, char* signs, Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x12, Int * ldx12, double complex* x21, Int * ldx21, double complex* x22, Int * ldx22, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* tauq2, double complex* work, Int * lwork, Int * info, int len_trans, int len_signs);
void hook_zunbdb(char* trans, char* signs, Int * m, Int * p, Int * q, double complex* x11, Int * ldx11, double complex* x12, Int * ldx12, double complex* x21, Int * ldx21, double complex* x22, Int * ldx22, double* theta, double* phi, double complex* taup1, double complex* taup2, double complex* tauq1, double complex* tauq2, double complex* work, Int * lwork, Int * info, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunbdb((void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) tauq2, (void*) work, (void*) lwork, (void*) info, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->zunbdb.timings[0] += (helpTimeStop - helpTime);
    data->zunbdb.calls[0]++;

    return ;
}



extern void flexiblas_chain_slatrs3 (char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, float* a, Int * lda, float* x, Int * ldx, float* scale, float* cnorm, float* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_slatrs3(char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, float* a, Int * lda, float* x, Int * ldx, float* scale, float* cnorm, float* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slatrs3((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) x, (void*) ldx, (void*) scale, (void*) cnorm, (void*) work, (void*) lwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->slatrs3.timings[0] += (helpTimeStop - helpTime);
    data->slatrs3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbevx (char* jobz, char* range, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dsbevx(char* jobz, char* range, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbevx.timings[0] += (helpTimeStop - helpTime);
    data->dsbevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhprfs (char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zhprfs(char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhprfs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhprfs.timings[0] += (helpTimeStop - helpTime);
    data->zhprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtplqt2 (Int * m, Int * n, Int * l, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, Int * info);
void hook_dtplqt2(Int * m, Int * n, Int * l, double* a, Int * lda, double* b, Int * ldb, double* t, Int * ldt, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtplqt2((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtplqt2.timings[0] += (helpTimeStop - helpTime);
    data->dtplqt2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarrj (Int * n, double* d, double* e2, Int * ifirst, Int * ilast, double* rtol, Int * offset, double* w, double* werr, double* work, Int * iwork, double* pivmin, double* spdiam, Int * info);
void hook_dlarrj(Int * n, double* d, double* e2, Int * ifirst, Int * ilast, double* rtol, Int * offset, double* w, double* werr, double* work, Int * iwork, double* pivmin, double* spdiam, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarrj((void*) n, (void*) d, (void*) e2, (void*) ifirst, (void*) ilast, (void*) rtol, (void*) offset, (void*) w, (void*) werr, (void*) work, (void*) iwork, (void*) pivmin, (void*) spdiam, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrj.timings[0] += (helpTimeStop - helpTime);
    data->dlarrj.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbtrs (char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_trans);
void hook_cgbtrs(char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float complex* ab, Int * ldab, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbtrs((void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->cgbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhegvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zhegvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhegvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhegvx.timings[0] += (helpTimeStop - helpTime);
    data->zhegvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetd2 (char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, Int * info, int len_uplo);
void hook_chetd2(char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetd2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetd2.timings[0] += (helpTimeStop - helpTime);
    data->chetd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zptsv (Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb, Int * info);
void hook_zptsv(Int * n, Int * nrhs, double* d, double complex* e, double complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zptsv((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zptsv.timings[0] += (helpTimeStop - helpTime);
    data->zptsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpoequb (Int * n, double* a, Int * lda, double* s, double* scond, double* amax, Int * info);
void hook_dpoequb(Int * n, double* a, Int * lda, double* s, double* scond, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpoequb((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dpoequb.timings[0] += (helpTimeStop - helpTime);
    data->dpoequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpftri (char* transr, char* uplo, Int * n, double* a, Int * info, int len_transr, int len_uplo);
void hook_dpftri(char* transr, char* uplo, Int * n, double* a, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpftri((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpftri.timings[0] += (helpTimeStop - helpTime);
    data->dpftri.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasq4 (Int * i0, Int * n0, float* z, Int * pp, Int * n0in, float* dmin, float* dmin1, float* dmin2, float* dn, float* dn1, float* dn2, float* tau, Int * ttype, float* g);
void hook_slasq4(Int * i0, Int * n0, float* z, Int * pp, Int * n0in, float* dmin, float* dmin1, float* dmin2, float* dn, float* dn1, float* dn2, float* tau, Int * ttype, float* g)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasq4((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) n0in, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dn1, (void*) dn2, (void*) tau, (void*) ttype, (void*) g);

    helpTimeStop = flexiblas_wtime();

    data->slasq4.timings[0] += (helpTimeStop - helpTime);
    data->slasq4.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqhp (char* uplo, Int * n, float complex* ap, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_claqhp(char* uplo, Int * n, float complex* ap, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqhp((void*) uplo, (void*) n, (void*) ap, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->claqhp.timings[0] += (helpTimeStop - helpTime);
    data->claqhp.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgbtrf (Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, Int * info);
void hook_cgbtrf(Int * m, Int * n, Int * kl, Int * ku, float complex* ab, Int * ldab, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgbtrf((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->cgbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgesdd (char* jobz, Int * m, Int * n, float complex* a, Int * lda, float* s, float complex* u, Int * ldu, float complex* vt, Int * ldvt, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info, int len_jobz);
void hook_cgesdd(char* jobz, Int * m, Int * n, float complex* a, Int * lda, float* s, float complex* u, Int * ldu, float complex* vt, Int * ldvt, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info, int len_jobz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgesdd((void*) jobz, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info, (int) len_jobz);

    helpTimeStop = flexiblas_wtime();

    data->cgesdd.timings[0] += (helpTimeStop - helpTime);
    data->cgesdd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtgsen (Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* q, Int * ldq, double* z, Int * ldz, Int * m, double* pl, double* pr, double* dif, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info);
void hook_dtgsen(Int * ijob, Int * wantq, Int * wantz, Int * select, Int * n, double* a, Int * lda, double* b, Int * ldb, double* alphar, double* alphai, double* beta, double* q, Int * ldq, double* z, Int * ldz, Int * m, double* pl, double* pr, double* dif, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtgsen((void*) ijob, (void*) wantq, (void*) wantz, (void*) select, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alphar, (void*) alphai, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) m, (void*) pl, (void*) pr, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dtgsen.timings[0] += (helpTimeStop - helpTime);
    data->dtgsen.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpftrf (char* transr, char* uplo, Int * n, float complex* a, Int * info, int len_transr, int len_uplo);
void hook_cpftrf(char* transr, char* uplo, Int * n, float complex* a, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpftrf((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpftrf.timings[0] += (helpTimeStop - helpTime);
    data->cpftrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqgb (Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed);
void hook_zlaqgb(Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqgb((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqgb.timings[0] += (helpTimeStop - helpTime);
    data->zlaqgb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasq4 (Int * i0, Int * n0, double* z, Int * pp, Int * n0in, double* dmin, double* dmin1, double* dmin2, double* dn, double* dn1, double* dn2, double* tau, Int * ttype, double* g);
void hook_dlasq4(Int * i0, Int * n0, double* z, Int * pp, Int * n0in, double* dmin, double* dmin1, double* dmin2, double* dn, double* dn1, double* dn2, double* tau, Int * ttype, double* g)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasq4((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) n0in, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dn1, (void*) dn2, (void*) tau, (void*) ttype, (void*) g);

    helpTimeStop = flexiblas_wtime();

    data->dlasq4.timings[0] += (helpTimeStop - helpTime);
    data->dlasq4.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlascl (char* type_bn, Int * kl, Int * ku, double* cfrom, double* cto, Int * m, Int * n, double complex* a, Int * lda, Int * info, int len_type_bn);
void hook_zlascl(char* type_bn, Int * kl, Int * ku, double* cfrom, double* cto, Int * m, Int * n, double complex* a, Int * lda, Int * info, int len_type_bn)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlascl((void*) type_bn, (void*) kl, (void*) ku, (void*) cfrom, (void*) cto, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_type_bn);

    helpTimeStop = flexiblas_wtime();

    data->zlascl.timings[0] += (helpTimeStop - helpTime);
    data->zlascl.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrti2 (char* uplo, char* diag, Int * n, float complex* a, Int * lda, Int * info, int len_uplo, int len_diag);
void hook_ctrti2(char* uplo, char* diag, Int * n, float complex* a, Int * lda, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrti2((void*) uplo, (void*) diag, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctrti2.timings[0] += (helpTimeStop - helpTime);
    data->ctrti2.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqsy (char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed);
void hook_claqsy(char* uplo, Int * n, float complex* a, Int * lda, float* s, float* scond, float* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqsy((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->claqsy.timings[0] += (helpTimeStop - helpTime);
    data->claqsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelqf (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgelqf(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqf.timings[0] += (helpTimeStop - helpTime);
    data->sgelqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgtts2 (Int * itrans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float complex* b, Int * ldb);
void hook_cgtts2(Int * itrans, Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* du2, Int * ipiv, float complex* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgtts2((void*) itrans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->cgtts2.timings[0] += (helpTimeStop - helpTime);
    data->cgtts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlatbs (char* uplo, char* trans, char* diag, char* normin, Int * n, Int * kd, double* ab, Int * ldab, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_dlatbs(char* uplo, char* trans, char* diag, char* normin, Int * n, Int * kd, double* ab, Int * ldab, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatbs((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->dlatbs.timings[0] += (helpTimeStop - helpTime);
    data->dlatbs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cppcon (char* uplo, Int * n, float complex* ap, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_cppcon(char* uplo, Int * n, float complex* ap, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cppcon((void*) uplo, (void*) n, (void*) ap, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cppcon.timings[0] += (helpTimeStop - helpTime);
    data->cppcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed1 (Int * n, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* work, Int * iwork, Int * info);
void hook_slaed1(Int * n, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed1((void*) n, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed1.timings[0] += (helpTimeStop - helpTime);
    data->slaed1.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsysvx (char* fact, char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, Int * lwork, double* rwork, Int * info, int len_fact, int len_uplo);
void hook_zsysvx(char* fact, char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, Int * lwork, double* rwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsysvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsysvx.timings[0] += (helpTimeStop - helpTime);
    data->zsysvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cppsv (char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_cppsv(char* uplo, Int * n, Int * nrhs, float complex* ap, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cppsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cppsv.timings[0] += (helpTimeStop - helpTime);
    data->cppsv.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlantp (char* norm, char* uplo, char* diag, Int * n, double complex* ap, double* work, int len_norm, int len_uplo, int len_diag);
double hook_zlantp(char* norm, char* uplo, char* diag, Int * n, double complex* ap, double* work, int len_norm, int len_uplo, int len_diag)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlantp((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->zlantp.timings[0] += (helpTimeStop - helpTime);
    data->zlantp.calls[0]++;

    return v;
}



extern void flexiblas_chain_chetrd (char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetrd(char* uplo, Int * n, float complex* a, Int * lda, float* d, float* e, float complex* tau, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrd((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrd.timings[0] += (helpTimeStop - helpTime);
    data->chetrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrf_aa (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetrf_aa(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrf_aa((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->zhetrf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgevc (char* side, char* howmny, Int * select, Int * n, double complex* s, Int * lds, double complex* p, Int * ldp, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * info, int len_side, int len_howmny);
void hook_ztgevc(char* side, char* howmny, Int * select, Int * n, double complex* s, Int * lds, double complex* p, Int * ldp, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, Int * mm, Int * m, double complex* work, double* rwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgevc((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) s, (void*) lds, (void*) p, (void*) ldp, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) rwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ztgevc.timings[0] += (helpTimeStop - helpTime);
    data->ztgevc.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasd4 (Int * n, Int * i, float* d, float* z, float* delta, float* rho, float* sigma, float* work, Int * info);
void hook_slasd4(Int * n, Int * i, float* d, float* z, float* delta, float* rho, float* sigma, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasd4((void*) n, (void*) i, (void*) d, (void*) z, (void*) delta, (void*) rho, (void*) sigma, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd4.timings[0] += (helpTimeStop - helpTime);
    data->slasd4.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetri (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo);
void hook_chetri(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetri.timings[0] += (helpTimeStop - helpTime);
    data->chetri.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytri (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * info, int len_uplo);
void hook_ssytri(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri.timings[0] += (helpTimeStop - helpTime);
    data->ssytri.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgemlqt (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans);
void hook_sgemlqt(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgemlqt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgemlqt.timings[0] += (helpTimeStop - helpTime);
    data->sgemlqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_chptri (char* uplo, Int * n, float complex* ap, Int * ipiv, float complex* work, Int * info, int len_uplo);
void hook_chptri(char* uplo, Int * n, float complex* ap, Int * ipiv, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chptri((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chptri.timings[0] += (helpTimeStop - helpTime);
    data->chptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhesv_rk (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhesv_rk(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* e, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhesv_rk((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhesv_rk.timings[0] += (helpTimeStop - helpTime);
    data->zhesv_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed0 (Int * icompq, Int * qsiz, Int * n, double* d, double* e, double* q, Int * ldq, double* qstore, Int * ldqs, double* work, Int * iwork, Int * info);
void hook_dlaed0(Int * icompq, Int * qsiz, Int * n, double* d, double* e, double* q, Int * ldq, double* qstore, Int * ldqs, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed0((void*) icompq, (void*) qsiz, (void*) n, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) qstore, (void*) ldqs, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed0.timings[0] += (helpTimeStop - helpTime);
    data->dlaed0.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarz (char* side, Int * m, Int * n, Int * l, float complex* v, Int * incv, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side);
void hook_clarz(char* side, Int * m, Int * n, Int * l, float complex* v, Int * incv, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarz((void*) side, (void*) m, (void*) n, (void*) l, (void*) v, (void*) incv, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->clarz.timings[0] += (helpTimeStop - helpTime);
    data->clarz.calls[0]++;

    return ;
}



extern void flexiblas_chain_srscl (Int * n, float* sa, float* sx, Int * incx);
void hook_srscl(Int * n, float* sa, float* sx, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_srscl((void*) n, (void*) sa, (void*) sx, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->srscl.timings[0] += (helpTimeStop - helpTime);
    data->srscl.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheev_2stage (char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo);
void hook_cheev_2stage(char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheev_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheev_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgecon (char* norm, Int * n, double complex* a, Int * lda, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_norm);
void hook_zgecon(char* norm, Int * n, double complex* a, Int * lda, double* anorm, double* rcond, double complex* work, double* rwork, Int * info, int len_norm)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgecon((void*) norm, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->zgecon.timings[0] += (helpTimeStop - helpTime);
    data->zgecon.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqz0 (char* wants, char* wantq, char* wantz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* alpha, double complex* beta, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * rec, Int * info, int len_wants, int len_wantq, int len_wantz);
void hook_zlaqz0(char* wants, char* wantq, char* wantz, Int * n, Int * ilo, Int * ihi, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* alpha, double complex* beta, double complex* q, Int * ldq, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * rec, Int * info, int len_wants, int len_wantq, int len_wantz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqz0((void*) wants, (void*) wantq, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) rec, (void*) info, (int) len_wants, (int) len_wantq, (int) len_wantz);

    helpTimeStop = flexiblas_wtime();

    data->zlaqz0.timings[0] += (helpTimeStop - helpTime);
    data->zlaqz0.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgejsv (char* joba, char* jobu, char* jobv, char* jobr, char* jobt, char* jobp, Int * m, Int * n, float complex* a, Int * lda, float* sva, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* cwork, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * info, int len_joba, int len_jobu, int len_jobv, int len_jobr, int len_jobt, int len_jobp);
void hook_cgejsv(char* joba, char* jobu, char* jobv, char* jobr, char* jobt, char* jobp, Int * m, Int * n, float complex* a, Int * lda, float* sva, float complex* u, Int * ldu, float complex* v, Int * ldv, float complex* cwork, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * info, int len_joba, int len_jobu, int len_jobv, int len_jobr, int len_jobt, int len_jobp)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgejsv((void*) joba, (void*) jobu, (void*) jobv, (void*) jobr, (void*) jobt, (void*) jobp, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) sva, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) cwork, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) info, (int) len_joba, (int) len_jobu, (int) len_jobv, (int) len_jobr, (int) len_jobt, (int) len_jobp);

    helpTimeStop = flexiblas_wtime();

    data->cgejsv.timings[0] += (helpTimeStop - helpTime);
    data->cgejsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytrs2 (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * info, int len_uplo);
void hook_zsytrs2(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytrs2((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytrs2.timings[0] += (helpTimeStop - helpTime);
    data->zsytrs2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbgst (char* vect, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* x, Int * ldx, double* work, Int * info, int len_vect, int len_uplo);
void hook_dsbgst(char* vect, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* x, Int * ldx, double* work, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbgst((void*) vect, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) x, (void*) ldx, (void*) work, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbgst.timings[0] += (helpTimeStop - helpTime);
    data->dsbgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqz1 (double* a, Int * lda, double* b, Int * ldb, double* sr1, double* sr2, double* si, double* beta1, double* beta2, double* v);
void hook_dlaqz1(double* a, Int * lda, double* b, Int * ldb, double* sr1, double* sr2, double* si, double* beta1, double* beta2, double* v)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqz1((void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sr1, (void*) sr2, (void*) si, (void*) beta1, (void*) beta2, (void*) v);

    helpTimeStop = flexiblas_wtime();

    data->dlaqz1.timings[0] += (helpTimeStop - helpTime);
    data->dlaqz1.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqge (Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, char* equed, int len_equed);
void hook_slaqge(Int * m, Int * n, float* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqge((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->slaqge.timings[0] += (helpTimeStop - helpTime);
    data->slaqge.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb (char* trans, char* signs, Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x12, Int * ldx12, float complex* x21, Int * ldx21, float complex* x22, Int * ldx22, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* tauq2, float complex* work, Int * lwork, Int * info, int len_trans, int len_signs);
void hook_cunbdb(char* trans, char* signs, Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x12, Int * ldx12, float complex* x21, Int * ldx21, float complex* x22, Int * ldx22, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* tauq2, float complex* work, Int * lwork, Int * info, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb((void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) tauq2, (void*) work, (void*) lwork, (void*) info, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsptrs (char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zsptrs(char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsptrs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsptrs.timings[0] += (helpTimeStop - helpTime);
    data->zsptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyconv (char* uplo, char* way, Int * n, float* a, Int * lda, Int * ipiv, float* e, Int * info, int len_uplo, int len_way);
void hook_ssyconv(char* uplo, char* way, Int * n, float* a, Int * lda, Int * ipiv, float* e, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyconv((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) e, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->ssyconv.timings[0] += (helpTimeStop - helpTime);
    data->ssyconv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgesvdq (char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, double* a, Int * lda, double* s, double* u, Int * ldu, double* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, double* work, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv);
void hook_dgesvdq(char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, double* a, Int * lda, double* s, double* u, Int * ldu, double* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, double* work, Int * lwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgesvdq((void*) joba, (void*) jobp, (void*) jobr, (void*) jobu, (void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) numrank, (void*) iwork, (void*) liwork, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_joba, (int) len_jobp, (int) len_jobr, (int) len_jobu, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->dgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->dgesvdq.calls[0]++;

    return ;
}



extern void flexiblas_chain_cungbr (char* vect, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info, int len_vect);
void hook_cungbr(char* vect, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info, int len_vect)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungbr((void*) vect, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_vect);

    helpTimeStop = flexiblas_wtime();

    data->cungbr.timings[0] += (helpTimeStop - helpTime);
    data->cungbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgbtrf (Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, Int * ipiv, Int * info);
void hook_dgbtrf(Int * m, Int * n, Int * kl, Int * ku, double* ab, Int * ldab, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgbtrf((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->dgbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarrc (char* jobt, Int * n, double* vl, double* vu, double* d, double* e, double* pivmin, Int * eigcnt, Int * lcnt, Int * rcnt, Int * info, int len_jobt);
void hook_dlarrc(char* jobt, Int * n, double* vl, double* vu, double* d, double* e, double* pivmin, Int * eigcnt, Int * lcnt, Int * rcnt, Int * info, int len_jobt)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarrc((void*) jobt, (void*) n, (void*) vl, (void*) vu, (void*) d, (void*) e, (void*) pivmin, (void*) eigcnt, (void*) lcnt, (void*) rcnt, (void*) info, (int) len_jobt);

    helpTimeStop = flexiblas_wtime();

    data->dlarrc.timings[0] += (helpTimeStop - helpTime);
    data->dlarrc.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbtrf (char* uplo, Int * n, Int * kd, double* ab, Int * ldab, Int * info, int len_uplo);
void hook_dpbtrf(char* uplo, Int * n, Int * kd, double* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbtrf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpbtrf.timings[0] += (helpTimeStop - helpTime);
    data->dpbtrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeev (char* jobvl, char* jobvr, Int * n, float complex* a, Int * lda, float complex* w, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobvl, int len_jobvr);
void hook_cgeev(char* jobvl, char* jobvr, Int * n, float complex* a, Int * lda, float complex* w, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeev((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->cgeev.timings[0] += (helpTimeStop - helpTime);
    data->cgeev.calls[0]++;

    return ;
}



extern void flexiblas_chain_dhsein (char* side, char* eigsrc, char* initv, Int * select, Int * n, double* h, Int * ldh, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, Int * mm, Int * m, double* work, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv);
void hook_dhsein(char* side, char* eigsrc, char* initv, Int * select, Int * n, double* h, Int * ldh, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, Int * mm, Int * m, double* work, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dhsein((void*) side, (void*) eigsrc, (void*) initv, (void*) select, (void*) n, (void*) h, (void*) ldh, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) ifaill, (void*) ifailr, (void*) info, (int) len_side, (int) len_eigsrc, (int) len_initv);

    helpTimeStop = flexiblas_wtime();

    data->dhsein.timings[0] += (helpTimeStop - helpTime);
    data->dhsein.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpttrf (Int * n, float* d, float complex* e, Int * info);
void hook_cpttrf(Int * n, float* d, float complex* e, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpttrf((void*) n, (void*) d, (void*) e, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cpttrf.timings[0] += (helpTimeStop - helpTime);
    data->cpttrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zunm2r (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans);
void hook_zunm2r(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* c, Int * ldc, double complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zunm2r((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zunm2r.timings[0] += (helpTimeStop - helpTime);
    data->zunm2r.calls[0]++;

    return ;
}



extern void flexiblas_chain_dppcon (char* uplo, Int * n, double* ap, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dppcon(char* uplo, Int * n, double* ap, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dppcon((void*) uplo, (void*) n, (void*) ap, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dppcon.timings[0] += (helpTimeStop - helpTime);
    data->dppcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztprfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ztprfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztprfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztprfs.timings[0] += (helpTimeStop - helpTime);
    data->ztprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dposv (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dposv(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dposv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dposv.timings[0] += (helpTimeStop - helpTime);
    data->dposv.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssygv_2stage (Int * itype, char* jobz, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_ssygv_2stage(Int * itype, char* jobz, char* uplo, Int * n, float* a, Int * lda, float* b, Int * ldb, float* w, float* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssygv_2stage((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssygv_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssygv_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlacgv (Int * n, double complex* x, Int * incx);
void hook_zlacgv(Int * n, double complex* x, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlacgv((void*) n, (void*) x, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->zlacgv.timings[0] += (helpTimeStop - helpTime);
    data->zlacgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zspsv (char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zspsv(char* uplo, Int * n, Int * nrhs, double complex* ap, Int * ipiv, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zspsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zspsv.timings[0] += (helpTimeStop - helpTime);
    data->zspsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarfb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, double* v, Int * ldv, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_dlarfb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, double* v, Int * ldv, double* t, Int * ldt, double* c, Int * ldc, double* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarfb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->dlarfb.timings[0] += (helpTimeStop - helpTime);
    data->dlarfb.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrs_rook (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chetrs_rook(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpbrfs (char* uplo, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dpbrfs(char* uplo, Int * n, Int * kd, Int * nrhs, double* ab, Int * ldab, double* afb, Int * ldafb, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpbrfs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dpbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dpbrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_csysv_rk (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csysv_rk(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csysv_rk((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csysv_rk.timings[0] += (helpTimeStop - helpTime);
    data->csysv_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpgst (Int * itype, char* uplo, Int * n, double complex* ap, double complex* bp, Int * info, int len_uplo);
void hook_zhpgst(Int * itype, char* uplo, Int * n, double complex* ap, double complex* bp, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpgst((void*) itype, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpgst.timings[0] += (helpTimeStop - helpTime);
    data->zhpgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_dppsvx (char* fact, char* uplo, Int * n, Int * nrhs, double* ap, double* afp, char* equed, double* s, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed);
void hook_dppsvx(char* fact, char* uplo, Int * n, Int * nrhs, double* ap, double* afp, char* equed, double* s, double* b, Int * ldb, double* x, Int * ldx, double* rcond, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_fact, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dppsvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) equed, (void*) s, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_fact, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->dppsvx.timings[0] += (helpTimeStop - helpTime);
    data->dppsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zppsv (char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zppsv(char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zppsv((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zppsv.timings[0] += (helpTimeStop - helpTime);
    data->zppsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqhp (char* uplo, Int * n, double complex* ap, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_zlaqhp(char* uplo, Int * n, double complex* ap, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqhp((void*) uplo, (void*) n, (void*) ap, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqhp.timings[0] += (helpTimeStop - helpTime);
    data->zlaqhp.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlapmr (Int * forwrd, Int * m, Int * n, double complex* x, Int * ldx, Int * k);
void hook_zlapmr(Int * forwrd, Int * m, Int * n, double complex* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlapmr((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->zlapmr.timings[0] += (helpTimeStop - helpTime);
    data->zlapmr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgelsy (Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * jpvt, float* rcond, Int * rank_bn, float* work, Int * lwork, Int * info);
void hook_sgelsy(Int * m, Int * n, Int * nrhs, float* a, Int * lda, float* b, Int * ldb, Int * jpvt, float* rcond, Int * rank_bn, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgelsy((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) jpvt, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelsy.timings[0] += (helpTimeStop - helpTime);
    data->sgelsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytri2 (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssytri2(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri2.timings[0] += (helpTimeStop - helpTime);
    data->ssytri2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spbtrs (char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, Int * info, int len_uplo);
void hook_spbtrs(char* uplo, Int * n, Int * kd, Int * nrhs, float* ab, Int * ldab, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spbtrs((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spbtrs.timings[0] += (helpTimeStop - helpTime);
    data->spbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggbak (char* job, char* side, Int * n, Int * ilo, Int * ihi, float* lscale, float* rscale, Int * m, float complex* v, Int * ldv, Int * info, int len_job, int len_side);
void hook_cggbak(char* job, char* side, Int * n, Int * ilo, Int * ihi, float* lscale, float* rscale, Int * m, float complex* v, Int * ldv, Int * info, int len_job, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggbak((void*) job, (void*) side, (void*) n, (void*) ilo, (void*) ihi, (void*) lscale, (void*) rscale, (void*) m, (void*) v, (void*) ldv, (void*) info, (int) len_job, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->cggbak.timings[0] += (helpTimeStop - helpTime);
    data->cggbak.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlacon (Int * n, double* v, double* x, Int * isgn, double* est, Int * kase);
void hook_dlacon(Int * n, double* v, double* x, Int * isgn, double* est, Int * kase)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlacon((void*) n, (void*) v, (void*) x, (void*) isgn, (void*) est, (void*) kase);

    helpTimeStop = flexiblas_wtime();

    data->dlacon.timings[0] += (helpTimeStop - helpTime);
    data->dlacon.calls[0]++;

    return ;
}



extern double flexiblas_chain_dlansp (char* norm, char* uplo, Int * n, double* ap, double* work, int len_norm, int len_uplo);
double hook_dlansp(char* norm, char* uplo, Int * n, double* ap, double* work, int len_norm, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlansp((void*) norm, (void*) uplo, (void*) n, (void*) ap, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlansp.timings[0] += (helpTimeStop - helpTime);
    data->dlansp.calls[0]++;

    return v;
}



extern void flexiblas_chain_dgebd2 (Int * m, Int * n, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* work, Int * info);
void hook_dgebd2(Int * m, Int * n, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgebd2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgebd2.timings[0] += (helpTimeStop - helpTime);
    data->dgebd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrs_aa (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetrs_aa(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, Int * ipiv, double complex* b, Int * ldb, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrs_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->zhetrs_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrs_aa_2stage (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* tb, Int * ltb, Int * ipiv, Int * ipiv2, float* b, Int * ldb, Int * info, int len_uplo);
void hook_ssytrs_aa_2stage(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* tb, Int * ltb, Int * ipiv, Int * ipiv2, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrs_aa_2stage((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqz3 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, Int * ns, Int * nd, float* alphar, float* alphai, float* beta, float* qc, Int * ldqc, float* zc, Int * ldzc, float* work, Int * lwork, Int * rec, Int * info);
void hook_slaqz3(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nw, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, Int * ns, Int * nd, float* alphar, float* alphai, float* beta, float* qc, Int * ldqc, float* zc, Int * ldzc, float* work, Int * lwork, Int * rec, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqz3((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nw, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) alphar, (void*) alphai, (void*) beta, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) rec, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqz3.timings[0] += (helpTimeStop - helpTime);
    data->slaqz3.calls[0]++;

    return ;
}



extern void flexiblas_chain_stprfb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_stprfb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, Int * l, float* v, Int * ldv, float* t, Int * ldt, float* a, Int * lda, float* b, Int * ldb, float* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stprfb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) l, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->stprfb.timings[0] += (helpTimeStop - helpTime);
    data->stprfb.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetri_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo);
void hook_chetri_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetri_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetri_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetri_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo);
void hook_csytri_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytri_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_slamswlq (char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_slamswlq(char* side, char* trans, Int * m, Int * n, Int * k, Int * mb, Int * nb, float* a, Int * lda, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slamswlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->slamswlq.timings[0] += (helpTimeStop - helpTime);
    data->slamswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_clarfx (char* side, Int * m, Int * n, float complex* v, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side);
void hook_clarfx(char* side, Int * m, Int * n, float complex* v, float complex* tau, float complex* c, Int * ldc, float complex* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clarfx((void*) side, (void*) m, (void*) n, (void*) v, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->clarfx.timings[0] += (helpTimeStop - helpTime);
    data->clarfx.calls[0]++;

    return ;
}



extern void flexiblas_chain_slag2d (Int * m, Int * n, float* sa, Int * ldsa, double* a, Int * lda, Int * info);
void hook_slag2d(Int * m, Int * n, float* sa, Int * ldsa, double* a, Int * lda, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slag2d((void*) m, (void*) n, (void*) sa, (void*) ldsa, (void*) a, (void*) lda, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slag2d.timings[0] += (helpTimeStop - helpTime);
    data->slag2d.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetc2 (Int * n, double* a, Int * lda, Int * ipiv, Int * jpiv, Int * info);
void hook_dgetc2(Int * n, double* a, Int * lda, Int * ipiv, Int * jpiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetc2((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) jpiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetc2.timings[0] += (helpTimeStop - helpTime);
    data->dgetc2.calls[0]++;

    return ;
}



extern void flexiblas_chain_chesv_rk (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chesv_rk(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chesv_rk((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chesv_rk.timings[0] += (helpTimeStop - helpTime);
    data->chesv_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsygv (Int * itype, char* jobz, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, double* w, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo);
void hook_dsygv(Int * itype, char* jobz, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, double* w, double* work, Int * lwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsygv((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsygv.timings[0] += (helpTimeStop - helpTime);
    data->dsygv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgesdd (char* jobz, Int * m, Int * n, double* a, Int * lda, double* s, double* u, Int * ldu, double* vt, Int * ldvt, double* work, Int * lwork, Int * iwork, Int * info, int len_jobz);
void hook_dgesdd(char* jobz, Int * m, Int * n, double* a, Int * lda, double* s, double* u, Int * ldu, double* vt, Int * ldvt, double* work, Int * lwork, Int * iwork, Int * info, int len_jobz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgesdd((void*) jobz, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobz);

    helpTimeStop = flexiblas_wtime();

    data->dgesdd.timings[0] += (helpTimeStop - helpTime);
    data->dgesdd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpocon (char* uplo, Int * n, float complex* a, Int * lda, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo);
void hook_cpocon(char* uplo, Int * n, float complex* a, Int * lda, float* anorm, float* rcond, float complex* work, float* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpocon((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) anorm, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpocon.timings[0] += (helpTimeStop - helpTime);
    data->cpocon.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetrd_2stage (char* vect, char* uplo, Int * n, double complex* a, Int * lda, double* d, double* e, double complex* tau, double complex* hous2, Int * lhous2, double complex* work, Int * lwork, Int * info, int len_vect, int len_uplo);
void hook_zhetrd_2stage(char* vect, char* uplo, Int * n, double complex* a, Int * lda, double* d, double* e, double complex* tau, double complex* hous2, Int * lhous2, double complex* work, Int * lwork, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetrd_2stage((void*) vect, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) hous2, (void*) lhous2, (void*) work, (void*) lwork, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhetrd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgelsy (Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * jpvt, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * info);
void hook_zgelsy(Int * m, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, Int * jpvt, double* rcond, Int * rank_bn, double complex* work, Int * lwork, double* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgelsy((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) jpvt, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgelsy.timings[0] += (helpTimeStop - helpTime);
    data->zgelsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_stzrzf (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_stzrzf(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stzrzf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stzrzf.timings[0] += (helpTimeStop - helpTime);
    data->stzrzf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsygst (Int * itype, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo);
void hook_dsygst(Int * itype, char* uplo, Int * n, double* a, Int * lda, double* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsygst((void*) itype, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsygst.timings[0] += (helpTimeStop - helpTime);
    data->dsygst.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlatps (char* uplo, char* trans, char* diag, char* normin, Int * n, double* ap, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_dlatps(char* uplo, char* trans, char* diag, char* normin, Int * n, double* ap, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatps((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) ap, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->dlatps.timings[0] += (helpTimeStop - helpTime);
    data->dlatps.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpbstf (char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, Int * info, int len_uplo);
void hook_cpbstf(char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpbstf((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpbstf.timings[0] += (helpTimeStop - helpTime);
    data->cpbstf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarzt (char* direct, char* storev, Int * n, Int * k, double complex* v, Int * ldv, double complex* tau, double complex* t, Int * ldt, int len_direct, int len_storev);
void hook_zlarzt(char* direct, char* storev, Int * n, Int * k, double complex* v, Int * ldv, double complex* tau, double complex* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarzt((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->zlarzt.timings[0] += (helpTimeStop - helpTime);
    data->zlarzt.calls[0]++;

    return ;
}



extern void flexiblas_chain_csycon_3 (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo);
void hook_csycon_3(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csycon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csycon_3.timings[0] += (helpTimeStop - helpTime);
    data->csycon_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspgst (Int * itype, char* uplo, Int * n, double* ap, double* bp, Int * info, int len_uplo);
void hook_dspgst(Int * itype, char* uplo, Int * n, double* ap, double* bp, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspgst((void*) itype, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspgst.timings[0] += (helpTimeStop - helpTime);
    data->dspgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeqlf (Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info);
void hook_zgeqlf(Int * m, Int * n, double complex* a, Int * lda, double complex* tau, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeqlf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->zgeqlf.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasd3 (Int * nl, Int * nr, Int * sqre, Int * k, float* d, float* q, Int * ldq, float* dsigma, float* u, Int * ldu, float* u2, Int * ldu2, float* vt, Int * ldvt, float* vt2, Int * ldvt2, Int * idxc, Int * ctot, float* z, Int * info);
void hook_slasd3(Int * nl, Int * nr, Int * sqre, Int * k, float* d, float* q, Int * ldq, float* dsigma, float* u, Int * ldu, float* u2, Int * ldu2, float* vt, Int * ldvt, float* vt2, Int * ldvt2, Int * idxc, Int * ctot, float* z, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasd3((void*) nl, (void*) nr, (void*) sqre, (void*) k, (void*) d, (void*) q, (void*) ldq, (void*) dsigma, (void*) u, (void*) ldu, (void*) u2, (void*) ldu2, (void*) vt, (void*) ldvt, (void*) vt2, (void*) ldvt2, (void*) idxc, (void*) ctot, (void*) z, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd3.timings[0] += (helpTimeStop - helpTime);
    data->slasd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtptri (char* uplo, char* diag, Int * n, double* ap, Int * info, int len_uplo, int len_diag);
void hook_dtptri(char* uplo, char* diag, Int * n, double* ap, Int * info, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtptri((void*) uplo, (void*) diag, (void*) n, (void*) ap, (void*) info, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtptri.timings[0] += (helpTimeStop - helpTime);
    data->dtptri.calls[0]++;

    return ;
}



extern void flexiblas_chain_spoequ (Int * n, float* a, Int * lda, float* s, float* scond, float* amax, Int * info);
void hook_spoequ(Int * n, float* a, Int * lda, float* s, float* scond, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spoequ((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->spoequ.timings[0] += (helpTimeStop - helpTime);
    data->spoequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrevc3 (char* side, char* howmny, Int * select, Int * n, float complex* t, Int * ldt, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * info, int len_side, int len_howmny);
void hook_ctrevc3(char* side, char* howmny, Int * select, Int * n, float complex* t, Int * ldt, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrevc3((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->ctrevc3.timings[0] += (helpTimeStop - helpTime);
    data->ctrevc3.calls[0]++;

    return ;
}



extern void flexiblas_chain_clacgv (Int * n, float complex* x, Int * incx);
void hook_clacgv(Int * n, float complex* x, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clacgv((void*) n, (void*) x, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->clacgv.timings[0] += (helpTimeStop - helpTime);
    data->clacgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggsvp3 (char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* tola, double* tolb, Int * k, Int * l, double complex* u, Int * ldu, double complex* v, Int * ldv, double complex* q, Int * ldq, Int * iwork, double* rwork, double complex* tau, double complex* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_zggsvp3(char* jobu, char* jobv, char* jobq, Int * m, Int * p, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* tola, double* tolb, Int * k, Int * l, double complex* u, Int * ldu, double complex* v, Int * ldv, double complex* q, Int * ldq, Int * iwork, double* rwork, double complex* tau, double complex* work, Int * lwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggsvp3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) p, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) tola, (void*) tolb, (void*) k, (void*) l, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) iwork, (void*) rwork, (void*) tau, (void*) work, (void*) lwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->zggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->zggsvp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqge (Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, char* equed, int len_equed);
void hook_claqge(Int * m, Int * n, float complex* a, Int * lda, float* r, float* c, float* rowcnd, float* colcnd, float* amax, char* equed, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqge((void*) m, (void*) n, (void*) a, (void*) lda, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) equed, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->claqge.timings[0] += (helpTimeStop - helpTime);
    data->claqge.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssfrk (char* transr, char* uplo, char* trans, Int * n, Int * k, float* alpha, float* a, Int * lda, float* beta, float* c, int len_transr, int len_uplo, int len_trans);
void hook_ssfrk(char* transr, char* uplo, char* trans, Int * n, Int * k, float* alpha, float* a, Int * lda, float* beta, float* c, int len_transr, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssfrk((void*) transr, (void*) uplo, (void*) trans, (void*) n, (void*) k, (void*) alpha, (void*) a, (void*) lda, (void*) beta, (void*) c, (int) len_transr, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ssfrk.timings[0] += (helpTimeStop - helpTime);
    data->ssfrk.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheev (char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo);
void hook_cheev(char* jobz, char* uplo, Int * n, float complex* a, Int * lda, float* w, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheev((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheev.timings[0] += (helpTimeStop - helpTime);
    data->cheev.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaswlq (Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info);
void hook_dlaswlq(Int * m, Int * n, Int * mb, Int * nb, double* a, Int * lda, double* t, Int * ldt, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaswlq((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaswlq.timings[0] += (helpTimeStop - helpTime);
    data->dlaswlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztfsm (char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, double complex* alpha, double complex* a, double complex* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag);
void hook_ztfsm(char* transr, char* side, char* uplo, char* trans, char* diag, Int * m, Int * n, double complex* alpha, double complex* a, double complex* b, Int * ldb, int len_transr, int len_side, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztfsm((void*) transr, (void*) side, (void*) uplo, (void*) trans, (void*) diag, (void*) m, (void*) n, (void*) alpha, (void*) a, (void*) b, (void*) ldb, (int) len_transr, (int) len_side, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztfsm.timings[0] += (helpTimeStop - helpTime);
    data->ztfsm.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgesvdx (char* jobu, char* jobvt, char* range, Int * m, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double complex* u, Int * ldu, double complex* vt, Int * ldvt, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info, int len_jobu, int len_jobvt, int len_range);
void hook_zgesvdx(char* jobu, char* jobvt, char* range, Int * m, Int * n, double complex* a, Int * lda, double* vl, double* vu, Int * il, Int * iu, Int * ns, double* s, double complex* u, Int * ldu, double complex* vt, Int * ldvt, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * info, int len_jobu, int len_jobvt, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgesvdx((void*) jobu, (void*) jobvt, (void*) range, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) ns, (void*) s, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info, (int) len_jobu, (int) len_jobvt, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->zgesvdx.timings[0] += (helpTimeStop - helpTime);
    data->zgesvdx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb2 (Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info);
void hook_cunbdb2(Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb2((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb2.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrttf (char* transr, char* uplo, Int * n, float complex* a, Int * lda, float complex* arf, Int * info, int len_transr, int len_uplo);
void hook_ctrttf(char* transr, char* uplo, Int * n, float complex* a, Int * lda, float complex* arf, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrttf((void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) arf, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ctrttf.timings[0] += (helpTimeStop - helpTime);
    data->ctrttf.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssptrd (char* uplo, Int * n, float* ap, float* d, float* e, float* tau, Int * info, int len_uplo);
void hook_ssptrd(char* uplo, Int * n, float* ap, float* d, float* e, float* tau, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssptrd((void*) uplo, (void*) n, (void*) ap, (void*) d, (void*) e, (void*) tau, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssptrd.timings[0] += (helpTimeStop - helpTime);
    data->ssptrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorbdb5 (Int * m1, Int * m2, Int * n, double* x1, Int * incx1, double* x2, Int * incx2, double* q1, Int * ldq1, double* q2, Int * ldq2, double* work, Int * lwork, Int * info);
void hook_dorbdb5(Int * m1, Int * m2, Int * n, double* x1, Int * incx1, double* x2, Int * incx2, double* q1, Int * ldq1, double* q2, Int * ldq2, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorbdb5((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb5.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb5.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztgsyl (char* trans, Int * ijob, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double complex* d, Int * ldd, double complex* e, Int * lde, double complex* f, Int * ldf, double* scale, double* dif, double complex* work, Int * lwork, Int * iwork, Int * info, int len_trans);
void hook_ztgsyl(char* trans, Int * ijob, Int * m, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* c, Int * ldc, double complex* d, Int * ldd, double complex* e, Int * lde, double complex* f, Int * ldf, double* scale, double* dif, double complex* work, Int * lwork, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztgsyl((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->ztgsyl.timings[0] += (helpTimeStop - helpTime);
    data->ztgsyl.calls[0]++;

    return ;
}



extern void flexiblas_chain_clasyf (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clasyf(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clasyf((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clasyf.timings[0] += (helpTimeStop - helpTime);
    data->clasyf.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqr5 (Int * wantt, Int * wantz, Int * kacc22, Int * n, Int * ktop, Int * kbot, Int * nshfts, float complex* s, float complex* h, Int * ldh, Int * iloz, Int * ihiz, float complex* z, Int * ldz, float complex* v, Int * ldv, float complex* u, Int * ldu, Int * nv, float complex* wv, Int * ldwv, Int * nh, float complex* wh, Int * ldwh);
void hook_claqr5(Int * wantt, Int * wantz, Int * kacc22, Int * n, Int * ktop, Int * kbot, Int * nshfts, float complex* s, float complex* h, Int * ldh, Int * iloz, Int * ihiz, float complex* z, Int * ldz, float complex* v, Int * ldv, float complex* u, Int * ldu, Int * nv, float complex* wv, Int * ldwv, Int * nh, float complex* wh, Int * ldwh)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqr5((void*) wantt, (void*) wantz, (void*) kacc22, (void*) n, (void*) ktop, (void*) kbot, (void*) nshfts, (void*) s, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) v, (void*) ldv, (void*) u, (void*) ldu, (void*) nv, (void*) wv, (void*) ldwv, (void*) nh, (void*) wh, (void*) ldwh);

    helpTimeStop = flexiblas_wtime();

    data->claqr5.timings[0] += (helpTimeStop - helpTime);
    data->claqr5.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztrrfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ztrrfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztrrfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztrrfs.timings[0] += (helpTimeStop - helpTime);
    data->ztrrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqp3 (Int * m, Int * n, float* a, Int * lda, Int * jpvt, float* tau, float* work, Int * lwork, Int * info);
void hook_sgeqp3(Int * m, Int * n, float* a, Int * lda, Int * jpvt, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqp3((void*) m, (void*) n, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->sgeqp3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspevd (char* jobz, char* uplo, Int * n, double* ap, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_dspevd(char* jobz, char* uplo, Int * n, double* ap, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspevd((void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspevd.timings[0] += (helpTimeStop - helpTime);
    data->dspevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetrf_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chetrf_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetrf_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgsvj0 (char* jobv, Int * m, Int * n, double* a, Int * lda, double* d, double* sva, Int * mv, double* v, Int * ldv, double* eps, double* sfmin, double* tol, Int * nsweep, double* work, Int * lwork, Int * info, int len_jobv);
void hook_dgsvj0(char* jobv, Int * m, Int * n, double* a, Int * lda, double* d, double* sva, Int * mv, double* v, Int * ldv, double* eps, double* sfmin, double* tol, Int * nsweep, double* work, Int * lwork, Int * info, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgsvj0((void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) eps, (void*) sfmin, (void*) tol, (void*) nsweep, (void*) work, (void*) lwork, (void*) info, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->dgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->dgsvj0.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpbsvx (char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, char* equed, float* s, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_uplo, int len_equed);
void hook_cpbsvx(char* fact, char* uplo, Int * n, Int * kd, Int * nrhs, float complex* ab, Int * ldab, float complex* afb, Int * ldafb, char* equed, float* s, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_fact, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpbsvx((void*) fact, (void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) afb, (void*) ldafb, (void*) equed, (void*) s, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->cpbsvx.timings[0] += (helpTimeStop - helpTime);
    data->cpbsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd8 (Int * icompq, Int * k, double* d, double* z, double* vf, double* vl, double* difl, double* difr, Int * lddifr, double* dsigma, double* work, Int * info);
void hook_dlasd8(Int * icompq, Int * k, double* d, double* z, double* vf, double* vl, double* difl, double* difr, Int * lddifr, double* dsigma, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd8((void*) icompq, (void*) k, (void*) d, (void*) z, (void*) vf, (void*) vl, (void*) difl, (void*) difr, (void*) lddifr, (void*) dsigma, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd8.timings[0] += (helpTimeStop - helpTime);
    data->dlasd8.calls[0]++;

    return ;
}



extern void flexiblas_chain_slapmt (Int * forwrd, Int * m, Int * n, float* x, Int * ldx, Int * k);
void hook_slapmt(Int * forwrd, Int * m, Int * n, float* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slapmt((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->slapmt.timings[0] += (helpTimeStop - helpTime);
    data->slapmt.calls[0]++;

    return ;
}



extern void flexiblas_chain_chpev (char* jobz, char* uplo, Int * n, float complex* ap, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * info, int len_jobz, int len_uplo);
void hook_chpev(char* jobz, char* uplo, Int * n, float complex* ap, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chpev((void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chpev.timings[0] += (helpTimeStop - helpTime);
    data->chpev.calls[0]++;

    return ;
}



extern void flexiblas_chain_spptrs (char* uplo, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, Int * info, int len_uplo);
void hook_spptrs(char* uplo, Int * n, Int * nrhs, float* ap, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spptrs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spptrs.timings[0] += (helpTimeStop - helpTime);
    data->spptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chptrf (char* uplo, Int * n, float complex* ap, Int * ipiv, Int * info, int len_uplo);
void hook_chptrf(char* uplo, Int * n, float complex* ap, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chptrf((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chptrf.timings[0] += (helpTimeStop - helpTime);
    data->chptrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunm2r (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans);
void hook_cunm2r(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunm2r((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunm2r.timings[0] += (helpTimeStop - helpTime);
    data->cunm2r.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbtrd (char* vect, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* d, double* e, double* q, Int * ldq, double* work, Int * info, int len_vect, int len_uplo);
void hook_dsbtrd(char* vect, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* d, double* e, double* q, Int * ldq, double* work, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbtrd((void*) vect, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) d, (void*) e, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbtrd.timings[0] += (helpTimeStop - helpTime);
    data->dsbtrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_slarrd (char* range, char* order, Int * n, float* vl, float* vu, Int * il, Int * iu, float* gers, float* reltol, float* d, float* e, float* e2, float* pivmin, Int * nsplit, Int * isplit, Int * m, float* w, float* werr, float* wl, float* wu, Int * iblock, Int * indexw, float* work, Int * iwork, Int * info, int len_range, int len_order);
void hook_slarrd(char* range, char* order, Int * n, float* vl, float* vu, Int * il, Int * iu, float* gers, float* reltol, float* d, float* e, float* e2, float* pivmin, Int * nsplit, Int * isplit, Int * m, float* w, float* werr, float* wl, float* wu, Int * iblock, Int * indexw, float* work, Int * iwork, Int * info, int len_range, int len_order)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slarrd((void*) range, (void*) order, (void*) n, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) gers, (void*) reltol, (void*) d, (void*) e, (void*) e2, (void*) pivmin, (void*) nsplit, (void*) isplit, (void*) m, (void*) w, (void*) werr, (void*) wl, (void*) wu, (void*) iblock, (void*) indexw, (void*) work, (void*) iwork, (void*) info, (int) len_range, (int) len_order);

    helpTimeStop = flexiblas_wtime();

    data->slarrd.timings[0] += (helpTimeStop - helpTime);
    data->slarrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaqtr (Int * ltran, Int * lreal, Int * n, float* t, Int * ldt, float* b, float* w, float* scale, float* x, float* work, Int * info);
void hook_slaqtr(Int * ltran, Int * lreal, Int * n, float* t, Int * ldt, float* b, float* w, float* scale, float* x, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaqtr((void*) ltran, (void*) lreal, (void*) n, (void*) t, (void*) ldt, (void*) b, (void*) w, (void*) scale, (void*) x, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqtr.timings[0] += (helpTimeStop - helpTime);
    data->slaqtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelsd (Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info);
void hook_cgelsd(Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float* s, float* rcond, Int * rank_bn, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelsd((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelsd.timings[0] += (helpTimeStop - helpTime);
    data->cgelsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggev3 (char* jobvl, char* jobvr, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* alpha, double complex* beta, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobvl, int len_jobvr);
void hook_zggev3(char* jobvl, char* jobvr, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* alpha, double complex* beta, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggev3((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->zggev3.timings[0] += (helpTimeStop - helpTime);
    data->zggev3.calls[0]++;

    return ;
}



extern void flexiblas_chain_chegvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_chegvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chegvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chegvx.timings[0] += (helpTimeStop - helpTime);
    data->chegvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsymv (char* uplo, Int * n, double complex* alpha, double complex* a, Int * lda, double complex* x, Int * incx, double complex* beta, double complex* y, Int * incy, int len_uplo);
void hook_zsymv(char* uplo, Int * n, double complex* alpha, double complex* a, Int * lda, double complex* x, Int * incx, double complex* beta, double complex* y, Int * incy, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsymv((void*) uplo, (void*) n, (void*) alpha, (void*) a, (void*) lda, (void*) x, (void*) incx, (void*) beta, (void*) y, (void*) incy, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsymv.timings[0] += (helpTimeStop - helpTime);
    data->zsymv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasd0 (Int * n, Int * sqre, double* d, double* e, double* u, Int * ldu, double* vt, Int * ldvt, Int * smlsiz, Int * iwork, double* work, Int * info);
void hook_dlasd0(Int * n, Int * sqre, double* d, double* e, double* u, Int * ldu, double* vt, Int * ldvt, Int * smlsiz, Int * iwork, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasd0((void*) n, (void*) sqre, (void*) d, (void*) e, (void*) u, (void*) ldu, (void*) vt, (void*) ldvt, (void*) smlsiz, (void*) iwork, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd0.timings[0] += (helpTimeStop - helpTime);
    data->dlasd0.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgemqr (char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_zgemqr(char* side, char* trans, Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* t, Int * tsize, double complex* c, Int * ldc, double complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgemqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->zgemqr.timings[0] += (helpTimeStop - helpTime);
    data->zgemqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zstemr (char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, Int * m, double* w, double complex* z, Int * ldz, Int * nzc, Int * isuppz, Int * tryrac, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_zstemr(char* jobz, char* range, Int * n, double* d, double* e, double* vl, double* vu, Int * il, Int * iu, Int * m, double* w, double complex* z, Int * ldz, Int * nzc, Int * isuppz, Int * tryrac, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zstemr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) nzc, (void*) isuppz, (void*) tryrac, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->zstemr.timings[0] += (helpTimeStop - helpTime);
    data->zstemr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgetf2 (Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info);
void hook_sgetf2(Int * m, Int * n, float* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgetf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetf2.timings[0] += (helpTimeStop - helpTime);
    data->sgetf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sopgtr (char* uplo, Int * n, float* ap, float* tau, float* q, Int * ldq, float* work, Int * info, int len_uplo);
void hook_sopgtr(char* uplo, Int * n, float* ap, float* tau, float* q, Int * ldq, float* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sopgtr((void*) uplo, (void*) n, (void*) ap, (void*) tau, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->sopgtr.timings[0] += (helpTimeStop - helpTime);
    data->sopgtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dspgvx (Int * itype, char* jobz, char* range, char* uplo, Int * n, double* ap, double* bp, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_dspgvx(Int * itype, char* jobz, char* range, char* uplo, Int * n, double* ap, double* bp, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double* z, Int * ldz, double* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dspgvx((void*) itype, (void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dspgvx.timings[0] += (helpTimeStop - helpTime);
    data->dspgvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyrfs (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo);
void hook_ssyrfs(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, float* af, Int * ldaf, Int * ipiv, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyrfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssyrfs.timings[0] += (helpTimeStop - helpTime);
    data->ssyrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqr1 (Int * n, double* h, Int * ldh, double* sr1, double* si1, double* sr2, double* si2, double* v);
void hook_dlaqr1(Int * n, double* h, Int * ldh, double* sr1, double* si1, double* sr2, double* si2, double* v)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqr1((void*) n, (void*) h, (void*) ldh, (void*) sr1, (void*) si1, (void*) sr2, (void*) si2, (void*) v);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr1.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr1.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtpttr (char* uplo, Int * n, double* ap, double* a, Int * lda, Int * info, int len_uplo);
void hook_dtpttr(char* uplo, Int * n, double* ap, double* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtpttr((void*) uplo, (void*) n, (void*) ap, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dtpttr.timings[0] += (helpTimeStop - helpTime);
    data->dtpttr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungtsqr_row (Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info);
void hook_zungtsqr_row(Int * m, Int * n, Int * mb, Int * nb, double complex* a, Int * lda, double complex* t, Int * ldt, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungtsqr_row((void*) m, (void*) n, (void*) mb, (void*) nb, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->zungtsqr_row.calls[0]++;

    return ;
}



extern void flexiblas_chain_zptcon (Int * n, double* d, double complex* e, double* anorm, double* rcond, double* rwork, Int * info);
void hook_zptcon(Int * n, double* d, double complex* e, double* anorm, double* rcond, double* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zptcon((void*) n, (void*) d, (void*) e, (void*) anorm, (void*) rcond, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zptcon.timings[0] += (helpTimeStop - helpTime);
    data->zptcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_sopmtr (char* side, char* uplo, char* trans, Int * m, Int * n, float* ap, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_uplo, int len_trans);
void hook_sopmtr(char* side, char* uplo, char* trans, Int * m, Int * n, float* ap, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_uplo, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sopmtr((void*) side, (void*) uplo, (void*) trans, (void*) m, (void*) n, (void*) ap, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_uplo, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sopmtr.timings[0] += (helpTimeStop - helpTime);
    data->sopmtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelq (Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* work, Int * lwork, Int * info);
void hook_cgelq(Int * m, Int * n, float complex* a, Int * lda, float complex* t, Int * tsize, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelq((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelq.timings[0] += (helpTimeStop - helpTime);
    data->cgelq.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgges (char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * sdim, double complex* alpha, double complex* beta, double complex* vsl, Int * ldvsl, double complex* vsr, Int * ldvsr, double complex* work, Int * lwork, double* rwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort);
void hook_zgges(char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, Int * sdim, double complex* alpha, double complex* beta, double complex* vsl, Int * ldvsl, double complex* vsr, Int * ldvsr, double complex* work, Int * lwork, double* rwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgges((void*) jobvsl, (void*) jobvsr, (void*) sort, (void*) selctg, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sdim, (void*) alpha, (void*) beta, (void*) vsl, (void*) ldvsl, (void*) vsr, (void*) ldvsr, (void*) work, (void*) lwork, (void*) rwork, (void*) bwork, (void*) info, (int) len_jobvsl, (int) len_jobvsr, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->zgges.timings[0] += (helpTimeStop - helpTime);
    data->zgges.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlacpy (char* uplo, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, int len_uplo);
void hook_dlacpy(char* uplo, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlacpy((void*) uplo, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dlacpy.timings[0] += (helpTimeStop - helpTime);
    data->dlacpy.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytf2_rook (char* uplo, Int * n, float* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_ssytf2_rook(char* uplo, Int * n, float* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytf2_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytf2_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaqz4 (Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, double* sr, double* si, double* ss, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, double* qc, Int * ldqc, double* zc, Int * ldzc, double* work, Int * lwork, Int * info);
void hook_dlaqz4(Int * ilschur, Int * ilq, Int * ilz, Int * n, Int * ilo, Int * ihi, Int * nshifts, Int * nblock_desired, double* sr, double* si, double* ss, double* a, Int * lda, double* b, Int * ldb, double* q, Int * ldq, double* z, Int * ldz, double* qc, Int * ldqc, double* zc, Int * ldzc, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaqz4((void*) ilschur, (void*) ilq, (void*) ilz, (void*) n, (void*) ilo, (void*) ihi, (void*) nshifts, (void*) nblock_desired, (void*) sr, (void*) si, (void*) ss, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) qc, (void*) ldqc, (void*) zc, (void*) ldzc, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqz4.timings[0] += (helpTimeStop - helpTime);
    data->dlaqz4.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrtrs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ctrtrs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrtrs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctrtrs.timings[0] += (helpTimeStop - helpTime);
    data->ctrtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_clacp2 (char* uplo, Int * m, Int * n, float* a, Int * lda, float complex* b, Int * ldb, int len_uplo);
void hook_clacp2(char* uplo, Int * m, Int * n, float* a, Int * lda, float complex* b, Int * ldb, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clacp2((void*) uplo, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clacp2.timings[0] += (helpTimeStop - helpTime);
    data->clacp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeqlf (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info);
void hook_sgeqlf(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeqlf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->sgeqlf.calls[0]++;

    return ;
}



extern void flexiblas_chain_chesvx (char* fact, char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, Int * lwork, float* rwork, Int * info, int len_fact, int len_uplo);
void hook_chesvx(char* fact, char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* af, Int * ldaf, Int * ipiv, float complex* b, Int * ldb, float complex* x, Int * ldx, float* rcond, float* ferr, float* berr, float complex* work, Int * lwork, float* rwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chesvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chesvx.timings[0] += (helpTimeStop - helpTime);
    data->chesvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlalsd (char* uplo, Int * smlsiz, Int * n, Int * nrhs, double* d, double* e, double complex* b, Int * ldb, double* rcond, Int * rank_bn, double complex* work, double* rwork, Int * iwork, Int * info, int len_uplo);
void hook_zlalsd(char* uplo, Int * smlsiz, Int * n, Int * nrhs, double* d, double* e, double complex* b, Int * ldb, double* rcond, Int * rank_bn, double complex* work, double* rwork, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlalsd((void*) uplo, (void*) smlsiz, (void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) rcond, (void*) rank_bn, (void*) work, (void*) rwork, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlalsd.timings[0] += (helpTimeStop - helpTime);
    data->zlalsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsysv_rook (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsysv_rook(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsysv_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_slapmr (Int * forwrd, Int * m, Int * n, float* x, Int * ldx, Int * k);
void hook_slapmr(Int * forwrd, Int * m, Int * n, float* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slapmr((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->slapmr.timings[0] += (helpTimeStop - helpTime);
    data->slapmr.calls[0]++;

    return ;
}



extern float flexiblas_chain_clansy (char* norm, char* uplo, Int * n, float complex* a, Int * lda, float* work, int len_norm, int len_uplo);
float hook_clansy(char* norm, char* uplo, Int * n, float complex* a, Int * lda, float* work, int len_norm, int len_uplo)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clansy((void*) norm, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clansy.timings[0] += (helpTimeStop - helpTime);
    data->clansy.calls[0]++;

    return v;
}



extern void flexiblas_chain_ztpttf (char* transr, char* uplo, Int * n, double complex* ap, double complex* arf, Int * info, int len_transr, int len_uplo);
void hook_ztpttf(char* transr, char* uplo, Int * n, double complex* ap, double complex* arf, Int * info, int len_transr, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztpttf((void*) transr, (void*) uplo, (void*) n, (void*) ap, (void*) arf, (void*) info, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ztpttf.timings[0] += (helpTimeStop - helpTime);
    data->ztpttf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlabrd (Int * m, Int * n, Int * nb, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* x, Int * ldx, double* y, Int * ldy);
void hook_dlabrd(Int * m, Int * n, Int * nb, double* a, Int * lda, double* d, double* e, double* tauq, double* taup, double* x, Int * ldx, double* y, Int * ldy)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlabrd((void*) m, (void*) n, (void*) nb, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tauq, (void*) taup, (void*) x, (void*) ldx, (void*) y, (void*) ldy);

    helpTimeStop = flexiblas_wtime();

    data->dlabrd.timings[0] += (helpTimeStop - helpTime);
    data->dlabrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgetrf2 (Int * m, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info);
void hook_zgetrf2(Int * m, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgetrf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->zgetrf2.calls[0]++;

    return ;
}



extern int flexiblas_chain_dlaneg (Int * n, double* d, double* lld, double* sigma, double* pivmin, Int * r);
int hook_dlaneg(Int * n, double* d, double* lld, double* sigma, double* pivmin, Int * r)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dlaneg((void*) n, (void*) d, (void*) lld, (void*) sigma, (void*) pivmin, (void*) r);

    helpTimeStop = flexiblas_wtime();

    data->dlaneg.timings[0] += (helpTimeStop - helpTime);
    data->dlaneg.calls[0]++;

    return v;
}



extern void flexiblas_chain_zlaqhb (char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_zlaqhb(char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqhb((void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqhb.timings[0] += (helpTimeStop - helpTime);
    data->zlaqhb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zung2r (Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zung2r(Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zung2r((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zung2r.timings[0] += (helpTimeStop - helpTime);
    data->zung2r.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrf_rk (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytrf_rk(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrf_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctpqrt (Int * m, Int * n, Int * l, Int * nb, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, float complex* work, Int * info);
void hook_ctpqrt(Int * m, Int * n, Int * l, Int * nb, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* t, Int * ldt, float complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctpqrt((void*) m, (void*) n, (void*) l, (void*) nb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->ctpqrt.timings[0] += (helpTimeStop - helpTime);
    data->ctpqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zspsvx (char* fact, char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact, int len_uplo);
void hook_zspsvx(char* fact, char* uplo, Int * n, Int * nrhs, double complex* ap, double complex* afp, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* rcond, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_fact, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zspsvx((void*) fact, (void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) afp, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) rcond, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_fact, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zspsvx.timings[0] += (helpTimeStop - helpTime);
    data->zspsvx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgemqrt (char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans);
void hook_sgemqrt(char* side, char* trans, Int * m, Int * n, Int * k, Int * nb, float* v, Int * ldv, float* t, Int * ldt, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgemqrt((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) nb, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->sgemqrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_ddisna (char* job, Int * m, Int * n, double* d, double* sep, Int * info, int len_job);
void hook_ddisna(char* job, Int * m, Int * n, double* d, double* sep, Int * info, int len_job)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ddisna((void*) job, (void*) m, (void*) n, (void*) d, (void*) sep, (void*) info, (int) len_job);

    helpTimeStop = flexiblas_wtime();

    data->ddisna.timings[0] += (helpTimeStop - helpTime);
    data->ddisna.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbgst (char* vect, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float complex* x, Int * ldx, float complex* work, float* rwork, Int * info, int len_vect, int len_uplo);
void hook_chbgst(char* vect, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float complex* x, Int * ldx, float complex* work, float* rwork, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbgst((void*) vect, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) x, (void*) ldx, (void*) work, (void*) rwork, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbgst.timings[0] += (helpTimeStop - helpTime);
    data->chbgst.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelq (Int * m, Int * n, double* a, Int * lda, double* t, Int * tsize, double* work, Int * lwork, Int * info);
void hook_dgelq(Int * m, Int * n, double* a, Int * lda, double* t, Int * tsize, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelq((void*) m, (void*) n, (void*) a, (void*) lda, (void*) t, (void*) tsize, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelq.timings[0] += (helpTimeStop - helpTime);
    data->dgelq.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgtts2 (Int * itrans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* du2, Int * ipiv, float* b, Int * ldb);
void hook_sgtts2(Int * itrans, Int * n, Int * nrhs, float* dl, float* d, float* du, float* du2, Int * ipiv, float* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgtts2((void*) itrans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->sgtts2.timings[0] += (helpTimeStop - helpTime);
    data->sgtts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpteqr (char* compz, Int * n, double* d, double* e, double* z, Int * ldz, double* work, Int * info, int len_compz);
void hook_dpteqr(char* compz, Int * n, double* d, double* e, double* z, Int * ldz, double* work, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpteqr((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->dpteqr.timings[0] += (helpTimeStop - helpTime);
    data->dpteqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_slasq5 (Int * i0, Int * n0, float* z, Int * pp, float* tau, float* sigma, float* dmin, float* dmin1, float* dmin2, float* dn, float* dnm1, float* dnm2, Int * ieee, float* eps);
void hook_slasq5(Int * i0, Int * n0, float* z, Int * pp, float* tau, float* sigma, float* dmin, float* dmin1, float* dmin2, float* dn, float* dnm1, float* dnm2, Int * ieee, float* eps)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slasq5((void*) i0, (void*) n0, (void*) z, (void*) pp, (void*) tau, (void*) sigma, (void*) dmin, (void*) dmin1, (void*) dmin2, (void*) dn, (void*) dnm1, (void*) dnm2, (void*) ieee, (void*) eps);

    helpTimeStop = flexiblas_wtime();

    data->slasq5.timings[0] += (helpTimeStop - helpTime);
    data->slasq5.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlag2 (double* a, Int * lda, double* b, Int * ldb, double* safmin, double* scale1, double* scale2, double* wr1, double* wr2, double* wi);
void hook_dlag2(double* a, Int * lda, double* b, Int * ldb, double* safmin, double* scale1, double* scale2, double* wr1, double* wr2, double* wi)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlag2((void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) safmin, (void*) scale1, (void*) scale2, (void*) wr1, (void*) wr2, (void*) wi);

    helpTimeStop = flexiblas_wtime();

    data->dlag2.timings[0] += (helpTimeStop - helpTime);
    data->dlag2.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed4 (Int * n, Int * i, float* d, float* z, float* delta, float* rho, float* dlam, Int * info);
void hook_slaed4(Int * n, Int * i, float* d, float* z, float* delta, float* rho, float* dlam, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed4((void*) n, (void*) i, (void*) d, (void*) z, (void*) delta, (void*) rho, (void*) dlam, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed4.timings[0] += (helpTimeStop - helpTime);
    data->slaed4.calls[0]++;

    return ;
}



extern void flexiblas_chain_sptrfs (Int * n, Int * nrhs, float* d, float* e, float* df, float* ef, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * info);
void hook_sptrfs(Int * n, Int * nrhs, float* d, float* e, float* df, float* ef, float* b, Int * ldb, float* x, Int * ldx, float* ferr, float* berr, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sptrfs((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) df, (void*) ef, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sptrfs.timings[0] += (helpTimeStop - helpTime);
    data->sptrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqp2 (Int * m, Int * n, Int * offset, double complex* a, Int * lda, Int * jpvt, double complex* tau, double* vn1, double* vn2, double complex* work);
void hook_zlaqp2(Int * m, Int * n, Int * offset, double complex* a, Int * lda, Int * jpvt, double complex* tau, double* vn1, double* vn2, double complex* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqp2((void*) m, (void*) n, (void*) offset, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->zlaqp2.timings[0] += (helpTimeStop - helpTime);
    data->zlaqp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlabad (double* small, double* large);
void hook_dlabad(double* small, double* large)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlabad((void*) small, (void*) large);

    helpTimeStop = flexiblas_wtime();

    data->dlabad.timings[0] += (helpTimeStop - helpTime);
    data->dlabad.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb5 (Int * m1, Int * m2, Int * n, float complex* x1, Int * incx1, float complex* x2, Int * incx2, float complex* q1, Int * ldq1, float complex* q2, Int * ldq2, float complex* work, Int * lwork, Int * info);
void hook_cunbdb5(Int * m1, Int * m2, Int * n, float complex* x1, Int * incx1, float complex* x2, Int * incx2, float complex* q1, Int * ldq1, float complex* q2, Int * ldq2, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb5((void*) m1, (void*) m2, (void*) n, (void*) x1, (void*) incx1, (void*) x2, (void*) incx2, (void*) q1, (void*) ldq1, (void*) q2, (void*) ldq2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb5.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb5.calls[0]++;

    return ;
}



extern void flexiblas_chain_zupgtr (char* uplo, Int * n, double complex* ap, double complex* tau, double complex* q, Int * ldq, double complex* work, Int * info, int len_uplo);
void hook_zupgtr(char* uplo, Int * n, double complex* ap, double complex* tau, double complex* q, Int * ldq, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zupgtr((void*) uplo, (void*) n, (void*) ap, (void*) tau, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zupgtr.timings[0] += (helpTimeStop - helpTime);
    data->zupgtr.calls[0]++;

    return ;
}



extern int flexiblas_chain_disnan (double* din);
int hook_disnan(double* din)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_disnan((void*) din);

    helpTimeStop = flexiblas_wtime();

    data->disnan.timings[0] += (helpTimeStop - helpTime);
    data->disnan.calls[0]++;

    return v;
}



extern void flexiblas_chain_clags2 (Int * upper, float* a1, float complex* a2, float* a3, float* b1, float complex* b2, float* b3, float* csu, float complex* snu, float* csv, float complex* snv, float* csq, float complex* snq);
void hook_clags2(Int * upper, float* a1, float complex* a2, float* a3, float* b1, float complex* b2, float* b3, float* csu, float complex* snu, float* csv, float complex* snv, float* csq, float complex* snq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clags2((void*) upper, (void*) a1, (void*) a2, (void*) a3, (void*) b1, (void*) b2, (void*) b3, (void*) csu, (void*) snu, (void*) csv, (void*) snv, (void*) csq, (void*) snq);

    helpTimeStop = flexiblas_wtime();

    data->clags2.timings[0] += (helpTimeStop - helpTime);
    data->clags2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetf2 (Int * m, Int * n, double* a, Int * lda, Int * ipiv, Int * info);
void hook_dgetf2(Int * m, Int * n, double* a, Int * lda, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetf2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetf2.timings[0] += (helpTimeStop - helpTime);
    data->dgetf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgels (char* trans, Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_trans);
void hook_dgels(char* trans, Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgels((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgels.timings[0] += (helpTimeStop - helpTime);
    data->dgels.calls[0]++;

    return ;
}



extern void flexiblas_chain_chegs2 (Int * itype, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_chegs2(Int * itype, char* uplo, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chegs2((void*) itype, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chegs2.timings[0] += (helpTimeStop - helpTime);
    data->chegs2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetrs (char* trans, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_trans);
void hook_dgetrs(char* trans, Int * n, Int * nrhs, double* a, Int * lda, Int * ipiv, double* b, Int * ldb, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetrs((void*) trans, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dgetrs.timings[0] += (helpTimeStop - helpTime);
    data->dgetrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chpgvd (Int * itype, char* jobz, char* uplo, Int * n, float complex* ap, float complex* bp, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_chpgvd(Int * itype, char* jobz, char* uplo, Int * n, float complex* ap, float complex* bp, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chpgvd((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chpgvd.timings[0] += (helpTimeStop - helpTime);
    data->chpgvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrd_2stage (char* vect, char* uplo, Int * n, double* a, Int * lda, double* d, double* e, double* tau, double* hous2, Int * lhous2, double* work, Int * lwork, Int * info, int len_vect, int len_uplo);
void hook_dsytrd_2stage(char* vect, char* uplo, Int * n, double* a, Int * lda, double* d, double* e, double* tau, double* hous2, Int * lhous2, double* work, Int * lwork, Int * info, int len_vect, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrd_2stage((void*) vect, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) hous2, (void*) lhous2, (void*) work, (void*) lwork, (void*) info, (int) len_vect, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbevd (char* jobz, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_zhbevd(char* jobz, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbevd((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbevd.timings[0] += (helpTimeStop - helpTime);
    data->zhbevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytd2 (char* uplo, Int * n, float* a, Int * lda, float* d, float* e, float* tau, Int * info, int len_uplo);
void hook_ssytd2(char* uplo, Int * n, float* a, Int * lda, float* d, float* e, float* tau, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytd2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) e, (void*) tau, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytd2.timings[0] += (helpTimeStop - helpTime);
    data->ssytd2.calls[0]++;

    return ;
}



extern void flexiblas_chain_sstevx (char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range);
void hook_sstevx(char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, float* work, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sstevx((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->sstevx.timings[0] += (helpTimeStop - helpTime);
    data->sstevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarfx (char* side, Int * m, Int * n, double complex* v, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_side);
void hook_zlarfx(char* side, Int * m, Int * n, double complex* v, double complex* tau, double complex* c, Int * ldc, double complex* work, int len_side)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarfx((void*) side, (void*) m, (void*) n, (void*) v, (void*) tau, (void*) c, (void*) ldc, (void*) work, (int) len_side);

    helpTimeStop = flexiblas_wtime();

    data->zlarfx.timings[0] += (helpTimeStop - helpTime);
    data->zlarfx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsycon_3 (char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dsycon_3(char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsycon_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsycon_3.timings[0] += (helpTimeStop - helpTime);
    data->dsycon_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeql2 (Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info);
void hook_dgeql2(Int * m, Int * n, double* a, Int * lda, double* tau, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeql2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeql2.timings[0] += (helpTimeStop - helpTime);
    data->dgeql2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlae2 (double* a, double* b, double* c, double* rt1, double* rt2);
void hook_dlae2(double* a, double* b, double* c, double* rt1, double* rt2)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlae2((void*) a, (void*) b, (void*) c, (void*) rt1, (void*) rt2);

    helpTimeStop = flexiblas_wtime();

    data->dlae2.timings[0] += (helpTimeStop - helpTime);
    data->dlae2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrrfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_ctrrfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* x, Int * ldx, float* ferr, float* berr, float complex* work, float* rwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrrfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ctrrfs.timings[0] += (helpTimeStop - helpTime);
    data->ctrrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_chetf2 (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_chetf2(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chetf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chetf2.timings[0] += (helpTimeStop - helpTime);
    data->chetf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_strevc (char* side, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * info, int len_side, int len_howmny);
void hook_strevc(char* side, char* howmny, Int * select, Int * n, float* t, Int * ldt, float* vl, Int * ldvl, float* vr, Int * ldvr, Int * mm, Int * m, float* work, Int * info, int len_side, int len_howmny)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_strevc((void*) side, (void*) howmny, (void*) select, (void*) n, (void*) t, (void*) ldt, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) info, (int) len_side, (int) len_howmny);

    helpTimeStop = flexiblas_wtime();

    data->strevc.timings[0] += (helpTimeStop - helpTime);
    data->strevc.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgeesx (char* jobvs, char* sort, Int * select, char* sense, Int * n, float complex* a, Int * lda, Int * sdim, float complex* w, float complex* vs, Int * ldvs, float* rconde, float* rcondv, float complex* work, Int * lwork, float* rwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense);
void hook_cgeesx(char* jobvs, char* sort, Int * select, char* sense, Int * n, float complex* a, Int * lda, Int * sdim, float complex* w, float complex* vs, Int * ldvs, float* rconde, float* rcondv, float complex* work, Int * lwork, float* rwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgeesx((void*) jobvs, (void*) sort, (void*) select, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) w, (void*) vs, (void*) ldvs, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) rwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->cgeesx.timings[0] += (helpTimeStop - helpTime);
    data->cgeesx.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgbtrs (char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, Int * ipiv, float* b, Int * ldb, Int * info, int len_trans);
void hook_sgbtrs(char* trans, Int * n, Int * kl, Int * ku, Int * nrhs, float* ab, Int * ldab, Int * ipiv, float* b, Int * ldb, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgbtrs((void*) trans, (void*) n, (void*) kl, (void*) ku, (void*) nrhs, (void*) ab, (void*) ldab, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->sgbtrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssptrs (char* uplo, Int * n, Int * nrhs, float* ap, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo);
void hook_ssptrs(char* uplo, Int * n, Int * nrhs, float* ap, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssptrs((void*) uplo, (void*) n, (void*) nrhs, (void*) ap, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssptrs.timings[0] += (helpTimeStop - helpTime);
    data->ssptrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarfg (Int * n, double* alpha, double* x, Int * incx, double* tau);
void hook_dlarfg(Int * n, double* alpha, double* x, Int * incx, double* tau)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarfg((void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) tau);

    helpTimeStop = flexiblas_wtime();

    data->dlarfg.timings[0] += (helpTimeStop - helpTime);
    data->dlarfg.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhptrf (char* uplo, Int * n, double complex* ap, Int * ipiv, Int * info, int len_uplo);
void hook_zhptrf(char* uplo, Int * n, double complex* ap, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhptrf((void*) uplo, (void*) n, (void*) ap, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhptrf.timings[0] += (helpTimeStop - helpTime);
    data->zhptrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaeda (Int * n, Int * tlvls, Int * curlvl, Int * curpbm, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double* q, Int * qptr, double* z, double* ztemp, Int * info);
void hook_dlaeda(Int * n, Int * tlvls, Int * curlvl, Int * curpbm, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double* q, Int * qptr, double* z, double* ztemp, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaeda((void*) n, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) q, (void*) qptr, (void*) z, (void*) ztemp, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaeda.timings[0] += (helpTimeStop - helpTime);
    data->dlaeda.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasdt (Int * n, Int * lvl, Int * nd, Int * inode, Int * ndiml, Int * ndimr, Int * msub);
void hook_dlasdt(Int * n, Int * lvl, Int * nd, Int * inode, Int * ndiml, Int * ndimr, Int * msub)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasdt((void*) n, (void*) lvl, (void*) nd, (void*) inode, (void*) ndiml, (void*) ndimr, (void*) msub);

    helpTimeStop = flexiblas_wtime();

    data->dlasdt.timings[0] += (helpTimeStop - helpTime);
    data->dlasdt.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytrs_rook (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo);
void hook_ssytrs_rook(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_clapmr (Int * forwrd, Int * m, Int * n, float complex* x, Int * ldx, Int * k);
void hook_clapmr(Int * forwrd, Int * m, Int * n, float complex* x, Int * ldx, Int * k)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clapmr((void*) forwrd, (void*) m, (void*) n, (void*) x, (void*) ldx, (void*) k);

    helpTimeStop = flexiblas_wtime();

    data->clapmr.timings[0] += (helpTimeStop - helpTime);
    data->clapmr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqr1 (Int * n, double complex* h, Int * ldh, double complex* s1, double complex* s2, double complex* v);
void hook_zlaqr1(Int * n, double complex* h, Int * ldh, double complex* s1, double complex* s2, double complex* v)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqr1((void*) n, (void*) h, (void*) ldh, (void*) s1, (void*) s2, (void*) v);

    helpTimeStop = flexiblas_wtime();

    data->zlaqr1.timings[0] += (helpTimeStop - helpTime);
    data->zlaqr1.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsyconvf (char* uplo, char* way, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, Int * info, int len_uplo, int len_way);
void hook_zsyconvf(char* uplo, char* way, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsyconvf((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->zsyconvf.timings[0] += (helpTimeStop - helpTime);
    data->zsyconvf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorgql (Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dorgql(Int * m, Int * n, Int * k, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorgql((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgql.timings[0] += (helpTimeStop - helpTime);
    data->dorgql.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed7 (Int * icompq, Int * n, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double* work, Int * iwork, Int * info);
void hook_dlaed7(Int * icompq, Int * n, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, double* d, double* q, Int * ldq, Int * indxq, double* rho, Int * cutpnt, double* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, double* givnum, double* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed7((void*) icompq, (void*) n, (void*) qsiz, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) qstore, (void*) qptr, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed7.timings[0] += (helpTimeStop - helpTime);
    data->dlaed7.calls[0]++;

    return ;
}



extern void flexiblas_chain_clahef_rk (char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo);
void hook_clahef_rk(char* uplo, Int * n, Int * nb, Int * kb, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* w, Int * ldw, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clahef_rk((void*) uplo, (void*) n, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) w, (void*) ldw, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->clahef_rk.timings[0] += (helpTimeStop - helpTime);
    data->clahef_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_sormbr (char* vect, char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_vect, int len_side, int len_trans);
void hook_sormbr(char* vect, char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_vect, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sormbr((void*) vect, (void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_vect, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sormbr.timings[0] += (helpTimeStop - helpTime);
    data->sormbr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zheequb (char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, double complex* work, Int * info, int len_uplo);
void hook_zheequb(char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheequb((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheequb.timings[0] += (helpTimeStop - helpTime);
    data->zheequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlacrm (Int * m, Int * n, double complex* a, Int * lda, double* b, Int * ldb, double complex* c, Int * ldc, double* rwork);
void hook_zlacrm(Int * m, Int * n, double complex* a, Int * lda, double* b, Int * ldb, double complex* c, Int * ldc, double* rwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlacrm((void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) rwork);

    helpTimeStop = flexiblas_wtime();

    data->zlacrm.timings[0] += (helpTimeStop - helpTime);
    data->zlacrm.calls[0]++;

    return ;
}



extern void flexiblas_chain_sormqr (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_sormqr(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sormqr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sormqr.timings[0] += (helpTimeStop - helpTime);
    data->sormqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztpttr (char* uplo, Int * n, double complex* ap, double complex* a, Int * lda, Int * info, int len_uplo);
void hook_ztpttr(char* uplo, Int * n, double complex* ap, double complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztpttr((void*) uplo, (void*) n, (void*) ap, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ztpttr.timings[0] += (helpTimeStop - helpTime);
    data->ztpttr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlals0 (Int * icompq, Int * nl, Int * nr, Int * sqre, Int * nrhs, double complex* b, Int * ldb, double complex* bx, Int * ldbx, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* poles, double* difl, double* difr, double* z, Int * k, double* c, double* s, double* rwork, Int * info);
void hook_zlals0(Int * icompq, Int * nl, Int * nr, Int * sqre, Int * nrhs, double complex* b, Int * ldb, double complex* bx, Int * ldbx, Int * perm, Int * givptr, Int * givcol, Int * ldgcol, double* givnum, Int * ldgnum, double* poles, double* difl, double* difr, double* z, Int * k, double* c, double* s, double* rwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlals0((void*) icompq, (void*) nl, (void*) nr, (void*) sqre, (void*) nrhs, (void*) b, (void*) ldb, (void*) bx, (void*) ldbx, (void*) perm, (void*) givptr, (void*) givcol, (void*) ldgcol, (void*) givnum, (void*) ldgnum, (void*) poles, (void*) difl, (void*) difr, (void*) z, (void*) k, (void*) c, (void*) s, (void*) rwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlals0.timings[0] += (helpTimeStop - helpTime);
    data->zlals0.calls[0]++;

    return ;
}



extern void flexiblas_chain_spotf2 (char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo);
void hook_spotf2(char* uplo, Int * n, float* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spotf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->spotf2.timings[0] += (helpTimeStop - helpTime);
    data->spotf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpbsv (char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* b, Int * ldb, Int * info, int len_uplo);
void hook_zpbsv(char* uplo, Int * n, Int * kd, Int * nrhs, double complex* ab, Int * ldab, double complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpbsv((void*) uplo, (void*) n, (void*) kd, (void*) nrhs, (void*) ab, (void*) ldab, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpbsv.timings[0] += (helpTimeStop - helpTime);
    data->zpbsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zheevd (char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double* w, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_zheevd(char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double* w, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zheevd((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zheevd.timings[0] += (helpTimeStop - helpTime);
    data->zheevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaln2 (Int * ltrans, Int * na, Int * nw, double* smin, double* ca, double* a, Int * lda, double* d1, double* d2, double* b, Int * ldb, double* wr, double* wi, double* x, Int * ldx, double* scale, double* xnorm, Int * info);
void hook_dlaln2(Int * ltrans, Int * na, Int * nw, double* smin, double* ca, double* a, Int * lda, double* d1, double* d2, double* b, Int * ldb, double* wr, double* wi, double* x, Int * ldx, double* scale, double* xnorm, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaln2((void*) ltrans, (void*) na, (void*) nw, (void*) smin, (void*) ca, (void*) a, (void*) lda, (void*) d1, (void*) d2, (void*) b, (void*) ldb, (void*) wr, (void*) wi, (void*) x, (void*) ldx, (void*) scale, (void*) xnorm, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaln2.timings[0] += (helpTimeStop - helpTime);
    data->dlaln2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zherfs (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo);
void hook_zherfs(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* af, Int * ldaf, Int * ipiv, double complex* b, Int * ldb, double complex* x, Int * ldx, double* ferr, double* berr, double complex* work, double* rwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zherfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) rwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zherfs.timings[0] += (helpTimeStop - helpTime);
    data->zherfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlaed9 (Int * k, Int * kstart, Int * kstop, Int * n, double* d, double* q, Int * ldq, double* rho, double* dlambda, double* w, double* s, Int * lds, Int * info);
void hook_dlaed9(Int * k, Int * kstart, Int * kstop, Int * n, double* d, double* q, Int * ldq, double* rho, double* dlambda, double* w, double* s, Int * lds, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaed9((void*) k, (void*) kstart, (void*) kstop, (void*) n, (void*) d, (void*) q, (void*) ldq, (void*) rho, (void*) dlambda, (void*) w, (void*) s, (void*) lds, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed9.timings[0] += (helpTimeStop - helpTime);
    data->dlaed9.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheevx (char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_cheevx(char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheevx((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheevx.timings[0] += (helpTimeStop - helpTime);
    data->cheevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri_3x (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo);
void hook_csytri_3x(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri_3x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri_3x.timings[0] += (helpTimeStop - helpTime);
    data->csytri_3x.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpteqr (char* compz, Int * n, double* d, double* e, double complex* z, Int * ldz, double* work, Int * info, int len_compz);
void hook_zpteqr(char* compz, Int * n, double* d, double* e, double complex* z, Int * ldz, double* work, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpteqr((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->zpteqr.timings[0] += (helpTimeStop - helpTime);
    data->zpteqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgges (char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, float* a, Int * lda, float* b, Int * ldb, Int * sdim, float* alphar, float* alphai, float* beta, float* vsl, Int * ldvsl, float* vsr, Int * ldvsr, float* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort);
void hook_sgges(char* jobvsl, char* jobvsr, char* sort, Int * selctg, Int * n, float* a, Int * lda, float* b, Int * ldb, Int * sdim, float* alphar, float* alphai, float* beta, float* vsl, Int * ldvsl, float* vsr, Int * ldvsr, float* work, Int * lwork, Int * bwork, Int * info, int len_jobvsl, int len_jobvsr, int len_sort)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgges((void*) jobvsl, (void*) jobvsr, (void*) sort, (void*) selctg, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) sdim, (void*) alphar, (void*) alphai, (void*) beta, (void*) vsl, (void*) ldvsl, (void*) vsr, (void*) ldvsr, (void*) work, (void*) lwork, (void*) bwork, (void*) info, (int) len_jobvsl, (int) len_jobvsr, (int) len_sort);

    helpTimeStop = flexiblas_wtime();

    data->sgges.timings[0] += (helpTimeStop - helpTime);
    data->sgges.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorbdb1 (Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* work, Int * lwork, Int * info);
void hook_dorbdb1(Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x21, Int * ldx21, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorbdb1((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb1.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb1.calls[0]++;

    return ;
}



extern void flexiblas_chain_claswp (Int * n, float complex* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx);
void hook_claswp(Int * n, float complex* a, Int * lda, Int * k1, Int * k2, Int * ipiv, Int * incx)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claswp((void*) n, (void*) a, (void*) lda, (void*) k1, (void*) k2, (void*) ipiv, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->claswp.timings[0] += (helpTimeStop - helpTime);
    data->claswp.calls[0]++;

    return ;
}



extern void flexiblas_chain_clacn2 (Int * n, float complex* v, float complex* x, float* est, Int * kase, Int * isave);
void hook_clacn2(Int * n, float complex* v, float complex* x, float* est, Int * kase, Int * isave)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clacn2((void*) n, (void*) v, (void*) x, (void*) est, (void*) kase, (void*) isave);

    helpTimeStop = flexiblas_wtime();

    data->clacn2.timings[0] += (helpTimeStop - helpTime);
    data->clacn2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssysv (char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssysv(char* uplo, Int * n, Int * nrhs, float* a, Int * lda, Int * ipiv, float* b, Int * ldb, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssysv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssysv.timings[0] += (helpTimeStop - helpTime);
    data->ssysv.calls[0]++;

    return ;
}



extern double flexiblas_chain_zlanhf (char* norm, char* transr, char* uplo, Int * n, double complex* a, double* work, int len_norm, int len_transr, int len_uplo);
double hook_zlanhf(char* norm, char* transr, char* uplo, Int * n, double complex* a, double* work, int len_norm, int len_transr, int len_uplo)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_zlanhf((void*) norm, (void*) transr, (void*) uplo, (void*) n, (void*) a, (void*) work, (int) len_norm, (int) len_transr, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zlanhf.timings[0] += (helpTimeStop - helpTime);
    data->zlanhf.calls[0]++;

    return v;
}



extern void flexiblas_chain_dlaorhr_col_getrfnp2 (Int * m, Int * n, double* a, Int * lda, double* d, Int * info);
void hook_dlaorhr_col_getrfnp2(Int * m, Int * n, double* a, Int * lda, double* d, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlaorhr_col_getrfnp2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaorhr_col_getrfnp2.timings[0] += (helpTimeStop - helpTime);
    data->dlaorhr_col_getrfnp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarrf (Int * n, double* d, double* l, double* ld, Int * clstrt, Int * clend, double* w, double* wgap, double* werr, double* spdiam, double* clgapl, double* clgapr, double* pivmin, double* sigma, double* dplus, double* lplus, double* work, Int * info);
void hook_dlarrf(Int * n, double* d, double* l, double* ld, Int * clstrt, Int * clend, double* w, double* wgap, double* werr, double* spdiam, double* clgapl, double* clgapr, double* pivmin, double* sigma, double* dplus, double* lplus, double* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarrf((void*) n, (void*) d, (void*) l, (void*) ld, (void*) clstrt, (void*) clend, (void*) w, (void*) wgap, (void*) werr, (void*) spdiam, (void*) clgapl, (void*) clgapr, (void*) pivmin, (void*) sigma, (void*) dplus, (void*) lplus, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrf.timings[0] += (helpTimeStop - helpTime);
    data->dlarrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgeev (char* jobvl, char* jobvr, Int * n, double complex* a, Int * lda, double complex* w, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobvl, int len_jobvr);
void hook_zgeev(char* jobvl, char* jobvr, Int * n, double complex* a, Int * lda, double complex* w, double complex* vl, Int * ldvl, double complex* vr, Int * ldvr, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgeev((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->zgeev.timings[0] += (helpTimeStop - helpTime);
    data->zgeev.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeesx (char* jobvs, char* sort, Int * select, char* sense, Int * n, double* a, Int * lda, Int * sdim, double* wr, double* wi, double* vs, Int * ldvs, double* rconde, double* rcondv, double* work, Int * lwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense);
void hook_dgeesx(char* jobvs, char* sort, Int * select, char* sense, Int * n, double* a, Int * lda, Int * sdim, double* wr, double* wi, double* vs, Int * ldvs, double* rconde, double* rcondv, double* work, Int * lwork, Int * iwork, Int * liwork, Int * bwork, Int * info, int len_jobvs, int len_sort, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeesx((void*) jobvs, (void*) sort, (void*) select, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) sdim, (void*) wr, (void*) wi, (void*) vs, (void*) ldvs, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) bwork, (void*) info, (int) len_jobvs, (int) len_sort, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->dgeesx.timings[0] += (helpTimeStop - helpTime);
    data->dgeesx.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarfb_gett (char* ident, Int * m, Int * n, Int * k, double* t, Int * ldt, double* a, Int * lda, double* b, Int * ldb, double* work, Int * ldwork, int len_ident);
void hook_dlarfb_gett(char* ident, Int * m, Int * n, Int * k, double* t, Int * ldt, double* a, Int * lda, double* b, Int * ldb, double* work, Int * ldwork, int len_ident)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarfb_gett((void*) ident, (void*) m, (void*) n, (void*) k, (void*) t, (void*) ldt, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) ldwork, (int) len_ident);

    helpTimeStop = flexiblas_wtime();

    data->dlarfb_gett.timings[0] += (helpTimeStop - helpTime);
    data->dlarfb_gett.calls[0]++;

    return ;
}



extern void flexiblas_chain_dptsv (Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, Int * info);
void hook_dptsv(Int * n, Int * nrhs, double* d, double* e, double* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dptsv((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dptsv.timings[0] += (helpTimeStop - helpTime);
    data->dptsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssytri_3 (char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo);
void hook_ssytri_3(char* uplo, Int * n, float* a, Int * lda, float* e, Int * ipiv, float* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssytri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_3.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpotf2 (char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo);
void hook_cpotf2(char* uplo, Int * n, float complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpotf2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cpotf2.timings[0] += (helpTimeStop - helpTime);
    data->cpotf2.calls[0]++;

    return ;
}



extern void flexiblas_chain_zcposv (char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* x, Int * ldx, double complex* work, float complex* swork, double* rwork, Int * iter, Int * info, int len_uplo);
void hook_zcposv(char* uplo, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* x, Int * ldx, double complex* work, float complex* swork, double* rwork, Int * iter, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zcposv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) work, (void*) swork, (void*) rwork, (void*) iter, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zcposv.timings[0] += (helpTimeStop - helpTime);
    data->zcposv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpgv (Int * itype, char* jobz, char* uplo, Int * n, double complex* ap, double complex* bp, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zhpgv(Int * itype, char* jobz, char* uplo, Int * n, double complex* ap, double complex* bp, double* w, double complex* z, Int * ldz, double complex* work, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpgv((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpgv.timings[0] += (helpTimeStop - helpTime);
    data->zhpgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgttrf (Int * n, double* dl, double* d, double* du, double* du2, Int * ipiv, Int * info);
void hook_dgttrf(Int * n, double* dl, double* d, double* du, double* du2, Int * ipiv, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgttrf((void*) n, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgttrf.timings[0] += (helpTimeStop - helpTime);
    data->dgttrf.calls[0]++;

    return ;
}



extern void flexiblas_chain_dorbdb (char* trans, char* signs, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x12, Int * ldx12, double* x21, Int * ldx21, double* x22, Int * ldx22, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* tauq2, double* work, Int * lwork, Int * info, int len_trans, int len_signs);
void hook_dorbdb(char* trans, char* signs, Int * m, Int * p, Int * q, double* x11, Int * ldx11, double* x12, Int * ldx12, double* x21, Int * ldx21, double* x22, Int * ldx22, double* theta, double* phi, double* taup1, double* taup2, double* tauq1, double* tauq2, double* work, Int * lwork, Int * info, int len_trans, int len_signs)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dorbdb((void*) trans, (void*) signs, (void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x12, (void*) ldx12, (void*) x21, (void*) ldx21, (void*) x22, (void*) ldx22, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) tauq2, (void*) work, (void*) lwork, (void*) info, (int) len_trans, (int) len_signs);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaed7 (Int * icompq, Int * n, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float* work, Int * iwork, Int * info);
void hook_slaed7(Int * icompq, Int * n, Int * qsiz, Int * tlvls, Int * curlvl, Int * curpbm, float* d, float* q, Int * ldq, Int * indxq, float* rho, Int * cutpnt, float* qstore, Int * qptr, Int * prmptr, Int * perm, Int * givptr, Int * givcol, float* givnum, float* work, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaed7((void*) icompq, (void*) n, (void*) qsiz, (void*) tlvls, (void*) curlvl, (void*) curpbm, (void*) d, (void*) q, (void*) ldq, (void*) indxq, (void*) rho, (void*) cutpnt, (void*) qstore, (void*) qptr, (void*) prmptr, (void*) perm, (void*) givptr, (void*) givcol, (void*) givnum, (void*) work, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed7.timings[0] += (helpTimeStop - helpTime);
    data->slaed7.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsytri_rook (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * info, int len_uplo);
void hook_zsytri_rook(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsytri_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->zsytri_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlanv2 (double* a, double* b, double* c, double* d, double* rt1r, double* rt1i, double* rt2r, double* rt2i, double* cs, double* sn);
void hook_dlanv2(double* a, double* b, double* c, double* d, double* rt1r, double* rt1i, double* rt2r, double* rt2i, double* cs, double* sn)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlanv2((void*) a, (void*) b, (void*) c, (void*) d, (void*) rt1r, (void*) rt1i, (void*) rt2r, (void*) rt2i, (void*) cs, (void*) sn);

    helpTimeStop = flexiblas_wtime();

    data->dlanv2.timings[0] += (helpTimeStop - helpTime);
    data->dlanv2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtgsyl (char* trans, Int * ijob, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* d, Int * ldd, double* e, Int * lde, double* f, Int * ldf, double* scale, double* dif, double* work, Int * lwork, Int * iwork, Int * info, int len_trans);
void hook_dtgsyl(char* trans, Int * ijob, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* d, Int * ldd, double* e, Int * lde, double* f, Int * ldf, double* scale, double* dif, double* work, Int * lwork, Int * iwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtgsyl((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) dif, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dtgsyl.timings[0] += (helpTimeStop - helpTime);
    data->dtgsyl.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlasrt (char* id, Int * n, double* d, Int * info, int len_id);
void hook_dlasrt(char* id, Int * n, double* d, Int * info, int len_id)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlasrt((void*) id, (void*) n, (void*) d, (void*) info, (int) len_id);

    helpTimeStop = flexiblas_wtime();

    data->dlasrt.timings[0] += (helpTimeStop - helpTime);
    data->dlasrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqr4 (Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info);
void hook_zlaqr4(Int * wantt, Int * wantz, Int * n, Int * ilo, Int * ihi, double complex* h, Int * ldh, double complex* w, Int * iloz, Int * ihiz, double complex* z, Int * ldz, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqr4((void*) wantt, (void*) wantz, (void*) n, (void*) ilo, (void*) ihi, (void*) h, (void*) ldh, (void*) w, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqr4.timings[0] += (helpTimeStop - helpTime);
    data->zlaqr4.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytri_3 (char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_csytri_3(char* uplo, Int * n, float complex* a, Int * lda, float complex* e, Int * ipiv, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytri_3((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytri_3.timings[0] += (helpTimeStop - helpTime);
    data->csytri_3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgejsv (char* joba, char* jobu, char* jobv, char* jobr, char* jobt, char* jobp, Int * m, Int * n, double* a, Int * lda, double* sva, double* u, Int * ldu, double* v, Int * ldv, double* work, Int * lwork, Int * iwork, Int * info, int len_joba, int len_jobu, int len_jobv, int len_jobr, int len_jobt, int len_jobp);
void hook_dgejsv(char* joba, char* jobu, char* jobv, char* jobr, char* jobt, char* jobp, Int * m, Int * n, double* a, Int * lda, double* sva, double* u, Int * ldu, double* v, Int * ldv, double* work, Int * lwork, Int * iwork, Int * info, int len_joba, int len_jobu, int len_jobv, int len_jobr, int len_jobt, int len_jobp)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgejsv((void*) joba, (void*) jobu, (void*) jobv, (void*) jobr, (void*) jobt, (void*) jobp, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) sva, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_joba, (int) len_jobu, (int) len_jobv, (int) len_jobr, (int) len_jobt, (int) len_jobp);

    helpTimeStop = flexiblas_wtime();

    data->dgejsv.timings[0] += (helpTimeStop - helpTime);
    data->dgejsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dopgtr (char* uplo, Int * n, double* ap, double* tau, double* q, Int * ldq, double* work, Int * info, int len_uplo);
void hook_dopgtr(char* uplo, Int * n, double* ap, double* tau, double* q, Int * ldq, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dopgtr((void*) uplo, (void*) n, (void*) ap, (void*) tau, (void*) q, (void*) ldq, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dopgtr.timings[0] += (helpTimeStop - helpTime);
    data->dopgtr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetri2 (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo);
void hook_zhetri2(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetri2((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetri2.timings[0] += (helpTimeStop - helpTime);
    data->zhetri2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ztbcon (char* norm, char* uplo, char* diag, Int * n, Int * kd, double complex* ab, Int * ldab, double* rcond, double complex* work, double* rwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_ztbcon(char* norm, char* uplo, char* diag, Int * n, Int * kd, double complex* ab, Int * ldab, double* rcond, double complex* work, double* rwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ztbcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) rcond, (void*) work, (void*) rwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->ztbcon.timings[0] += (helpTimeStop - helpTime);
    data->ztbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtgsy2 (char* trans, Int * ijob, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* d, Int * ldd, double* e, Int * lde, double* f, Int * ldf, double* scale, double* rdsum, double* rdscal, Int * iwork, Int * pq, Int * info, int len_trans);
void hook_dtgsy2(char* trans, Int * ijob, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* d, Int * ldd, double* e, Int * lde, double* f, Int * ldf, double* scale, double* rdsum, double* rdscal, Int * iwork, Int * pq, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtgsy2((void*) trans, (void*) ijob, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) d, (void*) ldd, (void*) e, (void*) lde, (void*) f, (void*) ldf, (void*) scale, (void*) rdsum, (void*) rdscal, (void*) iwork, (void*) pq, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dtgsy2.timings[0] += (helpTimeStop - helpTime);
    data->dtgsy2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dormhr (char* side, char* trans, Int * m, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_dormhr(char* side, char* trans, Int * m, Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* c, Int * ldc, double* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dormhr((void*) side, (void*) trans, (void*) m, (void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dormhr.timings[0] += (helpTimeStop - helpTime);
    data->dormhr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtprfs (char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* ap, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag);
void hook_dtprfs(char* uplo, char* trans, char* diag, Int * n, Int * nrhs, double* ap, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo, int len_trans, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtprfs((void*) uplo, (void*) trans, (void*) diag, (void*) n, (void*) nrhs, (void*) ap, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->dtprfs.timings[0] += (helpTimeStop - helpTime);
    data->dtprfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_clatrz (Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* tau, float complex* work);
void hook_clatrz(Int * m, Int * n, Int * l, float complex* a, Int * lda, float complex* tau, float complex* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_clatrz((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->clatrz.timings[0] += (helpTimeStop - helpTime);
    data->clatrz.calls[0]++;

    return ;
}



extern void flexiblas_chain_cggev3 (char* jobvl, char* jobvr, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobvl, int len_jobvr);
void hook_cggev3(char* jobvl, char* jobvr, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* alpha, float complex* beta, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, float complex* work, Int * lwork, float* rwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cggev3((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->cggev3.timings[0] += (helpTimeStop - helpTime);
    data->cggev3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsycon_rook (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dsycon_rook(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* anorm, double* rcond, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsycon_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsycon_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsycon_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgtts2 (Int * itrans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* du2, Int * ipiv, double* b, Int * ldb);
void hook_dgtts2(Int * itrans, Int * n, Int * nrhs, double* dl, double* d, double* du, double* du2, Int * ipiv, double* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgtts2((void*) itrans, (void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) du2, (void*) ipiv, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->dgtts2.timings[0] += (helpTimeStop - helpTime);
    data->dgtts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlatrz (Int * m, Int * n, Int * l, double* a, Int * lda, double* tau, double* work);
void hook_dlatrz(Int * m, Int * n, Int * l, double* a, Int * lda, double* tau, double* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatrz((void*) m, (void*) n, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->dlatrz.timings[0] += (helpTimeStop - helpTime);
    data->dlatrz.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyrfs (char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo);
void hook_dsyrfs(char* uplo, Int * n, Int * nrhs, double* a, Int * lda, double* af, Int * ldaf, Int * ipiv, double* b, Int * ldb, double* x, Int * ldx, double* ferr, double* berr, double* work, Int * iwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyrfs((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) af, (void*) ldaf, (void*) ipiv, (void*) b, (void*) ldb, (void*) x, (void*) ldx, (void*) ferr, (void*) berr, (void*) work, (void*) iwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyrfs.timings[0] += (helpTimeStop - helpTime);
    data->dsyrfs.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeql2 (Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info);
void hook_sgeql2(Int * m, Int * n, float* a, Int * lda, float* tau, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeql2((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeql2.timings[0] += (helpTimeStop - helpTime);
    data->sgeql2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlartgs (double* x, double* y, double* sigma, double* cs, double* sn);
void hook_dlartgs(double* x, double* y, double* sigma, double* cs, double* sn)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlartgs((void*) x, (void*) y, (void*) sigma, (void*) cs, (void*) sn);

    helpTimeStop = flexiblas_wtime();

    data->dlartgs.timings[0] += (helpTimeStop - helpTime);
    data->dlartgs.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgetsqrhrt (Int * m, Int * n, Int * mb1, Int * nb1, Int * nb2, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info);
void hook_cgetsqrhrt(Int * m, Int * n, Int * mb1, Int * nb1, Int * nb2, float complex* a, Int * lda, float complex* t, Int * ldt, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgetsqrhrt((void*) m, (void*) n, (void*) mb1, (void*) nb1, (void*) nb2, (void*) a, (void*) lda, (void*) t, (void*) ldt, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetsqrhrt.timings[0] += (helpTimeStop - helpTime);
    data->cgetsqrhrt.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgeevx (char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, double* a, Int * lda, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, Int * ilo, Int * ihi, double* scale, double* abnrm, double* rconde, double* rcondv, double* work, Int * lwork, Int * iwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense);
void hook_dgeevx(char* balanc, char* jobvl, char* jobvr, char* sense, Int * n, double* a, Int * lda, double* wr, double* wi, double* vl, Int * ldvl, double* vr, Int * ldvr, Int * ilo, Int * ihi, double* scale, double* abnrm, double* rconde, double* rcondv, double* work, Int * lwork, Int * iwork, Int * info, int len_balanc, int len_jobvl, int len_jobvr, int len_sense)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgeevx((void*) balanc, (void*) jobvl, (void*) jobvr, (void*) sense, (void*) n, (void*) a, (void*) lda, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) ilo, (void*) ihi, (void*) scale, (void*) abnrm, (void*) rconde, (void*) rcondv, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_balanc, (int) len_jobvl, (int) len_jobvr, (int) len_sense);

    helpTimeStop = flexiblas_wtime();

    data->dgeevx.timings[0] += (helpTimeStop - helpTime);
    data->dgeevx.calls[0]++;

    return ;
}



extern void flexiblas_chain_cposv (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_cposv(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cposv((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cposv.timings[0] += (helpTimeStop - helpTime);
    data->cposv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgbequ (Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info);
void hook_zgbequ(Int * m, Int * n, Int * kl, Int * ku, double complex* ab, Int * ldab, double* r, double* c, double* rowcnd, double* colcnd, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgbequ((void*) m, (void*) n, (void*) kl, (void*) ku, (void*) ab, (void*) ldab, (void*) r, (void*) c, (void*) rowcnd, (void*) colcnd, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbequ.timings[0] += (helpTimeStop - helpTime);
    data->zgbequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqps (Int * m, Int * n, Int * offset, Int * nb, Int * kb, double complex* a, Int * lda, Int * jpvt, double complex* tau, double* vn1, double* vn2, double complex* auxv, double complex* f, Int * ldf);
void hook_zlaqps(Int * m, Int * n, Int * offset, Int * nb, Int * kb, double complex* a, Int * lda, Int * jpvt, double complex* tau, double* vn1, double* vn2, double complex* auxv, double complex* f, Int * ldf)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqps((void*) m, (void*) n, (void*) offset, (void*) nb, (void*) kb, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf);

    helpTimeStop = flexiblas_wtime();

    data->zlaqps.timings[0] += (helpTimeStop - helpTime);
    data->zlaqps.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbgvd (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_dsbgvd(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbgvd((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbgvd.timings[0] += (helpTimeStop - helpTime);
    data->dsbgvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelqf (Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cgelqf(Int * m, Int * n, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelqf((void*) m, (void*) n, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelqf.timings[0] += (helpTimeStop - helpTime);
    data->cgelqf.calls[0]++;

    return ;
}



extern void flexiblas_chain_claqp2 (Int * m, Int * n, Int * offset, float complex* a, Int * lda, Int * jpvt, float complex* tau, float* vn1, float* vn2, float complex* work);
void hook_claqp2(Int * m, Int * n, Int * offset, float complex* a, Int * lda, Int * jpvt, float complex* tau, float* vn1, float* vn2, float complex* work)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_claqp2((void*) m, (void*) n, (void*) offset, (void*) a, (void*) lda, (void*) jpvt, (void*) tau, (void*) vn1, (void*) vn2, (void*) work);

    helpTimeStop = flexiblas_wtime();

    data->claqp2.timings[0] += (helpTimeStop - helpTime);
    data->claqp2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dladiv (double* a, double* b, double* c, double* d, double* p, double* q);
void hook_dladiv(double* a, double* b, double* c, double* d, double* p, double* q)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dladiv((void*) a, (void*) b, (void*) c, (void*) d, (void*) p, (void*) q);

    helpTimeStop = flexiblas_wtime();

    data->dladiv.timings[0] += (helpTimeStop - helpTime);
    data->dladiv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dladiv1 (double* a, double* b, double* c, double* d, double* p, double* q);
void hook_dladiv1(double* a, double* b, double* c, double* d, double* p, double* q)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dladiv1((void*) a, (void*) b, (void*) c, (void*) d, (void*) p, (void*) q);

    helpTimeStop = flexiblas_wtime();

    data->dladiv1.timings[0] += (helpTimeStop - helpTime);
    data->dladiv1.calls[0]++;

    return ;
}



extern double flexiblas_chain_dladiv2 (double* a, double* b, double* c, double* d, double* r, double* t);
double hook_dladiv2(double* a, double* b, double* c, double* d, double* r, double* t)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dladiv2((void*) a, (void*) b, (void*) c, (void*) d, (void*) r, (void*) t);

    helpTimeStop = flexiblas_wtime();

    data->dladiv2.timings[0] += (helpTimeStop - helpTime);
    data->dladiv2.calls[0]++;

    return v;
}



extern void flexiblas_chain_zhetri2x (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * nb, Int * info, int len_uplo);
void hook_zhetri2x(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, double complex* work, Int * nb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetri2x((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) nb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetri2x.timings[0] += (helpTimeStop - helpTime);
    data->zhetri2x.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlatdf (Int * ijob, Int * n, double* z, Int * ldz, double* rhs, double* rdsum, double* rdscal, Int * ipiv, Int * jpiv);
void hook_dlatdf(Int * ijob, Int * n, double* z, Int * ldz, double* rhs, double* rdsum, double* rdscal, Int * ipiv, Int * jpiv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatdf((void*) ijob, (void*) n, (void*) z, (void*) ldz, (void*) rhs, (void*) rdsum, (void*) rdscal, (void*) ipiv, (void*) jpiv);

    helpTimeStop = flexiblas_wtime();

    data->dlatdf.timings[0] += (helpTimeStop - helpTime);
    data->dlatdf.calls[0]++;

    return ;
}



extern void flexiblas_chain_chsein (char* side, char* eigsrc, char* initv, Int * select, Int * n, float complex* h, Int * ldh, float complex* w, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, float* rwork, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv);
void hook_chsein(char* side, char* eigsrc, char* initv, Int * select, Int * n, float complex* h, Int * ldh, float complex* w, float complex* vl, Int * ldvl, float complex* vr, Int * ldvr, Int * mm, Int * m, float complex* work, float* rwork, Int * ifaill, Int * ifailr, Int * info, int len_side, int len_eigsrc, int len_initv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chsein((void*) side, (void*) eigsrc, (void*) initv, (void*) select, (void*) n, (void*) h, (void*) ldh, (void*) w, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) mm, (void*) m, (void*) work, (void*) rwork, (void*) ifaill, (void*) ifailr, (void*) info, (int) len_side, (int) len_eigsrc, (int) len_initv);

    helpTimeStop = flexiblas_wtime();

    data->chsein.timings[0] += (helpTimeStop - helpTime);
    data->chsein.calls[0]++;

    return ;
}



extern void flexiblas_chain_chesv_aa (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo);
void hook_chesv_aa(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chesv_aa((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chesv_aa.timings[0] += (helpTimeStop - helpTime);
    data->chesv_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhpgvd (Int * itype, char* jobz, char* uplo, Int * n, double complex* ap, double complex* bp, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_zhpgvd(Int * itype, char* jobz, char* uplo, Int * n, double complex* ap, double complex* bp, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhpgvd((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) ap, (void*) bp, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhpgvd.timings[0] += (helpTimeStop - helpTime);
    data->zhpgvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_sstevr (char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range);
void hook_sstevr(char* jobz, char* range, Int * n, float* d, float* e, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float* z, Int * ldz, Int * isuppz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sstevr((void*) jobz, (void*) range, (void*) n, (void*) d, (void*) e, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range);

    helpTimeStop = flexiblas_wtime();

    data->sstevr.timings[0] += (helpTimeStop - helpTime);
    data->sstevr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqr3 (Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double complex* h, Int * ldh, Int * iloz, Int * ihiz, double complex* z, Int * ldz, Int * ns, Int * nd, double complex* sh, double complex* v, Int * ldv, Int * nh, double complex* t, Int * ldt, Int * nv, double complex* wv, Int * ldwv, double complex* work, Int * lwork);
void hook_zlaqr3(Int * wantt, Int * wantz, Int * n, Int * ktop, Int * kbot, Int * nw, double complex* h, Int * ldh, Int * iloz, Int * ihiz, double complex* z, Int * ldz, Int * ns, Int * nd, double complex* sh, double complex* v, Int * ldv, Int * nh, double complex* t, Int * ldt, Int * nv, double complex* wv, Int * ldwv, double complex* work, Int * lwork)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqr3((void*) wantt, (void*) wantz, (void*) n, (void*) ktop, (void*) kbot, (void*) nw, (void*) h, (void*) ldh, (void*) iloz, (void*) ihiz, (void*) z, (void*) ldz, (void*) ns, (void*) nd, (void*) sh, (void*) v, (void*) ldv, (void*) nh, (void*) t, (void*) ldt, (void*) nv, (void*) wv, (void*) ldwv, (void*) work, (void*) lwork);

    helpTimeStop = flexiblas_wtime();

    data->zlaqr3.timings[0] += (helpTimeStop - helpTime);
    data->zlaqr3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlatrs (char* uplo, char* trans, char* diag, char* normin, Int * n, double* a, Int * lda, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_dlatrs(char* uplo, char* trans, char* diag, char* normin, Int * n, double* a, Int * lda, double* x, double* scale, double* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlatrs((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) a, (void*) lda, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->dlatrs.timings[0] += (helpTimeStop - helpTime);
    data->dlatrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_ssyconvf_rook (char* uplo, char* way, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo, int len_way);
void hook_ssyconvf_rook(char* uplo, char* way, Int * n, float* a, Int * lda, float* e, Int * ipiv, Int * info, int len_uplo, int len_way)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssyconvf_rook((void*) uplo, (void*) way, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo, (int) len_way);

    helpTimeStop = flexiblas_wtime();

    data->ssyconvf_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssyconvf_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_zpotri (char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo);
void hook_zpotri(char* uplo, Int * n, double complex* a, Int * lda, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zpotri((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zpotri.timings[0] += (helpTimeStop - helpTime);
    data->zpotri.calls[0]++;

    return ;
}



extern void flexiblas_chain_dbbcsd (char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, Int * m, Int * p, Int * q, double* theta, double* phi, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* v2t, Int * ldv2t, double* b11d, double* b11e, double* b12d, double* b12e, double* b21d, double* b21e, double* b22d, double* b22e, double* work, Int * lwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans);
void hook_dbbcsd(char* jobu1, char* jobu2, char* jobv1t, char* jobv2t, char* trans, Int * m, Int * p, Int * q, double* theta, double* phi, double* u1, Int * ldu1, double* u2, Int * ldu2, double* v1t, Int * ldv1t, double* v2t, Int * ldv2t, double* b11d, double* b11e, double* b12d, double* b12e, double* b21d, double* b21e, double* b22d, double* b22e, double* work, Int * lwork, Int * info, int len_jobu1, int len_jobu2, int len_jobv1t, int len_jobv2t, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dbbcsd((void*) jobu1, (void*) jobu2, (void*) jobv1t, (void*) jobv2t, (void*) trans, (void*) m, (void*) p, (void*) q, (void*) theta, (void*) phi, (void*) u1, (void*) ldu1, (void*) u2, (void*) ldu2, (void*) v1t, (void*) ldv1t, (void*) v2t, (void*) ldv2t, (void*) b11d, (void*) b11e, (void*) b12d, (void*) b12e, (void*) b21d, (void*) b21e, (void*) b22d, (void*) b22e, (void*) work, (void*) lwork, (void*) info, (int) len_jobu1, (int) len_jobu2, (int) len_jobv1t, (int) len_jobv2t, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->dbbcsd.timings[0] += (helpTimeStop - helpTime);
    data->dbbcsd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgtsv (Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* b, Int * ldb, Int * info);
void hook_cgtsv(Int * n, Int * nrhs, float complex* dl, float complex* d, float complex* du, float complex* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgtsv((void*) n, (void*) nrhs, (void*) dl, (void*) d, (void*) du, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cgtsv.timings[0] += (helpTimeStop - helpTime);
    data->cgtsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zungl2 (Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info);
void hook_zungl2(Int * m, Int * n, Int * k, double complex* a, Int * lda, double complex* tau, double complex* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zungl2((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zungl2.timings[0] += (helpTimeStop - helpTime);
    data->zungl2.calls[0]++;

    return ;
}



extern void flexiblas_chain_dtrsyl (char* trana, char* tranb, Int * isgn, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* scale, Int * info, int len_trana, int len_tranb);
void hook_dtrsyl(char* trana, char* tranb, Int * isgn, Int * m, Int * n, double* a, Int * lda, double* b, Int * ldb, double* c, Int * ldc, double* scale, Int * info, int len_trana, int len_tranb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dtrsyl((void*) trana, (void*) tranb, (void*) isgn, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) scale, (void*) info, (int) len_trana, (int) len_tranb);

    helpTimeStop = flexiblas_wtime();

    data->dtrsyl.timings[0] += (helpTimeStop - helpTime);
    data->dtrsyl.calls[0]++;

    return ;
}



extern void flexiblas_chain_slatrs (char* uplo, char* trans, char* diag, char* normin, Int * n, float* a, Int * lda, float* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_slatrs(char* uplo, char* trans, char* diag, char* normin, Int * n, float* a, Int * lda, float* x, float* scale, float* cnorm, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slatrs((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) a, (void*) lda, (void*) x, (void*) scale, (void*) cnorm, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->slatrs.timings[0] += (helpTimeStop - helpTime);
    data->slatrs.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhegv (Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo);
void hook_zhegv(Int * itype, char* jobz, char* uplo, Int * n, double complex* a, Int * lda, double complex* b, Int * ldb, double* w, double complex* work, Int * lwork, double* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhegv((void*) itype, (void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) w, (void*) work, (void*) lwork, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhegv.timings[0] += (helpTimeStop - helpTime);
    data->zhegv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dpoequ (Int * n, double* a, Int * lda, double* s, double* scond, double* amax, Int * info);
void hook_dpoequ(Int * n, double* a, Int * lda, double* s, double* scond, double* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dpoequ((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dpoequ.timings[0] += (helpTimeStop - helpTime);
    data->dpoequ.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhbevx_2stage (char* jobz, char* range, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_zhbevx_2stage(char* jobz, char* range, char* uplo, Int * n, Int * kd, double complex* ab, Int * ldab, double complex* q, Int * ldq, double* vl, double* vu, Int * il, Int * iu, double* abstol, Int * m, double* w, double complex* z, Int * ldz, double complex* work, Int * lwork, double* rwork, Int * iwork, Int * ifail, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhbevx_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) q, (void*) ldq, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) iwork, (void*) ifail, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->zhbevx_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zsyswapr (char* uplo, Int * n, double complex* a, Int * lda, Int * i1, Int * i2, int len_uplo);
void hook_zsyswapr(char* uplo, Int * n, double complex* a, Int * lda, Int * i1, Int * i2, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zsyswapr((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) i1, (void*) i2, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zsyswapr.timings[0] += (helpTimeStop - helpTime);
    data->zsyswapr.calls[0]++;

    return ;
}



extern void flexiblas_chain_stbcon (char* norm, char* uplo, char* diag, Int * n, Int * kd, float* ab, Int * ldab, float* rcond, float* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag);
void hook_stbcon(char* norm, char* uplo, char* diag, Int * n, Int * kd, float* ab, Int * ldab, float* rcond, float* work, Int * iwork, Int * info, int len_norm, int len_uplo, int len_diag)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stbcon((void*) norm, (void*) uplo, (void*) diag, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) rcond, (void*) work, (void*) iwork, (void*) info, (int) len_norm, (int) len_uplo, (int) len_diag);

    helpTimeStop = flexiblas_wtime();

    data->stbcon.timings[0] += (helpTimeStop - helpTime);
    data->stbcon.calls[0]++;

    return ;
}



extern void flexiblas_chain_sptsv (Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb, Int * info);
void hook_sptsv(Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sptsv((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->sptsv.timings[0] += (helpTimeStop - helpTime);
    data->sptsv.calls[0]++;

    return ;
}



extern void flexiblas_chain_sptts2 (Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb);
void hook_sptts2(Int * n, Int * nrhs, float* d, float* e, float* b, Int * ldb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sptts2((void*) n, (void*) nrhs, (void*) d, (void*) e, (void*) b, (void*) ldb);

    helpTimeStop = flexiblas_wtime();

    data->sptts2.timings[0] += (helpTimeStop - helpTime);
    data->sptts2.calls[0]++;

    return ;
}



extern void flexiblas_chain_ctrsyl3 (char* trana, char* tranb, Int * isgn, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float* scale, float* swork, Int * ldswork, Int * info, int len_trana, int len_tranb);
void hook_ctrsyl3(char* trana, char* tranb, Int * isgn, Int * m, Int * n, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* c, Int * ldc, float* scale, float* swork, Int * ldswork, Int * info, int len_trana, int len_tranb)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ctrsyl3((void*) trana, (void*) tranb, (void*) isgn, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) c, (void*) ldc, (void*) scale, (void*) swork, (void*) ldswork, (void*) info, (int) len_trana, (int) len_tranb);

    helpTimeStop = flexiblas_wtime();

    data->ctrsyl3.timings[0] += (helpTimeStop - helpTime);
    data->ctrsyl3.calls[0]++;

    return ;
}



extern void flexiblas_chain_cungql (Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info);
void hook_cungql(Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cungql((void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cungql.timings[0] += (helpTimeStop - helpTime);
    data->cungql.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmlq (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cunmlq(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmlq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmlq.timings[0] += (helpTimeStop - helpTime);
    data->cunmlq.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlapll (Int * n, double complex* x, Int * incx, double complex* y, Int * incy, double* ssmin);
void hook_zlapll(Int * n, double complex* x, Int * incx, double complex* y, Int * incy, double* ssmin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlapll((void*) n, (void*) x, (void*) incx, (void*) y, (void*) incy, (void*) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->zlapll.timings[0] += (helpTimeStop - helpTime);
    data->zlapll.calls[0]++;

    return ;
}



extern void flexiblas_chain_dlarft (char* direct, char* storev, Int * n, Int * k, double* v, Int * ldv, double* tau, double* t, Int * ldt, int len_direct, int len_storev);
void hook_dlarft(char* direct, char* storev, Int * n, Int * k, double* v, Int * ldv, double* tau, double* t, Int * ldt, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dlarft((void*) direct, (void*) storev, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) tau, (void*) t, (void*) ldt, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->dlarft.timings[0] += (helpTimeStop - helpTime);
    data->dlarft.calls[0]++;

    return ;
}



extern void flexiblas_chain_zggglm (Int * n, Int * m, Int * p, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* d, double complex* x, double complex* y, double complex* work, Int * lwork, Int * info);
void hook_zggglm(Int * n, Int * m, Int * p, double complex* a, Int * lda, double complex* b, Int * ldb, double complex* d, double complex* x, double complex* y, double complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zggglm((void*) n, (void*) m, (void*) p, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) d, (void*) x, (void*) y, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zggglm.timings[0] += (helpTimeStop - helpTime);
    data->zggglm.calls[0]++;

    return ;
}



extern int flexiblas_chain_ilauplo (char* uplo, int len_uplo);
int hook_ilauplo(char* uplo, int len_uplo)
{
    int v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_ilauplo((void*) uplo, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ilauplo.timings[0] += (helpTimeStop - helpTime);
    data->ilauplo.calls[0]++;

    return v;
}



extern void flexiblas_chain_dsbevd (char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_dsbevd(char* jobz, char* uplo, Int * n, Int * kd, double* ab, Int * ldab, double* w, double* z, Int * ldz, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbevd((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbevd.timings[0] += (helpTimeStop - helpTime);
    data->dsbevd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgelst (char* trans, Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_trans);
void hook_cgelst(char* trans, Int * m, Int * n, Int * nrhs, float complex* a, Int * lda, float complex* b, Int * ldb, float complex* work, Int * lwork, Int * info, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgelst((void*) trans, (void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) work, (void*) lwork, (void*) info, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cgelst.timings[0] += (helpTimeStop - helpTime);
    data->cgelst.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgelss (Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double* work, Int * lwork, Int * info);
void hook_dgelss(Int * m, Int * n, Int * nrhs, double* a, Int * lda, double* b, Int * ldb, double* s, double* rcond, Int * rank_bn, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgelss((void*) m, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) s, (void*) rcond, (void*) rank_bn, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelss.timings[0] += (helpTimeStop - helpTime);
    data->dgelss.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbgv (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * info, int len_jobz, int len_uplo);
void hook_chbgv(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float complex* ab, Int * ldab, float complex* bb, Int * ldbb, float* w, float complex* z, Int * ldz, float complex* work, float* rwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbgv((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) rwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbgv.timings[0] += (helpTimeStop - helpTime);
    data->chbgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlarfb (char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev);
void hook_zlarfb(char* side, char* trans, char* direct, char* storev, Int * m, Int * n, Int * k, double complex* v, Int * ldv, double complex* t, Int * ldt, double complex* c, Int * ldc, double complex* work, Int * ldwork, int len_side, int len_trans, int len_direct, int len_storev)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlarfb((void*) side, (void*) trans, (void*) direct, (void*) storev, (void*) m, (void*) n, (void*) k, (void*) v, (void*) ldv, (void*) t, (void*) ldt, (void*) c, (void*) ldc, (void*) work, (void*) ldwork, (int) len_side, (int) len_trans, (int) len_direct, (int) len_storev);

    helpTimeStop = flexiblas_wtime();

    data->zlarfb.timings[0] += (helpTimeStop - helpTime);
    data->zlarfb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrf_aa_2stage (char* uplo, Int * n, double* a, Int * lda, double* tb, Int * ltb, Int * ipiv, Int * ipiv2, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrf_aa_2stage(char* uplo, Int * n, double* a, Int * lda, double* tb, Int * ltb, Int * ipiv, Int * ipiv2, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrf_aa_2stage((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) tb, (void*) ltb, (void*) ipiv, (void*) ipiv2, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_aa_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmrq (char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans);
void hook_cunmrq(char* side, char* trans, Int * m, Int * n, Int * k, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * lwork, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmrq((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) lwork, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmrq.timings[0] += (helpTimeStop - helpTime);
    data->cunmrq.calls[0]++;

    return ;
}



extern float flexiblas_chain_clange (char* norm, Int * m, Int * n, float complex* a, Int * lda, float* work, int len_norm);
float hook_clange(char* norm, Int * m, Int * n, float complex* a, Int * lda, float* work, int len_norm)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_clange((void*) norm, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) work, (int) len_norm);

    helpTimeStop = flexiblas_wtime();

    data->clange.timings[0] += (helpTimeStop - helpTime);
    data->clange.calls[0]++;

    return v;
}



extern void flexiblas_chain_ssbgvd (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_ssbgvd(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, float* ab, Int * ldab, float* bb, Int * ldbb, float* w, float* z, Int * ldz, float* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_ssbgvd((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->ssbgvd.timings[0] += (helpTimeStop - helpTime);
    data->ssbgvd.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunbdb4 (Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* phantom, float complex* work, Int * lwork, Int * info);
void hook_cunbdb4(Int * m, Int * p, Int * q, float complex* x11, Int * ldx11, float complex* x21, Int * ldx21, float* theta, float* phi, float complex* taup1, float complex* taup2, float complex* tauq1, float complex* phantom, float complex* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunbdb4((void*) m, (void*) p, (void*) q, (void*) x11, (void*) ldx11, (void*) x21, (void*) ldx21, (void*) theta, (void*) phi, (void*) taup1, (void*) taup2, (void*) tauq1, (void*) phantom, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb4.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb4.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgetri (Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info);
void hook_dgetri(Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgetri((void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetri.timings[0] += (helpTimeStop - helpTime);
    data->dgetri.calls[0]++;

    return ;
}



extern void flexiblas_chain_sorm2r (char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans);
void hook_sorm2r(char* side, char* trans, Int * m, Int * n, Int * k, float* a, Int * lda, float* tau, float* c, Int * ldc, float* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sorm2r((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->sorm2r.timings[0] += (helpTimeStop - helpTime);
    data->sorm2r.calls[0]++;

    return ;
}



extern void flexiblas_chain_cunmr3 (char* side, char* trans, Int * m, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans);
void hook_cunmr3(char* side, char* trans, Int * m, Int * n, Int * k, Int * l, float complex* a, Int * lda, float complex* tau, float complex* c, Int * ldc, float complex* work, Int * info, int len_side, int len_trans)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cunmr3((void*) side, (void*) trans, (void*) m, (void*) n, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) tau, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_side, (int) len_trans);

    helpTimeStop = flexiblas_wtime();

    data->cunmr3.timings[0] += (helpTimeStop - helpTime);
    data->cunmr3.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrf_aa (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrf_aa(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrf_aa((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_aa.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsbgv (char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo);
void hook_dsbgv(char* jobz, char* uplo, Int * n, Int * ka, Int * kb, double* ab, Int * ldab, double* bb, Int * ldbb, double* w, double* z, Int * ldz, double* work, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsbgv((void*) jobz, (void*) uplo, (void*) n, (void*) ka, (void*) kb, (void*) ab, (void*) ldab, (void*) bb, (void*) ldbb, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsbgv.timings[0] += (helpTimeStop - helpTime);
    data->dsbgv.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsyevd_2stage (char* jobz, char* uplo, Int * n, double* a, Int * lda, double* w, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_dsyevd_2stage(char* jobz, char* uplo, Int * n, double* a, Int * lda, double* w, double* work, Int * lwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsyevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) w, (void*) work, (void*) lwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsyevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_stgex2 (Int * wantq, Int * wantz, Int * n, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, Int * j1, Int * n1, Int * n2, float* work, Int * lwork, Int * info);
void hook_stgex2(Int * wantq, Int * wantz, Int * n, float* a, Int * lda, float* b, Int * ldb, float* q, Int * ldq, float* z, Int * ldz, Int * j1, Int * n1, Int * n2, float* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stgex2((void*) wantq, (void*) wantz, (void*) n, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) q, (void*) ldq, (void*) z, (void*) ldz, (void*) j1, (void*) n1, (void*) n2, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stgex2.timings[0] += (helpTimeStop - helpTime);
    data->stgex2.calls[0]++;

    return ;
}



extern void flexiblas_chain_spoequb (Int * n, float* a, Int * lda, float* s, float* scond, float* amax, Int * info);
void hook_spoequb(Int * n, float* a, Int * lda, float* s, float* scond, float* amax, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_spoequb((void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->spoequb.timings[0] += (helpTimeStop - helpTime);
    data->spoequb.calls[0]++;

    return ;
}



extern void flexiblas_chain_dgehrd (Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info);
void hook_dgehrd(Int * n, Int * ilo, Int * ihi, double* a, Int * lda, double* tau, double* work, Int * lwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dgehrd((void*) n, (void*) ilo, (void*) ihi, (void*) a, (void*) lda, (void*) tau, (void*) work, (void*) lwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->dgehrd.timings[0] += (helpTimeStop - helpTime);
    data->dgehrd.calls[0]++;

    return ;
}



extern void flexiblas_chain_stplqt (Int * m, Int * n, Int * l, Int * mb, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, float* work, Int * info);
void hook_stplqt(Int * m, Int * n, Int * l, Int * mb, float* a, Int * lda, float* b, Int * ldb, float* t, Int * ldt, float* work, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_stplqt((void*) m, (void*) n, (void*) l, (void*) mb, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) t, (void*) ldt, (void*) work, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->stplqt.timings[0] += (helpTimeStop - helpTime);
    data->stplqt.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqsy (char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed);
void hook_zlaqsy(char* uplo, Int * n, double complex* a, Int * lda, double* s, double* scond, double* amax, char* equed, int len_uplo, int len_equed)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqsy((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) scond, (void*) amax, (void*) equed, (int) len_uplo, (int) len_equed);

    helpTimeStop = flexiblas_wtime();

    data->zlaqsy.timings[0] += (helpTimeStop - helpTime);
    data->zlaqsy.calls[0]++;

    return ;
}



extern void flexiblas_chain_cheevr_2stage (char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, Int * isuppz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo);
void hook_cheevr_2stage(char* jobz, char* range, char* uplo, Int * n, float complex* a, Int * lda, float* vl, float* vu, Int * il, Int * iu, float* abstol, Int * m, float* w, float complex* z, Int * ldz, Int * isuppz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_range, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cheevr_2stage((void*) jobz, (void*) range, (void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) vl, (void*) vu, (void*) il, (void*) iu, (void*) abstol, (void*) m, (void*) w, (void*) z, (void*) ldz, (void*) isuppz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_range, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cheevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevr_2stage.calls[0]++;

    return ;
}



extern double flexiblas_chain_dzsum1 (Int * n, double complex* cx, Int * incx);
double hook_dzsum1(Int * n, double complex* cx, Int * incx)
{
    double v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_dzsum1((void*) n, (void*) cx, (void*) incx);

    helpTimeStop = flexiblas_wtime();

    data->dzsum1.timings[0] += (helpTimeStop - helpTime);
    data->dzsum1.calls[0]++;

    return v;
}



extern void flexiblas_chain_dggsvd3 (char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, double* a, Int * lda, double* b, Int * ldb, double* alpha, double* beta, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq);
void hook_dggsvd3(char* jobu, char* jobv, char* jobq, Int * m, Int * n, Int * p, Int * k, Int * l, double* a, Int * lda, double* b, Int * ldb, double* alpha, double* beta, double* u, Int * ldu, double* v, Int * ldv, double* q, Int * ldq, double* work, Int * lwork, Int * iwork, Int * info, int len_jobu, int len_jobv, int len_jobq)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dggsvd3((void*) jobu, (void*) jobv, (void*) jobq, (void*) m, (void*) n, (void*) p, (void*) k, (void*) l, (void*) a, (void*) lda, (void*) b, (void*) ldb, (void*) alpha, (void*) beta, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) q, (void*) ldq, (void*) work, (void*) lwork, (void*) iwork, (void*) info, (int) len_jobu, (int) len_jobv, (int) len_jobq);

    helpTimeStop = flexiblas_wtime();

    data->dggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->dggsvd3.calls[0]++;

    return ;
}



extern void flexiblas_chain_zgesvdq (char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, double complex* a, Int * lda, double* s, double complex* u, Int * ldu, double complex* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, double complex* cwork, Int * lcwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv);
void hook_zgesvdq(char* joba, char* jobp, char* jobr, char* jobu, char* jobv, Int * m, Int * n, double complex* a, Int * lda, double* s, double complex* u, Int * ldu, double complex* v, Int * ldv, Int * numrank, Int * iwork, Int * liwork, double complex* cwork, Int * lcwork, double* rwork, Int * lrwork, Int * info, int len_joba, int len_jobp, int len_jobr, int len_jobu, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zgesvdq((void*) joba, (void*) jobp, (void*) jobr, (void*) jobu, (void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) s, (void*) u, (void*) ldu, (void*) v, (void*) ldv, (void*) numrank, (void*) iwork, (void*) liwork, (void*) cwork, (void*) lcwork, (void*) rwork, (void*) lrwork, (void*) info, (int) len_joba, (int) len_jobp, (int) len_jobr, (int) len_jobu, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->zgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->zgesvdq.calls[0]++;

    return ;
}



extern void flexiblas_chain_csytrs_rook (char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo);
void hook_csytrs_rook(char* uplo, Int * n, Int * nrhs, float complex* a, Int * lda, Int * ipiv, float complex* b, Int * ldb, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csytrs_rook((void*) uplo, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) ipiv, (void*) b, (void*) ldb, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_csycon_rook (char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo);
void hook_csycon_rook(char* uplo, Int * n, float complex* a, Int * lda, Int * ipiv, float* anorm, float* rcond, float complex* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_csycon_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) anorm, (void*) rcond, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->csycon_rook.timings[0] += (helpTimeStop - helpTime);
    data->csycon_rook.calls[0]++;

    return ;
}



extern float flexiblas_chain_second (void);
float hook_second(void)
{
    float v;
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    v = flexiblas_chain_second();

    helpTimeStop = flexiblas_wtime();

    data->second.timings[0] += (helpTimeStop - helpTime);
    data->second.calls[0]++;

    return v;
}



extern void flexiblas_chain_dbdsqr (char* uplo, Int * n, Int * ncvt, Int * nru, Int * ncc, double* d, double* e, double* vt, Int * ldvt, double* u, Int * ldu, double* c, Int * ldc, double* work, Int * info, int len_uplo);
void hook_dbdsqr(char* uplo, Int * n, Int * ncvt, Int * nru, Int * ncc, double* d, double* e, double* vt, Int * ldvt, double* u, Int * ldu, double* c, Int * ldc, double* work, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dbdsqr((void*) uplo, (void*) n, (void*) ncvt, (void*) nru, (void*) ncc, (void*) d, (void*) e, (void*) vt, (void*) ldvt, (void*) u, (void*) ldu, (void*) c, (void*) ldc, (void*) work, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dbdsqr.timings[0] += (helpTimeStop - helpTime);
    data->dbdsqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlaqp3rk (Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double complex* a, Int * lda, Int * done, Int * kb, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double* vn1, double* vn2, double complex* auxv, double complex* f, Int * ldf, Int * iwork, Int * info);
void hook_zlaqp3rk(Int * m, Int * n, Int * nrhs, Int * ioffset, Int * nb, double* abstol, double* reltol, Int * kp1, double* maxc2nrm, double complex* a, Int * lda, Int * done, Int * kb, double* maxc2nrmk, double* relmaxc2nrmk, Int * jpiv, double complex* tau, double* vn1, double* vn2, double complex* auxv, double complex* f, Int * ldf, Int * iwork, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlaqp3rk((void*) m, (void*) n, (void*) nrhs, (void*) ioffset, (void*) nb, (void*) abstol, (void*) reltol, (void*) kp1, (void*) maxc2nrm, (void*) a, (void*) lda, (void*) done, (void*) kb, (void*) maxc2nrmk, (void*) relmaxc2nrmk, (void*) jpiv, (void*) tau, (void*) vn1, (void*) vn2, (void*) auxv, (void*) f, (void*) ldf, (void*) iwork, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->zlaqp3rk.timings[0] += (helpTimeStop - helpTime);
    data->zlaqp3rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_cgsvj0 (char* jobv, Int * m, Int * n, float complex* a, Int * lda, float complex* d, float* sva, Int * mv, float complex* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float complex* work, Int * lwork, Int * info, int len_jobv);
void hook_cgsvj0(char* jobv, Int * m, Int * n, float complex* a, Int * lda, float complex* d, float* sva, Int * mv, float complex* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float complex* work, Int * lwork, Int * info, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cgsvj0((void*) jobv, (void*) m, (void*) n, (void*) a, (void*) lda, (void*) d, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) eps, (void*) sfmin, (void*) tol, (void*) nsweep, (void*) work, (void*) lwork, (void*) info, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->cgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->cgsvj0.calls[0]++;

    return ;
}



extern void flexiblas_chain_zlatrs3 (char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* x, Int * ldx, double* scale, double* cnorm, double* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin);
void hook_zlatrs3(char* uplo, char* trans, char* diag, char* normin, Int * n, Int * nrhs, double complex* a, Int * lda, double complex* x, Int * ldx, double* scale, double* cnorm, double* work, Int * lwork, Int * info, int len_uplo, int len_trans, int len_diag, int len_normin)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zlatrs3((void*) uplo, (void*) trans, (void*) diag, (void*) normin, (void*) n, (void*) nrhs, (void*) a, (void*) lda, (void*) x, (void*) ldx, (void*) scale, (void*) cnorm, (void*) work, (void*) lwork, (void*) info, (int) len_uplo, (int) len_trans, (int) len_diag, (int) len_normin);

    helpTimeStop = flexiblas_wtime();

    data->zlatrs3.timings[0] += (helpTimeStop - helpTime);
    data->zlatrs3.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgeev (char* jobvl, char* jobvr, Int * n, float* a, Int * lda, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, float* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr);
void hook_sgeev(char* jobvl, char* jobvr, Int * n, float* a, Int * lda, float* wr, float* wi, float* vl, Int * ldvl, float* vr, Int * ldvr, float* work, Int * lwork, Int * info, int len_jobvl, int len_jobvr)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgeev((void*) jobvl, (void*) jobvr, (void*) n, (void*) a, (void*) lda, (void*) wr, (void*) wi, (void*) vl, (void*) ldvl, (void*) vr, (void*) ldvr, (void*) work, (void*) lwork, (void*) info, (int) len_jobvl, (int) len_jobvr);

    helpTimeStop = flexiblas_wtime();

    data->sgeev.timings[0] += (helpTimeStop - helpTime);
    data->sgeev.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetf2_rk (char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, Int * info, int len_uplo);
void hook_zhetf2_rk(char* uplo, Int * n, double complex* a, Int * lda, double complex* e, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetf2_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->zhetf2_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_cpteqr (char* compz, Int * n, float* d, float* e, float complex* z, Int * ldz, float* work, Int * info, int len_compz);
void hook_cpteqr(char* compz, Int * n, float* d, float* e, float complex* z, Int * ldz, float* work, Int * info, int len_compz)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cpteqr((void*) compz, (void*) n, (void*) d, (void*) e, (void*) z, (void*) ldz, (void*) work, (void*) info, (int) len_compz);

    helpTimeStop = flexiblas_wtime();

    data->cpteqr.timings[0] += (helpTimeStop - helpTime);
    data->cpteqr.calls[0]++;

    return ;
}



extern void flexiblas_chain_sgsvj1 (char* jobv, Int * m, Int * n, Int * n1, float* a, Int * lda, float* d, float* sva, Int * mv, float* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float* work, Int * lwork, Int * info, int len_jobv);
void hook_sgsvj1(char* jobv, Int * m, Int * n, Int * n1, float* a, Int * lda, float* d, float* sva, Int * mv, float* v, Int * ldv, float* eps, float* sfmin, float* tol, Int * nsweep, float* work, Int * lwork, Int * info, int len_jobv)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_sgsvj1((void*) jobv, (void*) m, (void*) n, (void*) n1, (void*) a, (void*) lda, (void*) d, (void*) sva, (void*) mv, (void*) v, (void*) ldv, (void*) eps, (void*) sfmin, (void*) tol, (void*) nsweep, (void*) work, (void*) lwork, (void*) info, (int) len_jobv);

    helpTimeStop = flexiblas_wtime();

    data->sgsvj1.timings[0] += (helpTimeStop - helpTime);
    data->sgsvj1.calls[0]++;

    return ;
}



extern void flexiblas_chain_slaln2 (Int * ltrans, Int * na, Int * nw, float* smin, float* ca, float* a, Int * lda, float* d1, float* d2, float* b, Int * ldb, float* wr, float* wi, float* x, Int * ldx, float* scale, float* xnorm, Int * info);
void hook_slaln2(Int * ltrans, Int * na, Int * nw, float* smin, float* ca, float* a, Int * lda, float* d1, float* d2, float* b, Int * ldb, float* wr, float* wi, float* x, Int * ldx, float* scale, float* xnorm, Int * info)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_slaln2((void*) ltrans, (void*) na, (void*) nw, (void*) smin, (void*) ca, (void*) a, (void*) lda, (void*) d1, (void*) d2, (void*) b, (void*) ldb, (void*) wr, (void*) wi, (void*) x, (void*) ldx, (void*) scale, (void*) xnorm, (void*) info);

    helpTimeStop = flexiblas_wtime();

    data->slaln2.timings[0] += (helpTimeStop - helpTime);
    data->slaln2.calls[0]++;

    return ;
}



extern void flexiblas_chain_cspr (char* uplo, Int * n, float complex* alpha, float complex* x, Int * incx, float complex* ap, int len_uplo);
void hook_cspr(char* uplo, Int * n, float complex* alpha, float complex* x, Int * incx, float complex* ap, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_cspr((void*) uplo, (void*) n, (void*) alpha, (void*) x, (void*) incx, (void*) ap, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->cspr.timings[0] += (helpTimeStop - helpTime);
    data->cspr.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytrf_rk (char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo);
void hook_dsytrf_rk(char* uplo, Int * n, double* a, Int * lda, double* e, Int * ipiv, double* work, Int * lwork, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_dsytrf_rk((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) e, (void*) ipiv, (void*) work, (void*) lwork, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_rk.calls[0]++;

    return ;
}



extern void flexiblas_chain_chbevd_2stage (char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo);
void hook_chbevd_2stage(char* jobz, char* uplo, Int * n, Int * kd, float complex* ab, Int * ldab, float* w, float complex* z, Int * ldz, float complex* work, Int * lwork, float* rwork, Int * lrwork, Int * iwork, Int * liwork, Int * info, int len_jobz, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_chbevd_2stage((void*) jobz, (void*) uplo, (void*) n, (void*) kd, (void*) ab, (void*) ldab, (void*) w, (void*) z, (void*) ldz, (void*) work, (void*) lwork, (void*) rwork, (void*) lrwork, (void*) iwork, (void*) liwork, (void*) info, (int) len_jobz, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->chbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbevd_2stage.calls[0]++;

    return ;
}



extern void flexiblas_chain_zhetf2_rook (char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_zhetf2_rook(char* uplo, Int * n, double complex* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();

    flexiblas_chain_zhetf2_rook((void*) uplo, (void*) n, (void*) a, (void*) lda, (void*) ipiv, (void*) info, (int) len_uplo);

    helpTimeStop = flexiblas_wtime();

    data->zhetf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->zhetf2_rook.calls[0]++;

    return ;
}



extern void flexiblas_chain_dsytf2_rook (char* uplo, Int * n, double* a, Int * lda, Int * ipiv, Int * info, int len_uplo);
void hook_dsytf2_rook(char* uplo, Int * n, double* a, Int * lda, Int * ipiv, Int * info, int len_uplo)
{

    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas