use clap::builder::NonEmptyStringValueParser;
use clap::{crate_authors, crate_description, crate_version, Arg, ArgAction, ArgMatches, Command};

pub fn launch() -> ArgMatches {
    Command::new("t-rec")
        .allow_missing_positional(true)
        .version(crate_version!())
        .author(crate_authors!())
        .about(crate_description!())
        .arg(
            Arg::new("verbose")
                .action(ArgAction::SetTrue)
                .short('v')
                .long("verbose")
                .required(false)
                .help("Enable verbose insights for the curious")
        )
        .arg(
            Arg::new("quiet")
                .action(ArgAction::SetTrue)
                .short('q')
                .long("quiet")
                .required(false)
                .help("Quiet mode, suppresses the banner: 'Press Ctrl+D to end recording'")
        )
        .arg(
            Arg::new("video")
                .action(ArgAction::SetTrue)
                .short('m')
                .long("video")
                .required(false)
                .help("Generates additionally to the gif a mp4 video of the recording")
        )
        .arg(
            Arg::new("video-only")
                .action(ArgAction::SetTrue)
                .short('M')
                .long("video-only")
                .required(false)
                .conflicts_with("video")
                .help("Generates only a mp4 video and not gif")
        )
        .arg(
            Arg::new("decor")
                .value_parser(["shadow", "none"])
                .default_value("none")
                .required(false)
                .short('d')
                .long("decor")
                .help("Decorates the animation with certain, mostly border effects")
        )
        .arg(
            Arg::new("wallpaper")
                .value_parser(NonEmptyStringValueParser::new())
                .required(false)
                .short('p')
                .long("wallpaper")
                .help("Wallpaper background. Use 'ventura' for built-in, or provide a path to a custom image (PNG, JPEG, TGA)")
        )
        .arg(
            Arg::new("wallpaper-padding")
                .value_parser(clap::value_parser!(u32).range(1..=500))
                .default_value("60")
                .required(false)
                .long("wallpaper-padding")
                .help("Padding in pixels around the recording when using --wallpaper (1-500, default: 60)")
        )
        .arg(
            Arg::new("bg")
                .value_parser(["white", "black", "transparent"])
                .default_value("transparent")
                .required(false)
                .short('b')
                .long("bg")
                .help("Background color when decors are used")
        )
        .arg(
            Arg::new("natural-mode")
                .action(ArgAction::SetTrue)
                .value_name("natural")
                .required(false)
                .short('n')
                .long("natural")
                .help("If you want a very natural typing experience and disable the idle detection and sampling optimization")
        )
        .arg(
            Arg::new("list-windows")
                .action(ArgAction::SetTrue)
                .value_name("list all visible windows with name and id")
                .required(false)
                .short('l')
                .long("ls-win")
                .long("ls")
                .help("If you want to see a list of windows available for recording by their id, you can set env var 'WINDOWID' or `--win-id` to record this specific window only"),
        )
        .arg(
            Arg::new("win-id")
                .value_parser(clap::value_parser!(u64))
                .short('w')
                .long("win-id")
                .required(false)
                .help("Window Id (see --ls-win) that should be captured, instead of the current terminal")
        )
        .arg(
            Arg::new("end-pause")
                .value_parser(NonEmptyStringValueParser::new())
                .value_name("s | ms | m")
                .required(false)
                .short('e')
                .long("end-pause")
                .help("to specify the pause time at the end of the animation, that time the gif will show the last frame"),
        )
        .arg(
            Arg::new("start-pause")
                .value_parser(NonEmptyStringValueParser::new())
                .value_name("s | ms | m")
                .required(false)
                .short('s')
                .long("start-pause")
                .help("to specify the pause time at the start of the animation, that time the gif will show the first frame"),
        )
        .arg(
            Arg::new("idle-pause")
                .value_parser(NonEmptyStringValueParser::new())
                .value_name("s | ms | m")
                .required(false)
                .short('i')
                .long("idle-pause")
                .default_value("3s")
                .help("to preserve natural pauses up to a maximum duration by overriding idle detection. Can enhance readability."),
        )
        .arg(
            Arg::new("file")
                .value_parser(NonEmptyStringValueParser::new())
                .required(false)
                .short('o')
                .long("output")
                .default_value("t-rec")
                .help("to specify the output file (without extension)"),
        )
        .arg(
            Arg::new("fps")
                .value_parser(clap::value_parser!(u8).range(4..=15))
                .default_value("4")
                .required(false)
                .short('f')
                .long("fps")
                .help("Capture framerate, 4-15 fps. Higher = smoother animations but larger files"),
        )
        .arg(
            Arg::new("program")
                .value_name("shell or program to launch")
                .value_parser(NonEmptyStringValueParser::new())
                .required(false)
                .help("If you want to start a different program than $SHELL you can pass it here. For example '/bin/sh'"),
        )
        .arg(
            Arg::new("profile")
                .value_parser(NonEmptyStringValueParser::new())
                .required(false)
                .long("profile")
                .help("Use a named profile from the config file"),
        )
        .arg(
            Arg::new("init-config")
                .action(ArgAction::SetTrue)
                .long("init-config")
                .help("Create a starter config file at ~/.config/t-rec/config.toml"),
        )
        .arg(
            Arg::new("list-profiles")
                .action(ArgAction::SetTrue)
                .long("list-profiles")
                .help("List available profiles from the config file"),
        )
        .get_matches()
}
