/*****************************************************************************
 * copy.c
 *****************************************************************************
 * Copyright (C) 2001, 2002, 2006 the VideoLAN team
 * $Id: b6046ac21e55d211d60068aed91041324bef249c $
 *
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 *          Eric Petit <titer@videolan.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <vlc_common.h>
#include <vlc_plugin.h>
#include <vlc_codec.h>
#include <vlc_block.h>

/*****************************************************************************
 * Module descriptor
 *****************************************************************************/
static int  Open ( vlc_object_t * );
static void Close( vlc_object_t * );

vlc_module_begin ()
    set_category( CAT_SOUT )
    set_subcategory( SUBCAT_SOUT_PACKETIZER )
    set_description( N_("Copy packetizer") )
    set_capability( "packetizer", 1 )
    set_callbacks( Open, Close )
vlc_module_end ()

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/
struct decoder_sys_t
{
    block_t *p_block;
};

static block_t *Packetize   ( decoder_t *, block_t ** );
static block_t *PacketizeSub( decoder_t *, block_t ** );

/*****************************************************************************
 * Open: probe the packetizer and return score
 *****************************************************************************
 * Tries to launch a decoder and return score so that the interface is able
 * to choose.
 *****************************************************************************/
static int Open( vlc_object_t *p_this )
{
    decoder_t     *p_dec = (decoder_t*)p_this;
    decoder_sys_t *p_sys;

    if( p_dec->fmt_in.i_cat != AUDIO_ES &&
        p_dec->fmt_in.i_cat != VIDEO_ES &&
        p_dec->fmt_in.i_cat != SPU_ES )
    {
        msg_Err( p_dec, "invalid ES type" );
        return VLC_EGENERIC;
    }

    if( p_dec->fmt_in.i_cat == SPU_ES )
        p_dec->pf_packetize = PacketizeSub;
    else
        p_dec->pf_packetize = Packetize;

    /* Create the output format */
    es_format_Copy( &p_dec->fmt_out, &p_dec->fmt_in );

    /* Fix the value of the fourcc for audio */
    if( p_dec->fmt_in.i_cat == AUDIO_ES )
    {
        p_dec->fmt_out.i_codec = vlc_fourcc_GetCodecAudio( p_dec->fmt_in.i_codec,
                                                           p_dec->fmt_in.audio.i_bitspersample );
        if( !p_dec->fmt_out.i_codec )
        {
            msg_Err( p_dec, "unknown raw audio sample size" );
            return VLC_EGENERIC;
        }
    }

    p_dec->p_sys = p_sys = malloc( sizeof(*p_sys) );
    p_sys->p_block    = NULL;

    return VLC_SUCCESS;
}

/*****************************************************************************
 * Close:
 *****************************************************************************/
static void Close( vlc_object_t *p_this )
{
    decoder_t     *p_dec = (decoder_t*)p_this;

    if( p_dec->p_sys->p_block )
    {
        block_ChainRelease( p_dec->p_sys->p_block );
    }

    es_format_Clean( &p_dec->fmt_out );
    free( p_dec->p_sys );
}

/*****************************************************************************
 * Packetize: packetize an unit (here copy a complete block )
 *****************************************************************************/
static block_t *Packetize ( decoder_t *p_dec, block_t **pp_block )
{
    block_t *p_block;
    block_t *p_ret = p_dec->p_sys->p_block;

    if( pp_block == NULL || *pp_block == NULL )
        return NULL;
    if( (*pp_block)->i_flags&(BLOCK_FLAG_DISCONTINUITY|BLOCK_FLAG_CORRUPTED) )
    {
        block_Release( *pp_block );
        return NULL;
    }

    p_block = *pp_block;
    *pp_block = NULL;

    if( p_block->i_dts <= VLC_TS_INVALID )
    {
        p_block->i_dts = p_block->i_pts;
    }

    if( p_block->i_dts <= VLC_TS_INVALID )
    {
        msg_Dbg( p_dec, "need valid dts" );
        block_Release( p_block );
        return NULL;
    }

    if( p_ret != NULL && p_block->i_pts > p_ret->i_pts )
    {
        p_ret->i_length = p_block->i_pts - p_ret->i_pts;
    }
    p_dec->p_sys->p_block = p_block;

    return p_ret;
}

/*****************************************************************************
 * PacketizeSub: packetize an unit (here copy a complete block )
 *****************************************************************************/
static block_t *PacketizeSub( decoder_t *p_dec, block_t **pp_block )
{
    block_t *p_block;

    if( pp_block == NULL || *pp_block == NULL )
        return NULL;
    if( (*pp_block)->i_flags&(BLOCK_FLAG_DISCONTINUITY|BLOCK_FLAG_CORRUPTED) )
    {
        block_Release( *pp_block );
        return NULL;
    }

    p_block = *pp_block;
    *pp_block = NULL;

    if( p_block->i_dts <= VLC_TS_INVALID )
    {
        p_block->i_dts = p_block->i_pts;
    }

    if( p_block->i_dts <= VLC_TS_INVALID )
    {
        msg_Dbg( p_dec, "need valid dts" );
        block_Release( p_block );
        return NULL;
    }

    return p_block;
}
