#!/usr/bin/env python
#---------------------------------------------------------------------------
"""
Usage:  build_renamers.py destdir modulename filename.xml

Scans the XML file produced by SWIG (see setup.py) and generate the
%rename directives needed to implement the new wx namespace.  The
rename directives are output in a file named _modulename_rename.i in
the destdir given.

Also output a reverse 'renamer' Python module located in
wxPython/filename.py (relative the the current dir) to make a
backwards compatibility interface for the old wxPython packages.
"""

import sys, os, tempfile, pprint
import xml.sax            
from distutils.spawn import spawn

try:
    import libxml2
    USE_LIBXML2 = True
except ImportError:
    USE_LIBXML2 = False

wxPythonDir = "wxPython"

#---------------------------------------------------------------------------


renamerTemplateStart = """\
// A bunch of %%rename directives generated by %s
// in order to remove the wx prefix from all global scope names.

#ifndef BUILDING_RENAMERS

"""

renamerTemplateEnd = """
#endif
"""

wxPythonTemplateStart = """\
## This file reverse renames symbols in the wx package to give
## them their wx prefix again, for backwards compatibility.
##
## Generated by %s

# This silly stuff here is so the wxPython.wx module doesn't conflict
# with the wx package.  We need to import modules from the wx package
# here, then we'll put the wxPython.wx entry back in sys.modules.
import sys
_wx = None
if sys.modules.has_key('wxPython.wx'):
    _wx = sys.modules['wxPython.wx']
    del sys.modules['wxPython.wx']

import wx.%s

sys.modules['wxPython.wx'] = _wx
del sys, _wx


# Now assign all the reverse-renamed names:
"""

wxPythonTemplateEnd = """

"""



#---------------------------------------------------------------------------

def main(args):
    # check args
    if len(args) < 3:
        print __doc__
        sys.exit(1)

    # check location (there should be a wxPython subdir)
    if not os.path.exists(wxPythonDir) or not os.path.isdir(wxPythonDir):
        print __doc__
        print "You should only run this script from the main wxPython source dir.\n"
        sys.exit(1)
        

    destdir = args[0]
    modname = args[1]
    xmlfile = args[2]
    
    swigDest = os.path.join(destdir, "_"+modname+"_rename.i")
    pyDest = os.path.join(wxPythonDir, modname + '.py')

    swigDestTemp = tempfile.mktemp('.tmp')
    swigFile = open(swigDestTemp, "w")
    swigFile.write(renamerTemplateStart % sys.argv[0])

    pyDestTemp = tempfile.mktemp('.tmp')
    pyFile = open(pyDestTemp, "w")
    pyFile.write(wxPythonTemplateStart % (sys.argv[0], modname))

    try:
        print "Parsing and building renamers", 
        if USE_LIBXML2:
            processXML(xmlfile, modname, swigFile, pyFile)
        else:
            print "using xml.sax..."
            xml.sax.parse(xmlfile, ContentHandler(modname, swigFile, pyFile))

    finally:

        checkOtherNames(pyFile, modname,
                        os.path.join(destdir, '_'+modname+'_reverse.txt'))
        pyFile.write(wxPythonTemplateEnd)        
        pyFile.close()

        swigFile.write(renamerTemplateEnd)
        swigFile.close()

        # Compare the files just created with the existing one and
        # blow away the old one if they are different.
        for dest, temp in [(swigDest, swigDestTemp),
                           (pyDest, pyDestTemp)]:
            if not os.path.exists(dest):
                os.rename(temp, dest)
            elif open(dest).read() != open(temp).read():
                os.unlink(dest)
                os.rename(temp, dest)
            else:
                print dest + " not changed."
                os.unlink(temp)

#---------------------------------------------------------------------------


def GetAttr(node, name):
    path = "./attributelist/attribute[@name='%s']/@value" % name
    n = node.xpathEval2(path)
    if len(n):
        return n[0].content
    else:
        return None
    

def processXML(xmlfile, modname, swigFile, pyFile):
    import libxml2
    print "using libxml2..."

    topnode = libxml2.parseFile(xmlfile).children

    # remove any import nodes as we don't need to do renamers for symbols found therein
    imports = topnode.xpathEval2("*/import")
    for n in imports:
        n.unlinkNode()
        n.freeNode()

    # do a depth first iteration over what's left
    for node in topnode:
        doRename = False
        doPtr = False
        addWX = False
        revOnly = False


        if node.name == "class":
            lastClassName = name = GetAttr(node, "name")
            lastClassSymName = sym_name = GetAttr(node, "sym_name")
            doRename = True
            doPtr = True
            if sym_name != name:
                name = sym_name
                addWX = True

        # renamed constructors
        elif node.name == "constructor":
            name     = GetAttr(node, "name")
            sym_name = GetAttr(node, "sym_name")
            if sym_name != name:
                name = sym_name
                addWX = True
                doRename = True

        # only enumitems at the top level
        elif node.name == "enumitem" and node.parent.parent.name == "include":
            name     = GetAttr(node, "name")
            sym_name = GetAttr(node, "sym_name")
            doRename = True


        elif node.name in ["cdecl", "constant"]:
            name      = GetAttr(node, "name")
            sym_name  = GetAttr(node, "sym_name")
            toplevel  = node.parent.name == "include"

            # top-level functions
            if toplevel and GetAttr(node, "view") == "globalfunctionHandler":
                doRename = True

            # top-level global vars
            elif toplevel and GetAttr(node, "feature_immutable") == "1":
                doRename = True

            # static methods
            elif GetAttr(node, "view") == "staticmemberfunctionHandler":
                name     = lastClassName + '_' + name
                sym_name = lastClassSymName + '_' + sym_name
                # only output the reverse renamer in this case
                doRename = revOnly = True
       
            if doRename and name != sym_name:
                name = sym_name
                addWX = True
               

        if doRename and name:
            old = new = name
            if old.startswith('wx') and not old.startswith('wxEVT_'):
                # remove all wx prefixes except wxEVT_ and write a %rename directive for it
                new = old[2:]
                if not revOnly:
                    swigFile.write("%%rename(%s)  %35s;\n" % (new, old))

            # Write assignments to import into the old wxPython namespace
            if addWX and not old.startswith('wx'):
                old = 'wx'+old
            pyFile.write("%s = wx.%s.%s\n" % (old, modname, new))
            if doPtr:
                pyFile.write("%sPtr = wx.%s.%sPtr\n" % (old, modname, new))
            

#---------------------------------------------------------------------------

def checkOtherNames(pyFile, moduleName, filename):
    if os.path.exists(filename):
        prefixes = []
        for line in file(filename):
            if line.endswith('\n'):
                line = line[:-1]
            if line and not line.startswith('#'):
                if line.endswith('*'):
                    prefixes.append(line[:-1])
                elif line.find('=') != -1:
                    pyFile.write("%s\n" % line)
                else:
                    wxname = 'wx' + line
                    if line.startswith('wx') or line.startswith('WX') or line.startswith('EVT'):
                        wxname = line
                    pyFile.write("%s = wx.%s.%s\n" % (wxname, moduleName, line))

        if prefixes:
            pyFile.write(
                "\n\nd = globals()\nfor k, v in wx.%s.__dict__.iteritems():"
                % moduleName)
            first = True
            for p in prefixes:
                if first:
                    pyFile.write("\n    if ")
                    first = False
                else:
                    pyFile.write("\n    elif ")
                pyFile.write("k.startswith('%s'):\n        d[k] = v" % p)
            pyFile.write("\ndel d, k, v\n\n")

                        
#---------------------------------------------------------------------------

interestingTypes = [ 'class', 'cdecl', 'enumitem', 'constructor', 'constant' ]
interestingAttrs = [ 'name', 'sym_name', 'decl', 'feature_immutable', 'module',
                     'storage', 'type' ]


class Element:
    def __init__(self, tagtype):
        self.tagtype   = tagtype
        self.level     = -1
        self.name      = None
        self.sym_name  = None
        self.decl      = None
        self.immutable = None
        self.klass     = None
        self.module    = None
        self.storage   = None
        self.type      = None
        self.startLine = -1


    def write(self, moduleName, swigFile, pyFile):
        doRename = False
        doPtr = False
        addWX = False
        revOnly = False

        #if self.name.find('DefaultPosition') != -1:
        #    pprint.pprint(self.__dict__)
       
        if self.tagtype in  ['cdecl', 'constant']:
            if self.storage == 'typedef':
                pass
            
            # top level functions
            elif self.level == 0 and self.decl != "":
                doRename = True

            # top level global vars
            elif self.level == 0 and self.immutable == '1':
                doRename = True

            # static methods
            elif self.storage == 'static':
                if not self.klass:
                    pprint.pprint(self.__dict__)
                else:
                    self.name     = self.klass + '_' + self.name
                    self.sym_name = self.sym_klass + '_' + self.sym_name
                    # only output the reverse renamer in this case
                    doRename = revOnly = True

                            

            if doRename and self.name != self.sym_name:
                #print "%-25s %-25s" % (self.name, self.sym_name)
                self.name = self.sym_name
                addWX = True
                

        elif self.tagtype == 'class' and self.module == moduleName:
            doRename = True
            doPtr = True
            if self.sym_name != self.klass:
                #print self.sym_name
                self.name = self.sym_name
                addWX = True

        elif self.tagtype == 'constructor':
            #print "%-25s  %-25s" % (self.name, self.sym_name)
            if self.sym_name != self.klass:
                #print self.sym_name
                self.name = self.sym_name
                addWX = True
                doRename = True

        elif self.tagtype == 'enumitem' and self.level == 0:
            doRename = True


        if doRename:
            #print "%-25s %-25s" % (self.name, self.sym_name)
            old = new = self.name
            if old.startswith('wx') and not old.startswith('wxEVT_'):
                # remove all wx prefixes except wxEVT_ and write a %rename directive for it
                new = old[2:]
                if not revOnly:
                    swigFile.write("%%rename(%s)  %35s;\n" % (new, old))

            # Write assignments to import into the old wxPython namespace
            if addWX and not old.startswith('wx'):
                old = 'wx'+old
            pyFile.write("%s = wx.%s.%s\n" % (old, moduleName, new))
            if doPtr:
                pyFile.write("%sPtr = wx.%s.%sPtr\n" % (old, moduleName, new))
                
            
            
        #else:
        #    text = "%07d  %d  %10s  %-35s %s\n" % (
        #        self.startLine, self.level, self.tagtype, self.name, self.decl)
        #    #rejects.write(text)
        #    print text,


#---------------------------------------------------------------------------

class ContentHandler(xml.sax.ContentHandler):
    def __init__(self, modname, swigFile, pyFile):
        xml.sax.ContentHandler.__init__(self)
        self.modname = modname
        self.swigFile = swigFile
        self.pyFile   = pyFile
        self.elements = []
        self.imports  = 0
        self.klass = None
        self.sym_klass = None


    def setDocumentLocator(self, locator):
        self.locator = locator



    def startElement(self, name, attrs):
        if name in interestingTypes:
            # start of a new element that we are interested in
            ce = Element(name)
            ce.startLine = self.locator.getLineNumber()
            ce.level = len(self.elements)
            if name == 'constructor':
                ce.klass = self.elements[0].name
            else:
                ce.klass = self.klass
                ce.sym_klass = self.sym_klass
            self.elements.insert(0, ce)


        elif len(self.elements) and name == 'attribute' and attrs['name'] in interestingAttrs:
            attrName = attrs['name']
            attrVal  = attrs['value']
            if attrName.startswith('feature_'):
                attrName = attrName.replace('feature_', '')
            ce = self.elements[0]
            if getattr(ce, attrName) is None:
                setattr(ce, attrName, attrVal)
            if ce.tagtype == 'class' and attrName == 'name' and self.klass is None:
                self.klass = attrVal
            if ce.tagtype == 'class' and attrName == 'sym_name' and self.sym_klass is None:
                self.sym_klass = attrVal
            

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'name':
##             # save the elements name
##             ce = self.elements[0]
##             if ce.name is None:
##                 ce.name = attrs['value']
##                 ce.nameLine = self.locator.getLineNumber()

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'sym_name':
##             # save the elements name
##             ce = self.elements[0]
##             if ce.sym_name is None:
##                 ce.sym_name = attrs['value']

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'decl':
##             # save the elements decl
##             ce = self.elements[0]
##             ce.decl = attrs['value']

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'feature_immutable':
##             # save the elements decl
##             ce = self.elements[0]
##             ce.immutable = int(attrs['value'])

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'module':
##             # save the elements decl
##             ce = self.elements[0]
##             ce.module = attrs['value']

        elif name == 'import':
            self.imports += 1

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'storage':
##             # save the elements decl
##             ce = self.elements[0]
##             ce.storage = attrs['value']

##         elif len(self.elements) and name == 'attribute' and attrs['name'] == 'type':
##             # save the elements decl
##             ce = self.elements[0]
##             ce.type = attrs['value']


    def endElement(self, name):
        if name in interestingTypes:
            # end of an element that we are interested in
            ce = self.elements.pop(0)
            
            if self.imports == 0:
                # only write for items that are in this file, not imported
                ce.write(self.modname, self.swigFile, self.pyFile)
            
        if name == 'import':
            self.imports -= 1

        if name == 'class':
            self.klass = None
            self.sym_klass = None
            

#---------------------------------------------------------------------------

if __name__ == "__main__":
    main(sys.argv[1:])

