/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */
#ifndef INCLUDED_SW_INC_TOX_HXX
#define INCLUDED_SW_INC_TOX_HXX

#include <cppuhelper/weakref.hxx>
#include <sal/log.hxx>

#include <i18nlangtag/lang.h>
#include <svl/poolitem.hxx>

#include <editeng/svxenum.hxx>
#include <swtypes.hxx>
#include <toxe.hxx>
#include <calbck.hxx>

#include <vector>

namespace com { namespace sun { namespace star {
    namespace text { class XDocumentIndexMark; }
} } }

class SwTOXType;
class SwTOXMark;
class SwTextTOXMark;
class SwDoc;

typedef std::vector<SwTOXMark*> SwTOXMarks;

// Entry of content index, alphabetical index or user defined index

extern const sal_Unicode C_NUM_REPL;
extern const sal_Unicode C_END_PAGE_NUM;
extern const OUString S_PAGE_DELI;

class SW_DLLPUBLIC SwTOXMark
    : public SfxPoolItem
    , public SwModify
{
    friend void _InitCore();
    friend class SwTextTOXMark;

    OUString aAltText;    // Text of caption is different.
    OUString aPrimaryKey;
    OUString aSecondaryKey;
    OUString aCitationKeyReading;

    // three more strings for phonetic sorting
    OUString aTextReading;
    OUString aPrimaryKeyReading;
    OUString aSecondaryKeyReading;

    SwTextTOXMark* pTextAttr;

    sal_uInt16  nLevel;
    OUString    m_aBookmarkName;
    OUString    m_aEntryTypeName;       // stored specific entry type name for INDEX field \f
    bool    bAutoGenerated : 1;     // generated using a concordance file
    bool    bMainEntry : 1;         // main entry emphasized by character style

    ::com::sun::star::uno::WeakReference<
        ::com::sun::star::text::XDocumentIndexMark> m_wXDocumentIndexMark;

    SwTOXMark();                    // to create the default attribute in _InitCore

protected:
    // SwClient
   virtual void Modify( const SfxPoolItem* pOld, const SfxPoolItem* pNew ) SAL_OVERRIDE;

public:
    TYPEINFO_OVERRIDE();   // rtti

    // single argument ctors shall be explicit.
    explicit SwTOXMark( const SwTOXType* pTyp );
    virtual ~SwTOXMark();

    SwTOXMark( const SwTOXMark& rCopy );
    SwTOXMark& operator=( const SwTOXMark& rCopy );

    // "pure virtual methods" of SfxPoolItem
    virtual bool            operator==( const SfxPoolItem& ) const SAL_OVERRIDE;
    virtual SfxPoolItem*    Clone( SfxItemPool* pPool = 0 ) const SAL_OVERRIDE;

    void InvalidateTOXMark();

    OUString                GetText() const;

    inline bool             IsAlternativeText() const;
    inline OUString         GetAlternativeText() const;

    inline void             SetAlternativeText( const OUString& rAlt );

    // content or user defined index
    inline void             SetLevel(sal_uInt16 nLevel);
    inline sal_uInt16           GetLevel() const;
    inline void             SetBookmarkName( const OUString& bName);
    inline OUString         GetBookmarkName() const;
    inline void             SetEntryTypeName( const OUString& sName);
    inline OUString         GetEntryTypeName() const;

    // for alphabetical index only
    inline void             SetPrimaryKey(const OUString& rStr );
    inline void             SetSecondaryKey(const OUString& rStr);
    inline void             SetTextReading(const OUString& rStr);
    inline void             SetPrimaryKeyReading(const OUString& rStr );
    inline void             SetSecondaryKeyReading(const OUString& rStr);
    inline void             SetCitationKeyReading(const OUString& rStr);

    inline OUString         GetPrimaryKey() const;
    inline OUString         GetSecondaryKey() const;
    inline OUString         GetTextReading() const;
    inline OUString         GetPrimaryKeyReading() const;
    inline OUString         GetSecondaryKeyReading() const;
    inline OUString         GetCitationKeyReading() const;

    bool                    IsAutoGenerated() const {return bAutoGenerated;}
    void                    SetAutoGenerated(bool bSet) {bAutoGenerated = bSet;}

    bool                    IsMainEntry() const {return bMainEntry;}
    void                    SetMainEntry(bool bSet) { bMainEntry = bSet;}

    inline const SwTOXType*    GetTOXType() const;

    const SwTextTOXMark* GetTextTOXMark() const   { return pTextAttr; }
          SwTextTOXMark* GetTextTOXMark()         { return pTextAttr; }

    SAL_DLLPRIVATE ::com::sun::star::uno::WeakReference<
        ::com::sun::star::text::XDocumentIndexMark> const& GetXTOXMark() const
            { return m_wXDocumentIndexMark; }
    SAL_DLLPRIVATE void SetXTOXMark(::com::sun::star::uno::Reference<
                    ::com::sun::star::text::XDocumentIndexMark> const& xMark)
            { m_wXDocumentIndexMark = xMark; }
    void DeRegister() { GetRegisteredInNonConst()->Remove( this ); }
    void RegisterToTOXType( SwTOXType& rMark );
    static void InsertTOXMarks( SwTOXMarks& aMarks, const SwTOXType& rType );
};

// index types
class SwTOXType : public SwModify
{
public:
    SwTOXType(TOXTypes eTyp, const OUString& aName);

    // @@@ public copy ctor, but no copy assignment?
    SwTOXType(const SwTOXType& rCopy);

    inline OUString         GetTypeName() const;
    inline TOXTypes         GetType() const;

private:
    OUString        aName;
    TOXTypes        eType;

    // @@@ public copy ctor, but no copy assignment?
    SwTOXType & operator= (const SwTOXType &) SAL_DELETED_FUNCTION;
};

// Structure of the index lines
#define FORM_TITLE              0
#define FORM_ALPHA_DELIMITTER   1
#define FORM_PRIMARY_KEY        2
#define FORM_SECONDARY_KEY      3
#define FORM_ENTRY              4

/*
 Pattern structure

 <E#> - entry number                    <E# CharStyleName,PoolId>
 <ET> - entry text                      <ET CharStyleName,PoolId>
 <E>  - entry text and number           <E CharStyleName,PoolId>
 <T>  - tab stop                        <T,,Position,Adjust>
 <C>  - chapter info n = {0, 1, 2, 3, 4 } values of SwChapterFormat <C CharStyleName,PoolId>
 <TX> - text token                      <X CharStyleName,PoolId, TOX_STYLE_DELIMITERTextContentTOX_STYLE_DELIMITER>
 <#>  - Page number                     <# CharStyleName,PoolId>
 <LS> - Link start                      <LS>
 <LE> - Link end                        <LE>
 <A00> - Authority entry field          <A02 CharStyleName, PoolId>
 */

// These enum values are stored and must not be changed!
enum FormTokenType
{
    TOKEN_ENTRY_NO,
    TOKEN_ENTRY_TEXT,
    TOKEN_ENTRY,
    TOKEN_TAB_STOP,
    TOKEN_TEXT,
    TOKEN_PAGE_NUMS,
    TOKEN_CHAPTER_INFO,
    TOKEN_LINK_START,
    TOKEN_LINK_END,
    TOKEN_AUTHORITY,
    TOKEN_END
};

struct SW_DLLPUBLIC SwFormToken
{
    OUString        sText;
    OUString        sCharStyleName;
    SwTwips         nTabStopPosition;
    FormTokenType   eTokenType;
    sal_uInt16          nPoolId;
    SvxTabAdjust    eTabAlign;
    sal_uInt16          nChapterFormat;     //SwChapterFormat;
    sal_uInt16          nOutlineLevel;//the maximum permitted outline level in numbering
    sal_uInt16          nAuthorityField;    //enum ToxAuthorityField
    sal_Unicode     cTabFillChar;
    bool        bWithTab;      // true: do generate tab
                                   // character only the tab stop
                                   // #i21237#

    SwFormToken(FormTokenType eType ) :
        nTabStopPosition(0),
        eTokenType(eType),
        nPoolId(USHRT_MAX),
        eTabAlign( SVX_TAB_ADJUST_LEFT ),
        nChapterFormat(0 /*CF_NUMBER*/),
        nOutlineLevel(MAXLEVEL),   //default to maximum outline level
        nAuthorityField(0 /*AUTH_FIELD_IDENTIFIER*/),
        cTabFillChar(' '),
        bWithTab(true)  // #i21237#
    {}

    OUString GetString() const;
};

struct SwFormTokenEqualToFormTokenType
{
    FormTokenType eType;

    SwFormTokenEqualToFormTokenType(FormTokenType _eType) : eType(_eType) {}
    bool operator()(const SwFormToken & rToken)
    {
        return rToken.eTokenType == eType;
    }
};

/// Vector of tokens.
typedef std::vector<SwFormToken> SwFormTokens;

/**
   Helper class that converts vectors of tokens to strings and vice
   versa.
 */
class SW_DLLPUBLIC SwFormTokensHelper
{
    /// the tokens
    SwFormTokens m_Tokens;

public:
    /**
       constructor

       @param rTokens       vector of tokens
    */
    SwFormTokensHelper(const SwFormTokens & rTokens) : m_Tokens(rTokens) {}

    /**
       constructor

       @param rStr   string representation of the tokens
    */
    SwFormTokensHelper(const OUString & rStr);

    /**
       Returns vector of tokens.

       @return vector of tokens
    */
    const SwFormTokens & GetTokens() const { return m_Tokens; }
};

class SW_DLLPUBLIC SwForm
{
    SwFormTokens    aPattern[ AUTH_TYPE_END + 1 ]; // #i21237#
    OUString  aTemplate[ AUTH_TYPE_END + 1 ];

    TOXTypes    eType;
    sal_uInt16      nFormMaxLevel;

    bool    bGenerateTabPos : 1;
    bool    bIsRelTabPos : 1;
    bool    bCommaSeparated : 1;

public:
    SwForm( TOXTypes eTOXType = TOX_CONTENT );
    SwForm( const SwForm& rForm );

    SwForm& operator=( const SwForm& rForm );

    inline void SetTemplate(sal_uInt16 nLevel, const OUString& rName);
    inline OUString GetTemplate(sal_uInt16 nLevel) const;

    // #i21237#
    void    SetPattern(sal_uInt16 nLevel, const SwFormTokens& rName);
    void    SetPattern(sal_uInt16 nLevel, const OUString& rStr);
    const SwFormTokens& GetPattern(sal_uInt16 nLevel) const;

    // fill tab stop positions from template to pattern- #i21237#
    void AdjustTabStops( SwDoc& rDoc );

    inline TOXTypes GetTOXType() const;
    inline sal_uInt16   GetFormMax() const;

    bool IsRelTabPos() const    {   return bIsRelTabPos; }
    void SetRelTabPos( bool b ) {   bIsRelTabPos = b;       }

    bool IsCommaSeparated() const       { return bCommaSeparated;}
    void SetCommaSeparated( bool b)     { bCommaSeparated = b;}

    static sal_uInt16 GetFormMaxLevel( TOXTypes eType );

    static OUString GetFormEntry();
    static OUString GetFormTab();
    static OUString GetFormPageNums();
    static OUString GetFormLinkStt();
    static OUString GetFormLinkEnd();
    static OUString GetFormEntryNum();
    static OUString GetFormEntryText();
    static OUString GetFormChapterMark();
    static OUString GetFormText();
    static OUString GetFormAuth();
};

// Content to create indexes of
typedef sal_uInt16 SwTOXElement;
namespace nsSwTOXElement
{
    const SwTOXElement TOX_MARK             = 1;
    const SwTOXElement TOX_OUTLINELEVEL     = 2;
    const SwTOXElement TOX_TEMPLATE         = 4;
    const SwTOXElement TOX_OLE              = 8;
    const SwTOXElement TOX_TABLE            = 16;
    const SwTOXElement TOX_GRAPHIC          = 32;
    const SwTOXElement TOX_FRAME            = 64;
    const SwTOXElement TOX_SEQUENCE         = 128;
    const SwTOXElement TOX_TABLEADER        = 256;
    const SwTOXElement TOX_TAB_IN_TOC       = 512;
    const SwTOXElement TOX_BOOKMARK         = 1024;
    const SwTOXElement TOX_NEWLINE          = 2048;
    const SwTOXElement TOX_PARAGRAPH_OUTLINE_LEVEL = 4096;
    const SwTOXElement TOX_INDEX_ENTRY_TYPE       = 8192;
}

typedef sal_uInt16 SwTOIOptions;
namespace nsSwTOIOptions
{
    const SwTOIOptions TOI_SAME_ENTRY       = 1;
    const SwTOIOptions TOI_FF               = 2;
    const SwTOIOptions TOI_CASE_SENSITIVE   = 4;
    const SwTOIOptions TOI_KEY_AS_ENTRY     = 8;
    const SwTOIOptions TOI_ALPHA_DELIMITTER = 16;
    const SwTOIOptions TOI_DASH             = 32;
    const SwTOIOptions TOI_INITIAL_CAPS     = 64;
}

//which part of the caption is to be displayed
enum SwCaptionDisplay
{
    CAPTION_COMPLETE,
    CAPTION_NUMBER,
    CAPTION_TEXT
};

typedef sal_uInt16 SwTOOElements;
namespace nsSwTOOElements
{
    const SwTOOElements TOO_MATH            = 0x01;
    const SwTOOElements TOO_CHART           = 0x02;
    const SwTOOElements TOO_CALC            = 0x08;
    const SwTOOElements TOO_DRAW_IMPRESS    = 0x10;
    const SwTOOElements TOO_OTHER           = 0x80;
}

#define TOX_STYLE_DELIMITER ((sal_Unicode)0x01)

// Class for all indexes
class SW_DLLPUBLIC SwTOXBase : public SwClient
{
    SwForm      aForm;              // description of the lines
    OUString    aName;              // unique name
    OUString    aTitle;             // title
    OUString    m_aBookmarkName;      //Bookmark Name
    OUString    m_aEntryTypeName;   // Type name

    OUString    sMainEntryCharStyle; // name of the character style applied to main index entries

    OUString    aStyleNames[MAXLEVEL]; // (additional) style names TOX_CONTENT, TOX_USER
    OUString    sSequenceName;      // FieldTypeName of a caption sequence

    LanguageType    eLanguage;
    OUString        sSortAlgorithm;

    union {
        sal_uInt16      nLevel;             // consider outline levels
        sal_uInt16      nOptions;           // options of alphabetical index
    } aData;

    sal_uInt16      nCreateType;        // sources to create the index from
    sal_uInt16      nOLEOptions;        // OLE sources
    SwCaptionDisplay eCaptionDisplay;
    bool        bProtected : 1;         // index protected ?
    bool        bFromChapter : 1;       // create from chapter or document
    bool        bFromObjectNames : 1;   // create a table or object index
                                    // from the names rather than the caption
    bool        bLevelFromChapter : 1; // User index: get the level from the source chapter

protected:
    // Add a data member, for record the TOC field expression of MS Word binary format
    // For keeping fedality and may giving a better exporting performance
    OUString maMSTOCExpression;
    bool mbKeepExpression;

public:
    SwTOXBase( const SwTOXType* pTyp, const SwForm& rForm,
               sal_uInt16 nCreaType, const OUString& rTitle );
    SwTOXBase( const SwTOXBase& rCopy, SwDoc* pDoc = 0 );
    virtual ~SwTOXBase();

    virtual bool GetInfo( SfxPoolItem& rInfo ) const SAL_OVERRIDE;

    // a kind of CopyCtor - check if the TOXBase is at TOXType of the doc.
    // If not, so create it an copy all other used things. The return is this
    SwTOXBase& CopyTOXBase( SwDoc*, const SwTOXBase& );

    const SwTOXType*    GetTOXType() const;

    sal_uInt16              GetCreateType() const;      // creation types

    OUString            GetTOXName() const {return aName;}
    void                SetTOXName(const OUString& rSet) {aName = rSet;}

    // for record the TOC field expression of MS Word binary format
    const OUString&     GetMSTOCExpression() const{return maMSTOCExpression;}
    void                SetMSTOCExpression(const OUString& rExp) {maMSTOCExpression = rExp;}
    void                EnableKeepExpression() {mbKeepExpression = true;}
    void                DisableKeepExpression() {mbKeepExpression = false;}

    OUString            GetTitle() const;           // Title
    OUString         GetBookmarkName() const;
    OUString            GetTypeName() const;        // Name
    const SwForm&       GetTOXForm() const;         // description of the lines

    void                SetCreate(sal_uInt16);
    void                SetTitle(const OUString& rTitle);
    void                SetTOXForm(const SwForm& rForm);
    void  SetBookmarkName(const OUString& bName);

    TOXTypes            GetType() const;

    OUString            GetMainEntryCharStyle() const {return sMainEntryCharStyle;}
    void                SetMainEntryCharStyle(const OUString& rSet)  {sMainEntryCharStyle = rSet;}

    // for record the Index field expression of MS Word
    OUString         GetEntryTypeName() const;
    void  SetEntryTypeName(const OUString& sName);

    // content index only
    inline void             SetLevel(sal_uInt16);                   // consider outline level
    inline sal_uInt16           GetLevel() const;

    // alphabetical index only
    inline sal_uInt16           GetOptions() const;                 // alphabetical index options
    inline void             SetOptions(sal_uInt16 nOpt);

    // index of objects
    sal_uInt16      GetOLEOptions() const {return nOLEOptions;}
    void        SetOLEOptions(sal_uInt16 nOpt) {nOLEOptions = nOpt;}

    // index of objects

    // user defined index only
    inline void             SetTemplateName(const OUString& rName); // Absatzlayout beachten

    OUString                GetStyleNames(sal_uInt16 nLevel) const
                                {
                                SAL_WARN_IF( nLevel >= MAXLEVEL, "sw", "Which level?");
                                return aStyleNames[nLevel];
                                }
    void                    SetStyleNames(const OUString& rSet, sal_uInt16 nLevel)
                                {
                                SAL_WARN_IF( nLevel >= MAXLEVEL, "sw", "Which level?");
                                aStyleNames[nLevel] = rSet;
                                }
    bool                    IsFromChapter() const { return bFromChapter;}
    void                    SetFromChapter(bool bSet) { bFromChapter = bSet;}

    bool                    IsFromObjectNames() const {return bFromObjectNames;}
    void                    SetFromObjectNames(bool bSet) {bFromObjectNames = bSet;}

    bool                    IsLevelFromChapter() const {return bLevelFromChapter;}
    void                    SetLevelFromChapter(bool bSet) {bLevelFromChapter = bSet;}

    bool                    IsProtected() const { return bProtected; }
    void                    SetProtected(bool bSet) { bProtected = bSet; }

    OUString                GetSequenceName() const {return sSequenceName;}
    void                    SetSequenceName(const OUString& rSet) {sSequenceName = rSet;}

    SwCaptionDisplay        GetCaptionDisplay() const { return eCaptionDisplay;}
    void                    SetCaptionDisplay(SwCaptionDisplay eSet) {eCaptionDisplay = eSet;}

    bool                    IsTOXBaseInReadonly() const;

    const SfxItemSet*       GetAttrSet() const;
    void                    SetAttrSet( const SfxItemSet& );

    LanguageType    GetLanguage() const {return eLanguage;}
    void            SetLanguage(LanguageType nLang)  {eLanguage = nLang;}

    OUString        GetSortAlgorithm()const {return sSortAlgorithm;}
    void            SetSortAlgorithm(const OUString& rSet) {sSortAlgorithm = rSet;}
    // #i21237#
    void AdjustTabStops( SwDoc & rDoc )
    {
        aForm.AdjustTabStops( rDoc );
    }

    SwTOXBase& operator=(const SwTOXBase& rSource);
    void RegisterToTOXType( SwTOXType& rMark );
};

//SwTOXMark

inline OUString SwTOXMark::GetAlternativeText() const
    {   return aAltText;    }

inline OUString SwTOXMark::GetBookmarkName() const
    {   return m_aBookmarkName;    }

inline OUString SwTOXMark::GetEntryTypeName() const
    {   return m_aEntryTypeName;    }

inline const SwTOXType* SwTOXMark::GetTOXType() const
    { return static_cast<const SwTOXType*>(GetRegisteredIn()); }

inline bool SwTOXMark::IsAlternativeText() const
    { return !aAltText.isEmpty(); }

inline void SwTOXMark::SetAlternativeText(const OUString& rAlt)
{
    aAltText = rAlt;
}

inline void SwTOXMark::SetBookmarkName(const OUString& bName)
{
    m_aBookmarkName = bName;
}

inline void SwTOXMark::SetEntryTypeName(const OUString& sName)
{
    m_aEntryTypeName = sName;
}
inline void SwTOXMark::SetLevel( sal_uInt16 nLvl )
{
    SAL_WARN_IF( GetTOXType() && GetTOXType()->GetType() == TOX_INDEX, "sw", "Wrong type");
    nLevel = nLvl;
}

inline void SwTOXMark::SetPrimaryKey( const OUString& rKey )
{
    SAL_WARN_IF( GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aPrimaryKey = rKey;
}

inline void SwTOXMark::SetSecondaryKey( const OUString& rKey )
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aSecondaryKey = rKey;
}

inline void SwTOXMark::SetTextReading( const OUString& rText )
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aTextReading = rText;
}

inline void SwTOXMark::SetPrimaryKeyReading( const OUString& rKey )
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aPrimaryKeyReading = rKey;
}

inline void SwTOXMark::SetCitationKeyReading( const OUString& rKey )
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_CITATION, "sw", "Wrong type");
    aCitationKeyReading = rKey;
}

inline void SwTOXMark::SetSecondaryKeyReading( const OUString& rKey )
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aSecondaryKeyReading = rKey;
}

inline sal_uInt16 SwTOXMark::GetLevel() const
{
    SAL_WARN_IF( GetTOXType() && GetTOXType()->GetType() == TOX_INDEX, "sw", "Wrong type");
    return nLevel;
}

inline OUString SwTOXMark::GetPrimaryKey() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aPrimaryKey;
}

inline OUString SwTOXMark::GetSecondaryKey() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aSecondaryKey;
}

inline OUString SwTOXMark::GetTextReading() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aTextReading;
}

inline OUString SwTOXMark::GetPrimaryKeyReading() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aPrimaryKeyReading;
}

inline OUString SwTOXMark::GetSecondaryKeyReading() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aSecondaryKeyReading;
}

inline OUString SwTOXMark::GetCitationKeyReading() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_CITATION, "sw", "Wrong type");
    return aCitationKeyReading;
}

//SwForm

inline void SwForm::SetTemplate(sal_uInt16 nLevel, const OUString& rTemplate)
{
    SAL_WARN_IF(nLevel >= GetFormMax(), "sw", "Index >= GetFormMax()");
    aTemplate[nLevel] = rTemplate;
}

inline OUString SwForm::GetTemplate(sal_uInt16 nLevel) const
{
    SAL_WARN_IF(nLevel >= GetFormMax(), "sw", "Index >= GetFormMax()");
    return aTemplate[nLevel];
}

inline TOXTypes SwForm::GetTOXType() const
{
    return eType;
}

inline sal_uInt16 SwForm::GetFormMax() const
{
    return nFormMaxLevel;
}

//SwTOXType

inline OUString SwTOXType::GetTypeName() const
    {   return aName;   }

inline TOXTypes SwTOXType::GetType() const
    {   return eType;   }

// SwTOXBase

inline const SwTOXType* SwTOXBase::GetTOXType() const
    { return static_cast<const SwTOXType*>(GetRegisteredIn()); }

inline sal_uInt16 SwTOXBase::GetCreateType() const
    { return nCreateType; }

inline OUString SwTOXBase::GetTitle() const
    { return aTitle; }

inline OUString SwTOXBase::GetBookmarkName() const
    { return m_aBookmarkName; }

inline OUString SwTOXBase::GetEntryTypeName() const
    { return m_aEntryTypeName; }

inline OUString SwTOXBase::GetTypeName() const
    { return GetTOXType()->GetTypeName();  }

inline const SwForm& SwTOXBase::GetTOXForm() const
    { return aForm; }

inline void SwTOXBase::SetCreate(sal_uInt16 nCreate)
    { nCreateType = nCreate; }

inline void SwTOXBase::SetTOXForm(const SwForm& rForm)
    {  aForm = rForm; }

inline TOXTypes SwTOXBase::GetType() const
    { return GetTOXType()->GetType(); }

inline void SwTOXBase::SetLevel(sal_uInt16 nLev)
{
    SAL_WARN_IF(GetTOXType()->GetType() == TOX_INDEX, "sw", "Wrong type");
    aData.nLevel = nLev;
}

inline sal_uInt16 SwTOXBase::GetLevel() const
{
    SAL_WARN_IF(GetTOXType()->GetType() == TOX_INDEX, "sw", "Wrong type");
    return aData.nLevel;
}

inline void SwTOXBase::SetTemplateName(const OUString& rName)
{
    SAL_WARN("sw", "SwTOXBase::SetTemplateName obsolete");
    aStyleNames[0] = rName;
}

inline sal_uInt16 SwTOXBase::GetOptions() const
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    return aData.nOptions;
}

inline void SwTOXBase::SetOptions(sal_uInt16 nOpt)
{
    SAL_WARN_IF(GetTOXType()->GetType() != TOX_INDEX, "sw", "Wrong type");
    aData.nOptions = nOpt;
}

#endif // INCLUDED_SW_INC_TOX_HXX

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
