/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#ifndef SC_XISTRING_HXX
#define SC_XISTRING_HXX

#include "xlstring.hxx"

// Byte/Unicode strings =======================================================

class XclImpStream;

/** This class represents an unformatted or formatted string and provides importing from stream. */
class XclImpString
{
public:
    /** Constructs an empty string. */
    explicit            XclImpString();
    /** Constructs an unformatted string. */
    explicit            XclImpString( const String& rString );

                        ~XclImpString();

    /** Reads a complete string from the passed stream. */
    void                Read( XclImpStream& rStrm, XclStrFlags nFlags = EXC_STR_DEFAULT );

    /** Sets the passed string data. */
    inline void         SetText( const String& rText ) { maString = rText; }
    /** Sets the passed formatting buffer. */
    inline void         SetFormats( const XclFormatRunVec& rFormats ) { maFormats = rFormats; }
    /** Insert a formatting run to the format buffer. */
    inline void         AppendFormat( sal_uInt16 nChar, sal_uInt16 nFontIdx ) { AppendFormat( maFormats, nChar, nFontIdx ); }
    /** Reads and appends the formatting information (run count and runs) from stream. */
    inline void         ReadFormats( XclImpStream& rStrm ) { ReadFormats( rStrm, maFormats ); }
    /** Reads and appends nRunCount formatting runs from stream. */
    inline void         ReadFormats( XclImpStream& rStrm, sal_uInt16 nRunCount ) { ReadFormats( rStrm, maFormats, nRunCount ); }
    /** Reads and appends formatting runs from an OBJ or TXO record. */
    inline void         ReadObjFormats( XclImpStream& rStrm, sal_uInt16 nFormatSize ) { ReadObjFormats( rStrm, maFormats, nFormatSize ); }

    /** Returns true, if the string is empty. */
    inline bool         IsEmpty() const { return maString.Len() == 0; }
    /** Returns the pure text data of the string. */
    inline const String& GetText() const { return maString; }

    /** Returns true, if the string contains formatting information. */
    inline bool         IsRich() const { return !maFormats.empty(); }
    /** Returns the formatting run vector. */
    inline const XclFormatRunVec& GetFormats() const { return maFormats; }

    /** Insert a formatting run to the passed format buffer. */
    static void         AppendFormat( XclFormatRunVec& rFormats, sal_uInt16 nChar, sal_uInt16 nFontIdx );
    /** Reads and appends the formatting information (run count and runs) from stream. */
    static void         ReadFormats( XclImpStream& rStrm, XclFormatRunVec& rFormats );
    /** Reads and appends nRunCount formatting runs from stream. */
    static void         ReadFormats( XclImpStream& rStrm, XclFormatRunVec& rFormats, sal_uInt16 nRunCount );
    /** Reads and appends formatting runs from an OBJ or TXO record. */
    static void         ReadObjFormats( XclImpStream& rStrm, XclFormatRunVec& rFormats, sal_uInt16 nFormatSize );

private:
    String              maString;       /// The text data of the string.
    XclFormatRunVec     maFormats;      /// All formatting runs.
};

// String iterator ============================================================

/** Iterates over formatted string portions. */
class XclImpStringIterator
{
public:
    explicit            XclImpStringIterator( const XclImpString& rString );

    /** Returns true, if the iterator references a valid text portion. */
    inline bool         Is() const { return mnTextBeg < mrText.Len(); }
    /** Returns the index of the current text portion. */
    inline size_t       GetPortionIndex() const { return mnPortion; }
    /** Returns the string of the current text portion. */
    String              GetPortionText() const;
    /** Returns the font index of the current text portion. */
    sal_uInt16          GetPortionFont() const;

    /** Moves iterator to next text portion. */
    XclImpStringIterator& operator++();

private:
    const String&       mrText;         /// The processed string.
    const XclFormatRunVec& mrFormats;   /// The vector of formatting runs.
    size_t              mnPortion;      /// Current text portion.
    xub_StrLen          mnTextBeg;      /// First character of current portion.
    xub_StrLen          mnTextEnd;      /// First character of next portion.
    size_t              mnFormatsBeg;   /// Formatting run index for current portion.
    size_t              mnFormatsEnd;   /// Formatting run index for next portion.
};

// ============================================================================

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
