/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include "sal/config.h"

#if defined MACOSX
#include <cassert>
#include <limits>
#include <unistd.h>
#endif

#include "osl/process.h"
#include "sal/main.h"
#include "sal/types.h"

extern "C" {

void sal_detail_initialize(int argc, char ** argv) {
#if defined MACOSX
    // On Mac OS X, soffice can restart itself via exec (see restartOnMac in
    // desktop/source/app/app.cxx), which leaves all file descriptors open,
    // which in turn can have unwanted effects (see
    // <https://bugs.freedesktop.org/show_bug.cgi?id=50603> "Unable to update
    // LibreOffice without resetting user profile").  But closing fds in
    // restartOnMac before calling exec does not work, as additional threads
    // might still be running then, wich can still use those fds and cause
    // crashes.  Therefore, the simples solution is to close fds at process
    // start (as early as possible, so that no other threads have been created
    // yet that might already have opened some fds); this is done for all kinds
    // of processes here, not just soffice, but hopefully none of our processes
    // rely on being spawned with certain fds already open.  Unfortunately, Mac
    // OS X appears to have no better interface to close all fds (like
    // closefrom):
    long openMax = sysconf(_SC_OPEN_MAX);
    if (openMax == -1) {
        // Some random value, but hopefully sysconf never returns -1 anyway:
        openMax = 1024;
    }
    assert(openMax >= 0 && openMax <= std::numeric_limits< int >::max());
    for (int fd = 3; fd < openMax; ++fd) {
        close(fd);
    }
#endif

    osl_setCommandArgs(argc, argv);
}

void sal_detail_deinitialize() {}

}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
