"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.backendToUiReportsList = exports.backendToUiReportDefinitionsList = exports.backendToUiReportDefinition = exports.backendToUiReport = void 0;

var _model = require("../../../model");

var _backendModel = require("../../../model/backendModel");

var _constants = require("../constants");

var _moment = _interopRequireDefault(require("moment"));

var _url = require("url");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const backendToUiReport = (backendReportInstance, serverBasePath) => {
  const {
    inContextDownloadUrlPath,
    beginTimeMs,
    endTimeMs,
    tenant,
    status,
    lastUpdatedTimeMs: reportLastUpdatedTimeMs,
    createdTimeMs: reportCreatedTimeMs,
    reportDefinitionDetails: backendReportDefinitionDetails
  } = backendReportInstance;
  const {
    reportDefinition: {
      source: {
        type: sourceType,
        id: sourceId
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  let report = {
    // inContextDownloadUrlPath may not exist for report instance created from scheduled job
    query_url: inContextDownloadUrlPath || getUiQueryUrl(baseUrl, beginTimeMs, endTimeMs, tenant),
    time_from: beginTimeMs,
    time_to: endTimeMs,
    last_updated: reportLastUpdatedTimeMs,
    time_created: reportCreatedTimeMs,
    state: getUiReportState(status, delivery),
    report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
  }; // Add severbasePath back to query_url

  report.query_url = serverBasePath + report.query_url;
  return report;
};

exports.backendToUiReport = backendToUiReport;

const backendToUiReportsList = (backendReportsList, serverBasePath) => {
  const res = backendReportsList.map(backendReport => {
    return {
      _id: backendReport.id,
      _source: backendToUiReport(backendReport, serverBasePath)
    };
  });
  return res;
};

exports.backendToUiReportsList = backendToUiReportsList;

const backendToUiReportDefinition = (backendReportDefinitionDetails, serverBasePath) => {
  const {
    lastUpdatedTimeMs,
    createdTimeMs,
    reportDefinition: {
      name,
      isEnabled,
      source: {
        type: sourceType,
        description,
        id: sourceId,
        origin
      },
      format: {
        fileFormat,
        duration,
        header,
        footer,
        limit
      },
      trigger: {
        triggerType,
        schedule
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  const reportSource = getUiReportSource(sourceType);
  let uiReportDefinition = {
    report_params: {
      report_name: name,
      report_source: reportSource,
      description: description,
      core_params: reportSource === _constants.REPORT_TYPE.savedSearch ? getDataReportCoreParams(limit, sourceId, fileFormat, duration, baseUrl, origin) : getVisualReportCoreParams(fileFormat, header, footer, duration, baseUrl, origin)
    },
    trigger: getUiTriggerParams(triggerType, schedule, createdTimeMs, isEnabled),
    delivery: getUiDeliveryParams(delivery),
    //TODO:
    time_created: createdTimeMs,
    last_updated: lastUpdatedTimeMs,
    status: getUiReportDefinitionStatus(isEnabled)
  }; // validate to assign default values to some fields for UI model

  uiReportDefinition = _model.reportDefinitionSchema.validate(uiReportDefinition);
  uiReportDefinition.report_params.core_params.base_url = serverBasePath + uiReportDefinition.report_params.core_params.base_url;
  return uiReportDefinition;
};

exports.backendToUiReportDefinition = backendToUiReportDefinition;

const backendToUiReportDefinitionsList = (backendReportDefinitionDetailsList, serverBasePath) => {
  const res = backendReportDefinitionDetailsList.map(backendReportDefinitionDetails => {
    return {
      _id: backendReportDefinitionDetails.id,
      _source: {
        // TODO: this property can be removed, but need UI changes as well
        report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
      }
    };
  });
  return res;
};

exports.backendToUiReportDefinitionsList = backendToUiReportDefinitionsList;

const getVisualReportCoreParams = (fileFormat, header = '', footer = '', duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    header: header,
    footer: footer,
    time_duration: duration,
    origin: origin
  };
  return res;
}; // queryUrl = baseUrl + time range


const getUiQueryUrl = (baseUrl, beginTimeMs, endTimeMs, tenant) => {
  const timeFrom = (0, _moment.default)(beginTimeMs).toISOString();
  const timeTo = (0, _moment.default)(endTimeMs).toISOString();
  let queryUrl = `${baseUrl}?_g=(time:(from:'${timeFrom}',to:'${timeTo}'))`;

  if (tenant !== undefined) {
    if (tenant === '') {
      tenant = 'global';
    } else if (tenant === '__user__') {
      tenant = 'private';
    }

    queryUrl = addTenantToURL(queryUrl, tenant);
  }

  return queryUrl;
};

const getBaseUrl = (sourceType, sourceId) => {
  //TODO: AES domain has different prefix, need figure out a general solution
  const baseUrl = `${_backendModel.URL_PREFIX_DICT[sourceType]}${sourceId}`;
  return baseUrl;
};

const getDataReportCoreParams = (limit = _constants.DEFAULT_MAX_SIZE, sourceId, fileFormat, duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    limit: limit,
    time_duration: duration,
    saved_search_id: sourceId,
    origin: origin
  };
  return res;
};

const getUiScheduleParams = (schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_params: {
      enabled_time: createdTimeMs,
      enabled: isEnabled,
      schedule_type: schedule && 'cron' in schedule ? _constants.SCHEDULE_TYPE.cron : _constants.SCHEDULE_TYPE.recurring,
      schedule: schedule
    }
  };
  return res;
};

const getUiTriggerType = backendField => {
  let res;

  for (let [ui, backendFieldList] of Object.entries(_backendModel.TRIGGER_TYPE_DICT)) {
    for (let item of backendFieldList) {
      if (item === backendField) {
        res = ui;
      }
    }
  }

  return res;
};

const getUiReportFormat = backendField => {
  let res;

  for (let [ui, backend] of Object.entries(_backendModel.REPORT_FORMAT_DICT)) {
    if (backend === backendField) {
      res = ui;
    }
  }

  return res;
};

const getUiReportState = (status, delivery) => {
  let res;

  for (let [ui, backend] of Object.entries(_backendModel.REPORT_STATE_DICT)) {
    if (backend === status) {
      // distinguish "shared" and "created"
      if (status === _backendModel.BACKEND_REPORT_STATE.success && delivery) {
        res = _constants.REPORT_STATE.shared;
      } else {
        res = ui;
      }
    } else if (status === _backendModel.BACKEND_REPORT_STATE.scheduled) {
      // corner case
      res = _constants.REPORT_STATE.pending;
    }
  }

  return res;
};

const getUiReportSource = type => {
  let res;

  for (let [ui, backend] of Object.entries(_backendModel.REPORT_SOURCE_DICT)) {
    if (backend === type) {
      res = ui;
    }
  }

  return res;
};

const getUiReportDefinitionStatus = isEnabled => {
  return isEnabled ? _constants.REPORT_DEFINITION_STATUS.active : _constants.REPORT_DEFINITION_STATUS.disabled;
};

const getUiTriggerParams = (triggerType, schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_type: getUiTriggerType(triggerType),
    ...(getUiTriggerType(triggerType) === _constants.TRIGGER_TYPE.schedule && getUiScheduleParams(schedule, createdTimeMs, isEnabled))
  };
  return res;
}; // Delivery


const getUiDeliveryParams = delivery => {
  const opensearchDashboardsUserDeliveryParams = {
    configIds: [],
    title: '',
    textDescription: '',
    htmlDescription: ''
  };
  let params;

  if (delivery) {
    const { ...rest
    } = delivery;
    params = { ...rest
    };
  } else {
    params = opensearchDashboardsUserDeliveryParams;
  }

  return params;
}; // helper function to add tenant info to url(if tenant is available)


const addTenantToURL = (url, userRequestedTenant) => {
  // build fake url from relative url
  const fakeUrl = `http://opensearch.com${url}`;
  const tenantKey = 'security_tenant';
  const tenantKeyAndValue = tenantKey + '=' + encodeURIComponent(userRequestedTenant);
  const {
    pathname,
    search
  } = (0, _url.parse)(fakeUrl);
  const queryDelimiter = !search ? '?' : '&'; // The url parser returns null if the search is empty. Change that to an empty
  // string so that we can use it to build the values later

  if (search && search.toLowerCase().indexOf(tenantKey) > -1) {
    // If we for some reason already have a tenant in the URL we skip any updates
    return url;
  } // A helper for finding the part in the string that we want to extend/replace


  const valueToReplace = pathname + (search || '');
  const replaceWith = valueToReplace + queryDelimiter + tenantKeyAndValue;
  return url.replace(valueToReplace, replaceWith);
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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